/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%                            DDDD   PPPP   X   X                              %
%                            D   D  P   P   X X                               %
%                            D   D  PPPP    XXX                               %
%                            D   D  P       X X                               %
%                            DDDD   P      X   X                              %
%                                                                             %
%                                                                             %
%                     Read/Write SMTPE DPX Image Format.                      %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                March 2001                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2005 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/script/license.php                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/attribute.h"
#include "magick/blob.h"
#include "magick/blob-private.h"
#include "magick/colorspace.h"
#include "magick/exception.h"
#include "magick/exception-private.h"
#include "magick/image.h"
#include "magick/image-private.h"
#include "magick/list.h"
#include "magick/magick.h"
#include "magick/memory_.h"
#include "magick/monitor.h"
#include "magick/static.h"
#include "magick/string_.h"

/*
  Typedef declaration.
*/
typedef struct _DPXFileInfo
{
  unsigned long
    magic,
    image_offset;

  char
    version[8];

  unsigned long
    file_size,
    ditto_key,
    generic_size,
    industry_size,
    user_size;

  char
    filename[100],
    timestamp[24],
    creator[100],
    project[200],
    copyright[200];

  unsigned long
    encrypt_key;

  char
    reserve[104];
} DPXFileInfo;

typedef struct _DPXFilmInfo
{
  char
    id[2],
    type[2],
    offset[2],
    prefix[6],
    count[4],
    format[32];

  unsigned long
    frame_position,
    sequence_length,
    held_count;

  float
    frame_rate,
    shutter_angle;

  char
    frame_id[32],
    slate[100],
    reserve[56];
} DPXFilmInfo;

typedef struct _DPXImageElement
{
  unsigned long
    data_sign,
    low_data;

  float
    low_quantity;

  unsigned long
    high_data;

  float
    high_quantity;

  unsigned char
    descriptor,
    transfer,
    colorimetric,
    bit_size;

  unsigned short
    packing,
    encoding;

  unsigned long
    data_offset,
    end_of_line_padding,
    end_of_image_padding;

  unsigned char
    description[52];
} DPXImageElement;

typedef struct _DPXImageInfo
{
  unsigned short
    orientation,
    number_elements;

  unsigned long
    pixels_per_line,
    lines_per_element;

  DPXImageElement
    image_element[8];

  unsigned char
    reserve[52];
} DPXImageInfo;

typedef struct _DPXOrientationInfo
{
  unsigned long
    x_offset,
    y_offset;

  float
    x_center,
    y_center;

  unsigned short
    x_size,
    y_size;

  unsigned char
    filename[100],
    timestamp[24],
    device[32],
    serial[32];

  unsigned short
    border[4];

  unsigned long
    aspect_ratio[2];

  unsigned char
    reserve[28];
} DPXOrientationInfo;

typedef struct _DPXTelevisionInfo
{
  unsigned long
    time_code,
    user_bits;

  unsigned char
    interlace,
    field_number,
    video_signal,
    padding;

  float
    horizontal_sample_rate,
    vertical_sample_rate,
    frame_rate,
    time_offset,
    gamma,
    black_level,
    black_gain,
    break_point,
    white_level,
    integration_times;

  char
    reserve[76];
} DPXTelevisionInfo;

typedef struct _DPXInfoInfo
{
  char
    id[32];
} DPXUserInfo;

/*
  Forward declaractions.
*/
static MagickBooleanType
  WriteDPXImage(const ImageInfo *,Image *);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I s D P X                                                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsDPX() returns MagickTrue if the image format type, identified by the
%  magick string, is DPX.
%
%  The format of the IsDPX method is:
%
%      MagickBooleanType IsDPX(const unsigned char *magick,const size_t length)
%
%  A description of each parameter follows:
%
%    o magick: This string is generally the first few bytes of an image file
%      or blob.
%
%    o length: Specifies the length of the magick string.
%
%
*/
static MagickBooleanType IsDPX(const unsigned char *magick,const size_t length)
{
  if (length < 4)
    return(MagickFalse);
  if (memcmp(magick,"SDPX",4) == 0)
    return(MagickTrue);
  if (memcmp(magick,"XPDS",4) == 0)
    return(MagickTrue);
  return(MagickFalse);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e a d D P X I m a g e                                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ReadDPXImage() reads an DPX X image file and returns it.  It
%  allocates the memory necessary for the new Image structure and returns a
%  pointer to the new image.
%
%  The format of the ReadDPXImage method is:
%
%      Image *ReadDPXImage(const ImageInfo *image_info,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o exception: return any errors or warnings in this structure.
%
%
*/
static Image *ReadDPXImage(const ImageInfo *image_info,ExceptionInfo *exception)
{
#define MonoColorType  6
#define RGBColorType  50

  char
    magick[4];

  DPXFileInfo
    dpx_file;

  DPXFilmInfo
    dpx_film;

  DPXImageInfo
    dpx_image;

  DPXOrientationInfo
    dpx_orientation;

  DPXTelevisionInfo
    dpx_television;

  DPXUserInfo
    dpx_user;

  Image
    *image;

  long
    y;

  MagickBooleanType
    status;

  QuantumType
    quantum_type;

  register const unsigned char
    *p;

  register long
    i;

  register PixelPacket
    *q;

  ssize_t
    count;

  size_t
    length;

  unsigned char
    colortype,
    *pixels;

  /*
    Open image file.
  */
  assert(image_info != (const ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"%s",
      image_info->filename);
  assert(exception != (ExceptionInfo *) NULL);
  assert(exception->signature == MagickSignature);
  image=AllocateImage(image_info);
  status=OpenBlob(image_info,image,ReadBinaryBlobMode,exception);
  if (status == MagickFalse)
    {
      image=DestroyImageList(image);
      return((Image *) NULL);
    }
  /*
    Read DPX file header.
  */
  count=ReadBlob(image,4,(unsigned char *) magick);
  if ((count != 4) || ((LocaleNCompare(magick,"SDPX",4) != 0) &&
      (LocaleNCompare((char *) magick,"XPDS",4) != 0)))
    ThrowReaderException(CorruptImageError,"ImproperImageHeader");
  image->endian=LSBEndian;
  if (LocaleNCompare(magick,"SDPX",4) == 0)
    image->endian=MSBEndian;
  dpx_file.image_offset=ReadBlobLong(image);
  (void) ReadBlob(image,8,(unsigned char *) dpx_file.version);
  (void) SetImageAttribute(image,"dpx:file.version",dpx_file.version);
  dpx_file.file_size=ReadBlobLong(image);
  dpx_file.ditto_key=ReadBlobLong(image);
  dpx_file.generic_size=ReadBlobLong(image);
  dpx_file.industry_size=ReadBlobLong(image);
  dpx_file.user_size=ReadBlobLong(image);
  (void) ReadBlob(image,100,(unsigned char *) dpx_file.filename);
  (void) SetImageAttribute(image,"dpx:file.filename",dpx_file.filename);
  (void) ReadBlob(image,24,(unsigned char *) dpx_file.timestamp);
  (void) SetImageAttribute(image,"dpx:file.timestamp",dpx_file.timestamp);
  (void) ReadBlob(image,100,(unsigned char *) dpx_file.creator);
  (void) SetImageAttribute(image,"dpx:file.creator",dpx_file.creator);
  (void) ReadBlob(image,200,(unsigned char *) dpx_file.project);
  (void) SetImageAttribute(image,"dpx:file.project",dpx_file.project);
  (void) ReadBlob(image,200,(unsigned char *) dpx_file.copyright);
  (void) SetImageAttribute(image,"dpx:file.copyright",dpx_file.copyright);
  dpx_file.encrypt_key=ReadBlobLong(image);
  (void) ReadBlob(image,104,(unsigned char *) dpx_file.reserve);
  dpx_image.orientation=ReadBlobShort(image);
  /*
    Read DPX image header.
  */
  switch (dpx_image.orientation)
  {
    default:
    case 0:  image->orientation=TopLeftOrientation; break;
    case 1:  image->orientation=TopRightOrientation; break;
    case 2:  image->orientation=BottomLeftOrientation; break;
    case 3:  image->orientation=BottomRightOrientation; break;
    case 4:  image->orientation=LeftTopOrientation; break;
    case 5:  image->orientation=RightTopOrientation; break;
    case 6:  image->orientation=LeftBottomOrientation; break;
    case 7:  image->orientation=RightBottomOrientation; break;
  }
  dpx_image.number_elements=ReadBlobShort(image);
  dpx_image.pixels_per_line=ReadBlobLong(image);
  image->columns=dpx_image.pixels_per_line;
  dpx_image.lines_per_element=ReadBlobLong(image);
  image->rows=dpx_image.lines_per_element;
  for (i=0; i < 8; i++)
  {
    dpx_image.image_element[i].data_sign=ReadBlobLong(image);
    dpx_image.image_element[i].low_data=ReadBlobLong(image);
    dpx_image.image_element[i].low_quantity=ReadBlobLong(image);
    dpx_image.image_element[i].high_data=ReadBlobLong(image);
    dpx_image.image_element[i].high_quantity=ReadBlobLong(image);
    dpx_image.image_element[i].descriptor=ReadBlobByte(image);
    dpx_image.image_element[i].transfer=ReadBlobByte(image);
    dpx_image.image_element[i].colorimetric=ReadBlobByte(image);
    dpx_image.image_element[i].bit_size=ReadBlobByte(image);
    dpx_image.image_element[i].packing=ReadBlobShort(image);
    dpx_image.image_element[i].encoding=ReadBlobShort(image);
    dpx_image.image_element[i].data_offset=ReadBlobLong(image);
    dpx_image.image_element[i].end_of_line_padding=ReadBlobLong(image);
    dpx_image.image_element[i].end_of_image_padding=ReadBlobLong(image);
    (void) ReadBlob(image,32,(unsigned char *)
      dpx_image.image_element[i].description);
  }
  (void) ReadBlob(image,52,(unsigned char *) dpx_image.reserve);
  colortype=dpx_image.image_element[0].descriptor;
  image->depth=dpx_image.image_element[0].bit_size;
  /*
    Read DPX orientation header.
  */
  dpx_orientation.x_offset=ReadBlobLong(image);
  dpx_orientation.y_offset=ReadBlobLong(image);
  dpx_orientation.x_center=ReadBlobLong(image);
  dpx_orientation.y_center=ReadBlobLong(image);
  dpx_orientation.x_size=ReadBlobLong(image);
  dpx_orientation.y_size=ReadBlobLong(image);
  (void) ReadBlob(image,100,(unsigned char *) dpx_orientation.filename);
  (void) ReadBlob(image,24,(unsigned char *) dpx_orientation.timestamp);
  (void) ReadBlob(image,32,(unsigned char *) dpx_orientation.device);
  (void) ReadBlob(image,32,(unsigned char *) dpx_orientation.serial);
  for (i=0; i < 4; i++)
    dpx_orientation.border[i]=ReadBlobShort(image);
  for (i=0; i < 2; i++)
    dpx_orientation.aspect_ratio[i]=ReadBlobLong(image);
  (void) ReadBlob(image,28,(unsigned char *) dpx_orientation.reserve);
  /*
    Read DPX film header.
  */
  (void) ReadBlob(image,2,(unsigned char *) dpx_film.id);
  (void) ReadBlob(image,2,(unsigned char *) dpx_film.type);
  (void) ReadBlob(image,2,(unsigned char *) dpx_film.offset);
  (void) ReadBlob(image,6,(unsigned char *) dpx_film.prefix);
  (void) ReadBlob(image,4,(unsigned char *) dpx_film.count);
  (void) ReadBlob(image,32,(unsigned char *) dpx_film.format);
  dpx_film.frame_position=ReadBlobLong(image);
  dpx_film.sequence_length=ReadBlobLong(image);
  dpx_film.held_count=ReadBlobLong(image);
  dpx_film.frame_rate=ReadBlobLong(image);
  dpx_film.shutter_angle=ReadBlobLong(image);
  (void) ReadBlob(image,32,(unsigned char *) dpx_film.frame_id);
  (void) ReadBlob(image,100,(unsigned char *) dpx_film.slate);
  (void) ReadBlob(image,56,(unsigned char *) dpx_film.reserve);
  /*
    Read DPX televison header.
  */
  dpx_television.time_code=ReadBlobLong(image);
  dpx_television.user_bits=ReadBlobLong(image);
  dpx_television.interlace=ReadBlobByte(image);
  dpx_television.field_number=ReadBlobByte(image);
  dpx_television.video_signal=ReadBlobByte(image);
  dpx_television.padding=ReadBlobByte(image);
  dpx_television.horizontal_sample_rate=ReadBlobLong(image);
  dpx_television.vertical_sample_rate=ReadBlobLong(image);
  dpx_television.frame_rate=ReadBlobLong(image);
  dpx_television.time_offset=ReadBlobLong(image);
  dpx_television.gamma=ReadBlobLong(image);
  dpx_television.black_level=ReadBlobLong(image);
  dpx_television.black_gain=ReadBlobLong(image);
  dpx_television.break_point=ReadBlobLong(image);
  dpx_television.white_level=ReadBlobLong(image);
  dpx_television.integration_times=ReadBlobLong(image);
  (void) ReadBlob(image,76,(unsigned char *) dpx_television.reserve);
  /*
    Read DPX user header.
  */
  (void) ReadBlob(image,32,(unsigned char *) dpx_user.id);
  /*
    Read DPX image header.
  */
  for (i=0; i < (long) (dpx_file.image_offset-2080); i++)
    (void) ReadBlobByte(image);
  if (image_info->ping != MagickFalse)
    {
      CloseBlob(image);
      return(GetFirstImageInList(image));
    }
  /*
    Convert DPX raster image to pixel packets.
  */
  length=4*image->columns*sizeof(*pixels);
  if (image->depth >= 12)
    length=6*image->columns*sizeof(*pixels);
  pixels=(unsigned char *) AcquireMagickMemory(length);
  if (pixels == (unsigned char *) NULL)
    ThrowReaderException(ResourceLimitError,"MemoryAllocationFailed");
  quantum_type=RGBPadQuantum;
  if (colortype == MonoColorType)
    quantum_type=GrayQuantum;
  for (y=0; y < (long) image->rows; y++)
  {
    q=SetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    p=ReadBlobStream(image,length,pixels,&count);
    if ((size_t) count != length)
      break;
    status=ExportQuantumPixels(image,quantum_type,0,p);
    if (status == MagickFalse)
      break;
    if (SyncImagePixels(image) == MagickFalse)
      break;
    if (image->previous == (Image *) NULL)
      if ((image->progress_monitor != (MagickProgressMonitor) NULL) &&
          (QuantumTick(y,image->rows) != MagickFalse))
        {
          status=image->progress_monitor(LoadImageTag,y,image->rows,
            image->client_data);
          if (status == MagickFalse)
            break;
        }
  }
  pixels=(unsigned char *) RelinquishMagickMemory(pixels);
  if (EOFBlob(image))
    ThrowFileException(exception,CorruptImageError,"UnexpectedEndOfFile",
      image->filename);
  image->colorspace=LogColorspace;
  CloseBlob(image);
  return(GetFirstImageInList(image));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e g i s t e r D P X I m a g e                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RegisterDPXImage() adds attributes for the DPX image format to
%  the list of supported formats.  The attributes include the image format
%  tag, a method to read and/or write the format, whether the format
%  supports the saving of more than one frame to the same file or blob,
%  whether the format supports native in-memory I/O, and a brief
%  description of the format.
%
%  The format of the RegisterDPXImage method is:
%
%      RegisterDPXImage(void)
%
*/
ModuleExport void RegisterDPXImage(void)
{
  MagickInfo
    *entry;

  static const char
    *DPXNote =
    {
      "File Format for Digital Moving-Picture Exchange (DPX) Version 2.0.\n"
      "See SMPTE 268M-2003 specification at http://www.smtpe.org\n"
    };

  entry=SetMagickInfo("DPX");
  entry->decoder=(DecoderHandler *) ReadDPXImage;
  entry->encoder=(EncoderHandler *) WriteDPXImage;
  entry->magick=(MagickHandler *) IsDPX;
  entry->description=AcquireString("SMPTE 268M-2003 (DPX)");
  entry->note=AcquireString(DPXNote);
  entry->module=AcquireString("DPX");
  (void) RegisterMagickInfo(entry);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   U n r e g i s t e r D P X I m a g e                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  UnregisterDPXImage() removes format registrations made by the
%  DPX module from the list of supported formats.
%
%  The format of the UnregisterDPXImage method is:
%
%      UnregisterDPXImage(void)
%
*/
ModuleExport void UnregisterDPXImage(void)
{
  (void) UnregisterMagickInfo("DPX");
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   W r i t e D P X I m a g e                                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  WriteDPXImage() writes an image in DPX encoded image format.
%
%  The format of the WriteDPXImage method is:
%
%      MagickBooleanType WriteDPXImage(const ImageInfo *image_info,Image *image)
%
%  A description of each parameter follows.
%
%    o image_info: The image info.
%
%    o image:  The image.
%
%
*/
static MagickBooleanType WriteDPXImage(const ImageInfo *image_info,Image *image)
{
  DPXFileInfo
    dpx_file;

  long
    y;

  MagickBooleanType
    status;

  register const PixelPacket
    *p;

  register long
    i;

  size_t
    length;

  ssize_t
    count;

  unsigned char
    *pixels;

  /*
    Open output image file.
  */
  assert(image_info != (const ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"%s",image->filename);
  status=OpenBlob(image_info,image,WriteBinaryBlobMode,&image->exception);
  if (status == MagickFalse)
    return(status);
  (void) ResetMagickMemory(&dpx_file,0,sizeof(dpx_file));
  (void) SetImageColorspace(image,LogColorspace);
  (void) WriteBlobLong(image,0x53445058);
  (void) WriteBlobLong(image,0x2000);
  (void) CopyMagickString(dpx_file.version,"V2.0",MaxTextExtent);
  (void) WriteBlob(image,8,(unsigned char *) &dpx_file.version);
  (void) WriteBlobLong(image,4*image->columns*image->rows+0x2000);
  (void) WriteBlobLong(image,0x00000001);
  (void) WriteBlobLong(image,0x00000680);
  (void) WriteBlobLong(image,0x00000180);
  (void) WriteBlobLong(image,0x00001800);
  for (i=0; i < 124; i++)
    (void) WriteBlobByte(image,0x00);
  (void) WriteBlobLong(image,0x496D6167);
  (void) WriteBlobLong(image,0x654D6167);
  (void) WriteBlobLong(image,0x69636B20);
  for (i=0; i < 599; i++)
    (void) WriteBlobByte(image,0x00);
  (void) WriteBlobByte(image,0x01);
  (void) WriteBlobLong(image,image->columns);
  (void) WriteBlobLong(image,image->rows);
  for (i=0; i < 20; i++)
    (void) WriteBlobByte(image,0x00);
  (void) WriteBlobByte(image,RGBColorType);
  (void) WriteBlobByte(image,0x00);
  (void) WriteBlobByte(image,0x00);
  (void) WriteBlobByte(image,image->depth);  /* bit depth */
  (void) WriteBlobByte(image,0x00);
  (void) WriteBlobByte(image,0x01);
  for (i=0; i < (0x2000-806); i++)
    (void) WriteBlobByte(image,0x00);
  /*
    Convert pixel packets to DPX raster image.
  */
  length=4*image->columns*sizeof(*pixels);
  if (image->depth >= 12)
    length=6*image->columns*sizeof(*pixels);
  pixels=(unsigned char *) AcquireMagickMemory(length);
  if (pixels == (unsigned char *) NULL)
    ThrowWriterException(ResourceLimitError,"MemoryAllocationFailed");
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    status=ImportQuantumPixels(image,RGBPadQuantum,0,pixels);
    if (status == MagickFalse)
      break;
    count=WriteBlob(image,length,pixels);
    if (count != (ssize_t) length)
      break;
  }
  pixels=(unsigned char *) RelinquishMagickMemory(pixels);
  CloseBlob(image);
  return(status);
}
