/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * callbacks_mainwindow_qsoframe.c - callbacks for the QSO frame, where all 
 * the entry widgets are ...
 */

#include <time.h>
#include <math.h>
#include <ctype.h>
#ifdef __FreeBSD__
	#include <machine/param.h>
	#include <sys/types.h>
#endif
#include <sys/ipc.h>
#include <sys/shm.h>
#include <errno.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#if WANT_HAMLIB
#include <hamlib/rig.h>
#include "hamlib-utils.h"
#endif

#include "callbacks_mainwindow_qsoframe.h"
#include "support.h"
#include "dxcc.h"
#include "types.h"
#include "utils.h"
#include "types.h"
#include "log.h"
#include "wwl.h"

/* defines and variables for shared memory support */
#define KEY 6146
#define SHMSIZE	40
void *shareCall;

extern GtkWidget *mainwindow, *b4dialog;
extern gint callid;
extern statetype state;
extern GList *logwindowlist;
extern preferencestype preferences;
extern GList *logwindowlist;
extern gchar **bandsplit, **modesplit;

/* QSO FRAME */
/* get current date and fill in the dateentry */
void
on_datebutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *nowdate;

	nowdate = getdate ();
	entry = lookup_widget (mainwindow, "dateentry");
	gtk_entry_set_text (GTK_ENTRY (entry), nowdate);
	g_free (nowdate);
}

/* get current time and fill in the gmtentry */
void
on_gmtbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *nowtime;

	entry = lookup_widget (mainwindow, "gmtentry");
	nowtime = gettime ();
	gtk_entry_set_text (GTK_ENTRY (entry), nowtime);
	g_free (nowtime);
}

void
on_endbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *nowtime;

	entry = lookup_widget (mainwindow, "endentry");
	nowtime = gettime ();
	gtk_entry_set_text (GTK_ENTRY (entry), nowtime);
	g_free (nowtime);
}

/* convert callsign to uppercase */
void
on_callentry_insert_text (GtkEditable * editable, gchar * new_text,
	gint new_text_length, gpointer position, gpointer user_data)
{
	gint i;
	gchar *result = g_new (gchar, new_text_length);

	for (i = 0; i < new_text_length; i++)
		result[i] = islower (new_text[i]) ? toupper (new_text[i]) : new_text[i];

	/* insert the new callsign and block insert_text, so we don't call this twice */
	g_signal_handlers_block_by_func (GTK_OBJECT (editable),	on_callentry_insert_text, user_data);
	gtk_editable_insert_text (editable, result, new_text_length, position);
	g_signal_handlers_unblock_by_func (GTK_OBJECT (editable),	on_callentry_insert_text, user_data);
	g_signal_stop_emission_by_name (GTK_OBJECT (editable), "insert_text"); 
	g_free (result);
}

static void
updateb4dialog_and_typeaheadfind (gchar * callsign)
{
	GtkWidget *b4treeview, *remarksvbox, *remarksentry, *namehbox, *nameentry,
		*qthhbox, *qthentry, *locatorhbox, *locatorentry;
	guint i, j;
	logtype *logw;
	gchar *logcallsign, **b4 = NULL, *temp, *entry;
	gboolean found = FALSE, valid = FALSE;
	GtkTreeIter iter, b4iter;
	GtkTreeModel *model, *b4model = NULL;
	GtkTreeViewColumn *column;

	if (b4dialog && GTK_WIDGET_VISIBLE (b4dialog))
	{
	/* clear the list */
		b4treeview = lookup_widget (b4dialog, "b4treeview");
		b4model = gtk_tree_view_get_model (GTK_TREE_VIEW (b4treeview));
		gtk_tree_store_clear (GTK_TREE_STORE (b4model));

		if (strlen(callsign) < 2) return; /* for responsiveness */

		b4 = g_new0 (gchar *, QSO_FIELDS + 1);
		for (j = 0; j < QSO_FIELDS + 1; j++)
			b4[j] = g_new0 (gchar, 100);
	}

	/* search all logs, if there is a match display it */
	if ((b4dialog && GTK_WIDGET_VISIBLE (b4dialog))|| (preferences.typeaheadfind == 1))
	{
		for (i = 0; i < g_list_length (logwindowlist); i++)
		{
			logw = g_list_nth_data (logwindowlist, i);
			model = gtk_tree_view_get_model (GTK_TREE_VIEW(logw->treeview));
			valid = gtk_tree_model_get_iter_first (model, &iter);
			while (valid)
			{
				gtk_tree_model_get (model, &iter, CALL, &logcallsign, -1);
				if ((preferences.typeaheadfind == 1) && !found && !g_ascii_strcasecmp (callsign, logcallsign))
				{ /* type ahead find */
					namehbox = lookup_widget(mainwindow, "namehbox");
					if (GTK_WIDGET_VISIBLE (namehbox))
					{
						nameentry = lookup_widget(mainwindow, "nameentry");
						entry = gtk_editable_get_chars (GTK_EDITABLE (nameentry), 0, -1);
						if (strlen (entry) == 0)
						{
							gtk_tree_model_get (model, &iter, NAME, &temp, -1);
							gtk_entry_set_text (GTK_ENTRY (nameentry), temp);
						}
						g_free (entry);
					}
					qthhbox = lookup_widget(mainwindow, "qthhbox");
					if (GTK_WIDGET_VISIBLE (qthhbox))
					{
						qthentry = lookup_widget(mainwindow, "qthentry");
						entry = gtk_editable_get_chars (GTK_EDITABLE (qthentry), 0, -1);
						if (strlen (entry) == 0)
						{
							gtk_tree_model_get (model, &iter, QTH, &temp, -1);
							gtk_entry_set_text (GTK_ENTRY (qthentry), temp);
						}
						g_free (entry);
					}
					locatorhbox = lookup_widget(mainwindow, "locatorhbox");
					if (GTK_WIDGET_VISIBLE (locatorhbox))
					{
						locatorentry = lookup_widget(mainwindow, "locatorentry");
						entry = gtk_editable_get_chars (GTK_EDITABLE (locatorentry), 0, -1);
						if (strlen (entry) == 0)
						{
 							gtk_tree_model_get (model, &iter, LOCATOR, &temp, -1);
							gtk_entry_set_text (GTK_ENTRY (locatorentry), temp);
						}
						g_free (entry);
					}
					remarksvbox = lookup_widget(mainwindow, "remarksvbox");
					if (!g_ascii_strcasecmp (preferences.defaultremarks, "?") && GTK_WIDGET_VISIBLE (remarksvbox))
					{
						remarksentry = lookup_widget(mainwindow, "remarksentry");
						entry = gtk_editable_get_chars (GTK_EDITABLE (remarksentry), 0, -1);
						if (strlen (entry) == 0)
						{
							gtk_tree_model_get (model, &iter, REMARKS, &temp, -1);
							gtk_entry_set_text (GTK_ENTRY (remarksentry), temp);
						}
						g_free (entry);
					}
					found = TRUE; /* lookup only once */
				}
 				if (b4dialog && GTK_WIDGET_VISIBLE (b4dialog) && !g_ascii_strncasecmp (callsign, logcallsign, strlen (callsign)))
				{	/* worked b4 */
					b4[0] = g_strdup (logw->logname);
					for (j = 0; j < QSO_FIELDS; j++)
					{
						column = gtk_tree_view_get_column (GTK_TREE_VIEW(logw->treeview), j);
						if (gtk_tree_view_column_get_visible (column))
							gtk_tree_model_get (model, &iter, j, &b4[j+1], -1);
						else b4[j+1] = g_strdup("");
					}
					gtk_tree_store_prepend (GTK_TREE_STORE(b4model), &b4iter, NULL);
					for (j = 0; j < QSO_FIELDS + 1; j++)
					if (strlen(b4[j]) > 0) /* for responsiveness */
						gtk_tree_store_set (GTK_TREE_STORE(b4model), &b4iter, j, b4[j], -1);
				}
				valid = gtk_tree_model_iter_next (model, &iter);
			}
		}
	}

	if (b4dialog && GTK_WIDGET_VISIBLE (b4dialog))
	{
		for (j = 0; j < QSO_FIELDS + 1; j++)
			g_free (b4[j]);
		g_free (b4);
	}
}


void
on_callentry_changed (GtkEditable * editable, gpointer user_data)
{
	gchar *call, *errorstr;

	call = gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1);

	updatedxccframe (call);
	updateb4dialog_and_typeaheadfind (call);

	/* twpsk support, if no IPC ID yet, create one */
	if (state.shmid == -1)	
	{
		if ((state.shmid = shmget (KEY, SHMSIZE, IPC_CREAT | 0600)) < 0)
		{
			errorstr = g_strdup_printf (_("shmget failed: %s"), g_strerror (errno));
			update_statusbar (errorstr);
			g_free (errorstr);
		}
		if ((shareCall = (shmat (state.shmid, NULL, 0))) == (gchar *) - 1)
		{
			errorstr = g_strdup_printf (_("shmat failed: %s"), g_strerror (errno));
			update_statusbar (errorstr);
			g_free (errorstr);
			state.shmid = -1;
		}
	}
	if (state.shmid != -1)
		strncpy (shareCall, call, SHMSIZE);	/* put call in shm */
	g_free (call);
}


/* catch return key when band-optionmenu has focus and show the menu */
gboolean
on_bandoptionmenu_key_press_event (GtkWidget * widget,
					 GdkEventKey * event, gpointer user_data)
{
	GtkWidget *bandoptionmenu, *menu;

	switch (event->keyval)
		{
		case GDK_Return:
			bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
			menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (bandoptionmenu));
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 0, 0);
			return TRUE;
			break;
		}
	return FALSE;
}

/* catch return key when mode-optionmenu has focus and show the menu */
gboolean
on_modeoptionmenu_key_press_event (GtkWidget * widget,
					 GdkEventKey * event, gpointer user_data)
{
	GtkWidget *modeoptionmenu, *menu;
	switch (event->keyval)
		{
		case GDK_Return:
			modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
			menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (modeoptionmenu));
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 0, 0);
			return TRUE;
			break;
		}
	return FALSE;
}

#if WANT_HAMLIB

/* check for polling, if it is enabled use the state struct,
	 otherwise get frequency from the rig when there is no 
	 default entry */
void
on_mhzbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry, *bandoptionmenu;
	GString *digits = g_string_new ("");
	gchar *bandcopy;
	guint bandindex = 0;

	entry = lookup_widget (mainwindow, "bandentry");
	bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
	bandsplit = g_strsplit (preferences.bands, ",", 0);
	
	/* set optionmenu and entry to the default */
	if (g_ascii_strcasecmp (preferences.defaultmhz, "?"))
	{
		gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaultmhz);
		if (strlen (preferences.defaultmhz) <= 2)
			bandcopy = g_strdup (preferences.defaultmhz);
		else
			bandcopy = finddot (preferences.defaultmhz);
		for (;;)
		{
			if (!bandsplit[bandindex])
				break;
			if (g_ascii_strcasecmp (bandcopy, bandsplit[bandindex]) == 0)
			{
				gtk_option_menu_set_history (GTK_OPTION_MENU(bandoptionmenu), 
					bandindex);
				break;
			}
			bandindex++;
		}
		g_free (bandcopy);
	}
	else /* there is no default */
	{
		if (state.rigfrequency != 0)
			digits = convert_frequency ();
		else
			g_string_printf (digits, "UNKNOWN");
		/* set entry and optionmenu */
		gtk_entry_set_text (GTK_ENTRY (entry), digits->str);
		if (strlen (digits->str) <= 2)
			bandcopy = g_strdup (digits->str);
		else
			bandcopy = finddot (digits->str);
		for (;;)
			{
				if (!bandsplit[bandindex])
					break;
				if (g_ascii_strcasecmp (bandcopy, bandsplit[bandindex]) == 0)
					{
						gtk_option_menu_set_history (GTK_OPTION_MENU
							(bandoptionmenu), bandindex);
						break;
					}
				bandindex++;
			}
		g_free (bandcopy);
		g_string_free (digits, TRUE);
	}
}

void
on_modebutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry, *modeoptionmenu;
	gchar *mode;
	guint modeindex = 0;

	entry = lookup_widget (mainwindow, "modeentry");
	modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
	modesplit = g_strsplit (preferences.modes, ",", 0);

	/* use default */
	if (g_ascii_strcasecmp (preferences.defaultmode, "?"))
	{
		gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaultmode);
		for (;;)
		{
			if (!modesplit[modeindex])
				break;
			if (g_ascii_strcasecmp (preferences.defaultmode, modesplit[modeindex]) == 0)
			{
				gtk_option_menu_set_history (GTK_OPTION_MENU (modeoptionmenu), 
					modeindex);
				break;
			}
			modeindex++;
		}
	}
	else	/* NO default */
	{
		mode = rigmode (state.rigmode);
		gtk_entry_set_text (GTK_ENTRY (entry), mode);
		for (;;)
		{
			if (!modesplit[modeindex])
				break;
			if (g_ascii_strcasecmp (mode, modesplit[modeindex]) == 0)
			{
				gtk_option_menu_set_history (GTK_OPTION_MENU (modeoptionmenu), 
					modeindex);
				break;
			}
			modeindex++;
		}
		g_free (mode);
	}
}

void
on_rstbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry;

	entry = lookup_widget (mainwindow, "rstentry");
	if (g_ascii_strcasecmp (preferences.defaulttxrst, "?"))
		gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaulttxrst);
	else if (state.rigrst != 0)
		gtk_entry_set_text (GTK_ENTRY (entry), state.rigrst);
	else
		gtk_entry_set_text (GTK_ENTRY (entry), "UNKNOWN");
}

void
on_powerbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *rigpower;

	entry = lookup_widget (mainwindow, "powerentry");
	if (g_ascii_strcasecmp (preferences.defaultpower, "?"))
		gtk_entry_set_text (GTK_ENTRY (entry), preferences.defaultpower);
	else if (state.rigpower != 0)
		{
			rigpower = g_strdup_printf ("%d", state.rigpower * 256 / 100000);
			gtk_entry_set_text (GTK_ENTRY (entry), rigpower);
			g_free (rigpower);
		}
	else
		gtk_entry_set_text (GTK_ENTRY (entry), "UNKNOWN");
}
#endif



/* check locator and calculate distance and azimuth */
void
on_locatorentry_changed (GtkEditable * editable, gpointer user_data)
{
	gchar *locator;

	locator = gtk_editable_get_chars (GTK_EDITABLE (editable), 0, -1);
	updatelocatorframe (locator);
	g_free (locator);
}
