/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *  Copyright (C) 2003 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon-popup.h"
#include "galeon-embed-event.h"
#include "galeon-embed-shell.h"
#include "galeon-embed-utils.h"
#include "galeon-shell.h"
#include "prefs-strings.h"
#include "eel-gconf-extensions.h"
#include "galeon-config.h"
#include "gul-download.h"
#include "gul-string.h"
#include "galeon-embed-helper-list.h"
#include "galeon-marshal.h"
#include "bookmarks-gtk-menu.h"
#include "gul-gui.h"
#include "pixbuf-cache.h"

#include "popup-commands.h"

#include <string.h>
#include <glib/gi18n.h>
#include <gtk/gtkclipboard.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkstock.h>

#include <libgnome/gnome-exec.h>
#include <libgnomevfs/gnome-vfs-uri.h>


typedef enum
{
	POPUP_INPUT,
	POPUP_SELECTION,
	POPUP_DOCUMENT_NORMAL,
	POPUP_DOCUMENT_FRAME,
	POPUP_DOCUMENT_SIDEBAR,
	POPUP_DOCUMENT_SIDEBAR_FRAME,
	POPUP_LINK,
	POPUP_EMAIL_LINK,
	POPUP_IMAGE,
	POPUP_IMAGE_LINK,
	POPUP_IMAGE_EMAIL_LINK
} PopupType;

struct GaleonPopupPrivate
{
	GaleonWindow *window;
	GtkWidget *menu;
	GSList *bookmark_menus;
	GaleonEmbedHelperList *helper_list;
	GaleonEmbedEvent *event;
	GaleonEmbed *embed;
	EmbedEventContext context;
	GtkUIManager *merge;
	GtkActionGroup *action_group;
	PopupType popup_type;
};

static void
galeon_popup_class_init (GaleonPopupClass *klass);
static void
galeon_popup_init (GaleonPopup *gep);
static void
galeon_popup_finalize (GObject *object);
static void
galeon_popup_set_window(GaleonPopup *p, 
			    GaleonWindow *window);
static void
galeon_popup_show_impl(GaleonPopup *pp,
		       GaleonEmbed *embed);
static void 
popup_copy_location_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_copy_email_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_copy_link_location_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_download_link_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_save_image_as_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_set_image_as_background_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_copy_image_location_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_save_page_as_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_save_background_as_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_open_frame_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void 
popup_reload_frame_cmd(GtkAction *action, 
			GaleonPopup *popup);

static void 
popup_open_image_cmd(GtkAction *action, 
			GaleonPopup *popup);
static void
popup_copy_to_clipboard (GaleonPopup *popup, const char *text);

const char *
galeon_popup_get_image_openwith_popup_path (GaleonPopup *p);
const char *
galeon_popup_get_doc_openwith_popup_path (GaleonPopup *p);

static GObjectClass *parent_class = NULL;

#define ACTION_ITEM(name) gtk_action_group_get_action(p->priv->action_group, name)

#define INPUT_POPUP_PATH                  "/GaleonEmbedInputPopup"
#define SELECTION_POPUP_PATH              "/GaleonEmbedSelectionPopup"
#define DOCUMENT_POPUP_NORMAL_PATH        "/GaleonEmbedDocumentNormalPopup"
#define DOCUMENT_POPUP_FRAME_PATH         "/GaleonEmbedDocumentFramePopup"
#define DOCUMENT_POPUP_SIDEBAR_PATH       "/GaleonEmbedDocumentSidebarPopup"
#define DOCUMENT_POPUP_SIDEBAR_FRAME_PATH "/GaleonEmbedDocumentSidebarFramePopup"
#define LINK_POPUP_PATH                   "/GaleonEmbedLinkPopup"
#define EMAIL_LINK_POPUP_PATH             "/GaleonEmbedEmailLinkPopup"
#define IMAGE_POPUP_PATH                  "/GaleonEmbedImagePopup"
#define IMAGE_LINK_POPUP_PATH             "/GaleonEmbedImageLinkPopup"
#define IMAGE_EMAIL_LINK_POPUP_PATH       "/GaleonEmbedImageEmailLinkPopup"

#define OPEN_WITH_MENU_ACTION "DPOpenWithMenuAction"
#define OPEN_IMAGE_WITH_MENU_ACTION "EPOpenImageWithMenuAction"
#define OPEN_IMAGE_ACTION "EPOpenImageAction"
#define SAVE_BACKGROUND_AS_ACTION "DPSaveBackgroundAsAction"

/*This command is from src/popup-commands.c but manipulating item label
 *text must be done here
 */
#define BLOCK_IMAGE_SITE_ACTION "EPBlockImageSiteAction"
#define ALLOW_IMAGE_SITE_ACTION "EPAllowImageSiteAction"

#define OPEN_IN_NEW_TAB_ACTION    "EPOpenInNewTabAction"
#define OPEN_IN_NEW_WINDOW_ACTION "EPOpenInNewWindowAction"
#define VIEW_SOURCE_ACTION        "EPViewSourceAction"

static GtkActionEntry entries[] = {
	{ "PopupToplevelAction", NULL, (""), NULL, NULL, NULL },
	{ OPEN_IN_NEW_WINDOW_ACTION, GTK_STOCK_OPEN, N_("Open in New Window"), NULL,
	  N_("Open the link in a new Galeon window"), G_CALLBACK(popup_cmd_new_window) },
	{ OPEN_IN_NEW_TAB_ACTION, NULL,  N_("Open in New Tab"), NULL,
	  N_("Open the link in a new tab in this window"), G_CALLBACK(popup_cmd_new_tab) },
	{ VIEW_SOURCE_ACTION, STOCK_VIEW_SOURCE, N_("_View Source"), "<control>u",
	  N_("Show the source of the current page"), G_CALLBACK(popup_cmd_view_source) },
	{ "EPAddBookmarkAction", STOCK_ADD_BOOKMARK,  N_("Add Bookmark..."), NULL,
	  N_("Add a bookmark for the link to the default folder"), G_CALLBACK(popup_cmd_add_bookmark) },
	{ "EPOpenImageInNewWindowAction", NULL,  N_("Open Image in New Window"), NULL,
	  N_("Open the image in a new Galeon window"), G_CALLBACK(popup_cmd_image_in_new_window) },
	{ "EPOpenImageInNewTabAction",	NULL, N_("Open Image in New Tab"), NULL,
	  N_("Open the image in a new tab in this window"), G_CALLBACK(popup_cmd_image_in_new_tab) },
	{ "DPOpenFrameInNewWindowAction", GTK_STOCK_OPEN,  N_("Open Frame in New Window"), NULL,
	  N_("Open the frame in a new Galeon window"), G_CALLBACK(popup_cmd_frame_in_new_window) },
	{ "DPOpenFrameInNewTabAction",	NULL, N_("Open Frame in New Tab"), NULL,
	  N_("Open the frame in a new tab in this window"), G_CALLBACK(popup_cmd_frame_in_new_tab) },
	{ "DPAddFrameBookmarkAction", STOCK_ADD_BOOKMARK,  N_("Add Bookmark for Frame..."), NULL,
	  N_("Add a bookmark for this frame to the default folder"), G_CALLBACK(popup_cmd_add_frame_bookmark) },
	{ OPEN_WITH_MENU_ACTION, NULL, N_("Open With"), NULL, NULL, NULL },
	{ OPEN_IMAGE_WITH_MENU_ACTION, NULL, N_("Open Image With"), NULL, NULL, NULL },

	{ "EPCopyLinkLocationAction", GTK_STOCK_COPY, N_("Copy Link Address"), NULL,
	  N_("Copy the link address to the clipboard"), G_CALLBACK(popup_copy_link_location_cmd) },
	{ "EPDownloadLinkAction", NULL, N_("Download Link"), NULL,
	  N_("Download the link"), G_CALLBACK(popup_download_link_cmd) },
	{ "EPOpenImageAction", GTK_STOCK_OPEN, N_("Open Image"), NULL,
	  N_("Open the image in this window"), G_CALLBACK(popup_open_image_cmd) },
	{ "EPSaveImageAsAction", GTK_STOCK_SAVE_AS, N_("Save Image As..."), NULL,
	  N_("Save the image"), G_CALLBACK(popup_save_image_as_cmd) },
	{ "EPSetImageAsBackgroundAction", NULL, N_("Use Image As Background"), NULL,
	  N_("Use the image as the desktop background image"), G_CALLBACK(popup_set_image_as_background_cmd) },
	{ "EPCopyImageLocationAction", GTK_STOCK_COPY, N_("Copy Image Address"), NULL,
	  N_("Copy the image address to the clipboard"), G_CALLBACK(popup_copy_image_location_cmd) },
	{ "EPCopyEmailAction", GTK_STOCK_COPY, N_("Copy Email Address"), NULL,
	  N_("Copy the Email address to the clipboard"), G_CALLBACK(popup_copy_email_cmd) },
	{ "DPCopyLocationAction", NULL, N_("Copy Page Address"), NULL,
	  N_("Copy the current page address to the clipboard"), G_CALLBACK(popup_copy_location_cmd) },
	{ "DPSavePageAsAction", GTK_STOCK_SAVE_AS, N_("Save Page As..."), NULL,
	  N_("Save the page"), G_CALLBACK(popup_save_page_as_cmd) },
	{ "DPSaveBackgroundAsAction", NULL, N_("Save Background As..."), NULL,
	  N_("Save the page's background"), G_CALLBACK(popup_save_background_as_cmd) },
	{ "DPOpenFrameAction", NULL, N_("Open Frame"), NULL,
	  N_("Open the frame in this window"), G_CALLBACK(popup_open_frame_cmd) },
	{ "DPReloadFrameAction", GTK_STOCK_REFRESH, N_("Reload Frame"), NULL,
	  N_("Display the latest content of the frame"), G_CALLBACK(popup_reload_frame_cmd) },

	{ "EPBlockImageSiteAction", NULL,  N_("Block Images From This Site"), NULL,
          N_("Block images from this image's originating host"), G_CALLBACK(popup_cmd_image_block_site) },
	{ "EPAllowImageSiteAction", NULL,  N_("Allow Images From This Site"), NULL,
          N_("Allow images from this image's originating host"), G_CALLBACK(popup_cmd_image_block_site) }
};
static guint n_entries = G_N_ELEMENTS(entries);

GType
galeon_popup_get_type (void)
{
       static GType galeon_popup_type = 0;

        if (galeon_popup_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (GaleonPopupClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) galeon_popup_class_init,
                        NULL, /* class_finalize */
                        NULL, /* class_data */
                        sizeof (GaleonPopup),
                        0,    /* n_preallocs */
                        (GInstanceInitFunc) galeon_popup_init
                };

		
                galeon_popup_type = g_type_register_static (G_TYPE_OBJECT,
								  "GaleonPopup",
								  &our_info, 0);
        }

        return galeon_popup_type;
}

static void
galeon_popup_class_init (GaleonPopupClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = (GObjectClass *) g_type_class_peek_parent (klass);

        object_class->finalize = galeon_popup_finalize;

	klass->show = galeon_popup_show_impl;
}

static void
galeon_popup_init (GaleonPopup *gep)
{
	GtkAction *action;

        gep->priv = g_new0 (GaleonPopupPrivate, 1);
	gep->priv->embed = NULL;
	gep->priv->event = NULL;
	gep->priv->helper_list = galeon_embed_helper_list_new ();
	gep->priv->window = NULL;
	gep->priv->menu = NULL;
	gep->priv->bookmark_menus = NULL;

	gep->priv->action_group = gtk_action_group_new("EmbedPopupActions");
	gtk_action_group_set_translation_domain (gep->priv->action_group, NULL);
	gtk_action_group_add_actions(gep->priv->action_group,
				     entries, n_entries, gep);

	action = gtk_action_group_get_action (gep->priv->action_group, "EPOpenImageWithMenuAction");
	g_object_set (action, "hide_if_empty", FALSE, NULL);

	action = gtk_action_group_get_action (gep->priv->action_group, "DPOpenWithMenuAction");
	g_object_set (action, "hide_if_empty", FALSE, NULL);
}

static void
galeon_popup_finalize (GObject *object)
{
	GaleonPopup *gep;
	GSList *li;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_GALEON_POPUP (object));

        gep = GALEON_POPUP (object);

        g_return_if_fail (gep->priv != NULL);

	for (li = gep->priv->bookmark_menus; li; li = li->next)
	{
		g_object_unref (li->data);
	}
	g_slist_free (gep->priv->bookmark_menus);

	if (gep->priv->event)
	{
		g_object_unref (G_OBJECT (gep->priv->event));
	}	
	
	if (gep->priv->helper_list)
	{
		g_object_unref (G_OBJECT (gep->priv->helper_list));
	}

	if (gep->priv->action_group)
	{
		g_object_unref(G_OBJECT(gep->priv->action_group));
	}
	
        g_free (gep->priv);
	
        G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
galeon_popup_set_window(GaleonPopup *p, 
			GaleonWindow *window)
{
	p->priv->window = window;
	/* crispin: urgh */
	g_object_set_data (G_OBJECT(p), "GaleonWindow", window);
}

GaleonPopup *
galeon_popup_new(GaleonWindow *window)
{
	GaleonPopup *p;
	
        p = GALEON_POPUP (g_object_new(GALEON_POPUP_TYPE, NULL));
	
        g_return_val_if_fail(p->priv != NULL, NULL);
	
	galeon_popup_set_window (p, window);
        return p;
}

static void
setup_image_openwith (GaleonPopup *p)
{
	const gchar *path = galeon_popup_get_image_openwith_popup_path (p);
	GaleonEmbedEvent *info;
	const char *url;
	const GValue *value;
	GaleonEmbedHelperList *hl;
	GtkWidget *menuitem;
	GtkWidget *menu;
	g_return_if_fail (path != NULL);

	menuitem = gtk_ui_manager_get_widget(p->priv->window->merge, path);
	menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (menuitem));
	info = galeon_popup_get_event (p);
	galeon_embed_event_get_property (info, "image", &value);
	url = g_value_get_string (value);
	hl = p->priv->helper_list;
	if (!galeon_embed_helper_list_set_uri (hl, url))
	{
		/* it must be an image, so let's try with png */
		galeon_embed_helper_list_add_mime_type (hl, "image/png");
	}

	gul_gui_remove_all_children (GTK_CONTAINER (menu));
	galeon_embed_helper_list_add_to_gtk_menu (hl, GTK_MENU_SHELL (menu));
}

static void 
setup_element_menu (GaleonPopup *p)
{
	gboolean is_image, is_sidebar, is_link;
	
	is_image = p->priv->context & EMBED_CONTEXT_IMAGE;
	is_sidebar = p->priv->context & EMBED_CONTEXT_SIDEBAR;
	is_link = p->priv->context & EMBED_CONTEXT_LINK;

	/* Hide items that doesnt make sense to show in the sidebar */
	g_object_set(G_OBJECT(ACTION_ITEM(OPEN_IMAGE_ACTION)),
		     "visible", !is_sidebar, NULL);

	if (is_image)
	{
		GaleonEmbedEvent *info;
		const GValue *value;
		char *open_image_str, *block_host_str,
		     *esc_file = NULL, *file = NULL;
		const char *host = NULL;
		GnomeVFSURI *uri = NULL;
		const char *show_action, *hide_action;
		PermissionActionType perm_action;

		info = galeon_popup_get_event(p);

		galeon_embed_event_get_property(info, "image", &value);

		uri = gnome_vfs_uri_new(g_value_get_string(value));

		if (uri)
		{
			host = gnome_vfs_uri_get_host_name(uri);
			file = gnome_vfs_uri_extract_short_name(uri);
		}

		if (!host)
			host = _("This Site");

		if (file)
		{
			esc_file = gul_string_double_underscores (file);
			g_free(file);
		}

		galeon_embed_shell_test_permission(embed_shell,
						   g_value_get_string(value),
						   IMAGES_PERMISSION,
						   &perm_action);

		if (perm_action == DENY_ACTION)
		{
			block_host_str = g_strdup_printf(_("Allow Images From %s"),
							 host);
			show_action = ALLOW_IMAGE_SITE_ACTION;
			hide_action = BLOCK_IMAGE_SITE_ACTION;
		}
		else
		{
			block_host_str = g_strdup_printf(_("Block Images From %s"),
							 host);
			hide_action = ALLOW_IMAGE_SITE_ACTION;
			show_action = BLOCK_IMAGE_SITE_ACTION;
		}

		g_object_set(G_OBJECT(ACTION_ITEM(show_action)),
			     "label", block_host_str, NULL);
		g_object_set(G_OBJECT(ACTION_ITEM(show_action)),
			     "visible", TRUE, NULL);
		g_object_set(G_OBJECT(ACTION_ITEM(hide_action)),
			     "visible", FALSE, NULL);

		open_image_str = g_strdup_printf(_("Open Image (%s)"),
						 esc_file);

		g_object_set(G_OBJECT(ACTION_ITEM(OPEN_IMAGE_ACTION)),
			     "label", open_image_str, NULL);

		if (esc_file)
			g_free(esc_file);

		if (uri)
			gnome_vfs_uri_unref(uri);

		g_free(block_host_str);
		g_free(open_image_str);

		setup_image_openwith (p);
	}

	if (is_link)
	{
		gboolean is_web_link = 
			galeon_embed_event_has_property (p->priv->event, 
							 "link-can-open-in-new-tab");

		g_object_set(G_OBJECT(ACTION_ITEM(OPEN_IN_NEW_TAB_ACTION)),
			     "sensitive", is_web_link, NULL);
		g_object_set(G_OBJECT(ACTION_ITEM(OPEN_IN_NEW_WINDOW_ACTION)),
			     "sensitive", is_web_link, NULL);
	}
}

static void
setup_doc_openwith (GaleonPopup *popup)
{
	const gchar *path = galeon_popup_get_doc_openwith_popup_path (popup);
	char *location;
	GaleonEmbedHelperList *hl = popup->priv->helper_list;
	GtkWidget *menuitem;
	GtkWidget *menu;

	g_return_if_fail (popup->priv->embed != NULL);
	g_return_if_fail (path != NULL);

	menuitem = gtk_ui_manager_get_widget(popup->priv->window->merge, path);
	menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (menuitem));

	galeon_embed_get_location (popup->priv->embed, FALSE, 
				   FALSE, &location);
	if (!galeon_embed_helper_list_set_uri (hl, location))
	{
		/* well, it's very probably html */
		galeon_embed_helper_list_add_mime_type (hl, "text/html");
	}
	
	g_free (location);
	
	galeon_embed_helper_list_add_mime_type (hl, "text/plain");
	
	gul_gui_remove_all_children (GTK_CONTAINER (menu));
	galeon_embed_helper_list_add_to_gtk_menu (hl, GTK_MENU_SHELL (menu));
}

static void
setup_document_menu (GaleonPopup *p)
{	
	gboolean has_background, can_view_source;
	GaleonEmbed *embed;

	embed = galeon_popup_get_embed (p);
	can_view_source = (galeon_embed_can_view_source (embed) == G_OK);
	g_object_set(G_OBJECT(ACTION_ITEM(VIEW_SOURCE_ACTION)),
		     "sensitive", can_view_source, NULL);
	
	has_background = galeon_embed_event_has_property (p->priv->event,
							  "background_image");
	g_object_set(G_OBJECT(ACTION_ITEM(SAVE_BACKGROUND_AS_ACTION)),
		     "visible", has_background, NULL);

	setup_doc_openwith (p);
}

void 
galeon_popup_set_event(GaleonPopup *p,
		       GaleonEmbedEvent *event)
{
	EmbedEventContext context;

	if (p->priv->event)
	{
		g_object_unref (G_OBJECT (p->priv->event));
	}

	galeon_embed_event_get_context (event, &context); 
	
	p->priv->context = context;

	p->priv->event = event;
	g_object_ref (G_OBJECT(event));
}

static void
galeon_popup_setup_menu (GaleonPopup *p)
{
	const GValue *value;
	gboolean is_framed;

	g_return_if_fail (IS_GALEON_EMBED_EVENT (p->priv->event));
	g_return_if_fail (IS_GALEON_EMBED (p->priv->embed));

	galeon_embed_event_get_property (p->priv->event, 
					 "framed_page", &value);
	is_framed = g_value_get_int (value);

	if (p->priv->context & EMBED_CONTEXT_EMAIL_LINK &&
	    p->priv->context & EMBED_CONTEXT_IMAGE)
	{
		p->priv->popup_type = POPUP_IMAGE_EMAIL_LINK;
		setup_element_menu(p);
	}
	else if (p->priv->context & EMBED_CONTEXT_EMAIL_LINK)
	{
		setup_element_menu(p);
		p->priv->popup_type = POPUP_EMAIL_LINK;
	}
	else if (p->priv->context & EMBED_CONTEXT_LINK &&
		 p->priv->context & EMBED_CONTEXT_IMAGE)
	{
		p->priv->popup_type = POPUP_IMAGE_LINK;
		setup_element_menu(p);
	}
	else if (p->priv->context & EMBED_CONTEXT_IMAGE)
	{
		p->priv->popup_type = POPUP_IMAGE;
		setup_element_menu(p);
	}
	else if (p->priv->context & EMBED_CONTEXT_LINK)
	{
		p->priv->popup_type = POPUP_LINK;
		setup_element_menu(p);
	}
	else if (p->priv->context & EMBED_CONTEXT_INPUT)
	{
		p->priv->popup_type = POPUP_INPUT;
	}
	else if (galeon_embed_event_has_property (p->priv->event, "selection"))
	{
		/* TODO here we could set up some additional menu items that
		 * use the selected text, like smart bookmarks for example */
		p->priv->popup_type = POPUP_SELECTION;
	}
	else if ((p->priv->context & (EMBED_CONTEXT_SIDEBAR)) &&
		 is_framed)
	{
		p->priv->popup_type = POPUP_DOCUMENT_SIDEBAR_FRAME;
		setup_document_menu(p);
	}
	else if (is_framed)
	{
		p->priv->popup_type = POPUP_DOCUMENT_FRAME;
		setup_document_menu(p);
	}
	else if (p->priv->context & (EMBED_CONTEXT_SIDEBAR))
	{
		p->priv->popup_type = POPUP_DOCUMENT_SIDEBAR;
		setup_document_menu(p);
	}
	else
	{
		p->priv->popup_type = POPUP_DOCUMENT_NORMAL;
		setup_document_menu (p);
	}
}

void
galeon_popup_set_embed(GaleonPopup *p, 
		       GaleonEmbed *e)
{
	p->priv->embed = e;
	galeon_embed_helper_list_set_embed (p->priv->helper_list, e);
}

GaleonEmbed *
galeon_popup_get_embed (GaleonPopup *p)
{
	return p->priv->embed;
}

void
galeon_popup_show (GaleonPopup *p,
			 GaleonEmbed *embed)
{
	GaleonPopupClass *klass = GALEON_POPUP_GET_CLASS (p);
	return klass->show (p, embed);

}

void 
galeon_popup_connect_verbs(GaleonPopup *p,
			   GtkUIManager *merge)
{
	p->priv->merge = merge;
	gtk_ui_manager_insert_action_group(merge, p->priv->action_group, 0);
}

GaleonEmbedEvent *
galeon_popup_get_event (GaleonPopup *p)
{
	g_return_val_if_fail (IS_GALEON_POPUP (p), NULL);

	return p->priv->event;
}

static void 
popup_copy_location_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	char *location;
	galeon_embed_get_location (popup->priv->embed, FALSE, 
				   FALSE, &location);
	popup_copy_to_clipboard (popup, location);
	g_free (location);
}

static void 
popup_copy_email_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_popup_get_event (popup);
	galeon_embed_event_get_property (info, "email", &value);
	location = g_value_get_string (value);
	popup_copy_to_clipboard (popup, location);
}

static void 
popup_copy_link_location_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_popup_get_event (popup);
	galeon_embed_event_get_property (info, "link", &value);
	location = g_value_get_string (value);
	popup_copy_to_clipboard (popup, location);
}

const char *
galeon_popup_get_image_openwith_popup_path (GaleonPopup *p)
{
	const char *result = NULL;

	switch (p->priv->popup_type)
	{
		case POPUP_INPUT:
			result = NULL;
			break;
		case POPUP_SELECTION:
			result = NULL;
			break;
		case POPUP_LINK:
			result = NULL;
			break;
		case POPUP_EMAIL_LINK:
			result = NULL;
			break;
		case POPUP_IMAGE:
			result = IMAGE_POPUP_PATH "/EPOpenImageWith";
			break;
		case POPUP_IMAGE_LINK:
			result = IMAGE_LINK_POPUP_PATH  "/EPOpenImageWith";
			break;
		case POPUP_IMAGE_EMAIL_LINK:
			result = IMAGE_EMAIL_LINK_POPUP_PATH  "/EPOpenImageWith";
			break;
		case POPUP_DOCUMENT_NORMAL:
			result = NULL;
			break;
		case POPUP_DOCUMENT_FRAME:
			result = NULL;
			break;
		case POPUP_DOCUMENT_SIDEBAR:
			result = NULL;
			break;
		case POPUP_DOCUMENT_SIDEBAR_FRAME:
			result = NULL;
			break;
	}
	
	return result;
}

const char *
galeon_popup_get_doc_openwith_popup_path (GaleonPopup *p)
{
	const char *result = NULL;

	switch (p->priv->popup_type)
	{
		case POPUP_INPUT:
			result = NULL;
			break;
		case POPUP_SELECTION:
			result = NULL;
			break;
		case POPUP_LINK:
			result = NULL;
			break;
		case POPUP_EMAIL_LINK:
			result = NULL;
			break;
		case POPUP_IMAGE:
			result = NULL;
			break;
		case POPUP_IMAGE_LINK:
			result = NULL;
			break;
		case POPUP_IMAGE_EMAIL_LINK:
			result = NULL;
			break;
		case POPUP_DOCUMENT_NORMAL:
			result = DOCUMENT_POPUP_NORMAL_PATH "/DPOpenWith";
			break;
		case POPUP_DOCUMENT_FRAME:
			result = DOCUMENT_POPUP_FRAME_PATH "/DPOpenWith";
			break;
		case POPUP_DOCUMENT_SIDEBAR:
			result = DOCUMENT_POPUP_SIDEBAR_PATH "/DPOpenWith";
			break;
		case POPUP_DOCUMENT_SIDEBAR_FRAME:
			result = DOCUMENT_POPUP_SIDEBAR_FRAME_PATH "/DPOpenWith";
			break;
	}
	
	return result;
}

const char *
galeon_popup_get_popup_path (GaleonPopup *p)
{
	const char *result = NULL;

	switch (p->priv->popup_type)
	{
		case POPUP_INPUT:
			result = INPUT_POPUP_PATH;
			break;
		case POPUP_SELECTION:
			result = SELECTION_POPUP_PATH;
			break;
		case POPUP_LINK:
			result = LINK_POPUP_PATH;
			break;
		case POPUP_EMAIL_LINK:
			result = EMAIL_LINK_POPUP_PATH;
			break;
		case POPUP_IMAGE:
			result = IMAGE_POPUP_PATH;
			break;
		case POPUP_IMAGE_LINK:
			result = IMAGE_LINK_POPUP_PATH;
			break;
		case POPUP_IMAGE_EMAIL_LINK:
			result = IMAGE_EMAIL_LINK_POPUP_PATH;
			break;
		case POPUP_DOCUMENT_NORMAL:
			result = DOCUMENT_POPUP_NORMAL_PATH;
			break;
		case POPUP_DOCUMENT_FRAME:
			result = DOCUMENT_POPUP_FRAME_PATH;
			break;
		case POPUP_DOCUMENT_SIDEBAR:
			result = DOCUMENT_POPUP_SIDEBAR_PATH;
			break;
		case POPUP_DOCUMENT_SIDEBAR_FRAME:
			result = DOCUMENT_POPUP_SIDEBAR_FRAME_PATH;
			break;
	}
	
	return result;
}

/* commands */

static void 
popup_download_link_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *event = galeon_popup_get_event(popup);
	GaleonEmbed *embed = galeon_popup_get_embed(popup);

	galeon_embed_utils_download_event_property(embed, event,
						   FALSE, NULL,
						   "link");
}

static void 
popup_save_image_as_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *event = galeon_popup_get_event(popup);
	GaleonEmbed *embed = galeon_popup_get_embed(popup);
	
	galeon_embed_utils_save_event_property (embed,event,
						TRUE, /* always_ask_dir */
						FALSE, /* show_progress */
						_("Save Image As..."),
						"image");
}

#define CONF_DESKTOP_BG_PICTURE "/desktop/gnome/background/picture_filename"
#define CONF_DESKTOP_BG_TYPE "/desktop/gnome/background/picture_options"

static void
background_download_completed (GaleonEmbedPersist *persist,
			       gpointer data)
{
	const char *bg;
	char *type;
	
	galeon_embed_persist_get_dest (persist, &bg);
	eel_gconf_set_string (CONF_DESKTOP_BG_PICTURE, bg);

	type = eel_gconf_get_string (CONF_DESKTOP_BG_TYPE);
	if (type && strcmp (type, "none") == 0)
	{
		eel_gconf_set_string (CONF_DESKTOP_BG_TYPE,
				      "wallpaper");
	}

	g_free (type);
}

static void 
popup_set_image_as_background_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *info;
	const char *location;
	char *dest, *base;
	const GValue *value;
	GaleonEmbedPersist *persist;
	
	info = galeon_popup_get_event (popup);
	galeon_embed_event_get_property (info, "image", &value);
	location = g_value_get_string (value);

	persist = galeon_embed_persist_new (popup->priv->embed);
	
	base = g_path_get_basename (location);
	dest = g_build_filename (g_get_home_dir (), 
				 GALEON_DIR, 
				 base, NULL);
	
	galeon_embed_persist_set_source (persist, location);
	galeon_embed_persist_set_dest (persist, dest);

	g_signal_connect (persist, "completed",
			  G_CALLBACK (background_download_completed),
			  NULL);

	galeon_embed_persist_save (persist);
	g_object_unref (persist);
	
	g_free (dest);
	g_free (base);
}

static void 
popup_copy_image_location_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_popup_get_event (popup);
	galeon_embed_event_get_property (info, "image", &value);
	location = g_value_get_string (value);
	popup_copy_to_clipboard (popup, location);
}

static void 
popup_save_page_as_cmd(GtkAction *action, 
		       GaleonPopup *popup)
{
	GtkWidget *window;
	GaleonEmbedPersist *persist;

	window = gtk_widget_get_toplevel (GTK_WIDGET (popup->priv->embed));
	
	persist = galeon_embed_persist_new (popup->priv->embed);
	galeon_embed_persist_set_flags (persist, EMBED_PERSIST_MAINDOC
					| EMBED_PERSIST_SAVE_CONTENT
					| EMBED_PERSIST_ASK_DESTINATION);

	galeon_embed_persist_set_fc_parent (persist, window);
	galeon_embed_persist_set_fc_title (persist, _("Save Page As..."));
	galeon_embed_persist_save (persist);

	g_object_unref (persist);
}

static void 
popup_save_background_as_cmd(GtkAction *action, 
			     GaleonPopup *popup)
{
	GaleonEmbedEvent *event = galeon_popup_get_event(popup);
	GaleonEmbed *embed = galeon_popup_get_embed(popup);
	
	galeon_embed_utils_save_event_property(embed,event,
					       TRUE, /* always_ask_dest */
					       FALSE, /* show_progress */
					       _("Save Background As..."),
					       "background_image");
}

static void 
popup_open_frame_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	char *location;
	
	galeon_embed_get_location (popup->priv->embed,
				   FALSE, FALSE, &location);

	galeon_embed_load_url (popup->priv->embed, location);
}

static void
popup_reload_frame_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	galeon_embed_reload(popup->priv->embed,
			    EMBED_RELOAD_FRAME);
}

static void 
popup_open_image_cmd(GtkAction *action, 
			GaleonPopup *popup)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_popup_get_event (popup);
	galeon_embed_event_get_property (info, "image", &value);
	location = g_value_get_string (value);

	galeon_embed_load_url (popup->priv->embed, location);	
}

static void
popup_copy_to_clipboard (GaleonPopup *popup, const char *text)
{
	gtk_clipboard_set_text (gtk_clipboard_get (GDK_NONE),
				text, -1);
	gtk_clipboard_set_text (gtk_clipboard_get (GDK_SELECTION_PRIMARY),
				text, -1);
}

static void
galeon_window_bookmark_activated_cb (GObject *sender,
				     GbBookmarkEventActivated *ev,
				     GaleonWindow *w)
{
	galeon_window_bookmark_activate (w, ev);
}

static void
popup_menu_at_coords(GtkMenu *menu, gint *x, gint *y, gboolean *push_in,
		     gpointer user_data)
{
	GaleonEmbedEvent *event = user_data;

	*x = event->x;
	*y = event->y;
	*push_in = FALSE;
}

static void
galeon_popup_show_impl(GaleonPopup *p,
		       GaleonEmbed *embed)
{
	GtkMenuPositionFunc posfunc = NULL;
	GaleonEmbedEvent *info = NULL;
	GdkEvent *event;
	guint button;
	GbBookmarkSet *bs = galeon_shell_get_bookmark_set (galeon_shell);

	galeon_popup_set_embed(p, embed);
	galeon_popup_setup_menu(p);
	
	/* If and only if the popup is initiated by a mouse button press should
	 * we pass the button to gtk_menu_popup, otherwise items in _sub_menus
	 * don't activate properly. */
	button = 0;
	event = gtk_get_current_event();
	if (event != NULL)
	{
		if (event->type == GDK_BUTTON_PRESS
		    || event->type == GDK_2BUTTON_PRESS
		    || event->type == GDK_3BUTTON_PRESS)
		{
			button = event->button.button;
		}
		else if (event->type == GDK_KEY_PRESS)
		{
			info = galeon_popup_get_event(p);
			if (info != NULL && info->keycode != 0)
			{
				posfunc = popup_menu_at_coords;
			}
		}
		gdk_event_free(event);
	}

	p->priv->menu = gtk_ui_manager_get_widget(p->priv->window->merge,
						  galeon_popup_get_popup_path(GALEON_POPUP(p)));
	g_return_if_fail(p->priv->menu != NULL);

	if (g_object_get_data (G_OBJECT (p->priv->menu), "GbGtkMenu") == NULL)
	{
		GbGtkMenu *ggm = gb_gtk_menu_new_context_only (bs, GTK_MENU_SHELL (p->priv->menu));
		gb_gtk_menu_set_location_source (ggm, GB_LOCATION_SOURCE (p->priv->window));
		gb_gtk_menu_fill_children_submenus (ggm);
		g_signal_connect (ggm, "bookmark-activated", 
				  G_CALLBACK (galeon_window_bookmark_activated_cb), p->priv->window);
		g_object_set_data (G_OBJECT (p->priv->menu), "GbGtkMenu", ggm);
		p->priv->bookmark_menus = g_slist_prepend (p->priv->bookmark_menus, ggm);
	}

	gtk_menu_popup(GTK_MENU(p->priv->menu),
	 	       NULL, NULL, posfunc, info,
		       button, gtk_get_current_event_time());
}

