"""
SciPy: A scientific computing package for Python
================================================

Documentation is available in the docstrings and
online at https://docs.scipy.org.

Contents
--------
SciPy imports all the functions from the NumPy namespace, and in
addition provides:

Subpackages
-----------
Using any of these subpackages requires an explicit import.  For example,
``import scipy.cluster``.

::

 cluster                      --- Vector Quantization / Kmeans
 fftpack                      --- Discrete Fourier Transform algorithms
 integrate                    --- Integration routines
 interpolate                  --- Interpolation Tools
 io                           --- Data input and output
 linalg                       --- Linear algebra routines
 linalg.blas                  --- Wrappers to BLAS library
 linalg.lapack                --- Wrappers to LAPACK library
 misc                         --- Various utilities that don't have
                                  another home.
 ndimage                      --- n-dimensional image package
 odr                          --- Orthogonal Distance Regression
 optimize                     --- Optimization Tools
 signal                       --- Signal Processing Tools
 sparse                       --- Sparse Matrices
 sparse.linalg                --- Sparse Linear Algebra
 sparse.linalg.dsolve         --- Linear Solvers
 sparse.linalg.dsolve.umfpack --- :Interface to the UMFPACK library:
                                  Conjugate Gradient Method (LOBPCG)
 sparse.linalg.eigen          --- Sparse Eigenvalue Solvers
 sparse.linalg.eigen.lobpcg   --- Locally Optimal Block Preconditioned
                                  Conjugate Gradient Method (LOBPCG)
 spatial                      --- Spatial data structures and algorithms
 special                      --- Special functions
 stats                        --- Statistical Functions

Utility tools
-------------
::

 test              --- Run scipy unittests
 show_config       --- Show scipy build configuration
 show_numpy_config --- Show numpy build configuration
 __version__       --- Scipy version string
 __numpy_version__ --- Numpy version string

"""
from __future__ import division, print_function, absolute_import

__all__ = ['test']

from numpy import show_config as show_numpy_config
if show_numpy_config is None:
    raise ImportError("Cannot import scipy when running from numpy source directory.")
from numpy import __version__ as __numpy_version__

# Import numpy symbols to scipy name space
import numpy as _num
linalg = None
from numpy import *
from numpy.random import rand, randn
from numpy.fft import fft, ifft
from numpy.lib.scimath import *

__all__ += _num.__all__
__all__ += ['randn', 'rand', 'fft', 'ifft']

del _num
# Remove the linalg imported from numpy so that the scipy.linalg package can be
# imported.
del linalg
__all__.remove('linalg')

# We first need to detect if we're being called as part of the scipy
# setup procedure itself in a reliable manner.
try:
    __SCIPY_SETUP__
except NameError:
    __SCIPY_SETUP__ = False


if __SCIPY_SETUP__:
    import sys as _sys
    _sys.stderr.write('Running from scipy source directory.\n')
    del _sys
else:
    try:
        from scipy.__config__ import show as show_config
    except ImportError:
        msg = """Error importing scipy: you cannot import scipy while
        being in scipy source directory; please exit the scipy source
        tree first, and relaunch your python intepreter."""
        raise ImportError(msg)

    from scipy.version import version as __version__
    from scipy._lib._version import NumpyVersion as _NumpyVersion
    if _NumpyVersion(__numpy_version__) < '1.8.2':
        import warnings
        warnings.warn("Numpy 1.8.2 or above is recommended for this version of "
                      "scipy (detected version %s)" % __numpy_version__,
                      UserWarning)

    del _NumpyVersion

    from scipy._lib._ccallback import LowLevelCallable

    from numpy.testing import Tester

    def test(*a, **kw):
        # Nose never recurses into directories with underscores prefix, so we
        # need to list those explicitly. Note that numpy.testing.Tester inserts
        # the top-level package path determined from __file__ to argv unconditionally,
        # so we only need to add the part that is not otherwise recursed into.
        import os
        underscore_modules = ['_lib', '_build_utils']
        base_dir = os.path.abspath(os.path.dirname(__file__))
        underscore_paths = [os.path.join(base_dir, name) for name in underscore_modules]
        kw['extra_argv'] = list(kw.get('extra_argv', [])) + underscore_paths
        return test._tester.test(*a, **kw)

    test._tester = Tester()
    test.__doc__ = test._tester.test.__doc__
    test.__test__ = False  # Prevent nose from treating test() as a test
