/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     assemble_dowb.c                                                */
/*                                                                          */
/* description:  fe-space independent assemblation routines of matrices     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Claus-Justus Heine                                           */
/*             Abteilung fuer Angewandte Mathematik                         */
/*             Albert-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg im Breisgau, Germany                        */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERT                        */
 /*                                                                          */
 /*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
 /*                                                                          */
 /*--------------------------------------------------------------------------*/

#include "alberta.h"

static const REAL TOO_SMALL = 1.e-15;

/* <<< struct dowb_fill_info */

/* Could name it "FILL_INFO", too, but this way it is easier to debug */
typedef struct dowb_fill_info  DOWB_FILL_INFO;
struct dowb_fill_info
{
  const FE_SPACE     *psi_fe;
  const FE_SPACE     *phi_fe;
  const QUAD         *quad[3];

  const Q11_PSI_PHI  *q11_psi_phi;
  const Q01_PSI_PHI  *q01_psi_phi;
  const Q10_PSI_PHI  *q10_psi_phi;
  const Q00_PSI_PHI  *q00_psi_phi;

  const QUAD_FAST    *psi_quad_fast[3];
  const QUAD_FAST    *phi_quad_fast[3];

  int        n_row, n_col;
  void       **element_matrix;

  PARAMETRIC *parametric;

  int        (*init_element)(const EL_INFO *, const QUAD *[3], void *);

  void       (*fast_second_order)(const EL_INFO *, const DOWB_FILL_INFO *);
  void       (*fast_first_order)(const EL_INFO *, const DOWB_FILL_INFO *);
  void       (*fast_zero_order)(const EL_INFO *, const DOWB_FILL_INFO *);

  void       (*slow_second_order)(const EL_INFO *, const DOWB_FILL_INFO *);
  void       (*slow_first_order)(const EL_INFO *, const DOWB_FILL_INFO *);
  void       (*slow_zero_order)(const EL_INFO *, const DOWB_FILL_INFO *);

  union {
    const REAL_DD  (*(*full)(const EL_INFO *,
			     const QUAD *, int, void *))[N_LAMBDA];
    const REAL_DDS (*(*symm)(const EL_INFO *,
			     const QUAD *, int, void *))[N_LAMBDA];
    const REAL_D   (*(*diag)(const EL_INFO *,
			     const QUAD *, int, void *))[N_LAMBDA];
  } LALt;
  int           LALt_symmetric;
  union {
    const REAL_DD  *(*full)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_DDS *(*symm)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_D   *(*diag)(const EL_INFO *, const QUAD *, int, void *);
  } Lb0;
  union {
    const REAL_DD  *(*full)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_DDS *(*symm)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_D   *(*diag)(const EL_INFO *, const QUAD *, int, void *);
  } Lb1;
  int           Lb0_Lb1_anti_symmetric;
  union {
    const REAL_D   *(*full)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_DDS *(*symm)(const EL_INFO *, const QUAD *, int, void *);
    const REAL     *(*diag)(const EL_INFO *, const QUAD *, int, void *);
  } c;
  int           c_symmetric;

  DOWBM_TYPE    type;

  void          *user_data;

  DOWB_FILL_INFO  *next;
};

/* >>> */

/* <<< functions for calculating element stiffness matrices */

/*--------------------------------------------------------------------------*/
/*  functions for calculating element stiffness matrices                    */
/*--------------------------------------------------------------------------*/

#define DECL_DOWB_BB(thing)			\
union						\
{						\
  const REAL_DD  (*full)[N_LAMBDA];		\
  const REAL_DDS (*symm)[N_LAMBDA];		\
  const REAL_D   (*diag)[N_LAMBDA];		\
} thing
#define DECL_DOWB_B(thing)			\
union						\
{						\
  const REAL_DD  *full;				\
  const REAL_DDS *symm;				\
  const REAL_D   *diag;				\
} thing
#define DECL_DOWB_C(thing)			\
union						\
{						\
  const REAL_D   *full;				\
  const REAL_DDS *symm;				\
  const REAL     *diag;				\
} thing
#define DECL_SOMEVAR(thing)			\
union						\
{						\
  REAL_DD  full;				\
  REAL_DDS symm;				\
  REAL_D   diag;				\
} thing
#define ASSIGN_MAT(thing, val)			\
union						\
{						\
  REAL_DD  **full;				\
  REAL_DDS **symm;				\
  REAL_D   **diag;				\
} thing = { (REAL_DD **)val }

#define GET_VALUE(type, dst, src, args) dst.type = (src).type args

#define SWITCH_TYPE(type, body_f, body_s, body_d)	\
  switch (type) {					\
  case dowbm_full: body_f; break;			\
  case dowbm_symm: body_s; break;			\
  case dowbm_diag: body_d; break;			\
  }

#define GET_VALUES(type, dst, src, args)	\
  SWITCH_TYPE(type,				\
	      GET_VALUE(full, dst, src, args),	\
	      GET_VALUE(symm, dst, src, args),	\
	      GET_VALUE(diag, dst, src, args))

/* BODY(F, CAST, PRE, SUF) is supposed to be a "multiplex" macro where
 * BLAS routines are accessed via F##AXPY(..., CAST PRE var##SUF, ...)
 */
#define EMIT_BODY_SWITCH(type)			\
  SWITCH_TYPE(type,				\
	      BODY(M, (REAL_D *),, full),	\
	      BODY(SM, ,& , symm),		\
	      BODY(DM, , , diag))

/* <<< ... with pre-computed integrals */

/* <<< pre_2() */

static void pre_2(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  const int      **n_entries, *k, *l;
  const REAL     *values;
  int            i, j, m;
  DOWBM_TYPE     type = info->type;
  DECL_SOMEVAR(val);
  ASSIGN_MAT(mat, info->element_matrix);

  GET_VALUES(type, LALt, info->LALt, (el_info, info->quad[2], 0, info->user_data));
  n_entries = info->q11_psi_phi->n_entries;

  if (info->LALt_symmetric) {
    /* Careful: we have to tranpose the blocks belonging to the lower
     * triangle (thx, Dani!)
     */
#undef BODY
#define BODY(F, C, P, S)						\
    for (i = 0; i < info->n_row; i++) {					\
      k      = info->q11_psi_phi->k[i][i];				\
      l      = info->q11_psi_phi->l[i][i];				\
      values = info->q11_psi_phi->values[i][i];				\
      for ( m = 0; m < n_entries[i][i]; m++)				\
	F##AXPY_DOW(values[m], C P LALt.S[k[m]][l[m]], C P mat.S[i][i]); \
									\
      for (j = i+1; j < info->n_col; j++) {				\
	k      = info->q11_psi_phi->k[i][j];				\
	l      = info->q11_psi_phi->l[i][j];				\
	values = info->q11_psi_phi->values[i][j];			\
	F##SET_DOW(0.0, C P val.S);					\
	for (m = 0; m < n_entries[i][j]; m++)				\
	  F##AXPY_DOW(values[m], C P LALt.S[k[m]][l[m]], C P val.S);	\
	F##AXPY_DOW(1.0, C P val.S, C P mat.S[i][j]);			\
	F##AXTPY_DOW(1.0, C P val.S, C P mat.S[j][i]);		\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  } else { /*  A not symmetric or psi != phi        */
#undef BODY
#define BODY(F, C, P, S)						\
    for (i = 0; i < info->n_row; i++) {					\
      for (j = 0; j < info->n_col; j++) {				\
	k      = info->q11_psi_phi->k[i][j];				\
	l      = info->q11_psi_phi->l[i][j];				\
	values = info->q11_psi_phi->values[i][j];			\
	for (m = 0; m < n_entries[i][j]; m++)				\
	  F##AXPY_DOW(values[m],					\
		      C P LALt.S[k[m]][l[m]], C P mat.S[i][j]);	\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* <<< pre_01() */

static void pre_01(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb0);
  const int     **n_entries, *l;
  const REAL    *values;
  int           i, j, m;
  DOWBM_TYPE    type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  
  GET_VALUES(type, Lb0, info->Lb0, (el_info, info->quad[1], 0, info->user_data));
  n_entries = info->q01_psi_phi->n_entries;

#undef BODY
#define BODY(F, C, P, S)						\
  for (i = 0; i < info->n_row; i++) {					\
    for (j = 0; j < info->n_col; j++) {					\
      l      = info->q01_psi_phi->l[i][j];				\
      values = info->q01_psi_phi->values[i][j];				\
      for (m = 0; m < n_entries[i][j]; m++) {				\
	F##AXPY_DOW(values[m], C P Lb0.S[l[m]], C P mat.S[i][j]);	\
      }									\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< pre_10() */

static void pre_10(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb1);
  const int     **n_entries, *k;
  const REAL    *values;
  int           i, j, m;
  DOWBM_TYPE    type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  
  GET_VALUES(type, Lb1, info->Lb1, (el_info, info->quad[1], -1, info->user_data));
  n_entries = info->q10_psi_phi->n_entries;

#undef BODY
#define BODY(F, C, P, S)						\
  for (i = 0; i < info->n_row; i++) {					\
    for (j = 0; j < info->n_col; j++) {					\
      k      = info->q10_psi_phi->k[i][j];				\
      values = info->q10_psi_phi->values[i][j];				\
      for (m = 0; m < n_entries[i][j]; m++) {				\
	F##AXPY_DOW(values[m], C P Lb1.S[k[m]], C P mat.S[i][j]);	\
      }									\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< pre_11() */

static void pre_11(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb0);
  DECL_DOWB_B(Lb1);
  const int     **n_entries01, **n_entries10, *k, *l;
  const REAL    *values;
  int           i, j, m;
  DOWBM_TYPE    type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);

  GET_VALUES(type, Lb0, info->Lb0, (el_info, info->quad[1], 0, info->user_data));
  GET_VALUES(type, Lb1, info->Lb1, (el_info, info->quad[1], 0, info->user_data));
  n_entries01 = info->q01_psi_phi->n_entries;
  n_entries10 = info->q10_psi_phi->n_entries;

#undef BODY
#define BODY(F, C, P, S)						\
  for (i = 0; i < info->n_row; i++) {					\
    for (j = 0; j < info->n_col; j++) {					\
      l      = info->q01_psi_phi->l[i][j];				\
      values = info->q01_psi_phi->values[i][j];				\
      for (m = 0; m < n_entries01[i][j]; m++)				\
	F##AXPY_DOW(values[m], C P Lb0.S[l[m]], C P mat.S[i][j]);	\
									\
      k      = info->q10_psi_phi->k[i][j];				\
      values = info->q10_psi_phi->values[i][j];				\
      for (m = 0; m < n_entries10[i][j]; m++)				\
	F##AXPY_DOW(values[m], C P Lb1.S[k[m]], C P mat.S[i][j]);	\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< pre_0() */

static void pre_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  int          i, j;
  DECL_SOMEVAR(val);
  DECL_DOWB_C(c);
  const REAL   **values;
  DOWBM_TYPE   type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  
  GET_VALUES(type, c, info->c, (el_info, info->quad[0], 0, info->user_data));
  values = info->q00_psi_phi->values;

  if (info->c_symmetric) {
#undef BODY
#define BODY(F, C, P, S)					\
    for (i = 0; i < info->n_row; i++) {				\
      F##AXPY_DOW(values[i][i], C c.S, C P mat.S[i][i]);	\
      for (j = i+1; j < info->n_col; j++) {			\
	F##AXEY_DOW(values[i][j], C c.S, C P val.S);		\
	F##AXPY_DOW(1.0, C P val.S, C P mat.S[i][j]);		\
	F##AXPY_DOW(1.0, C P val.S, C P mat.S[j][i]);		\
      }								\
    }
    EMIT_BODY_SWITCH(type);
  } else {
#undef BODY
#define BODY(F, C, P, S)					\
    for (i = 0; i < info->n_row; i++)				\
      for (j = 0; j < info->n_col; j++)				\
	F##AXPY_DOW(values[i][j], C c.S, P mat.S[i][j]);
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* >>> */

/* <<< ... without pre-computed integrals */

/* <<< utAv, btv, REAL_DD versions */

/* component-wise n_lambda scp, b is matrix valued */
#undef BODY
#define BODY(F, T, RT, C, P)					\
  static inline RT F##btv(const int n_lambda, const T b[],	\
			  const REAL_B v, RT r)	                \
  {								\
    int  i;							\
								\
    F##AXEY_DOW(v[0], C P b[0], r);				\
    for (i = 1; i < n_lambda; i++) {				\
      F##AXPY_DOW(v[i], C P b[i], r);				\
    }								\
    return r;							\
  }

BODY(M, REAL_DD, REAL_D *, (REAL_D *),)
BODY(SM, REAL_DDS, REAL_DDS *, ,&)
BODY(DM, REAL_D, REAL *,,)

/* A is a (N_LAMBDA)x(N_LAMBDA) block-matrix with DOWxDOW entries */
#undef BODY

#define BODY(F, T, RT, C, P)						\
  static inline RT F##utAv(const int n_lambda,				\
			   const REAL_B u,				\
			   const T (*A)[N_LAMBDA],			\
			   const REAL_B v,				\
			   RT r)					\
  {									\
    int     i;								\
    T tmp;								\
									\
    F##AX_DOW(u[0], C F##btv(n_lambda, A[0], v, r));			\
    for (i = 1; i < n_lambda; i++) {					\
      F##AXPY_DOW(u[i], C F##btv(n_lambda, A[i], v, C P tmp), r);	\
    }									\
    return r;								\
  }

BODY(M, REAL_DD, REAL_D *, (REAL_D *),)
BODY(SM, REAL_DDS, REAL_DDS *, ,&)
BODY(DM, REAL_D, REAL *,,)

/* >>> */

/* <<< quad_2() */

static void quad_2(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  REAL             (*grd_psi)[N_LAMBDA], (*grd_phi)[N_LAMBDA];
  int              iq, i, j;
  int              n_lambda = el_info->mesh->dim + 1;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  DOWBM_TYPE       type = info->type;
  DECL_SOMEVAR(val);
  DECL_SOMEVAR(tmp);
  ASSIGN_MAT(mat, info->element_matrix);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric) {
#undef BODY
#define BODY(F, C, P, S)					\
    for (iq = 0; iq < quad->n_points; iq++) {			\
      GET_VALUE(S, LALt, info->LALt,				\
		(el_info, quad, iq, info->user_data));		\
      grd_psi = psi_fast->grd_phi[iq];				\
      grd_phi = phi_fast->grd_phi[iq];				\
      for (i = 0; i < info->n_row; i++) {			\
	F##AXPY_DOW(quad->w[iq],				\
		    F##utAv(n_lambda, grd_psi[i],		\
			    LALt.S, grd_phi[i], C P tmp.S),	\
		    C P mat.S[i][i]);				\
	for (j = i+1; j < info->n_col; j++) {			\
	  F##AX_DOW(quad->w[iq],				\
		    F##utAv(n_lambda, grd_psi[i],		\
			    LALt.S, grd_phi[j], C P val.S));	\
	  F##AXPY_DOW(1.0, C P val.S, C P mat.S[i][j]);	\
	  F##AXTPY_DOW(1.0, C P val.S, C P mat.S[j][i]);	\
	}							\
      }								\
    }
    EMIT_BODY_SWITCH(type);
  } else {      /*  non symmetric assembling   */
#undef BODY
#define BODY(F, C, P, S)					\
    for (iq = 0; iq < quad->n_points; iq++){			\
      GET_VALUE(S, LALt, info->LALt,				\
		(el_info, quad, iq, info->user_data));		\
      grd_psi = psi_fast->grd_phi[iq];				\
      grd_phi = phi_fast->grd_phi[iq];				\
								\
      for (i = 0; i < info->n_row; i++) {			\
	for (j = 0; j < info->n_col; j++) {			\
	  F##AXPY_DOW(quad->w[iq],				\
		      F##utAv(n_lambda, grd_psi[i],		\
			      LALt.S, grd_phi[j], C P tmp.S),	\
		      C P mat.S[i][j]);			\
	}							\
      }								\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* <<< quad_01() */

static void quad_01(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb0);
  REAL             *psi, (*grd_phi)[N_LAMBDA];
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int               n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE        type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(tmp);

  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

#undef BODY
#define BODY(F, C, P, S)						\
  for (iq = 0; iq < quad->n_points; iq++) {				\
    GET_VALUE(S, Lb0, info->Lb0, (el_info, quad, iq, info->user_data)); \
									\
    grd_phi = phi_fast->grd_phi[iq];					\
    psi     = psi_fast->phi[iq];					\
									\
    for (i = 0; i < info->n_row; i++) {					\
      for (j = 0; j < info->n_col; j++)					\
	F##AXPY_DOW(quad->w[iq]*psi[i],					\
		    F##btv(n_lambda, Lb0.S, grd_phi[j], C P tmp.S),	\
		    C P mat.S[i][j]);					\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_10() */

static void quad_10(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb1);
  REAL             (*grd_psi)[N_LAMBDA], *phi;
  int              iq, i, j;
  int               n_lambda = el_info->mesh->dim + 1;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(tmp);

  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

#undef BODY
#define BODY(F, C, P, S)						\
  for (iq = 0; iq < quad->n_points; iq++)  {				\
    GET_VALUE(S, Lb1, info->Lb1, (el_info, quad, iq, info->user_data)); \
									\
    phi     = phi_fast->phi[iq];					\
    grd_psi = psi_fast->grd_phi[iq];					\
									\
    for (i = 0; i < info->n_row; i++) {					\
      for (j = 0; j < info->n_col; j++)					\
	F##AXPY_DOW(quad->w[iq]*phi[j],					\
		    F##btv(n_lambda, Lb1.S, grd_psi[i], C P tmp.S),	\
		    C P mat.S[i][j]);					\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_11() */

static void quad_11(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb0);
  DECL_DOWB_B(Lb1);
  REAL             *psi, (*grd_psi)[N_LAMBDA], (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(tmp1);
  DECL_SOMEVAR(tmp2);
  DECL_SOMEVAR(val);

  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  if (info->Lb0_Lb1_anti_symmetric) {
#undef BODY
#define BODY(F, C, P, S)					\
    for (iq = 0; iq < quad->n_points; iq++) {			\
      GET_VALUE(S, Lb0, info->Lb0,				\
		(el_info, quad, iq, info->user_data));		\
      GET_VALUE(S, Lb1, info->Lb1,				\
		(el_info, quad, iq, info->user_data));		\
								\
      grd_phi = phi_fast->grd_phi[iq];				\
      phi     = phi_fast->phi[iq];				\
      grd_psi = psi_fast->grd_phi[iq];				\
      psi     = psi_fast->phi[iq];				\
								\
      for (i = 0; i < info->n_row; i++) {			\
	for (j = i+1; j < info->n_col; j++) {			\
	  F##AXPBY_DOW(quad->w[iq]*psi[i],			\
		       F##btv(n_lambda, Lb0.S, grd_phi[j],     \
			      C P tmp1.S),                     \
		       quad->w[iq]*phi[j],			\
		       F##btv(n_lambda, Lb1.S, grd_psi[i],     \
			      C P tmp2.S),               	\
		       C P val.S);				\
	  F##AXPY_DOW( 1.0, C P val.S, C P mat.S[i][j]);	\
	  F##AXTPY_DOW(-1.0, C P val.S, C P mat.S[j][i]);	\
	}							\
      }								\
    }
    EMIT_BODY_SWITCH(type);
  } else {
#undef BODY
#define BODY(F, C, P, S)					\
    for (iq = 0; iq < quad->n_points; iq++)  {			\
      GET_VALUE(S, Lb0, info->Lb0,				\
		(el_info, quad, iq, info->user_data));		\
      GET_VALUE(S, Lb1, info->Lb1,				\
		(el_info, quad, iq, info->user_data));		\
								\
      grd_phi = phi_fast->grd_phi[iq];				\
      phi     = phi_fast->phi[iq];				\
      grd_psi = psi_fast->grd_phi[iq];				\
      psi     = psi_fast->phi[iq];				\
								\
      for (i = 0; i < info->n_row; i++) {			\
	for (j = 0; j < info->n_col; j++) {			\
	  F##AXPBYP_DOW(quad->w[iq]*psi[i],			\
			F##btv(n_lambda, Lb0.S, grd_phi[j],	\
			       C P tmp1.S),			\
			quad->w[iq]*phi[j],			\
			F##btv(n_lambda, Lb1.S, grd_psi[i],	\
			       C P tmp2.S),			\
			C P mat.S[i][j]);			\
	}							\
      }								\
    }
    EMIT_BODY_SWITCH(type);
  }

}

/* >>> */

/* <<< quad_0() */

static void quad_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_C(c);
  REAL             *psi, *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  DOWBM_TYPE       type = info->type;
  DECL_SOMEVAR(val);
  ASSIGN_MAT(mat, info->element_matrix);

  quad          = info->quad[0];
  psi_fast = info->psi_quad_fast[0];
  phi_fast = info->phi_quad_fast[0];

  if (info->c_symmetric) {
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, c, info->c, (el_info, quad, iq, info->user_data));	\
      psi = psi_fast->phi[iq];						\
      phi = phi_fast->phi[iq];						\
      									\
      for (i = 0; i < info->n_row; i++) {				\
	F##AXPY_DOW(quad->w[iq] * psi[i] * phi[i], C c.S, C P mat.S[i][i]); \
									\
	for (j = i+1; j < info->n_col; j++) {				\
	  F##AXEY_DOW(quad->w[iq] * psi[i] * phi[j], C c.S, C P val.S); \
	  F##AXPY_DOW(1.0, C P val.S, C P mat.S[i][j]);		\
	  F##AXPY_DOW(1.0, C P val.S, C P mat.S[j][i]);		\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  } else {      /*  non symmetric assembling   */
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, c, info->c, (el_info, quad, iq, info->user_data));	\
      psi = psi_fast->phi[iq];						\
      phi = phi_fast->phi[iq];						\
      									\
      for (i = 0; i < info->n_row; i++) {				\
	for (j = 0; j < info->n_col; j++)				\
	  F##AXPY_DOW(quad->w[iq] * psi[i] * phi[j], C c.S, C P mat.S[i][j]);	\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* <<< quad_2_01() */

static void quad_2_01(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_B(Lb0);
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA];
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type; 
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(tmpA);
  DECL_SOMEVAR(tmpb);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

#undef BODY
#define BODY(F, C, P, S)					\
  for (iq = 0; iq < quad->n_points; iq++)			\
  {								\
    GET_VALUE(S, LALt, info->LALt,				\
	      (el_info, quad, iq, info->user_data));		\
    GET_VALUE(S, Lb0, info->Lb0,				\
	      (el_info, quad, iq, info->user_data));		\
								\
    grd_psi = psi_fast->grd_phi[iq];				\
    grd_phi = phi_fast->grd_phi[iq];				\
								\
    psi     = psi_fast->phi[iq];				\
								\
    for (i = 0; i < info->n_row; i++) {				\
      for (j = 0; j < info->n_col; j++) {			\
	F##AXPBYP_DOW(quad->w[iq]*psi[i],			\
		      F##btv(n_lambda, Lb0.S, grd_phi[j],	\
			     C P tmpb.S),			\
		      quad->w[iq],				\
		      F##utAv(n_lambda, grd_psi[i], LALt.S,	\
			      grd_phi[j], C P tmpA.S),		\
		      C P mat.S[i][j]);			\
      }								\
    }								\
  }
  EMIT_BODY_SWITCH(type);
}  

/* >>> */

/* <<< quad_2_10() */

static void quad_2_10(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_B(Lb1);
  REAL             (*grd_psi)[N_LAMBDA], (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(tmpA);
  DECL_SOMEVAR(tmpb);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

#undef BODY
#define BODY(F, C, P, S)					\
  for (iq = 0; iq < quad->n_points; iq++) {			\
    GET_VALUE(S, LALt, info->LALt,				\
	      (el_info, quad, iq, info->user_data));		\
    GET_VALUE(S, Lb1, info->Lb1,				\
	      (el_info, quad, iq, info->user_data));		\
								\
    grd_psi = psi_fast->grd_phi[iq];				\
    grd_phi = phi_fast->grd_phi[iq];				\
								\
    phi     = phi_fast->phi[iq];				\
								\
    for (i = 0; i < info->n_row; i++) {				\
      for (j = 0; j < info->n_col; j++) {			\
	F##AXPBYP_DOW(quad->w[iq]*phi[j],			\
		      F##btv(n_lambda, Lb1.S, grd_psi[i],	\
			     C P tmpb.S),			\
		      quad->w[iq],				\
		      F##utAv(n_lambda, grd_psi[i], LALt.S,	\
			      grd_phi[j], C P tmpA.S),		\
		      C P mat.S[i][j]);			\
      }								\
    }								\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_2_11() */

static void quad_2_11(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_B(Lb0);
  DECL_DOWB_B(Lb1);
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val1);
  DECL_SOMEVAR(val2);
  DECL_SOMEVAR(tmp1);
  DECL_SOMEVAR(tmp2);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric  &&  info->Lb0_Lb1_anti_symmetric) {
#undef BODY
#define BODY(F, C, P, S)					\
    for (iq = 0; iq < quad->n_points; iq++) {			\
      GET_VALUE(S, LALt, info->LALt,				\
		(el_info, quad, iq, info->user_data));		\
      GET_VALUE(S, Lb0, info->Lb0,				\
		(el_info, quad, iq, info->user_data));		\
      GET_VALUE(S, Lb1, info->Lb1,				\
		(el_info, quad, iq, info->user_data));		\
								\
      grd_phi = phi_fast->grd_phi[iq];				\
      phi     = phi_fast->phi[iq];				\
      								\
      grd_psi = psi_fast->grd_phi[iq];				\
      psi     = psi_fast->phi[iq];				\
								\
      for (i = 0; i < info->n_row; i++) {			\
	F##AXPY_DOW(quad->w[iq],				\
		    F##utAv(n_lambda, grd_psi[i], LALt.S,	\
			    grd_phi[i], C P tmp1.S),		\
		    C P mat.S[i][i]);				\
								\
	for (j = i+1; j < info->n_col; j++) {			\
	  F##AX_DOW(quad->w[iq],				\
		    F##utAv(n_lambda, grd_psi[i], LALt.S,	\
			    grd_phi[j], C P val2.S));		\
	  F##AXPY_DOW(1.0, C P val2.S, C P mat.S[i][j]);	\
	  F##AXTPY_DOW(1.0, C P val2.S, C P mat.S[j][i]);	\
								\
	  F##AXPBY_DOW(quad->w[iq]*psi[i],			\
		       F##btv(n_lambda, Lb0.S, grd_phi[j],	\
			      C P tmp1.S),			\
		       quad->w[iq]*phi[j],			\
		       F##btv(n_lambda, Lb1.S,grd_psi[i],	\
			      C P tmp2.S),			\
		       C P val1.S);				\
	  F##AXPY_DOW( 1.0, C P val1.S, C P mat.S[i][j]);	\
	  F##AXTPY_DOW(-1.0, C P val1.S, C P mat.S[j][i]);	\
	}							\
      }								\
    }
    EMIT_BODY_SWITCH(type);
  } else {
#undef BODY
#define BODY(F, C, P, S)   						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, LALt, info->LALt,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb0, info->Lb0,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb1, info->Lb1,					\
		(el_info, quad, iq, info->user_data));			\
									\
      grd_phi = phi_fast->grd_phi[iq];					\
      phi     = phi_fast->phi[iq];					\
      									\
      grd_psi = psi_fast->grd_phi[iq];					\
      psi     = psi_fast->phi[iq];					\
									\
      for (i = 0; i < info->n_row; i++) {				\
	for (j = 0; j < info->n_col; j++) {				\
	  F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[j],		\
		  C P val2.S);						\
	  F##AXPY_DOW(psi[i],						\
		      F##btv(n_lambda, Lb0.S, grd_phi[j], C P val1.S), \
		      C P val2.S);					\
	  F##AXPY_DOW(phi[j],						\
		      F##btv(n_lambda, Lb1.S, grd_psi[i], C P val1.S), \
		      C P val2.S);					\
	  F##AXPY_DOW(quad->w[iq], C P val2.S, C P mat.S[i][j]);	\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* <<< quad_2_0() */

static void quad_2_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_C(c);
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j, n_lambda = el_info->mesh->dim + 1;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val);

  quad          = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric) {  /* => psi == phi => c_symmetric  */
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, LALt, info->LALt,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, c, info->c,						\
		(el_info, quad, iq, info->user_data));			\
									\
      grd_psi = psi_fast->grd_phi[iq];					\
      grd_phi = phi_fast->grd_phi[iq];					\
									\
      psi     = psi_fast->phi[iq];					\
      phi     = phi_fast->phi[iq];					\
									\
      for (i = 0; i < info->n_row; i++) {				\
	F##AXPY_DOW(psi[i]*phi[i], C c.S,				\
		    F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[i],	\
			    C P val.S));				\
	F##AXPY_DOW(quad->w[iq], C P val.S, C P mat.S[i][i]);		\
	for (j = i+1; j < info->n_col; j++) {				\
	  F##AXPY_DOW(psi[i]*phi[j], C c.S,				\
		      F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[j], \
			      C P val.S));				\
	  F##AX_DOW(quad->w[iq], C P val.S);				\
	  F##AXPY_DOW(1.0, C P val.S, C P mat.S[i][j]);		\
	  F##AXTPY_DOW(1.0, C P val.S, C P mat.S[j][i]);		\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  } else {      /*  non symmetric assembling   */
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, LALt, info->LALt,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, c, info->c,						\
		(el_info, quad, iq, info->user_data));			\
									\
      grd_psi = psi_fast->grd_phi[iq];					\
      grd_phi = phi_fast->grd_phi[iq];					\
									\
      psi     = psi_fast->phi[iq];					\
      phi     = phi_fast->phi[iq];					\
      									\
      for (i = 0; i < info->n_row; i++) {				\
	for (j = 0; j < info->n_col; j++) {				\
	  F##AXPY_DOW(psi[i]*phi[j], C c.S,				\
		      F##utAv(n_lambda, grd_psi[i], LALt.S,		\
			      grd_phi[j], C P val.S));			\
	  F##AXPY_DOW(quad->w[iq], C P val.S, C P mat.S[i][j]);	\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* <<< quad_01_0() */

static void quad_01_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb0);
  DECL_DOWB_C(c);
  REAL             *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val);

  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

#undef BODY
#define BODY(F, C, P, S)						\
  for (iq = 0; iq < quad->n_points; iq++) {				\
    GET_VALUE(S, Lb0, info->Lb0,					\
	      (el_info, quad, iq, info->user_data));			\
    GET_VALUE(S, c, info->c,						\
	      (el_info, quad, iq, info->user_data));			\
									\
    grd_phi = phi_fast->grd_phi[iq];					\
    psi     = psi_fast->phi[iq];					\
    phi     = phi_fast->phi[iq];					\
									\
    for (i = 0; i < info->n_row; i++) {					\
      for (j = 0; j < info->n_col; j++) {				\
	F##AXPY_DOW(phi[j], C c.S, F##btv(n_lambda, Lb0.S, grd_phi[j], \
					   C P val.S));		\
	F##AXPY_DOW(quad->w[iq]*psi[i], C P val.S, C P mat.S[i][j]);	\
      }									\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_10_0 */

static void quad_10_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb1);
  DECL_DOWB_C(c);
  REAL             (*grd_psi)[N_LAMBDA], *psi, *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val);

  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

#undef BODY
#define BODY(F, C, P, S)						\
  for (iq = 0; iq < quad->n_points; iq++) {				\
    GET_VALUE(S, Lb1, info->Lb1,					\
	      (el_info, quad, iq, info->user_data));			\
    GET_VALUE(S, c, info->c,						\
	      (el_info, quad, iq, info->user_data));			\
									\
    grd_psi = psi_fast->grd_phi[iq];					\
    psi     = psi_fast->phi[iq];					\
    phi     = phi_fast->phi[iq];					\
									\
    for (i = 0; i < info->n_row; i++) {					\
      for (j = 0; j < info->n_col; j++) {				\
	F##AXPY_DOW(psi[i], C c.S, F##btv(n_lambda, Lb1.S, grd_psi[i], \
					   C P val.S));		\
	F##AXPY_DOW(quad->w[iq]*phi[j], C P val.S, C P mat.S[i][j]);	\
      }									\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_11_0() */

static void quad_11_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_B(Lb0);
  DECL_DOWB_B(Lb1);
  DECL_DOWB_C(c);
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val0);
  DECL_SOMEVAR(val1);
  DECL_SOMEVAR(tmp1);
  DECL_SOMEVAR(tmp2);

  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  if (info->Lb0_Lb1_anti_symmetric) { /* => psi == phi => c symmetric  */

#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, Lb0, info->Lb0,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb1, info->Lb1,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, c, info->c,						\
		(el_info, quad, iq, info->user_data));			\
									\
      grd_psi = psi_fast->grd_phi[iq];					\
      psi     = psi_fast->phi[iq];					\
      grd_phi = phi_fast->grd_phi[iq];					\
      phi     = phi_fast->phi[iq];					\
									\
      for (i = 0; i < info->n_row; i++) {				\
	F##AXPY_DOW(quad->w[iq]*psi[i]*phi[i], C c.S, C P mat.S[i][i]); \
									\
	for (j = i+1; j < info->n_col; j++) {				\
	  F##AXPBY_DOW(quad->w[iq]*psi[i],				\
		       F##btv(n_lambda,Lb0.S, grd_phi[j], C P tmp1.S), \
		       quad->w[iq]*phi[j],				\
		       F##btv(n_lambda,Lb1.S, grd_psi[i], C P tmp2.S), \
		       C P val1.S);					\
	  F##AXPY_DOW( 1.0, C P val1.S, C P mat.S[i][j]);		\
	  F##AXTPY_DOW(-1.0, C P val1.S, C P mat.S[j][i]);		\
									\
	  F##AXEY_DOW(quad->w[iq]*psi[i]*phi[j], C c.S, C P val0.S);	\
	  F##AXPY_DOW(1.0, C P val0.S, C P mat.S[i][j]);		\
	  F##AXTPY_DOW(1.0, C P val0.S, C P mat.S[j][i]);		\
									\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  } else {
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, Lb0, info->Lb0,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb1, info->Lb1,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, c, info->c, (el_info, quad, iq, info->user_data));	\
									\
      grd_psi = psi_fast->grd_phi[iq];					\
      psi     = psi_fast->phi[iq];					\
      grd_phi = phi_fast->grd_phi[iq];					\
      phi     = phi_fast->phi[iq];					\
									\
      for (i = 0; i < info->n_row; i++) {				\
	for (j = 0; j < info->n_col; j++) {				\
	  F##AXPBY_DOW(psi[i],						\
		       F##btv(n_lambda,Lb0.S, grd_phi[j], C P tmp1.S), \
		       phi[j],						\
		       F##btv(n_lambda,Lb1.S, grd_psi[i], C P tmp2.S), \
		       C P val1.S);					\
	  F##AXPY_DOW(psi[i]*phi[j], C c.S, C P val1.S);		\
	  F##AXPY_DOW(quad->w[iq], C P val1.S, C P mat.S[i][j]);	\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* <<< quad_2_01_0() */

static void quad_2_01_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_B(Lb0);
  DECL_DOWB_C(c);
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val);
  DECL_SOMEVAR(tmp);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

#undef BODY
#define BODY(F, C, P, S)						\
  for (iq = 0; iq < quad->n_points; iq++) {				\
    GET_VALUE(S, LALt, info->LALt,					\
	      (el_info, quad, iq, info->user_data));			\
    GET_VALUE(S, Lb0, info->Lb0,					\
	      (el_info, quad, iq, info->user_data));			\
    GET_VALUE(S, c, info->c,						\
	      (el_info, quad, iq, info->user_data));			\
									\
    grd_psi = psi_fast->grd_phi[iq];					\
    grd_phi = phi_fast->grd_phi[iq];					\
									\
    psi     = psi_fast->phi[iq];					\
    phi     = phi_fast->phi[iq];					\
									\
    for (i = 0; i < info->n_row; i++)  {				\
      for (j = 0; j < info->n_col; j++) {				\
	F##AXPY_DOW(psi[i],						\
		    F##AXPY_DOW(phi[j], C c.S,				\
				F##btv(n_lambda, Lb0.S, grd_phi[j],	\
				       C P tmp.S)),			\
		    F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[j],	\
			    C P val.S));				\
	F##AXPY_DOW(quad->w[iq], C P val.S, C P mat.S[i][j]);		\
      }									\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_2_10_0() */

static void quad_2_10_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_B(Lb1);
  DECL_DOWB_C(c);
  REAL  (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val);
  DECL_SOMEVAR(tmp);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

#undef BODY
#define BODY(F, C, P, S)						\
  for (iq = 0; iq < quad->n_points; iq++) {				\
    GET_VALUE(S, LALt, info->LALt,					\
	      (el_info, quad, iq, info->user_data));			\
    GET_VALUE(S, Lb1, info->Lb1,					\
	      (el_info, quad, iq, info->user_data));			\
    GET_VALUE(S, c, info->c,						\
	      (el_info, quad, iq, info->user_data));			\
									\
    grd_psi = psi_fast->grd_phi[iq];					\
    grd_phi = phi_fast->grd_phi[iq];					\
									\
    psi     = psi_fast->phi[iq];					\
    phi     = phi_fast->phi[iq];					\
									\
    for (i = 0; i < info->n_row; i++) {					\
      for (j = 0; j < info->n_col; j++) {				\
	F##AXPY_DOW(phi[j],						\
		    F##AXPY_DOW(psi[i], C c.S,				\
				F##btv(n_lambda, Lb1.S, grd_psi[i],	\
				       C P tmp.S)),			\
		    F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[j],	\
			    C P val.S));				\
	F##AXPY_DOW(quad->w[iq], C P val.S, C P mat.S[i][j]);		\
      }									\
    }									\
  }
  EMIT_BODY_SWITCH(type);
}

/* >>> */

/* <<< quad_2_11_0 */

static void quad_2_11_0(const EL_INFO *el_info, const DOWB_FILL_INFO *info)
{
  DECL_DOWB_BB(LALt);
  DECL_DOWB_B(Lb0);
  DECL_DOWB_B(Lb1);
  DECL_DOWB_C(c);
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  DOWBM_TYPE       type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);
  DECL_SOMEVAR(val20);
  DECL_SOMEVAR(val1);
  DECL_SOMEVAR(tmp1);
  DECL_SOMEVAR(tmp2);

  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric  &&  info->Lb0_Lb1_anti_symmetric) {
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++)  {				\
      GET_VALUE(S, LALt, info->LALt,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb0, info->Lb0,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb1, info->Lb1,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, c, info->c,						\
		(el_info, quad, iq, info->user_data));			\
									\
      grd_psi = psi_fast->grd_phi[iq];					\
      grd_phi = phi_fast->grd_phi[iq];					\
      									\
      psi     = psi_fast->phi[iq];					\
      phi     = phi_fast->phi[iq];					\
									\
      for (i = 0; i < info->n_row; i++) {				\
	F##AXPY_DOW(psi[i]*phi[i], C c.S,				\
		    F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[i],	\
			    C P val20.S));				\
	F##AXPY_DOW(quad->w[iq], C P val20.S, C P mat.S[i][i]);	\
									\
	for (j = i+1; j < info->n_col; j++) {				\
	  F##AXPY_DOW(psi[i]*phi[j], C c.S,				\
		      F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[j], \
			      C P val20.S));				\
	  F##AX_DOW(quad->w[iq], C P val20.S);				\
	  F##AXPY_DOW(1.0, C P val20.S, C P mat.S[i][j]);		\
	  F##AXTPY_DOW(1.0, C P val20.S, C P mat.S[j][i]);		\
									\
	  F##AXPBY_DOW(quad->w[iq]*psi[i],				\
		       F##btv(n_lambda,Lb0.S, grd_phi[j], C P tmp1.S), \
		       quad->w[iq]*phi[j],				\
		       F##btv(n_lambda,Lb1.S, grd_psi[i], C P tmp2.S), \
		       C P val1.S);					\
	  F##AXPY_DOW( 1.0, C P val1.S, C P mat.S[i][j]);		\
	  F##AXTPY_DOW(-1.0, C P val1.S, C P mat.S[i][j]);		\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  } else {
#undef BODY
#define BODY(F, C, P, S)						\
    for (iq = 0; iq < quad->n_points; iq++) {				\
      GET_VALUE(S, LALt, info->LALt,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb0, info->Lb0,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, Lb1, info->Lb1,					\
		(el_info, quad, iq, info->user_data));			\
      GET_VALUE(S, c, info->c,						\
		(el_info, quad, iq, info->user_data));			\
									\
      grd_psi = psi_fast->grd_phi[iq];					\
      grd_phi = phi_fast->grd_phi[iq];					\
      									\
      psi     = psi_fast->phi[iq];					\
      phi     = phi_fast->phi[iq];					\
									\
      for (i = 0; i < info->n_row; i++) {				\
	for (j = 0; j < info->n_col; j++) {				\
	  F##AXPY_DOW(psi[i]*phi[j], C c.S,				\
		      F##utAv(n_lambda, grd_psi[i], LALt.S, grd_phi[j], \
			      C P val20.S));				\
	  F##AXPY_DOW(psi[i],						\
		      F##btv(n_lambda, Lb0.S, grd_phi[j], C P val1.S), \
		      C P val20.S);					\
	  F##AXPY_DOW(phi[j],						\
		      F##btv(n_lambda, Lb1.S, grd_psi[i], C P val1.S), \
		      C P val20.S);					\
	  F##AXPY_DOW(quad->w[iq], C P val20.S, C P mat.S[i][j]);	\
	}								\
      }									\
    }
    EMIT_BODY_SWITCH(type);
  }
}

/* >>> */

/* >>> */

static const void **element_matrix(const EL_INFO *el_info, void *fill_info)
{
  DOWB_FILL_INFO *info = (DOWB_FILL_INFO *)fill_info;
  int            i, j, use_slow = 0;
  DOWBM_TYPE     type = info->type;
  ASSIGN_MAT(mat, info->element_matrix);

#undef BODY
#define BODY(F, C, P, S)			\
  for(i = 0; i < info->n_row; i++) 		\
    for (j = 0; j < info->n_col; j++)		\
      F##SET_DOW(0.0, C P mat.S[i][j]);
  EMIT_BODY_SWITCH(type);

  /* Use the return value to determine if we are on a parametric element. */
  if (info->init_element)  
    use_slow = (*info->init_element)(el_info, info->quad, info->user_data);

  if (!use_slow && info->fast_second_order)  
    (*info->fast_second_order)(el_info, info);
  else if (info->slow_second_order)  
    (*info->slow_second_order)(el_info, info);

  if (!use_slow && info->fast_first_order)  
    (*info->fast_first_order)(el_info, info);
  else if (info->slow_first_order)  
    (*info->slow_first_order)(el_info, info);

  if (!use_slow && info->fast_zero_order)  
    (*info->fast_zero_order)(el_info, info);
  else if (info->slow_zero_order)  
    (*info->slow_zero_order)(el_info, info);

  return (const void **)info->element_matrix;
}

/* >>> */

/* <<< fill_dowb_matrix_info(), the exported function */

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* Just some defines to make the code below a little bit more readable.
 * also, use |= below instead of += so that the reader knows: it's not
 * magic, it's just a flag value.
 */
#define FLAG_C    0x8
#define FLAG_LB1  0x4
#define FLAG_LB0  0x2
#define FLAG_LALt 0x1

const EL_DOWB_MATRIX_INFO *
fill_dowb_matrix_info(const DOWB_OPERATOR_INFO *operator_info, 
		      EL_DOWB_MATRIX_INFO *matrix_info)
{
  FUNCNAME("fill_dowb_matrix_info");
  static DOWB_FILL_INFO *first_fill_info = nil;
  DOWB_FILL_INFO        *fill_info;
  DOWB_OPERATOR_INFO    oinfo[1];
  const BAS_FCTS        *psi, *phi;
  PARAMETRIC            *parametric = nil;
  int                   dim, not_all_param = false;
  int                   psi_deg, phi_deg;
  int                   pre2, pre1, pre0, quad2, quad1, quad0;

  *oinfo = *operator_info;

  TEST_EXIT(oinfo->type == dowbm_full ||
	    oinfo->type == dowbm_symm ||
	    oinfo->type == dowbm_diag,
	    "Unknown matrix type: %d\n", oinfo->type);

  if (!oinfo->row_fe_space && !oinfo->col_fe_space)
  {
    ERROR("both pointer to row and column FE_SPACEs nil\n");
    ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
    return(nil);
  }
  else if (!oinfo->row_fe_space)
    oinfo->row_fe_space = oinfo->col_fe_space;
  else if (!oinfo->col_fe_space)
    oinfo->col_fe_space = oinfo->row_fe_space;
  psi = oinfo->row_fe_space->bas_fcts;
  phi = oinfo->col_fe_space->bas_fcts;

  if(phi->dim != psi->dim) {
    ERROR("Support dimensions of phi and psi do not match!\n");
    ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
    return(nil);
  }

  dim = phi->dim;

  psi_deg = psi->degree;
  phi_deg = phi->degree;

  TEST_EXIT(oinfo->row_fe_space->mesh == oinfo->col_fe_space->mesh,
	    "Mesh must be the same for row and column fe_space!\n");

  parametric = oinfo->row_fe_space->mesh->parametric;

  if (!oinfo->c.full && !oinfo->Lb0.full &&
      !oinfo->Lb1.full && !oinfo->LALt.full)
  {
    ERROR("no function for 2nd, 1st, and 0 order term;\n");
    ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
    return(nil);
  }

  if(parametric) {
    if(!oinfo->quad[0] && !oinfo->quad[1] && !oinfo->quad[2]) {
      ERROR("User is responsible for providing at least one quadrature\n");
      ERROR("when using a parametric mesh!\n");
      ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
      return(nil);
    }
    not_all_param = parametric->not_all;
  }

  if (psi != phi)
  {
    oinfo->LALt_symmetric = 0;
    oinfo->Lb0_Lb1_anti_symmetric = 0;
  }

  if (oinfo->LALt.full && !oinfo->quad[2])
    oinfo->quad[2] = get_quadrature(dim, psi_deg + phi_deg - 2);
  else if (!oinfo->LALt.full)
    oinfo->quad[2] = nil;

  if ((oinfo->Lb0.full || oinfo->Lb1.full) && !oinfo->quad[1])
  {
    if ((!oinfo->Lb0_pw_const || !oinfo->Lb1_pw_const) && oinfo->quad[2])
      oinfo->quad[1] = oinfo->quad[2];
    else
      oinfo->quad[1] = get_quadrature(dim, psi_deg + phi_deg - 1);
  }
  else if (!oinfo->Lb0.full && !oinfo->Lb1.full)
    oinfo->quad[1] = nil;

  if (oinfo->c.full && !oinfo->quad[0])
  {
    if (!oinfo->c_pw_const && oinfo->quad[2])
      oinfo->quad[0] = oinfo->quad[2];
    else if (!oinfo->c_pw_const && oinfo->quad[1])
      oinfo->quad[0] = oinfo->quad[1];
    else
      oinfo->quad[0] = get_quadrature(dim, psi_deg + phi_deg);
  }
  else if (!oinfo->c.full)
    oinfo->quad[0] = nil;

/*--------------------------------------------------------------------------*/
/*  look for an existing fill_info                                          */
/*--------------------------------------------------------------------------*/

  for (fill_info = first_fill_info; fill_info; fill_info = fill_info->next)
  {
    if (fill_info->psi_fe != oinfo->row_fe_space) continue;
    if (fill_info->phi_fe != oinfo->col_fe_space) continue;
    if (fill_info->quad[2] != oinfo->quad[2]) continue;
    if (fill_info->quad[1] != oinfo->quad[1]) continue;
    if (fill_info->quad[0] != oinfo->quad[0]) continue;
    if (fill_info->parametric != parametric) continue;
    if (fill_info->init_element != oinfo->init_element) continue;
    if (fill_info->LALt.full != oinfo->LALt.full) continue;
    if (fill_info->LALt_symmetric != oinfo->LALt_symmetric) continue;
    if (fill_info->Lb0.full != oinfo->Lb0.full) continue;
    if (fill_info->Lb1.full != oinfo->Lb1.full) continue;
    if (fill_info->Lb0_Lb1_anti_symmetric != oinfo->Lb0_Lb1_anti_symmetric)
      continue;
    if (fill_info->c.full != oinfo->c.full) continue;
    if (fill_info->user_data != oinfo->user_data) continue;
    if (fill_info->type != oinfo->type) continue;

    break;
  }

  if (!fill_info)
  {
/*--------------------------------------------------------------------------*/
/*  create a new fill_info                                                  */
/*--------------------------------------------------------------------------*/
    const QUAD_FAST **psi_fast, **phi_fast;
    int             n_row = psi->n_bas_fcts, n_col = phi->n_bas_fcts;

    fill_info = MEM_CALLOC(1, DOWB_FILL_INFO);
    fill_info->next = first_fill_info;
    first_fill_info = fill_info;

    fill_info->psi_fe = oinfo->row_fe_space;
    fill_info->phi_fe = oinfo->col_fe_space;

    fill_info->quad[2] = oinfo->quad[2];
    fill_info->quad[1] = oinfo->quad[1];
    fill_info->quad[0] = oinfo->quad[0];

    fill_info->n_row = n_row;
    fill_info->n_col = n_col;
    switch (oinfo->type) {
    case dowbm_full:
      fill_info->element_matrix = (void **)MAT_ALLOC(n_row, n_col, REAL_DD);
      break;
    case dowbm_symm:
      fill_info->element_matrix = (void **)MAT_ALLOC(n_row, n_col, REAL_DDS);
      break;
    case dowbm_diag:
      fill_info->element_matrix = (void **)MAT_ALLOC(n_row, n_col, REAL_D);
      break;
    default:
      ERROR_EXIT("Unknown matrix type: %d\n", oinfo->type);
    }

    psi_fast = fill_info->psi_quad_fast;
    phi_fast = fill_info->phi_quad_fast;

#undef ASSIGN
#define ASSIGN(dst, src, what)			\
    (dst)->what.full = (src)->what.full;	\
    (dst)->what.symm = (src)->what.symm;	\
    (dst)->what.diag = (src)->what.diag;

    fill_info->parametric = parametric;

    fill_info->init_element = oinfo->init_element;
    ASSIGN(fill_info, oinfo, LALt);
    fill_info->LALt_symmetric = oinfo->LALt_symmetric;
    ASSIGN(fill_info, oinfo, Lb0);
    ASSIGN(fill_info, oinfo, Lb1);
    fill_info->Lb0_Lb1_anti_symmetric = oinfo->Lb0_Lb1_anti_symmetric;
    ASSIGN(fill_info, oinfo, c);
    fill_info->c_symmetric = (phi == psi);

    fill_info->user_data = oinfo->user_data;

    fill_info->type = oinfo->type;

    psi_fast[2] = phi_fast[2] = nil;
    pre2 = quad2 = 0;

    if (fill_info->LALt.full)
    {
      if (oinfo->LALt_pw_const)
      {
	pre2  |= FLAG_LALt;
	fill_info->q11_psi_phi = get_q11_psi_phi(psi, phi, oinfo->quad[2]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant LALt but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->LALt_pw_const || parametric) {
	quad2 |= FLAG_LALt;
	psi_fast[2] = get_quad_fast(psi, oinfo->quad[2], INIT_GRD_PHI);
	if (psi != phi)
	  phi_fast[2] = get_quad_fast(phi, oinfo->quad[2], INIT_GRD_PHI);
	else
	  phi_fast[2] = psi_fast[2];
      }
    }

    psi_fast[1] = phi_fast[1] = nil;
    pre1 = quad1 = 0;
    if (fill_info->Lb0.full)
    {
      if (oinfo->Lb0_pw_const)
      {
	pre1  |= FLAG_LB0;
	fill_info->q01_psi_phi = get_q01_psi_phi(psi, phi, oinfo->quad[1]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant Lb0 but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->Lb0_pw_const || parametric) {
	if (quad2 && oinfo->quad[1] == oinfo->quad[2])
	  quad2 |= FLAG_LB0;
	else
	  quad1 |= FLAG_LB0;

	psi_fast[1] = get_quad_fast(psi, oinfo->quad[1], INIT_PHI);
	phi_fast[1] = get_quad_fast(phi, oinfo->quad[1], INIT_GRD_PHI);
      }
    }

    if (fill_info->Lb1.full)
    {
      if (oinfo->Lb1_pw_const)
      {
	pre1  |= FLAG_LB1;
	fill_info->q10_psi_phi = get_q10_psi_phi(psi, phi, oinfo->quad[1]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant Lb1 but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->Lb1_pw_const || parametric) {
	if (quad2 && oinfo->quad[1] == oinfo->quad[2])
	  quad2 |= FLAG_LB1;
	else
	  quad1 |= FLAG_LB1;

	psi_fast[1] = get_quad_fast(psi, oinfo->quad[1], INIT_GRD_PHI);
	phi_fast[1] = get_quad_fast(phi, oinfo->quad[1], INIT_PHI);
      }
    }

    psi_fast[0] = phi_fast[0] = nil;
    pre0 = quad0 = 0;

    if (fill_info->c.full)
    {
      if (oinfo->c_pw_const)
      {
	pre0 |= FLAG_C;
	fill_info->q00_psi_phi = get_q00_psi_phi(psi, phi, oinfo->quad[0]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant c but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->c_pw_const || parametric) {
	if (quad2 && oinfo->quad[0] == oinfo->quad[2])
	  quad2 |= FLAG_C;
	else if (quad1 && oinfo->quad[0] == oinfo->quad[1])
	  quad1 |= FLAG_C;
	else
	  quad0 |= FLAG_C;

	psi_fast[0] = get_quad_fast(psi, oinfo->quad[0], INIT_PHI);
	if (psi != phi)
	  phi_fast[0] = get_quad_fast(phi, oinfo->quad[0], INIT_PHI);
	else
	  phi_fast[0] = psi_fast[0];
      }
    }

    if (pre2)
    {
      fill_info->fast_second_order = pre_2;
      INFO(0,2,"using second order pre_2\n");
    }

    switch (pre1)
    {
    case FLAG_LB0:
      fill_info->fast_first_order = pre_01;
      INFO(0,2,"using first order pre_01\n");
      break;
    case FLAG_LB1:
      fill_info->fast_first_order = pre_10;
      INFO(0,2,"using first order pre_10\n");
      break;
    case FLAG_LB0|FLAG_LB1:
      fill_info->fast_first_order = pre_11;
      INFO(0,2,"using first order pre_11\n");
      break;
    }

    if (pre0 /* == FLAG_C */)
    {
      fill_info->fast_zero_order = pre_0;
      INFO(0,2,"using zero order pre_0\n");
    }

    switch (quad2)
    {
    case FLAG_LALt: 
      fill_info->slow_second_order = quad_2;
      INFO(0,2,"using second order quad_2\n");
      break;
    case FLAG_LALt|FLAG_LB0:
      fill_info->slow_second_order = quad_2_01;
      INFO(0,2,"using second order quad_2_01\n");
      break;
    case FLAG_LALt|FLAG_LB1:
      fill_info->slow_second_order = quad_2_10;
      INFO(0,2,"using second order quad_2_10\n");
      break;
    case FLAG_LALt|FLAG_LB1|FLAG_LB0:
      fill_info->slow_second_order = quad_2_11;
      INFO(0,2,"using second order quad_2_11\n");
      break;
    case FLAG_LALt|FLAG_C:
      fill_info->slow_second_order = quad_2_0;
      INFO(0,2,"using second order quad_2_0\n");
      break;
    case FLAG_LALt|FLAG_LB0|FLAG_C:
      fill_info->slow_second_order = quad_2_01_0;
      INFO(0,2,"using second order quad_2_01_0\n");
      break;
    case FLAG_LALt|FLAG_LB1|FLAG_C:
      fill_info->slow_second_order = quad_2_10_0;
      INFO(0,2,"using second order quad_2_10_0\n");
      break;
    case FLAG_LALt|FLAG_LB1|FLAG_LB0|FLAG_C:
      fill_info->slow_second_order = quad_2_11_0;
      INFO(0,2,"using second order quad2_11_0\n");
      break;
    }

    switch (quad1)
    {
    case FLAG_LB0: 
      fill_info->slow_first_order = quad_01;
      INFO(0,2,"using first order quad_01\n");
      break;
    case FLAG_LB1: 
      fill_info->slow_first_order = quad_10;
      INFO(0,2,"using first order quad_10\n");
      break;
    case FLAG_LB0|FLAG_LB1: 
      fill_info->slow_first_order = quad_11;
      INFO(0,2,"using first order quad_11\n");
      break;
    case FLAG_LB0|FLAG_C: 
      fill_info->slow_first_order = quad_01_0;
      INFO(0,2,"using first order quad_01_0\n");
      break;
    case FLAG_LB1|FLAG_C:
      fill_info->slow_first_order = quad_10_0;
      INFO(0,2,"using first order quad_10_0\n");
      break;
    case FLAG_LB0|FLAG_LB1|FLAG_C: 
      fill_info->slow_first_order = quad_11_0;
      INFO(0,2,"using first order quad_11_0\n");
      break;
    }

    if (quad0 /* == FLAG_C */)
    {
      fill_info->slow_zero_order = quad_0;
      INFO(0,2,"using zero order quad_0\n");
    }
  }

  if (!matrix_info)
    matrix_info = MEM_CALLOC(1, EL_DOWB_MATRIX_INFO);

  matrix_info->n_row = psi->n_bas_fcts;
  matrix_info->row_admin = oinfo->row_fe_space->admin;
  matrix_info->col_admin = oinfo->col_fe_space->admin;
  matrix_info->get_row_dof = psi->get_dof_indices;
  matrix_info->type = oinfo->type;

  if (psi != phi)
  {
    matrix_info->n_col = phi->n_bas_fcts;
    matrix_info->get_col_dof = phi->get_dof_indices;
    matrix_info->get_bound = nil;
    matrix_info->fill_flag = oinfo->fill_flag;
  }
  else if (oinfo->use_get_bound)
  {
    matrix_info->get_bound = psi->get_bound;
    matrix_info->fill_flag = oinfo->fill_flag|FILL_BOUND;
  }
  else
  {
    matrix_info->get_bound = nil;
    matrix_info->fill_flag = oinfo->fill_flag;
  }

  matrix_info->factor = 1.0;
  matrix_info->el_matrix_fct = element_matrix;
  matrix_info->fill_info = fill_info;

  return matrix_info;
}

/* >>> */

