/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     element.c                                                      */
/*                                                                          */
/*                                                                          */
/* description:  routines on elements that depend on the dimension in 3d    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  world_to_coord_3d():  return -1 if inside, otherwise index of lambda<0  */
/*--------------------------------------------------------------------------*/

int world_to_coord_3d(const EL_INFO *el_info, const REAL *xy,
		      REAL_B lambda)
{
  FUNCNAME("world_to_coord_3d");
  REAL  edge[3][DIM_OF_WORLD], x[DIM_OF_WORLD];
  REAL  x0, det, det0, det1, det2, adet, lmin;
  int   i, j, k;

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->world_to_coord() to access the parametric mesh\n");

/*  wir haben das gleichungssystem zu loesen: */
/*       ( q1x q2x q3x)  (lambda1)     (qx)      */
/*       ( q1y q2y q3y)  (lambda2)  =  (qy)      */
/*       ( q1z q2z q3z)  (lambda3)     (qz)      */
/*      mit qi=pi-p3, q=xy-p3                 */

  for (j=0; j<DIM_OF_WORLD; j++) {
    x0 = el_info->coord[3][j];
    x[j] = xy[j] - x0;
    for (i=0; i<3; i++)
      edge[i][j] = el_info->coord[i][j] - x0;
  }

  det =  edge[0][0] * edge[1][1] * edge[2][2]
       + edge[0][1] * edge[1][2] * edge[2][0]
       + edge[0][2] * edge[1][0] * edge[2][1]
       - edge[0][2] * edge[1][1] * edge[2][0]
       - edge[0][0] * edge[1][2] * edge[2][1]
       - edge[0][1] * edge[1][0] * edge[2][2];
  det0 =       x[0] * edge[1][1] * edge[2][2]
       +       x[1] * edge[1][2] * edge[2][0]
       +       x[2] * edge[1][0] * edge[2][1]
       -       x[2] * edge[1][1] * edge[2][0]
       -       x[0] * edge[1][2] * edge[2][1]
       -       x[1] * edge[1][0] * edge[2][2];
  det1 = edge[0][0] *       x[1] * edge[2][2]
       + edge[0][1] *       x[2] * edge[2][0]
       + edge[0][2] *       x[0] * edge[2][1]
       - edge[0][2] *       x[1] * edge[2][0]
       - edge[0][0] *       x[2] * edge[2][1]
       - edge[0][1] *       x[0] * edge[2][2];
  det2 = edge[0][0] * edge[1][1] *       x[2]
       + edge[0][1] * edge[1][2] *       x[0]
       + edge[0][2] * edge[1][0] *       x[1]
       - edge[0][2] * edge[1][1] *       x[0]
       - edge[0][0] * edge[1][2] *       x[1]
       - edge[0][1] * edge[1][0] *       x[2];

  adet = ABS(det);

  if (adet < 1.E-20) {
    ERROR_EXIT("det = %le; abort\n", det);
    return(-2);
  }

  lambda[0] = det0 / det;
  lambda[1] = det1 / det;
  lambda[2] = det2 / det;
  lambda[3] = 1.0 - lambda[0] - lambda[1] - lambda[2];

  k = -1;
  lmin = 0.0;
  j = 0;
  for (i = 0; i <= 3; i++) {
    if ((lambda[i]*adet) < -1.E-15) {
      if (lambda[i] < lmin) {
	k = i;
	lmin = lambda[i];
      }
      j++;
    }
  }

  return(k);
}

/*--------------------------------------------------------------------------*/
/*  transform local coordinates l to world coordinates; if w is non nil     */
/*  store them at w otherwise return a pointer to some local static         */
/*  area containing world coordintes                                        */
/*--------------------------------------------------------------------------*/

const REAL *coord_to_world_3d(const EL_INFO *el_info, const REAL *l, REAL_D w)
{
  FUNCNAME("coord_to_world_3d");
  static REAL_D world;
  REAL         *ret;
  int           i;

#if DIM_OF_WORLD < 3
  ERROR_EXIT("Does not make sense for DIM_OF_WORLD < 3!\n");
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->coord_to_world() to access the parametric mesh\n");

  ret = w ? w : world;

  for(i = 0; i < DIM_OF_WORLD; i++)
    ret[i] = el_info->coord[0][i] * l[0]
           + el_info->coord[1][i] * l[1]
           + el_info->coord[2][i] * l[2]
           + el_info->coord[3][i] * l[3];

  return((const REAL *) ret);
}

/*--------------------------------------------------------------------------*/
/*  compute volume of an element                                            */
/*--------------------------------------------------------------------------*/

REAL el_det_3d(const EL_INFO *el_info)
{
  FUNCNAME("el_det_3d");
  REAL        e1[DIM_OF_WORLD], e2[DIM_OF_WORLD], e3[DIM_OF_WORLD], det;
  int         i;
  const REAL  *v0;

#if DIM_OF_WORLD < 3
  ERROR_EXIT("Does not make sense for DIM_OF_WORLD < 3!\n");
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->el_det() to access the parametric mesh\n");

  v0 = el_info->coord[0];
  for (i = 0; i < DIM_OF_WORLD; i++)  {
    e1[i] = el_info->coord[1][i] - v0[i];
    e2[i] = el_info->coord[2][i] - v0[i];
    e3[i] = el_info->coord[3][i] - v0[i];
  }

  det =   e1[0] * (e2[1]*e3[2] - e2[2]*e3[1])
        - e1[1] * (e2[0]*e3[2] - e2[2]*e3[0])
        + e1[2] * (e2[0]*e3[1] - e2[1]*e3[0]);
  det = ABS(det);

  return(det);
}

REAL el_volume_3d(const EL_INFO *el_info)
{
  return el_det_3d(el_info) / 6.0;
}

/*--------------------------------------------------------------------------*/
/*  compute gradients of basis functions on element                         */
/*  - returns abs(determinant)                                              */
/*--------------------------------------------------------------------------*/

REAL el_grd_lambda_3d(const EL_INFO *el_info,
		   REAL grd_lam[N_LAMBDA][DIM_OF_WORLD])
{
  FUNCNAME("el_grd_lambda_3d");
  int         i, j;
  REAL        e1[DIM_OF_WORLD], e2[DIM_OF_WORLD], e3[DIM_OF_WORLD];
  const REAL  *v0;
  REAL        det, adet;
  REAL        a11, a12, a13, a21, a22, a23, a31, a32, a33;

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->grd_lambda() to access the parametric mesh\n");

  v0 = el_info->coord[0];
  for (i = 0; i < DIM_OF_WORLD; i++) {
    e1[i] = el_info->coord[1][i] - v0[i];
    e2[i] = el_info->coord[2][i] - v0[i];
    e3[i] = el_info->coord[3][i] - v0[i];
  }

#if DIM_OF_WORLD == 3
  det =   e1[0] * (e2[1]*e3[2] - e2[2]*e3[1])
        - e1[1] * (e2[0]*e3[2] - e2[2]*e3[0])
        + e1[2] * (e2[0]*e3[1] - e2[1]*e3[0]);
  adet = ABS(det);
  if (adet < 1.0E-25)
  {
    MSG("abs(det) = %lf\n",adet);
    for (i = 0; i < N_VERTICES_3D; i++)
      for (j = 0; j < DIM_OF_WORLD; j++)
	grd_lam[i][j] = 0.0;
  }
  else 
  {
    det = 1.0 / det;
    a11 = (e2[1]*e3[2] - e2[2]*e3[1]) * det;    /* (a_ij) = A^{-T} */
    a12 = (e2[2]*e3[0] - e2[0]*e3[2]) * det;
    a13 = (e2[0]*e3[1] - e2[1]*e3[0]) * det;
    a21 = (e1[2]*e3[1] - e1[1]*e3[2]) * det;
    a22 = (e1[0]*e3[2] - e1[2]*e3[0]) * det;
    a23 = (e1[1]*e3[0] - e1[0]*e3[1]) * det;
    a31 = (e1[1]*e2[2] - e1[2]*e2[1]) * det;
    a32 = (e1[2]*e2[0] - e1[0]*e2[2]) * det;
    a33 = (e1[0]*e2[1] - e1[1]*e2[0]) * det;

    grd_lam[1][0] = a11;
    grd_lam[1][1] = a12;
    grd_lam[1][2] = a13;
    grd_lam[2][0] = a21;
    grd_lam[2][1] = a22;
    grd_lam[2][2] = a23;
    grd_lam[3][0] = a31;
    grd_lam[3][1] = a32;
    grd_lam[3][2] = a33;

    grd_lam[0][0] = -grd_lam[1][0] -grd_lam[2][0] -grd_lam[3][0];
    grd_lam[0][1] = -grd_lam[1][1] -grd_lam[2][1] -grd_lam[3][1];
    grd_lam[0][2] = -grd_lam[1][2] -grd_lam[2][2] -grd_lam[3][2];
  }


#else
  ERROR_EXIT("Does not make sense for DIM_OF_WORLD < 3!\n");
#endif

  return(adet);
}


/*--------------------------------------------------------------------------*/
/*  calculate a normal to a triangular face side of a tetrahedron with      */
/*  coordinates coord; return the absolute value of the determinant from    */
/*  the transformation to the reference element                             */
/*--------------------------------------------------------------------------*/

REAL get_wall_normal_3d(const EL_INFO *el_info, int i0, REAL *normal)
{
  FUNCNAME("get_wall_normal_3d");
#if DIM_OF_WORLD == 3
  static int   ind[7] = {0, 1, 2, 3, 0, 1, 2};
  REAL         det, scale;
  int          n, i1 = ind[i0+1], i2 = ind[i0+2], i3 = ind[i0+3];
  const REAL_D *coord = el_info->coord;

  REAL_D e0, e1, e2;
  DEBUG_TEST_FLAG(FILL_COORDS, el_info);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    e0[n] = coord[i2][n] - coord[i1][n];
    e1[n] = coord[i3][n] - coord[i1][n];
    e2[n] = coord[i0][n] - coord[i1][n];
  }

  WEDGE_DOW(e0, e1, normal);

  det = NORM_DOW(normal);
  TEST_EXIT(det > 1.e-30, "det = 0 on face %d\n", i0);

  scale = SCP_DOW(e2,normal) < 0.0 ? 1.0/det : -1.0/det;
  AX_DOW(scale, normal);

  return(det);
#else
  MSG("not implemented for DIM_OF_WORLD = %d in 3d\n", DIM_OF_WORLD);
  return(-1.0);
#endif
}

/*--------------------------------------------------------------------------*/
/* orient the vertices of walls                                             */
/* used by estimator for the jumps => same quadrature nodes from both sides!*/
/*--------------------------------------------------------------------------*/

int *sorted_wall_indices_3d(int wall, int permno)
{
  /* FUNCNAME("sorted_wall_indices_3d"); */
  static int  sorted[4][6][3] = 
      {{{1,3,2}, {2,1,3}, {1,2,3}, {3,2,1}, {3,1,2}, {2,3,1}},
       {{0,3,2}, {2,0,3}, {0,2,3}, {3,2,0}, {3,0,2}, {2,3,0}},
       {{0,3,1}, {1,0,3}, {0,1,3}, {3,1,0}, {3,0,1}, {1,3,0}},
       {{0,2,1}, {1,0,2}, {0,1,2}, {2,1,0}, {2,0,1}, {1,2,0}}};

  return sorted[wall][permno];
}


int wall_orientation_3d(const EL *el, int wall, int **vec)
{
  FUNCNAME("wall_orientation_3d");
  static int  vertex_of_wall[4][3] = {{1,2,3}, {0,2,3}, {0,1,3}, {0,1,2}};
  int    no;
  int    *vof = vertex_of_wall[wall];
  DOF    **dof = el->dof;

  no = -1;
  if (dof[vof[0]][0] < dof[vof[1]][0])
    no++;
  if (dof[vof[1]][0] < dof[vof[2]][0])
    no += 2;
  if (dof[vof[2]][0] < dof[vof[0]][0])
    no += 4;
  
  if (no >= 0  &&  no <= 5)
  {
    if (vec)
      *vec = sorted_wall_indices_3d(wall, no);
      
  }
  else
  {
    ERROR_EXIT("cannot sort wall indices of element %d at wall %d\n", 
	       INDEX(el), wall);
  }
  return no;
}

int *sort_wall_indices_3d(const EL *el, int wall, int *vec)
{
  /* FUNCNAME("sort_wall_indices_3d"); */
  int    i, *val  = nil;

  (void)wall_orientation_3d(el, wall, &val);
  if (vec)
  {
    for (i = 0; i < 3; i++)
	vec[i] = val[i];
    return val;
  }
  else
  {
    return vec;
  }
}
