/**
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2012-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


/*!
    \page lcmaps_ban_dn.mod ban_dn plugin

    \section ban_dnsection1 SYNOPSIS
    \b lcmaps_ban_dn.mod
        [-banmapfile \<location banmapfile\>]

    \section ban_dnsection2 DESCRIPTION

    This plugin is an Banning Plugin and will provide the LCMAPS system a credential banning feature based on the Distinghuished Name (DN).
    A banmapfile will be read, if the DN is listed, this means the DN is banned and the plug-in will register a negative result. This means a LCMAPS_MOD_FAIL.

    \section ban_dnoptions OPTIONS
    \subsection ban_dnoptie1 -banmapfile <banmapfile>
        When this option is set it will override the default path of the banmapfile.
        It is advised to use an absolute path to the banmapfile to avoid usage of the wrong file(path).

\section ban_dnReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure


\section ban_dnErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

\section ban_dnSeeAlso SEE ALSO
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_ban_dn.c
    \brief  Interface to the LCMAPS plugins
    \author Oscar Koeroo

    This file contains the code for ban_dn plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

#define _XOPEN_SOURCE	500

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>

#include "lcmaps_plugins_basic_config.h"

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include "lcmaps_gridmapfile.h"


/************************************************************************
 * defines
 ************************************************************************/

#define PLUGIN_PREFIX	"lcmaps_ban_dn"

#define PLUGIN_RUN	0   /* full run mode */
#define PLUGIN_VERIFY	1   /* verify-only mode */


/************************************************************************
 * global variables
 ************************************************************************/

static char *banmapfile = NULL; /* Filename of banning file */
static int  use_wildcard = 1;    /* default use wildcards */


/************************************************************************
 * private prototypes
 ************************************************************************/

/* called by plugin_run() and plugin_verify() */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode);


/************************************************************************
 * public functions
 ************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(int argc, char **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_initialize()";
    int i;

    /* Log commandline parameters on debug */
    lcmaps_log(LOG_DEBUG,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
	lcmaps_log(LOG_DEBUG,"%s: arg %d is %s\n", logstr, i, argv[i]);

    /* Parse arguments, argv[0] = name of plugin, so start with i = 1 */
    for (i = 1; i < argc; i++) {
	/* check banmapfile option (or similar variations) */
        if ( strcmp(argv[i], "-banmapfile") == 0 ||
             strcmp(argv[i], "-BANMAPFILE") == 0 ||
             strcmp(argv[i], "-ban_mapfile") == 0 ||
             strcmp(argv[i], "-BAN_MAPFILE") == 0 ||
             strcmp(argv[i], "-gridmapfile") == 0 ||
             strcmp(argv[i], "-GRIDMAPFILE") == 0 ||
             strcmp(argv[i], "-gridmap") == 0 ||
             strcmp(argv[i], "-GRIDMAP") == 0 )
	{
	    /* check valid filename argument */
            if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by valid filename\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(banmapfile); banmapfile=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (banmapfile = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &banmapfile)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using ban file \"%s\".\n",
		    logstr, banmapfile);
            i++;
        }
	/* check whether to disable wildcard matching */
	else if (strcmp(argv[i], "-disablewildcard") == 0 ||
                 strcmp(argv[i], "--disablewildcard") == 0 ||
                 strcmp(argv[i], "-disablewildcards") == 0 ||
                 strcmp(argv[i], "--disablewildcards") == 0 ||
                 strcmp(argv[i], "-no_wildcard") == 0)
	{
	    lcmaps_log(LOG_DEBUG, "%s: disabling wildcard matching\n", logstr);
            use_wildcard = 0;
        }
	/* any other argument is an error */
	else
	{
            lcmaps_log(LOG_ERR,
		    "%s: Unknown argument for plugin: %s (failure)\n",
		    logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Check we have a banmapfile */
    if (banmapfile==NULL || banmapfile[0]=='\0')    {
        lcmaps_log(LOG_ERR,
	    "%s: Error: No banmapfile was provided. "
	    "Use the option -banmapfile <banmapfile> to set one.\n", logstr);
        goto fail_init_module;
    }

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    free(banmapfile);
    banmapfile = NULL;

    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(int *argc, lcmaps_argument_t **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
	{"user_dn","char *", 1,NULL},
	{NULL	  ,NULL	   ,-1,NULL}
    };

    lcmaps_log(LOG_DEBUG,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log(LOG_DEBUG,"%s: address first argument: %p\n",
	    logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void) {
    const char * logstr = PLUGIN_PREFIX"-plugin_terminate()";

    lcmaps_log(LOG_DEBUG,"%s: terminating\n", logstr);

    free(banmapfile);
    banmapfile=NULL;

    return LCMAPS_MOD_SUCCESS;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Actual run/verify function. Called by both plugin_run and plugin_verify
 * with different lcmaps_mode.
 */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode) {
    const char *        logstr       = NULL;
    void *              value        = NULL;
    char **             user_dn_list = NULL;
    char *              dn           = NULL;
    int                 dn_cnt       = 0;
    unsigned short      options      = 0;
    int                 rc           = 0;
    char *              username     = NULL;
    int                 i;

    /* Set suitable logstr */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = PLUGIN_PREFIX"-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = PLUGIN_PREFIX"-plugin_verify()";
    else {
        lcmaps_log(LOG_ERR, PLUGIN_PREFIX"-plugin_run_or_verify(): "
		"attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_plugin;
    }

    /* Try to get DN from LCMAPS */
    /* First try to obtain DN from the credential data (i.e. stored by other
     * plugins */
    user_dn_list = getCredentialData(DN, &dn_cnt);
    if (dn_cnt>0)   {
	/* Log already registered DNs */
	for (i=0; i<dn_cnt; i++)
	    lcmaps_log(LOG_DEBUG,"%s: found registered DN[%d/%d]: %s\n",
		    logstr, i+1, dn_cnt, user_dn_list[i]);
	dn=user_dn_list[0];
    } else {
	/* No DNs are registered, use the introspect/run arguments */
	value=lcmaps_getArgValue("user_dn", "char *", argc, argv);
	if (value == NULL || (dn = *(char **)value) == NULL ) {
	    lcmaps_log(LOG_WARNING,"%s: could not get value of dn !\n", logstr);
	    return -1;
	}
	
	/* push it to the end-result registry */
	lcmaps_log(LOG_DEBUG, "%s: Adding DN to LCMAPS framework: %s\n",
		logstr, dn);
	addCredentialData(DN, &dn);
    }

    /* Log the found DN */
    lcmaps_log(LOG_DEBUG,"%s: found dn: %s\n", logstr, dn);

    /* Check if we have a banmapfile */
    if (banmapfile==NULL || banmapfile[0]=='\0')   {
        lcmaps_log(LOG_ERR, "%s: Error: No (valid) banmapfile assigned.\n",
		logstr);
        goto fail_plugin;
    }

    /* Log filename */
    lcmaps_log(LOG_DEBUG, "%s: banmapfile is: %s\n", logstr, banmapfile);

    /* Set the matching options */

    /* match DN only and ban-mapfile must exist */
    options = MATCH_ONLY_DN|REQUIRE_MAPFILE;

    /* check if we should use wildcards ... */
    if (use_wildcard)
	options = options|MATCH_WILD_CHARS;

    /* Do lookup for DN in ban mapfile */
    rc = lcmaps_gridmapfile(banmapfile, dn, NULL, options, &username);

    /* parse return value of lcmaps_gridmapfile */
    if (rc==-1)	{
	/* error */
	lcmaps_log(LOG_ERR, "%s: error while looking up DN in banmapfile %s\n",
	    logstr, banmapfile);
	goto fail_plugin;
    }
    if (rc==1)	{
	/* match */
	lcmaps_log(LOG_NOTICE,
		"%s: DN \"%s\" is found on the ban list.\n", logstr, dn);
	goto fail_plugin;
    }

    /* did not find match */
    lcmaps_log(LOG_DEBUG,
	    "%s: No entry found in the ban list: user is NOT banned.\n",
	    logstr);

    /* success */
    free(username);

    lcmaps_log(LOG_INFO,"%s: ban_dn plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_plugin:
    free(username);

    lcmaps_log(LOG_INFO,"%s: ban_dn plugin failed\n", logstr);

    return LCMAPS_MOD_FAIL;
}
