/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var ENV = require( '@stdlib/process/env' );
var now = require( '@stdlib/time/now' );
var minstd = require( './../../../base/minstd' );
var kstest = require( '@stdlib/stats/kstest' );
var UINT32_MAX = require( '@stdlib/constants/uint32/max' );
var Uint32Array = require( '@stdlib/array/uint32' );
var isUint32Array = require( '@stdlib/assert/is-uint32array' );
var typedarray2json = require( '@stdlib/array/to-json' );
var factory = require( './../lib/factory.js' );


// VARIABLES //

var opts = {
	'skip': ( ENV.TEST_MODE === 'coverage' )
};


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof factory, 'function', 'main export is a function' );
	t.end();
});

tape( 'if provided a value which is not an object, the factory function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( value );
		};
	}
});

tape( 'if provided a `prng` option which is not a function, the factory function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'prng': value
			});
		};
	}
});

tape( 'if provided a `copy` option which is not a boolean, the factory function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'copy': value
			});
		};
	}
});

tape( 'if provided a `seed` which is not a positive integer or a non-empty array-like object, the factory function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		0.0,
		-5.0,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'seed': value
			});
		};
	}
});

tape( 'the function throws a range error if provided a `seed` which is an integer greater than the maximum unsigned 32-bit integer', function test( t ) {
	var values;
	var i;

	values = [
		UINT32_MAX + 1,
		UINT32_MAX + 2,
		UINT32_MAX + 3
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws a range error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'seed': value
			});
		};
	}
});

tape( 'if provided a `state` option which is not a Uint32Array, the factory function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'state': value
			});
		};
	}
});

tape( 'if provided an invalid `state` option, the factory function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		new Uint32Array( 0 ),
		new Uint32Array( 10 ),
		new Uint32Array( 100 )
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'state': value
			});
		};
	}
});

tape( 'the function returns a pseudorandom number generator (no seed)', function test( t ) {
	var randn;
	var r;
	var i;

	randn = factory();
	for ( i = 0; i < 1e3; i++ ) {
		r = randn();
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function returns a pseudorandom number generator (options; no seed)', function test( t ) {
	var randn;
	var r;
	var i;

	randn = factory( {} );
	for ( i = 0; i < 1e3; i++ ) {
		r = randn();
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function returns a seeded pseudorandom number generator (integer seed)', function test( t ) {
	var randn1;
	var randn2;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	randn1 = factory({
		'seed': seed
	});
	randn2 = factory({
		'seed': seed
	});

	t.notEqual( randn1, randn2, 'separate generators' );

	for ( i = 0; i < 1e3; i++ ) {
		r1 = randn1();
		r2 = randn2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function returns a seeded pseudorandom number generator (array seed)', function test( t ) {
	var randn1;
	var randn2;
	var seed;
	var r1;
	var r2;
	var i;

	seed = [ minstd(), minstd(), minstd() ];

	randn1 = factory({
		'seed': seed
	});
	randn2 = factory({
		'seed': seed
	});

	t.notEqual( randn1, randn2, 'separate generators' );

	for ( i = 0; i < 1e3; i++ ) {
		r1 = randn1();
		r2 = randn2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'attached to the returned function is the generator name', function test( t ) {
	var randn = factory();
	t.equal( randn.NAME, 'box-muller', 'has property' );
	t.end();
});

tape( 'attached to the returned function is the underlying PRNG', function test( t ) {
	var randn = factory();
	t.equal( typeof randn.PRNG, 'function', 'has property' );
	t.end();
});

tape( 'attached to the returned function is the minimum possible generated number', function test( t ) {
	var randn = factory();

	t.equal( typeof randn.MIN, 'number', 'has property' );
	t.equal( randn.MIN < 0.0, true, 'less than 0' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.MIN, null, 'equal to `null`' );

	t.end();
});

tape( 'attached to the returned function is the maximum possible generated number', function test( t ) {
	var randn = factory();

	t.equal( typeof randn.MAX, 'number', 'has property' );
	t.equal( randn.MAX > 0.0, true, 'greater than 0' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.MAX, null, 'equal to `null`' );

	t.end();
});

tape( 'attached to the returned function is the generator seed (integer seed)', function test( t ) {
	var actual;
	var randn;

	randn = factory({
		'seed': 12345
	});
	actual = randn.seed;

	t.equal( isUint32Array( actual ), true, 'has property' );
	t.equal( actual[ 0 ], 12345, 'equal to provided seed' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.seed, null, 'equal to `null`' );

	t.end();
});

tape( 'attached to the returned function is the generator seed (array seed)', function test( t ) {
	var actual;
	var randn;
	var seed;
	var i;

	seed = [ minstd(), minstd(), minstd(), minstd(), minstd() ];
	randn = factory({
		'seed': seed
	});

	actual = randn.seed;
	t.equal( isUint32Array( actual ), true, 'has property' );
	for ( i = 0; i < seed.length; i++ ) {
		t.equal( actual[ i ], seed[ i ], 'returns expected value for word '+i );
	}
	t.end();
});

tape( 'attached to the returned function is the generator seed length', function test( t ) {
	var randn = factory();
	t.equal( typeof randn.seedLength, 'number', 'has property' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.seedLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state', function test( t ) {
	var randn = factory();
	t.equal( isUint32Array( randn.state ), true, 'has property' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.state, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state length', function test( t ) {
	var randn = factory();
	t.equal( typeof randn.stateLength, 'number', 'has property' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.stateLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state size', function test( t ) {
	var randn = factory();
	t.equal( typeof randn.byteLength, 'number', 'has property' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( randn.byteLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is a method to serialize the generator as a JSON object', function test( t ) {
	var randn;
	var o;

	randn = factory();
	t.equal( typeof randn.toJSON, 'function', 'has method' );

	o = randn.toJSON();
	t.equal( o.type, 'PRNG', 'has property' );
	t.equal( o.name, randn.NAME, 'has property' );
	t.deepEqual( o.state, typedarray2json( randn.state ), 'has property' );
	t.deepEqual( o.params, [], 'has property' );

	randn = factory({
		'prng': Math.random // eslint-disable-line stdlib/no-builtin-math
	});
	t.equal( typeof randn.toJSON, 'function', 'has method' );
	t.equal( randn.toJSON(), null, 'returns expected value' );

	t.end();
});

tape( 'if the `state` property is set to a value other than a Uint32Array, an error is thrown', function test( t ) {
	var values;
	var randn;
	var i;

	randn = factory();

	values = [
		'3',
		3,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{},
		function noop() {}
	];
	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when set to '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			randn.state = value;
		};
	}
});

tape( 'if the `state` property is set to an invalid value, an error is thrown', function test( t ) {
	var values;
	var randn;
	var i;

	randn = factory();

	values = [
		new Uint32Array( 0 ),
		new Uint32Array( 10 ),
		new Uint32Array( 20 )
	];
	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws an error when set to '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			randn.state = value;
		};
	}
});

tape( 'the function supports specifying the underlying PRNG', function test( t ) {
	var randn;
	var r;
	var i;

	randn = factory({
		'prng': minstd.normalized
	});

	for ( i = 0; i < 1e3; i++ ) {
		r = randn();
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function supports providing a seeded underlying PRNG', function test( t ) {
	var randn1;
	var randn2;
	var randu;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	randu = minstd.factory({
		'seed': seed
	});
	randn1 = factory({
		'prng': randu.normalized
	});

	randu = minstd.factory({
		'seed': seed
	});
	randn2 = factory({
		'prng': randu.normalized
	});

	t.notEqual( randn1, randn2, 'separate generators' );

	for ( i = 0; i < 1e3; i++ ) {
		r1 = randn1();
		r2 = randn2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function returns pseudorandom numbers from a standard normal distribution', opts, function test( t ) {
	var threshold;
	var count;
	var npass;
	var N;
	var x;

	threshold = 0.10;

	x = new Array( 1e3 );
	N = 300;

	count = -1;
	npass = 0;

	gof();

	function gof() {
		var rejected;
		var pValue;
		var randn;
		var bool;
		var i;
		var j;

		count += 1;
		rejected = 0;
		for ( i = 0; i < N; i++ ) {
			randn = factory();
			t.ok( true, 'seed: '+randn.seed );
			for ( j = 0; j < x.length; j++ ) {
				x[ j ] = randn();
			}
			// Test using Kolmogorov-Smirnov goodness-of-fit test:
			pValue = kstest( x, 'normal', 0.0, 1.0 ).pValue;
			t.equal( typeof pValue, 'number', 'returns a p-value: '+pValue );
			if ( pValue < 0.05 ) {
				rejected += 1;
			}
		}
		// Account for small sample size and few repeats...
		bool = ( rejected / N < threshold );

		// If we succeed the first time, we are done...
		if ( count === 0 && bool ) {
			return done( bool, rejected );
		}
		// Retry mode...
		if ( bool ) {
			npass += 1;
		}
		// Retry twice...
		if ( count < 2 ) {
			return gof();
		}
		// Both retries must succeed for test to pass:
		bool = ( npass >= 2 );
		return done( bool, rejected );
	}

	function done( bool, rejected ) {
		t.ok( bool, 'null hypothesis (i.e., that numbers are drawn from Normal(0,1)) is rejected in less than '+(threshold*100)+'% of cases ('+rejected+' of '+N+'). Repeats: '+npass+' of '+count+'.' );
		t.end();
	}
});

tape( 'the function supports specifying the generator state', function test( t ) {
	var randn;
	var state;
	var arr;
	var i;

	randn = factory();

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		randn();
	}
	// Capture the current state:
	state = randn.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( randn() );
	}

	// Create another PRNG using the captured state:
	randn = factory({
		'state': state
	});

	// Replay previously generated values...
	for ( i = 0; i < 100; i++ ) {
		t.equal( randn(), arr[ i ], 'returns expected value. i: '+i+'.' );
	}
	t.end();
});

tape( 'the function supports specifying a shared generator state', function test( t ) {
	var shared;
	var randn;
	var state;
	var rand1;
	var rand2;
	var arr;
	var v1;
	var v2;
	var i;
	var j;

	randn = factory();

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		randn();
	}
	// Capture the current state:
	state = randn.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( randn() );
	}

	// Create a copy of the state (to prevent mutation) which will be shared by more than one PRNG:
	shared = new Uint32Array( state );

	// Create PRNGs using the captured state:
	rand1 = factory({
		'state': shared,
		'copy': false
	});
	rand2 = factory({
		'state': shared,
		'copy': false
	});

	// Replay previously generated values...
	j = 0;
	for ( i = 0; i < 25; i++ ) {
		// NOTE: more complicated to test due to the generation of two pseudorandom variates at a time...
		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j ], 'returns expected value. i: '+j+'.' );
		t.equal( v2, arr[ j+2 ], 'returns expected value. i: '+(j+2)+'.' );

		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j+1 ], 'returns expected value. i: '+(j+1)+'.' );
		t.equal( v2, arr[ j+3 ], 'returns expected value. i: '+(j+3)+'.' );

		j += 4; // stride
	}

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		v2 = rand2();
	}

	// Reset the (shared) state:
	rand1.state = state;

	// Replay previously generated values...
	j = 0;
	for ( i = 0; i < 25; i++ ) {
		// NOTE: more complicated to test due to the generation of two pseudorandom variates at a time...
		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j ], 'returns expected value. i: '+j+'.' );
		t.equal( v2, arr[ j+2 ], 'returns expected value. i: '+(j+2)+'.' );

		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j+1 ], 'returns expected value. i: '+(j+1)+'.' );
		t.equal( v2, arr[ j+3 ], 'returns expected value. i: '+(j+3)+'.' );

		j += 4; // stride
	}
	t.end();
});

tape( 'the returned function supports setting the generator state', function test( t ) {
	var randn;
	var state;
	var arr;
	var i;

	randn = factory();

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		randn();
	}
	// Capture the current state:
	state = randn.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( randn() );
	}
	// Set the state:
	randn.state = state;

	// Replay previously generated values...
	for ( i = 0; i < 100; i++ ) {
		t.equal( randn(), arr[ i ], 'returns expected value. i: '+i+'.' );
	}
	t.end();
});
