@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-queue-list: List available queues
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $queueName = $ARGV[0];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

if ( not($queueName) ) {

	# Get the list of queues
	my $queues = $controlChannel->getQueueList($token);

	# Display the information for each queue
	foreach $queue ( $queues->valueof('//getQueueListResponse/return') ) {
		printf "%-15s %-15s %s (INIT=%s,WAIT=%s,EXEC=%s,DONE=%s,STOP=%s)\n", $queue->{queueName}, $queue->{schedulerName}, $queue->{comment},
		  $queue->{numberOfJobsInInitState}, $queue->{numberOfJobsInWaitState}, $queue->{numberOfJobsInExecState}, $queue->{numberOfJobsInDoneState},
		  $queue->{numberOfJobsInStopState};
	}
}
else {

	# Get the detail for the specified queue
	foreach $queue ( $controlChannel->getQueueDetail( $token, $queueName )->valueof('//getQueueDetailResponse/return') ) {
		printf "%-20s %s\n", $queue->{name}, $queue->{comment};
	}
}

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-queue-list - List available queues

=head1 SYNOPSIS

B<ecaccess-queue-list -version|-help|-manual>

B<ecaccess-queue-list [-debug]> B<[>I<queue-name>B<]>

=head1 DESCRIPTION

Two different types of queues are in use with ECaccess:

B<1> - The ECaccess batch queues, which correspond to one system at ECMWF with its specific batch environment, e.g. ecgate will
be the ECaccess queue, which will redirect jobs to ecgate at ECMWF, running LoadLeveler. This ECaccess queue will be given
as argument when submitting a batch job.

B<2> - The batch queues (or classes) on the systems at ECMWF will be given in the batch job with #@ class = for LoadLeveler.

The names of the ECaccess queues and associated batch queues at ECMWF can be seen with this command. When an ECaccess
I<queue-name> is specified then the details for this queue are displayed (including the list of classes).

The first ECaccess batch queue in the list is the default queue when submitting a batch job with the B<ecaccess-job-submit>
command.

=head1 ARGUMENTS

=over 8

=item I<queue-name> (optional)

The name of the ECaccess queue to retrieve the details.

=back

=head1 OPTIONS

=over 8

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-queue-list>

List all the ECaccess batch queues (or systems supporting job submission at ECMWF).

B<ecaccess-queue-list> I<ecgate>

List all the batch queues (or classes) on the ECMWF system I<ecgate>.

=head1 SEE ALSO

B<ecaccess>.

=cut

__END__
:endofperl
