
/*
 * rd.mkminix.c
 *
 * 1998-01-10
 * Chainsawed and converted by Dave Cinege
 *
 * 1998-01-17
 * Just some K&R styling 
 *
 * 1998-02-19
 * Sanity check for 2.1 code
 * Convert to 2.1 file ops
 *
 * 1998-10-17
 * Change kmalloc to vmalloc, to allow >12MB FS. Minix FS make check.
 *
 * mkfs.c - make a linux (minix) file-system.
 *
 * (C) 1991 Linus Torvalds, et al...
 *
 */

#ifndef CONFIG_MINIX_FS
	#error: "This won't do you much good without minix fs support!"
#endif

#include <linux/time.h>
#include <linux/unistd.h>
#include <linux/stat.h>
#include <linux/fcntl.h>
 
#include <linux/stddef.h>
#include <linux/minix_fs.h>
#include <linux/vmalloc.h>

#define MINIX_ROOT_INO 1
#define MAX_GOOD_BLOCKS 512

#define UPPER(size,n) ((size+((n)-1))/(n))
#define INODE_SIZE (sizeof(struct minix_inode))
#define INODE_BLOCKS UPPER(INODES, (MINIX_INODES_PER_BLOCK))
#define INODE_BUFFER_SIZE (INODE_BLOCKS * BLOCK_SIZE)
#define BITS_PER_BLOCK (BLOCK_SIZE<<3)
#define Inode (((struct minix_inode *) inode_buffer)-1)
#define Super (*(struct minix_super_block *)super_block_buffer)
#define INODES ((unsigned long)Super.s_ninodes)
#define ZONES ((unsigned long)(Super.s_nzones))
#define IMAPS ((unsigned long)Super.s_imap_blocks)
#define ZMAPS ((unsigned long)Super.s_zmap_blocks)
#define FIRSTZONE ((unsigned long)Super.s_firstdatazone)
#define ZONESIZE ((unsigned long)Super.s_log_zone_size)
#define MAXSIZE ((unsigned long)Super.s_max_size)
#define MAGIC (Super.s_magic)
#define NORM_FIRSTZONE (2+IMAPS+ZMAPS+INODE_BLOCKS)

#define zone_in_use(x) (bit(zone_map,(x)-FIRSTZONE+1))
#define mark_inode(x) (setbit(inode_map,(x)))
#define unmark_inode(x) (clrbit(inode_map,(x)))
#define mark_zone(x) (setbit(zone_map,(x)-FIRSTZONE+1))
#define unmark_zone(x) (clrbit(zone_map,(x)-FIRSTZONE+1))

static long BLOCKS = 0;

static int dirsize = 32;
static int magic = MINIX_SUPER_MAGIC2;

static char root_block[BLOCK_SIZE] = "\0";

static char * inode_buffer = NULL;

static char super_block_buffer[BLOCK_SIZE];
static char boot_block_buffer[512];
static char inode_map[BLOCK_SIZE * MINIX_I_MAP_SLOTS];
static char zone_map[BLOCK_SIZE * MINIX_Z_MAP_SLOTS];

static unsigned short good_blocks_table[MAX_GOOD_BLOCKS];
static int used_good_blocks = 0;
static unsigned long req_nr_inodes = 0;

static int mkminixfs(void);
static void write_tables(void);
static int get_free_block(void);
static void make_root_inodes(void);
static void setup_tables(void);

static inline void write_block(int blk, char * buffer);
static inline void mark_good_blocks(void);
static inline int next(int zone);

static inline int bit(char * addr,unsigned int nr); 
static inline int setbit(char * addr,unsigned int nr);
static inline int clrbit(char * addr,unsigned int nr);

extern int sys_time(int * tloc);

static
int mkminixfs(void)
{

	char * tmp;

	//sanity check
	outfile.f_pos = 0;

	magic = MINIX_SUPER_MAGIC2;

	BLOCKS = rd_size;
      
	if (BLOCKS > 65535)
		BLOCKS = 65535;

	tmp = root_block;
	*(short *)tmp = 1;
	strcpy(tmp+2,".");
	tmp += dirsize;
	*(short *)tmp = 1;
	strcpy(tmp+2,"..");
	tmp += dirsize;
	*(short *)tmp = 2;
	strcpy(tmp+2,".badblocks");

	setup_tables();

	make_root_inodes();
	mark_good_blocks();
	write_tables();

	return (0);
}

static
void write_tables(void)
{
	/* Mark the super block valid. */
	Super.s_state |= MINIX_VALID_FS;
	Super.s_state &= ~MINIX_ERROR_FS;
	
	outfile.f_pos = 0;
	outfile.f_op->write(&outfile, boot_block_buffer, 512, &outfile.f_pos);
	outfile.f_pos = BLOCK_SIZE;
	outfile.f_op->write(&outfile, super_block_buffer, BLOCK_SIZE, &outfile.f_pos);
	outfile.f_op->write(&outfile, inode_map, IMAPS*BLOCK_SIZE, &outfile.f_pos);
	outfile.f_op->write(&outfile, zone_map, ZMAPS*BLOCK_SIZE, &outfile.f_pos);
	outfile.f_op->write(&outfile, inode_buffer, INODE_BUFFER_SIZE, &outfile.f_pos);
}

static
int get_free_block(void)
{
	int blk;

	if (used_good_blocks)
		blk = good_blocks_table[used_good_blocks-1]+1;
	else
		blk = FIRSTZONE;
	while (blk < ZONES && zone_in_use(blk))
		blk++;
	good_blocks_table[used_good_blocks] = blk;
	used_good_blocks++;

	return blk;
}

static
void make_root_inodes(void)
{
	struct minix_inode * inode;

	inode = vmalloc(INODE_BUFFER_SIZE);
	memset(inode,0,INODE_BUFFER_SIZE);
	inode = &Inode[MINIX_ROOT_INO];
		
	mark_inode(MINIX_ROOT_INO);

	inode->i_zone[0] = get_free_block();	
	inode->i_nlinks = 2;
	inode->i_time = sys_time(NULL);
		
	root_block[2*dirsize] = '\0';
	root_block[2*dirsize+1] = '\0';
	inode->i_size = 2*dirsize;
	
	inode->i_mode = S_IFDIR + 0755;
	write_block(inode->i_zone[0],root_block);
}

static
void setup_tables(void)
{
	int i;
	unsigned long inodes;

	memset(inode_map,0xff,sizeof(inode_map));
	memset(zone_map,0xff,sizeof(zone_map));
	memset(super_block_buffer,0,BLOCK_SIZE);
	memset(boot_block_buffer,0,512);
	MAGIC = magic;
	ZONESIZE = 0;
	MAXSIZE = 0x7fffffff;
	ZONES = BLOCKS;
/* some magic nrs: 1 inode / 3 blocks */
	if ( req_nr_inodes == 0 ) 
		inodes = BLOCKS/3;
	else
		inodes = req_nr_inodes;
	if (inodes > 65535)
		inodes = 65535;
	INODES = inodes;
/* I don't want some off-by-one errors, so this hack... */
	if ((INODES & 8191) > 8188)
		INODES -= 5;
	if ((INODES & 8191) < 10)
		INODES -= 20;
	IMAPS = UPPER(INODES,BITS_PER_BLOCK);
	ZMAPS = 0;
	while (ZMAPS != UPPER(BLOCKS - NORM_FIRSTZONE,BITS_PER_BLOCK))
		ZMAPS = UPPER(BLOCKS - NORM_FIRSTZONE,BITS_PER_BLOCK);
	if (ZMAPS > 64)
		printk ("Filesystem too big for Linux to handle");
	FIRSTZONE = NORM_FIRSTZONE;
	for (i = FIRSTZONE ; i<ZONES ; i++)
		unmark_zone(i);
	for (i = MINIX_ROOT_INO ; i<INODES ; i++)
		unmark_inode(i);
	inode_buffer = vmalloc(INODE_BUFFER_SIZE);
	if (!inode_buffer)
		printk("unable to allocate buffer for inodes");
	memset(inode_buffer,0,INODE_BUFFER_SIZE);
	
	printk("minix: %ldi %ldbk %ldfdz(%ld) %dzs %ldms\n",INODES,ZONES,FIRSTZONE,
		NORM_FIRSTZONE,BLOCK_SIZE<<ZONESIZE,MAXSIZE);
}


static
inline
void write_block(int blk, char * buffer)
{
	outfile.f_pos=blk*BLOCK_SIZE;
	outfile.f_op->write(&outfile, buffer, BLOCK_SIZE, &outfile.f_pos);
}

static
inline
void mark_good_blocks(void)
{
	int blk;

	for (blk=0 ; blk < used_good_blocks ; blk++)
		mark_zone(good_blocks_table[blk]);
}

static
inline
int next(int zone)
{
	if (!zone)
		zone = FIRSTZONE-1;
	while (++zone < ZONES)
		if (zone_in_use(zone))
			return zone;
	return 0;
}

static
inline
int bit(char * addr,unsigned int nr) 
{
	return (addr[nr >> 3] & (1<<(nr & 7))) != 0;
}

static
inline
int setbit(char * addr,unsigned int nr)
{
	int __res = bit(addr, nr);
	addr[nr >> 3] |= (1<<(nr & 7));
	return __res != 0; \
}

static
inline
int clrbit(char * addr,unsigned int nr)
{
	int __res = bit(addr, nr);
	addr[nr >> 3] &= ~(1<<(nr & 7));
	return __res != 0;
}

