/*****************************************************************************
* cpipemon.c	Cisco HDLC Monitor
*
* Author:	David Fong
*
* Copyright:	(c) 1995-1997 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Aug 06, 1998	David Fong	Initial version based on ppipemon
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <linux/config.h>
#include "cpipemon.h"

#define TIMEOUT 1
#define TRUE 1
#define FALSE 0
#define MDATALEN 2024

/* Structures for data casting */
#pragma pack(1)
typedef struct {
	unsigned char link_address;	/* Broadcast or Unicast */
	unsigned char control;
	unsigned short packet_type;	/* IP/SLARP/CDP */
} cisco_header_t;
#pragma pack()

/* Link addresses */
#define CISCO_ADDR_UNICAST	0x0F
#define CISCO_ADDR_BCAST	0x8F

/* Packet Types */
#define CISCO_PACKET_IP		0x0800
#define CISCO_PACKET_SLARP	0x8035
#define CISCO_PACKET_CDP	0x2000
 
#pragma pack(1)
typedef struct {
	unsigned long code;	/* Slarp Control Packet Code */
	union {
		struct {
			unsigned long address;	/* IP address */
			unsigned long mask;	/* IP mask    */
			unsigned short reserved[3];
		} address;
		struct {
			unsigned long my_sequence;
			unsigned long your_sequence;
			unsigned short reliability;
			unsigned short t1; 	/* time alive (upper) */
			unsigned short t2; 	/* time alive (lower) */
		} keepalive;
	} un;
} cisco_slarp_t;
#pragma pack()

/* Packet Codes */
#define SLARP_REQUEST 	0
#define SLARP_REPLY	1
#define SLARP_KEEPALIVE 2


/* Prototypes */
void ResetCB(udp_management_packet_t *c1);
int MakeConnection( void );
int ObtainConfiguration( void );
unsigned char DoCommand( udp_management_packet_t *cb );
void SaveCB( udp_management_packet_t *dest, udp_management_packet_t *src );
void init( int argv, char *argc[]);
unsigned long decode_bps( unsigned char bps );

/* global for now */
struct sockaddr_in soin;
udp_management_packet_t cb, cbconfig;
unsigned int frame_count;
int is_508;
int raw_data = FALSE;

/* for S508/FT1 test commands */
static int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int off_counter, green_counter, red_counter;
int loop_counter;


/* defines for now */
char ipaddress[16];
int udp_port = 9000;
static int sock;

void ResetCB(udp_management_packet_t *c1)
{
	memset((void *)&c1->cblock.opp_flag, 0, sizeof(udp_management_packet_t )+ MDATALEN -16);
}

int MakeConnection( void )
{
	sock = socket( PF_INET, SOCK_DGRAM, 0 );

	if( sock < 0 ) {
		printf("Error: Unable to open socket!\n");
		return( FALSE );
	} /* if */

	soin.sin_family = AF_INET;
	soin.sin_addr.s_addr = inet_addr(ipaddress);
     	
	if(soin.sin_addr.s_addr < 0){
                printf("Error: Invalid IP address!\n");
                return( FALSE );
        }
	
	soin.sin_port = htons((u_short)udp_port);

	if( !connect( sock, (struct sockaddr *)&soin, sizeof(soin)) == 0 ) {
		printf("Error: Cannot make connection!\nMake sure the IP address is correct\n");
		return( FALSE );
	} /* if */

	if( !ObtainConfiguration() ) {
		printf("Error: Unable to obtain CHDLC information.\nMake sure the IP and UDP port are correct.\n");
		close( sock );
		return( FALSE );
	} /* if */   

	return( TRUE );
}; /* MakeConnection */

int ObtainConfiguration( void )
{
	unsigned char codeversion[10];
	unsigned char x;
   
	x = 0;
	cb.cblock.command= READ_CHDLC_CONFIGURATION;
	cb.cblock.buffer_length = 0;

	while (DoCommand(&cb) != 0 && ++x < 4) {
		if (cb.cblock.return_code == 0xaa) {
			printf("Error: Command timeout occurred"); 
			return(FALSE);
		}

		if (cb.cblock.return_code == 0xCC ) return(FALSE);
	}

	if (x >= 4) return(FALSE);

	if( cb.cblock.buffer_length == sizeof(CHDLC_CONFIGURATION_STRUCT)) {
		is_508 = TRUE;
	} else {
		is_508 = FALSE;
	} 
   
	strcpy(codeversion, "?.??");
   
	cb.cblock.command= READ_CHDLC_CODE_VERSION;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);
	if (cb.cblock.return_code == 0) {
		cb.data[cb.cblock.buffer_length] = 0;
		strcpy(codeversion, cb.data);
	}

	return(TRUE);
}; /* ObtainConfiguration */


unsigned char DoCommand( udp_management_packet_t *cb )
{
        static unsigned char id = 0;
        fd_set ready;
        struct timeval to;
        int x;

        for( x = 0; x < 4; x += 1 ) {
                cb->cblock.opp_flag = 0;
                cb->request_reply = 0x01;
                cb->id = id;
                /* 0xAA is our special return code indicating packet timeout */
                cb->cblock.return_code = 0xaa;

                send(sock, cb, 35+cb->cblock.buffer_length, 0);

                FD_ZERO(&ready);
                FD_SET(sock,&ready);
                to.tv_sec = 5;
                to.tv_usec = 0;

                if(select(sock + 1,&ready, NULL, NULL, &to)) {
                        recv(sock, cb->signature, 35+MDATALEN,0);
                        break;
                } else {
			printf("Error: No Data received from the connected socket.\n");
                } /* if */
        } //for

        /* make sure the id is correct (returning results from a previous
           call may be disastrous if not recognized)
           also make sure it is a reply
        */

        if (cb->id != id || cb->request_reply != 0x02){
        	printf("RET BB\n"); 
	       cb->cblock.return_code = 0xbb;
	}
        id++;
        if (cb->cblock.return_code == 0xCC) {
                printf("Error: Code is not running on the board!");
                exit(1);
        }

        return(cb->cblock.return_code);

}; /* DoCommand */


void init( int argv, char *argc[])
{
	cb.id = 0;

	/* signature for UDP Management */
	strcpy( cb.signature, "CTPIPEAB");
	
	if( (argc[1][0] == 'f')  || (argc[1][0] == 't')) {
		if( argv == 5 || argv == 4) {
                        strcpy(ipaddress,argc[3]);
                        if( argv == 5)
                                udp_port = atoi(argc[4]);
                } else {
                        printf("Error: Invalid number of arguments\n");
                        exit(1);
                }
	} else {
	  	if( argv == 4 || argv == 3) {
                        strcpy(ipaddress,argc[2]);
                        if( argv == 4 )
                                udp_port = atoi(argc[3]);
                } else {
                        printf("Error: Invalid number of arguments\n");
                        exit(1);
                }

	}
};


void error( void ) 
{

	printf("Error: Command failed!\n");

}; /* error */


void global_stats (void)
{
	GLOBAL_STATS_STRUCT* global_stats;
	cb.cblock.command= READ_GLOBAL_STATISTICS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code == 0 ) {
		global_stats = (GLOBAL_STATS_STRUCT *)&cb.data[0];
		printf("Times application did not respond to IRQ: %d",
			global_stats->app_IRQ_timeout_count);
	} else {
		error();
	}
};  /* global stats */

void flush_global_stats (void)
{
	cb.cblock.command= FLUSH_GLOBAL_STATISTICS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code != 0 ) error();
};  /* flush global stats */

void modem( void )
{
	unsigned char cts_dcd;
   
	cb.cblock.command= READ_MODEM_STATUS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code == 0 ) {
		memcpy(&cts_dcd,&cb.data[0],1);
		printf("DCD: ");
		(cts_dcd & 0x08) ? printf("High\n") : printf("Low\n");
		printf("CTS: ");
		(cts_dcd & 0x20) ? printf("High\n") : printf("Low\n");
	} else {
		error();
	}
}; /* modem */


void comm_stats (void)
{
	COMMS_ERROR_STATS_STRUCT* comm_stats;
	cb.cblock.command= READ_COMMS_ERROR_STATS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code == 0 ) {
		comm_stats = (COMMS_ERROR_STATS_STRUCT *)&cb.data[0];

		printf("Number of receiver overrun errors:  %d\n",
				comm_stats->Rx_overrun_err_count);
		printf("    Number of receiver CRC errors:  %d\n",
				comm_stats->CRC_err_count);
		printf("  Number of abort frames received:  %d\n",
				comm_stats->Rx_abort_count);
		printf("Number of times receiver disabled:  %d\n",
				comm_stats->Rx_dis_pri_bfrs_full_count);
		printf("     Number of transmit underruns:  %d\n",
				comm_stats->missed_Tx_und_int_count);
		printf("Number of times DCD changed state:  %d\n",
				comm_stats->DCD_state_change_count);
		printf("Number of times CTS changed state:  %d\n",
				comm_stats->CTS_state_change_count);
	} else {
		error();
	}
}; /* comm_stats */


void flush_comm_stats( void ) 
{
	cb.cblock.command= FLUSH_COMMS_ERROR_STATS;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code != 0 ) error();
}; /* flush_comm_stats */


void read_code_version (void)
{
	cb.cblock.command= READ_CHDLC_CODE_VERSION; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if (cb.cblock.return_code == 0) {
		unsigned char version[10];
		memcpy(version,&cb.data,cb.cblock.buffer_length);
		version[cb.cblock.buffer_length] = '\0';
		printf("Cisco HDLC Code version: %s\n", version);
	}
}; /* read code version */

void chdlc_configuration (void)
{
	CHDLC_CONFIGURATION_STRUCT *chdlc_cfg;
	cb.cblock.command = READ_CHDLC_CONFIGURATION; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if (cb.cblock.return_code == 0) {
		chdlc_cfg = (CHDLC_CONFIGURATION_STRUCT *)&cb.data[0];

		printf("Baud rate: %ld", chdlc_cfg->baud_rate);

		printf("\nLine configuration: ");
		     switch (chdlc_cfg->line_config_options) {
			case INTERFACE_LEVEL_V35:
				printf("V.35");
				break;
			case INTERFACE_LEVEL_RS232:
				printf("RS-232");
				break;
		     }

		printf("\n\nLink State depends on:");
		     if (chdlc_cfg->CHDLC_protocol_options & IGNORE_DCD_FOR_LINK_STAT) 				printf ("\tDCD:  NO");
		     else	printf ("\tDCD:  YES");
		     if (chdlc_cfg->CHDLC_protocol_options & IGNORE_CTS_FOR_LINK_STAT) 				printf ("\n\t\t\tCTS:  NO\n");
		     else	printf ("\n\t\t\tCTS:  YES\n");
		     if (chdlc_cfg->CHDLC_protocol_options & IGNORE_KPALV_FOR_LINK_STAT) 			printf ("\tKeepalive Reception:  NO\n");
		     else	printf ("\tKeepalive Reception:  YES\n");

		printf("\n\n                 Maximum data field length:  %d",
			chdlc_cfg->max_CHDLC_data_field_length);

		printf(  "\n          Keepalive transmit interval (ms):  %d",
			chdlc_cfg->transmit_keepalive_timer);

		printf(  "\nExpected keepalive reception interval (ms):  %d",
			chdlc_cfg->receive_keepalive_timer);
		
		printf(  "\n       Keepalive reception error tolerance:  %d",
			chdlc_cfg->keepalive_error_tolerance);

		printf(  "\n      SLARP request transmit interval (ms):  %d",
			chdlc_cfg->SLARP_request_timer);
		
	} else {
		error();
	}
}; /* chdlc_configuration */


void link_status (void)
{
	CHDLC_LINK_STATUS_STRUCT *status;
	cb.cblock.command= READ_CHDLC_LINK_STATUS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if (cb.cblock.return_code == 0) {
		status = (CHDLC_LINK_STATUS_STRUCT *)&cb.data[0];

		printf("                          Link status:  ");
		     switch (status->CHDLC_link_status) {
			case CHDLC_LINK_INACTIVE:
				printf("Inactive\n");
				break;
			case CHDLC_LINK_ACTIVE:
				printf("Active\n");
				break;
		     }

		printf("Available data frames for application:  %d\n",
			status->no_Data_frms_for_app);

		printf("                      Receiver status:  ");
		     if (status->receiver_status & 0x01)
			printf("Disabled due to flow control restrictions\n");
		     else printf("Enabled\n");
	} else {
		error();
	}
}; /* Link Status */		


void operational_stats (void)
{
	CHDLC_OPERATIONAL_STATS_STRUCT *stats;
	cb.cblock.command= READ_CHDLC_OPERATIONAL_STATS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if (cb.cblock.return_code == 0) {
		stats = (CHDLC_OPERATIONAL_STATS_STRUCT *)&cb.data[0];
 
		printf(    "             Number of frames transmitted:   %ld",stats->Data_frames_Tx_count);
		printf(  "\n              Number of bytes transmitted:   %ld",stats->Data_bytes_Tx_count);
		printf(  "\n                      Transmit Throughput:   %ld",stats->Data_Tx_throughput);
		printf(  "\n Transmit frames discarded (length error):   %ld",stats->Tx_Data_discard_lgth_err_count);

		printf("\n\n                Number of frames received:   %ld",stats->Data_frames_Rx_count);
		printf(  "\n                 Number of bytes received:   %ld",stats->Data_bytes_Rx_count);
		printf(  "\n                       Receive Throughput:   %ld",stats->Data_Rx_throughput);
		printf(  "\n    Received frames discarded (too short):   %ld",stats->Rx_Data_discard_short_count);
		printf(  "\n     Received frames discarded (too long):   %ld",stats->Rx_Data_discard_long_count);
		printf(  "\nReceived frames discarded (link inactive):   %ld",stats->Rx_Data_discard_inactive_count);


		printf("\n\nIncoming frames with format errors");
		printf(  "\n               Frames of excessive length:   %d", stats->Rx_frms_too_long_count);	
		printf(  "\n                  Incomplete Cisco Header:   %d", stats->Rx_frm_incomp_CHDLC_hdr_count);
		printf(  "\n          Invalid Cisco HDLC link address:   %d", stats->Rx_invalid_CHDLC_addr_count);
		printf(  "\n               Invalid Cisco HDLC control:   %d", stats->Rx_invalid_CHDLC_ctrl_count);
		printf(  "\n            Invalid Cisco HDLC frame type:   %d", stats->Rx_invalid_CHDLC_type_count);

		printf("\n\nCisco HDLC link active/inactive and loopback statistics");
		printf(  "\n                           Times that the link went active:   %d", stats->link_active_count);	
		printf(  "\n         Times that the link went inactive (modem failure):   %d", stats->link_inactive_modem_count);
		printf(  "\n     Times that the link went inactive (keepalive failure):   %d", stats->link_inactive_keepalive_count);
		printf(  "\n                                         link looped count:   %d", stats->link_looped_count);
	} else {
		error();
	}
} /* Operational_stats */


void flush_operational_stats( void ) 
{
	cb.cblock.command= FLUSH_CHDLC_OPERATIONAL_STATS;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code != 0 ) {
		error();
	}

}; /* flush_operational_stats */

void slarp_stats (void)
{
	CHDLC_OPERATIONAL_STATS_STRUCT *stats;
	cb.cblock.command= READ_CHDLC_OPERATIONAL_STATS; 
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if (cb.cblock.return_code == 0) {
		stats = (CHDLC_OPERATIONAL_STATS_STRUCT *)&cb.data[0];

		printf("\n SLARP frame transmission/reception statistics");
		printf(  "\n       SLARP request packets transmitted:   %ld",
 stats->CHDLC_SLARP_REQ_Tx_count);
		printf(  "\n          SLARP request packets received:   %ld",
 stats->CHDLC_SLARP_REQ_Rx_count);
		printf("\n\n         SLARP Reply packets transmitted:   %ld",
 stats->CHDLC_SLARP_REPLY_Tx_count);
		printf(  "\n            SLARP Reply packets received:   %ld",
 stats->CHDLC_SLARP_REPLY_Rx_count);
		printf("\n\n     SLARP keepalive packets transmitted:   %ld",
 stats->CHDLC_SLARP_KPALV_Tx_count);
		printf(  "\n        SLARP keepalive packets received:   %ld",
 stats->CHDLC_SLARP_KPALV_Rx_count);

		printf(  "\n\nIncoming SLARP Packets with format errors");
		printf(  "\n                      Invalid SLARP Code:   %d", stats->Rx_SLARP_invalid_code_count);
		printf(  "\n                Replies with bad IP addr:   %d", stats->Rx_SLARP_Reply_bad_IP_addr);
		printf(  "\n                Replies with bad netmask:   %d",stats->Rx_SLARP_Reply_bad_netmask);
		printf(  "\n\nSLARP timeout/retry statistics");
		printf(  "\n                  SLARP Request timeouts:   %d",stats->SLARP_Request_TO_count);
		printf(  "\n            keepalive reception timeouts:   %d",stats->SLARP_Rx_keepalive_TO_count);

		printf("\n\nCisco Discovery Protocol frames");
		printf(  "\n                             Transmitted:   %ld", stats->CHDLC_CDP_Tx_count);
		printf(  "\n                                Received:   %ld", stats->CHDLC_CDP_Rx_count);

	} else {
		error();
	}

} /* slarp_stats */



void line_trace() 
{
	unsigned char num_frames, num_chars;
	unsigned short curr_pos = 0;
	frame_data_t *frame_info;
	unsigned int i, j;
	unsigned char outstr[2000];
	int recv_buff = sizeof(udp_management_packet_t ) + MDATALEN + 100;
	fd_set ready;
	struct timeval to;
	cisco_header_t *cisco_header;
 
	setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );

	/* Disable trace to ensure that the buffers are flushed */
	cb.cblock.command= CPIPE_DISABLE_TRACING;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	cb.cblock.command= CPIPE_ENABLE_TRACING;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);

	if( cb.cblock.return_code == 0 ) { 
		printf("Starting trace...(Press ENTER to exit)\n");
	} else if( cb.cblock.return_code == 0xCD ) {
		printf("Cannot Enable Line Tracing from Underneath.\n");
		return;
	} else if( cb.cblock.return_code == 0x01 ) {
		printf("Starting trace...(although it's already enabled!)\n");
		printf("Press ENTER to exit.\n");
	} else {
		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", 
			cb.cblock.return_code );
		return;
	}

	for(;;) {
		FD_ZERO(&ready);
		FD_SET(0,&ready);
		to.tv_sec = 1;
		to.tv_usec = 0;
	
		if(select(1,&ready, NULL, NULL, &to)) {
			break;
		} /* if */

		cb.cblock.command = CPIPE_GET_TRACE_INFO;
		cb.cblock.buffer_length = 0;
		DoCommand(&cb);
		if (cb.cblock.return_code == 0 && cb.cblock.buffer_length) { 
		     num_frames = cb.num_frames;
		     for ( i = 0; i < num_frames; i++) {
			frame_info= (frame_data_t *)(&cb.data[0] + curr_pos);

			/*  frame type */
			if (frame_info->status & 0x01) {
				sprintf(outstr,"OUTGOING\t\0");
			} else {
				if (frame_info->status & 0x10) { 
					sprintf(outstr,"INCOMING - Aborted\t\0");
				} else if (frame_info->status & 0x20) {
					sprintf(outstr,"INCOMING - CRC Error\t\0");
				} else if (frame_info->status & 0x40) {
					sprintf(outstr,"INCOMING - Overrun Error\t\0");
				} else {
					sprintf(outstr,"INCOMING\t\0");
				}
			}

			/* real length and time stamp */
			sprintf(outstr+strlen(outstr), "%d\t%d\t\0", 
			     frame_info->real_length, frame_info->time_stamp);

			/* first update curr_pos */
			curr_pos += sizeof(frame_data_t);
			if (frame_info->data_avail == 0) {
				sprintf( outstr+strlen(outstr), "the frame data is not available" );
			} else {
				/* update curr_pos again */
				curr_pos += frame_info->real_length - 1;
				num_chars = (unsigned char)
					((frame_info->real_length <= 25)
					? frame_info->real_length : 25);

				if (raw_data) { /*  show raw data */
				     for( j=0; j<num_chars; j++ ) {
					sprintf(outstr+strlen(outstr), "%02X \0", (unsigned char)frame_info->data[j]);
				     }
				     outstr[strlen(outstr)-1] = '\0';
				} else { /* show int data */
				     cisco_header = (cisco_header_t *)
							&frame_info->data[0];
				     switch(ntohs(cisco_header->packet_type)) {

					case CISCO_PACKET_IP:
					     sprintf(outstr+strlen(outstr),
						"IP packet from %ld.%ld.%ld.%ld to %ld.%ld.%ld.%ld",
	((ip_packet_t *)(cisco_header + 1))->ip_src_address & 0x000000FF,
	(((ip_packet_t *)(cisco_header + 1))->ip_src_address & 0x0000FF00) >> 8,
	(((ip_packet_t *)(cisco_header + 1))->ip_src_address & 0x00FF0000) >>16,
	(((ip_packet_t *)(cisco_header + 1))->ip_src_address & 0xFF000000) >>24,

	((ip_packet_t *)(cisco_header + 1))->ip_dst_address & 0x000000FF,
	(((ip_packet_t *)(cisco_header + 1))->ip_dst_address & 0x0000FF00) >> 8,
	(((ip_packet_t *)(cisco_header + 1))->ip_dst_address & 0x00FF0000) >>16,
	(((ip_packet_t *)(cisco_header + 1))->ip_dst_address & 0xFF000000) >>24
);
					     break; 

					case CISCO_PACKET_SLARP: { 
					     cisco_slarp_t *cisco_slarp =
					       (cisco_slarp_t*)(cisco_header+1);

					     switch(ntohl(cisco_slarp->code)) {
						case SLARP_REQUEST:
					     	  sprintf(outstr+strlen(outstr),
						     "SLARP Request packet");
						  break;

						case SLARP_REPLY:
					     	  sprintf(outstr+strlen(outstr),
						     "SLARP Reply packet");
						  break;

						case SLARP_KEEPALIVE:
					     	  sprintf(outstr+strlen(outstr),
						     "SLARP Keepalive packet");
						  break;

						default:
					     	  sprintf(outstr+strlen(outstr),
						   "Unrecognized SLARP packet");
						  break;
					     } /* Slarp code switch */
					     break;
					} /* slarp case */

					case CISCO_PACKET_CDP: 
					     sprintf(outstr+strlen(outstr),
					     "Cisco Discover Protocol packet");
					     break;

					default:
						sprintf(outstr+strlen(outstr),"Unknown type");
				     } /* Switch packet type */
				} //if
			     } //if
 
			     printf("%s\n",outstr);
			} //for
		} //if
		curr_pos = 0;

		if (!cb.ismoredata) sleep(TIMEOUT);
	}

	cb.cblock.command= CPIPE_DISABLE_TRACING;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb);
}; /* line_trace */


void usage(void)
{
	printf("cpipemon [f](lush)/ [t](1 features) option ip-address [port]\n\n");
	printf("One of the following:\n");
	printf("\tm  Modem Status\n");
	printf("\tv  Read Code Version\n");
	printf("\tc  Cisco HDLC Configuration\n");
	printf("\tx  Link Status\n");
	printf("\tq  Line Trace: display Interpreted data\n");
	printf("\tqr Line Trace: display RAW data\n");
	printf("\tu  Display Router UP time\n");
	printf("\nOptions that can be flushed:\n");
	printf("\ti Global Statistics\n");
	printf("\te Communication Error Statistics\n");
	printf("\tl Operational Statistics\n");
	printf("\tp SLARP and CDP Statistics\n");
	printf("\nOptions for S508/FT1 only\n");
	printf("\tv View Status\n");
	printf("\ts Self Test\n");
	printf("\tl Line Loop Test\n");
	printf("\td Digital Loop Test\n");
	printf("\tr Remote Test\n");
	printf("\to Operational Mode\n");

}


/* This subroutine enables the FT1 monitor on the board */

void set_FT1_monitor_status( unsigned char status)
{
	fail = 0;
	cb.cblock.command= FT1_MONITOR_STATUS_CTRL;
	cb.cblock.buffer_length = 1;
	cb.data[0] = status; 	
	DoCommand(&cb);
	
	if( cb.cblock.return_code != 0 && status){
		fail = 1;
		printf("This command is only possible with S508/FT1 board!");
	}
} /* set_FT1_monitor_status */


/* Subroutine for changing modes on a FT1 boards */

void set_FT1_mode( void )
{
	for(;;){ 
		cb.cblock.command= SET_FT1_MODE;
		cb.cblock.buffer_length = 0;
		DoCommand(&cb);
		if(cb.cblock.return_code == 0){
			break;
		}
	}
} /* set_FT1_mode */

/* Read the data for status of all the lights */

void read_FT1_status( void )
{
	int i;
	unsigned long delay;
	struct timeval tv;

	i = gettimeofday(&tv,NULL);
	delay = tv.tv_usec;

	for(;;){
		i = gettimeofday(&tv,NULL);
		if(abs((tv.tv_usec - delay)) > 90000 )
			break;  
	}

	cb.cblock.command= CPIPE_FT1_READ_STATUS;
	cb.cblock.buffer_length = 0;
	DoCommand(&cb); 

	if( cb.cblock.return_code == 0 ){
		par_port_A_byte = cb.data[0];
		par_port_B_byte = cb.data[1];
	}

} /* read_FT1_status */

/* Display the status of all the lights */

void view_FT1_status( void )
{
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;
	loop_counter = 0;

	/* check for INS light */
	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x30) == 0x30)
			off_counter++;
		if((par_port_B_byte & 0x10) == 0x00)
			red_counter++;
		if((par_port_B_byte & 0x20) == 0x00)
			green_counter++;

		/* Is the INS light red ? */
		if(red_counter != 0 && off_counter != 0 && loop_counter ==30) {
			printf("Unit is not IN SERVICE\n");
			break;
		}
		/* Is the INS light green ? */ 	
		if(green_counter != 0 && off_counter == 0 && loop_counter ==30){
			printf("Unit is IN SERVICE\n");
			break;
		}
	
		/* Is the INS light flashing green ? */
		if(green_counter != 0 && off_counter != 0 && loop_counter ==30){
			printf("INS is flasing green\n");
			break;
		}
	
		/* Is the INS light OFF ? */
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 30){
			printf("INS is off\n");
			break;
		}
	
		loop_counter++;
    	 }	
     
	/* check for ERR light */
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;
	loop_counter = 0;

	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x0C) == 0x0C)
			off_counter++;
		if((par_port_B_byte & 0x08) == 0x00)
			red_counter++;
		if((par_port_B_byte & 0x04) == 0x00)
			green_counter++;

		/* Is the ERR light off ? */
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 30){
			printf("No Line ERROR being received or Valid Line\n");
			break;
		}
		
		/* Is the ERR light red ? */
		if(red_counter != 0 && off_counter == 0 && green_counter == 0 
							&& loop_counter == 30){
			printf("Line ERROR being received or Invalid Line\n");
			break;
		}

		loop_counter++;
     	}

	/* check TXD light */
	loop_counter = 0;
	off_counter = 0;
	green_counter = 0;

	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x02) == 0x02)
			off_counter++;
		if((par_port_B_byte & 0x02) == 0x00)
			green_counter++;
	
		/* Is the TXD light off ? */
		if(off_counter != 0 && green_counter == 0 && loop_counter ==20){
			printf("Transmit data is not present\n");	
			break;
		}
	
		/* Is the TXD light flashing green ? */
		if(off_counter != 0 && green_counter != 0 && loop_counter ==20){
			printf("Transmit data is present \n");
			break;
		}
		loop_counter++;
     	}

	/* check RXD light */
	loop_counter = 0;
	off_counter = 0;
	green_counter = 0;

	for(;;){
		read_FT1_status();
		if((par_port_B_byte & 0x01) == 0x01)
			off_counter++;
		if((par_port_B_byte & 0x01) == 0x00)
			green_counter++;

		/* Is the RXD light off ? */
		if(off_counter != 0 && green_counter == 0 && loop_counter ==20){
			printf("Receive data is not present\n");	
			break;
		}

		/* Is the RXD light flashing green ? */
		if(off_counter != 0 && green_counter != 0 && loop_counter ==20){
			printf("Receive data is present\n");
			break;
		}
		loop_counter++;
     	}

}/* view_FT1_status */


void FT1_operational_mode(void)
{
	printf("Operational Mode has been selected\n");
	printf("Putting S508/FT1 in operational mode....");
	loop_counter = 0;
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;

	for(;;){
		read_FT1_status();
     		/* ST light is OFF */
     		if((par_port_B_byte & 0xc0) == 0xc0 ){
			off_counter++;
     		}
     	
		/* ST light is GREEN */
     		if((par_port_B_byte & 0x40) == 0x00){
			green_counter++;
			red_counter = 0;
     		}
     
		/* ST light is RED */
     		if((par_port_B_byte & 0x80) == 0x00){
			red_counter++;
			green_counter = 0;
   		}
		/* Is ST light off ? - then check next light */
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 20){
			break;
		}
	
		/* Is ST light red or green ? - then push button */
		if((red_counter != 0 || green_counter !=0) && loop_counter==20){
			set_FT1_mode();
			break;
		}

   		loop_counter++;
     	} /* end of for */
       
	loop_counter = 0;
	off_counter = 0;
	red_counter = 0;
	green_counter = 0;

	for(;;){
   		read_FT1_status();
  	 	/* DL light is OFF */
   		if((par_port_A_byte & 0x08) == 0x08){
			off_counter++;
		}
       		/* DL light is RED */
   		if((par_port_A_byte & 0x08) == 0x00){
			red_counter++;
		}

		/* Is DL light off ? - then check next light */
		if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
			break;
		}
	
		/* Is DL light red ? - then push button */
		if(red_counter != 0 && loop_counter == 20){
			set_FT1_mode();
			break;
		}
      		loop_counter++; 
     	}	 
     
	loop_counter = 0;
     	off_counter = 0;
     	red_counter = 0;
     	green_counter = 0;
     
	for(;;){
   		read_FT1_status();
   		/* LL light is off */
   		if((par_port_A_byte & 0x04) == 0x04){
			off_counter++;
		}
   		
		if((par_port_A_byte & 0x04) == 0x00){
			red_counter++;
		}
	
		/* Is LL light off ? - then check next light */
		if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
			break;
		}
	
		/* Is LL light red ? - then push button */
		if(red_counter != 0 && loop_counter == 20){
			set_FT1_mode();
			break;
		}
        
		loop_counter++;
     	}
     
	loop_counter = 0;
     	off_counter = 0;
     	red_counter = 0;
     	green_counter = 0;
     
	for(;;){
   		read_FT1_status();
		/* RT light is OFF */
   		if((par_port_A_byte & 0x03) == 0x03){
			off_counter++;
		}
		/* RT light is RED */
   		if((par_port_A_byte & 0x01) == 0x00){
			red_counter++;
		}
		/* RT light is GREEN */
   		if((par_port_A_byte & 0x02) == 0x00){
   			green_counter++;
		}

		/* Is RT light off ? - then it is in operational mode */ 
		if(off_counter != 0 && red_counter == 0 && green_counter == 0 
							&& loop_counter == 20){
			printf("Done!\n");
			break;
		}

		/* Is RT light flashing red or green ? - then push button */	
		if((red_counter != 0 || green_counter != 0) && off_counter != 0
 							&& loop_counter == 20){
			set_FT1_mode();
			printf("Done!\n");
			break;
		}

		/* Is RT light solid green ? - then can't push the button */
		if(off_counter == 0 && green_counter != 0 && loop_counter ==20){
			printf("Failed!\n");
			printf("Remote End is running Remote Test\n");
			printf("Exit Remote Test at remote end\n");
			break;
		}

		loop_counter++;
     	}        	

} /* FT1_operational_mode */


void FT1_self_test(void)
{
     	int started = 0; 
     	int selftest = 0;
 
     	set_FT1_mode();
     	off_counter = 0;
    	green_counter = 0;
     	red_counter = 0;
     	loop_counter = 0;
     	printf("Self Test has been selected\n");
     
	for(;;){
	  	read_FT1_status();
	  	/* ST light is OFF */
	  	if((par_port_B_byte & 0xc0) == 0xc0){
			off_counter++;
	  	}
	  
		/* ST light is GREEN */
	  	if((par_port_B_byte & 0x40) == 0x00){
			green_counter++;
	  	}
	  
		/* ST light is RED */
	  	if((par_port_B_byte & 0x80) == 0x00){
			red_counter++;
	  	}
         
		/* Is ST light flashing red ? - if not then push button */
		if(red_counter == 0 && loop_counter == 3){
			set_FT1_mode();    
			off_counter = 0;
			red_counter = 0;
			green_counter = 0;
			loop_counter = -1;
     			printf("Selecting Self Test....\r");
			selftest++;
	  		if( selftest == 10){
				  printf("\t\t\tFailed!\n");
				  printf("Self Test will not operate while connected to a valid line\n");
		 		 FT1_operational_mode();
				  break;
	      		}
	  	}
	 
		/* Is ST light flashing red ? */ 
		if(red_counter != 0 && off_counter != 0 && (loop_counter % 2)){
			printf("Performing Self Test....\r");
	        	started = 1;
	  	}
	  
		/* Is ST light flashing red ? */
		if(green_counter != 0 && off_counter != 0 && started){
	   		printf("\t\t\tDone!\n");
			FT1_operational_mode();
			break;
	  	}
          
		loop_counter++;	  
     	}/* end of for */     
} /* FT1_self_test */


void FT1_dl_test( void )
{
	int dl_test=0;

     	set_FT1_mode();
     	off_counter = 0;
     	red_counter = 0;
     	loop_counter = 0;
     
	printf("Bi-directional Digital Loop has been selected\n"); 
     
	for(;;){
		read_FT1_status();
		if((par_port_A_byte & 0x08) == 0x08){
			off_counter++;  
		}
	
		if((par_port_A_byte & 0x08) == 0x00){
			red_counter++;
		}

		/* Is DL light flashing red ? - if not then push button */	
		if(red_counter == 0 && loop_counter == 20){
			set_FT1_mode();
			off_counter = 0;
			red_counter = 0;
			loop_counter = -1;
			printf("Selecting Digital Loop Test....\r");
			dl_test++;
		
			if(dl_test==10){
	 			printf("\t\t\t\tFailed\n");
				printf("Remote End might be running Remote Test\n");
				break;
			}
		}
	
		/* Is DL light flashing red ? -if yes then DL test in progress*/
		if(red_counter != 0){
			off_counter = 0;
			red_counter = 0;
			green_counter = 0;
			loop_counter = 0;
			printf("Performing Digital Loop Test....\r");
			
			for(;;){
				read_FT1_status();
				printf("Performing Digital Loop Test....\r");
				/* check INS light */
				if((par_port_B_byte & 0x30) == 0x30)
					off_counter++;
				if((par_port_B_byte & 0x10) == 0x00){
					red_counter++;
					green_counter = 0;
				}
				if((par_port_B_byte & 0x20) == 0x00){
					green_counter++;
					red_counter = 0;
				}

				/* Is INS light red ? - if yes then DL test 
				   failed */
				if(red_counter != 0 && (par_port_B_byte & 0x08) 
								== 0x00 ){
					printf("\t\t\t\tFailed!\n");
					printf("Bi-directional Digital Loop test has failed\n");
					printf("Either the unit is not connected or the line is Invalid\n");
					break;
				}
			
				/* Is INS light green ? - if yes then DL test 
				   passed */
				if(green_counter != 0  && (par_port_B_byte & 
					0x0C) == 0x0C && loop_counter == 100 ){
					printf("\t\t\t\tDone!\n");
					printf("Bi-directional Digital Loop test has been successfully\n");
					printf("completed\n");
					break;
		
				}
			
				loop_counter++;
			} /* end of for */
			break;	
		} 
		loop_counter++;
     	} /* end of for */

} /* FT1_dl_test */

void FT1_ll_test( void )
{
    	int ll_test = 0;

    	set_FT1_mode();	
    	off_counter = 0;
    	red_counter = 0;
    	loop_counter = 0;
    
	printf("Line Loop Test has been selected\n");
    
	for(;;){
		read_FT1_status();
	
		if((par_port_A_byte & 0x04) == 0x04){
			off_counter++;
		}
		if((par_port_A_byte & 0x04) == 0x00){
			red_counter++;
		}

		/* Is LL light not flashing red ? - if not then push button */
		if(red_counter == 0 && off_counter != 0 && loop_counter == 20){
			set_FT1_mode();
			off_counter = 0;
			red_counter = 0;
			loop_counter = -1;
			printf("Selecting Line Loop Test....\r");
			ll_test++;
			if(ll_test == 10){
				printf("\t\t\t\tFailed!\n");
				printf("Line Loop Test will not operate while connected to a valid line\n");
				FT1_operational_mode();
		 		break;	
			}
		}
	
		/* Is LL light flashing red ? - if yes then it is selected */
		if(red_counter != 0){	
			off_counter = 0;
			red_counter = 0;
			loop_counter = 0;
		
			for(;;){
				printf("Performing Line Loop Test....\r");
				read_FT1_status();	
			
				/* check INS light */
				if((par_port_B_byte & 0x30) == 0x30)
					off_counter++;
			
				if((par_port_B_byte & 0x10) == 0x00){
					red_counter++;
					green_counter = 0;
				}
			
				if((par_port_B_byte & 0x20) == 0x00){
					green_counter++;
					red_counter = 0;
				}
			
				/* Is INS light green ? - if yes then test 
				   passed */
				if(green_counter != 0 && red_counter == 0){
					printf("\t\t\t\tDone!\n");
					printf("Line Loop Test has been successfully completed\n");
					break;
				}
				
				/* Is INS light red ? - if yes then test 
				   failed  */
				if(red_counter != 0 && green_counter == 0 && loop_counter == 100){
					printf("\t\t\t\tFailed!\n");
					break;
				}
			
				loop_counter++;
			} /* end of for */
			break;
		} /* end of if */
		loop_counter++;
    	} /* end of for */

} /* FT1_ll_test */

void FT1_rt_test( void )
{
    	off_counter = 0;
   	red_counter = 0;
    	green_counter = 0;	  	
    	loop_counter = 0;
    	printf("Remote Test has been selected\n");
    	
	for(;;){
		read_FT1_status();
		if((par_port_A_byte & 0x03) == 0x03)
			off_counter++;
		if((par_port_A_byte & 0x01) == 0x00)
			red_counter++;
		if((par_port_A_byte & 0x02) == 0x00)
			green_counter++;

		/* Is RT light red ? - if not then push button */
		if(red_counter == 0 && loop_counter == 20){
			set_FT1_mode();
			off_counter = 0;
			red_counter = 0;
			green_counter = 0;
			loop_counter = -1;
			printf("Selecting Remote Test....\r");		
		}

		/* Is RT light flashing green ? - if yes then RT test is 
	 	   selected */
		if(green_counter != 0 && loop_counter == 10){
			loop_counter = 0;
	   		off_counter = 0;
	   
			for(;;) {
				read_FT1_status();
				if((par_port_A_byte & 0x03) == 0x03)
					off_counter++;

				/* RT test was already started at the remote
				   end */
	        		if(off_counter == 0 && loop_counter == 20){
		   			printf("Remote End is currently sending Remote Test data\n");
		   			printf("Exit from the Remote Test mode at the remote end\n");
		   			break;
				}
				/* RT test has started */
				if(off_counter != 0 && loop_counter == 20) {
		   			printf("This unit is currently in Remote Testing Mode\n");
		   			break;
				}
	        		loop_counter++;
	  		}
	   		break;
		}
	
		if(red_counter != 0){
			printf("Waiting for a valid response from remote end....\r");
			loop_counter = 0;
			for(;;){
				read_FT1_status();
				if((par_port_A_byte & 0x02) == 0x00){
					printf("\t\t\t\t\t\tDone!\n");
					printf("Valid response has been received from remote end\n");
					printf("Remote Test has been successfully completed\n");
					break;	
				}
				if((par_port_B_byte & 0x08) == 0x00){
					printf("\t\t\t\t\t\tFailed!\n");
					printf("The test can only be successful between two Sangoma\n");
					printf("S508/FT1 units configured the SAME WAY\n");

					break;
				}
			} /* end of for */
			break;
		}
    		loop_counter++;	
    	} /* end of for */	
} /* FT1_rt_test */


void chdlc_router_up_time( void )
{
     	unsigned long time;
     	cb.cblock.command= CPIPE_ROUTER_UP_TIME;
     	cb.cblock.buffer_length = 0;
     	cb.data[0] = 0;
     	DoCommand(&cb);
     
     	time = *(unsigned long*)&cb.data[0];

     	if (time < 3600) {
		if (time<60) 
     			printf("    Router UP Time:  %ld seconds\n", time);
		else
     			printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     	}else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
}

void main(int argv, char* argc[])
{
	int proceed;

   	printf("\n");
   	if( argv > 2 ) {
     
		init( argv, argc);
     		proceed = MakeConnection();
     		
		if(proceed == TRUE){
      			switch( argc[1][0] ) {
      				case 'm':
	 				modem();
		 			break;
      				case 'i':
	 				global_stats();
	 				break;
      				case 'e':
	 				comm_stats();
	 				break;
      				case 'v':
	 				read_code_version();
	 				break;
      				case 'c':
	 				chdlc_configuration();
	 				break;
      				case 'x':
	 				link_status();
	 				break;
      				case 'l':
	 				operational_stats();
	 				break;
      				case 'p':
	 				slarp_stats();
	 				break;
     				case 'q':
	 				if( argc[1][1] == 'r' )
	   					raw_data = TRUE;
	 				line_trace();
	 				break;
      				case 'u':
	 				chdlc_router_up_time();
		 			break;
      				case 'f':
	 				if( argv > 2 ) {
	    					switch( argc[2][0] ) {
	    					case 'l':
	       						flush_operational_stats();
	      		 				operational_stats();
	       						break;
	    					case 'p':
	       						flush_operational_stats();
	      		 				slarp_stats();
	       						break;
	    					case 'i':
	       						flush_global_stats();
	       						global_stats();
	      		 				break;
	    					case 'e':
	       						flush_comm_stats();
	       						comm_stats();
	      		 				break;
    					default:
	       						usage();
	    					} //switch
	 				} else {
	    					usage();
	 				}//if
	 				break;
      				case 't':
      					if( argv > 2){
           					switch( argc[2][0] ){
	   					case 'v':
	     						set_FT1_monitor_status(0x01);
	     						if(!fail){
								view_FT1_status();
	    						 }
							set_FT1_monitor_status(0x00);
	     						break;
	  	 				case 's':
	     						set_FT1_monitor_status(0x01);
	     						if(!fail){	 	
								FT1_self_test();
	    	 					}
							set_FT1_monitor_status(0x00);
	     						break;
           					case 'l':
	     						set_FT1_monitor_status(0x01);
	     						if(!fail){
	    							FT1_ll_test();
             						}
							set_FT1_monitor_status(0x00);
             						break;
           					case 'd':
             						set_FT1_monitor_status(0x01);
	     						if(!fail){
								FT1_dl_test();
	     						}
							set_FT1_monitor_status(0x00);
             						break;
           					case 'r':
	     						set_FT1_monitor_status(0x01);
	     						if(!fail){
								FT1_rt_test();
	     						}
							set_FT1_monitor_status(0x00);
	     						break;
	   					case 'o':
	     						set_FT1_monitor_status(0x01);
	     						if(!fail){
								FT1_operational_mode();
	    		 				}
							set_FT1_monitor_status(0x00);
	     						break;
	   					default:
	     						usage();
	   					}
					} else{
  	   					usage();
					} 
        				break; 
      				default:
	 				usage();
      			} //switch
     		} 
     		close( sock );
   	} else {
      		usage();
   	} //if
   printf("\n");
}; //main

/*
 * EOF ppipemon.c
 */
