#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <termios.h>
#include <unistd.h>
#include <errno.h>
#include <utmp.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <grp.h>
#include <sys/types.h>
#include <ctype.h>

#include "shadow_.h"
#include "internal.h"




static const char login_usage[] =
    "login [OPTION]... [username] [ENV=VAR ...]\n\n"
    "Begin a new session on the system.\n\n"
    "Options:\n"
    "\t-f\tDo not perform authentication (user already authenticated\n"
    "\t-h\tName of the remote host for this login.\n"
    "\t-p\tPreserve environment.\n";


// login defines
#define LOGIN_PROMPT "\n%s login: "


/* From env.c */
extern char **newenvp;
extern size_t newenvc;


/* Stuff global to this file */
struct utmp utent;
struct passwd pwent;


static void check_nologin();
static int check_tty();
static int is_my_tty();
static void login_prompt();
static void motd();
static int pw_auth();
static int set_uid_gid();

static void alarm_handler()
{
    fprintf(stderr, "\nLogin timed out after %d seconds.\n", TIMEOUT);
    exit(0);
}




extern int 
login_main(int argc, char **argv)
{
    char name[32];
    char tty[BUFSIZ];
    char full_tty[200];
    char fromhost[512];
    char *host = "";
    char *cp = NULL;
    char *tmp;
    int amroot;
    int flag;
    int fflg = 0, hflg = 0, pflg = 0;
    int failed;
    struct passwd *pwd;
    struct spwd *spwd = NULL;
    char **envp = environ;
    initenv();
    name[0] = '\0';
    amroot = (getuid() == 0);
    signal(SIGALRM, alarm_handler);
    while ((flag = getopt(argc, argv, "f:h:d:p")) != EOF) {
	switch (flag) {
	case 'p':
	    pflg++;
	    break;
	case 'f':
	    /*
	     * username must be a seperate token
	     * (-f root, *NOT* -froot). --marekm
	     */
	    if (optarg != argv[optind - 1]) {
		usage(login_usage);
	    }
	    if (!amroot) {	/* Auth bypass only if real UID is zero */
		fprintf(stderr, "login: -f permission denied\n");
		exit(1);
	    }
	    fflg++;
	    STRFCPY(name, optarg);
	    break;
	case 'h':
	    hflg++;
	    host = optarg;
	    break;
	default:
	    usage(login_usage);
	}
    }
    if (!isatty(0) || !isatty(1) || !isatty(2)) {
	exit(1);		/* Must be a terminal */
    }
    checkutmp(!amroot);
    STRFCPY(tty, utent.ut_line);
    if (amroot) {
	bzero(utent.ut_host, sizeof utent.ut_host);
    }
    if (hflg) {
	strncpy(utent.ut_host, host, sizeof(utent.ut_host));
	cp = host;
    }
    openlog("login", LOG_PID | LOG_CONS | LOG_NOWAIT, LOG_AUTH);
    if (pflg) {
	while (*envp) {
	    addenv(*envp++, NULL);
	}
    }
    if (!pflg && (tmp = getenv("TERM"))) {
	addenv("TERM", tmp);
    }
    if (optind < argc) {
	STRFCPY(name, argv[optind]);
	optind++;
    }
    if (optind < argc) {	// Set command line variables
	set_env(argc - optind, &argv[optind]);
    }
    if (cp != NULL) {
	snprintf(fromhost, sizeof(fromhost), " on `%.100s' from `%.200s'",
		 tty, cp);
    } else {
	snprintf(fromhost, sizeof(fromhost), " on `%.100s'", tty);
    }
    if (TIMEOUT > 0) {
	alarm(TIMEOUT);
    }
    environ = newenvp;
    while (1) {
	failed = 0;
	if (!name[0]) {
	    login_prompt(LOGIN_PROMPT, name, sizeof name);
	}
	if (!(pwd = getpwnam(name))) {
	    pwent.pw_name = name;
	    pwent.pw_passwd = "!";
	    pwent.pw_shell = "/bin/sh";
	    fflg = 0;
	    failed = 1;
	} else {
	    pwent = *pwd;
	}
	spwd = NULL;
	if (pwd
	    && ((strcmp(pwd->pw_passwd, "x") == 0)
		|| (strcmp(pwd->pw_passwd, "*") == 0))) {
	    spwd = getspnam(name);
	    if (spwd) {
		pwent.pw_passwd = spwd->sp_pwdp;
	    } else {
		syslog(LOG_WARNING, "no shadow password for `%s'%s\n",
		       name, fromhost);
	    }
	}
	if (pwent.pw_passwd[0] == '!' || pwent.pw_passwd[0] == '*') {
	    failed = 1;
	}
	if (fflg) {
	    fflg--;
	    goto auth_ok;
	}
	if ((pwent.pw_uid == 0) && (check_tty(tty))) {
	    failed = 1;
	}
	if (pwent.pw_passwd[0] == '\0') {
	    goto auth_ok;
	}
	if (pw_auth(pwent.pw_passwd, name) == 0) {
	    goto auth_ok;
	}
	syslog(LOG_WARNING, "invalid password for `%s'%s\n",
	       pwd ? name : "UNKNOWN", fromhost);
	failed = 1;
      auth_ok:
	if (!failed) {
	    break;
	}
	if (pwent.pw_passwd[0] == '\0') {
	    pw_auth("!", name);
	}
	bzero(name, sizeof name);
	sleep(FAIL_DELAY);
	puts("Login incorrect");
    }
    (void) alarm(0);
    check_nologin();
    if (getenv("IFS")) {
	addenv("IFS= \t\n", NULL);
    }
    setutmp(name, tty);
    if (*tty != '/') {
	snprintf(full_tty, sizeof full_tty, "/dev/%s", tty);
    } else {
	strncpy(full_tty, tty, sizeof full_tty);
    }
    if (!is_my_tty(full_tty)) {
	syslog(LOG_ERR, "unable to determine TTY name, got %s\n",
	       full_tty);
    }
    if (chown(full_tty, pwent.pw_uid, pwent.pw_gid)
	|| chmod(full_tty, 0600)) {
	perror("unable to change TTY");
	syslog(LOG_ERR, "unable to change tty `%s' for user `%s'\n", tty,
	       pwent.pw_name);
	closelog();
	exit(1);
    }
    if (set_uid_gid() != 0) {
	exit(1);
    }
    setup_env(&pwent);
    motd();
    signal(SIGINT, SIG_DFL);	/* default interrupt signal */
    signal(SIGQUIT, SIG_DFL);	/* default quit signal */
    signal(SIGTERM, SIG_DFL);	/* default terminate signal */
    signal(SIGALRM, SIG_DFL);	/* default alarm signal */
    signal(SIGHUP, SIG_DFL);	/* added this.  --marekm */
    endpwent();			/* stop access to password file */
    endgrent();			/* stop access to group file */
    endspent();			/* stop access to shadow passwd file */
//  endsgent();                    /* stop access to shadow group file */
    if (pwent.pw_uid == 0) {
	syslog(LOG_INFO, "root login from `%s'\n", fromhost);
    }
    closelog();
    shell(pwent.pw_shell, (char *) 0);	/* exec the shell finally. */
     /*NOTREACHED*/ return (0);
}





static void check_nologin()
{
    if (access(NOLOGIN_FILE, F_OK) == 0) {
	FILE *nlfp;
	int c;
	if ((nlfp = fopen(NOLOGIN_FILE, "r"))) {
	    while ((c = getc(nlfp)) != EOF) {
		if (c == '\n')
		    putchar('\r');
		putchar(c);
	    }
	    fflush(stdout);
	    fclose(nlfp);
	} else {
	    printf("\r\nSystem closed for routine maintenance.\r\n");
	}
	if (pwent.pw_uid != 0) {
	    closelog();
	    exit(0);
	}
	printf("\r\n[Disconnect bypassed -- root login allowed.]\r\n");
    }
}

static int check_tty(const char *tty)
{
    FILE *fp;
    int i;
    char buf[BUFSIZ];
    if ((fp = fopen("/etc/securetty", "r")) == NULL) {
	syslog(LOG_WARNING, "cannot open securetty file.\n");
	return 0;
    }
    while (fgets(buf, sizeof(buf), fp) != NULL) {
	for (i = strlen(buf) - 1; i >= 0; --i) {
	    if (!isspace(buf[i])) {
		break;
	    }
	}
	buf[++i] = '\0';
	if (buf[0] == '\0' || buf[0] == '#') {
	    continue;
	}
	if (strcmp(buf, tty) == 0) {
	    fclose(fp);
	    return 0;
	}
    }
    fclose(fp);
    return 1;
}

static int is_my_tty(const char *tty)
{
    struct stat by_name, by_fd;
    if (stat(tty, &by_name) || fstat(0, &by_fd)) {
	return 0;
    }
    if (by_name.st_rdev != by_fd.st_rdev) {
	return 0;
    } else {
	return 1;
    }
}

static void login_prompt(const char *prompt, char *name, int namesize)
{
    char buf[1024];
    char *cp;
    int i;
    void (*sigquit) ();
    sigquit = signal(SIGQUIT, _exit);
    if (prompt) {
	gethostname(buf, sizeof buf);
	printf(prompt, buf);
	fflush(stdout);
    }
    bzero(buf, sizeof buf);
    if (fgets(buf, sizeof buf, stdin) != buf) {
	exit(1);
    }
    cp = strchr(buf, '\n');
    if (!cp) {
	exit(1);
    }
    *cp = '\0';

    for (cp = buf; *cp == ' ' || *cp == '\t'; cp++);
    for (i = 0; i < namesize - 1 && isgraph(*cp); name[i++] = *cp++);
    while (isgraph(*cp)) {
	cp++;
    }
    if (*cp) {
	cp++;
    }
    name[i] = '\0';
    signal(SIGQUIT, sigquit);
}

static void motd()
{
    FILE *fp;
    register int c;
    if ((fp = fopen("/etc/motd", "r")) != NULL) {
	while ((c = getc(fp)) != EOF) {
	    putchar(c);
	}
	fclose(fp);
    }
}

static int pw_auth(const char *cipher, const char *user)
{
    char *clear = NULL;
    int retval;
    if (cipher == (char *) 0 || *cipher == '\0') {
	return 0;
    }
    clear = getpass("Password: ");
    if (!clear) {
	static char c[1];
	c[0] = '\0';
	clear = c;
    }
    retval = strcmp(pw_encrypt(clear, cipher), cipher);
    bzero(clear, strlen(clear));
    return retval;
}

static int set_uid_gid()
{
    if (initgroups(pwent.pw_name, pwent.pw_gid) == -1) {
	perror("initgroups");
	syslog(LOG_ERR, "initgroups failed for user `%s': %,\n",
	       pwent.pw_name);
	closelog();
	return -1;
    }
    if (setgid(pwent.pw_gid) == -1) {
	perror("setgid");
	syslog(LOG_ERR, "bad group ID `%d' for user `%s': %m\n",
	       pwent.pw_gid, pwent.pw_name);
	closelog();
	return -1;
    }
    if (setuid(pwent.pw_uid)) {
	perror("setuid");
	syslog(LOG_ERR, "bad user ID `%d' for user `%s': %m\n",
	       pwent.pw_uid, pwent.pw_name);
	closelog();
	return -1;
    }
    return 0;
}

