/* command interface to Pluto
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: whack.c,v 1.42 1999/04/11 00:44:24 dhr Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <getopt.h>
#include <assert.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "version.c"
#include "connections.h"
#include "whack.h"	/* need connections.h for struct end */

static void
help(void)
{
    fprintf(stderr,
	"Usage:\n\n"
	"all forms:"
	    " [--optionsfrom <filename>]"
	    " [--ctlbase <path>]"
	    "\n\n"
	"help: whack"
	    " [--help]"
	    " [--version]"
	    "\n\n"
	"connection: whack"
	    " --name <connection_name>"
	    " \\\n   "
	    " --host <ip-address>"
	    " [--ikeport <port-number>]"
	    " \\\n   "
	    "   "
	    " [--nexthop <ip-address>]"
	    " [--client <subnet>]"
	    " [--firewall]"
	    " \\\n   "
	    " --to"
	    " --host <ip-address>"
	    " [--ikeport <port-number>]"
	    " \\\n   "
	    "   "
	    " [--nexthop <ip-address>]"
	    " [--client <subnet>]"
	    " [--firewall]"
	    " \\\n   "
	    " [--encrypt]"
	    " [--authenticate]"
	    " [--tunnel]"
	    " [--pfs]"
	    " \\\n   "
	    " [--ikelifetime <seconds>]"
	    " [--ipseclifetime <seconds>]"
	    " \\\n   "
	    " [--reykeywindow <seconds>]"
	    " [--keyingtries <count>]"
	    "\n\n"
	"routing: whack"
	    " (--route | --unroute)"
	    " --name <connection_name>"
	    "\n\n"
	"initiation: whack"
	    " (--initiate | --terminate)"
	    " --name <connection_name>"
	    " [--asynchronous]"
	    "\n\n"
	"delete: whack"
	    " --delete"
	    " --name <connection_name>"
	    "\n\n"
#ifdef DEBUG
	"debug: whack"
	    " \\\n   "
	    " [--debug-none]"
	    " [--debug-all]"
	    " \\\n   "
	    " [--debug-raw]"
	    " [--debug-crypt]"
	    " [--debug-parsing]"
	    " \\\n   "
	    " [--debug-emitting]"
	    " [--debug-control]"
	    " [--debug-klips]"
	    "\n\n"
#endif
	"listen: whack"
	    " --listen"
	    "\n\n"
	"status: whack"
	    " --status"
	    "\n\n"
	"shutdown: whack"
	    " --shutdown"
	    "\n\n"
	"FreeS/WAN %s\n",
	freeswan_version);
}

/* print a string as a diagnostic, then a usage message, then exits whack */
static void
diag(const char *mess)
{
    if (mess != NULL && *mess != '\0')
	fprintf(stderr, "whack: %s\n", mess);

    fprintf(stderr, "whack: for usage information, use --help\n");
    exit(mess == NULL? 0 : RC_WHACK_PROBLEM);
}

/* conditially calls diag; prints second arg as quoted string */
static void
diagq(const char *ugh, const char *this)
{
    if (ugh != NULL)
    {
	char buf[100];

	snprintf(buf, sizeof(buf), "%s \"%s\"", ugh, this);
	diag(buf);
    }
}

/* complex combined operands return one of these enumerated values
 * Note: we are close to the limit of 32 flags in an unsigned long!
 */
enum {
    OPT_CTLBASE,
    OPT_NAME,

    OPT_TO,
    OPT_HOST,	/* first per-end */
    OPT_IKEPORT,
    OPT_NEXTHOP,
    OPT_CLIENT,
    OPT_FIREWALL,	/* last per-end */

    OPT_ENCRYPT,	/* same order as POLICY_* */
    OPT_AUTHENTICATE,	/* same order as POLICY_* */
    OPT_TUNNEL,	/* same order as POLICY_* */
    OPT_PFS,	/* same order as POLICY_* */

    OPT_IKELIFETIME,
    OPT_IPSECLIFETIME,
    OPT_RKWINDOW,
    OPT_KTRIES,

    OPT_ROUTE,
    OPT_UNROUTE,

    OPT_INITIATE,
    OPT_TERMINATE,
    OPT_DELETE,
    OPT_LISTEN,
    OPT_STATUS,
    OPT_SHUTDOWN,

    OPT_ASYNC

#ifdef DEBUG	/* must be last so others are less than 32 to fit in lset_t */
    ,
    OPT_DEBUG_NONE,
    OPT_DEBUG_ALL,

    OPT_DEBUG_RAW,	/* same order as DBG_* */
    OPT_DEBUG_CRYPT,	/* same order as DBG_* */
    OPT_DEBUG_PARSING,	/* same order as DBG_* */
    OPT_DEBUG_EMITTING,	/* same order as DBG_* */
    OPT_DEBUG_CONTROL,	/* same order as DBG_* */
    OPT_DEBUG_KLIPS	/* same order as DBG_* */
#endif
};

#define OPTION_OFFSET	256	/* to get out of the way of letter options */

static const struct option long_opts[] = {
    /* name, has_arg, flag, val */

    { "help", no_argument, NULL, 'h' },
    { "version", no_argument, NULL, 'v' },
    { "optionsfrom", required_argument, NULL, '+' },

    { "ctlbase", required_argument, NULL, OPT_CTLBASE + OPTION_OFFSET },
    { "name", required_argument, NULL, OPT_NAME + OPTION_OFFSET },

    { "to", no_argument, NULL, OPT_TO + OPTION_OFFSET },

    { "host", required_argument, NULL, OPT_HOST + OPTION_OFFSET },
    { "ikeport", required_argument, NULL, OPT_IKEPORT + OPTION_OFFSET },
    { "nexthop", required_argument, NULL, OPT_NEXTHOP + OPTION_OFFSET },
    { "client", required_argument, NULL, OPT_CLIENT + OPTION_OFFSET },
    { "firewall", no_argument, NULL, OPT_FIREWALL + OPTION_OFFSET },

    { "encrypt", no_argument, NULL, OPT_ENCRYPT + OPTION_OFFSET },
    { "authenticate", no_argument, NULL, OPT_AUTHENTICATE + OPTION_OFFSET },
    { "tunnel", no_argument, NULL, OPT_TUNNEL + OPTION_OFFSET },
    { "pfs", no_argument, NULL, OPT_PFS + OPTION_OFFSET },

    { "ikelifetime", required_argument, NULL, OPT_IKELIFETIME + OPTION_OFFSET },
    { "ipseclifetime", required_argument, NULL, OPT_IPSECLIFETIME + OPTION_OFFSET },
    { "rekeywindow", required_argument, NULL, OPT_RKWINDOW + OPTION_OFFSET },
    { "keyingtries", required_argument, NULL, OPT_KTRIES + OPTION_OFFSET },

    { "route", no_argument, NULL, OPT_ROUTE + OPTION_OFFSET },
    { "unroute", no_argument, NULL, OPT_UNROUTE + OPTION_OFFSET },

    { "initiate", no_argument, NULL, OPT_INITIATE + OPTION_OFFSET },
    { "terminate", no_argument, NULL, OPT_TERMINATE + OPTION_OFFSET },
    { "delete", no_argument, NULL, OPT_DELETE + OPTION_OFFSET },
    { "listen", no_argument, NULL, OPT_LISTEN + OPTION_OFFSET },
    { "status", no_argument, NULL, OPT_STATUS + OPTION_OFFSET },
    { "shutdown", no_argument, NULL, OPT_SHUTDOWN + OPTION_OFFSET },

    { "asynchronous", no_argument, NULL, OPT_ASYNC + OPTION_OFFSET },

#ifdef DEBUG
    { "debug-none", no_argument, NULL, OPT_DEBUG_NONE + OPTION_OFFSET },
    { "debug-all]", no_argument, NULL, OPT_DEBUG_ALL + OPTION_OFFSET },
    { "debug-raw", no_argument, NULL, OPT_DEBUG_RAW + OPTION_OFFSET },
    { "debug-crypt", no_argument, NULL, OPT_DEBUG_CRYPT + OPTION_OFFSET },
    { "debug-parsing", no_argument, NULL, OPT_DEBUG_PARSING + OPTION_OFFSET },
    { "debug-emitting", no_argument, NULL, OPT_DEBUG_EMITTING + OPTION_OFFSET },
    { "debug-control", no_argument, NULL, OPT_DEBUG_CONTROL + OPTION_OFFSET },
    { "debug-klips", no_argument, NULL, OPT_DEBUG_KLIPS + OPTION_OFFSET },
#endif
    { 0,0,0,0 }
};

struct sockaddr_un ctl_addr = { AF_UNIX, DEFAULT_CTLBASE CTL_SUFFIX };

/* This is a hack for initiating ISAKMP exchanges. */

int
main(int argc, char **argv)
{
    struct whack_message msg;
    lset_t
	opts_seen = LEMPTY,
	pre_to_opts_seen;
    memset(&msg, '\0', sizeof(msg));
    msg.magic = WHACK_MAGIC;
    msg.right.host_port = IKE_UDP_PORT;

    msg.sa_ike_life_seconds = OAKLEY_ISAKMP_SA_LIFETIME_DEFAULT;
    msg.sa_ipsec_life_seconds = SA_LIFE_DURATION_DEFAULT;
    msg.sa_rekey_window = SA_REPLACEMENT_WINDOW_DEFAULT;
    msg.sa_keying_tries = SA_REPLACEMENT_RETRIES_DEFAULT;

    for (;;)
    {
	int long_index;
	unsigned long opt_whole;	/* numeric argument for some flags */

	int c;

	/* Note: we don't like the way short options get parsed
	 * by getopt_long, so we simply pass an empty string as
	 * the list.  It could be "hp:d:c:o:eatfs" "NARXPECK".
	 */
	c = getopt_long(argc, argv, "", long_opts, &long_index) - OPTION_OFFSET;

	/* all of our non-letter flags except OPT_DEBUG_* get
	 * added to opts_seen.  The OP_DEBUG_* exception is because
	 * we have too many to fit in a 32-bit string!
	 */

	if (0 <= c
#ifdef DEBUG
	&& c < OPT_DEBUG_NONE
#endif
	)
	{
	    lset_t f = LELEM(c);

	    if (opts_seen & f)
		diagq("duplicated flag", long_opts[long_index].name);
	    opts_seen |= f;

	    /* decode a numeric argument, if expected */
	    if (f & (LELEM(OPT_IKEPORT)
	    | LELEM(OPT_IKELIFETIME) | LELEM(OPT_IPSECLIFETIME)
	    | LELEM(OPT_RKWINDOW) | LELEM(OPT_KTRIES)))
	    {
		char *endptr;
		opt_whole = strtoul(optarg, &endptr, 0);

		if (*endptr != '\0' || endptr == optarg)
		    diagq("badly formed numeric argument", optarg);
	    }
	}

	/* Note: "breaking" from case terminates loop */
	switch (c)
	{
	case EOF - OPTION_OFFSET:	/* end of flags */
	    break;

	case 0 - OPTION_OFFSET: /* long option already handled */
	    continue;

	case ':' - OPTION_OFFSET:	/* diagnostic already printed by getopt_long */
	case '?' - OPTION_OFFSET:	/* diagnostic already printed by getopt_long */
	    diag("");	/* print null diagnostic, but exit sadly */
	    break;	/* not actually reached */

	case 'h' - OPTION_OFFSET:	/* --help */
	    help();
	    return 0;	/* GNU coding standards say to stop here */

	case 'v' - OPTION_OFFSET:	/* --version */
	    fprintf(stderr, "FreeS/WAN %s\n", freeswan_version);
	    return 0;	/* GNU coding standards say to stop here */

	case '+' - OPTION_OFFSET:	/* --optionsfrom <filename> */
	    optionsfrom(optarg, &argc, &argv, optind, stderr);
	    /* does not return on error */
	    continue;

	/* the rest of the options combine in complex ways */

	case OPT_CTLBASE:	/* --port <ctlbase> */
	    if (snprintf(ctl_addr.sun_path, sizeof(ctl_addr.sun_path)
	    , "%s%s", optarg, CTL_SUFFIX) == -1)
		diag("<ctlbase>" CTL_SUFFIX " must be fit in a sun_addr");
	    continue;

	case OPT_NAME:	/* --name <connection-name> */
	    if (strlen(optarg) >= CONNECTION_NAME_LIMIT)
		diagq("connection name too long", optarg);
	    strcpy(msg.name, optarg);
	    continue;

	case OPT_HOST:	/* --host <ip-address> */
	    diagq(atoaddr(optarg, 0, &msg.right.host), optarg);
	    continue;

	case OPT_IKEPORT:	/* --ikeport <port-number> */
	    if (opt_whole<=0 || opt_whole >= 0x10000)
		diagq("<port-number> must be a number between 1 and 65535", optarg);
	    msg.right.host_port = opt_whole;
	    continue;

	case OPT_NEXTHOP:	/* --nexthop <ip-address> */
	    diagq(atoaddr(optarg, 0, &msg.right.host_nexthop), optarg);
	    continue;

	case OPT_CLIENT:	/* --client <subnet> */
	    diagq(atosubnet(optarg, 0, &msg.right.client_net, &msg.right.client_mask), optarg);
	    msg.right.has_client = TRUE;
	    continue;

	case OPT_FIREWALL:
	    msg.right.firewall = TRUE;
	    continue;


	case OPT_TO:		/* --to */
	    /* process right end, move it to left, reset it */
	    if (!LALLIN(opts_seen, LELEM(OPT_HOST)))
		diag("connection missing --host (before --to)");
	    msg.left = msg.right;
	    memset(&msg.right, '\0', sizeof(msg.right));
	    msg.right.host_port = IKE_UDP_PORT;
	    pre_to_opts_seen = opts_seen;
	    opts_seen = (opts_seen & ~LRANGE(OPT_HOST,OPT_FIREWALL));
	    continue;

	case OPT_ENCRYPT:	/* --encrypt */
	case OPT_AUTHENTICATE:	/* --authenticate */
	case OPT_TUNNEL:	/* --tunnel */
	case OPT_PFS:	/* -- pfs */
	    msg.policy |= LELEM(c - OPT_ENCRYPT);
	    continue;

	case OPT_IKELIFETIME:	/* --ikelifetime <seconds> */
	    msg.sa_ike_life_seconds = opt_whole;
	    continue;

	case OPT_IPSECLIFETIME:	/* --ipseclifetime <seconds> */
	    msg.sa_ipsec_life_seconds = opt_whole;
	    continue;

	case OPT_RKWINDOW:	/* --rekeywindow <seconds> */
	    msg.sa_rekey_window = opt_whole;
	    continue;

	case OPT_KTRIES:	/* --keyingtries <count> */
	    msg.sa_keying_tries = opt_whole;
	    continue;

	case OPT_ROUTE:	/* --route */
	    msg.whack_route = TRUE;
	    continue;

	case OPT_UNROUTE:	/* --unroute */
	    msg.whack_unroute = TRUE;
	    continue;

	case OPT_INITIATE:	/* --initiate */
	    msg.whack_initiate = TRUE;
	    continue;

	case OPT_TERMINATE:	/* --terminate */
	    msg.whack_terminate = TRUE;
	    continue;

	case OPT_DELETE:	/* --delete */
	    msg.whack_delete = TRUE;
	    continue;

	case OPT_LISTEN:	/* --listen */
	    msg.whack_listen = TRUE;
	    continue;

	case OPT_STATUS:	/* --status */
	    msg.whack_status = TRUE;
	    continue;

	case OPT_SHUTDOWN:	/* --shutdown */
	    msg.whack_shutdown = TRUE;
	    continue;

	case OPT_ASYNC:
	    msg.whack_async = TRUE;
	    continue;

#ifdef DEBUG
	case OPT_DEBUG_NONE:	/* --debug-none */
	    msg.whack_options = TRUE;
	    msg.debugging = DBG_NONE;
	    continue;

	case OPT_DEBUG_ALL:	/* --debug-all */
	    msg.whack_options = TRUE;
	    msg.debugging = DBG_ALL;
	    continue;

	case OPT_DEBUG_RAW:	/* --debug-raw */
	case OPT_DEBUG_CRYPT:	/* --debug-crypt */
	case OPT_DEBUG_PARSING:	/* --debug-parsing */
	case OPT_DEBUG_EMITTING:	/* --debug-emitting */
	case OPT_DEBUG_CONTROL:	/* --debug-control */
	case OPT_DEBUG_KLIPS:	/* --debug-klips */
	    msg.whack_options = TRUE;
	    msg.debugging |= LELEM(c-DBG_RAW);
	    continue;
#endif
	default:
	    assert(FALSE);	/* unknown return value */
	}
	break;
    }

    if (optind != argc)
	diagq("unexpected argument", argv[optind]);

    /* For each possible form of the command, figure out if an argument
     * suggests whether that form was intended, and if so, whether all
     * required information was supplied.
     */

    if (opts_seen & LRANGE(OPT_TO,OPT_KTRIES))
    {
	if (!LALLIN(opts_seen, LELEM(OPT_HOST)))
	    diag("connection missing --host (after --to)");

	if ((pre_to_opts_seen & LELEM(OPT_NEXTHOP)) == 0)
	{
	    if (msg.right.host.s_addr == 0)
		diag("left nexthop must be specified when right is Road Warrior");
	    msg.left.host_nexthop = msg.right.host;
	}
	if ((opts_seen & LELEM(OPT_NEXTHOP)) == 0)
	{
	    if (msg.left.host.s_addr == 0)
		diag("right nexthop must be specified when left is Road Warrior");
	    msg.right.host_nexthop = msg.left.host;
	}

	msg.whack_connection = TRUE;
    }

    if (opts_seen & (LELEM(OPT_ROUTE) | LELEM(OPT_UNROUTE)
    | LELEM(OPT_INITIATE) | LELEM(OPT_TERMINATE)
    | LELEM(OPT_DELETE) | LELEM(OPT_TO)))
    {
	if ((opts_seen & LELEM(OPT_NAME)) == 0)
	    diag("missing --name <connection_name>");
    }
    else
    {
	if ((opts_seen & LELEM(OPT_NAME)) != 0)
	    diag("no reason for --name");
    }

    if (!(msg.whack_connection || msg.whack_delete
    || msg.whack_initiate || msg.whack_terminate
    || msg.whack_route || msg.whack_unroute
    || msg.whack_listen || msg.whack_status
    || msg.whack_options || msg.whack_shutdown))
	    diag(NULL);	/* nothing specified -- suggest help */

    if (msg.sa_ike_life_seconds <= msg.sa_rekey_window)
	diag("ikelifetime must be greater than rekeywindow");

    if (msg.sa_ipsec_life_seconds <= msg.sa_rekey_window)
	diag("ipseclifetime must be greater than rekeywindow");

    /* send message to Pluto */
    {
	int sock = socket(AF_UNIX, SOCK_STREAM, 0);
	int exit_status = 0;

	if (sock == -1)
	{
	    int e = errno;

	    fprintf(stderr, "whack: socket() failed (%d %s)\n", e, strerror(e));
	    exit(RC_WHACK_PROBLEM);
	}

	if (connect(sock, (struct sockaddr *)&ctl_addr
	, offsetof(struct sockaddr_un, sun_path) + strlen(ctl_addr.sun_path)) < 0)
	{
	    int e = errno;

	    fprintf(stderr, "whack: connect() for %s failed (%d %s)\n"
		, ctl_addr.sun_path, e, strerror(e));
	    exit(RC_WHACK_PROBLEM);
	}

	if (write(sock, &msg, sizeof(msg)) != sizeof(msg))
	{
	    int e = errno;

	    fprintf(stderr, "whack: write() failed (%d %s)\n", e, strerror(e));
	    exit(RC_WHACK_PROBLEM);
	}

	/* for now, just copy reply back to stdout */

	for (;;)
	{
	    char buf[513];
	    ssize_t len = read(sock, buf, sizeof(buf)-1);

	    if (len < 0)
	    {
		int e = errno;

		fprintf(stderr, "whack: read() failed (%d %s)\n", e, strerror(e));
		exit(RC_WHACK_PROBLEM);
	    }
	    if (len == 0)
		break;

	    /* figure out prefix number
	     * and how it should affect our exit status
	     */
	    buf[len] = '\0';
	    {
		unsigned long s = strtoul(buf, NULL, 10);

		switch (s)
		{
		case RC_COMMENT:
		case RC_LOG:
		    /* ignore */
		    break;
		case RC_SUCCESS:
		    /* be happy */
		    exit_status = 0;
		    break;
		default:
		    /* pass through */
		    exit_status = s;
		    break;
		}
	    }

	    write(1, buf, len);
	}
	return exit_status;
    }
}
