/* randomness machinery
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: rnd.c,v 1.9 1999/04/11 00:44:21 dhr Exp $
 */

#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <fcntl.h>

#include "sha1.h"
#include "constants.h"
#include "defs.h"
#include "rnd.h"
#include "log.h"
#include "timer.h"

#ifdef linux
static int       linux_random;
#endif

#define RANDOM_POOL_SIZE   20
static u_char random_pool[RANDOM_POOL_SIZE];
static volatile sig_atomic_t i, j, k;

/*
 * Signal handler.
 */
static void
rnd_handler(int ignore_me)
{
    k <<= 1;        /* Shift left by 1 */
    j++;
    k |= (i & 0x1); /* Get lsbit of counter */

    if (j != 8)
	signal(SIGVTALRM, rnd_handler);
}

/*
 * Do the clock skew trick.
 */
static u_char
generate_rnd_byte(void)
{
    struct itimerval tmval, ntmval;
#ifdef linux
    u_char c;
#endif

/*
 * #ifdef linux
 *  int mask = siggetmask();
 *
 *  mask |= SIGVTALRM;
 *  sigsetmask(mask);
 * #endif
 */

#ifdef linux
    if (read(linux_random, &c, sizeof(c)) == -1)
	exit_log_errno((e, "read() failed in get_rnd_byte()"));

    return c;
#endif

    i = 0;
    j = 0;

    ntmval.it_interval.tv_sec = 0;
    ntmval.it_interval.tv_usec = 1;
    ntmval.it_value.tv_sec = 0;
    ntmval.it_value.tv_usec = 1;
    signal(SIGVTALRM, rnd_handler);
    setitimer(ITIMER_VIRTUAL, &ntmval, &tmval);

    while (j != 8)
	i++;

    setitimer(ITIMER_VIRTUAL, &tmval, &ntmval);
    signal(SIGVTALRM, SIG_IGN);

/*
 * #ifdef linux
 *  mask ^= SIGVTALRM;
 *  sigsetmask(mask);
 * #endif
 */
    return k;
}

/*
 * Get a single random byte.
 */
static u_char
get_rnd_byte(void)
{
    SHA1_CTX ctx;

    SHA1Init(&ctx);

    random_pool[RANDOM_POOL_SIZE - 1] = generate_rnd_byte();
    random_pool[0] = generate_rnd_byte();

    SHA1Update(&ctx, random_pool, RANDOM_POOL_SIZE);
    SHA1Final(random_pool, &ctx);

    return random_pool[0];
}

/*
 * A random number generator based on clock skew. I first heard of
 * the idea by John Ioannidis, who heard it from Matt Blaze and/or
 * Jack Lacy. Seems to work.
 * If we're on OpenBSD, use the arc4random() generator instead; if
 * on Linux, use /dev/random.
 */
void
get_rnd_bytes(u_char *buffer, int length)
{
    int i;
#ifdef __OpenBSD__
    for (i = 0; i < length; i++)
	buffer[i] = arc4random() % 256;
#else
    for (i = 0; i < length; i++)
	buffer[i] = get_rnd_byte();
#endif
}

/*
 * Initialize the random pool.
 */
void
init_rnd_pool(void)
{
    SHA1_CTX ctx;

    SHA1Init(&ctx);

#ifdef linux
    DBG(DBG_KLIPS, DBG_log("opening %s", LINUX_RANDOM_FILE));
    linux_random = open(LINUX_RANDOM_FILE, O_RDONLY);
    if (linux_random == -1)
	exit_log_errno((e, "open() failed in init_rnd_pool()"));
#endif

    get_rnd_bytes(random_pool, RANDOM_POOL_SIZE);

    SHA1Update(&ctx, random_pool, RANDOM_POOL_SIZE);
    SHA1Final(random_pool, &ctx);
}

u_char    secret_of_the_day[SECRET_VALUE_LENGTH];

void
init_secret(void)
{
    /*
     * Generate the secret value for responder cookies, and
     * schedule an event for refresh.
     */
    get_rnd_bytes(secret_of_the_day, SECRET_VALUE_LENGTH);
    event_schedule(EVENT_REINIT_SECRET, EVENT_REINIT_SECRET_DELAY, NULL);
}
