/* IPsec DOI and Oakley resolution routines
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: ipsec_doi.c,v 1.57 1999/04/11 00:44:18 dhr Exp $
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>
#include <des.h>

#include "constants.h"
#include "defs.h"
#include "state.h"
#include "connections.h"
#include "preshared.h"
#include "packet.h"
#include "demux.h"	/* needs packet.h */
#include "kernel.h"
#include "log.h"
#include "cookie.h"
#include "server.h"
#include "spdb.h"
#include "timer.h"
#include "rnd.h"
#include "ipsec_doi.h"	/* needs demux.h and state.h */
#include "whack.h"

#include "sha1.h"
#include "md5.h"
#include "crypto.h" /* requires sha1.h and md5.h */

static void quick_outI1(int whack_sock, struct state *phase1_st
    , struct connection *c, lset_t policy, unsigned long klives);

/* pad_up(n, m) is the amount to add to n to make it a multiple of m */
#define pad_up(n, m) (((m) - 1) - (((n) + (m) - 1) % (m)))

/* var := mod(base ** exp, mod), ensuring var is mpz_inited */
#define mpz_init_powm(flag, var, base, exp, mod) { \
    if (!(flag)) \
	mpz_init(&(var)); \
    (flag) = TRUE; \
    mpz_powm(&(var), &(base), &(exp), (mod)); \
    }

/* Convert MP_INT to network form (binary octets, big-endian).
 * We do the malloc; caller must eventually do free.
 */
static void
mpz_to_n(u_char **rptr, size_t *rlen, const MP_INT *mp)
{
    MP_INT temp1, temp2;
    size_t len = (mpz_sizeinbase(mp, 16) + 1) / 2;	/* can't ask for base 256 */
    u_char *p = alloc_bytes(len, "host representation of large integer");
    int i;

    mpz_init(&temp1);
    mpz_init(&temp2);

    mpz_set(&temp1, mp);

    for (i = len-1; i >= 0; i--)
    {
	p[i] = mpz_mdivmod_ui(&temp2, NULL, &temp1, 1 << BITS_PER_BYTE);
	mpz_set(&temp1, &temp2);
    }

    mpz_clear(&temp1);
    mpz_clear(&temp2);

    *rptr = p;
    *rlen = len;
}

/* Convert network form (binary bytes, big-endian) to MP_INT
 * The flag indicates whether mp is already mpz_inited; we set it.
 */
static void
n_to_mpz(u_int8_t *flag, MP_INT *mp, const u_char *nbytes, size_t nlen)
{
    size_t i;

    if (*flag)
	mpz_clear(mp);
    *flag = TRUE;
    mpz_init_set_ui(mp, 0);

    for (i = 0; i != nlen; i++)
    {
	mpz_mul_ui(mp, mp, 1 << BITS_PER_BYTE);
	mpz_add_ui(mp, mp, nbytes[i]);
    }
}

/* Dump number in network order hex */

#ifdef DEBUG
static void
DBG_dump_mpz(const char *label, const MP_INT *mp)
{
    u_char *p;
    size_t l;

    mpz_to_n(&p, &l, mp);

    DBG_dump(label, p, l);
    pfree(p);
}
#endif

/* Figure out which interface / IP address we are using for this exchange.
 * It used to be that we didn't know until we get a message
 * from our peer (in main_inI1_outR1 or main_inR1_outI2)!
 */
static void
glean_myidentity(struct state *st)
{
    replace(st->st_myidentity, clone_thing(st->st_connection->this.host
	, "st_myidentity in glean_myidentity1()"));
    st->st_myidentity_len = sizeof(st->st_connection->this.host);
    st->st_myidentity_type = ID_IPV4_ADDR;
    DBG(DBG_CONTROL, DBG_log("my identity is %s"
	, inet_ntoa(st->st_connection->this.host)));
}

/*
 * Send a notification to the peer. We could make a decision on
 * whether to send the notification, based on the type and the
 * destination, if we care to.
 * XXX It doesn't handle DELETE notifications (which are also
 * XXX informational exchanges).
 */
#if 0 /* not currently used */
//static void
//send_notification(int sock,
//    u_int16_t type,
//    u_char *spi,
//    u_char spilen,
//    u_char protoid,
//    u_char *icookie,
//    u_char *rcookie,
//    msgid_t /*network order*/ msgid,
//    struct sockaddr sa)
//{
//    u_char buffer[sizeof(struct isakmp_hdr) +
//		 sizeof(struct isakmp_notification)];
//    struct isakmp_hdr *isa = (struct isakmp_hdr *) buffer;
//    struct isakmp_notification *isan = (struct isakmp_notification *)
//				       (buffer + sizeof(struct isakmp_hdr));
//
//    memset(buffer, '\0', sizeof(struct isakmp_hdr) +
//	  sizeof(struct isakmp_notification));
//
//    if (icookie != (u_char *) NULL)
//	memcpy(isa->isa_icookie, icookie, COOKIE_SIZE);
//
//    if (rcookie != (u_char *) NULL)
//	memcpy(isa->isa_rcookie, rcookie, COOKIE_SIZE);
//
//    /* Standard header */
//    isa->isa_np = ISAKMP_NEXT_N;
//    isa->isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
//    isa->isa_xchg = ISAKMP_XCHG_INFO;
//    isa->isa_msgid = msgid;
//    isa->isa_length = htonl(sizeof(struct isakmp_hdr) +
//			    sizeof(struct isakmp_notification) +
//			    spilen);
//
//    /* Notification header */
//    isan->isan_type = htons(type);
//    isan->isan_doi = htonl(ISAKMP_DOI_IPSEC);
//    isan->isan_length = htons(sizeof(struct isakmp_notification) + spilen);
//    isan->isan_spisize = spilen;
//    memcpy((u_char *)isan + sizeof(struct isakmp_notification), spi, spilen);
//    isan->isan_protoid = protoid;
//
//    DBG(DBG_CONTROL, DBG_log("sending INFO type %s to %s",
//	enum_show(&notification_names, type),
//	show_sa(&sa)));
//
//    if (sendto(sock, buffer, ntohl(isa->isa_length), 0, &sa,
//	       sizeof(sa)) != ntohl(isa->isa_length))
//	log_errno((e, "sendto() failed in send_notification() to %s",
//	    show_sa(&sa)));
//    else
//    {
//	DBG(DBG_CONTROL, DBG_log("transmitted %d bytes", ntohl(isa->isa_length)));
//    }
//}
#endif /* not currently used */

/* The whole message must be a multiple of 4 octets.
 * I'm not sure where this is spelled out, but look at
 * rfc2408 3.6 Transform Payload.
 * Note: it talks about 4 BYTE boundaries!
 */
static void
close_message(pb_stream *pbs)
{
    size_t padding =  pad_up(pbs_offset(pbs), 4);

    if (padding != 0)
	(void) out_zero(padding, pbs, "message padding");
    close_output_pbs(pbs);
}

/* Initiate an Oakley exchange.
 * --> HDR;SA
 */
static void
main_outI1(
	int whack_sock,
	struct connection *c,
	bool pending_quick,
	lset_t policy,
	unsigned long klives)
{
    u_char space[8192];	/* NOTE: we assume 8192 is big enough to build the packet */
    pb_stream reply_pbs;	/* not actually a reply, but you know what I mean */
    pb_stream r_hdr_pbs;

    struct isakmp_hdr hdr;
    struct state *st;

    /* set up new state */
    cur_state = st = new_state();

    st->st_connection = c;
    st->st_pending_quick = pending_quick;
    st->st_policy = policy;
    st->st_whack_sock = whack_sock;
    st->st_klives = klives;
    st->st_state = STATE_MAIN_I1;

    get_cookie(ISAKMP_INITIATOR, st->st_icookie, COOKIE_SIZE, c->that.host);

    insert_state(st);	/* needs cookies, connection, and msgid (0) */

    /* an event will be scheduled for st before we return */

    log("initiating Main Mode");

    /* set up reply_pbs */
    init_pbs(&reply_pbs, space, sizeof(space), "reply packet");

    /* HDR out */

    memset(&hdr, '\0', sizeof(hdr));	/* default to 0 */

    hdr.isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
    hdr.isa_np = ISAKMP_NEXT_SA;
    hdr.isa_xchg = ISAKMP_XCHG_IDPROT;
    memcpy(hdr.isa_icookie, st->st_icookie, COOKIE_SIZE);
    /* R-cookie, flags and MessageID are left zero */

    if (!out_struct(&hdr, &isakmp_hdr_desc, &reply_pbs, &r_hdr_pbs))
	passert(FALSE);

    /* SA out */
    {
	u_char *sa_start = r_hdr_pbs.cur;

	if (!out_sa(&r_hdr_pbs, &oakley_sadb, st, TRUE, ISAKMP_NEXT_NONE))
	    passert(FALSE);

	/* save initiator SA for later HASH */
	passert(st->st_p1isa == NULL);	/* no leak! */
	st->st_p1isa_len = r_hdr_pbs.cur - sa_start;
	st->st_p1isa = clone_bytes(sa_start, st->st_p1isa_len, "sa in main_outI1");
    }

    close_message(&r_hdr_pbs);
    close_output_pbs(&reply_pbs);

    st->st_tpacket = clone_bytes(reply_pbs.start, pbs_offset(&reply_pbs), "reply packet for main_outI1");
    st->st_tpacket_len = pbs_offset(&reply_pbs);

    /* Transmit */

    DBG_cond_dump(DBG_RAW, "sending:\n", st->st_tpacket, st->st_tpacket_len);

    send_packet(st, "main_outI1");

    /* Set up a retransmission event, half a minute henceforth */
    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st);

    whack_log(st->st_whack_sock, RC_NEW_STATE + STATE_MAIN_I1
	, "%s: initiate", enum_name(&state_names, st->st_state));
    cur_state = NULL;
}

void
ipsecdoi_initiate(
	int whack_sock,
	struct connection *c,
	bool pending_quick,
	lset_t policy,
	unsigned long klives)
{
    /* If there's already an ISAKMP SA established, use that and
     * go directly to Quick Mode.
     * XXX If an ISAKMP SA is *being* established, we foolishly
     * XXX try to establish another one, in parallel.  We could
     * XXX issue an event to wait for it to finish and then try
     * XXX to establish it.
     */
    struct state *st = find_phase1_state(c->that.host);

    if (st == NULL)
    {
	main_outI1(whack_sock, c, pending_quick, policy, klives);
    }
    else if (pending_quick)
    {
	/* ??? we assume that peer_nexthop_sin isn't important:
	 * we already have it from when we negotiated the ISAKMP SA!
	 */
	quick_outI1(whack_sock, st, c, policy, klives);
    }
    else if (whack_sock != NULL_FD)
    {
	close(whack_sock);
    }
}

/* Replace SA with a fresh one that is similar
 *
 * Shares some logic with ipsecdoi_initiate, but not the same!
 * - we must not reuse the ISAKMP SA if we are trying to replace it!
 * - if trying to replace IPSEC SA, use ipsecdoi_initiate to build
 *   ISAKMP SA if needed.
 * - duplicate whack fd, if live.
 * Does not delete the old state -- someone else will do that.
 */
void
ipsecdoi_replace(struct state *st, unsigned long klives)
{
    int whack_sock = st->st_whack_sock == NULL_FD? NULL_FD : dup(st->st_whack_sock);
    lset_t policy = st->st_policy;

    if (IS_PHASE1(st->st_state))
    {
	main_outI1(whack_sock, st->st_connection, st->st_pending_quick
	    , policy, klives);
    }
    else
    {
	/* Add features of actual old state to policy.  This ensures
	 * that rekeying doesn't downgrade security.  I admit that
	 * this doesn't capture everything.
	 */
	if (st->st_pfs_group != OAKLEY_GROUP_NONE)
	    policy |= POLICY_PFS;
	if (st->st_ah.present)
	{
	    policy |= POLICY_AUTHENTICATE;
	    if (st->st_ah.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
		policy |= POLICY_TUNNEL;
	}
	if (st->st_esp.present && st->st_esp.attrs.transid != ESP_NULL)
	{
	    policy |= POLICY_ENCRYPT;
	    if (st->st_esp.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
		policy |= POLICY_TUNNEL;
	}
	ipsecdoi_initiate(whack_sock, st->st_connection, TRUE, policy, klives);
    }
}


/*
 * SKEYID for preshared keys.
 */
static void
skeyid_preshared(struct state *st, const u_char *id, int length)
{
    struct hmac_ctx ctx;

    hmac_init(&ctx, st->st_oakley.hasher, id, length);

    replace(st->st_skeyid, alloc_bytes(ctx.hmac_digest_len, "st_skeyid in skeyid_preshared()"));
    st->st_skeyid_len = ctx.hmac_digest_len;

    hmac_update(&ctx, st->st_ni, st->st_ni_len);
    hmac_update(&ctx, st->st_nr, st->st_nr_len);
    hmac_final(st->st_skeyid, &ctx);
}


/*
 * Generate the SKEYID_* and new IV
 */
static void
generate_skeyids_iv(struct state *st)
{
    switch (st->st_oakley.auth)
    {
	case OAKLEY_PRESHARED_KEY:
	    /* Generate the SKEYID */
	    {
		size_t len;
		const u_char *id = get_preshared_secret(st, &len);

		if (id == NULL)
		    log("preshared secret disappeared!");
		skeyid_preshared(st, id, len);
	    }
	    break;

	case OAKLEY_DSS_SIG:
	case OAKLEY_RSA_SIG:
	    /* XXX */

	case OAKLEY_RSA_ENC:
	case OAKLEY_RSA_ENC_REV:
	    /* XXX */

	default:
	    exit_log("generate_skeyids_iv(): unsupported authentication method %s",
		enum_show(&oakley_auth_names, st->st_oakley.auth));
    }

    {
	u_char *gxy;
	size_t gxylen;
	struct hmac_ctx ctx;

	/* Get the g^xy for HMAC'ing */
	mpz_to_n(&gxy, &gxylen, &st->st_shared);

	DBG(DBG_CRYPT, DBG_log("size of g^xy is %d", gxylen));

	hmac_init(&ctx, st->st_oakley.hasher, st->st_skeyid, st->st_skeyid_len);

	st->st_skeyid_d_len = ctx.hmac_digest_len;
	replace(st->st_skeyid_d, alloc_bytes(ctx.hmac_digest_len, "st_skeyid_d in generate_skeyids_iv()"));

	st->st_skeyid_a_len = ctx.hmac_digest_len;
	replace(st->st_skeyid_a, alloc_bytes(ctx.hmac_digest_len, "st_skeyid_a in generate_skeyids_iv()"));

	st->st_skeyid_e_len = ctx.hmac_digest_len;
	replace(st->st_skeyid_e, alloc_bytes(ctx.hmac_digest_len, "st_skeyid_e in generate_skeyids_iv()"));

	/* SKEYID_D */
	hmac_update(&ctx, gxy, gxylen);
	hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
	hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
	hmac_update(&ctx, "\0", 1);
	hmac_final(st->st_skeyid_d, &ctx);

	/* SKEYID_A */
	hmac_reinit(&ctx);
	hmac_update(&ctx, st->st_skeyid_d, st->st_skeyid_d_len);
	hmac_update(&ctx, gxy, gxylen);
	hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
	hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
	hmac_update(&ctx, "\1", 1);
	hmac_final(st->st_skeyid_a, &ctx);

	/* SKEYID_E */
	hmac_reinit(&ctx);
	hmac_update(&ctx, st->st_skeyid_a, st->st_skeyid_a_len);
	hmac_update(&ctx, gxy, gxylen);
	hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
	hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
	hmac_update(&ctx, "\2", 1);
	hmac_final(st->st_skeyid_e, &ctx);

	pfree(gxy);
    }

    /* IV */
    {
	u_char *gi, *gr;
	size_t gilen, grlen;
	union hash_ctx hash_ctx;
	const struct hash_desc *h = st->st_oakley.hasher;

	mpz_to_n(&gi, &gilen, &st->st_gi);
	mpz_to_n(&gr, &grlen, &st->st_gr);

	st->st_new_iv_len = h->hash_digest_len;
	passert(st->st_new_iv_len <= sizeof(st->st_new_iv));

	h->hash_init(&hash_ctx);
	h->hash_update(&hash_ctx, gi, gilen);
	h->hash_update(&hash_ctx, gr, grlen);
	h->hash_final(st->st_new_iv, &hash_ctx);

	pfree(gi);
	pfree(gr);
    }

    /* Oakley Keying Material
     * Derived from Skeyid_e: if it is not big enough, generate more
     * using the PRF.
     * See draft-ietf-ipsec-isakmp-oakley-07.txt Appendix B
     */
    {
	const size_t keysize = st->st_oakley.encrypter->keysize;

	st->st_enc_key_len = keysize;
	if (keysize <= st->st_skeyid_e_len)
	{
	    replace(st->st_enc_key, clone_bytes(st->st_skeyid_e, keysize, "st_enc_key 1"));
	}
	else
	{
	    u_char keytemp[MAX_OAKLEY_KEY_LEN + MAX_DIGEST_LEN];
	    struct hmac_ctx ctx;
	    int i = 0;

	    hmac_init(&ctx, st->st_oakley.hasher, st->st_skeyid_e, st->st_skeyid_e_len);
	    hmac_update(&ctx, "\0", 1);
	    for (;;)
	    {
		hmac_final(&keytemp[i], &ctx);
		i += ctx.hmac_digest_len;
		if (i >= keysize)
		    break;
		hmac_reinit(&ctx);
		hmac_update(&ctx, &keytemp[i - ctx.hmac_digest_len], ctx.hmac_digest_len);
	    }
	    replace(st->st_enc_key, clone_bytes(keytemp, keysize, "st_enc_key 2"));
	}
    }

    DBG(DBG_CRYPT,
	DBG_dump("Skeyid:  ", st->st_skeyid, st->st_skeyid_len);
	DBG_dump("Skeyid_d:", st->st_skeyid_d, st->st_skeyid_d_len);
	DBG_dump("Skeyid_a:", st->st_skeyid_a, st->st_skeyid_a_len);
	DBG_dump("Skeyid_e:", st->st_skeyid_e, st->st_skeyid_e_len);
	DBG_dump("enc key:", st->st_enc_key, st->st_enc_key_len);
	DBG_dump("IV:", st->st_new_iv, st->st_new_iv_len));
}

/* Generate HASH_I or HASH_R for ISAKMP Phase I.
 * This will *not* generate other hash payloads (eg. Phase II of Quick Mode,
 * New Group Mode, or ISAKMP Informational Exchanges).
 * If the hashi argument is TRUE, generate HASH_I; if FALSE generate HASH_R.
 * If hashus argument is TRUE, we're generating a hash for our end.
 */
static void
main_mode_hash(struct state *st, u_char *hash_val, int *hash_len, bool hashi, bool hashus)
{
    struct hmac_ctx ctx;

    hmac_init(&ctx, st->st_oakley.hasher, st->st_skeyid, st->st_skeyid_len);

    *hash_len = ctx.hmac_digest_len;

    {
	u_char *gi, *gr;
	size_t gilen, grlen;

	mpz_to_n(&gi, &gilen, &st->st_gi);
	mpz_to_n(&gr, &grlen, &st->st_gr);
	if (hashi)
	{
	    hmac_update(&ctx, gi, gilen);
	    hmac_update(&ctx, gr, grlen);
	    hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
	    hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
	}
	else
	{
	    hmac_update(&ctx, gr, grlen);
	    hmac_update(&ctx, gi, gilen);
	    hmac_update(&ctx, st->st_rcookie, COOKIE_SIZE);
	    hmac_update(&ctx, st->st_icookie, COOKIE_SIZE);
	}
	pfree(gi);
	pfree(gr);
    }

    DBG(DBG_CRYPT, DBG_log("hashing %d bytes of SA",
	    st->st_p1isa_len - sizeof(struct isakmp_generic)));

    /* SA_b */
    hmac_update(&ctx, st->st_p1isa + sizeof(struct isakmp_generic),
	 st->st_p1isa_len - sizeof(struct isakmp_generic));

    /* IDio_b (o stands for originator: i or r) */
    {
	/* hash identification payload, without generic payload header */
	struct isakmp_ipsec_id id;	/* body used, in network order! */

	memset(&id, '\0', sizeof(id));

	if (hashus) {
	    id.isaiid_idtype = st->st_myidentity_type;
	    /* leave protoid and port empty (zero) */
	} else {
	    id.isaiid_idtype = st->st_peeridentity_type;
	    id.isaiid_protoid = st->st_peeridentity_protocol;
	    id.isaiid_port = htons(st->st_peeridentity_port);
	}
	DBG(DBG_CRYPT,
	    DBG_log("Hashing %s ID: Type %s, Protocol %d, Port %d",
		hashus? "my" : "his",
		enum_show(&ident_names, id.isaiid_idtype),
		id.isaiid_protoid, htons(id.isaiid_port)));

	/* NOTE: hash does NOT include the generic payload part of
	 * Identity Payload
	 */
	hmac_update(&ctx,
	    (u_char *)&id + sizeof(struct isakmp_generic),
	    sizeof(id) - sizeof(struct isakmp_generic));
    }

    if (hashus)
	hmac_update(&ctx, st->st_myidentity, st->st_myidentity_len);
    else
	hmac_update(&ctx, st->st_peeridentity, st->st_peeridentity_len);

    hmac_final(hash_val, &ctx);
}

/* encrypt message, sans fixed part of header
 * IV is fetched from st->st_new_iv and stored into st->st_iv.
 * The theory is that there will be no "backing out", so we commit to IV.
 * We also close the pbs.
 */
static bool
encrypt_message(pb_stream *pbs, struct state *st)
{
    const struct encrypt_desc *e = st->st_oakley.encrypter;
    u_int8_t *enc_start = pbs->start + sizeof(struct isakmp_hdr);
    size_t enc_len = pbs_offset(pbs) - sizeof(struct isakmp_hdr);

    DBG_cond_dump(DBG_CRYPT | DBG_RAW, "encrypting:\n", enc_start, enc_len);

    /* pad up to multiple of encryption blocksize */
    {
	size_t padding = pad_up(enc_len, e->blocksize);

	if (padding != 0)
	{
	    if (!out_zero(padding, pbs, "encryption padding"))
		return FALSE;
	    enc_len += padding;
	}
    }

    DBG(DBG_CRYPT, DBG_log("encrypting using %s", enum_show(&oakley_enc_names, st->st_oakley.encrypt)));

    e->crypt(TRUE, enc_start, enc_len, st);

    update_iv(st);
    DBG_cond_dump(DBG_CRYPT, "next IV:", st->st_iv, st->st_iv_len);
    close_message(pbs);
    return TRUE;
}

/* Compute HASH(1), HASH(2) of Quick Mode.
 * HASH(1) is part of Quick I1 message.
 * HASH(2) is part of Quick R1 message.
 * Used by: quick_outI1, quick_inI1_outR1 (twice), quick_inR1_outI2
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
static void
quick_mode_hash12(
    u_char *dest, size_t *dest_len,
    const u_char *start, const u_char *roof,
    const struct state *st, bool hash2)
{
    struct hmac_ctx ctx;

    hmac_init(&ctx, st->st_oakley.hasher, st->st_skeyid_a, st->st_skeyid_a_len);
    hmac_update(&ctx, (u_char *) &st->st_msgid, sizeof(st->st_msgid));
    if (hash2)
    {
	/* include Ni_b in the hash */
	hmac_update(&ctx, st->st_ni, st->st_ni_len);
    }
    hmac_update(&ctx, start, roof-start);
    hmac_final(dest, &ctx);

    if (dest_len != NULL)
	*dest_len = ctx.hmac_digest_len;

    DBG(DBG_CRYPT,
	DBG_log("HASH(%d) computed:", hash2 + 1);
	DBG_dump("", dest, ctx.hmac_digest_len));
}

/* Compute Phase 2 IV.
 * Uses Phase 1 IV from st_iv; puts result in st_new_iv.
 */
void
init_phase2_iv(struct state *st, const msgid_t *msgid)
{
    const struct hash_desc *h = st->st_oakley.hasher;
    union hash_ctx ctx;

    st->st_new_iv_len = h->hash_digest_len;
    passert(st->st_new_iv_len <= sizeof(st->st_new_iv));

    h->hash_init(&ctx);
    h->hash_update(&ctx, st->st_iv, st->st_iv_len);
    passert(*msgid != 0);
    h->hash_update(&ctx, (u_char *)msgid, sizeof(*msgid));
    h->hash_final(st->st_new_iv, &ctx);

    DBG_cond_dump(DBG_CRYPT, "computed Phase 2 IV:"
	, st->st_new_iv, st->st_new_iv_len);
}

/* Initiate quick mode.
 * --> HDR*, HASH(1), SA, Nr [, KE ] [, IDci, IDcr ]
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
static void
quick_outI1(
	int whack_sock,
	struct state *phase1_st,
	struct connection *c,
	lset_t policy,
	unsigned long klives)
{
    struct state *nst = duplicate_state(phase1_st);
    u_char space[8192];	/* NOTE: we assume 8192 is big enough to build the packet */
    pb_stream reply_pbs;	/* not really a reply */
    pb_stream r_hdr_pbs;
    u_char
	*r_hashval,	/* where in reply to jam hash value */
	*r_hash_start;	/* start of what is to be hashed */
    bool has_client = c->this.has_client ||  c->that.has_client;

    cur_state = nst;
    nst->st_whack_sock = whack_sock;
    nst->st_connection = c;
    nst->st_policy = policy;
    nst->st_klives = klives;

    nst->st_myuserprotoid = nst->st_peeruserprotoid = 0;
    nst->st_myuserport = nst->st_peeruserport = 0;

    nst->st_msgid = generate_msgid(c->that.host);
    nst->st_state = STATE_QUICK_I1;

    insert_state(nst);	/* needs cookies, connection, and msgid */

    /* an event will be scheduled for nst before we return */

    log("initiating Quick Mode %s", bitnamesof(sa_policy_bit_names, policy));

    /* set up reply_pbs */
    init_pbs(&reply_pbs, space, sizeof(space), "reply packet");

    /* HDR* */
    {
	struct isakmp_hdr hdr;

	hdr.isa_version = ISAKMP_MAJOR_VERSION << ISA_MAJ_SHIFT | ISAKMP_MINOR_VERSION;
	hdr.isa_np = ISAKMP_NEXT_HASH;
	hdr.isa_xchg = ISAKMP_XCHG_QUICK;
	hdr.isa_msgid = nst->st_msgid;
	hdr.isa_flags = ISAKMP_FLAG_ENCRYPTION;
	memcpy(hdr.isa_icookie, nst->st_icookie, COOKIE_SIZE);
	memcpy(hdr.isa_rcookie, nst->st_rcookie, COOKIE_SIZE);
	if (!out_struct(&hdr, &isakmp_hdr_desc, &reply_pbs, &r_hdr_pbs))
	    passert(FALSE);
    }

    /* HASH(1) -- space to be filled later */
    {
	pb_stream hash_pbs;

	if (!out_generic(ISAKMP_NEXT_SA, &isakmp_hash_desc, &r_hdr_pbs, &hash_pbs))
	    passert(FALSE);
	r_hashval = hash_pbs.cur;	/* remember where to plant value */
	if (!out_zero(nst->st_oakley.hasher->hash_digest_len, &hash_pbs, "HASH(1)"))
	    passert(FALSE);
	close_output_pbs(&hash_pbs);
	r_hash_start = r_hdr_pbs.cur;	/* hash from after HASH(1) */
    }

    /* SA */
    if (policy & POLICY_PFS)
    {
	/* same group as during Phase 1: since no negotiation is
	 * possible, we pick one that is very likely supported.
	 */
	nst->st_pfs_group = phase1_st->st_oakley.group;
	nst->st_pfs_modp_modulus = phase1_st->st_oakley.modp_modulus;
    }
    else
    {
	nst->st_pfs_group = OAKLEY_GROUP_NONE;
	nst->st_pfs_modp_modulus = NULL;
    }

    if (!out_sa(&r_hdr_pbs
    , &ipsec_sadb[policy & (POLICY_ENCRYPT | POLICY_AUTHENTICATE | POLICY_TUNNEL)]
    , nst, FALSE, ISAKMP_NEXT_NONCE))
	passert(FALSE);

    /* Ni */
    {
	pb_stream nonce_pbs;
	u_int8_t nonce_bytes[DEFAULTNONCESIZE];

	/* next payload depends on whether KE or ID payloads will be sent.
	 */
	if (!out_generic(policy & POLICY_PFS? ISAKMP_NEXT_KE : has_client? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE
	, &isakmp_nonce_desc, &r_hdr_pbs, &nonce_pbs))
	    passert(FALSE);
	get_rnd_bytes(nonce_bytes, DEFAULTNONCESIZE);
	if (!out_raw(nonce_bytes, DEFAULTNONCESIZE, &nonce_pbs, "nonce value"))
	    passert(FALSE);
	close_output_pbs(&nonce_pbs);

	/* Copy Ni into the state object */
	replace(nst->st_ni, clone_bytes(nonce_bytes, DEFAULTNONCESIZE,
	    "st_ni in quick_outI1()"));
	nst->st_ni_len = DEFAULTNONCESIZE;

	DBG_cond_dump(DBG_CRYPT, "Ni sent:", nonce_bytes, DEFAULTNONCESIZE);
    }

    /* [ KE ] (needed for PFS) */

    if (policy & POLICY_PFS)
    {
	u_char tmp[LOCALSECRETSIZE];
	pb_stream r_keyex_pbs;
	u_char *gi;
	size_t gilen;

	get_rnd_bytes(tmp, LOCALSECRETSIZE);
	n_to_mpz(&nst->st_sec_in_use, &nst->st_sec, tmp, LOCALSECRETSIZE);

	mpz_init_powm(nst->st_gi_in_use, nst->st_gi
	    , groupgenerator, nst->st_sec, nst->st_pfs_modp_modulus);

	DBG(DBG_CRYPT,
		DBG_dump_mpz("PFS Local secret:\n", &nst->st_sec);
		DBG_dump_mpz("PFS Public value sent:\n", &nst->st_gi));

	if (!out_generic(has_client? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE
	, &isakmp_keyex_desc, &r_hdr_pbs, &r_keyex_pbs))
	    passert(FALSE);

	mpz_to_n(&gi, &gilen, &nst->st_gi);

	if (!out_raw(gi, gilen, &r_keyex_pbs, "keyex value"))
	    passert(FALSE);

	pfree(gi);
	close_output_pbs(&r_keyex_pbs);
    }

    /* [ IDci, IDcr ] */
    if (has_client)
    {
	struct isakmp_ipsec_id id;
	pb_stream id_pbs;

	/* IDci (we are initiator) */
	id.isaiid_np = ISAKMP_NEXT_ID;
	id.isaiid_idtype = ID_IPV4_ADDR_SUBNET;
	id.isaiid_protoid = nst->st_myuserprotoid;
	id.isaiid_port = nst->st_myuserport;

	if (!out_struct(&id, &isakmp_ipsec_identification_desc, &r_hdr_pbs, &id_pbs))
	    passert(FALSE);
	if (!out_raw(&c->this.client_net
	, sizeof(c->this.client_net)
	, &id_pbs, "initiator's client network"))
	    passert(FALSE);
	if (!out_raw(&c->this.client_mask.s_addr
	, sizeof(c->this.client_mask.s_addr)
	, &id_pbs, "initiator's client mask"))
	    passert(FALSE);
	close_output_pbs(&id_pbs);

	/* IDcr (peer is responder) */
	id.isaiid_np = ISAKMP_NEXT_NONE;
	id.isaiid_idtype = ID_IPV4_ADDR_SUBNET;
	id.isaiid_protoid = nst->st_peeruserprotoid;
	id.isaiid_port = nst->st_peeruserport;
	if (!out_struct(&id, &isakmp_ipsec_identification_desc, &r_hdr_pbs, &id_pbs))
	    passert(FALSE);
	if (!out_raw(&c->that.client_net
	, sizeof(c->that.client_net)
	, &id_pbs, "peer's client network"))
	    passert(FALSE);
	if (!out_raw(&c->that.client_mask, sizeof(c->that.client_mask)
	, &id_pbs, "peer's client mask"))
	    passert(FALSE);
	close_output_pbs(&id_pbs);
    }

    /* finish computing HASH(1) */
    quick_mode_hash12(r_hashval, NULL, r_hash_start, r_hdr_pbs.cur, nst, FALSE);

    /* encrypt message, except for fixed part of header */

    init_phase2_iv(phase1_st, &nst->st_msgid);
    nst->st_new_iv_len = phase1_st->st_new_iv_len;
    memcpy(nst->st_new_iv, phase1_st->st_new_iv, nst->st_new_iv_len);

    if (!encrypt_message(&r_hdr_pbs, nst))
	passert(FALSE);

    /* save packet, now that we know its size */
    nst->st_tpacket = clone_bytes(reply_pbs.start, pbs_offset(&reply_pbs),
	"reply packet from quick_outI1");
    nst->st_tpacket_len = pbs_offset(&reply_pbs);

    /* send the packet */

    DBG_cond_dump(DBG_RAW, "sending:\n", nst->st_tpacket, nst->st_tpacket_len);

    send_packet(nst, "quick_outI1");

    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, nst);

    whack_log(nst->st_whack_sock, RC_NEW_STATE + STATE_QUICK_I1
	, "%s: initiate", enum_name(&state_names, nst->st_state));
    cur_state = NULL;
}

/* Decode the ID payload of Phase 1 (main_inI3_outR3 and main_inR3) */
static bool
decode_peer_id(struct msg_digest *md)
{
    struct state *const st = md->st;
    struct payload_digest *const id_pld = md->chain[ISAKMP_NEXT_ID];
    pb_stream *const id_pbs = &id_pld->pbs;
    struct isakmp_id *const id = &id_pld->payload.id;

    /* XXX Check for valid ID types? */
    switch (id->isaid_idtype)
    {
    case ID_IPV4_ADDR:
	/* I think that RFC2407 (IPSEC DOI) 4.6.2 is confused.
	 * It talks about the protocol ID and Port fields of the ID
	 * Payload, but they don't exist as such in Phase 1.
	 * We use more appropriate names.
	 * isaid_doi_specific_a is in place of Protocol ID.
	 * isaid_doi_specific_b is in place of Port.
	 * Besides, there is no good reason for allowing these to be
	 * other than 0 in Phase 1.
	 */
	if (!(id->isaid_doi_specific_a == 0 && id->isaid_doi_specific_b == 0)
	&& !(id->isaid_doi_specific_a == IPPROTO_UDP && id->isaid_doi_specific_b == IKE_UDP_PORT))
	{
	    log("protocol/port in Phase 1 ID payload must be 0/0 or %d/%d but are %d/%d"
		, IPPROTO_UDP, IKE_UDP_PORT
		, id->isaid_doi_specific_a, id->isaid_doi_specific_b);
	    return FALSE;
	}
	if (pbs_left(id_pbs) != sizeof(struct in_addr))
	{
	    log("size of ID_IPV4_ADDR identification should be %u but is %d in ID payload"
		, (unsigned) sizeof(struct in_addr)
		, (unsigned) pbs_left(id_pbs));
	    /* XXX Could send notification back */
	    return FALSE;
	}
	st->st_peeridentity_protocol = id->isaid_doi_specific_a;
	st->st_peeridentity_port = id->isaid_doi_specific_b;
	st->st_peeridentity_len = pbs_left(id_pbs);
	replace(st->st_peeridentity, clone_bytes(id_pbs->cur, st->st_peeridentity_len,
	    "st_peeridentity in decode_peer_id()"));

	DBG(DBG_PARSING,
	    DBG_log("Peer's ID type is %s: %s",
		enum_show(&ident_names, id->isaid_idtype),
		inet_ntoa(*(struct in_addr *)st->st_peeridentity)));
	break;

    default:
	/* XXX Could send notification back */
	log("Unacceptable identity type (%s) in ID payload"
	    , enum_show(&ident_names, id->isaid_idtype));
	return FALSE;
    }
    st->st_peeridentity_type = id->isaid_idtype;
    return TRUE;
}

/* Decode the variable part of an ID packet in (during Quick Mode).
 * This is designed for packets that identify clients, not peers.
 * Currently this will only accept two forms.
 */
static bool
decode_net_id(
    struct isakmp_ipsec_id *id,
    pb_stream *id_pbs,
    struct in_addr *net, struct in_addr *mask,
    const char *which, const struct in_addr from)
{
    switch (id->isaiid_idtype)
    {
	case ID_IPV4_ADDR:
	    if (pbs_left(id_pbs) != sizeof(*net))
	    {
		log("%s ID payload ID_IPV4_ADDR wrong length in Quick I1", which);
		/* XXX Could send notification back */
		return FALSE;
	    }
	    memcpy(net, id_pbs->cur, sizeof(*net));
	    *mask = mask32.sin_addr;
	    DBG(DBG_PARSING | DBG_CONTROL,
		DBG_log("%s is IP address %s", which, inet_ntoa(*net)));
	    break;

	case ID_IPV4_ADDR_SUBNET:
	    if (pbs_left(id_pbs) != sizeof(*net) + sizeof(*mask))
	    {
		log("%s ID payload ID_IPV4_ADDR_SUBNET wrong length in Quick I1"
		    , which);
		/* XXX Could send notification back */
		return FALSE;
	    }
	    memcpy(net, id_pbs->cur, sizeof(*net));
	    memcpy(mask, id_pbs->cur + sizeof(*net), sizeof(*mask));
	    DBG(DBG_PARSING | DBG_CONTROL,
		{
		    char buf[SUBNETTOA_BUF];

		    subnettoa(*net, *mask, 0, buf, sizeof(buf));
		    DBG_log("%s is IP subnet %s", which, buf);
		});
	    break;

	default:
	    /* XXX support more */
	    log("bad id type %s", enum_show(&ident_names, id->isaiid_idtype));
	    /* XXX Could send notification back */
	    return FALSE;
    }
    return TRUE;
}

/* like decode, but checks that what is received matches what was sent */
static bool
check_net_id(
    struct isakmp_ipsec_id *id,
    pb_stream *id_pbs,
    u_int8_t *protoid, u_int16_t *port,
    struct in_addr *net, struct in_addr *mask,
    const char *which, const struct in_addr from)
{
    struct in_addr net_temp, mask_temp;

    if (!decode_net_id(id, id_pbs, &net_temp, &mask_temp, which, from))
	return FALSE;

    if (net->s_addr != net_temp.s_addr
    || mask->s_addr != mask_temp.s_addr
    || *protoid != id->isaiid_protoid || *port != id->isaiid_port)
    {
	log("%s ID returned doesn't match my proposal", which);
	return FALSE;
    }
    return TRUE;
}

/*
 * Produce the new key material of Quick Mode.
 * draft-ietf-ipsec-isakmp-oakley-06.txt section 5.5
 * specifies how this is to be done.
 */
static void
compute_proto_keymat(
    struct state *st,
    u_int8_t protoid,
    struct ipsec_proto_info *pi)
{
    size_t needed_len; /* bytes of keying material needed */

    /* Add up the requirements for keying material
     * (It probably doesn't matter if we produce too much!)
     */
    switch (protoid)
    {
    case PROTO_IPSEC_ESP:
	    switch (pi->attrs.transid)
	    {
	    case ESP_NULL:
		needed_len = 0;
		break;
	    case ESP_DES:
		needed_len = DES_CBC_BLOCK_SIZE;
		break;
	    case ESP_3DES:
		needed_len = DES_CBC_BLOCK_SIZE * 3;
		break;
	    default:
		exit_log("transform %s not implemented yet",
		    enum_show(&esp_transformid_names, pi->attrs.transid));
	    }

	    switch (pi->attrs.auth)
	    {
	    case AUTH_ALGORITHM_NONE:
		break;
	    case AUTH_ALGORITHM_HMAC_MD5:
		needed_len += HMAC_MD5_KEY_LEN;
		break;
	    case AUTH_ALGORITHM_HMAC_SHA1:
		needed_len += HMAC_SHA1_KEY_LEN;
		break;
	    case AUTH_ALGORITHM_DES_MAC:
	    default:
		exit_log("AUTH algorithm %s not implemented yet",
		    enum_show(&auth_alg_names, pi->attrs.auth));
	    }
	    break;

    case PROTO_IPSEC_AH:
	    switch (pi->attrs.transid)
	    {
	    case AH_MD5:
		needed_len = HMAC_MD5_KEY_LEN;
		break;
	    case AH_SHA:
		needed_len = HMAC_SHA1_KEY_LEN;
		break;
	    default:
		exit_log("transform %s not implemented yet",
		    enum_show(&ah_transformid_names, pi->attrs.transid));
	    }
	    break;

    default:
	exit_log("protocol %s not implemented yet",
	    enum_show(&protocol_names, protoid));
	break;
    }

    pi->keymat_len = needed_len;

    /* Allocate space for the keying material.
     * Although only needed_len bytes are desired, we
     * must round up to a multiple of ctx.hmac_digest_len
     * so that our buffer isn't overrun.
     */
    {
	struct hmac_ctx ctx_me, ctx_peer;
	size_t needed_space;	/* space needed for keying material (rounded up) */
	size_t i;

	/* XXX PFS requires different hasher */
	hmac_init(&ctx_me, st->st_oakley.hasher, st->st_skeyid_d, st->st_skeyid_d_len);
	ctx_peer = ctx_me;	/* duplicate initial conditions */

	needed_space = needed_len + pad_up(needed_len, ctx_me.hmac_digest_len);
	replace(pi->our_keymat, alloc_bytes(needed_space, "keymat in compute_keymat()"));
	replace(pi->peer_keymat, alloc_bytes(needed_space, "peer_keymat in quick_inI1_outR1()"));

	for (i = 0;; )
	{
	    if (st->st_shared_in_use)
	    {
		/* PFS: include the g^xy */
		u_char *gxy;
		size_t gxylen;

		mpz_to_n(&gxy, &gxylen, &st->st_shared);
		hmac_update(&ctx_me, gxy, gxylen);
		hmac_update(&ctx_peer, gxy, gxylen);
		pfree(gxy);
	    }
	    hmac_update(&ctx_me, &protoid, sizeof(protoid));
	    hmac_update(&ctx_peer, &protoid, sizeof(protoid));

	    hmac_update(&ctx_me, (u_char *)&pi->our_spi, sizeof(pi->our_spi));
	    hmac_update(&ctx_peer, (u_char *)&pi->attrs.spi, sizeof(pi->attrs.spi));

	    hmac_update(&ctx_me, st->st_ni, st->st_ni_len);
	    hmac_update(&ctx_peer, st->st_ni, st->st_ni_len);

	    hmac_update(&ctx_me, st->st_nr, st->st_nr_len);
	    hmac_update(&ctx_peer, st->st_nr, st->st_nr_len);

	    hmac_final(pi->our_keymat + i, &ctx_me);
	    hmac_final(pi->peer_keymat + i, &ctx_peer);

	    i += ctx_me.hmac_digest_len;
	    if (i >= needed_space)
		break;

	    /* more keying material needed: prepare to go around again */

	    hmac_reinit(&ctx_me);
	    hmac_reinit(&ctx_peer);

	    hmac_update(&ctx_me, pi->our_keymat + i - ctx_me.hmac_digest_len,
		ctx_me.hmac_digest_len);
	    hmac_update(&ctx_peer, pi->peer_keymat + i - ctx_peer.hmac_digest_len,
		ctx_peer.hmac_digest_len);
	}
    }

    DBG(DBG_CRYPT,
	DBG_dump("KEYMAT computed:\n", pi->our_keymat, pi->keymat_len);
	DBG_dump("Peer KEYMAT computed:\n", pi->peer_keymat, pi->keymat_len));
}

static void
compute_keymats(struct state *st)
{
    if (st->st_ah.present)
	compute_proto_keymat(st, PROTO_IPSEC_AH, &st->st_ah);
    if (st->st_esp.present)
	compute_proto_keymat(st, PROTO_IPSEC_ESP, &st->st_esp);
}

/* Compute HASH(3) in Quick Mode (part of quick I2 message).
 * Used by: quick_inR1_outI2, quick_inI2
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
static void
quick_mode_hash3(u_char *dest, size_t *destlen, struct state *st)
{
    struct hmac_ctx ctx;

    hmac_init(&ctx, st->st_oakley.hasher, st->st_skeyid_a, st->st_skeyid_a_len);
    hmac_update(&ctx, "\0", 1);
    hmac_update(&ctx, (u_char *) &st->st_msgid, sizeof(st->st_msgid));
    hmac_update(&ctx, st->st_ni, st->st_ni_len);
    hmac_update(&ctx, st->st_nr, st->st_nr_len);
    hmac_final(dest, &ctx);

    if (destlen != NULL)
	*destlen = ctx.hmac_digest_len;

    DBG_cond_dump(DBG_CRYPT, "HASH(3) computed:", dest, ctx.hmac_digest_len);
}

/* State Transition Functions.
 * - Called from comm_handle;
 * - state_check[state].processor points to these
 * - these routines are in state-order
 */

/* Handle a Main Mode Oakley first packet (responder side).
 * HDR;SA --> HDR;SA
 */
stf_status
main_inI1_outR1(struct msg_digest *md)
{
    struct payload_digest *const sa_pd = md->chain[ISAKMP_NEXT_SA];
    struct state *st;
    struct connection *c = find_host_connection(md->iface->addr, md->sin.sin_addr);

    pb_stream r_hdr_pbs;
    pb_stream r_sa_pbs;

    if (c == NULL)
    {
#ifdef ROAD_WARRIOR_FUDGE
	/* see if a wildcarded connection can be found */
	c = find_host_connection(md->iface->addr, mask0.sin_addr);
	if (c != NULL)
	{
	    /* create a temporary connection that is a copy of this one */
	    c = rw_connection(c, md->sin.sin_addr);
	}
	else
#endif /* ROAD_WARRIOR_FUDGE */
	{
	    log("packet from %s but no connection has been authorized", inet_ntoa(md->sin.sin_addr));
	    /* XXX notification is in order! */
	    return STF_IGNORE;
	}
    }

    /* Set up state */
    cur_state = md->st = st = new_state();	/* (caller will reset) */
    st->st_connection = c;
    st->st_klives = 1;	/* not our job to try again from start */

    memcpy(st->st_icookie, md->hdr.isa_icookie, COOKIE_SIZE);
    get_cookie(ISAKMP_RESPONDER, st->st_rcookie, COOKIE_SIZE, md->sin.sin_addr);

    insert_state(st);	/* needs cookies, connection, and msgid (0) */

    /* put a very short fuse on this state object
     * in case things don't work out.
     */
    event_schedule(EVENT_SA_EXPIRE, 0, st);

    glean_myidentity(st);

    st->st_doi = ISAKMP_DOI_IPSEC;
    st->st_situation = SIT_IDENTITY_ONLY; /* We only support this */
    st->st_state = STATE_MAIN_R1;

#ifdef ROAD_WARRIOR_FUDGE
    if (c->rw_instance)
    {
	log("responding to Main Mode from Road Warrior %s"
	    , inet_ntoa(c->that.host));
    }
    else
#endif /* ROAD_WARRIOR_FUDGE */
    {
	log("responding to Main Mode");
    }

    /* parse_isakmp_sa also spits out a winning SA into our reply,
     * so we have to build our md->reply and emit HDR before calling it.
     */

    /* HDR out */
    {
	struct isakmp_hdr r_hdr = md->hdr;

	memcpy(r_hdr.isa_rcookie, st->st_rcookie, COOKIE_SIZE);
	r_hdr.isa_np = ISAKMP_NEXT_SA;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    passert(FALSE);
    }

    /* start of SA out */
    {
	struct isakmp_sa r_sa = sa_pd->payload.sa;

	r_sa.isasa_np = ISAKMP_NEXT_NONE;
	if (!out_struct(&r_sa, &isakmp_sa_desc, &r_hdr_pbs, &r_sa_pbs))
	    passert(FALSE);
    }

    /* SA body in and out */

    {
	notification_t r = parse_isakmp_sa_body(&sa_pd->pbs, &sa_pd->payload.sa, &r_sa_pbs
	    , FALSE, st, st->st_rcookie, st->st_icookie);

	if (r != NOTHING_WRONG)
	    return STF_FAIL + r;
    }

    close_message(&r_hdr_pbs);

    /* save initiator SA for HASH */
    passert(st->st_p1isa == NULL);	/* no leak! */
    st->st_p1isa_len = pbs_room(&sa_pd->pbs);
    replace(st->st_p1isa, clone_bytes(sa_pd->pbs.start, pbs_room(&sa_pd->pbs), "sa in main_inI1_outR1()"));

    return STF_REPLY;
}

/* Handle HDR;SA from responder (verify proposal),
 * and send back HDR;KE;Ni.
 */
stf_status
main_inR1_outI2(struct msg_digest *md)
{
    struct state *const st = md->st;
    pb_stream r_hdr_pbs;    /* reply message pbs */

    glean_myidentity(st);

    {
	struct payload_digest *const sapd = md->chain[ISAKMP_NEXT_SA];
	notification_t r = parse_isakmp_sa_body(&sapd->pbs
	    , &sapd->payload.sa, NULL, TRUE, st
	    , st->st_icookie, md->hdr.isa_rcookie);

	if (r != NOTHING_WRONG)
	    return STF_FAIL + r;
    }

    /* st->st_sec := localsecret */
    {
	u_char tmp[LOCALSECRETSIZE];

	get_rnd_bytes(tmp, LOCALSECRETSIZE);
	n_to_mpz(&st->st_sec_in_use, &st->st_sec, tmp, LOCALSECRETSIZE);
    }

    /* st->st_gi := groupgenerator ** st->st_sec  %  st->st_oakley.modp_modulus */
    mpz_init_powm(st->st_gi_in_use, st->st_gi, groupgenerator, st->st_sec, st->st_oakley.modp_modulus);

    DBG(DBG_CRYPT,
	DBG_dump_mpz("Local secret:", &st->st_sec);
	DBG_dump_mpz("Public value sent:", &st->st_gi));

    /**************** build output packet HDR;KE;Ni ****************/

    /* HDR */
    {
	struct isakmp_hdr r_hdr = md->hdr;	/* mostly same as incoming header */

	r_hdr.isa_np = ISAKMP_NEXT_KE;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    return STF_INTERNAL_ERROR;
    }

    /* KE */
    {
	pb_stream r_keyex_pbs;

	u_char *gi;
	size_t gilen;

	if (!out_generic(ISAKMP_NEXT_NONCE, &isakmp_keyex_desc, &r_hdr_pbs, &r_keyex_pbs))
	    return STF_INTERNAL_ERROR;

	mpz_to_n(&gi, &gilen, &st->st_gi);

	if (!out_raw(gi, gilen, &r_keyex_pbs, "keyex value"))
	    return STF_INTERNAL_ERROR;

	pfree(gi);
	close_output_pbs(&r_keyex_pbs);
    }

    /* Ni */
    {
	pb_stream nonce_pbs;
	u_int8_t nonce_bytes[DEFAULTNONCESIZE];

	if (!out_generic(ISAKMP_NEXT_NONE, &isakmp_nonce_desc, &r_hdr_pbs, &nonce_pbs))
	    return STF_INTERNAL_ERROR;

	get_rnd_bytes(nonce_bytes, DEFAULTNONCESIZE);
	if (!out_raw(nonce_bytes, DEFAULTNONCESIZE, &nonce_pbs, "nonce value"))
	    return STF_INTERNAL_ERROR;

	close_output_pbs(&nonce_pbs);

	/* Copy the Ni nonce into the state object */
	replace(st->st_ni, clone_bytes(nonce_bytes, DEFAULTNONCESIZE,
	    "st_ni in main_inR1_outI2()"));
	st->st_ni_len = DEFAULTNONCESIZE;
    }

    /* finish message */
    close_message(&r_hdr_pbs);

    /* Reinsert the state, using the responder cookie we just received */
    unhash_state(st);
    memcpy(st->st_rcookie, md->hdr.isa_rcookie, COOKIE_SIZE);
    insert_state(st);	/* needs cookies, connection, and msgid (0) */

    st->st_state = STATE_MAIN_I2;

    return STF_REPLY;
}

/* Handle HDR;KE;Ni from Initiator.  Send a HDR;KE;Nr back.
 */
stf_status
main_inI2_outR2(struct msg_digest *md)
{
    struct state *const st = md->st;

    pb_stream *keyex_pbs = &md->chain[ISAKMP_NEXT_KE]->pbs;
    pb_stream r_hdr_pbs;    /* reply message pbs */

    /* KE */

    if (pbs_left(keyex_pbs) < MINIMUM_PUBLIC_VALUE_SIZE)
    {
	log("too short public size (%d)", pbs_left(keyex_pbs));
	/* XXX Could send notification back */
	return STF_FAIL + INVALID_KEY_INFORMATION;
    }

    /* Ni */

    /* ??? should we check for a minimum or maximum length? */

    /* Decode Gi from network form */
    n_to_mpz(&st->st_gi_in_use, &st->st_gi, keyex_pbs->cur, pbs_left(keyex_pbs));

    DBG(DBG_CRYPT,
	DBG_dump_mpz("public value received:", &st->st_gi));

    /* st->st_sec := localsecret */
    {
	u_char tmp[LOCALSECRETSIZE];

	get_rnd_bytes(tmp, LOCALSECRETSIZE);
	n_to_mpz(&st->st_sec_in_use, &st->st_sec, tmp, LOCALSECRETSIZE);
    }

    /* st->st_gr := groupgenerator ** st->st_sec  %  st->st_oakley.modp_modulus */
    mpz_init_powm(st->st_gr_in_use, st->st_gr, groupgenerator, st->st_sec, st->st_oakley.modp_modulus);

    /* st->st_shared := st->st_gi ** st->st_sec  %  st->st_oakley.modp_modulus */
    mpz_init_powm(st->st_shared_in_use, st->st_shared, st->st_gi, st->st_sec, st->st_oakley.modp_modulus);

    DBG(DBG_CRYPT,
	DBG_dump_mpz("our secret value:", &st->st_sec);
	DBG_dump_mpz("our public value:", &st->st_gr);
	DBG_dump_mpz("shared secret:", &st->st_shared));

    /* Save Ni */
    {
        pb_stream *nonce_pbs = &md->chain[ISAKMP_NEXT_NONCE]->pbs;

	replace(st->st_ni, clone_bytes(nonce_pbs->cur, pbs_left(nonce_pbs),
	    "st_ni in main_inI2_outR2()"));
	st->st_ni_len = pbs_left(nonce_pbs);
    }

    /**************** build output packet HDR;KE;Nr ****************/

    /* HDR  */
    {
	struct isakmp_hdr r_hdr = md->hdr;	/* mostly same as incoming header */

	r_hdr.isa_np = ISAKMP_NEXT_KE;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    return STF_INTERNAL_ERROR;
    }

    /* KE */
    {
	pb_stream r_keyex_pbs;
	u_char *gr;
	size_t grlen;
	bool r;

	if (!out_generic(ISAKMP_NEXT_NONCE, &isakmp_keyex_desc, &r_hdr_pbs, &r_keyex_pbs))
	    return STF_INTERNAL_ERROR;

	mpz_to_n(&gr, &grlen, &st->st_gr);
	r = out_raw(gr, grlen, &r_keyex_pbs, "keyex value");
	pfree(gr);

	if (!r)
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&r_keyex_pbs);
    }

    /* Nr */
    {
	pb_stream r_nonce_pbs;
	u_int8_t nonce_bytes[DEFAULTNONCESIZE];

	if (!out_generic(ISAKMP_NEXT_NONE, &isakmp_nonce_desc, &r_hdr_pbs, &r_nonce_pbs))
	    return STF_INTERNAL_ERROR;
	get_rnd_bytes(nonce_bytes, DEFAULTNONCESIZE);
	if (!out_raw(nonce_bytes, DEFAULTNONCESIZE, &r_nonce_pbs, "nonce value"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&r_nonce_pbs);

	/* Copy Nr into the state object */
	replace(st->st_nr, clone_bytes(nonce_bytes, DEFAULTNONCESIZE,
	    "st_nr in main_inR1_outI2()"));
	st->st_nr_len = DEFAULTNONCESIZE;
    }

    /* finish message */
    close_message(&r_hdr_pbs);

    /* next message will be encrypted, but not this one */
    generate_skeyids_iv(st);
    update_iv(st);

    /* Advance state */
    st->st_state = STATE_MAIN_R2;

    return STF_REPLY;
}

/* Handle HDR;KE;Nr from responder.  Send a HDR*;IDii;HASH_I back.
 */
stf_status
main_inR2_outI3(struct msg_digest *md)
{
    struct state *const st = md->st;

    pb_stream *const keyex_pbs = &md->chain[ISAKMP_NEXT_KE]->pbs;
    pb_stream *const nonce_pbs = &md->chain[ISAKMP_NEXT_NONCE]->pbs;
    pb_stream r_hdr_pbs;    /* reply message pbs */

    if (pbs_left(keyex_pbs) < MINIMUM_PUBLIC_VALUE_SIZE)
    {
	log("too short public size (%d)", pbs_left(keyex_pbs));
	/* XXX Could send notification back */
	return STF_FAIL + INVALID_KEY_INFORMATION;
    }

    DBG_cond_dump(DBG_CRYPT, "Ni:", nonce_pbs->cur, pbs_left(nonce_pbs));

    /* done parsing */

    /* decode Gr from network form */
    n_to_mpz(&st->st_gr_in_use, &st->st_gr, keyex_pbs->cur, pbs_left(keyex_pbs));

    /* st->st_shared := st->st_gr ** st->st_sec  %  st->st_oakley.modp_modulus */
    mpz_init_powm(st->st_shared_in_use, st->st_shared, st->st_gr, st->st_sec, st->st_oakley.modp_modulus);

    DBG(DBG_CRYPT,
	DBG_dump_mpz("public value received:", &st->st_gr);
	DBG_dump_mpz("shared secret:", &st->st_shared));

    /* Save Nr */
    replace(st->st_nr, clone_bytes(nonce_pbs->cur, pbs_left(nonce_pbs),
	"st_nr in main_inR2_outI3()"));
    st->st_nr_len = pbs_left(nonce_pbs);

    /* Generate SKEYID, SKEYID_A, SKEYID_D, SKEYID_E, and st_new_iv  */
    generate_skeyids_iv(st);

    /**************** build output packet HDR*;IDii;HASH_I ****************/
    /* ??? NOTE: this is almost the same as main_inI3_outR3's code */

    /* HDR*  */
    {
	struct isakmp_hdr r_hdr = md->hdr;	/* mostly same as incoming header */

	r_hdr.isa_flags |= ISAKMP_FLAG_ENCRYPTION;
	r_hdr.isa_np = ISAKMP_NEXT_ID;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    return STF_INTERNAL_ERROR;
    }

    /* IDii */
    {
	struct isakmp_ipsec_id id;
	pb_stream id_pbs;

	id.isaiid_np = ISAKMP_NEXT_HASH;
	id.isaiid_idtype = st->st_myidentity_type;
	id.isaiid_protoid = 0;	/* ??? is this right?  IPPROTO_UDP? */
	id.isaiid_port = 0;	/* ??? is this right?  our_port? */
	if (!out_struct(&id, &isakmp_ipsec_identification_desc, &r_hdr_pbs, &id_pbs)
	|| !out_raw(st->st_myidentity, st->st_myidentity_len, &id_pbs, "my identity"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&id_pbs);
    }

    /* HASH_I */
    {
	u_char hash_val[MAX_DIGEST_LEN];
	int hash_len;
	pb_stream hash_pbs;

	main_mode_hash(st, hash_val, &hash_len, TRUE, TRUE);
	DBG_cond_dump(DBG_CRYPT, "HASH_I sent:", hash_val, hash_len);

	if (!out_generic(ISAKMP_NEXT_NONE, &isakmp_hash_desc, &r_hdr_pbs, &hash_pbs)
	|| !out_raw(hash_val, hash_len, &hash_pbs, "HASH_I"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&hash_pbs);
    }

    /* encrypt message, except for fixed part of header */

    /* st_new_iv was computed by generate_skeyids_iv */
    if (!encrypt_message(&r_hdr_pbs, st))
	return STF_INTERNAL_ERROR;	/* ??? we may be partly committed */

    /* Advance state */
    st->st_state = STATE_MAIN_I3;

    return STF_REPLY;
}

/*
 * Handle HDR*;IDii;HASH_I from initiator. Send a HDR*;IDir;HASH_R back.
 */
stf_status
main_inI3_outR3(struct msg_digest *md)
{
    struct state *const st = md->st;

    pb_stream r_hdr_pbs;

    /* input code similar to main_inR3 -- should be factored */

    /* IDii */
    if (!decode_peer_id(md))
	return STF_FAIL + INVALID_ID_INFORMATION;

    /* HASH_I */

    {
	pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs;
	u_char hash_val[MAX_DIGEST_LEN];
	int hash_len;

	main_mode_hash(st, hash_val, &hash_len, TRUE, FALSE);

	DBG(DBG_CRYPT,
	    DBG_dump("computed HASH_I:", hash_val, hash_len);
	    DBG_dump("received HASH_I:", hash_pbs->cur, pbs_left(hash_pbs)));

	if (pbs_left(hash_pbs) != hash_len
	|| memcmp(hash_pbs->cur, hash_val, hash_len) != 0)
	{
	    log("received HASH_I does not match computed value in main I3");
	    /* XXX Could send notification back */
	    return STF_FAIL + INVALID_HASH_INFORMATION;
	}
    }

    /**************** build output packet HDR*;IDir;HASH_R ****************/
    /* ??? NOTE: this is almost the same as main_inR2_outI3's code */

    /* HDR*  */
    {
	struct isakmp_hdr r_hdr = md->hdr;	/* mostly same as incoming header */

	r_hdr.isa_flags |= ISAKMP_FLAG_ENCRYPTION;
	r_hdr.isa_np = ISAKMP_NEXT_ID;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    return STF_INTERNAL_ERROR;
    }

    /* IDii */
    {
	struct isakmp_ipsec_id r_id;
	pb_stream r_id_pbs;

	r_id.isaiid_np = ISAKMP_NEXT_HASH;
	r_id.isaiid_idtype = st->st_myidentity_type;
	r_id.isaiid_protoid = 0;	/* ??? is this right? */
	r_id.isaiid_port = 0;	/* ??? is this right? */
	if (!out_struct(&r_id, &isakmp_ipsec_identification_desc, &r_hdr_pbs, &r_id_pbs)
	|| !out_raw(st->st_myidentity, st->st_myidentity_len, &r_id_pbs, "my identity"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&r_id_pbs);
    }

    /* HASH_I */
    {
	u_char hash_val[MAX_DIGEST_LEN];
	int hash_len;
	pb_stream r_hash_pbs;

	main_mode_hash(st, hash_val, &hash_len, FALSE, TRUE);
	DBG_cond_dump(DBG_CRYPT, "HASH_R sent:", hash_val, hash_len);

	if (!out_generic(ISAKMP_NEXT_NONE, &isakmp_hash_desc, &r_hdr_pbs, &r_hash_pbs)
	|| !out_raw(hash_val, hash_len, &r_hash_pbs, "HASH_R"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&r_hash_pbs);
    }

    /* encrypt message, sans fixed part of header */

    if (!encrypt_message(&r_hdr_pbs, st))
	return STF_INTERNAL_ERROR;	/* ??? we may be partly committed */

    /* Last block of Phase 1 (R3), kept for Phase 2 IV generation */
    DBG_cond_dump(DBG_CRYPT, "last encrypted block of Phase 1:"
	, st->st_new_iv, st->st_new_iv_len);

    /* Advance state */
    st->st_state = STATE_MAIN_R3;
    st->st_connection->newest_isakmp_sa = st->st_serialno;

    return STF_REPLY;
}

/*
 * Handle HDR*;IDir;HASH_R from responder.
 * It then calls quick_outI1() if there is a pending_quick.
 */
stf_status
main_inR3(struct msg_digest *md)
{
    struct state *const st = md->st;
    struct connection *c = st->st_connection;

    /* input code similar to main_inI3_outR3 -- should be factored */

    /* IDir */
    if (!decode_peer_id(md))
	return STF_FAIL + INVALID_ID_INFORMATION;

    /* HASH_R */

    {
	pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs;
	u_char hash_val[MAX_DIGEST_LEN];
	int hash_len;

	main_mode_hash(st, hash_val, &hash_len, FALSE, FALSE);

	DBG(DBG_CRYPT,
	    DBG_dump("computed HASH_R:", hash_val, hash_len);
	    DBG_dump("received HASH_R:", hash_pbs->cur, pbs_left(hash_pbs)));

	if (pbs_left(hash_pbs) != hash_len
	|| memcmp(hash_pbs->cur, hash_val, hash_len) != 0)
	{
	    log("received HASH_R does not match computed value in main R3");
	    /* XXX Could send notification back */
	    return STF_FAIL + INVALID_HASH_INFORMATION;
	}
    }

    /**************** done input ****************/

    /* Advance state (may be overridden by quick_outI1()) */
    st->st_state = STATE_MAIN_I4;
    c->newest_isakmp_sa = st->st_serialno;

    update_iv(st);	/* finalize our Phase 1 IV */

    if (st->st_pending_quick)
    {
	DBG(DBG_CONTROL, DBG_log("Doing Quick Mode with %s \"%s\""
	     , inet_ntoa(c->that.host), c->name));
	quick_outI1(st->st_whack_sock == NULL_FD? NULL_FD : dup(st->st_whack_sock)
	    , st, c, st->st_policy, c->sa_keying_tries);
	st->st_pending_quick = FALSE;
    }
    return STF_NO_REPLY;
}

/* Handle first message of Phase 2 -- Quick Mode.
 * HDR*, HASH(1), SA, Ni [, KE ] [, IDci, IDcr ] -->
 * HDR*, HASH(2), SA, Nr [, KE ] [, IDci, IDcr ]
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_inI1_outR1(struct msg_digest *md)
{
    /* we build reply packet as we parse the message since
     * the parse_ipsec_sa_body emits the reply SA
     */

    struct state *st = duplicate_state(md->st);
    struct connection *const c = st->st_connection;

    /* optional KE payload for PFS */
    struct payload_digest *const ke_pd = md->chain[ISAKMP_NEXT_KE];

    struct payload_digest *const id_pd = md->chain[ISAKMP_NEXT_ID];
    struct in_addr our_net, our_mask, peer_net, peer_mask;

    pb_stream r_hdr_pbs;    /* reply header */

    u_char
	*r_hashval,	/* where in reply to jam hash value */
	*r_hash_start;	/* from where to start hashing */


    cur_state = st;	/* (caller will reset) */
    /* first: fill in missing bits of our new state object */

    st->st_msgid = md->hdr.isa_msgid;
    insert_state(st);	/* needs cookies, connection, and msgid */

    st->st_new_iv_len = md->st->st_new_iv_len;
    memcpy(st->st_new_iv, md->st->st_new_iv, st->st_new_iv_len);

    md->st = st;	/* feed back new state */

    st->st_policy = c->policy;	/* somebody has got to do it */

    /* put a very short fuse on this state object
     * in case things don't work out.
     */
    event_schedule(EVENT_SA_EXPIRE, 0, st);


    /* demand a KE if PFS requested; accept no more than one */

    if (ke_pd == NULL)
    {
	if (st->st_policy & POLICY_PFS)
	{
	    log("we want PFS but Quick I1 message contains no KE payload");
	    return STF_FAIL + NO_PROPOSAL_CHOSEN;	/* ??? */
	}
    }
    else if (ke_pd->next != NULL)
    {
	log("Quick I1 message contains several KE payloads");
	return STF_FAIL + INVALID_KEY_INFORMATION;	/* ??? */
    }

    /* emit HDR* */
    {
	struct isakmp_hdr r_hdr = md->hdr;	/* mostly same as incoming header */

	r_hdr.isa_flags |= ISAKMP_FLAG_ENCRYPTION;
	r_hdr.isa_np = ISAKMP_NEXT_HASH;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    return STF_INTERNAL_ERROR;
    }

    /* accept HASH(1) */
    {
	pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs;
	u_char hashval[MAX_DIGEST_LEN];
	size_t hashlen;

	DBG_cond_dump(DBG_CRYPT, "received HASH(1):\n", hash_pbs->cur, pbs_left(hash_pbs));

	quick_mode_hash12(hashval, &hashlen, hash_pbs->roof, md->message_pbs.roof, st, FALSE);

	if (hashlen != pbs_left(hash_pbs)
	|| memcmp(hashval, hash_pbs->cur, hashlen) != 0)
	{
	    log("HASH(1) does not verify");
	    /* XXX Could send notification back */
	    return STF_FAIL + INVALID_HASH_INFORMATION;
	}
	DBG(DBG_CRYPT, DBG_log("HASH(1) verified"));
    }

    /* emit HASH(2) -- first pass */
    {
	pb_stream hash_pbs;

	if (!out_generic(ISAKMP_NEXT_SA, &isakmp_hash_desc, &r_hdr_pbs, &hash_pbs))
	    return STF_INTERNAL_ERROR;
	r_hashval = hash_pbs.cur;	/* remember where to plant value */
	if (!out_zero(st->st_oakley.hasher->hash_digest_len, &hash_pbs, "HASH(2)"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&hash_pbs);
	r_hash_start = r_hdr_pbs.cur;	/* hash from after HASH*/
    }

    /* process SA (in and out) */
    {
	struct payload_digest *const sapd = md->chain[ISAKMP_NEXT_SA];
	pb_stream r_sa_pbs;
	struct isakmp_sa sa = sapd->payload.sa;

	/* sa header is unchanged -- except for np */
	sa.isasa_np = ISAKMP_NEXT_NONCE;
	if (!out_struct(&sa, &isakmp_sa_desc, &r_hdr_pbs, &r_sa_pbs))
	    return STF_INTERNAL_ERROR;

	/* parse and accept body */
	{
	    notification_t r = parse_ipsec_sa_body(&sapd->pbs
		, &sapd->payload.sa, &r_sa_pbs, FALSE, st);

	    if (r != NOTHING_WRONG)
		return STF_FAIL + r;
	}
    }

    /* accept Ni payload */

    /* accept optional KE payload -- needed for PFS */

    if (ke_pd != NULL)
    {
	if (st->st_pfs_group == OAKLEY_GROUP_NONE)
	{
	    log("Quick Mode I1 message KE payload requires a GROUP_DESCRIPTION attribute in SA");
	    return STF_FAIL + INVALID_KEY_INFORMATION;
	}

	/* first parse incoming values and put them in state */

	if (pbs_left(&ke_pd->pbs) < MINIMUM_PUBLIC_VALUE_SIZE)
	{
		log("too short public size (%d)", pbs_left(&ke_pd->pbs));
		/* XXX Could send notification back */
		return STF_FAIL + INVALID_KEY_INFORMATION;
	}

	/* Decode Gi from network form */
	n_to_mpz(&st->st_gi_in_use, &st->st_gi, ke_pd->pbs.cur, pbs_left(&ke_pd->pbs));

	DBG(DBG_CRYPT,
	    DBG_dump_mpz("PFS public value received:\n", &st->st_gi));
    }
    else
    {
	if (st->st_pfs_group != OAKLEY_GROUP_NONE)
	{
	    log("missing KE payload in Quick Mode I1 message");
	    return STF_FAIL + INVALID_KEY_INFORMATION;
	}
    }

    /* accept optional IDci, IDcr */

    if (id_pd != NULL)
    {
	/* ??? we are assuming IPSEC_DOI */

	/* IDci (initiator is peer) */

	if (!decode_net_id(&id_pd->payload.ipsec_id, &id_pd->pbs
	, &peer_net, &peer_mask, "peer client", c->that.host))
	    return STF_FAIL + INVALID_ID_INFORMATION;

	st->st_peeruserprotoid = id_pd->payload.ipsec_id.isaiid_protoid;
	st->st_peeruserport = id_pd->payload.ipsec_id.isaiid_port;

	/* IDcr (we are responder) */

	if (!decode_net_id(&id_pd->next->payload.ipsec_id, &id_pd->next->pbs
	, &our_net, &our_mask, "our client", c->that.host))
	    return STF_FAIL + INVALID_ID_INFORMATION;

	st->st_myuserprotoid = id_pd->next->payload.ipsec_id.isaiid_protoid;
	st->st_myuserport = id_pd->next->payload.ipsec_id.isaiid_port;
    }
    else
    {
	/* implicit IDci and IDcr: peer and self */
	our_net = c->this.host;
	our_mask = mask32.sin_addr;
	peer_net = c->that.host;
	peer_mask = mask32.sin_addr;
    }

    /* Now that we have identities of client subnets, we must look for
     * a suitable connection (our current one only matches for hosts).
     */
    {
	struct connection *p = find_client_connection(c,
	    our_net, our_mask, peer_net, peer_mask);

	if (p == NULL)
	{
	    char
		me[ADDRTOA_BUF], mine[SUBNETTOA_BUF],
		he[ADDRTOA_BUF], his[SUBNETTOA_BUF];

	    addrtoa(c->this.host, 0, me, sizeof(me));
	    subnettoa(our_net, our_mask, 0, mine, sizeof(mine));
	    addrtoa(c->that.host, 0, he, sizeof(he));
	    subnettoa(peer_net, peer_mask, 0, his, sizeof(mine));
	    log("no connection known for %s<->%s<-->%s<->%s",
		mine, me, he, his);
	    return STF_FAIL + INVALID_ID_INFORMATION;
	}
	st->st_connection = p;	/* ??? more plumbing needed? */
	DBG(DBG_CONTROL, DBG_log("using connection \"%s\"", p->name));
    }

    log("responding to Quick Mode");

    /* copy Ni into the state */
    {
	pb_stream *const nonce_pbs = &md->chain[ISAKMP_NEXT_NONCE]->pbs;

	replace(st->st_ni, clone_bytes(nonce_pbs->cur, pbs_left(nonce_pbs),
	    "st_ni in quick_inI1_outR1()"));
	st->st_ni_len = pbs_left(nonce_pbs);
    }

    /**** finish reply packet: Nr [, KE ] [, IDci, IDcr ] ****/

    /* reply Nr */
    {
	pb_stream r_nonce_pbs;
	u_int8_t nonce_bytes[DEFAULTNONCESIZE];

	/* next payload depends on whether KE or ID payloads will be sent.
	 * ??? But we don't do KE at the moment.
	 */
	if (!out_generic(ke_pd != NULL? ISAKMP_NEXT_KE : id_pd != NULL? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE
	, &isakmp_nonce_desc, &r_hdr_pbs, &r_nonce_pbs))
	    return STF_INTERNAL_ERROR;
	get_rnd_bytes(nonce_bytes, DEFAULTNONCESIZE);
	if (!out_raw(nonce_bytes, DEFAULTNONCESIZE, &r_nonce_pbs, "nonce value"))
	    return STF_INTERNAL_ERROR;
	close_output_pbs(&r_nonce_pbs);

	/* Copy Nr into the state object */
	replace(st->st_nr, clone_bytes(nonce_bytes, DEFAULTNONCESIZE,
	    "st_nr in quick_inI1_outR1()"));
	st->st_nr_len = DEFAULTNONCESIZE;

	DBG(DBG_CRYPT, DBG_dump("Nr sent:", nonce_bytes, DEFAULTNONCESIZE));
    }

    /* reply [, KE ] */

    if (ke_pd != NULL)
    {
	/* generate own values for DH */

	u_char tmp[LOCALSECRETSIZE];
	pb_stream r_keyex_pbs;
	u_char *gr;
	size_t grlen;
	bool r;
		
	/* MPZ-Operations might be done after sending the packet... */

	/* own secret ... */
	get_rnd_bytes(tmp, LOCALSECRETSIZE);
	n_to_mpz(&st->st_sec_in_use, &st->st_sec, tmp, LOCALSECRETSIZE);

	/* and public value... */
	mpz_init_powm(st->st_gr_in_use
	    , st->st_gr, groupgenerator, st->st_sec, st->st_pfs_modp_modulus);

	/* and shared secret ... */
	mpz_init_powm(st->st_shared_in_use, st->st_shared
	    , st->st_gi, st->st_sec, st->st_pfs_modp_modulus);

	DBG(DBG_CRYPT,
		DBG_dump_mpz("PFS Local secret:\n", &st->st_sec);
		DBG_dump_mpz("PFS Public value sent:\n", &st->st_gr);
		DBG_dump_mpz("PFS Shared secret:\n", &st->st_shared));

	if (!out_generic(id_pd != NULL? ISAKMP_NEXT_ID : ISAKMP_NEXT_NONE
	, &isakmp_keyex_desc, &r_hdr_pbs, &r_keyex_pbs))
		return STF_INTERNAL_ERROR;

	mpz_to_n(&gr, &grlen, &st->st_gr);    		
	r = out_raw(gr, grlen, &r_keyex_pbs, "keyex value");
	pfree(gr);
	if (!r)
	    return STF_INTERNAL_ERROR;

	close_output_pbs(&r_keyex_pbs);
    }

    /* reply [, IDci, IDcr ] */
    if  (id_pd != NULL)
    {
	struct isakmp_ipsec_id *p = (void *)r_hdr_pbs.cur;	/* UGH! */

	if (!out_raw(id_pd->pbs.start, pbs_room(&id_pd->pbs), &r_hdr_pbs, "IDci"))
	    return STF_INTERNAL_ERROR;
	p->isaiid_np = ISAKMP_NEXT_ID;

	p = (void *)r_hdr_pbs.cur;	/* UGH! */

	if (!out_raw(id_pd->next->pbs.start, pbs_room(&id_pd->next->pbs), &r_hdr_pbs, "IDcr"))
	    return STF_INTERNAL_ERROR;
	p->isaiid_np = ISAKMP_NEXT_NONE;
    }

    /* Compute reply HASH(2) */
    quick_mode_hash12(r_hashval, NULL, r_hash_start, r_hdr_pbs.cur, st, TRUE);

    /* encrypt message, except for fixed part of header */

    if (!encrypt_message(&r_hdr_pbs, st))
	return STF_INTERNAL_ERROR;	/* ??? we may be partly committed */

    /* Update state of exchange */
    st->st_state = STATE_QUICK_R1;

    return STF_REPLY;
}

/* Handle (the single) message from Responder in Quick Mode.
 * HDR*, HASH(2), SA, Nr [, KE ] [, IDci, IDcr ] -->
 * HDR*, HASH(3)
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_inR1_outI2(struct msg_digest *md)
{
    struct state *const st = md->st;

    struct payload_digest *const sa_pd = md->chain[ISAKMP_NEXT_SA];

    /* optional KE payload for PFS */
    struct payload_digest *const ke_pd = md->chain[ISAKMP_NEXT_KE];

    struct payload_digest *const id_pd = md->chain[ISAKMP_NEXT_ID];

    pb_stream r_hdr_pbs;    /* reply header */

    /* accept no more than one KE */
    if (ke_pd != NULL && ke_pd->next != NULL)
    {
	log("Quick I1 message contains several KE payloads");
	return STF_FAIL + INVALID_KEY_INFORMATION;	/* ??? */
    }

    /* HASH(2) */
    {
	pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs;
	u_char hashval[MAX_DIGEST_LEN];
	size_t hashlen;

	DBG_cond_dump(DBG_CRYPT, "received HASH(2):\n", hash_pbs->cur, pbs_left(hash_pbs));

	quick_mode_hash12(hashval, &hashlen, hash_pbs->roof, md->message_pbs.roof, st, TRUE);

	if (hashlen != pbs_left(hash_pbs)
	|| memcmp(hashval, hash_pbs->cur, hashlen) != 0)
	{
	    log("HASH(2) does not verify");
	    /* XXX Could send notification back */
	    return STF_FAIL + INVALID_HASH_INFORMATION;
	}
	DBG(DBG_CRYPT, DBG_log("HASH(2) verified"));
    }

    /* SA payload */
    {
	notification_t r = parse_ipsec_sa_body(&sa_pd->pbs
	    , &sa_pd->payload.sa, NULL, TRUE, st);

	if (r != NOTHING_WRONG)
	    return STF_FAIL + r;
    }

    /* Ni payload -- nothing to do yet */

    /* Handle optional KE payload -- needed for PFS */

    if (ke_pd != NULL)
    {
	if (st->st_pfs_group == OAKLEY_GROUP_NONE)
	{
	    log("Quick Mode R1 message KE payload requires a GROUP_DESCRIPTION attribute in SA");
	    return STF_FAIL + INVALID_KEY_INFORMATION;
	}

	if (pbs_left(&ke_pd->pbs) < MINIMUM_PUBLIC_VALUE_SIZE)
	{
	    log("too short public size (%d)", pbs_left(&ke_pd->pbs));
	    /* XXX Could send notification back */
	    return STF_FAIL + INVALID_KEY_INFORMATION;
	}

	/* decode Gr from network form */
	n_to_mpz(&st->st_gr_in_use, &st->st_gr, ke_pd->pbs.cur, pbs_left(&ke_pd->pbs));

	/* st->st_shared := st->st_gr ** st->st_sec  %  st->st_pfs_modp_modulus */
	mpz_init_powm(st->st_shared_in_use, st->st_shared
	    , st->st_gr, st->st_sec, st->st_pfs_modp_modulus);

	DBG(DBG_CRYPT,
	    DBG_dump_mpz("PFS public value received:\n", &st->st_gr);
	    DBG_dump_mpz("PFS local secret:\n", &st->st_sec);
	    DBG_dump_mpz("PFS shared secret:\n", &st->st_shared));
    }
    else
    {
	if (st->st_pfs_group != OAKLEY_GROUP_NONE)
	{
	    log("missing KE payload in Quick Mode R1 message");
	    return STF_FAIL + INVALID_KEY_INFORMATION;
	}
    }

    /* handle optional IDci, IDcr; these must match what we sent */

    if (id_pd != NULL)
    {
	/* ??? we are assuming IPSEC_DOI */

	/* IDci (we are initiator) */

	if (!check_net_id(&id_pd->payload.ipsec_id, &id_pd->pbs
	, &st->st_myuserprotoid, &st->st_myuserport
	, &st->st_connection->this.client_net
	, &st->st_connection->this.client_mask
	, "our client", st->st_connection->this.host))
	    return STF_FAIL + INVALID_ID_INFORMATION;

	/* IDcr (responder is peer) */

	if (!check_net_id(&id_pd->next->payload.ipsec_id, &id_pd->next->pbs
	, &st->st_peeruserprotoid, &st->st_peeruserport
	, &st->st_connection->that.client_net
	, &st->st_connection->that.client_mask
	, "peer client", st->st_connection->this.host))
	    return STF_FAIL + INVALID_ID_INFORMATION;
    }
    else
    {
	/* No IDci, IDcr: we must check that the defaults match our proposal.
	 * Parallels a sequence of assignments in quick_outI1.
	 */
	const struct connection *c = st->st_connection;

	if (c->this.client_net.s_addr != c->this.host.s_addr
	|| c->this.client_mask.s_addr != mask32.sin_addr.s_addr
	|| c->that.client_net.s_addr != c->that.host.s_addr
	|| c->that.client_mask.s_addr != mask32.sin_addr.s_addr)
	{
	    log("IDci, IDcr payloads missing in message but default does not match proposal");
	    return STF_FAIL + INVALID_ID_INFORMATION;
	}
    }

    /* copy Nr into the state */
    {
	pb_stream *const nonce_pbs = &md->chain[ISAKMP_NEXT_NONCE]->pbs;

	replace(st->st_nr, clone_bytes(nonce_pbs->cur, pbs_left(nonce_pbs),
	    "st_nr in quick_inR1_outI2()"));
	st->st_nr_len = pbs_left(nonce_pbs);
	DBG_cond_dump(DBG_CRYPT, "Nr received:", nonce_pbs->cur
	    , pbs_left(nonce_pbs));
    }


    /* Copy accepted proposal into the state.  The whole payload! */
    replace(st->st_proposal, clone_bytes(sa_pd->pbs.start, pbs_room(&sa_pd->pbs),
	"st_proposal in quick_inR1_outI2()"));

    /**************** build reply packet HDR*, HASH(3) ****************/

    /* HDR* */

    {
	struct isakmp_hdr r_hdr = md->hdr;	/* mostly same as incoming header */

	r_hdr.isa_flags |= ISAKMP_FLAG_ENCRYPTION;
	r_hdr.isa_np = ISAKMP_NEXT_HASH;
	if (!out_struct(&r_hdr, &isakmp_hdr_desc, &md->reply, &r_hdr_pbs))
	    return STF_INTERNAL_ERROR;
    }

    /* HASH(3) -- since this is the only contents, no passes needed */

    {
	pb_stream hash_pbs;
	u_char *r_hashval;	/* where in reply to jam hash value */

	if (!out_generic(ISAKMP_NEXT_NONE, &isakmp_hash_desc, &r_hdr_pbs, &hash_pbs))
	    return STF_INTERNAL_ERROR;
	r_hashval = hash_pbs.cur;	/* remember where to plant value */
	if (!out_zero(st->st_oakley.hasher->hash_digest_len, &hash_pbs, "HASH(3)"))
	    return STF_INTERNAL_ERROR;
	quick_mode_hash3(r_hashval, NULL, st);
	close_output_pbs(&hash_pbs);
    }

    /* Derive new keying material */
    compute_keymats(st);

    /* XXX Unless the commit bit is set, we need to send a message
     * XXX to the kernel to establish the new SA.
     * We do this before any state updating so that
     * failure won't look like success.
     */
    if (!install_ipsec_sa(st, TRUE))
	return STF_INTERNAL_ERROR;

    /* encrypt message, except for fixed part of header */

    if (!encrypt_message(&r_hdr_pbs, st))
	return STF_INTERNAL_ERROR;	/* ??? we may be partly committed */

    /* Update state of exchange */
    st->st_state = STATE_QUICK_I2;
    st->st_connection->newest_ipsec_sa = st->st_serialno;

    return STF_REPLY;
}

/* Handle last message of Quick Mode.
 * HDR*, HASH(3) -> done
 * (see draft-ietf-ipsec-isakmp-oakley-07.txt 5.5)
 */
stf_status
quick_inI2(struct msg_digest *md)
{
    struct state *const st = md->st;

    /* HASH(3) */
    {
	pb_stream *const hash_pbs = &md->chain[ISAKMP_NEXT_HASH]->pbs;
	u_char hashval[MAX_DIGEST_LEN];
	size_t hashlen;

	DBG_cond_dump(DBG_CRYPT, "received HASH(3):"
	    , hash_pbs->cur, pbs_left(hash_pbs));

	quick_mode_hash3(hashval, &hashlen, st);

	if (hashlen != pbs_left(hash_pbs)
	|| memcmp(hashval, hash_pbs->cur, hashlen) != 0)
	{
	    log("HASH(3) does not verify");
	    /* XXX Could send notification back */
	    return STF_FAIL + INVALID_HASH_INFORMATION;
	}
	DBG(DBG_CRYPT, DBG_log("HASH(3) authentication succeeded"));
    }

    /* Derive new keying material */
    compute_keymats(st);

    /* XXX Unless the commit bit is set, we need to send a message
     * XXX to the kernel to establish the new SA.
     * We do this before any state updating so that
     * failure won't look like success.
     */
    if (!install_ipsec_sa(st, FALSE))
	return STF_INTERNAL_ERROR;

    /* Advance state */
    st->st_state = STATE_QUICK_R2;
    st->st_connection->newest_ipsec_sa = st->st_serialno;

    update_iv(st);	/* not actually used, but tidy */

    return STF_NO_REPLY;
}
