/* demultiplex incoming IKE messages
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: demux.c,v 1.51 1999/04/11 00:44:17 dhr Exp $
 */

/* Ordering Constraints on Payloads
 *
 * rfc2409: The Internet Key Exchange (IKE)
 *
 * 5 Exchanges:
 *   "The SA payload MUST precede all other payloads in a phase 1 exchange."
 *
 *   "Except where otherwise noted, there are no requirements for ISAKMP
 *    payloads in any message to be in any particular order."
 *
 * 5.3 Phase 1 Authenticated With a Revised Mode of Public Key Encryption:
 *
 *   "If the HASH payload is sent it MUST be the first payload of the
 *    second message exchange and MUST be followed by the encrypted
 *    nonce. If the HASH payload is not sent, the first payload of the
 *    second message exchange MUST be the encrypted nonce."
 *
 *   "Save the requirements on the location of the optional HASH payload
 *    and the mandatory nonce payload there are no further payload
 *    requirements. All payloads-- in whatever order-- following the
 *    encrypted nonce MUST be encrypted with Ke_i or Ke_r depending on the
 *    direction."
 *
 * 5.5 Phase 2 - Quick Mode
 *
 *   "In Quick Mode, a HASH payload MUST immediately follow the ISAKMP
 *    header and a SA payload MUST immediately follow the HASH."
 *   [NOTE: there may be more than one SA payload, so this is not
 *    totally reasonable.  Probably all SAs should be so constrained.]
 *
 *   "If ISAKMP is acting as a client negotiator on behalf of another
 *    party, the identities of the parties MUST be passed as IDci and
 *    then IDcr."
 *
 *   "With the exception of the HASH, SA, and the optional ID payloads,
 *    there are no payload ordering restrictions on Quick Mode."
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>
#include <des.h>

#include "constants.h"
#include "defs.h"
#include "cookie.h"
#include "connections.h"
#include "state.h"
#include "packet.h"
#include "md5.h"
#include "sha1.h"
#include "crypto.h" /* requires sha1.h and md5.h */
#include "log.h"
#include "demux.h"	/* needs packet.h */
#include "ipsec_doi.h"	/* needs demux.h and state.h */
#include "timer.h"
#include "whack.h"	/* requires connections.h */
#include "server.h"

/* This file does basic header checking and demux of
 * incoming packets.
 */

struct state_microcode {
    lset_t flags;
    lset_t req_payloads;	/* required payloads (allows just one) */
    lset_t opt_payloads;	/* optional payloads (any mumber) */
    u_int8_t timeout_event;
    state_transition_fn *processor;
};

/* State Microcode Flags */

#define SMF_FIRST_ENCRYPTED_INPUT	LELEM(0)
#define SMF_INPUT_ENCRYPTED	LELEM(1)
#define SMF_OUTPUT_ENCRYPTED	LELEM(2)

static const struct state_microcode state_microcode[] = {
#define P(n) LELEM(ISAKMP_NEXT_##n)

    /* No state for main_outI1: --> HDR;SA */

    /* STATE_MAIN_R0: HDR;SA --> HDR;SA */
    { LEMPTY,
      P(SA), P(VID), EVENT_RETRANSMIT, main_inI1_outR1},

    /* STATE_MAIN_I1: HDR;SA --> HDR;KE;Ni */
    { LEMPTY,
      P(SA), P(VID), EVENT_RETRANSMIT, main_inR1_outI2 },

    /* STATE_MAIN_R1: HDR;KE;Ni --> HDR;KE;Nr */
    { LEMPTY,
      P(KE) | P(NONCE), P(VID), EVENT_RETRANSMIT, main_inI2_outR2 },

    /* for states from here on, output message must be encrypted */

    /* STATE_MAIN_I2: HDR;KE;Nr --> HDR*;IDii;HASH_I */
    { SMF_OUTPUT_ENCRYPTED,
      P(KE) | P(NONCE), P(VID), EVENT_RETRANSMIT, main_inR2_outI3 },

    /* for states from here on, input message must be encrypted */

    /* STATE_MAIN_R2: HDR*;IDii;HASH_I --> HDR*;IDir;HASH_R */
    { SMF_FIRST_ENCRYPTED_INPUT | SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      P(ID) | P(HASH), P(VID), EVENT_SA_EXPIRE, main_inI3_outR3 },

    /* STATE_MAIN_I3: HDR*;IDir;HASH_R --> done
     * May initiate quick mode by calling quick_outI1
     */
    { SMF_FIRST_ENCRYPTED_INPUT | SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      P(ID) | P(HASH), P(VID), EVENT_SA_REPLACE, main_inR3 },

    /* STATE_MAIN_R3: can only get here due to packet loss */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      LEMPTY, LEMPTY,  EVENT_NULL, NULL},

    /* STATE_MAIN_I4: can only get here due to packet loss */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      LEMPTY, LEMPTY, EVENT_NULL, NULL },


    /* No state for quick_outI1:
     * --> HDR*, HASH(1), SA, Nr [, KE ] [, IDci, IDcr ]
     */

    /* STATE_QUICK_R0:
     * HDR*, HASH(1), SA, Ni [, KE ] [, IDci, IDcr ] -->
     * HDR*, HASH(2), SA, Nr [, KE ] [, IDci, IDcr ]
     * ??? it is legal to have multiple SAs, but we don't support it yet.
     */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      P(HASH) | P(SA) | P(NONCE), /* P(SA) | */ P(KE) | P(ID)
    , EVENT_RETRANSMIT, quick_inI1_outR1 },

    /* STATE_QUICK_I1:
     * HDR*, HASH(2), SA, Nr [, KE ] [, IDci, IDcr ] -->
     * HDR*, HASH(3)
     * ??? it is legal to have multiple SAs, but we don't support it yet.
     */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      P(HASH) | P(SA) | P(NONCE), /* P(SA) | */ P(KE) | P(ID)
    , EVENT_SA_REPLACE, quick_inR1_outI2 },

    /* STATE_QUICK_R1: HDR*, HASH(3) --> done */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      P(HASH), LEMPTY, EVENT_SA_EXPIRE, quick_inI2 },

    /* STATE_QUICK_I2: can only happen due to lost packet */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      LEMPTY, LEMPTY, EVENT_NULL, NULL },

    /* STATE_QUICK_R2: can only happen due to lost packet */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      LEMPTY, LEMPTY, EVENT_NULL, NULL },


    /* informational messages: */

    /* STATE_INFO: */
    { LEMPTY,
      LEMPTY, LEMPTY, EVENT_NULL, NULL },

    /* STATE_INFO_PROTECTED: */
    { SMF_INPUT_ENCRYPTED | SMF_OUTPUT_ENCRYPTED,
      P(HASH), LEMPTY, EVENT_NULL, NULL },

#undef P
};

bool
send_packet(struct state *st, const char *where)
{
    struct sockaddr_in peer;
    struct connection *c = st->st_connection;

    mksin(peer, c->that.host.s_addr, c->that.host_port);
    if (sendto(c->interface->fd
    , st->st_tpacket, st->st_tpacket_len, 0
    , (struct sockaddr *)&peer, sizeof(peer)) != st->st_tpacket_len)
    {
	log_errno((e, "sendto() on %s to %s failed in %s"
	    , c->interface->rname, show_sa_in(&peer), where));
	return FALSE;
    }
    else
    {
	return TRUE;
    }
}

/* free the resources in a message digest */

static void
free_md(struct msg_digest *md)
{
    if (md->raw_packet != NULL)
    {
	pfree(md->raw_packet);
	md->raw_packet = NULL;
    }
    passert(md->packet_pbs.start != NULL);
    pfree(md->packet_pbs.start);
    md->packet_pbs.start = NULL;
    whack_log_fd = NULL_FD;
    cur_state = NULL;
    cur_connection = NULL;
    cur_from = NULL;
}

/* Receive a packet. If we pass buffer to a routine that does not return
 * failure indication (e.g. the packet handling routines), it's up to them
 * to free it; otherwise this routine does.
 */
void
comm_handle(const struct iface *ifp)
{
    struct msg_digest md;
    const struct state_microcode *smc;
    bool new_iv_set = FALSE;

    /* initialize md */
    md.iface = ifp;
    md.from_state = 0;		/* not yet valid */
    md.st = NULL;
    md.note = NOTHING_WRONG;
    md.raw_packet = NULL;

    md.encrypted = FALSE;
    md.digest_roof = md.digest;

    {
	int i;
	for (i = 0; i != ISAKMP_NEXT_ROOF; i++)
	    md.chain[i] = NULL;
    }

    /* Now really read the message.
     * Since we don't know its size, we read it into
     * an overly large buffer and then copy it to a
     * new, properly sized buffer.
     */
    {
	int sin_len;
	int packet_len;
	/* ??? this buffer seems *way* too big */
	u_int8_t bigbuffer[UDP_SIZE];

	mksin(md.sin, 0, 0);
	sin_len = sizeof(md.sin);

	packet_len = recvfrom(ifp->fd, bigbuffer, sizeof(bigbuffer), 0
	    , (struct sockaddr *)&md.sin, &sin_len);
	if (packet_len == -1)
	{
	    log_errno((e, "recvfrom() on %s from %s failed in comm_handle"
		, ifp->rname, show_sa_in(&md.sin)));
	    return;
	}

	cur_from = &md.sin;

	/* Allocate exactly that much space
	 * and set up md.packet_pbs to describe it.
	 */
	init_pbs(&md.packet_pbs,
	    clone_bytes(bigbuffer, packet_len, "message buffer in comm_handle()"),
	    packet_len, "packet");
    }

    DBG(DBG_RAW,
	DBG_log("read %d bytes", (int) pbs_room(&md.packet_pbs));
	DBG_dump("", md.packet_pbs.start, pbs_room(&md.packet_pbs)));

    if ((pbs_room(&md.packet_pbs) & 03) != 0)
    {
	/* We don't care.  Should we?
	 * ??? We don't even take care to make our output packets conform.
	 */
	log("packet has %d bytes, not a multiple of 4"
	    , (int) pbs_room(&md.packet_pbs));
    }

    if (!in_struct(&md.hdr, &isakmp_hdr_desc, &md.packet_pbs, &md.message_pbs))
    {
	/* XXX specific failures (special notification?):
	 * - bad ISAKMP major/minor version numbers
	 * - size of packet vs size of message
	 */
	free_md(&md);
	return;
    }

    if (md.packet_pbs.roof != md.message_pbs.roof)
    {
	log("size (%u) differs from size specified in ISAKMP HDR (%u)"
	    , (unsigned) pbs_room(&md.packet_pbs), md.hdr.isa_length);
	free_md(&md);
	return;
    }

    if (pbs_room(&md.message_pbs) & 03)
    {
	log("ISAKMP message: size should be a multiple of 4 but is %u"
	    , (unsigned) pbs_room(&md.packet_pbs));
    }

    if (md.hdr.isa_flags & ISAKMP_FLAG_COMMIT)
    {
	/* XXX we should handle this, whatever it means */
	log("Pluto doesn't implement ISAKMP_FLAG_ENCRYPTION");
	free_md(&md);
	return;
    }

    switch (md.hdr.isa_xchg)
    {
#ifdef NOTYET
    case ISAKMP_XCHG_NONE:
    case ISAKMP_XCHG_BASE:
#endif

    case ISAKMP_XCHG_IDPROT:	/* part of a Main Mode exchange */
	if (md.hdr.isa_msgid != 0)
	{
	    log("message id should be zero (was 0x%08lx) in Main Mode",
		(unsigned long) md.hdr.isa_msgid);
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}

	if (is_zero_cookie(md.hdr.isa_icookie))
	{
	    log("initiator cookie must not be zero in Main Mode message");
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}

	if (is_zero_cookie(md.hdr.isa_rcookie))
	{
	    /* initial message from initiator
	     * ??? what if this is a duplicate of another message?
	     */
	    if (md.hdr.isa_flags & ISAKMP_FLAG_ENCRYPTION)
	    {
		log("initial Main Mode message is encrypted");
		free_md(&md);
		return;
	    }
	    if (md.hdr.isa_np != ISAKMP_NEXT_SA)
	    {
		log("invalid payload %s (expected SA in main I1)"
		    , enum_show(&payload_names, md.hdr.isa_np));
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    /* don't build a state until the message looks tasty */
	    md.from_state = STATE_MAIN_R0;
	}
	else
	{
	    /* not an initial message */

	    cur_state = md.st = find_state(md.hdr.isa_icookie, md.hdr.isa_rcookie
		, md.sin.sin_addr, md.hdr.isa_msgid);

	    if (md.st == NULL)
	    {
		/* perhaps this is a first message from the responder
		 * and contains a responder cookie that we've not yet seen.
		 */
		cur_state = md.st = find_state(md.hdr.isa_icookie, zero_cookie
		    , md.sin.sin_addr, md.hdr.isa_msgid);

		if (md.st == NULL)
		{
		    log("Main Mode message is part of an unknown exchange");
		    /* XXX Could send notification back */
		    free_md(&md);
		    return;
		}
	    }
	    md.from_state = md.st->st_state;
	    whack_log_fd = md.st->st_whack_sock;
	}
	break;

#ifdef NOTYET
    case ISAKMP_XCHG_AO:
    case ISAKMP_XCHG_AGGR:
#endif

    case ISAKMP_XCHG_INFO:	/* an informational exchange */
	cur_state = md.st = find_state(md.hdr.isa_icookie, md.hdr.isa_rcookie
	    , md.sin.sin_addr, 0);

	if (md.hdr.isa_flags & ISAKMP_FLAG_ENCRYPTION)
	{
	    if (md.st == NULL)
	    {
		log("informational exchange for an unknown SA");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    whack_log_fd = md.st->st_whack_sock;

	    if (!IS_ISAKMP_SA_ESTABLISHED(md.st->st_state))
	    {
		log("encrypted Informational Exchange message is for incomplete ISAKMP SA");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    if (md.hdr.isa_msgid == 0)
	    {
		log("informational exchange with 0 message ID");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    if (!reserve_msgid(md.st->st_connection->that.host, md.hdr.isa_msgid))
	    {
		log("informational exchange with previously used message ID (0x%08lx)"
		    , (unsigned long)md.hdr.isa_msgid);
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    init_phase2_iv(md.st, &md.hdr.isa_msgid);
	    new_iv_set = TRUE;

	    md.from_state = STATE_INFO_PROTECTED;
	}
	else
	{
	    if (md.st != NULL && IS_ISAKMP_SA_ESTABLISHED(md.st->st_state))
	    {
		log("Informational Exchange message is for ISAKMP SA must be encrypted");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }
	    md.from_state = STATE_INFO;
	}
	break;

    case ISAKMP_XCHG_QUICK:	/* part of a Quick Mode exchange */
	if (is_zero_cookie(md.hdr.isa_icookie))
	{
	    log("initiator cookie must not be zero in Quick Mode message");
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}

	if (is_zero_cookie(md.hdr.isa_icookie))
	{
	    log("responder cookie must not be zero in Quick Mode message");
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}

	if (md.hdr.isa_msgid == 0)
	{
	    log("message id must not be zero in Quick Mode message");
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}

	cur_state = md.st = find_state(md.hdr.isa_icookie, md.hdr.isa_rcookie
	    , md.sin.sin_addr, md.hdr.isa_msgid);

	if (md.st == NULL)
	{
	    /* No appropriate Quick Mode state.
	     * See if we have a Main Mode state.
	     * ??? what if this is a duplicate of another message?
	     */
	    cur_state = md.st = find_state(md.hdr.isa_icookie, md.hdr.isa_rcookie
		, md.sin.sin_addr, 0);

	    if (md.st == (struct state *) NULL)
	    {
		log("Quick Mode message is for a non-existent (expired?) ISAKMP SA");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    if (!IS_ISAKMP_SA_ESTABLISHED(md.st->st_state))
	    {
		log("Quick Mode message is for incomplete ISAKMP SA");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    /* only accept this new Quick Mode exchange if it has a unique message ID */
	    if (!reserve_msgid(md.sin.sin_addr, md.hdr.isa_msgid))
	    {
		log("Quick Mode I1 message with previously used message ID 0x%08lx (perhaps duplicated packet)"
		    , (unsigned long) md.hdr.isa_msgid);
		/* XXX Could send notification INVALID_MESSAGE_ID back */
		free_md(&md);
		return;
	    }

	    /* Quick Mode Initial IV */
	    init_phase2_iv(md.st, &md.hdr.isa_msgid);
	    new_iv_set = TRUE;

	    md.from_state = STATE_QUICK_R0;
	}
	else
	{
	    md.from_state = md.st->st_state;
	}

	whack_log_fd = md.st->st_whack_sock;

	break;

#ifdef NOTYET
    case ISAKMP_XCHG_NGRP:
#endif

    default:
	log("unsupported exchange type %s in message"
	    , enum_show(&exchange_names, md.hdr.isa_xchg));
	free_md(&md);
	return;
    }

    /* We have found a from_state: set smc to describe its properties.
     * (We may not have a state object -- if we need to build one,
     * we wait until the package has been sanity checked.)
     */
    passert(STATE_MAIN_R0 <= md.from_state && md.from_state <= STATE_INFO_PROTECTED);
    smc = &state_microcode[md.from_state - STATE_MAIN_R0];

    /* If this packet is a duplicate, discard it.
     * This won't work for the initial packet of an exchange
     * because we won't have a state object to remember it.
     * ??? Notification packets are like exchanges -- I hope that
     * they are idempotent!
     */
    if (md.st != NULL
    && md.st->st_rpacket != NULL
    && md.st->st_rpacket_len == (size_t) (md.packet_pbs.roof - md.packet_pbs.start)
    && memcmp(md.st->st_rpacket, md.packet_pbs.start, md.st->st_rpacket_len) == 0)
    {
	/* the new packet was identical to the last accepted one -- ignore */
	log("discarding duplicate packet");
	free_md(&md);
	return;
    }

    if (md.hdr.isa_flags & ISAKMP_FLAG_ENCRYPTION)
    {
	DBG(DBG_CRYPT, DBG_log("received encrypted packet from %s",
	    show_sa_in(&md.sin)));

	if (md.st == NULL
	|| md.st->st_skeyid_e == (u_char *) NULL)
	{
	    log("unexpected encrypted packet received");
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}

	/* Mark as encrypted */
	md.encrypted = TRUE;

	DBG(DBG_CRYPT, DBG_log("decrypting %u bytes using algorithm %s",
	    (unsigned) pbs_left(&md.message_pbs),
	    enum_show(&oakley_enc_names, md.st->st_oakley.encrypt)));

	/* do the specified decryption
	 *
	 * IV is from md.st->st_iv (md.st->st_iv_len)
	 * The new iv is placed in md.st->st_new_iv
	 *
	 * See draft-ietf-ipsec-isakmp-oakley-07.txt Appendix B
	 *
	 * XXX The IV should only be updated really if the packet
	 * is successfully processed.
	 * We should keep this value, check for a success return
	 * value from the parsing routines and then replace.
	 *
	 * Each post phase 1 exchange generates IVs from
	 * the last phase 1 block, not the last block sent.
	 */
	{
	    const struct encrypt_desc *e = md.st->st_oakley.encrypter;

	    if (pbs_left(&md.message_pbs) % e->blocksize != 0)
	    {
		log("message not a multiple of encryption blocksize");
		/* XXX Could send notification back */
		free_md(&md);
		return;
	    }

	    /* XXX Detect weak keys */

	    /* grab a copy of raw packet (for duplicate packet detection) */
	    md.raw_packet_len = pbs_room(&md.packet_pbs);
	    md.raw_packet = clone_bytes(md.packet_pbs.start, md.raw_packet_len
		, "raw packet");

	    /* Decrypt everything after header */
	    if (!new_iv_set)
	    {
		/* use old IV */
		passert(md.st->st_iv_len <= sizeof(md.st->st_new_iv));
		md.st->st_new_iv_len = md.st->st_iv_len;
		memcpy(md.st->st_new_iv, md.st->st_iv, md.st->st_new_iv_len);
	    }
	    e->crypt(FALSE, md.message_pbs.cur, pbs_left(&md.message_pbs)
		, md.st);
	}

	DBG_cond_dump(DBG_CRYPT, "decrypted:\n", md.message_pbs.cur,
	    md.message_pbs.roof - md.message_pbs.cur);

	DBG_cond_dump(DBG_CRYPT, "next IV:"
	    , md.st->st_new_iv, md.st->st_new_iv_len);
    }
    else
    {
	/* packet was not encryped -- should it have been? */

	if (smc->flags & SMF_INPUT_ENCRYPTED)
	{
	    log("packet should have been encrypted");
	    /* XXX Could send notification back */
	    free_md(&md);
	    return;
	}
    }

    /* Digest the message.
     * Padding must be removed to make hashing work.
     * Padding comes from encryption (so this code must be after decryption)
     * and from the requirement that messages must be "aligned" (padded
     * to a multiple of 4 octets).
     */
    {
	struct payload_digest *pd = md.digest;
	int np = md.hdr.isa_np;
	lset_t needed = smc->req_payloads;

	while (np != ISAKMP_NEXT_NONE)
	{
	    struct_desc *sd = np < ISAKMP_NEXT_ROOF? payload_descs[np] : NULL;

	    if (pd == &md.digest[PAYLIMIT])
	    {
		log("more than %d payloads in message; ignored", PAYLIMIT);
		free_md(&md);
		return;
	    }

	    if (sd == NULL)
	    {
		switch (np)
		{
		case ISAKMP_NEXT_ID:
		    sd = IS_PHASE1(md.from_state)
			? &isakmp_identification_desc : &isakmp_ipsec_identification_desc;
		    break;
		default:
		    log("unknown or unexpected payload type (%s) at outer level of packet -- ignored"
			, enum_show(&payload_names, np));
		    free_md(&md);
		    return;
		}
	    }

	    {
		lset_t s = LELEM(np);

		if (0 == (s & (needed | smc->opt_payloads
		| LELEM(ISAKMP_NEXT_N) | LELEM(ISAKMP_NEXT_D))))
		{
		    log("unexpected payload type (%s); packet -- ignored"
			, enum_show(&payload_names, np));
		    free_md(&md);
		    return;
		}
		needed &= ~s;
	    }

	    if (!in_struct(&pd->payload, sd, &md.message_pbs, &pd->pbs))
	    {
		log(smc->flags & SMF_FIRST_ENCRYPTED_INPUT
		    ? "probable authentication (preshared secret) failure: "
			"malformed payload in first encrypted message"
		    : "malformed payload in packet");
		free_md(&md);
		return;
	    }

	    /* place this payload at the end of the chain for this type */
	    {
		struct payload_digest **p;

		for (p = &md.chain[np]; *p != NULL; p = &(*p)->next)
		    ;
		*p = pd;
		pd->next = NULL;
	    }

	    np = pd->payload.generic.isag_np;
	    pd++;
	}

	md.digest_roof = pd;

	DBG(DBG_PARSING,
	    if (pbs_left(&md.message_pbs) != 0)
		DBG_log("removing %d bytes of padding", (int) pbs_left(&md.message_pbs)));

	md.message_pbs.roof = md.message_pbs.cur;

	/* check that all mandatory payloads appeared */

	if (needed != 0)
	{
	    log("message for %s is missing payloads %s"
		, enum_show(&state_names, md.from_state)
		, bitnamesof(payload_name, needed));
	    free_md(&md);
	    return;
	}
    }

    /* more sanity checking: enforce most ordering constraints */

    if (IS_PHASE1(md.from_state))
    {
	/* rfc2409: The Internet Key Exchange (IKE), 5 Exchanges:
	 * "The SA payload MUST precede all other payloads in a phase 1 exchange."
	 */
	if (md.chain[ISAKMP_NEXT_SA] != NULL
	&& md.hdr.isa_np != ISAKMP_NEXT_SA)
	{
	    log("Phase 1 message does not start with SA payload");
	    free_md(&md);
	    return;
	}
    }
    else if (IS_QUICK(md.from_state))
    {
	/* rfc2409: The Internet Key Exchange (IKE), 5.5 Phase 2 - Quick Mode
	 *
	 * "In Quick Mode, a HASH payload MUST immediately follow the ISAKMP
	 *  header and a SA payload MUST immediately follow the HASH."
	 * [NOTE: there may be more than one SA payload, so this is not
	 *  totally reasonable.  Probably all SAs should be so constrained.]
	 *
	 * "If ISAKMP is acting as a client negotiator on behalf of another
	 *  party, the identities of the parties MUST be passed as IDci and
	 *  then IDcr."
	 *
	 * "With the exception of the HASH, SA, and the optional ID payloads,
	 *  there are no payload ordering restrictions on Quick Mode."
	 */

	if (md.hdr.isa_np != ISAKMP_NEXT_HASH)
	{
	    log("Quick Mode message does not start with a HASH payload");
	    free_md(&md);
	    return;
	}

	{
	    struct payload_digest *p;
	    int i;

	    for (p = md.chain[ISAKMP_NEXT_SA], i = 1; p != NULL
	    ; p = p->next, i++)
	    {
		if (p != &md.digest[i])
		{
		    log("Quick Mode message has SA payload in wrong position");
		    free_md(&md);
		    return;
		}
	    }
	}

	/* rfc2409: The Internet Key Exchange (IKE), 5.5 Phase 2 - Quick Mode:
	 * "If ISAKMP is acting as a client negotiator on behalf of another
	 *  party, the identities of the parties MUST be passed as IDci and
	 *  then IDcr."
	 */
	{
	    struct payload_digest *id = md.chain[ISAKMP_NEXT_ID];

	    if (id != NULL
	    && (id+1 != id->next || id->next->next != NULL))
	    {
		log("Quick mode requires two ID payloads, and they must be adjacent");
		free_md(&md);
		return;
	    }
	}
    }

    /* Handle (ignore!) Delete/Notification/VendorID Payloads */
    /* XXX Handle deletions */
    /* XXX Handle Notifications */
    /* XXX Handle VID payloads */
    {
	struct payload_digest *p;

	for (p = md.chain[ISAKMP_NEXT_N]; p != NULL; p = p->next)
	{
	    log("ignoring informational payload, type %s"
		, enum_show(&ipsec_notification_names, p->payload.notification.isan_type));
	    DBG_cond_dump(DBG_PARSING, "info:", p->pbs.cur, pbs_left(&p->pbs));
	}

	for (p = md.chain[ISAKMP_NEXT_D]; p != NULL; p = p->next)
	{
	    log("ignoring delete SA payload");
	    DBG_cond_dump(DBG_PARSING, "del:", p->pbs.cur, pbs_left(&p->pbs));
	}

	for (p = md.chain[ISAKMP_NEXT_VID]; p != NULL; p = p->next)
	{
	    log("ignoring Vendor ID payload");
	    DBG_cond_dump(DBG_PARSING, "VID:", p->pbs.cur, pbs_left(&p->pbs));
	}
    }

    /* XXX Handle Commit Bit set and unset it */

    /* invoke state transition function to do the state-specific things.
     * These include:
     * - process and judge payloads
     * - advance st_state
     * - update st_iv (result of decryption is in st_new_iv)
     * - build reply packet
     * - indicate to us whether to transmit the reply packet
     */
    {
	stf_status result;

	/* ??? this buffer seems *way* too big */
	u_int8_t reply_buffer[UDP_SIZE];

	/* set up reply pb_stream */
	init_pbs(&md.reply, reply_buffer, sizeof(reply_buffer), "reply packet");

	/* do the real work! */
	result = smc->processor == NULL? STF_IGNORE : smc->processor(&md);
	cur_state = md.st;	/* may be fiddled (eg. by quick_outI1) */
	switch (result)
	{
	    case STF_IGNORE:
		break;
	    case STF_NO_REPLY:
	    case STF_REPLY:
		/* Delete previous retransmission event.
		 * New event will be scheduled below.
		 */
		delete_event(md.st);

		/* replace previous receive packet with latest */

		if (md.st->st_rpacket != NULL)
		    pfree(md.st->st_rpacket);

		if (md.encrypted)
		{
		    md.st->st_rpacket = md.raw_packet;
		    md.st->st_rpacket_len = md.raw_packet_len;
		    md.raw_packet = NULL;
		}
		else
		{
		    md.st->st_rpacket_len = pbs_room(&md.packet_pbs);
		    md.st->st_rpacket = clone_bytes(md.packet_pbs.start
			, md.st->st_rpacket_len, "raw packet");
		}

		/* free previous transmit packet */
		if (md.st->st_tpacket != NULL)
		{
		    pfree(md.st->st_tpacket);
		    md.st->st_tpacket = NULL;
		}

		/* if requested, send the new reply packet */
		if (result == STF_REPLY)
		{
		    close_output_pbs(&md.reply);   /* good form, but actually a no-op */

		    md.st->st_tpacket = clone_bytes(md.reply.start, pbs_offset(&md.reply),
			"reply packet");
		    md.st->st_tpacket_len = pbs_offset(&md.reply);

		    DBG_cond_dump(DBG_RAW, "sending:\n", md.st->st_tpacket, md.st->st_tpacket_len);

		    /* actually send the packet
		     * Note: this is a great place to implement "impairments"
		     * for testing purposes.  Suppress or duplicate the
		     * send_packet call depending on md.st->st_state.
		     */
		    send_packet(md.st, "STF_REPLY");
		}

		/* Schedule for whatever timeout is specified */
		{
		    u_int32_t delay;
		    int kind = smc->timeout_event;

		    switch (kind)
		    {
		    case EVENT_RETRANSMIT:	/* Retransmit packet */
			delay = EVENT_RETRANSMIT_DELAY;
			break;

		    case EVENT_SA_REPLACE:	/* SA replacement event */
		    case EVENT_SA_EXPIRE:	/* SA expiration event */
			/* if the lifetime(s) specified in negotiation
			 * (i.e. dictated by initiator) is longer
			 * than we will accept, force EVENT_SA_REPLACE
			 * so that we take on the job of rekeying.
			 * This may result in redundant SAs.
			 * To prevent this getting out of hand, the
			 * EVENT_SA_REPLACE code will only rekey an SA
			 * that is the newest for its connection.
			 * The accepted technique is to negotiate-down
			 * the lifetime with after-exchange NOTIFY,
			 * but we don't yet do this.
			 */
			if (IS_PHASE1(md.st->st_state))
			{
			    delay = md.st->st_connection->sa_ike_life_seconds;
			    if (delay >= md.st->st_oakley.life_seconds)
				delay = md.st->st_oakley.life_seconds;
			    else
				kind = EVENT_SA_REPLACE;
			}
			else
			{
			    /* Delay is min of up to three things:
			     * each can limit the lifetime.
			     */
			    delay = md.st->st_connection->sa_ipsec_life_seconds;
			    if (md.st->st_ah.present
			    && delay >= md.st->st_ah.attrs.life_seconds)
				delay = md.st->st_ah.attrs.life_seconds;
			    else if (md.st->st_esp.present
			    && delay >= md.st->st_esp.attrs.life_seconds)
				delay = md.st->st_esp.attrs.life_seconds;
			    else
				kind = EVENT_SA_REPLACE;
			}

			if (kind == EVENT_SA_REPLACE)
			{
			    /* If we have enough time, shave some for
			     * replacement.  Otherwise, don't attempt.
			     * In fact, we should always have time.
			     * Whack enforces this restriction on our
			     * own lifetime.  If a smaller liftime comes
			     * from the other IKE, we won't have
			     * EVENT_SA_REPLACE.
			     */
			    if (delay > md.st->st_connection->sa_rekey_window)
				delay -= md.st->st_connection->sa_rekey_window;
			    else
				kind = EVENT_SA_EXPIRE;
			}
			break;

		    case EVENT_NULL:	/* non-event */
		    case EVENT_REINIT_SECRET:	/* Refresh cookie secret */
		    default:
			passert(FALSE);
		    }
		    event_schedule(kind, delay, md.st);
		}

		/* tell whack what just happened */
		switch (md.st->st_state)
		{
		case STATE_MAIN_R3:
		case STATE_MAIN_I4:
		case STATE_QUICK_I2:
		case STATE_QUICK_R2:
		    /* Tell Whack about our success,
		     * and close socket to let Whack finish.
		     * Note: if the pending_quick directed us to follow
		     * with Phase 2, the socket fd will have been
		     * dup'ed by main_inR3().
		     */
		    log("%s", state_story[md.st->st_state - STATE_MAIN_R0]);
		    whack_log(md.st->st_whack_sock, RC_SUCCESS
			, "%s: SA established"
			, enum_name(&state_names, md.st->st_state));
		    release_whack(md.st);
		    break;

		default:
		    whack_log(md.st->st_whack_sock, RC_NEW_STATE + md.st->st_state
			, "%s: from %s; %s"
			, enum_name(&state_names, md.st->st_state)
			, enum_name(&state_names, md.from_state)
			, state_story[md.st->st_state - STATE_MAIN_R0]);
		    break;
		}
		break;

	    case STF_INTERNAL_ERROR:
		whack_log(md.st->st_whack_sock, RC_INTERNALERR + md.note
		    , "%s: internal error"
		    , enum_name(&state_names, md.st->st_state));

		DBG(DBG_CONTROL,
		    DBG_log("state transition function for %s had internal error",
			enum_name(&state_names, md.from_state)));
		break;

	    default:	/* a shortcut to STF_FAIL, setting md.note */
		md.note = result - STF_FAIL;
		result = STF_FAIL;
		/* FALL THROUGH ... */
	    case STF_FAIL:
		/* XXX Could send notification back
		 * As it is, we act as if this message never happened:
		 * whatever retrying was in place, remains in place.
		 */
		whack_log(md.st->st_whack_sock, RC_NOTIFICATION + md.note
		    , "%s: %s", enum_name(&state_names, md.st->st_state)
		    , enum_name(&ipsec_notification_names, md.note));

		DBG(DBG_CONTROL,
		    DBG_log("state transition function for %s failed: %s"
			, enum_name(&state_names, md.from_state)
			, enum_name(&ipsec_notification_names, md.note)));
		break;
	}
    }
    free_md(&md);
}
