/*
 * SA grouping
 * Copyright (C) 1996  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char spigrp_c_version[] = "RCSID $Id: spigrp.c,v 1.18 1999/04/11 00:12:09 henry Exp $";


#include <sys/types.h>
#include <linux/types.h> /* new */
#include <string.h>
#include <errno.h>

#include <sys/socket.h>

#include <netinet/in.h>
#include <arpa/inet.h>
/* #include <linux/ip.h> */

#include <unistd.h>
#include <stdio.h>
#include <netdb.h>
#include "../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_netlink.h"
#include "ipsec_ah.h"


__u32 bigbuf[1024];
char *program_name;

inline int x2i(char *s)
{
	char ss[3];
	char *endptr;
	__u32 n;
	ss[0] = s[0];
	ss[1] = s[1];
	ss[2] = 0;

	n = strtoul(ss, &endptr, 16);
	if(!(endptr == ss + strlen(ss)))
	{
		fprintf(stderr, "%s: Invalid character in numerical input stream: %s\n",
			program_name, ss);
		exit (1);
	}
	return n;
}

int
resolve_ip(char *arg, struct in_addr *ip)
{
	struct hostent *hostp;
	struct netent *netp;
	
	if (!strcmp(arg, "default")) {/* default is really 0.0.0.0 */
		ip->s_addr = htonl(INADDR_ANY);
		return(1);
	}
	if (!(strcmp(arg, "255.255.255.255"))) {
		/* inet_addr() returns -1 for this, as it should, but that means an error */
		ip->s_addr = htonl(INADDR_BROADCAST);
		return (1);
	}
	if ((netp = getnetbyname(arg)) != (struct netent *)NULL) {
		ip->s_addr = htonl(netp->n_net);
		return(1);
	}
	if ((hostp = gethostbyname(arg)) == (struct hostent *)NULL) {
		errno = h_errno;
		return(-1);
	}
	memcpy((char *) ip, (char *) hostp->h_addr_list[0], hostp->h_length);
	return(0);
}

void
usage(char *s)
{
	fprintf(stdout, "usage: %s [ --label <label> ] dst1 spi1 proto1 [ dst2 spi2 proto2 [ dst3 spi3 proto3 [ dst4 spi4 proto4 ] ] ] \n", s);
	fprintf(stdout, "usage: %s --help\n", s);
	fprintf(stdout, "usage: %s --version\n", s);
	exit(1);
}

	
int
main(argc, argv)
int argc;
char **argv;
{
	int fd;
	int ret;
	int i, j, dlen, nspis;
	char *endptr;

	struct encap_msghdr *em = (struct encap_msghdr *)bigbuf;

	program_name = argv[0];
	if (argc == 1)
                usage(program_name);

        if(strcmp(argv[1], "--help") == 0) {
                usage(program_name);
        }

        if(strcmp(argv[1], "--version") == 0) {
                fprintf(stderr, "%s, %s\n", program_name, spigrp_c_version);
                exit(1);
        }

        if(strcmp(argv[1], "--label") == 0) {
		if(argc > 2) {
			program_name = argv[2];
			argv += 2;
			argc -= 2;
		} else {
			fprintf(stderr, "%s: --label option requires an argument.\n",
				program_name);
			exit(1);
		}
        }

	if ((argc < 4) || (argc > 13) || ((argc % 3) != 1))
		usage(program_name);

	nspis = argc / 3;

	em->em_magic = EM_MAGIC;
	em->em_version = 0;
	if(nspis == 1)
		em->em_type = EMT_UNGRPSPIS;
	else
		em->em_type = EMT_GRPSPIS;
	em->em_msglen = EMT_GRPSPIS_FLEN +
		nspis *
#if 0
		(sizeof(struct sa_id) + sizeof(struct tdb*));
#else
		(sizeof(em->em_rel[0]));
#endif
	for(i = 0; i < nspis; i++) {
		if(!strcmp(argv[i*3+3], "ah")) {
			em->em_rel[i].emr_proto = SA_AH;
		}
		if(!strcmp(argv[i*3+3], "esp")) {
			em->em_rel[i].emr_proto = SA_ESP;
		}
		if(!strcmp(argv[i*3+3], "tun")) {
			em->em_rel[i].emr_proto = SA_IPIP;
		}
		if(em->em_rel[i].emr_proto == 0) {
			fprintf(stderr, "%s: Badly formed proto: %s\n",
				program_name, argv[i*3+3]);
			exit(1);
		}
		em->em_rel[i].emr_spi = htonl(strtoul(argv[i*3+2], &endptr, 0));
		if(!(endptr == argv[i*3+2] + strlen(argv[i*3+2]))) {
			fprintf(stderr, "%s: Badly formed spi: %s\n",
				program_name, argv[i*3+2]);
			exit(1);
		}
		if(ret = resolve_ip(argv[i*3+1] /*optarg*/, &(em->em_rel[i].emr_dst)) != 0) {
			if(ret == 1) {
				fprintf(stderr, "%s: Destination address cannot be a network: %s\n",
					program_name, argv[i*3+1]/*optarg*/);
				exit (1);
			} else {
				fprintf(stderr, "%s: Invalid formation of destination address: %s\n",
					program_name, argv[i*3+1]/*optarg*/);
				exit (1);
			}
		}
#ifdef DEBUG
		fprintf(stdout, "SA (size:%d) %d contains: ",
			(sizeof(struct sa_id) + sizeof(struct tdb*)), i+1);
		for(j = 0; j < sizeof(struct sa_id); j++) {
			fprintf(stdout, " %02x",
				((char*)(em))
				[8 + i * (sizeof(struct sa_id) + sizeof(struct tdb*)) + j]);
		}
		fprintf(stdout, "\n");
		fprintf(stdout, "proto = %d\n", em->em_rel[i].emr_said.proto);
		fprintf(stdout, "spi = %08x\n", em->em_rel[i].emr_said.spi);
		fprintf(stdout, "edst = %08x\n", em->em_rel[i].emr_said.dst);
#endif
	}	

	fd = open("/dev/ipsec", 2);
	if (fd < 0) {
		printf("%s: Could not open /dev/ipsec -- ", program_name);
		switch(errno) {
		case ENOENT:
			printf("device does not exist.  See FreeS/WAN installation procedure.\n");
		case EACCES:
			printf("access denied.  ");
			if(getuid() == 0)
			{
				printf("Check permissions.  Should be 600.\n");
			}
			else
			{
				printf("You must be root to open this file.\n");
			}
			break;
		case EUNATCH:
			printf("Netlink not enabled OR KLIPS not loaded.\n");
			break;
		default:
			printf("Unknown file open error %d\n", errno);
		}
		exit(1);
	}
	if (write(fd, (caddr_t)em, em->em_msglen) != em->em_msglen) {
		close(fd);
		printf("%s: Had trouble writing to /dev/ipsec -- ", program_name);
		switch(errno) {
		case EINVAL:
			printf("Invalid argument, check kernel log messages for specifics.\n");
			break;
		case ENXIO:
			printf("SA not found.  Check kernel log messages for specifics.\n");
			break;
		case EBUSY:
			printf("SA in use.  Check kernel log messages for specifics.\n");
			break;
		default:
			printf("Unknown file write error %d\n", errno);
		}
		close(fd);
		exit(1);
	}
	close(fd);
	exit(0);
}
/*
 * $Log: spigrp.c,v $
 * Revision 1.18  1999/04/11 00:12:09  henry
 * GPL boilerplate
 *
 * Revision 1.17  1999/04/06 04:54:39  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.16  1999/03/17 15:40:54  rgb
 * Make explicit main() return type of int.
 *
 * Revision 1.15  1999/01/28 23:20:49  rgb
 * Replace hard-coded numbers in macros and code with meaningful values
 * automatically generated from sizeof() and offsetof() to further the
 * goal of platform independance.
 *
 * Revision 1.14  1999/01/22 06:36:46  rgb
 * 64-bit clean-up.
 *
 * Revision 1.13  1998/11/12 21:08:04  rgb
 * Add --label option to identify caller from scripts.
 *
 * Revision 1.12  1998/10/26 01:28:38  henry
 * use SA_* protocol names, not IPPROTO_*, to avoid compile problems
 *
 * Revision 1.11  1998/10/25 02:47:09  rgb
 * Fix bug in size of stucture passed in from user space for grpspi command.
 * Added debugging code to find spigrp stucture size mismatch bug.
 * Convert switch to loop for more efficient coding and redundant code elimination.
 *
 * Revision 1.10  1998/10/19 18:58:56  rgb
 * Added inclusion of freeswan.h.
 * a_id structure implemented and used: now includes protocol.
 *
 * Revision 1.9  1998/10/09 04:36:32  rgb
 * Changed help output from stderr to stdout.
 * Avoid use of argv[0] after first use.
 *
 * Revision 1.8  1998/08/05 22:24:45  rgb
 * Change includes to accomodate RH5.x
 *
 * Revision 1.7  1998/07/29 21:43:17  rgb
 * Convert to 0x-prefixed spis.
 * Support dns lookups for hostnames.
 *
 * Revision 1.6  1998/07/14 18:24:05  rgb
 * Remove unused skbuff header.
 *
 * Revision 1.5  1998/07/09 18:14:11  rgb
 * Added error checking to IP's and keys.
 * Made most error messages more specific rather than spamming usage text.
 * Added more descriptive kernel error return codes and messages.
 * Converted all spi translations to unsigned.
 * Removed all invocations of perror.
 *
 * Revision 1.4  1998/06/30 18:04:32  rgb
 * Fix compiler warning: couldn't find 'struct option' prototype.
 *
 * Revision 1.3  1998/05/27 18:48:20  rgb
 * Adding --help and --version directives.
 *
 * Revision 1.2  1998/05/18 21:14:16  rgb
 * Modifications to be able to ungroup spi's.
 *
 * Revision 1.1.1.1  1998/04/08 05:35:09  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.3  1996/11/20 14:51:32  ji
 * Fixed problems with #include paths.
 * Changed (incorrect) references to ipsp into ipsec.
 *
 * Revision 0.2  1996/11/08 15:46:29  ji
 * First limited release.
 *
 *
 */
