/*
 * IPSEC <> netlink interface
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char ipsec_netlink_c_version[] = "RCSID $Id: ipsec_netlink.c,v 1.24 1999/04/11 00:28:58 henry Exp $";

#include <linux/config.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/skbuff.h>
#include <asm/checksum.h>
#include <net/ip.h>
#include <net/netlink.h>

#include "../../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"

#include "ipsec_rcv.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

#ifdef DEBUG_IPSEC
#include "ipsec_tunnel.h"
#endif

#ifdef DEBUG_IPSEC
int debug_netlink = 0;
#endif

#define SENDERR(_x) do { len = -(_x); goto errlab; } while (0)

int 
ipsec_callback(struct sk_buff *skb)
{
	/*
	 * this happens when we write to /dev/ipsec (c 36 10)
	 */
	int len = skb->len;
	u_char *dat = (u_char *)skb->data;
	struct encap_msghdr *em	= (struct encap_msghdr *)dat;
	struct tdb *tdbp, *tprev;
	int i, nspis, error;
#ifdef DEBUG_IPSEC
	struct eroute *eret;
	char sa[SATOA_BUF];
	size_t sa_len;

	sa_len = satoa(em->em_said, 0, sa, SATOA_BUF);

	if(debug_netlink) {
		printk("klips_debug:ipsec_callback: skb=0x%p skblen=%ld em_magic=%d em_type=%d\n",
		       skb, skb->len, em->em_magic, em->em_type);
		switch(em->em_type) {
		case EMT_SETDEBUG:
			printk("klips_debug:ipsec_callback: set ipsec_debug level\n");
			break;
		case EMT_DELEROUTE:
		case EMT_CLREROUTE:
		case EMT_CLRSPIS:
			break;
		default:
			printk("klips_debug:ipsec_callback: called for SA:%s\n", sa);
		}
	}
#endif	
	/*	em = (struct encap_msghdr *)dat; */
	if (em->em_magic != EM_MAGIC) {
		SENDERR(EINVAL);
	}
	switch (em->em_type) {
	case EMT_SETDEBUG:
#ifdef DEBUG_IPSEC
		if(em->em_db_nl >> (sizeof(em->em_db_nl) * 8 - 1)) {
			if(debug_netlink)
				printk("klips_debug:ipsec_callback: set\n");
			debug_tunnel  |= em->em_db_tn;
			debug_netlink |= em->em_db_nl;
			debug_xform   |= em->em_db_xf;
			debug_eroute  |= em->em_db_er;
			debug_spi     |= em->em_db_sp;
			debug_radij   |= em->em_db_rj;
			debug_esp     |= em->em_db_es;
			debug_ah      |= em->em_db_ah;
			debug_rcv     |= em->em_db_rx;
		} else {
			if(debug_netlink)
				printk("klips_debug:ipsec_callback: unset\n");
			debug_tunnel  &= em->em_db_tn;
			debug_netlink &= em->em_db_nl;
			debug_xform   &= em->em_db_xf;
			debug_eroute  &= em->em_db_er;
			debug_spi     &= em->em_db_sp;
			debug_radij   &= em->em_db_rj;
			debug_esp     &= em->em_db_es;
			debug_ah      &= em->em_db_ah;
			debug_rcv     &= em->em_db_rx;
		}
#else
		printk("klips_debug:ipsec_callback: debugging not enabled\n");
		SENDERR(EINVAL);
#endif	
		break;

	case EMT_SETEROUTE:
		if ((error = ipsec_makeroute(&(em->em_eaddr), &(em->em_emask), em->em_ersaid)))
			SENDERR(error);
		break;

	case EMT_RPLACEROUTE:
		if ((error = ipsec_breakroute(&(em->em_eaddr), &(em->em_emask))) == EINVAL) {
				SENDERR(error);
		}
		if ((error = ipsec_makeroute(&(em->em_eaddr), &(em->em_emask), em->em_ersaid)))
			SENDERR(error);
		break;

	case EMT_DELEROUTE:
		if ((error = ipsec_breakroute(&(em->em_eaddr), &(em->em_emask))))
			SENDERR(error);
		break;

	case EMT_CLREROUTE:
		if ((error = ipsec_cleareroutes()))
			SENDERR(error);
		break;

#ifdef DEBUG_IPSEC
	case EMT_TESTROUTE:
		      
		printk("klips_debug:ipsec_callback: testroute: trying to locate 0x%08x->0x%08x", (u_int)em->em_eaddr.sen_ip_src.s_addr, (u_int)em->em_eaddr.sen_ip_dst.s_addr);

		eret = ipsec_findroute(&(em->em_eaddr));
		
		if (eret == NULL) {
			printk(" not found\n");
		} else {
			printk(" via SA: %s\n", sa);
		}
		break;
#endif
	case EMT_SETSPI:
		if (em->em_if >= 5)	/* XXX -- why 5? */
			SENDERR(ENODEV);
		
		tdbp = gettdb(em->em_said);
		if (tdbp == NULL) {
			KLIPS_PRINT(debug_netlink & DB_NL_TDBCB,
				    "klips_debug:ipsec_callback: existing Tunnel Descriptor Block not found (this \n"
				    "klips_debug:                is good) for SA: %s, %s-bound, allocating.\n",
				    sa, (em->em_flags & EMT_INBOUND) ? "in" : "out");
			tdbp = (struct tdb *)kmalloc(sizeof (*tdbp), GFP_ATOMIC);

			if (tdbp == NULL)
				SENDERR(ENOBUFS);
			
			memset((caddr_t)tdbp, 0, sizeof(*tdbp));
			
			tdbp->tdb_said = em->em_said;
			tdbp->tdb_flags = em->em_flags;
			
/* XXX  		tdbp->tdb_rcvif = &(enc_softc[em->em_if].enc_if);*/
			tdbp->tdb_rcvif = NULL;
			puttdb(tdbp);
		} else {
			KLIPS_PRINT(debug_netlink & DB_NL_TDBCB,
				    "klips_debug:ipsec_callback: EMT_SETSPI found an old Tunnel Descriptor Block\n"
				    "klips_debug:                for SA: %s, delete it first.\n", sa);
			SENDERR(EEXIST);
		}
		
		if ((error = tdb_init(tdbp, em))) {
			KLIPS_PRINT(debug_netlink & DB_NL_TDBCB,
				    "klips_debug:ipsec_callback: EMT_SETSPI not successful for SA: %s, deleting.\n", sa);
			ipsec_tdbwipe(tdbp);
			deltdb(tdbp);
			memset((caddr_t)tdbp, 0, sizeof(*tdbp));
			kfree_s(tdbp, sizeof(*tdbp));
			
			SENDERR(error);
		}
		KLIPS_PRINT(debug_netlink & DB_NL_TDBCB,
			    "klips_debug:ipsec_callback: EMT_SETSPI successful for SA: %s\n", sa);
		break;
		
	case EMT_DELSPI:
		if (em->em_if >= 5)	/* XXX -- why 5? */
			SENDERR(ENODEV);
		
		tdbp = gettdb(em->em_said);
		if (tdbp == NULL) {
			KLIPS_PRINT(debug_netlink & DB_NL_TDBCB,
				    "klips_debug:ipsec_callback: EMT_DELSPI Tunnel Descriptor Block not found for SA:\n"
				    "klips_debug:                %s, could not delete.\n", sa);
			SENDERR(ENXIO);  /* XXX -- wrong error message... */
		} else {
			while(tdbp->tdb_onext) {
				tdbp = tdbp->tdb_onext;
			}
			while(tdbp) {
				tprev = tdbp;
				tdbp = tdbp->tdb_inext;
				tprev->tdb_inext = NULL;
				if(tdbp)
					tdbp->tdb_onext = NULL;
				ipsec_tdbwipe(tprev);
				deltdb(tprev);
				memset((caddr_t)tprev, 0, sizeof(*tprev));
				kfree_s(tprev, sizeof(*tprev));
			}
		}
		break;
		
	case EMT_GRPSPIS:
		nspis = (len - EMT_GRPSPIS_FLEN) / sizeof(em->em_rel[0]);
		if ((nspis * (sizeof(em->em_rel[0]))) != (len - EMT_GRPSPIS_FLEN)) {
			printk("klips_debug:ipsec_callback: EMT_GRPSPI message size incorrect, expected nspis(%d)*%d, got %d.\n",
			       nspis,
			       sizeof(em->em_rel[0]),
			       (len - EMT_GRPSPIS_FLEN));
			SENDERR(EINVAL);
			break;
		}
		
		for (i = 0; i < nspis; i++) {
			KLIPS_PRINT(debug_netlink,
				    "klips_debug:ipsec_callback: EMT_GRPSPI for SA(%d)\n"
				    "klips_debug:                %s,\n", i, sa);
			if ((tdbp = gettdb(em->em_rel[i].emr_said)) == NULL) {
				KLIPS_PRINT(debug_netlink,
					    "klips_debug:ipsec_callback: EMT_GRPSPI Tunnel Descriptor Block not found for SA:\n"
					    "klips_debug:                %s, could not group.\n", sa);
				SENDERR(ENXIO);
			} else {
				if(tdbp->tdb_inext || tdbp->tdb_onext) {
					KLIPS_PRINT(debug_netlink,
						    "klips_debug:ipsec_callback: EMT_GRPSPI Tunnel Descriptor Block already grouped  \n"
						    "klips_debug:                for SA: %s, can't regroup.\n", sa);
					SENDERR(EBUSY);
				}
				em->em_rel[i].emr_tdb = tdbp;
			}
		}
		tprev = em->em_rel[0].emr_tdb;
		tprev->tdb_inext = NULL;
		for (i = 1; i < nspis; i++) {
			tdbp = em->em_rel[i].emr_tdb;
			tprev->tdb_onext = tdbp;
			tdbp->tdb_inext = tprev;
			tprev = tdbp;
		}
		tprev->tdb_onext = NULL;
		error = 0;
		break;
		
	case EMT_UNGRPSPIS:
		if (len != (8 + (sizeof(struct sa_id) + sizeof(struct tdb *)) /* 12 */) ) {
			SENDERR(EINVAL);
			break;
		}
		
		if ((tdbp = gettdb(em->em_rel[0].emr_said)) == NULL) {
			KLIPS_PRINT(debug_netlink,
				    "klips_debug:ipsec_callback: EMT_UGRPSPI Tunnel Descriptor Block not found for SA:\n"
				    "klips_debug:                %s, could not ungroup.\n", sa);
			SENDERR(ENXIO);
		}
		while(tdbp->tdb_onext) {
			tdbp = tdbp->tdb_onext;
		}
		while(tdbp->tdb_inext) {
			tprev = tdbp;
			tdbp = tdbp->tdb_inext;
			tprev->tdb_inext = NULL;
			tdbp->tdb_onext = NULL;
		}

		error = 0;
		break;
		
	case EMT_CLRSPIS:
		KLIPS_PRINT(debug_netlink,
			    "klips_debug:ipsec_callback: spi clear called.\n");
		if (em->em_if >= 5)	/* XXX -- why 5? */
			SENDERR(ENODEV);
		ipsec_tdbcleanup();
		break;
	default:
		KLIPS_PRINT(debug_netlink,
			    "klips_debug:ipsec_callback: unknown message type\n");
		SENDERR(EINVAL);
	}
 errlab:
	kfree_skb(skb, FREE_WRITE);
	return len;
}

/*
 * $Log: ipsec_netlink.c,v $
 * Revision 1.24  1999/04/11 00:28:58  henry
 * GPL boilerplate
 *
 * Revision 1.23  1999/04/07 17:44:21  rgb
 * Fix ipsec_callback memory leak, skb not freed after use.
 *
 * Revision 1.22  1999/04/06 04:54:26  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.21  1999/02/17 16:50:11  rgb
 * Consolidate satoa()s for space and speed efficiency.
 * Convert DEBUG_IPSEC to KLIPS_PRINT
 * Clean out unused cruft.
 *
 * Revision 1.20  1999/01/28 23:20:49  rgb
 * Replace hard-coded numbers in macros and code with meaningful values
 * automatically generated from sizeof() and offsetof() to further the
 * goal of platform independance.
 *
 * Revision 1.19  1999/01/26 02:07:07  rgb
 * Removed CONFIG_IPSEC_ALGO_SWITCH macro.
 * Remove ah/esp switching on include files.
 * Removed dead code.
 *
 * Revision 1.18  1999/01/22 06:20:36  rgb
 * Cruft clean-out.
 * 64-bit clean-up.
 * Added algorithm switch code.
 *
 * Revision 1.17  1998/12/02 03:09:39  rgb
 * Clean up debug printing conditionals to compile with debugging off.
 *
 * Revision 1.16  1998/12/01 05:56:57  rgb
 * Add support for debug printing of version info.
 * Fail on unknown error for breakroute in replace command.
 *
 * Revision 1.15  1998/11/30 13:22:54  rgb
 * Rationalised all the klips kernel file headers.  They are much shorter
 * now and won't conflict under RH5.2.
 *
 * Revision 1.14  1998/11/10 05:36:14  rgb
 * Clean up debug output.
 * Add direction to spi setup debug code.
 * Add support for SA direction flag.
 *
 * Revision 1.13  1998/10/31 06:51:56  rgb
 * Get zeroize to return something useful.
 * Clean up code to isolate 'spi --add/del' memory leak.
 * Fixed up comments in #endif directives.
 *
 * Revision 1.12  1998/10/27 00:35:02  rgb
 * Supressed debug output during normal operation.
 *
 * Revision 1.11  1998/10/25 02:40:21  rgb
 * Selective debug printing, depending upon called service.
 * Institute more precise error return codes from eroute commands.
 * Fix bug in size of stucture passed in from user space for grpspi command.
 *
 * Revision 1.10  1998/10/22 06:44:58  rgb
 * Convert to use satoa for printk.
 * Moved break; in 'set debug level code to avoid undetected bug.
 * Fixed run-on error message to fit 80 columns.
 *
 * Revision 1.9  1998/10/19 14:44:28  rgb
 * Added inclusion of freeswan.h.
 * sa_id structure implemented and used: now includes protocol.
 *
 * Revision 1.8  1998/10/09 04:29:51  rgb
 * Added support for '-replace' option to eroute.
 * Fixed spiungroup bug.
 * Added 'klips_debug' prefix to all klips printk debug statements.
 *
 * Revision 1.7  1998/08/12 00:10:06  rgb
 * Fixed minor error return code syntax.
 *
 * Revision 1.6  1998/07/29 20:22:57  rgb
 * Cosmetic cleanup.
 *
 * Revision 1.5  1998/07/27 21:53:11  rgb
 * Check for proper return code from eroute clear command.
 * Use appropriate error return codes from kernel.
 * Add an option to clear the SA table.
 *
 * Revision 1.4  1998/07/14 18:02:40  rgb
 * Add a command to clear the eroute table.
 * Clean up some error codes.
 *
 * Revision 1.3  1998/06/25 19:52:33  rgb
 * Code cosmetic changes only.
 *
 * Revision 1.2  1998/06/23 02:57:58  rgb
 * Clean up after an error condition in setspi.
 *
 * Revision 1.9  1998/06/18 21:29:06  henry
 * move sources from klips/src to klips/net/ipsec, to keep stupid kernel
 * build scripts happier in presence of symbolic links
 *
 * Revision 1.8  1998/06/08 17:57:15  rgb
 * Very minor spacing change.
 *
 * Revision 1.7  1998/05/18 21:46:45  rgb
 * Clean up for numerical consistency of output.
 *
 * Added debugging switch output.
 *
 * SETSPI will refuse to overwrite a previous SA.  This is to make it
 * consistent with the eroute command.
 *
 * spidel now deletes entire chain of spi's.
 *
 * spigrp can now ungroup a set of spi's.
 *
 * spigrp will not regroup a previously grouped spi.
 *
 * Key data is properly cleaned up, ie. zeroed.
 *
 * Revision 1.6  1998/05/07 20:36:27  rgb
 * Fixed case where debugging not enabled that caused ipsec_netlink.c to
 * not compile.
 *
 * Revision 1.5  1998/05/06 03:34:21  rgb
 * Updated debugging output statements.
 *
 * Revision 1.4  1998/04/23 21:03:59  rgb
 * Completed kernel development for userspace access to klips kernel debugging
 * switches.
 * Added detail to the kernel error message when trying to group non-existant
 * spi's.
 *
 * Revision 1.3  1998/04/21 21:29:06  rgb
 * Rearrange debug switches to change on the fly debug output from user
 * space.  Only kernel changes checked in at this time.  radij.c was also
 * changed to temporarily remove buggy debugging code in rj_delete causing
 * an OOPS and hence, netlink device open errors.
 *
 * Revision 1.2  1998/04/12 22:03:23  rgb
 * Updated ESP-3DES-HMAC-MD5-96,
 * 	ESP-DES-HMAC-MD5-96,
 * 	AH-HMAC-MD5-96,
 * 	AH-HMAC-SHA1-96 since Henry started freeswan cvs repository
 * from old standards (RFC182[5-9] to new (as of March 1998) drafts.
 *
 * Fixed eroute references in /proc/net/ipsec*.
 *
 * Started to patch module unloading memory leaks in ipsec_netlink and
 * radij tree unloading.
 *
 * Revision 1.1  1998/04/09 03:06:08  henry
 * sources moved up from linux/net/ipsec
 *
 * Revision 1.1.1.1  1998/04/08 05:35:02  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.4  1997/01/15 01:28:15  ji
 * No changes.
 *
 * Revision 0.3  1996/11/20 14:39:04  ji
 * Minor cleanups.
 * Rationalized debugging code.
 *
 * Revision 0.2  1996/11/02 00:18:33  ji
 * First limited release.
 *
 *
 */
