/*
 * Initialization code, and /proc file system interface code.
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char ipsec_init_c_version[] = "RCSID $Id: ipsec_init.c,v 1.17 1999/04/11 00:28:57 henry Exp $";

#include <linux/config.h>

#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/in.h>          /* struct sockaddr_in */
#include <linux/skbuff.h>
#include <asm/checksum.h>
#include <net/ip.h>
#include <linux/proc_fs.h>
#include <net/netlink.h>

#include "../../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_tunnel.h"

#include "version.c"

#include "ipsec_ah.h"
#include "ipsec_esp.h"

#ifdef DEBUG_IPSEC
int debug_eroute = 0;
int debug_spi = 0;
#endif /* DEBUG_IPSEC */

unsigned int ipsec_spi_next = 0x1000;

static int
ipsec_eroute_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
	struct wsbuf w = {buffer, length, offset, 0, 0, 0, 0};

#ifdef DEBUG_IPSEC
	if (debug_radij & DB_RJ_DUMPTREES)
	  rj_dumptrees();			/* XXXXXXXXX */
#endif /* DEBUG_IPSEC */

	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_eroute_get_info: buffer=0x%p,"
		    " *start=0x%x, offset=%d, length=%d\n",
		    buffer, (u_int)*start, (int)offset, length);

	rj_walktree(rnh, ipsec_rj_walker_procprint, &w);
/*	rj_walktree(mask_rjhead, ipsec_rj_walker_procprint, &w); */

	*start = buffer + (offset - w.begin);	/* Start of wanted data */
	w.len -= (offset - w.begin);			/* Start slop */
	if (w.len > length)
		w.len = length;
	return w.len;
}

static int
ipsec_spi_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
	int len = 0;
	off_t pos = 0, begin = 0;
	int i;
	struct tdb *tdbp;
	char sa[SATOA_BUF];
	char buf_s[16], buf_d[16];
	size_t sa_len;

	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_spi_get_info: buffer=0x%p,"
		    "*start=0x%x, offset=%d, length=%d\n",
		    buffer, (u_int)*start, (int)offset, length);
	
	for (i = 0; i < TDB_HASHMOD; i++)
		for (tdbp = tdbh[i]; tdbp; tdbp = tdbp->tdb_hnext) {
			sa_len = satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);
			len += sprintf(buffer + len, "%s ", sa);
			switch(tdbp->tdb_proto) {
			case IPPROTO_AH:
				len += sprintf(buffer + len, "AH");
				break;
			case IPPROTO_ESP:
				len += sprintf(buffer + len, "ESP");
				break;
			case IPPROTO_IPIP:
				len += sprintf(buffer + len, "IPIP");
				break;
#if 0 /* don't get your hopes up just yet... */
			case IPPROTO_COMP:
				len += sprintf(buffer + len, "COMP");
				break;
#endif
			default:
				len += sprintf(buffer + len, "UNKNOWN");
			}

			len += sprintf(buffer + len, "_proto");

			if(tdbp->tdb_encalg) {
				len += sprintf(buffer + len, "_");
				switch(tdbp->tdb_encalg) {
				case ESP_DES:
					len += sprintf(buffer + len, "DES");
					break;
				case ESP_3DES:
					len += sprintf(buffer + len, "3DES");
					break;
				case ESP_NULL:
					len += sprintf(buffer + len, "NULL");
					break;
				default:
					len += sprintf(buffer + len, "UNKNOWN");
				}
				
				len += sprintf(buffer + len, "_encrypt");
			}

			if(tdbp->tdb_encalg || tdbp->tdb_authalg) {
				len += sprintf(buffer + len, "_");
				switch(tdbp->tdb_authalg) {
				case AH_NONE:
					len += sprintf(buffer + len, "NO");
					break;
				case AH_MD5:
					len += sprintf(buffer + len, "MD5");
					break;
				case AH_SHA:
					len += sprintf(buffer + len, "SHA1");
					break;
				default:
					len += sprintf(buffer + len, "UNKNOWN");
				}
				
				len += sprintf(buffer + len, "_auth");
			}

			len += sprintf(buffer + len, ":");
			len += sprintf(buffer + len, "  dir=%s",
				       (tdbp->tdb_flags & EMT_INBOUND) ?
				       "in" : "out");
			if(tdbp->tdb_iv_bits) {
				int j;
				len += sprintf(buffer + len, "  ivlen=%dbits  iv=0x",
					       tdbp->tdb_iv_bits);
				for(j = 0; j < tdbp->tdb_iv_bits / sizeof(__u8); j++) {
					len += sprintf(buffer + len, "%02x",
						       (__u32)((__u8*)(tdbp->tdb_iv))[j]);
				}
			}
			if(tdbp->tdb_encalg || tdbp->tdb_authalg) {
				len += sprintf(buffer + len, "  win=%d",
					       tdbp->tdb_replaywin);
				if(/* tdbp->tdb_replaywin */ 1) {
					len += sprintf(buffer + len, "  seq=%d  bit=0x%08Lx",
						       tdbp->tdb_replaywin_lastseq,
						       tdbp->tdb_replaywin_bitmap);
				}
			}
			len += sprintf(buffer + len, "  flags=0x%x",
				       tdbp->tdb_flags);
			if(tdbp->tdb_flags & ~EMT_INBOUND) {
				len += sprintf(buffer + len, "<");
				/* flag printing goes here */
				len += sprintf(buffer + len, ">");
			}
			if(tdbp->tdb_auth_bits) {
				len += sprintf(buffer + len, "  alen=%dbits",
					       tdbp->tdb_auth_bits);
			}
			if(tdbp->tdb_key_bits_a) {
				len += sprintf(buffer + len, "  aklen=%dbits",
					       tdbp->tdb_key_bits_a);
			}
			if(tdbp->tdb_key_bits_e) {
				len += sprintf(buffer + len, "  eklen=%dbits",
					       tdbp->tdb_key_bits_e);
			}
			if(tdbp->tdb_addr_s && tdbp->tdb_addr_d) {
				addrtoa(((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr,
					0, buf_s, sizeof(buf_s));
				addrtoa(((struct sockaddr_in*)(tdbp->tdb_addr_d))->sin_addr,
					0, buf_d, sizeof(buf_d));
				len += sprintf(buffer + len, "   %s -> %s",
					       buf_s, buf_d);
			}
			len += sprintf(buffer + len, "\n");
			pos = begin + len;
			if(pos < offset) {
				len = 0;
				begin = pos;
			}
			if (pos > offset + length) {
				goto done_spi_i;
			}
		}
 done_spi_i:	
	*start = buffer + (offset - begin);	/* Start of wanted data */
	len -= (offset - begin);			/* Start slop */
	if (len > length)
		len = length;
	return len;
}

static int
ipsec_spigrp_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{

	int len = 0;
	off_t pos = 0, begin = 0;
	int i;
	struct tdb *tdbp, *tdbp2;
	char sa[SATOA_BUF];
	size_t sa_len;

	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_spigrp_get_info: buffer=0x%p,"
		    " *start=0x%x, offset=%d, length=%d\n",
		    buffer, (u_int)*start, (int)offset, length);

	for (i = 0; i < TDB_HASHMOD; i++)
		for (tdbp = tdbh[i]; tdbp; tdbp = tdbp->tdb_hnext)
		{
			if(!tdbp->tdb_inext)
			{
				tdbp2 = tdbp;
				while(tdbp2) {
					sa_len = satoa(tdbp2->tdb_said, 0, sa, SATOA_BUF);
					len += sprintf(buffer + len, "%s ",
						       sa);
					tdbp2 = tdbp2->tdb_onext;
				}
				len += sprintf(buffer + len, "\n");
				pos = begin + len;
				if(pos < offset) {
					len = 0;
					begin = pos;
				}
				if (pos > offset + length) {
					goto done_spigrp_i;
				}
			}
		}
 done_spigrp_i:	
	*start = buffer + (offset - begin);	/* Start of wanted data */
	len -= (offset - begin);			/* Start slop */
	if (len > length)
		len = length;
	return len;
}

static int
ipsec_tncfg_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
	int len = 0;
	off_t pos = 0, begin = 0;
	int i;
	char name[9];
	struct device *dev, *privdev;
	struct ipsecpriv *priv;

	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_tncfg_get_info: buffer=0x%p,"
		    "*start=0x%x, offset=%d, length=%d\n",
		    buffer, (u_int)*start, (int)offset, length);

	for(i = 0; i < IPSEC_NUM_IF; i++)
	{
		sprintf(name, "ipsec%d", i);
		dev = dev_get(name);
		if(dev)
		{
			priv = (struct ipsecpriv *)(dev->priv);
			privdev = (struct device *)(priv->dev);
			len += sprintf(buffer + len, "%s -> %s\n",
				       dev->name, privdev ? privdev->name : "NULL");
			pos = begin + len;
			if(pos < offset) {
				len = 0;
				begin = pos;
			}
			else if (pos > offset + length)	{
				break;
			}
		}
	}
	*start = buffer + (offset - begin);	/* Start of wanted data */
	len -= (offset - begin);			/* Start slop */
	if (len > length)
		len = length;
	return len;
}

static int
ipsec_spi_get_new(char *buffer, char **start, off_t offset, int length, int dummy)
{

	int len = 0;
	off_t pos = 0, begin = 0;

	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_spi_get_new: buffer=0x%p,"
		    "*start=0x%x, offset=%d, length=%d\n",
		    buffer, (u_int)*start, (int)offset, length);

	len += sprintf(buffer + len, "0x%x\n", ipsec_spi_next++);

	pos = begin + len;
	if(pos < offset) {
		len = 0;
		begin = pos;
	}
	else if (pos > offset + length)	{
		goto done_spinew;
	}
	
 done_spinew:	
	*start = buffer + (offset - begin);	/* Start of wanted data */
	len -= (offset - begin);			/* Start slop */
	if (len > length)
		len = length;
	return len;

}

static int
ipsec_version_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{

	int len = 0;
	off_t begin = 0;

	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_version_get_info: buffer=0x%p,"
		    "*start=0x%x, offset=%d, length=%d\n",
		    buffer, (u_int)*start, (int)offset, length);

	len += sprintf(buffer + len, "FreeS/WAN version: %s\n", freeswan_version);
	KLIPS_PRINT(debug_tunnel & DB_TN_PROCFS,
		    "klips_debug:ipsec_init CVS version %s\n",
		    ipsec_init_c_version);

	*start = buffer + (offset - begin);	/* Start of wanted data */
	len -= (offset - begin);			/* Start slop */
	if (len > length)
		len = length;
	return len;

}

struct proc_dir_entry ipsec_eroute =
{
	PROC_NET_IPSEC_EROUTE,
	12, "ipsec_eroute",
	S_IFREG | S_IRUGO, 1, 0, 0, 0,
	&proc_net_inode_operations,
	ipsec_eroute_get_info,
	NULL, NULL, NULL, NULL, NULL
};

struct proc_dir_entry ipsec_spi =
{
	PROC_NET_IPSEC_SPI,
	9, "ipsec_spi",
	S_IFREG | S_IRUGO, 1, 0, 0, 0,
	&proc_net_inode_operations,
	ipsec_spi_get_info,
	NULL, NULL, NULL, NULL, NULL
};

struct proc_dir_entry ipsec_spigrp =
{
	PROC_NET_IPSEC_SPIGRP,
	12, "ipsec_spigrp",
	S_IFREG | S_IRUGO, 1, 0, 0, 0,
	&proc_net_inode_operations,
	ipsec_spigrp_get_info,
	NULL, NULL, NULL, NULL, NULL
};

struct proc_dir_entry ipsec_tncfg =
{
	PROC_NET_IPSEC_TNCFG,
	11, "ipsec_tncfg",
	S_IFREG | S_IRUGO, 1, 0, 0, 0,
	&proc_net_inode_operations,
	ipsec_tncfg_get_info,
	NULL, NULL, NULL, NULL, NULL
};

struct proc_dir_entry ipsec_spinew =
{
	PROC_NET_IPSEC_SPINEW,
	12, "ipsec_spinew",
	S_IFREG | S_IRUGO, 1, 0, 0, 0,
	&proc_net_inode_operations,
	ipsec_spi_get_new,
	NULL, NULL, NULL, NULL, NULL
};

struct proc_dir_entry ipsec_version =
{
	PROC_NET_IPSEC_VERSION,
	13, "ipsec_version",
	S_IFREG | S_IRUGO, 1, 0, 0, 0,
	&proc_net_inode_operations,
	ipsec_version_get_info,
	NULL, NULL, NULL, NULL, NULL
};

void
ipsec_init(void)
{
	(void)netlink_attach(NETLINK_IPSEC, ipsec_callback);

	ipsec_radijinit();
}	


void ipsec_cleanup(void)
{
	netlink_detach(NETLINK_IPSEC);

	ipsec_tdbcleanup();
	ipsec_radijcleanup();
}

#ifdef MODULE
int init_module(void)
{
	proc_net_register(&ipsec_eroute);
	proc_net_register(&ipsec_spi);
	proc_net_register(&ipsec_spigrp);
	proc_net_register(&ipsec_tncfg);
	proc_net_register(&ipsec_spinew);
	proc_net_register(&ipsec_version);

	printk("klips_debug:ipsec_init: ipsec module loading. freeswan version: %s\n",
	       freeswan_version);
	KLIPS_PRINT(1, "klips_debug:ipsec_init: ipsec_init version: %s\n",
		    ipsec_init_c_version);
	KLIPS_PRINT(1, "klips_debug:ipsec_init: ipsec_tunnel version: %s\n",
		    ipsec_tunnel_c_version);
	KLIPS_PRINT(1, "klips_debug:ipsec_init: ipsec_netlink version: %s\n",
		    ipsec_netlink_c_version);

	ipsec_init();

#ifdef CONFIG_IPSEC_ESP
	inet_add_protocol(&esp_protocol);
#endif /* CONFIG_IPSEC_ESP */

#ifdef CONFIG_IPSEC_AH
	inet_add_protocol(&ah_protocol);
#endif /* CONFIG_IPSEC_AH */

	ipsec_tunnel_init_module();
	return 0;
}

int cleanup_module(void)
{
	ipsec_tunnel_cleanup_module();

#ifdef CONFIG_IPSEC_AH
	if ( inet_del_protocol(&ah_protocol) < 0 )
		printk(KERN_INFO "klips_debug:ah close: can't remove protocol\n");
#endif /* CONFIG_IPSEC_AH */
#ifdef CONFIG_IPSEC_ESP
	if ( inet_del_protocol(&esp_protocol) < 0 )
		printk(KERN_INFO "klips_debug:esp close: can't remove protocol\n");
#endif /* CONFIG_IPSEC_ESP */

	if (proc_net_unregister(ipsec_version.low_ino) != 0)
		printk("klips_debug:ipsec_cleanup: cannot unregister /proc/net/ipsec-version\n");
	if (proc_net_unregister(ipsec_spinew.low_ino) != 0)
		printk("klips_debug:ipsec_cleanup: cannot unregister /proc/net/ipsec-spinew\n");
	if (proc_net_unregister(ipsec_eroute.low_ino) != 0)
		printk("klips_debug:ipsec_cleanup: cannot unregister /proc/net/ipsec-tncfg\n");
	if (proc_net_unregister(ipsec_spi.low_ino) != 0)
		printk("klips_debug:ipsec_cleanup: cannot unregister /proc/net/ipsec-spigrp\n");
	if (proc_net_unregister(ipsec_spigrp.low_ino) != 0)
		printk("klips_debug:ipsec_cleanup: cannot unregister /proc/net/ipsec-spi\n");
	if (proc_net_unregister(ipsec_tncfg.low_ino) != 0)
		printk("klips_debug:ipsec_cleanup: cannot unregister /proc/net/ipsec-eroute\n");

	ipsec_cleanup();

	KLIPS_PRINT(1, "klips_debug:ipsec_cleanup: ipsec module unloaded.\n");
	KLIPS_PRINT(debug_tunnel & DB_TN_INIT,
		    "klips_debug:ipsec_tunnel_cleanup_module: unloaded module ipsec\n");
	return 0;
}
#endif /* MODULE */

/*
 * $Log: ipsec_init.c,v $
 * Revision 1.17  1999/04/11 00:28:57  henry
 * GPL boilerplate
 *
 * Revision 1.16  1999/04/06 04:54:25  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.15  1999/02/24 20:15:07  rgb
 * Update output format.
 *
 * Revision 1.14  1999/02/17 16:49:39  rgb
 * Convert DEBUG_IPSEC to KLIPS_PRINT
 * Ditch NET_IPIP dependancy.
 *
 * Revision 1.13  1999/01/26 02:06:37  rgb
 * Remove ah/esp switching on include files.
 * Removed CONFIG_IPSEC_ALGO_SWITCH macro.
 * Removed dead code.
 * Remove references to INET_GET_PROTOCOL.
 *
 * Revision 1.12  1999/01/22 06:19:18  rgb
 * Cruft clean-out.
 * 64-bit clean-up.
 * Added algorithm switch code.
 *
 * Revision 1.11  1998/12/01 05:54:53  rgb
 * Cleanup and order debug version output.
 *
 * Revision 1.10  1998/11/30 13:22:54  rgb
 * Rationalised all the klips kernel file headers.  They are much shorter
 * now and won't conflict under RH5.2.
 *
 * Revision 1.9  1998/11/10 05:35:13  rgb
 * Print direction in/out flag from /proc/net/ipsec_spi.
 *
 * Revision 1.8  1998/10/27 13:48:10  rgb
 * Cleaned up /proc/net/ipsec_* filesystem for easy parsing by scripts.
 * Fixed less(1) truncated output bug.
 * Code clean-up.
 *
 * Revision 1.7  1998/10/22 06:43:16  rgb
 * Convert to use satoa for printk.
 *
 * Revision 1.6  1998/10/19 14:24:35  rgb
 * Added inclusion of freeswan.h.
 *
 * Revision 1.5  1998/10/09 04:43:35  rgb
 * Added 'klips_debug' prefix to all klips printk debug statements.
 *
 * Revision 1.4  1998/07/27 21:50:22  rgb
 * Not necessary to traverse mask tree for /proc/net/ipsec_eroute.
 *
 * Revision 1.3  1998/06/25 19:51:20  rgb
 * Clean up #endif comments.
 * Shift debugging comment control for procfs to debug_tunnel.
 * Make proc_dir_entries visible to rest of kernel for static link.
 * Replace hardwired fileperms with macros.
 * Use macros for procfs inode numbers.
 * Rearrange initialisations between ipsec_init and module_init as appropriate
 * for static loading.
 *
 * Revision 1.2  1998/06/23 02:55:43  rgb
 * Slightly quieted init-time messages.
 * Re-introduced inet_add_protocol after it mysteriously disappeared...
 * Check for and warn of absence of IPIP protocol on install of module.
 * Move tdbcleanup to ipsec_xform.c.
 *
 * Revision 1.10  1998/06/18 21:29:04  henry
 * move sources from klips/src to klips/net/ipsec, to keep stupid kernel
 * build scripts happier in presence of symbolic links
 *
 * Revision 1.9  1998/06/14 23:49:40  rgb
 * Clarify version reporting on module loading.
 *
 * Revision 1.8  1998/06/11 05:54:23  rgb
 * Added /proc/net/ipsec_version to report freeswan and transform versions.
 * Added /proc/net/ipsec_spinew to generate new and unique spi's..
 * Fixed /proc/net/ipsec_tncfg bug.
 *
 * Revision 1.7  1998/05/25 20:23:13  rgb
 * proc_register changed to dynamic registration to avoid arbitrary inode
 * numbers.
 *
 * Implement memory recovery from tdb and eroute tables.
 *
 * Revision 1.6  1998/05/21 13:08:58  rgb
 * Rewrote procinfo subroutines to avoid *bad things* when more that 3k of
 * information is available for printout.
 *
 * Revision 1.5  1998/05/18 21:29:48  rgb
 * Cleaned up /proc/net/ipsec_* output, including a title line, algorithm
 * names instead of numbers, standard format for numerical output base,
 * whitespace for legibility, and the names themselves for consistency.
 *
 * Added /proc/net/ipsec_spigrp and /proc/net/ipsec_tncfg.
 *
 * Revision 1.4  1998/04/30 15:42:24  rgb
 * Silencing attach for normal operations with #ifdef IPSEC_DEBUG.
 *
 * Revision 1.3  1998/04/21 21:28:58  rgb
 * Rearrange debug switches to change on the fly debug output from user
 * space.  Only kernel changes checked in at this time.  radij.c was also
 * changed to temporarily remove buggy debugging code in rj_delete causing
 * an OOPS and hence, netlink device open errors.
 *
 * Revision 1.2  1998/04/12 22:03:22  rgb
 * Updated ESP-3DES-HMAC-MD5-96,
 * 	ESP-DES-HMAC-MD5-96,
 * 	AH-HMAC-MD5-96,
 * 	AH-HMAC-SHA1-96 since Henry started freeswan cvs repository
 * from old standards (RFC182[5-9] to new (as of March 1998) drafts.
 *
 * Fixed eroute references in /proc/net/ipsec*.
 *
 * Started to patch module unloading memory leaks in ipsec_netlink and
 * radij tree unloading.
 *
 * Revision 1.1  1998/04/09 03:06:05  henry
 * sources moved up from linux/net/ipsec
 *
 * Revision 1.1.1.1  1998/04/08 05:35:02  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.4  1997/01/15 01:28:15  ji
 * No changes.
 *
 * Revision 0.3  1996/11/20 14:39:04  ji
 * Fixed problem with node names of /proc/net entries.
 * Other minor cleanups.
 * Rationalized debugging code.
 *
 * Revision 0.2  1996/11/02 00:18:33  ji
 * First limited release.
 *
 *
 */
