/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 2000 */
/* See the file NOTICE for conditions of use and distribution. */

#include "../exim.h"
#include "dnsdb.h"



/* Ancient systems (e.g. SunOS4) don't appear to have T_TXT defined in their
header files. */

#ifndef T_TXT
#define T_TXT 16
#endif

/* Table of recognized DNS record types and their integer values. */

static char *type_names[] = {
  "a",
#if HAVE_IPV6
  "aaaa",
#endif
  "cname",
  "mx",
  "ns",
  "ptr",
  "txt" };

static int type_values[] = {
  T_A,
#if HAVE_IPV6
  T_AAAA,
#endif
  T_CNAME,
  T_MX,
  T_NS,
  T_PTR,
  T_TXT };


/*************************************************
*              Open entry point                  *
*************************************************/

/* See local README for interface description. */

void *
dnsdb_open(char *filename, char **errmsg)
{
filename = filename;   /* Keep picky compilers happy */
errmsg = errmsg;       /* Ditto */
return (void *)(-1);   /* Any non-0 value */
}



/*************************************************
*           Find entry point for dnsdb           *
*************************************************/

/* See local README for interface description. */

int
dnsdb_find(void *handle, char *filename, char *keystring, int length,
  char **result, char **errmsg)
{
int rc;
int size = 0;
int ptr = 0;
int type = T_TXT;
char *yield = NULL;
char *equals = strchr(keystring, '=');
char buffer[256];

dns_record *rr;

handle = handle;           /* Keep picky compilers happy */
filename = filename;
length = length;

/* If the keystring contains an = this is preceded by a type name. */

if (equals != NULL)
  {
  int i;
  int len = equals - keystring;
  for (i = 0; i < sizeof(type_names)/sizeof(char *); i++)
    {
    if (len == (int)strlen(type_names[i]) &&
        strncmpic(keystring, type_names[i], len) == 0)
      {
      type = type_values[i];
      break;
      }
    }
  if (i >= sizeof(type_names)/sizeof(char *))
    {
    *errmsg = "unsupported DNS record type";
    return DEFER;
    }
  keystring += len + 1;
  }

/* If the type is PTR, we have to construct the relevant magic lookup
key. */

if (type == T_PTR)
  {
  char *p = keystring + (int)strlen(keystring);
  char *pp = buffer;

  /* Handle IPv4 address */

  #if HAVE_IPV6
  if (strchr(keystring, ':') == NULL)
  #endif
    {
    int i;
    for (i = 0; i < 4; i++)
      {
      char *ppp = p;
      while (ppp > keystring && ppp[-1] != '.') ppp--;
      strncpy(pp, ppp, p - ppp);
      pp += p - ppp;
      *pp++ = '.';
      p = ppp - 1;
      }
    strcpy(pp, "in-addr.arpa");
    }

  /* Handle IPv6 address; convert to binary so as to fill out any
  abbreviation in the textual form. */

  #if HAVE_IPV6
  else
    {
    int i;
    int v6[4];
    (void)host_aton(keystring, v6);
    for (i = 3; i >= 0; i--)
      {
      int j;
      for (j = 0; j < 32; j += 4)
        {
        sprintf(pp, "%x.", (v6[i] >> j) & 15);
        pp += 2;
        }
      }
    strcpy(pp, "ip6.int");
    }
  #endif

  /* Point at constructed key */

  keystring = buffer;
  }

DEBUG(1) debug_printf("dnsdb key: %s\n", keystring);

/* Initialize the resolver, in case this is the first time it is used
in this run. Then do the lookup and sort out the result. */

dns_init(FALSE, FALSE);
rc = dns_lookup(keystring, type, NULL);

if (rc == DNS_NOMATCH) return FAIL;
if (rc != DNS_SUCCEED) return DEFER;

for (rr = dns_next_rr(RESET_ANSWERS); rr != NULL;
     rr = dns_next_rr(RESET_NEXT))
  {
  int len = 0;        /* Stop picky compilers warning */
  char *s = NULL;

  if (rr->type != type) continue;

  switch(type)
    {
    case T_A:
    case T_AAAA:
    s = dns_address_from_rr(rr);
    len = (int)strlen(s);
    break;

    case T_TXT:
    len = (rr->data)[0];
    s = string_copyn((char *)(rr->data+1), len);
    break;

    case T_CNAME:
    case T_MX:
    case T_NS:
    case T_PTR:
      {
      uschar *p = (uschar *)(rr->data);
      int ssize = 264;
      s = store_get(ssize);
      if (type == T_MX)
        {
        int pref;
        GETSHORT(pref, p);      /* pointer is advanced */
        sprintf(s, "%d ", pref);
        len = (int)strlen(s);
        }
      len += dns_expand(p, s + len, ssize - len);
      store_reset(s + len + 1);
      }
    break;
    }

  if (yield == NULL)
    {
    yield = s;
    ptr = len;
    size = ptr + 1;
    }
  else
    {
    yield = string_cat(yield, &size, &ptr, "\n", 1);
    yield = string_cat(yield, &size, &ptr, s, len);
    }
  }

yield[ptr] = 0;
*result = yield;

return OK;
}

/* End of lookups/dnsdb.c */
