/*
 *      IPv6 address translation routines inet_ntop() and inet_pton()
 *      as defined in draft-ietf-ipngwg-bsd-api-05
 *
 *      Author: Peter Belding <pbelding@qualcomm.com>
 *
 *     This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <linux/in6.h>

static char inet_ntop_buffer[46];
static char hexc[] = "0123456789abcdef";
char *inet_ntop(int af, const void *ap, size_t len, char *cp)
{
	int value;
	if (ap == NULL) 
		return NULL;
	if (cp == NULL)
		cp = inet_ntop_buffer;
	if (af == AF_INET) {
		char *p = (char *)ap;
		if (len < 4)
			return NULL;
		sprintf(cp, "%d.%d.%d.%d",p[0], p[1], p[2], p[3]);
		return cp;
	} else if (af == AF_INET6) {
		struct in6_addr *addr = (struct in6_addr *)ap;
		char *tmp = cp;
                int start = -1, count = 0, i;

		if (len < 16)
			return NULL;

		/* weird special cases for IPv4 address */
		if ((addr->s6_addr32[0] == 0) && 
		    (addr->s6_addr32[1] == 0)) {
			if (addr->s6_addr32[2] == 0) {
				if (addr->s6_addr32[3] == 0) {
                                        strcpy(cp,"::");
				} else if (ntohl(addr->s6_addr32[3]) <=  0xff) {
					sprintf(cp,"::%x",addr->s6_addr[15]);
				} else
					sprintf(cp,"::%d.%d.%d.%d", addr->s6_addr[12],
						addr->s6_addr[13], addr->s6_addr[14],
						addr->s6_addr[15]);
				return cp;
			} else if (addr->s6_addr32[2] == htonl(0xffff)) {
				sprintf(cp,"::ffff:%d.%d.%d.%d",
					addr->s6_addr[12], addr->s6_addr[13],
					addr->s6_addr[14], addr->s6_addr[15]);
				return cp;
			}
		}
                /*
                 * Find the first string of consecutive __u16 zeros
                 */
                for (i = 0; i < 8; i++) {
                        if ((addr->s6_addr[2*i] == 0) &&
                            (addr->s6_addr[2*i+1] == 0)) {
                                if (start > -1) {
                                        count++;
                                } else {
                                        start = i;
                                        count = 1;
                                }
                        } else {
                                if (count > 1) {
                                        break;
                                } else {
                                        start = -1;
                                        count = 0;
                                }
                        }
                }
                if (start == 0) 
                        *tmp++ = ':';
                for (i = 0; i < 8; i++) {
                        if (i == start) {
                                *tmp++ = ':';
                                i += count-1;
                        } else {
                                int value = (addr->s6_addr[2*i] << 8) |
                                        addr->s6_addr[2*i+1];
				if (value >> 12)
					*tmp++ = hexc[(value>>12)&0xf];
				if (value >> 8)
					*tmp++ = hexc[(value>>8)&0xf];
				if (value >> 4)
					*tmp++ = hexc[(value>>4)&0xf];
				*tmp++ = hexc[value & 0xf];
                                if (i != 7)
                                        *tmp++ = ':';
                        }
                }
		*tmp = '\0';
		return cp;
	} else 
		return NULL;
}

int inet_pton(int af, const char *s, void *ap) 
{

	if (ap == NULL) 
		return -1;
	if (af == AF_INET) {
		unsigned long addr = inet_addr(s);
		*(unsigned long *)ap = addr;
		return 4;
	} else if (af == AF_INET6) {
		const char *tmp = s;
		int ccount = 0, pos = 0;
		int value = 0, lastcolon = 0;
		struct in6_addr addr;
                int v4[4];
                
		/* quick and dirty tests for IPv4 compatible addresses */
		memset(&addr,0,sizeof(struct in6_addr));
		if (sscanf(s, "::%d.%d.%d.%d",&v4[0],&v4[1],&v4[2],&v4[3]) == 4 ) {
                        for (pos=0; pos < 4;pos++) 
                                addr.s6_addr[pos+12] = v4[pos] & 0xff;
			memcpy(ap, &addr, sizeof(addr));
			return sizeof(addr);
		}
		if (sscanf(s, "::%x:%d.%d.%d.%d", &value, &v4[0], &v4[1],
                           &v4[2], &v4[3]) == 5) {
                        for (pos=0; pos < 4;pos++) 
                                addr.s6_addr[pos+12] = v4[pos] & 0xff;
			addr.s6_addr[10] = (value >> 8) & 0xff;
			addr.s6_addr[11] = value & 0xff;
			memcpy(ap, &addr, sizeof(addr));
			return sizeof(addr);
		}
		for (tmp = s; *tmp != '\0'; tmp++)
			if (*tmp == ':') 
				ccount++;
                tmp = s;
		value = 0;
		while((*tmp != '\0') && (pos < 16)) {
			if (*tmp == ':') {
				if (lastcolon) {
					pos += 16 - 2*ccount;
					lastcolon = 0;
				} else {
					addr.s6_addr[pos++] = (value >> 8) & 0xff;
					addr.s6_addr[pos++] = value & 0xff;
					value = 0;
					lastcolon = 1;
				}
			} else {
				lastcolon = 0;
				if ((*tmp >= '0') && (*tmp <= '9')) {
					value = (value << 4) | (*tmp - '0');
				} else if ((*tmp >= 'A') && (*tmp <= 'F')) {
					value = (value << 4) | (*tmp - 'A' + 10);
				} else if ((*tmp >= 'a') && (*tmp <= 'f')) {
					value = (value << 4) | (*tmp - 'a' + 10);
				} else {
					break;
				}
			}
			tmp++;
		}
		addr.s6_addr[pos++] = (value >> 8) & 0xff;
		addr.s6_addr[pos++] = value & 0xff;

		if (pos < 16)
			return -1;
		memcpy(ap, &addr, sizeof(addr));
		return sizeof(struct in6_addr);
	}
	return -1;
}
