/*
 * SPCA501 chip based cameras initialization data
 *
 * V4L2 by Jean-Francois Moine <http://moinejf.free.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#define MODULE_NAME "spca501"

#include "gspca.h"

MODULE_AUTHOR("Michel Xhaard <mxhaard@users.sourceforge.net>");
MODULE_DESCRIPTION("GSPCA/SPCA501 USB Camera Driver");
MODULE_LICENSE("GPL");

/* specific webcam descriptor */
struct sd {
	struct gspca_dev gspca_dev;	/* !! must be the first item */

	unsigned short contrast;
	__u8 brightness;
	__u8 colors;
	__u8 blue_balance;
	__u8 red_balance;

	char subtype;
#define Arowana300KCMOSCamera 0
#define IntelCreateAndShare 1
#define KodakDVC325 2
#define MystFromOriUnknownCamera 3
#define SmileIntlCamera 4
#define ThreeComHomeConnectLite 5
#define ViewQuestM318B 6
};

/* V4L2 controls supported by the driver */
static int sd_setbrightness(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getbrightness(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setcontrast(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getcontrast(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setcolors(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getcolors(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setblue_balance(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getblue_balance(struct gspca_dev *gspca_dev, __s32 *val);
static int sd_setred_balance(struct gspca_dev *gspca_dev, __s32 val);
static int sd_getred_balance(struct gspca_dev *gspca_dev, __s32 *val);

static const struct ctrl sd_ctrls[] = {
#define MY_BRIGHTNESS 0
	{
	    {
		.id      = V4L2_CID_BRIGHTNESS,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Brightness",
		.minimum = 0,
		.maximum = 127,
		.step    = 1,
		.default_value = 0,
	    },
	    .set = sd_setbrightness,
	    .get = sd_getbrightness,
	},
#define MY_CONTRAST 1
	{
	    {
		.id      = V4L2_CID_CONTRAST,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Contrast",
		.minimum = 0,
		.maximum = 64725,
		.step    = 1,
		.default_value = 64725,
	    },
	    .set = sd_setcontrast,
	    .get = sd_getcontrast,
	},
#define MY_COLOR 2
	{
	    {
		.id      = V4L2_CID_SATURATION,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Color",
		.minimum = 0,
		.maximum = 63,
		.step    = 1,
		.default_value = 20,
	    },
	    .set = sd_setcolors,
	    .get = sd_getcolors,
	},
#define MY_BLUE_BALANCE 3
	{
	    {
		.id      = V4L2_CID_BLUE_BALANCE,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Blue Balance",
		.minimum = 0,
		.maximum = 127,
		.step    = 1,
		.default_value = 0,
	    },
	    .set = sd_setblue_balance,
	    .get = sd_getblue_balance,
	},
#define MY_RED_BALANCE 4
	{
	    {
		.id      = V4L2_CID_RED_BALANCE,
		.type    = V4L2_CTRL_TYPE_INTEGER,
		.name    = "Red Balance",
		.minimum = 0,
		.maximum = 127,
		.step    = 1,
		.default_value = 0,
	    },
	    .set = sd_setred_balance,
	    .get = sd_getred_balance,
	},
};

static const struct v4l2_pix_format vga_mode[] = {
	{160, 120, V4L2_PIX_FMT_SPCA501, V4L2_FIELD_NONE,
		.bytesperline = 160,
		.sizeimage = 160 * 120 * 3 / 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 2},
	{320, 240, V4L2_PIX_FMT_SPCA501, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 3 / 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 1},
	{640, 480, V4L2_PIX_FMT_SPCA501, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 3 / 2,
		.colorspace = V4L2_COLORSPACE_SRGB,
		.priv = 0},
};

#define SPCA50X_REG_USB 0x2	/* spca505 501 */
/*
 * Data to initialize a SPCA501. From a capture file provided by Bill Roehl
 * With SPCA501 chip description
 */
#define CCDSP_SET		/* set CCDSP parameters */
#define TG_SET			/* set time generator set */
#undef DSPWIN_SET		/* set DSP windows parameters */
#undef ALTER_GAMA	/* Set alternate set to YUV transform coeffs. */
#define SPCA501_SNAPBIT 0x80
#define SPCA501_SNAPCTRL 0x10
/* Frame packet header offsets for the spca501 */
#define SPCA501_OFFSET_GPIO   1
#define SPCA501_OFFSET_TYPE   2
#define SPCA501_OFFSET_TURN3A 3
#define SPCA501_OFFSET_FRAMSEQ 4
#define SPCA501_OFFSET_COMPRESS 5
#define SPCA501_OFFSET_QUANT 6
#define SPCA501_OFFSET_QUANT2 7
#define SPCA501_OFFSET_DATA 8

#define SPCA501_PROP_COMP_ENABLE(d) ((d) & 1)
#define SPCA501_PROP_SNAP(d) ((d) & 0x40)
#define SPCA501_PROP_SNAP_CTRL(d) ((d) & 0x10)
#define SPCA501_PROP_COMP_THRESH(d) (((d) & 0x0e) >> 1)
#define SPCA501_PROP_COMP_QUANT(d) (((d) & 0x70) >> 4)

/* SPCA501 CCDSP control */
#define SPCA501_REG_CCDSP 0x01
/* SPCA501 control/status registers */
#define SPCA501_REG_CTLRL 0x02

/* registers for color correction and YUV transformation */
#define SPCA501_A11 0x08
#define SPCA501_A12 0x09
#define SPCA501_A13 0x0A
#define SPCA501_A21 0x0B
#define SPCA501_A22 0x0C
#define SPCA501_A23 0x0D
#define SPCA501_A31 0x0E
#define SPCA501_A32 0x0F
#define SPCA501_A33 0x10

/* Data for video camera initialization before capturing */
static const __u16 spca501_open_data[][3] = {
	/* bmRequest,value,index */

	{0x2, 0x50, 0x00},	/* C/S enable soft reset */
	{0x2, 0x40, 0x00},	/* C/S disable soft reset */
	{0x2, 0x02, 0x05},	/* C/S general purpose I/O data */
	{0x2, 0x03, 0x05},	/* C/S general purpose I/O data */

#ifdef CCDSP_SET
	{0x1, 0x38, 0x01},	/* CCDSP options */
	{0x1, 0x05, 0x02}, /* CCDSP Optical black level for user settings */
	{0x1, 0xC0, 0x03},	/* CCDSP Optical black settings */

	{0x1, 0x67, 0x07},
	{0x1, 0x63, 0x3f},	/* CCDSP CCD gamma enable */
	{0x1, 0x03, 0x56},	/* Add gamma correction */

	{0x1, 0xFF, 0x15},	/* CCDSP High luminance for white balance */
	{0x1, 0x01, 0x16},	/* CCDSP Low luminance for white balance */

/* Color correction and RGB-to-YUV transformation coefficients changing */
#ifdef ALTER_GAMA
	{0x0, 0x00, 0x08},	/* A11 */
	{0x0, 0x00, 0x09},	/* A12 */
	{0x0, 0x90, 0x0A},	/* A13 */
	{0x0, 0x12, 0x0B},	/* A21 */
	{0x0, 0x00, 0x0C},	/* A22 */
	{0x0, 0x00, 0x0D},	/* A23 */
	{0x0, 0x00, 0x0E},	/* A31 */
	{0x0, 0x02, 0x0F},	/* A32 */
	{0x0, 0x00, 0x10},	/* A33 */
#else
	{0x1, 0x2a, 0x08},	/* A11 0x31 */
	{0x1, 0xf8, 0x09},	/* A12 f8 */
	{0x1, 0xf8, 0x0A},	/* A13 f8 */
	{0x1, 0xf8, 0x0B},	/* A21 f8 */
	{0x1, 0x14, 0x0C},	/* A22 0x14 */
	{0x1, 0xf8, 0x0D},	/* A23 f8 */
	{0x1, 0xf8, 0x0E},	/* A31 f8 */
	{0x1, 0xf8, 0x0F},	/* A32 f8 */
	{0x1, 0x20, 0x10},	/* A33 0x20 */
#endif
	{0x1, 0x00, 0x11},	/* R offset */
	{0x1, 0x00, 0x12},	/* G offset */
	{0x1, 0x00, 0x13},	/* B offset */
	{0x1, 0x00, 0x14},	/* GB offset */

#endif

#ifdef TG_SET
	/* Time generator manipulations */
	{0x0, 0xfc, 0x0},	/* Set up high bits of shutter speed */
	{0x0, 0x01, 0x1},	/* Set up low bits of shutter speed */

	{0x0, 0xe4, 0x04},	/* DCLK*2 clock phase adjustment */
	{0x0, 0x08, 0x05},	/* ADCK phase adjustment, inv. ext. VB */
	{0x0, 0x03, 0x06},	/* FR phase adjustment */
	{0x0, 0x01, 0x07},	/* FCDS phase adjustment */
	{0x0, 0x39, 0x08},	/* FS phase adjustment */
	{0x0, 0x88, 0x0a},	/* FH1 phase and delay adjustment */
	{0x0, 0x03, 0x0f},	/* pixel identification */
	{0x0, 0x00, 0x11},	/* clock source selection (default) */

	/*VERY strange manipulations with
	 * select DMCLP or OBPX to be ADCLP output (0x0C)
	 * OPB always toggle or not (0x0D) but they allow
	 * us to set up brightness
	 */
	{0x0, 0x01, 0x0c},
	{0x0, 0xe0, 0x0d},
	/* Done */
#endif

#ifdef DSPWIN_SET
	{0x1, 0xa0, 0x01},	/* Setting image processing parameters */
	{0x1, 0x1c, 0x17},	/* Changing Windows positions X1 */
	{0x1, 0xe2, 0x19},	/* X2 */
	{0x1, 0x1c, 0x1b},	/* X3 */
	{0x1, 0xe2, 0x1d},	/* X4 */
	{0x1, 0x5f, 0x1f},	/* X5 */
	{0x1, 0x32, 0x20},	/* Y5 */
	{0x1, 0x01, 0x10},	/* Changing A33 */
#endif

	{0x2, 0x204a, 0x07},/* Setting video compression & resolution 160x120 */
	{0x2, 0x94, 0x06},	/* Setting video no compression */
	{}
};

/*
   The SPCAxxx docs from Sunplus document these values
   in tables, one table per register number.  In the data
   below, dmRequest is the register number, index is the Addr,
   and value is a combination of Bit values.
   Bit  Value (hex)
   0    01
   1    02
   2    04
   3    08
   4    10
   5    20
   6    40
   7    80
 */

/* Data for chip initialization (set default values) */
static const __u16 spca501_init_data[][3] = {
	/* Set all the values to powerup defaults */
	/* bmRequest,value,index */
	{0x0, 0xAA, 0x00},
	{0x0, 0x02, 0x01},
	{0x0, 0x01, 0x02},
	{0x0, 0x02, 0x03},
	{0x0, 0xCE, 0x04},
	{0x0, 0x00, 0x05},
	{0x0, 0x00, 0x06},
	{0x0, 0x00, 0x07},
	{0x0, 0x00, 0x08},
	{0x0, 0x00, 0x09},
	{0x0, 0x90, 0x0A},
	{0x0, 0x12, 0x0B},
	{0x0, 0x00, 0x0C},
	{0x0, 0x00, 0x0D},
	{0x0, 0x00, 0x0E},
	{0x0, 0x02, 0x0F},
	{0x0, 0x00, 0x10},
	{0x0, 0x00, 0x11},
	{0x0, 0x00, 0x12},
	{0x0, 0x00, 0x13},
	{0x0, 0x00, 0x14},
	{0x0, 0x00, 0x15},
	{0x0, 0x00, 0x16},
	{0x0, 0x00, 0x17},
	{0x0, 0x00, 0x18},
	{0x0, 0x00, 0x19},
	{0x0, 0x00, 0x1A},
	{0x0, 0x00, 0x1B},
	{0x0, 0x00, 0x1C},
	{0x0, 0x00, 0x1D},
	{0x0, 0x00, 0x1E},
	{0x0, 0x00, 0x1F},
	{0x0, 0x00, 0x20},
	{0x0, 0x00, 0x21},
	{0x0, 0x00, 0x22},
	{0x0, 0x00, 0x23},
	{0x0, 0x00, 0x24},
	{0x0, 0x00, 0x25},
	{0x0, 0x00, 0x26},
	{0x0, 0x00, 0x27},
	{0x0, 0x00, 0x28},
	{0x0, 0x00, 0x29},
	{0x0, 0x00, 0x2A},
	{0x0, 0x00, 0x2B},
	{0x0, 0x00, 0x2C},
	{0x0, 0x00, 0x2D},
	{0x0, 0x00, 0x2E},
	{0x0, 0x00, 0x2F},
	{0x0, 0x00, 0x30},
	{0x0, 0x00, 0x31},
	{0x0, 0x00, 0x32},
	{0x0, 0x00, 0x33},
	{0x0, 0x00, 0x34},
	{0x0, 0x00, 0x35},
	{0x0, 0x00, 0x36},
	{0x0, 0x00, 0x37},
	{0x0, 0x00, 0x38},
	{0x0, 0x00, 0x39},
	{0x0, 0x00, 0x3A},
	{0x0, 0x00, 0x3B},
	{0x0, 0x00, 0x3C},
	{0x0, 0x00, 0x3D},
	{0x0, 0x00, 0x3E},
	{0x0, 0x00, 0x3F},
	{0x0, 0x00, 0x40},
	{0x0, 0x00, 0x41},
	{0x0, 0x00, 0x42},
	{0x0, 0x00, 0x43},
	{0x0, 0x00, 0x44},
	{0x0, 0x00, 0x45},
	{0x0, 0x00, 0x46},
	{0x0, 0x00, 0x47},
	{0x0, 0x00, 0x48},
	{0x0, 0x00, 0x49},
	{0x0, 0x00, 0x4A},
	{0x0, 0x00, 0x4B},
	{0x0, 0x00, 0x4C},
	{0x0, 0x00, 0x4D},
	{0x0, 0x00, 0x4E},
	{0x0, 0x00, 0x4F},
	{0x0, 0x00, 0x50},
	{0x0, 0x00, 0x51},
	{0x0, 0x00, 0x52},
	{0x0, 0x00, 0x53},
	{0x0, 0x00, 0x54},
	{0x0, 0x00, 0x55},
	{0x0, 0x00, 0x56},
	{0x0, 0x00, 0x57},
	{0x0, 0x00, 0x58},
	{0x0, 0x00, 0x59},
	{0x0, 0x00, 0x5A},
	{0x0, 0x00, 0x5B},
	{0x0, 0x00, 0x5C},
	{0x0, 0x00, 0x5D},
	{0x0, 0x00, 0x5E},
	{0x0, 0x00, 0x5F},
	{0x0, 0x00, 0x60},
	{0x0, 0x00, 0x61},
	{0x0, 0x00, 0x62},
	{0x0, 0x00, 0x63},
	{0x0, 0x00, 0x64},
	{0x0, 0x00, 0x65},
	{0x0, 0x00, 0x66},
	{0x0, 0x00, 0x67},
	{0x0, 0x00, 0x68},
	{0x0, 0x00, 0x69},
	{0x0, 0x00, 0x6A},
	{0x0, 0x00, 0x6B},
	{0x0, 0x00, 0x6C},
	{0x0, 0x00, 0x6D},
	{0x0, 0x00, 0x6E},
	{0x0, 0x00, 0x6F},
	{0x0, 0x00, 0x70},
	{0x0, 0x00, 0x71},
	{0x0, 0x00, 0x72},
	{0x0, 0x00, 0x73},
	{0x0, 0x00, 0x74},
	{0x0, 0x00, 0x75},
	{0x0, 0x00, 0x76},
	{0x0, 0x00, 0x77},
	{0x0, 0x00, 0x78},
	{0x0, 0x00, 0x79},
	{0x0, 0x00, 0x7A},
	{0x0, 0x00, 0x7B},
	{0x0, 0x00, 0x7C},
	{0x0, 0x00, 0x7D},
	{0x0, 0x00, 0x7E},
	{0x0, 0x00, 0x7F},
	{0x0, 0x00, 0x80},
	{0x0, 0x00, 0x81},
	{0x0, 0x00, 0x82},
	{0x0, 0x00, 0x83},
	{0x0, 0x00, 0x84},
	{0x0, 0x00, 0x85},
	{0x0, 0x00, 0x86},
	{0x0, 0x00, 0x87},
	{0x0, 0x00, 0x88},
	{0x0, 0x00, 0x89},
	{0x0, 0x00, 0x8A},
	{0x0, 0x00, 0x8B},
	{0x0, 0x00, 0x8C},
	{0x0, 0x00, 0x8D},
	{0x0, 0x00, 0x8E},
	{0x0, 0x00, 0x8F},
	{0x0, 0x00, 0x90},
	{0x0, 0x00, 0x91},
	{0x0, 0x00, 0x92},
	{0x0, 0x00, 0x93},
	{0x0, 0x00, 0x94},
	{0x0, 0x00, 0x95},
	{0x0, 0x00, 0x96},
	{0x0, 0x00, 0x97},
	{0x0, 0x00, 0x98},
	{0x0, 0x00, 0x99},
	{0x0, 0x00, 0x9A},
	{0x0, 0x00, 0x9B},
	{0x0, 0x00, 0x9C},
	{0x0, 0x00, 0x9D},
	{0x0, 0x00, 0x9E},
	{0x0, 0x00, 0x9F},
	{0x0, 0x00, 0xA0},
	{0x0, 0x00, 0xA1},
	{0x0, 0x00, 0xA2},
	{0x0, 0x00, 0xA3},
	{0x0, 0x00, 0xA4},
	{0x0, 0x00, 0xA5},
	{0x0, 0x00, 0xA6},
	{0x0, 0x00, 0xA7},
	{0x0, 0x00, 0xA8},
	{0x0, 0x00, 0xA9},
	{0x0, 0x00, 0xAA},
	{0x0, 0x00, 0xAB},
	{0x0, 0x00, 0xAC},
	{0x0, 0x00, 0xAD},
	{0x0, 0x00, 0xAE},
	{0x0, 0x00, 0xAF},
	{0x0, 0x00, 0xB0},
	{0x0, 0x00, 0xB1},
	{0x0, 0x00, 0xB2},
	{0x0, 0x00, 0xB3},
	{0x0, 0x00, 0xB4},
	{0x0, 0x00, 0xB5},
	{0x0, 0x00, 0xB6},
	{0x0, 0x00, 0xB7},
	{0x0, 0x00, 0xB8},
	{0x0, 0x00, 0xB9},
	{0x0, 0x00, 0xBA},
	{0x0, 0x00, 0xBB},
	{0x0, 0x00, 0xBC},
	{0x0, 0x00, 0xBD},
	{0x0, 0x00, 0xBE},
	{0x0, 0x00, 0xBF},
	{0x0, 0x00, 0xC0},
	{0x0, 0x00, 0xC1},
	{0x0, 0x00, 0xC2},
	{0x0, 0x00, 0xC3},
	{0x0, 0x00, 0xC4},
	{0x0, 0x00, 0xC5},
	{0x0, 0x00, 0xC6},
	{0x0, 0x00, 0xC7},
	{0x0, 0x00, 0xC8},
	{0x0, 0x00, 0xC9},
	{0x0, 0x00, 0xCA},
	{0x0, 0x00, 0xCB},
	{0x0, 0x00, 0xCC},
	{0x1, 0xF4, 0x00},
	{0x1, 0x38, 0x01},
	{0x1, 0x40, 0x02},
	{0x1, 0x0A, 0x03},
	{0x1, 0x40, 0x04},
	{0x1, 0x40, 0x05},
	{0x1, 0x40, 0x06},
	{0x1, 0x67, 0x07},
	{0x1, 0x31, 0x08},
	{0x1, 0x00, 0x09},
	{0x1, 0x00, 0x0A},
	{0x1, 0x00, 0x0B},
	{0x1, 0x14, 0x0C},
	{0x1, 0x00, 0x0D},
	{0x1, 0x00, 0x0E},
	{0x1, 0x00, 0x0F},
	{0x1, 0x1E, 0x10},
	{0x1, 0x00, 0x11},
	{0x1, 0x00, 0x12},
	{0x1, 0x00, 0x13},
	{0x1, 0x00, 0x14},
	{0x1, 0xFF, 0x15},
	{0x1, 0x01, 0x16},
	{0x1, 0x32, 0x17},
	{0x1, 0x23, 0x18},
	{0x1, 0xCE, 0x19},
	{0x1, 0x23, 0x1A},
	{0x1, 0x32, 0x1B},
	{0x1, 0x8D, 0x1C},
	{0x1, 0xCE, 0x1D},
	{0x1, 0x8D, 0x1E},
	{0x1, 0x00, 0x1F},
	{0x1, 0x00, 0x20},
	{0x1, 0xFF, 0x3E},
	{0x1, 0x02, 0x3F},
	{0x1, 0x00, 0x40},
	{0x1, 0x00, 0x41},
	{0x1, 0x00, 0x42},
	{0x1, 0x00, 0x43},
	{0x1, 0x00, 0x44},
	{0x1, 0x00, 0x45},
	{0x1, 0x00, 0x46},
	{0x1, 0x00, 0x47},
	{0x1, 0x00, 0x48},
	{0x1, 0x00, 0x49},
	{0x1, 0x00, 0x4A},
	{0x1, 0x00, 0x4B},
	{0x1, 0x00, 0x4C},
	{0x1, 0x00, 0x4D},
	{0x1, 0x00, 0x4E},
	{0x1, 0x00, 0x4F},
	{0x1, 0x00, 0x50},
	{0x1, 0x00, 0x51},
	{0x1, 0x00, 0x52},
	{0x1, 0x00, 0x53},
	{0x1, 0x00, 0x54},
	{0x1, 0x00, 0x55},
	{0x1, 0x00, 0x56},
	{0x1, 0x00, 0x57},
	{0x1, 0x00, 0x58},
	{0x1, 0x00, 0x59},
	{0x1, 0x00, 0x5A},
	{0x2, 0x03, 0x00},
	{0x2, 0x00, 0x01},
	{0x2, 0x00, 0x05},
	{0x2, 0x00, 0x06},
	{0x2, 0x00, 0x07},
	{0x2, 0x00, 0x10},
	{0x2, 0x00, 0x11},
	/* Strange - looks like the 501 driver doesn't do anything
	 * at insert time except read the EEPROM
	 */
	{}
};

/* Data for video camera init before capture.
 * Capture and decoding by Colin Peart.
 * This is is for the 3com HomeConnect Lite which is spca501a based.
 */
static const __u16 spca501_3com_open_data[][3] = {
	/* bmRequest,value,index */
	{0x2, 0x0050, 0x0000},	/* C/S Enable TG soft reset, timing mode=010 */
	{0x2, 0x0043, 0x0000},	/* C/S Disable TG soft reset, timing mode=010 */
	{0x2, 0x0002, 0x0005},	/* C/S GPIO */
	{0x2, 0x0003, 0x0005},	/* C/S GPIO */

#ifdef CCDSP_SET
	{0x1, 0x0020, 0x0001},	/* CCDSP Options */

	{0x1, 0x0020, 0x0002},	/* CCDSP Black Level */
	{0x1, 0x006e, 0x0007},	/* CCDSP Gamma options */
	{0x1, 0x0090, 0x0015},	/* CCDSP Luminance Low */
	{0x1, 0x00ff, 0x0016},	/* CCDSP Luminance High */
	{0x1, 0x0003, 0x003F},	/* CCDSP Gamma correction toggle */

#ifdef ALTER_GAMMA
	{0x1, 0x0010, 0x0008},	/* CCDSP YUV A11 */
	{0x1, 0x0000, 0x0009},	/* CCDSP YUV A12 */
	{0x1, 0x0000, 0x000a},	/* CCDSP YUV A13 */
	{0x1, 0x0000, 0x000b},	/* CCDSP YUV A21 */
	{0x1, 0x0010, 0x000c},	/* CCDSP YUV A22 */
	{0x1, 0x0000, 0x000d},	/* CCDSP YUV A23 */
	{0x1, 0x0000, 0x000e},	/* CCDSP YUV A31 */
	{0x1, 0x0000, 0x000f},	/* CCDSP YUV A32 */
	{0x1, 0x0010, 0x0010},	/* CCDSP YUV A33 */
	{0x1, 0x0000, 0x0011},	/* CCDSP R Offset */
	{0x1, 0x0000, 0x0012},	/* CCDSP G Offset */
	{0x1, 0x0001, 0x0013},	/* CCDSP B Offset */
	{0x1, 0x0001, 0x0014},	/* CCDSP BG Offset */
	{0x1, 0x003f, 0x00C1},	/* CCDSP Gamma Correction Enable */
#endif
#endif

#ifdef TG_SET
	{0x0, 0x00fc, 0x0000},	/* TG Shutter Speed High Bits */
	{0x0, 0x0000, 0x0001},	/* TG Shutter Speed Low Bits */
	{0x0, 0x00e4, 0x0004},	/* TG DCLK*2 Adjust */
	{0x0, 0x0008, 0x0005},	/* TG ADCK Adjust */
	{0x0, 0x0003, 0x0006},	/* TG FR Phase Adjust */
	{0x0, 0x0001, 0x0007},	/* TG FCDS Phase Adjust */
	{0x0, 0x0039, 0x0008},	/* TG FS Phase Adjust */
	{0x0, 0x0088, 0x000a},	/* TG MH1 */
	{0x0, 0x0003, 0x000f},	/* TG Pixel ID */

	/* Like below, unexplained toglleing */
	{0x0, 0x0080, 0x000c},
	{0x0, 0x0000, 0x000d},
	{0x0, 0x0080, 0x000c},
	{0x0, 0x0004, 0x000d},
	{0x0, 0x0000, 0x000c},
	{0x0, 0x0000, 0x000d},
	{0x0, 0x0040, 0x000c},
	{0x0, 0x0017, 0x000d},
	{0x0, 0x00c0, 0x000c},
	{0x0, 0x0000, 0x000d},
	{0x0, 0x0080, 0x000c},
	{0x0, 0x0006, 0x000d},
	{0x0, 0x0080, 0x000c},
	{0x0, 0x0004, 0x000d},
	{0x0, 0x0002, 0x0003},
#endif

#ifdef DSPWIN_SET
	{0x1, 0x001c, 0x0017},	/* CCDSP W1 Start X */
	{0x1, 0x00e2, 0x0019},	/* CCDSP W2 Start X */
	{0x1, 0x001c, 0x001b},	/* CCDSP W3 Start X */
	{0x1, 0x00e2, 0x001d},	/* CCDSP W4 Start X */
	{0x1, 0x00aa, 0x001f},	/* CCDSP W5 Start X */
	{0x1, 0x0070, 0x0020},	/* CCDSP W5 Start Y */
#endif
	{0x0, 0x0001, 0x0010},	/* TG Start Clock */

/*	{0x2, 0x006a, 0x0001},	 * C/S Enable ISOSYNCH Packet Engine */
	{0x2, 0x0068, 0x0001},	/* C/S Diable ISOSYNCH Packet Engine */
	{0x2, 0x0000, 0x0005},
	{0x2, 0x0043, 0x0000},	/* C/S Set Timing Mode, Disable TG soft reset */
	{0x2, 0x0043, 0x0000},	/* C/S Set Timing Mode, Disable TG soft reset */
	{0x2, 0x0002, 0x0005},	/* C/S GPIO */
	{0x2, 0x0003, 0x0005},	/* C/S GPIO */

	{0x2, 0x006a, 0x0001},	/* C/S Enable ISOSYNCH Packet Engine */
	{}
};

/*
 * Data used to initialize a SPCA501C with HV7131B sensor.
 * From a capture file taken with USBSnoop v 1.5
 * I have a "SPCA501C pc camera chipset" manual by sunplus, but some
 * of the value meanings are obscure or simply "reserved".
 * to do list:
 * 1) Understand what every value means
 * 2) Understand why some values seem to appear more than once
 * 3) Write a small comment for each line of the following arrays.
 */
static const __u16 spca501c_arowana_open_data[][3] = {
	/* bmRequest,value,index */
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x01, 0x0006, 0x0011},
	{0x01, 0x00ff, 0x0012},
	{0x01, 0x0014, 0x0013},
	{0x01, 0x0000, 0x0014},
	{0x01, 0x0042, 0x0051},
	{0x01, 0x0040, 0x0052},
	{0x01, 0x0051, 0x0053},
	{0x01, 0x0040, 0x0054},
	{0x01, 0x0000, 0x0055},
	{0x00, 0x0025, 0x0000},
	{0x00, 0x0026, 0x0000},
	{0x00, 0x0001, 0x0000},
	{0x00, 0x0027, 0x0000},
	{0x00, 0x008a, 0x0000},
	{}
};

static const __u16 spca501c_arowana_init_data[][3] = {
	/* bmRequest,value,index */
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x01, 0x0006, 0x0011},
	{0x01, 0x00ff, 0x0012},
	{0x01, 0x0014, 0x0013},
	{0x01, 0x0000, 0x0014},
	{0x01, 0x0042, 0x0051},
	{0x01, 0x0040, 0x0052},
	{0x01, 0x0051, 0x0053},
	{0x01, 0x0040, 0x0054},
	{0x01, 0x0000, 0x0055},
	{0x00, 0x0025, 0x0000},
	{0x00, 0x0026, 0x0000},
	{0x00, 0x0001, 0x0000},
	{0x00, 0x0027, 0x0000},
	{0x00, 0x008a, 0x0000},
	{0x02, 0x0000, 0x0005},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x000c, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0000, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},
	{0x00, 0x0000, 0x0024},
	{0x00, 0x00d5, 0x0025},
	{0x00, 0x0000, 0x0026},
	{0x00, 0x000b, 0x0027},
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
	{0xff, 0x0000, 0x00d0},
	{0xff, 0x00d8, 0x00d1},
	{0xff, 0x0000, 0x00d4},
	{0xff, 0x0000, 0x00d5},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0x00fd, 0x000a},
	{0x01, 0x0038, 0x000b},
	{0x01, 0x00d1, 0x000c},
	{0x01, 0x00f7, 0x000d},
	{0x01, 0x00ed, 0x000e},
	{0x01, 0x00d8, 0x000f},
	{0x01, 0x0038, 0x0010},
	{0x01, 0x00ff, 0x0015},
	{0x01, 0x0001, 0x0016},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},
	{0x01, 0x00ff, 0x003e},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0060, 0x0057},
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x100a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x000c, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0000, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},
	{0x00, 0x0000, 0x0024},
	{0x00, 0x00d5, 0x0025},
	{0x00, 0x0000, 0x0026},
	{0x00, 0x000b, 0x0027},
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
	{0xff, 0x0000, 0x00d0},
	{0xff, 0x00d8, 0x00d1},
	{0xff, 0x0000, 0x00d4},
	{0xff, 0x0000, 0x00d5},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0x00fd, 0x000a},
	{0x01, 0x0038, 0x000b},
	{0x01, 0x00d1, 0x000c},
	{0x01, 0x00f7, 0x000d},
	{0x01, 0x00ed, 0x000e},
	{0x01, 0x00d8, 0x000f},
	{0x01, 0x0038, 0x0010},
	{0x01, 0x00ff, 0x0015},
	{0x01, 0x0001, 0x0016},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},
	{0x01, 0x00ff, 0x003e},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0060, 0x0057},
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x100a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0xfffd, 0x000a},
	{0x01, 0x0023, 0x000b},
	{0x01, 0xffea, 0x000c},
	{0x01, 0xfff4, 0x000d},
	{0x01, 0xfffc, 0x000e},
	{0x01, 0xffe3, 0x000f},
	{0x01, 0x001f, 0x0010},
	{0x01, 0x00a8, 0x0001},
	{0x01, 0x0067, 0x0007},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x00c8, 0x0015},
	{0x01, 0x0032, 0x0016},
	{0x01, 0x0000, 0x0011},
	{0x01, 0x0000, 0x0012},
	{0x01, 0x0000, 0x0013},
	{0x01, 0x000a, 0x0003},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xc000, 0x0001},
	{0x02, 0x0000, 0x0005},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x000c, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0000, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},
	{0x00, 0x0000, 0x0024},
	{0x00, 0x00d5, 0x0025},
	{0x00, 0x0000, 0x0026},
	{0x00, 0x000b, 0x0027},
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
	{0xff, 0x0000, 0x00d0},
	{0xff, 0x00d8, 0x00d1},
	{0xff, 0x0000, 0x00d4},
	{0xff, 0x0000, 0x00d5},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0x00fd, 0x000a},
	{0x01, 0x0038, 0x000b},
	{0x01, 0x00d1, 0x000c},
	{0x01, 0x00f7, 0x000d},
	{0x01, 0x00ed, 0x000e},
	{0x01, 0x00d8, 0x000f},
	{0x01, 0x0038, 0x0010},
	{0x01, 0x00ff, 0x0015},
	{0x01, 0x0001, 0x0016},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},
	{0x01, 0x00ff, 0x003e},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0060, 0x0057},
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x100a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x000c, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0000, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},
	{0x00, 0x0000, 0x0024},
	{0x00, 0x00d5, 0x0025},
	{0x00, 0x0000, 0x0026},
	{0x00, 0x000b, 0x0027},
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
	{0xff, 0x0000, 0x00d0},
	{0xff, 0x00d8, 0x00d1},
	{0xff, 0x0000, 0x00d4},
	{0xff, 0x0000, 0x00d5},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0x00fd, 0x000a},
	{0x01, 0x0038, 0x000b},
	{0x01, 0x00d1, 0x000c},
	{0x01, 0x00f7, 0x000d},
	{0x01, 0x00ed, 0x000e},
	{0x01, 0x00d8, 0x000f},
	{0x01, 0x0038, 0x0010},
	{0x01, 0x00ff, 0x0015},
	{0x01, 0x0001, 0x0016},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},
	{0x01, 0x00ff, 0x003e},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0060, 0x0057},
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x100a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x000f, 0x0000},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0xfffd, 0x000a},
	{0x01, 0x0023, 0x000b},
	{0x01, 0xffea, 0x000c},
	{0x01, 0xfff4, 0x000d},
	{0x01, 0xfffc, 0x000e},
	{0x01, 0xffe3, 0x000f},
	{0x01, 0x001f, 0x0010},
	{0x01, 0x00a8, 0x0001},
	{0x01, 0x0067, 0x0007},
	{0x01, 0x0042, 0x0051},
	{0x01, 0x0051, 0x0053},
	{0x01, 0x000a, 0x0003},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xc000, 0x0001},
	{0x02, 0x0000, 0x0005},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x000c, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0000, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},
	{0x00, 0x0000, 0x0024},
	{0x00, 0x00d5, 0x0025},
	{0x00, 0x0000, 0x0026},
	{0x00, 0x000b, 0x0027},
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
	{0xff, 0x0000, 0x00d0},
	{0xff, 0x00d8, 0x00d1},
	{0xff, 0x0000, 0x00d4},
	{0xff, 0x0000, 0x00d5},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0x00fd, 0x000a},
	{0x01, 0x0038, 0x000b},
	{0x01, 0x00d1, 0x000c},
	{0x01, 0x00f7, 0x000d},
	{0x01, 0x00ed, 0x000e},
	{0x01, 0x00d8, 0x000f},
	{0x01, 0x0038, 0x0010},
	{0x01, 0x00ff, 0x0015},
	{0x01, 0x0001, 0x0016},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},
	{0x01, 0x00ff, 0x003e},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0060, 0x0057},
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x100a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x000c, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0000, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},
	{0x00, 0x0000, 0x0024},
	{0x00, 0x00d5, 0x0025},
	{0x00, 0x0000, 0x0026},
	{0x00, 0x000b, 0x0027},
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
	{0xff, 0x0000, 0x00d0},
	{0xff, 0x00d8, 0x00d1},
	{0xff, 0x0000, 0x00d4},
	{0xff, 0x0000, 0x00d5},
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0x00fd, 0x000a},
	{0x01, 0x0038, 0x000b},
	{0x01, 0x00d1, 0x000c},
	{0x01, 0x00f7, 0x000d},
	{0x01, 0x00ed, 0x000e},
	{0x01, 0x00d8, 0x000f},
	{0x01, 0x0038, 0x0010},
	{0x01, 0x00ff, 0x0015},
	{0x01, 0x0001, 0x0016},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},
	{0x01, 0x00ff, 0x003e},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0060, 0x0057},
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x100a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x001e, 0x0000},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x0011, 0x0008},
	{0x01, 0x0032, 0x0009},
	{0x01, 0xfffd, 0x000a},
	{0x01, 0x0023, 0x000b},
	{0x01, 0xffea, 0x000c},
	{0x01, 0xfff4, 0x000d},
	{0x01, 0xfffc, 0x000e},
	{0x01, 0xffe3, 0x000f},
	{0x01, 0x001f, 0x0010},
	{0x01, 0x00a8, 0x0001},
	{0x01, 0x0067, 0x0007},
	{0x01, 0x0042, 0x0051},
	{0x01, 0x0051, 0x0053},
	{0x01, 0x000a, 0x0003},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x0007, 0x0005},
	{0x01, 0x0042, 0x0051},
	{0x01, 0x0051, 0x0053},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x002d, 0x0000},
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0001, 0x0056},
	{0x02, 0xc000, 0x0001},
	{0x02, 0x0000, 0x0005},
	{}
};

/* Unknown camera from Ori Usbid 0x0000:0x0000 */
/* Based on snoops from Ori Cohen */
static const __u16 spca501c_mysterious_open_data[][3] = {
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
/* DSP Registers */
	{0x01, 0x0016, 0x0011},	/* RGB offset */
	{0x01, 0x0000, 0x0012},
	{0x01, 0x0006, 0x0013},
	{0x01, 0x0078, 0x0051},
	{0x01, 0x0040, 0x0052},
	{0x01, 0x0046, 0x0053},
	{0x01, 0x0040, 0x0054},
	{0x00, 0x0025, 0x0000},
/*	{0x00, 0x0000, 0x0000 }, */
/* Part 2 */
/* TG Registers */
	{0x00, 0x0026, 0x0000},
	{0x00, 0x0001, 0x0000},
	{0x00, 0x0027, 0x0000},
	{0x00, 0x008a, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x2000, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0015, 0x0001},
	{0x05, 0x00ea, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0023, 0x0001},
	{0x05, 0x0003, 0x0000},
	{0x05, 0x0030, 0x0001},
	{0x05, 0x002b, 0x0000},
	{0x05, 0x0031, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0032, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0033, 0x0001},
	{0x05, 0x0023, 0x0000},
	{0x05, 0x0034, 0x0001},
	{0x05, 0x0002, 0x0000},
	{0x05, 0x0050, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0051, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0052, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0054, 0x0001},
	{0x05, 0x0001, 0x0000},
	{}
};

/* Based on snoops from Ori Cohen */
static const __u16 spca501c_mysterious_init_data[][3] = {
/* Part 3 */
/* TG registers */
/*	{0x00, 0x0000, 0x0000}, */
	{0x00, 0x0000, 0x0001},
	{0x00, 0x0000, 0x0002},
	{0x00, 0x0006, 0x0003},
	{0x00, 0x0000, 0x0004},
	{0x00, 0x0090, 0x0005},
	{0x00, 0x0000, 0x0006},
	{0x00, 0x0040, 0x0007},
	{0x00, 0x00c0, 0x0008},
	{0x00, 0x004a, 0x0009},
	{0x00, 0x0000, 0x000a},
	{0x00, 0x0000, 0x000b},
	{0x00, 0x0001, 0x000c},
	{0x00, 0x0001, 0x000d},
	{0x00, 0x0000, 0x000e},
	{0x00, 0x0002, 0x000f},
	{0x00, 0x0001, 0x0010},
	{0x00, 0x0000, 0x0011},
	{0x00, 0x0001, 0x0012},
	{0x00, 0x0002, 0x0020},
	{0x00, 0x0080, 0x0021},	/* 640 */
	{0x00, 0x0001, 0x0022},
	{0x00, 0x00e0, 0x0023},	/* 480 */
	{0x00, 0x0000, 0x0024},	/* Offset H hight */
	{0x00, 0x00d3, 0x0025},	/* low */
	{0x00, 0x0000, 0x0026},	/* Offset V */
	{0x00, 0x000d, 0x0027},	/* low */
	{0x00, 0x0000, 0x0046},
	{0x00, 0x0000, 0x0047},
	{0x00, 0x0000, 0x0048},
	{0x00, 0x0000, 0x0049},
	{0x00, 0x0008, 0x004a},
/* DSP Registers	*/
	{0x01, 0x00a6, 0x0000},
	{0x01, 0x0028, 0x0001},
	{0x01, 0x0000, 0x0002},
	{0x01, 0x000a, 0x0003},	/* Level Calc bit7 ->1 Auto */
	{0x01, 0x0040, 0x0004},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x000f, 0x0008},	/* A11 Color correction coeff */
	{0x01, 0x002d, 0x0009},	/* A12 */
	{0x01, 0x0005, 0x000a},	/* A13 */
	{0x01, 0x0023, 0x000b},	/* A21 */
	{0x01, 0x00e0, 0x000c},	/* A22 */
	{0x01, 0x00fd, 0x000d},	/* A23 */
	{0x01, 0x00f4, 0x000e},	/* A31 */
	{0x01, 0x00e4, 0x000f},	/* A32 */
	{0x01, 0x0028, 0x0010},	/* A33 */
	{0x01, 0x00ff, 0x0015},	/* Reserved */
	{0x01, 0x0001, 0x0016},	/* Reserved */
	{0x01, 0x0032, 0x0017},	/* Win1 Start begin */
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x0000, 0x001f},
	{0x01, 0x0000, 0x0020},	/* Win1 Start end */
	{0x01, 0x00ff, 0x003e},	/* Reserved begin */
	{0x01, 0x0002, 0x003f},
	{0x01, 0x0000, 0x0040},
	{0x01, 0x0035, 0x0041},
	{0x01, 0x0053, 0x0042},
	{0x01, 0x0069, 0x0043},
	{0x01, 0x007c, 0x0044},
	{0x01, 0x008c, 0x0045},
	{0x01, 0x009a, 0x0046},
	{0x01, 0x00a8, 0x0047},
	{0x01, 0x00b4, 0x0048},
	{0x01, 0x00bf, 0x0049},
	{0x01, 0x00ca, 0x004a},
	{0x01, 0x00d4, 0x004b},
	{0x01, 0x00dd, 0x004c},
	{0x01, 0x00e7, 0x004d},
	{0x01, 0x00ef, 0x004e},
	{0x01, 0x00f8, 0x004f},
	{0x01, 0x00ff, 0x0050},
	{0x01, 0x0003, 0x0056},	/* Reserved end */
	{0x01, 0x0060, 0x0057},	/* Edge Gain */
	{0x01, 0x0040, 0x0058},
	{0x01, 0x0011, 0x0059},	/* Edge Bandwidth */
	{0x01, 0x0001, 0x005a},
	{0x02, 0x0007, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x02, 0x0007, 0x0005},
	{0x02, 0x0015, 0x0006},
	{0x02, 0x200a, 0x0007},
	{0x02, 0xa048, 0x0000},
	{0x02, 0xc000, 0x0001},
	{0x02, 0x000f, 0x0005},
	{0x02, 0xa048, 0x0000},
	{0x05, 0x0022, 0x0004},
	{0x05, 0x0025, 0x0001},
	{0x05, 0x0000, 0x0000},
/* Part 4 */
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0001, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x05, 0x0021, 0x0001},
	{0x05, 0x00d2, 0x0000},
	{0x05, 0x0020, 0x0001},
	{0x05, 0x0000, 0x0000},
	{0x00, 0x0090, 0x0005},
	{0x01, 0x00a6, 0x0000},
	{0x02, 0x0000, 0x0005},
	{0x05, 0x0026, 0x0001},
	{0x05, 0x0001, 0x0000},
	{0x05, 0x0027, 0x0001},
	{0x05, 0x004e, 0x0000},
/* Part 5 */
	{0x01, 0x0003, 0x003f},
	{0x01, 0x0001, 0x0056},
	{0x01, 0x000f, 0x0008},
	{0x01, 0x002d, 0x0009},
	{0x01, 0x0005, 0x000a},
	{0x01, 0x0023, 0x000b},
	{0x01, 0xffe0, 0x000c},
	{0x01, 0xfffd, 0x000d},
	{0x01, 0xfff4, 0x000e},
	{0x01, 0xffe4, 0x000f},
	{0x01, 0x0028, 0x0010},
	{0x01, 0x00a8, 0x0001},
	{0x01, 0x0066, 0x0007},
	{0x01, 0x0032, 0x0017},
	{0x01, 0x0023, 0x0018},
	{0x01, 0x00ce, 0x0019},
	{0x01, 0x0023, 0x001a},
	{0x01, 0x0032, 0x001b},
	{0x01, 0x008d, 0x001c},
	{0x01, 0x00ce, 0x001d},
	{0x01, 0x008d, 0x001e},
	{0x01, 0x00c8, 0x0015},	/* c8 Poids fort Luma */
	{0x01, 0x0032, 0x0016},	/* 32 */
	{0x01, 0x0016, 0x0011},	/* R 00 */
	{0x01, 0x0016, 0x0012},	/* G 00 */
	{0x01, 0x0016, 0x0013},	/* B 00 */
	{0x01, 0x000a, 0x0003},
	{0x02, 0xc002, 0x0001},
	{0x02, 0x0007, 0x0005},
	{}
};

static int reg_write(struct usb_device *dev,
		     __u16 req, __u16 index, __u16 value)
{
	int ret;

	ret = usb_control_msg(dev,
			usb_sndctrlpipe(dev, 0),
			req,
			USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			value, index, NULL, 0, 500);
	PDEBUG(D_USBO, "reg write: 0x%02x 0x%02x 0x%02x",
		req, index, value);
	if (ret < 0)
		err("reg write: error %d", ret);
	return ret;
}


static int write_vector(struct gspca_dev *gspca_dev,
			const __u16 data[][3])
{
	struct usb_device *dev = gspca_dev->dev;
	int ret, i = 0;

	while (data[i][0] != 0 || data[i][1] != 0 || data[i][2] != 0) {
		ret = reg_write(dev, data[i][0], data[i][2], data[i][1]);
		if (ret < 0) {
			PDEBUG(D_ERR,
				"Reg write failed for 0x%02x,0x%02x,0x%02x",
				data[i][0], data[i][1], data[i][2]);
			return ret;
		}
		i++;
	}
	return 0;
}

static void setbrightness(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_write(gspca_dev->dev, SPCA501_REG_CCDSP, 0x12, sd->brightness);
}

static void setcontrast(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_write(gspca_dev->dev, 0x00, 0x00,
				  (sd->contrast >> 8) & 0xff);
	reg_write(gspca_dev->dev, 0x00, 0x01,
				  sd->contrast & 0xff);
}

static void setcolors(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_write(gspca_dev->dev, SPCA501_REG_CCDSP, 0x0c, sd->colors);
}

static void setblue_balance(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_write(gspca_dev->dev, SPCA501_REG_CCDSP, 0x11, sd->blue_balance);
}

static void setred_balance(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	reg_write(gspca_dev->dev, SPCA501_REG_CCDSP, 0x13, sd->red_balance);
}

/* this function is called at probe time */
static int sd_config(struct gspca_dev *gspca_dev,
			const struct usb_device_id *id)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct cam *cam;

	cam = &gspca_dev->cam;
	cam->cam_mode = vga_mode;
	cam->nmodes = ARRAY_SIZE(vga_mode);
	sd->subtype = id->driver_info;
	sd->brightness = sd_ctrls[MY_BRIGHTNESS].qctrl.default_value;
	sd->contrast = sd_ctrls[MY_CONTRAST].qctrl.default_value;
	sd->colors = sd_ctrls[MY_COLOR].qctrl.default_value;

	return 0;
}

/* this function is called at probe and resume time */
static int sd_init(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	switch (sd->subtype) {
	case Arowana300KCMOSCamera:
	case SmileIntlCamera:
		/* Arowana 300k CMOS Camera data */
		if (write_vector(gspca_dev, spca501c_arowana_init_data))
			goto error;
		break;
	case MystFromOriUnknownCamera:
		/* Unknown Ori CMOS Camera data */
		if (write_vector(gspca_dev, spca501c_mysterious_open_data))
			goto error;
		break;
	default:
		/* generic spca501 init data */
		if (write_vector(gspca_dev, spca501_init_data))
			goto error;
		break;
	}
	PDEBUG(D_STREAM, "Initializing SPCA501 finished");
	return 0;
error:
	return -EINVAL;
}

static int sd_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	struct usb_device *dev = gspca_dev->dev;
	int mode;

	switch (sd->subtype) {
	case ThreeComHomeConnectLite:
		/* Special handling for 3com data */
		write_vector(gspca_dev, spca501_3com_open_data);
		break;
	case Arowana300KCMOSCamera:
	case SmileIntlCamera:
		/* Arowana 300k CMOS Camera data */
		write_vector(gspca_dev, spca501c_arowana_open_data);
		break;
	case MystFromOriUnknownCamera:
		/* Unknown CMOS Camera data */
		write_vector(gspca_dev, spca501c_mysterious_init_data);
		break;
	default:
		/* Generic 501 open data */
		write_vector(gspca_dev, spca501_open_data);
	}

	/* memorize the wanted pixel format */
	mode = gspca_dev->cam.cam_mode[(int) gspca_dev->curr_mode].priv;

	/* Enable ISO packet machine CTRL reg=2,
	 * index=1 bitmask=0x2 (bit ordinal 1) */
	reg_write(dev, SPCA50X_REG_USB, 0x6, 0x94);
	switch (mode) {
	case 0: /* 640x480 */
		reg_write(dev, SPCA50X_REG_USB, 0x07, 0x004a);
		break;
	case 1: /* 320x240 */
		reg_write(dev, SPCA50X_REG_USB, 0x07, 0x104a);
		break;
	default:
/*	case 2:  * 160x120 */
		reg_write(dev, SPCA50X_REG_USB, 0x07, 0x204a);
		break;
	}
	reg_write(dev, SPCA501_REG_CTLRL, 0x01, 0x02);

	/* HDG atleast the Intel CreateAndShare needs to have one of its
	 * brightness / contrast / color set otherwise it assumes what seems
	 * max contrast. Note that strange enough setting any of these is
	 * enough to fix the max contrast problem, to be sure we set all 3 */
	setbrightness(gspca_dev);
	setcontrast(gspca_dev);
	setcolors(gspca_dev);
	return 0;
}

static void sd_stopN(struct gspca_dev *gspca_dev)
{
	/* Disable ISO packet
	 * machine CTRL reg=2, index=1 bitmask=0x0 (bit ordinal 1) */
	reg_write(gspca_dev->dev, SPCA501_REG_CTLRL, 0x01, 0x00);
}

/* called on streamoff with alt 0 and on disconnect */
static void sd_stop0(struct gspca_dev *gspca_dev)
{
	if (!gspca_dev->present)
		return;
	reg_write(gspca_dev->dev, SPCA501_REG_CTLRL, 0x05, 0x00);
}

static void sd_pkt_scan(struct gspca_dev *gspca_dev,
			u8 *data,			/* isoc packet */
			int len)			/* iso packet length */
{
	switch (data[0]) {
	case 0:				/* start of frame */
		gspca_frame_add(gspca_dev, LAST_PACKET, NULL, 0);
		data += SPCA501_OFFSET_DATA;
		len -= SPCA501_OFFSET_DATA;
		gspca_frame_add(gspca_dev, FIRST_PACKET, data, len);
		return;
	case 0xff:			/* drop */
/*		gspca_dev->last_packet_type = DISCARD_PACKET; */
		return;
	}
	data++;
	len--;
	gspca_frame_add(gspca_dev, INTER_PACKET, data, len);
}

static int sd_setbrightness(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->brightness = val;
	if (gspca_dev->streaming)
		setbrightness(gspca_dev);
	return 0;
}

static int sd_getbrightness(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->brightness;
	return 0;
}

static int sd_setcontrast(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->contrast = val;
	if (gspca_dev->streaming)
		setcontrast(gspca_dev);
	return 0;
}

static int sd_getcontrast(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->contrast;
	return 0;
}

static int sd_setcolors(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->colors = val;
	if (gspca_dev->streaming)
		setcolors(gspca_dev);
	return 0;
}

static int sd_getcolors(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->colors;
	return 0;
}

static int sd_setblue_balance(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->blue_balance = val;
	if (gspca_dev->streaming)
		setblue_balance(gspca_dev);
	return 0;
}

static int sd_getblue_balance(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->blue_balance;
	return 0;
}

static int sd_setred_balance(struct gspca_dev *gspca_dev, __s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	sd->red_balance = val;
	if (gspca_dev->streaming)
		setred_balance(gspca_dev);
	return 0;
}

static int sd_getred_balance(struct gspca_dev *gspca_dev, __s32 *val)
{
	struct sd *sd = (struct sd *) gspca_dev;

	*val = sd->red_balance;
	return 0;
}

/* sub-driver description */
static const struct sd_desc sd_desc = {
	.name = MODULE_NAME,
	.ctrls = sd_ctrls,
	.nctrls = ARRAY_SIZE(sd_ctrls),
	.config = sd_config,
	.init = sd_init,
	.start = sd_start,
	.stopN = sd_stopN,
	.stop0 = sd_stop0,
	.pkt_scan = sd_pkt_scan,
};

/* -- module initialisation -- */
static const struct usb_device_id device_table[] = {
	{USB_DEVICE(0x040a, 0x0002), .driver_info = KodakDVC325},
	{USB_DEVICE(0x0497, 0xc001), .driver_info = SmileIntlCamera},
	{USB_DEVICE(0x0506, 0x00df), .driver_info = ThreeComHomeConnectLite},
	{USB_DEVICE(0x0733, 0x0401), .driver_info = IntelCreateAndShare},
	{USB_DEVICE(0x0733, 0x0402), .driver_info = ViewQuestM318B},
	{USB_DEVICE(0x1776, 0x501c), .driver_info = Arowana300KCMOSCamera},
	{USB_DEVICE(0x0000, 0x0000), .driver_info = MystFromOriUnknownCamera},
	{}
};
MODULE_DEVICE_TABLE(usb, device_table);

/* -- device connect -- */
static int sd_probe(struct usb_interface *intf,
			const struct usb_device_id *id)
{
	return gspca_dev_probe(intf, id, &sd_desc, sizeof(struct sd),
				THIS_MODULE);
}

static struct usb_driver sd_driver = {
	.name = MODULE_NAME,
	.id_table = device_table,
	.probe = sd_probe,
	.disconnect = gspca_disconnect,
#ifdef CONFIG_PM
	.suspend = gspca_suspend,
	.resume = gspca_resume,
#endif
};

/* -- module insert / remove -- */
static int __init sd_mod_init(void)
{
	return usb_register(&sd_driver);
}
static void __exit sd_mod_exit(void)
{
	usb_deregister(&sd_driver);
}

module_init(sd_mod_init);
module_exit(sd_mod_exit);
