/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#include "fc_os.h"

#include "fc_hw.h"
#include "fc.h"

#include "fcdiag.h"
#include "fcfgparm.h"
#include "fcmsg.h"
#include "fc_crtn.h"   /* Core - external routine definitions */
#include "fc_ertn.h"   /* Environment - external routine definitions */

extern uint32           fcPAGESIZE;
extern fc_dd_ctl_t      DD_CTL;
extern iCfgParam icfgparam[];
extern int      fc_max_els_sent;


/*
 * Define the following to Enable SANITY check logic in routines 
 * fc_getvaddr() and fc_mapvaddr().
 * #define FC_DBG_VADDR_SANITY_CHK
 */

/* Routine Declaration - Local */
/* There currently are no local routine declarations */
/* End Routine Declaration - Local */

/***************************************************/
/**  fc_malloc_buffer                             **/
/**                                               **/
/**  This routine will allocate iocb/data buffer  **/
/**  space and setup the buffers for all rings on **/
/**  the specified board to use. The data buffers **/
/**  can be posted to the ring with the           **/
/**  fc_post_buffer routine.  The iocb buffers    **/
/**  are used to make a temp copy of the response **/
/**  ring iocbs. Returns 0 if not enough memory,  **/
/**  Returns 1 if successful.                     **/
/***************************************************/
_static_ int
fc_malloc_buffer(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO * binfo = &BINFO;
   iCfgParam   * clp;
   int  i, j;
   uchar * bp;
   uchar * oldbp;
   RING * rp;
   MEMSEG * mp;
   MATCHMAP * matp;
   MBUF_INFO * buf_info;
   MBUF_INFO bufinfo;
   unsigned long iflag;

   buf_info = &bufinfo;
   clp = DD_CTL.p_config[binfo->fc_brd_no];


   for (i = 0; i < binfo->fc_ffnumrings; i++) {
      rp = &binfo->fc_ring[i];
      rp->fc_mpon = 0;
      rp->fc_mpoff = 0;
   }

   if(clp[CFG_FCP_ON].a_current) {
      buf_info->size = (MAX_FCP_CMDS * sizeof(void *));
      buf_info->flags = 0;
      buf_info->align = sizeof(void *);
      buf_info->dma_handle = 0;

      /* Create a table to relate FCP iotags to fc_buf addresses */
      fc_malloc(p_dev_ctl, buf_info);
      if (buf_info->virt == NULL) {
         fc_free_buffer(p_dev_ctl);
         return(0);
      }
      binfo->fc_table = (FCPTBL * )buf_info->virt;
      fc_bzero((char *)binfo->fc_table, MAX_FCP_CMDS * sizeof(void *));
   }

   /* Initialize xmit/receive buffer structure */
   /* Three buffers per response entry will initially be posted to ELS ring */
   iflag = lpfc_mempool_disable_lock(p_dev_ctl);
   mp = &binfo->fc_memseg[MEM_BUF];
   mp->fc_memsize = FCELSSIZE;
   if(clp[CFG_NUM_BUFS].a_current < 50)
      mp->fc_numblks = 50;
   else
      mp->fc_numblks = (ushort)clp[CFG_NUM_BUFS].a_current;

   /* MEM_BUF is same pool as MEM_BPL */
   if (binfo->fc_sli == 2)
      mp->fc_numblks += MAX_SLI2_IOCB;

   mp->fc_memflag = FC_MEM_DMA;
   mp->fc_lowmem = (3 * fc_max_els_sent) + 8;

   if((2*mp->fc_lowmem) > mp->fc_numblks)
      mp->fc_lowmem = (mp->fc_numblks / 2);

   /* Initialize mailbox cmd buffer structure */
   mp = &binfo->fc_memseg[MEM_MBOX];
   mp->fc_memsize = sizeof(MAILBOXQ);
   mp->fc_numblks = (short)clp[CFG_NUM_NODES].a_current + 32;
   mp->fc_memflag = 0;
   mp->fc_lowmem = (2 * fc_max_els_sent) + 8;

   /* Initialize iocb buffer structure */
   mp = &binfo->fc_memseg[MEM_IOCB];
   mp->fc_memsize = sizeof(IOCBQ);
   mp->fc_numblks = (ushort)clp[CFG_NUM_IOCBS].a_current + MIN_CLK_BLKS;
   mp->fc_memflag = 0;
   mp->fc_lowmem = (2 * fc_max_els_sent) + 8;

   /* Initialize iocb buffer structure */
   mp = &binfo->fc_memseg[MEM_NLP];
   mp->fc_memsize = sizeof(NODELIST);
   mp->fc_numblks = (short)clp[CFG_NUM_NODES].a_current + 2;
   mp->fc_memflag = 0;
   mp->fc_lowmem = 0;


   /* Allocate buffer pools for above buffer structures */
   for (j = 0; j < FC_MAX_SEG; j++) {
      mp = &binfo->fc_memseg[j];


      mp->fc_memptr = 0;
      mp->fc_endmemptr = 0;
      mp->fc_memhi = 0;
      mp->fc_memlo = 0;

      for (i = 0; i < mp->fc_numblks; i++) {
         /* If this is a DMA buffer we need alignment on a page so we don't
          * want to worry about buffers spanning page boundries when mapping
          * memory for the adapter.
          */
         if (mp->fc_memflag & FC_MEM_DMA) {
            buf_info->size = sizeof(MATCHMAP);
            buf_info->flags = 0;
            buf_info->align = sizeof(void *);
            buf_info->dma_handle = 0;

            fc_malloc(p_dev_ctl, buf_info);
            if (buf_info->virt == NULL) {
               lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
               fc_free_buffer(p_dev_ctl);
               return(0);
            }

            matp = (MATCHMAP * )buf_info->virt;
            fc_bzero(matp, sizeof(MATCHMAP));
            if((ulong)matp > (ulong)(mp->fc_memhi))
               mp->fc_memhi = (uchar *)matp;
            if(mp->fc_memlo == 0)
               mp->fc_memlo = (uchar *)matp;
            else {
               if((ulong)matp < (ulong)(mp->fc_memlo))
                  mp->fc_memlo = (uchar *)matp;
            }

            buf_info->size = mp->fc_memsize;
            buf_info->flags = FC_MBUF_DMA;
            buf_info->dma_handle = 0;

            switch (mp->fc_memsize) {
            case sizeof(FCP_CMND):
               buf_info->align = sizeof(FCP_CMND);
               break;

            case 1024:
               buf_info->align = 1024;
               break;

            case 2048:
               buf_info->align = 2048;
               break;

            case 4096:
               buf_info->align = 4096;
               break;

            default:
               buf_info->align = sizeof(void *);
               break;
            }

            fc_malloc(p_dev_ctl, buf_info);
            if (buf_info->virt == NULL) {
               fc_free_buffer(p_dev_ctl);
               return(0);
            }
            bp = (uchar * )buf_info->virt;
            fc_bzero(bp, mp->fc_memsize);

            /* Link buffer into beginning of list. The first pointer in
             * each buffer is a forward pointer to the next buffer.
             */
            oldbp = mp->fc_memptr;
            if(oldbp == 0)
               mp->fc_endmemptr = (uchar *)matp;
            mp->fc_memptr = (uchar * )matp;
            matp->fc_mptr = oldbp;
            matp->virt = bp;
            if (buf_info->dma_handle) {
               matp->dma_handle = buf_info->dma_handle;
               matp->data_handle = buf_info->data_handle;
            }
            matp->phys = (uchar * )buf_info->phys;
         } else {
            buf_info->size = mp->fc_memsize;
            buf_info->flags = 0;
            buf_info->align = sizeof(void *);
            buf_info->dma_handle = 0;

            fc_malloc(p_dev_ctl, buf_info);
            if (buf_info->virt == NULL) {
               lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
               fc_free_buffer(p_dev_ctl);
               return(0);
            }
            bp = (uchar * )buf_info->virt;
            fc_bzero(bp, mp->fc_memsize);
            if((ulong)bp > (ulong)(mp->fc_memhi))
               mp->fc_memhi = (uchar *)bp;
            if(mp->fc_memlo == 0)
               mp->fc_memlo = (uchar *)bp;
            else {
               if((ulong)bp < (ulong)(mp->fc_memlo))
                  mp->fc_memlo = (uchar *)bp;
            }

            /* Link buffer into beginning of list. The first pointer in
             * each buffer is a forward pointer to the next buffer.
             */
            oldbp = mp->fc_memptr;
            if(oldbp == 0)
               mp->fc_endmemptr = bp;
            mp->fc_memptr = bp;
            *((uchar * *)bp) = oldbp;
         }
      }

      /* free blocks = total blocks right now */
      mp->fc_free = i;
   }

   lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
   return(1);
}       /* End fc_malloc_buffer */


/***************************************************/
/**  fc_free_buffer                               **/
/**                                               **/
/**  This routine will free iocb/data buffer space */
/***************************************************/
_static_ int
fc_free_buffer(
fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO * binfo = &BINFO;
   int  j, ipri;
   uchar * bp;
   MEMSEG * mp;
   MATCHMAP * mm;
   NODELIST * ndlp;
   NODELIST * ondlp;
   RING * rp;
   IOCBQ * iocbq, *save;
   MAILBOXQ * mbox, *mbsave;
   MBUF_INFO * buf_info;
   MBUF_INFO bufinfo;
   FCCLOCK_INFO * clock_info;
   FCCLOCK * cb;
   FCCLOCK * nextcb;
   unsigned long iflag;

   buf_info = &bufinfo;

   /* free the mapped address match area for each ring */
   for (j = 0; j < binfo->fc_ffnumrings; j++) {
      rp = &binfo->fc_ring[j];

      /* Free everything on tx queue */
      iocbq = (IOCBQ * )(rp->fc_tx.q_first);
      while (iocbq) {
         save  = iocbq;
         iocbq = (IOCBQ * )iocbq->q;
         fc_mem_put(binfo, MEM_IOCB, (uchar * )save);
      }

      if (j != FC_FCP_RING) {
         /* Free everything on txp queue */
         unsigned long iflag;

         iflag = lpfc_q_disable_lock(p_dev_ctl);
         iocbq = (IOCBQ * )(rp->fc_txp.q_first);
         while (iocbq) {
            save  = iocbq;
            iocbq = (IOCBQ * )iocbq->q;
            fc_mem_put(binfo, MEM_IOCB, (uchar * )save);
         }
         lpfc_q_unlock_enable(p_dev_ctl, iflag);

         while (rp->fc_mpoff) {
            uchar * addr;

            addr = 0;
            mm = (MATCHMAP * )(rp->fc_mpoff);
            if (j == FC_IP_RING)
               addr = (uchar * )(fcnextpkt((fcipbuf_t * )mm));
            else if (j == FC_ELS_RING)
               addr = mm->phys;
            if ((mm = fc_getvaddr(p_dev_ctl, rp, addr))) {
               if (j == FC_ELS_RING) {
                  fc_mem_put(binfo, MEM_BUF, (uchar * )mm);
               }
               else if (j == FC_IP_RING) {
                  fcipbuf_t  * mbuf;

                  mbuf = (fcipbuf_t * )mm;
                  fcnextdata(mbuf) = 0;
                  fcnextpkt(mbuf) = 0;
                  m_freem(mbuf);
               }
            }
         }
      }
   }

   /* Free any delayed ELS xmits */
   if(binfo->fc_delayxmit) {
      iocbq = binfo->fc_delayxmit;
      binfo->fc_delayxmit = 0;
      while(iocbq) {
         mm = (MATCHMAP * )iocbq->bp;
         if (binfo->fc_flag & FC_SLI2) {
            fc_mem_put(binfo, MEM_BPL, (uchar * )iocbq->bpl);
         }
         fc_mem_put(binfo, MEM_BUF, (uchar * )mm);
         fc_mem_put(binfo, MEM_BUF, (uchar * )iocbq->info);
         save = iocbq;
         iocbq = (IOCBQ *)save->q;
         fc_mem_put(binfo, MEM_IOCB, (uchar * )save);
      }
   }

   if (binfo->fc_table) {
      buf_info->size = (MAX_FCP_CMDS * sizeof(void *));
      buf_info->virt = (uint32 * )binfo->fc_table;
      buf_info->phys = 0;
      buf_info->flags = 0;
      buf_info->dma_handle = 0;
      fc_free(p_dev_ctl, buf_info);
      binfo->fc_table = 0;
   }

   /* Free everything on mbox queue */
   mbox = (MAILBOXQ * )(binfo->fc_mbox.q_first);
   while (mbox) {
      mbsave = mbox;
      mbox = (MAILBOXQ * )mbox->q;
      fc_mem_put(binfo, MEM_MBOX, (uchar * )mbsave);
   }
   binfo->fc_mbox.q_first = 0;
   binfo->fc_mbox.q_last = 0;
   binfo->fc_mbox_active = 0;

   /* Free everything on iocb plogi queue */
   iocbq = (IOCBQ * )(binfo->fc_plogi.q_first);
   while (iocbq) {
      save = iocbq;
      iocbq = (IOCBQ * )iocbq->q;
      fc_mem_put(binfo, MEM_IOCB, (uchar * )save);
   }
   binfo->fc_plogi.q_first = 0;
   binfo->fc_plogi.q_last = 0;

   /* Now cleanup unexpired clock blocks */
   clock_info = &DD_CTL.fc_clock_info;
   ipri = disable_lock(FC_LVL, &CLOCK_LOCK);

   cb = clock_info->fc_clkhdr.cl_f;
   while (cb != (FCCLOCK * ) & clock_info->fc_clkhdr) {
      nextcb = cb->cl_fw;
      if(cb->cl_p_dev_ctl == (void *)p_dev_ctl) {
         fc_clock_deque(cb);
         /* Release clock block */
         fc_clkrelb(p_dev_ctl, cb);
         /* start over */
      }
      cb = nextcb;
   }
   unlock_enable(ipri, &CLOCK_LOCK);

   /* Free all node table entries */
   ndlp = binfo->fc_nlpbind_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
     ndlp = binfo->fc_nlpunmap_start;
   if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
      ndlp = binfo->fc_nlpmap_start;
   while(ndlp != (NODELIST *)&binfo->fc_nlpmap_start) {
      ondlp = ndlp;
      ndlp = (NODELIST *)ndlp->nlp_listp_next;
      if(ndlp == (NODELIST *)&binfo->fc_nlpbind_start)
        ndlp = binfo->fc_nlpunmap_start;
      if(ndlp == (NODELIST *)&binfo->fc_nlpunmap_start)
         ndlp = binfo->fc_nlpmap_start;
      fc_mem_put(binfo, MEM_NLP, (uchar * )ondlp);
   }
   binfo->fc_nlpbind_start  = (NODELIST *)&binfo->fc_nlpbind_start;
   binfo->fc_nlpbind_end    = (NODELIST *)&binfo->fc_nlpbind_start;
   binfo->fc_nlpmap_start   = (NODELIST *)&binfo->fc_nlpmap_start;
   binfo->fc_nlpmap_end     = (NODELIST *)&binfo->fc_nlpmap_start;
   binfo->fc_nlpunmap_start = (NODELIST *)&binfo->fc_nlpunmap_start;
   binfo->fc_nlpunmap_end   = (NODELIST *)&binfo->fc_nlpunmap_start;

   iflag = lpfc_mempool_disable_lock(p_dev_ctl);
   /* Loop through all memory buffer pools */
   for (j = 0; j < FC_MAX_SEG; j++) {
      mp = &binfo->fc_memseg[j];
      /* Free memory associated with all buffers on free buffer pool */
      while ((bp = mp->fc_memptr) != NULL) {
         mp->fc_memptr = *((uchar * *)bp);
         if (mp->fc_memflag & FC_MEM_DMA) {
            mm = (MATCHMAP * )bp;
            bp = mm->virt;
            buf_info->size = mp->fc_memsize;
            buf_info->virt = (uint32 * )bp;
            buf_info->phys = (uint32 * )mm->phys;
            buf_info->flags = FC_MBUF_DMA;
            if (mm->dma_handle) {
               buf_info->dma_handle = mm->dma_handle;
               buf_info->data_handle = mm->data_handle;
            }
            fc_free(p_dev_ctl, buf_info);

            buf_info->size = sizeof(MATCHMAP);
            buf_info->virt = (uint32 * )mm;
            buf_info->phys = 0;
            buf_info->flags = 0;
            buf_info->dma_handle = 0;
            fc_free(p_dev_ctl, buf_info);
         } else {
            buf_info->size = mp->fc_memsize;
            buf_info->virt = (uint32 * )bp;
            buf_info->phys = 0;
            buf_info->flags = 0;
            buf_info->dma_handle = 0;
            fc_free(p_dev_ctl, buf_info);
         }
      }
      mp->fc_endmemptr = NULL;
      mp->fc_free = 0;
   }
   lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
   return(0);
}       /* End fc_free_buffer */



/**************************************************/
/**  fc_mem_get                                  **/
/**                                              **/
/**  This routine will get a free memory buffer. **/
/**  seg identifies which buffer pool to use.    **/
/**  Returns the free buffer ptr or 0 for no buf **/
/**************************************************/
_static_ uchar *
fc_mem_get(
FC_BRD_INFO *binfo,
uint32      arg)
{
   uchar * bp;
   MEMSEG * mp;
   uint32 seg = arg & MEM_SEG_MASK;
   int  low;
   fc_dev_ctl_t *p_dev_ctl;
   unsigned long iflag;

   /* range check on seg argument */
   if (seg >= FC_MAX_SEG)
      return((uchar * )0);

   p_dev_ctl = (fc_dev_ctl_t *)binfo->fc_p_dev_ctl;
   iflag = lpfc_mempool_disable_lock(p_dev_ctl);
   mp = &binfo->fc_memseg[seg];

   if ((low = (!(arg & MEM_PRI) && (mp->fc_free <= mp->fc_lowmem)))) {
      /* Memory Buffer Pool is below low water mark */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0406,                  /* ptr to msg structure */
              fc_mes0406,                     /* ptr to msg */
               fc_msgBlk0406.msgPreambleStr,  /* begin varargs */
                seg,
                 mp->fc_lowmem,
                  low);                       /* end varargs */
      /* Low priority request and not enough buffers, so fail */
      lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
      return((uchar * )0);
   }

   bp = mp->fc_memptr;

   if (bp) {
      if(((ulong)bp > (ulong)(mp->fc_memhi)) || ((ulong)bp < (ulong)(mp->fc_memlo))) {
         /* Memory Buffer Pool is corrupted */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0407,                  /* ptr to msg structure */
                 fc_mes0407,                     /* ptr to msg */
                  fc_msgBlk0407.msgPreambleStr,  /* begin varargs */
                   seg,
                    (ulong)bp,
                     (ulong)mp->fc_memhi,
                      (ulong)mp->fc_memlo);      /* end varargs */
         mp->fc_memptr = 0;
         lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
         return((uchar * )0);
      }

      /* If a memory block exists, take it off freelist 
       * and return it to the user.
       */
      if(mp->fc_endmemptr == bp) {
         mp->fc_endmemptr = 0;
      }
      mp->fc_memptr = *((uchar * *)bp);
      *((uchar * *)bp) = 0;
      mp->fc_free--;
   } else {
      /* Memory Buffer Pool is out of buffers */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0409,                  /* ptr to msg structure */
              fc_mes0409,                     /* ptr to msg */
               fc_msgBlk0409.msgPreambleStr,  /* begin varargs */
                seg,
                 mp->fc_free,
                  binfo->fc_mbox.q_cnt,
                   (ulong)(mp->fc_memhi));    /* end varargs */
      FCSTATCTR.memAllocErr++;
   }

   if (seg == MEM_NLP) {
      /* GET nodelist */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0927,                  /* ptr to msg structure */
              fc_mes0927,                     /* ptr to msg */
               fc_msgBlk0927.msgPreambleStr,  /* begin varargs */
                (ulong)bp,
                  mp->fc_free);               /* end varargs */
   }

   lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
   return(bp);
}       /* End fc_mem_get */


/**************************************************/
/**  fc_mem_put                                  **/
/**                                              **/
/**  This routine will put a memory buffer back  **/
/**  on the freelist.                            **/
/**  seg identifies which buffer pool to use.    **/
/**************************************************/
_static_ uchar *
fc_mem_put(
FC_BRD_INFO *binfo,
uint32      seg,
uchar       *bp)
{
   MEMSEG * mp;
   uchar * oldbp;
   fc_dev_ctl_t *p_dev_ctl;
   unsigned long iflag;
   /* range check on seg argument */
   if (seg >= FC_MAX_SEG)
      return((uchar * )0);

   p_dev_ctl = (fc_dev_ctl_t *)binfo->fc_p_dev_ctl;
   iflag = lpfc_mempool_disable_lock(p_dev_ctl);
   mp = &binfo->fc_memseg[seg];

   if (bp) {

      if(((ulong)bp > (ulong)(mp->fc_memhi)) || ((ulong)bp < (ulong)(mp->fc_memlo))) {
         /* Memory Buffer Pool is corrupted */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0408,                  /* ptr to msg structure */
                 fc_mes0408,                     /* ptr to msg */
                  fc_msgBlk0408.msgPreambleStr,  /* begin varargs */
                   seg,
                    (ulong)bp,
                     (ulong)mp->fc_memhi,
                      (ulong)mp->fc_memlo);      /* end varargs */
         lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
         return((uchar * )0);
      }
      /* If a memory block exists, put it on freelist 
       * and return it to the user.
       */
      oldbp = mp->fc_memptr;
      mp->fc_memptr = bp;
      *((uchar * *)bp) = oldbp;
      if(oldbp == 0)
         mp->fc_endmemptr = bp;
      mp->fc_free++;
   }

   if (seg == MEM_NLP) {
      /* PUT nodelist */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0928,                  /* ptr to msg structure */
                 fc_mes0928,                     /* ptr to msg */
                  fc_msgBlk0928.msgPreambleStr,  /* begin varargs */
                   (ulong)bp,
                    mp->fc_free);                /* end varargs */
   }

   lpfc_mempool_unlock_enable(p_dev_ctl, iflag);
   return(bp);
}       /* End fc_mem_put */


/* Look up the virtual address given a mapped address */
_static_ MATCHMAP *
fc_getvaddr(
fc_dev_ctl_t *p_dev_ctl,
RING        *rp,
uchar       *mapbp)
{
   FC_BRD_INFO * binfo;

   binfo = &BINFO;
   /* While there are available slots in the list */
   if (rp->fc_ringno == FC_ELS_RING) {
      MATCHMAP * mp;
      MATCHMAP * mpoff;

      mpoff = (MATCHMAP * )rp->fc_mpoff;
      mp = 0;

      while (mpoff) {
         /* Check for a match */
         if (mpoff->phys == mapbp) {
            /* If we matched on the first slot */
            if (mp == 0) {
               rp->fc_mpoff = mpoff->fc_mptr;
            } else {
               mp->fc_mptr = mpoff->fc_mptr;
            }

            if (rp->fc_mpon == (uchar * )mpoff) {
               rp->fc_mpon = (uchar * )mp;
            }
            rp->fc_bufcnt--;
            mpoff->fc_mptr = 0;

            fc_mpdata_sync(mpoff->dma_handle, 0, 0, DDI_DMA_SYNC_FORKERNEL);

            /* Return entry */
            return(mpoff);
         }
         mp = mpoff;
         mpoff = (MATCHMAP * )mpoff->fc_mptr;
      }
   }

   if (rp->fc_ringno == FC_IP_RING) {
      fcipbuf_t * mb;
      fcipbuf_t * mboff;

      mboff = (fcipbuf_t * )rp->fc_mpoff;
      mb = 0;

      while (mboff) {
         /* Check for a match */
         if (fcnextpkt(mboff) == (fcipbuf_t * )mapbp) {
            /* If we matched on the first slot */
            if (mb == 0) {
               rp->fc_mpoff = (uchar * )fcnextdata(mboff);
            } else {
               fcnextdata(mb) = fcnextdata(mboff);
            }

            if (rp->fc_mpon == (uchar * )mboff) {
               rp->fc_mpon = (uchar * )mb;
            }
            rp->fc_bufcnt--;

            if (fcnextpkt(mboff)) {
               MBUF_INFO         * buf_info;
               MBUF_INFO        bufinfo;

               buf_info = &bufinfo;
               buf_info->dma_handle = (ulong * )fcgethandle(mboff);
               if (buf_info->dma_handle) {
                  fc_mpdata_sync(buf_info->dma_handle, 0, 0,
                      DDI_DMA_SYNC_FORKERNEL);
               }
               buf_info->virt = 0;
               buf_info->flags  = (FC_MBUF_PHYSONLY | FC_MBUF_DMA);
               buf_info->phys = (uint32 * )fcnextpkt(mboff);
               buf_info->size = fcPAGESIZE;
               fc_free(p_dev_ctl, buf_info);
            }

            fcsethandle(mboff, 0);
            fcnextpkt(mboff) = 0;
            fcnextdata(mboff) = 0;
            /* Return entry */
            return((MATCHMAP * )mboff);
         }
         mb = mboff;
         mboff = (fcipbuf_t * )fcnextdata(mboff);
      }
   }
   /* Cannot find virtual addr for mapped buf on ring (num) */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk0410,                  /* ptr to msg structure */
           fc_mes0410,                     /* ptr to msg */
            fc_msgBlk0410.msgPreambleStr,  /* begin varargs */
             rp->fc_ringno,
               (ulong)mapbp,
                (ulong)rp->fc_mpoff,
                 (ulong)rp->fc_mpon);      /* end varargs */
   FCSTATCTR.noVirtPtr++;
   return(0);
}       /* End fc_getvaddr */


/* Given a virtual address, bp, generate the physical mapped address
 * and place it where addr points to. Save the address pair for lookup later.
 */
_static_ void
fc_mapvaddr(
FC_BRD_INFO *binfo,
RING        *rp,
MATCHMAP    *mp,
uint32      *haddr,
uint32      *laddr)
{
   fcipbuf_t * mbuf;

   switch (rp->fc_ringno) {
   case FC_ELS_RING:
      mp->fc_mptr = 0;
      /* Update slot fc_mpon points to then bump it */
      if (rp->fc_mpoff == 0) {
         rp->fc_mpoff = (uchar * )mp;
         rp->fc_mpon = (uchar * )mp;
      } else {
         ((MATCHMAP * )(rp->fc_mpon))->fc_mptr = (uchar * )mp;
         rp->fc_mpon = (uchar * )mp;
      }
      if (binfo->fc_flag & FC_SLI2) {
         *haddr = (uint32)putPaddrHigh(mp->phys);   /* return mapped address */
         *laddr = (uint32)putPaddrLow(mp->phys);   /* return mapped address */
      } else {
         *laddr = (uint32)putPaddrLow(mp->phys);   /* return mapped address */
      }
      break;

   case FC_IP_RING:
      mbuf = (fcipbuf_t * )mp;
      fcnextdata(mbuf) = 0;
      /* Update slot fc_mpon points to then bump it */
      if (rp->fc_mpoff == 0) {
         rp->fc_mpoff = (uchar * )mbuf;
         rp->fc_mpon = (uchar * )mbuf;
      } else {
         fcnextdata((fcipbuf_t * )(rp->fc_mpon)) = mbuf;
         rp->fc_mpon = (uchar * )mbuf;
      }
      if (binfo->fc_flag & FC_SLI2) {
         *haddr = (uint32)putPaddrHigh(fcnextpkt(mbuf));
         *laddr = (uint32)putPaddrLow(fcnextpkt(mbuf));
      } else {
         *laddr = (uint32)putPaddrLow(fcnextpkt(mbuf));
      }
      break;
   }

   rp->fc_bufcnt++;
   return;
}       /* End fc_mapvaddr */



