/*
 *      Binding Cache
 *
 *      Authors:
 *      Juha Mynttinen            <jmynttin@cc.hut.fi>
 *
 *      $Id: s.bcache.c 1.68 02/12/19 13:57:09+02:00 vnuorval@dsl-hkigw1d8c.dial.inet.fi $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 */

/*
 *	Changes:
 *
 *	Nanno Langstraat	:	Timer code cleaned up, active socket
 *					test rewritten
 */

#include <linux/autoconf.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/in6.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/proc_fs.h>
#include <net/ipv6.h>
#include <net/addrconf.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <net/mipv6.h>

#include "bcache.h"
#include "hashlist.h"
#include "debug.h"
#include "mobhdr.h"
#include "util.h"
#include "tunnel.h"
#include "ha.h"

#define TIMERDELAY HZ/10

struct mipv6_bcache {
	struct hashlist *entries;
	__u32 size;
	struct timer_list callback_timer;
};

static rwlock_t bcache_lock = RW_LOCK_UNLOCKED;

static struct mipv6_bcache bcache;

static int bcache_proc_info(char *buffer, char **start, off_t offset,
			    int length);

static void set_timer(void);


#define MIPV6_BCACHE_HASHSIZE  32

/* Moment of transmission of a BR, in seconds before bcache entry expiry */
#define BCACHE_BR_SEND_LEAD  3

/* No BR is sent if the bcache entry hasn't been used in the last
 * BCACHE_BR_SEND_THRESHOLD seconds before expiry */
#define BCACHE_BR_SEND_THRESHOLD  10

/* 
 * Internal functions.
 *
 * Assume that synchronization is taken care by the callers of these
 * functions, in the top level of the module. This is to avoid
 * deadlocks, when called function tries to get the same lock with the
 * caller.
 */

/*
 * Callback for hashlist_iterate
 */

struct cache_entry_iterator_args {
	struct mipv6_bcache_entry **entry;
};

static int find_first_cache_entry_iterator(void *data, void *args,
					   unsigned long *lifetime)
{
	struct mipv6_bcache_entry *entry =
	    (struct mipv6_bcache_entry *) data;
	struct cache_entry_iterator_args *state =
	    (struct cache_entry_iterator_args *) args;

	if (entry == NULL) {
		DEBUG(DBG_ERROR, "iterator called with NULL argument");
		return ITERATOR_ERR;	/* continue iteration ?? */
	}

	if (entry->type == CACHE_ENTRY) {
		*(state->entry) = entry;
		return ITERATOR_STOP;	/* stop iteration */
	} else {
		return ITERATOR_CONT;	/* continue iteration */
	}
}


/* 
 * Get memory for a new bcache entry.  If bcache is full, a cache
 * entry may be deleted to get space for a home registration, but not
 * vice versa.
 */
static struct mipv6_bcache_entry *mipv6_bcache_get_entry(__u8 type)
{
	struct mipv6_bcache_entry *entry;
	struct cache_entry_iterator_args args;

	DEBUG_FUNC();

	entry = (struct mipv6_bcache_entry *)
		hashlist_alloc(bcache.entries, SLAB_ATOMIC);

	/* Different cache replacement policies for HOME_REGISTRATION
           and CACHE_ENTRY could be implemented here, but for now we
           always replace the CACHE_ENTRY closest to expiration.  Type
           HOME_REGISTRATION entry may never be deleted before
           expiration. */
	if (entry == NULL) {
		/* cache full, try to delete a CACHE_ENTRY */
		args.entry = &entry;
		hashlist_iterate(bcache.entries, &args,
				 find_first_cache_entry_iterator);
		if (entry == NULL)
			return NULL;
		hashlist_delete(bcache.entries,
				(struct hashlist_entry *)entry);
		entry = (struct mipv6_bcache_entry *)
			hashlist_alloc(bcache.entries, SLAB_ATOMIC);
	}
	return entry;
}

/*
 * Frees entry's memory allocated with mipv6_bcache_get_entry
 */
static void mipv6_bcache_entry_free(struct mipv6_bcache_entry *entry)
{
	hashlist_free(bcache.entries, (void *) entry);
}

static int is_valid_type(__u8 type)
{
	if (type != CACHE_ENTRY && type != HOME_REGISTRATION)
		return 0;
	else
		return 1;
}

/*
 * Removes all expired entries 
 */
static void expire(void)
{
	struct mipv6_bcache_entry *entry;
	unsigned long now = jiffies;
	struct br_addrs {
		struct in6_addr daddr;
		struct in6_addr saddr;
		struct br_addrs *next;
	};
	struct br_addrs *br_info = NULL;

	DEBUG_FUNC();

	write_lock(&bcache_lock);

	while ((entry = (struct mipv6_bcache_entry *)
		hashlist_get_first(bcache.entries)) != NULL) {
		if (entry->callback_time <= now) {
			DEBUG(DBG_INFO, "an entry expired");

			if (entry->type & HOME_REGISTRATION) {
				del_proxy(&entry->home_addr, entry);
			}
			hashlist_delete(bcache.entries, (void *)entry);
			mipv6_bcache_entry_free(entry);
			entry = NULL;
		} else if (entry->br_callback_time != 0 &&
			   entry->br_callback_time <= now &&
			   entry->type & !HOME_REGISTRATION) {
			if (now - entry->last_used <
			    BCACHE_BR_SEND_THRESHOLD * HZ) {
				struct br_addrs *tmp;

				tmp = br_info;
				DEBUG(DBG_INFO,
				      "bcache entry recently used. Sending BR.");
				/* queue for sending */
				br_info =
				    kmalloc(sizeof(struct br_addrs),
					    GFP_ATOMIC);
				if (br_info) {
					ipv6_addr_copy(&br_info->saddr,
						       &entry->our_addr);
					ipv6_addr_copy(&br_info->daddr,
						       &entry->home_addr);
					br_info->next = tmp;
					entry->last_br = now;
				} else {
					br_info = tmp;
					DEBUG(DBG_ERROR, "Out of memory");
				}
			}
			entry->br_callback_time = 0;
		} else {
			break;
		}
	}
	write_unlock(&bcache_lock);

	while (br_info) {
		struct br_addrs *tmp = br_info->next;
		if (mipv6_send_brr(&br_info->saddr, &br_info->daddr, 0, NULL) < 0)
			DEBUG(DBG_WARNING,
			      "BR send for %x:%x:%x:%x:%x:%x:%x:%x failed",
			      NIPV6ADDR(&br_info->daddr));
		kfree(br_info);
		br_info = tmp;
	}
}

/* 
 * The function that is scheduled to do the callback functions. May be
 * modified e.g to allow Binding Requests, now only calls expire() and
 * schedules a new timer.
 *
 * Important: This function is a 'top-level' function in this module,
 * it is not called from any other function inside the module although
 * it is static. So it must take care of the syncronization, like the
 * other static functions need not.  
 */
static void timer_handler(unsigned long dummy)
{
	expire();
	write_lock(&bcache_lock);
	set_timer();
	write_unlock(&bcache_lock);
}

static void set_timer(void)
{
	struct mipv6_bcache_entry *entry;
	unsigned long callback_time;

	DEBUG_FUNC();

	entry = (struct mipv6_bcache_entry *)
		hashlist_get_first(bcache.entries);
	if (entry != NULL) {
		if (entry->callback_time == EXPIRE_INFINITE) {
			DEBUG(DBG_WARNING, "expire at infinity, "
			      "not setting a new timer");
		} else {
			if (entry->br_callback_time > 0)
				callback_time = entry->br_callback_time;
			else if (entry->callback_time > jiffies)
				callback_time = entry->callback_time;
			else {
				DEBUG(DBG_WARNING, 
				      "bcache timer attempted to schedule"
				      " for a historical jiffies count!");
				callback_time = jiffies + TIMERDELAY;
			}

			DEBUG(DBG_INFO, "setting timer to now");
			mod_timer(&bcache.callback_timer, callback_time);
		}
	} else {
		del_timer(&bcache.callback_timer);
		DEBUG(DBG_INFO, "BC empty, not setting a new timer");
	}
}

/*
 * Generic list handling routine :
 * 
 * On each element on the prefix list, apply the home_addr's 2 low order words
 * to create a full IPv6 address, and call the passed function with this
 * address and the argument.
 * 
 * Assumes EUI64 prefix/ID.
 *
 * Returns :
 * 	on success, the number of elements on the list for which the function
 *		returned success.
 * 	on failure, the negative value of the number of elements on the list
 *		processed before the failure occurred.
 *
 */
int mipv6_execute_list(struct prefix_info *plist, struct in6_addr *home_addr,
		int count, int (*func)(struct in6_addr *, void *), void *arg)
{
	int i = 1;
	struct in6_addr addr;
	DEBUG_FUNC();
	ipv6_addr_set(&addr, 0, 0, home_addr->s6_addr32[2],
			home_addr->s6_addr32[3]);
	while (i <= count) {
		addr.s6_addr32[0] = plist->prefix.s6_addr32[0];
		addr.s6_addr32[1] = plist->prefix.s6_addr32[1];
		if (func(&addr, arg) < 0)
			return -i;
		plist ++;
		i ++;
	}
	return count;
}

int mipv6_get_prefix_entries(struct in6_addr *home_addr,
			struct prefix_info **plist, int single, int ifindex)
{
	int count;

	extern int ipv6_get_prefix_entries(struct prefix_info **plist,
			int ifindex, int plen);
	DEBUG_FUNC();
	if (single || (count = ipv6_get_prefix_entries(plist,
				ifindex, 64)) <= 0) {
		*plist = kmalloc(sizeof(struct prefix_info), GFP_ATOMIC);
		if (*plist) {
			ipv6_addr_copy(&((*plist)->prefix), home_addr);
			(*plist)->prefix_len = 64;
			return 1;
		} else {
			return 0;
		}
	}
	return count;
}

static int plist_cleanup(struct in6_addr *home_addr, void *arg)
{
	struct in6_addr *ha_addr = (struct in6_addr *)arg;
	struct in6_addr_pair hashkey;
	struct mipv6_bcache_entry *entry;

	hashkey.a1 = home_addr;
	hashkey.a2 = ha_addr;

	if ((entry = (struct mipv6_bcache_entry *)
	     hashlist_get(bcache.entries, &hashkey)) != NULL) {
		if (entry->type & HOME_REGISTRATION) {
			del_proxy(home_addr, entry);
		}
		hashlist_delete(bcache.entries, (void *)entry);
		mipv6_bcache_entry_free(entry);
	}
	return 0;
}

/*
 * Interface functions visible to other modules
 */

/**
 * mipv6_bcache_add - add Binding Cache entry
 * @ifindex: interface index
 * @our_addr: own address
 * @home_addr_org: MN's home address
 * @coa: MN's care-of address
 * @lifetime: lifetime for this binding
 * @prefix: prefix length
 * @seq: sequence number
 * @single: single address bit
 * @type: type of entry
 *
 * Adds an entry for this @home_addr_org in the Binding Cache.  If entry
 * already exists, old entry is updated.  @type may be %CACHE_ENTRY or
 * %HOME_REGISTRATION.
 **/
int mipv6_bcache_add(int ifindex,
		     struct in6_addr *our_addr,
		     struct in6_addr *home_addr_org,
		     struct in6_addr *coa,
		     __u32 lifetime,
		     __u8 prefix, __u16 seq, __u8 single, __u8 type)
{
	struct mipv6_bcache_entry *entry;
	int update = 0;
	int create_tunnel = 0;
	unsigned long now = jiffies;
	struct in6_addr_pair hashkey;
	int count, count_org;
	struct prefix_info *plist, *plist_org;
	struct in6_addr home_addr;	/* calculated home address */
	int processed_count = 0;
	int ret = -1;
	int update_pneigh_tbl = 0;

	DEBUG_FUNC();

	if (type != HOME_REGISTRATION) {
		/*
		 * If we are not doing a home registration, don't defend
		 * multiple addresses.
		 */
		single = 1;
		count_org = count = 1;
		if ((plist = kmalloc(sizeof(struct prefix_info), GFP_ATOMIC)) 
		    == NULL)
			return -ENOMEM;

		plist_org = plist;
		ipv6_addr_copy(&plist->prefix, home_addr_org);
		plist->prefix_len = 64;
	} else {
		single = 1;
		if ((count =
		     mipv6_get_prefix_entries(home_addr_org, &plist, single,
					      ifindex)) <= 0) {
			/* Memory failure, return failure */
			DEBUG(DBG_INFO, "could not allocate"
			      " memory for creating bcache entries");
			return -ENOMEM;
		}
		count_org = count;
		plist_org = plist;
	}

	hashkey.a2 = our_addr;

	write_lock(&bcache_lock);

	if (unlikely(bcache.entries == NULL)) {
		ret = -ENOMEM;
		goto err;
	}
process_next_home_address:
	ipv6_addr_set(&home_addr, plist->prefix.s6_addr32[0],
		      plist->prefix.s6_addr32[1],
		      home_addr_org->s6_addr32[2],
		      home_addr_org->s6_addr32[3]);
	hashkey.a1 = &home_addr;
	plist++;

	if ((entry = (struct mipv6_bcache_entry *)
	     hashlist_get(bcache.entries, &hashkey)) != NULL) {
		/* if an entry for this home_addr exists (with smaller
		 * seq than the new seq), update it by removing it
		 * first
		 */
		if (modGT65536(seq, entry->seq)) {
			DEBUG(DBG_INFO, "updating an existing entry");
			update = 1;
			create_tunnel = (type == HOME_REGISTRATION &&
					 (entry->type != HOME_REGISTRATION ||
					  ipv6_addr_cmp(&entry->coa, coa) ||
					  entry->ifindex != ifindex || 
					  entry->prefix != prefix || 
					  entry->single != single));
			mipv6_check_entry(entry, create_tunnel, type, single,
						&update_pneigh_tbl);
		} else {
			DEBUG(DBG_INFO, "smaller seq than existing, not updating");
			goto out;
		}
	} else {
		/* no entry for this home_addr, try to create a new entry */
		DEBUG(DBG_INFO, "creating a new entry");
		entry = mipv6_bcache_get_entry(type);

		update = 0;

		if (entry == NULL) {
			DEBUG(DBG_INFO, "cache full, entry not added");
			goto err;
		}

		create_tunnel = (type == HOME_REGISTRATION);
	}

	ipv6_addr_copy(&(entry->our_addr), our_addr);
	ipv6_addr_copy(&(entry->home_addr), &home_addr);
	ipv6_addr_copy(&(entry->coa), coa);
	entry->ifindex = ifindex;
	entry->prefix = prefix;
	entry->seq = seq;
	entry->type = type;
	entry->last_used = 0;

	hashkey.a1 = &entry->home_addr;
	hashkey.a2 = &entry->our_addr;

	if ((ret = mipv6_create_tnl(type, single, create_tunnel, ifindex,
					prefix, entry, coa,
					our_addr, &home_addr,
					update_pneigh_tbl))) {
		if (ret == 1)
			goto err_plist;
		else goto err_tnl;
	}

	entry->last_br = 0;
	if (lifetime == EXPIRE_INFINITE) {
		entry->callback_time = EXPIRE_INFINITE;
	} else {
		entry->callback_time = now + lifetime * HZ;
		if (entry->type & HOME_REGISTRATION)
			entry->br_callback_time = 0;
		else
			entry->br_callback_time = now +
				(lifetime - BCACHE_BR_SEND_LEAD) * HZ;
	}
	
	if (update) {
		DEBUG(DBG_INFO, "updating entry : %x", entry);
		hashlist_reposition(bcache.entries, (void *)entry, 
				    entry->callback_time);
	} else {
		DEBUG(DBG_INFO, "adding entry: %x", entry);
		if ((hashlist_add(bcache.entries,
				  &hashkey,
				  entry->callback_time, entry)) < 0) {
			
			DEBUG(DBG_ERROR, "Hash add failed");
			goto err_proxy;
		}
	}
	processed_count++;
	if (--count) {
		goto process_next_home_address;
	}
	
	set_timer();
	
out:
	write_unlock(&bcache_lock);
	kfree(plist_org);
	return 0;
	
err_proxy:
	if (create_tunnel || update_pneigh_tbl) {
		bcache_proxy_nd_rem(entry);
	}
err_tnl:
#ifdef CONFIG_IPV6_MOBILITY_HA
	if (create_tunnel) {
		mipv6_del_tnl_to_mn(coa, our_addr, &home_addr);
	}
#endif
err_plist:
	mipv6_execute_list(plist_org, &home_addr, 
			   processed_count,
			   plist_cleanup, our_addr);
	
	if (update) {
		hashlist_delete(bcache.entries, (void *)entry);
	}
	mipv6_bcache_entry_free(entry);
err:
	write_unlock(&bcache_lock);
	kfree(plist_org);
	return ret;
}

static int __mipv6_bcache_delete(struct in6_addr *addr, void *arg)
{
	struct in6_addr_pair hashkey;
	struct mipv6_bcache_entry *entry;

	hashkey.a1 = addr;
	hashkey.a2 = (struct in6_addr *) arg;
	entry = (struct mipv6_bcache_entry *)
	    hashlist_get(bcache.entries, &hashkey);
	if (entry == NULL) {
		DEBUG(DBG_INFO, "No such entry");
		return 0;
	}
	if (entry->type == HOME_REGISTRATION) {
		del_proxy(&entry->home_addr, entry);
	}
	hashlist_delete(bcache.entries, (void *) entry);
	mipv6_bcache_entry_free(entry);
	return 0;
}

/**
 * mipv6_bcache_delete - delete Binding Cache entry
 * @home_addr: MN's home address
 * @our_addr: our address
 * @type: type of entry
 *
 * Deletes an entry associated with @home_addr from Binding Cache.
 * Valid values for @type are %CACHE_ENTRY, %HOME_REGISTRATION and
 * %ANY_ENTRY.  %ANY_ENTRY deletes any type of entry.
 **/
int mipv6_bcache_delete(struct in6_addr *home_addr,
			struct in6_addr *our_addr, __u8 type)
{
	struct mipv6_bcache_entry *entry;
	struct in6_addr_pair hashkey;
	int count;
	struct prefix_info *plist;
	int err = 0;

	DEBUG_FUNC();

	if (home_addr == NULL || our_addr == NULL || !is_valid_type(type)) {
		DEBUG(DBG_INFO, "error in arguments");
		return -EINVAL;
	}

	hashkey.a1 = home_addr;
	hashkey.a2 = our_addr;

	write_lock(&bcache_lock);

	if (unlikely(bcache.entries == NULL) ||
	    (entry = (struct mipv6_bcache_entry *)
	     hashlist_get(bcache.entries, &hashkey)) == NULL ||
	    !(entry->type & type)) {
		DEBUG(DBG_INFO, "No matching entry found");
		err = -ENOENT;
		goto out;
	}
	if (type != HOME_REGISTRATION) {
		count = 1;
		if ((plist = kmalloc(sizeof(struct prefix_info), GFP_ATOMIC)) == NULL) {
			err = -ENOMEM;
			goto out;
		}
		ipv6_addr_copy(&plist->prefix, home_addr);
		plist->prefix_len = 64;
	} else {
		if ((count = mipv6_get_prefix_entries(home_addr, &plist,
						      entry->single,
						      entry->ifindex)) <= 0) {
			err = -ENOMEM;
			goto out;
		}
	}

	mipv6_execute_list(plist, home_addr, count, __mipv6_bcache_delete,
			   our_addr);
	kfree(plist);
	
	set_timer();
out:
	write_unlock(&bcache_lock);
	return err;
}

/**
 * mipv6_bcache_exists - check if entry exists
 * @home_addr: home address to check
 * @our_addr: our address
 *
 * Determines if a binding exists for @home_addr.  Returns type of the
 * entry or negative if entry does not exist.
 **/
int mipv6_bcache_exists(struct in6_addr *home_addr,
			struct in6_addr *our_addr)
{
	struct mipv6_bcache_entry *entry;
	struct in6_addr_pair hashkey;
	int type = -ENOENT;

	DEBUG_FUNC();

	if (home_addr == NULL || our_addr == NULL)
		return -EINVAL;

	hashkey.a1 = home_addr;
	hashkey.a2 = our_addr;

	read_lock(&bcache_lock);
	if (likely(bcache.entries != NULL) &&
	    (entry = (struct mipv6_bcache_entry *)
	     hashlist_get(bcache.entries, &hashkey)) != NULL) {
		type = entry->type;
	}
	read_unlock(&bcache_lock);

	return type;
}

/**
 * mipv6_bcache_get - get entry from Binding Cache
 * @home_addr: home address to search
 * @our_addr: our address
 * @entry: pointer to buffer
 *
 * Gets a copy of Binding Cache entry for @home_addr.  Entry's
 * @last_used field is updated.  If entry exists entry is copied to
 * @entry and zero is returned.  Otherwise returns negative.
 **/
int mipv6_bcache_get(struct in6_addr *home_addr,
		     struct in6_addr *our_addr,
		     struct mipv6_bcache_entry *entry)
{
	struct mipv6_bcache_entry *entry2;
	struct in6_addr_pair hashkey;
	int ret = -1;

	DEBUG_FUNC();

	if (home_addr == NULL || our_addr == NULL || entry == NULL)
		return -1;

	hashkey.a1 = home_addr;
	hashkey.a2 = our_addr;

	read_lock_bh(&bcache_lock);

	entry2 = (struct mipv6_bcache_entry *)
	    hashlist_get(bcache.entries, &hashkey);
	if (entry2 != NULL) {
		entry2->last_used = jiffies;
		memcpy(entry, entry2, sizeof(struct mipv6_bcache_entry));
		ret = 0;
	}
	read_unlock_bh(&bcache_lock);
	return ret;
}

int mipv6_bcache_iterate(hashlist_iterator_t func, void *args)
{
	int ret;

	read_lock_bh(&bcache_lock);
	ret = hashlist_iterate(bcache.entries, args, func);
	read_unlock_bh(&bcache_lock);

	return ret;
}

/*
 * Proc-filesystem functions
 */

#define BC_INFO_LEN 90

struct procinfo_iterator_args {
	char *buffer;
	int offset;
	int length;
	int skip;
	int len;
};

static int procinfo_iterator(void *data, void *args, unsigned long *pref)
{
	struct procinfo_iterator_args *arg =
	    (struct procinfo_iterator_args *) args;
	struct mipv6_bcache_entry *entry =
	    (struct mipv6_bcache_entry *) data;

	DEBUG_FUNC();

	if (entry == NULL)
		return ITERATOR_ERR;

	if (arg->skip < arg->offset / BC_INFO_LEN) {
		arg->skip++;
		return ITERATOR_CONT;
	}

	if (arg->len >= arg->length)
		return ITERATOR_CONT;

	arg->len += sprintf(arg->buffer + arg->len,
			    "h=%04x%04x%04x%04x%04x%04x%04x%04x "
			    "c=%04x%04x%04x%04x%04x%04x%04x%04x "
			    "(e=%010lu,t=%02d)\n",
			    NIPV6ADDR(&entry->home_addr),
			    NIPV6ADDR(&entry->coa),
			    ((entry->callback_time) - jiffies) / HZ,
			    (int) entry->type);

	return ITERATOR_CONT;
}

 /*
  * Callback function for proc filesystem.
  */
static int bcache_proc_info(char *buffer, char **start, off_t offset,
			    int length)
{
	struct procinfo_iterator_args args;

	args.buffer = buffer;
	args.offset = offset;
	args.length = length;
	args.skip = 0;
	args.len = 0;

	read_lock_bh(&bcache_lock);
	hashlist_iterate(bcache.entries, &args, procinfo_iterator);
	read_unlock_bh(&bcache_lock);

	*start = buffer;
	if (offset)
		*start += offset % BC_INFO_LEN;

	args.len -= offset % BC_INFO_LEN;

	if (args.len > length)
		args.len = length;
	if (args.len < 0)
		args.len = 0;

	return args.len;
}

/* Router bit leftovers.  See if we need this later on with mobile
 * networks.  Prototype removed. */

static int bcache_compare(void *data, void *hashkey)
{
	struct in6_addr_pair *p = (struct in6_addr_pair *) hashkey;
	struct mipv6_bcache_entry *e = (struct mipv6_bcache_entry *) data;

	if (ipv6_addr_cmp(&e->home_addr, p->a1) == 0
	    && ipv6_addr_cmp(&e->our_addr, p->a2) == 0)
		return 0;
	else
		return -1;
}

static __u32 bcache_hash(void *hashkey)
{
	struct in6_addr_pair *p = (struct in6_addr_pair *) hashkey;

	return p->a1->s6_addr32[0] ^ p->a1->s6_addr32[1] ^
		p->a2->s6_addr32[2] ^ p->a2->s6_addr32[3];
}

static void *bcache_hashkey(void *data)
{
	struct in6_addr_pair *p;
	struct mipv6_bcache_entry *e = (struct mipv6_bcache_entry *) data;

	if ((p = kmalloc(sizeof(struct in6_addr_pair), GFP_KERNEL)) == NULL)
		return NULL;

	p->a1 = &e->home_addr;
	p->a2 = &e->our_addr;

	return p;
}

/* 
 * Initialization and shutdown functions
 */

int __init mipv6_bcache_init(__u32 size)
{
	if (size < 1) {
		DEBUG(DBG_ERROR, "Binding cache size must be at least 1");
		return -EINVAL;
	}
	bcache.entries = hashlist_create(MIPV6_BCACHE_HASHSIZE, size,
					 sizeof(struct mipv6_bcache_entry),
					 "mip6_bcache", NULL, NULL,
					 bcache_compare, bcache_hash,
					 bcache_hashkey);

	if (bcache.entries == NULL) {
		DEBUG(DBG_ERROR, "Failed to initialize hashlist");
		return -ENOMEM;
	}

	init_timer(&bcache.callback_timer);
	bcache.callback_timer.data = 0;
	bcache.callback_timer.function = timer_handler;
	bcache.size = size;

	proc_net_create("mip6_bcache", 0, bcache_proc_info);

	DEBUG(DBG_INFO, "Binding cache initialized");
	return 0;
}

int __exit mipv6_bcache_exit(void)
{
	struct mipv6_bcache_entry *entry;
	struct hashlist *entries;

	DEBUG_FUNC();

	proc_net_remove("mip6_bcache");

	write_lock_bh(&bcache_lock);
	DEBUG(DBG_INFO, "Stopping the bcache timer");
	del_timer(&bcache.callback_timer);

	while ((entry = (struct mipv6_bcache_entry *)
		hashlist_get_first(bcache.entries)) != NULL) {
		
		/*      hashlist_delete_first(bcache.entries); */
		if (entry->type == HOME_REGISTRATION) {
			del_proxy(&entry->home_addr, entry);
		}
		hashlist_delete(bcache.entries, (void *) entry);
		mipv6_bcache_entry_free(entry);
	}
	entries = bcache.entries;
	bcache.entries = NULL;
	write_unlock_bh(&bcache_lock);

	hashlist_destroy(entries);
	return 0;
}
