/*
 * Interface handling module
 *
 * Authors: Michael Bellion and Thomas Heinz
 * (c) 2002-2003 by the hipac core team <nf@hipac.org>:
 *      +-----------------------+----------------------+
 *      |   Michael Bellion     |     Thomas Heinz     |
 *      | <mbellion@hipac.org>  |  <creatix@hipac.org> |
 *      +-----------------------+----------------------+
 * Licenced under the GNU General Public Licence, version >= 2.
 */

#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/netdevice.h>
#include "nfhp_com.h"
#include "nfhp_mod.h"
#include "nfhp_dev.h"
#include "hipac.h"

#define MIN_INDEX_START (BIT_INIFACE == BIT_U32 ? INT_MAX : 65535)
#define MAX_INDEX_START 1

struct dev_list
{
	struct list_head head;
	char ifname[IFNAMSIZ];
	int ifindex;
};

/* list of used but non-existing device names and their mapping to
   an unused ifindex */
static struct dev_list nex_dev = {LIST_HEAD_INIT(nex_dev.head), {0}, 0};
static int min_index = MIN_INDEX_START;
static rwlock_t nex_dev_lock = RW_LOCK_UNLOCKED;

/* list existing device names and their corresponding interface indexes */
static struct dev_list xdev = {LIST_HEAD_INIT(xdev.head), {0}, 0};
static int max_index = MAX_INDEX_START;
static rwlock_t xdev_lock = RW_LOCK_UNLOCKED;



/*
 * helper function
 */

static inline int
dev_add_tail(const char *ifname, int ifindex, struct dev_list *dl)
{
	struct dev_list *d;

	if (ifindex <= 0 || ifindex >= hipac_maxkey(BIT_INIFACE)) {
		printk(KERN_ERR "%s: invalid ifindex\n", __FUNCTION__);
		return -EINVAL;
	}
	d = kmalloc(sizeof(*d), GFP_KERNEL);
	if (d == NULL) {
		return -ENOMEM;
	}
	strncpy(d->ifname, ifname, sizeof(d->ifname));
	d->ifindex = ifindex;
	list_add_tail(&d->head, &dl->head);
	return ifindex;
}


/*
 * functions for manipulating existing device list
 */

static int
xdev_add_tail(const char *ifname, int ifindex)
{
	int ifind;

	if (ifindex >= min_index) {
		printk(KERN_WARNING "%s: ifindex already in use\n",
		       __FUNCTION__);
		return -EINVAL;
	}
	ifind = dev_add_tail(ifname, ifindex, &xdev);
	if (ifind > max_index) {
		max_index = ifind;
	}
	return ifind;
}

static void
xdev_del_entry(struct dev_list *e)
{
	int recomp = e->ifindex == max_index;
	struct list_head *lh;
	struct dev_list *d;

	list_del(&e->head);
	kfree(e);
	if (recomp) {
		max_index = MAX_INDEX_START;
		list_for_each(lh, &xdev.head) {
			d = list_entry(lh, struct dev_list, head);
			if (d->ifindex > max_index) {
				max_index = d->ifindex;
			}
		}
	}
}

static void
xdev_del_tail(void)
{
	struct dev_list *d;
	
	if (list_empty(&xdev.head)) {
		return;
	}
	d = list_entry(xdev.head.prev, struct dev_list, head);
	xdev_del_entry(d);
}

static struct dev_list *
xdev_get_by_name(const char *ifname)
{
	struct list_head *lh;
	struct dev_list *d;

	list_for_each(lh, &xdev.head) {
		d = list_entry(lh, struct dev_list, head);
		if (strncmp(d->ifname, ifname, sizeof(d->ifname)) == 0) {
			return d;
		}
	}
	return NULL;
}

static struct dev_list *
xdev_get_by_index(int ifindex)
{
	struct list_head *lh;
	struct dev_list *d;

	list_for_each (lh, &xdev.head) {
		d = list_entry(lh, struct dev_list, head);
		if (d->ifindex == ifindex) {
			return d;
		}
	}
	return NULL;
}


/*
 * functions for manipulating non-existing device list
 */

static int
nex_dev_add_tail(const char *ifname, int ifindex)
{
	int ifind;

	if (ifindex > 0 && ifindex <= max_index) {
		printk(KERN_WARNING "%s: ifindex already in use\n",
		       __FUNCTION__);
		return -EINVAL;
	}
	ifind = dev_add_tail(ifname, ifindex, &nex_dev);
	if (ifind > 0 && ifind < min_index) {
		min_index = ifind;
	}
	return ifind;
}

static void
nex_dev_del_entry(struct dev_list *e)
{
	int recomp = e->ifindex == min_index;
	struct list_head *lh;
	struct dev_list *d;

	list_del(&e->head);
	kfree(e);
	if (recomp) {
		min_index = MIN_INDEX_START;
		list_for_each(lh, &nex_dev.head) {
			d = list_entry(lh, struct dev_list, head);
			if (d->ifindex < min_index) {
				min_index = d->ifindex;
			}
		}
	}
}

static void
nex_dev_del_tail(void)
{
	struct dev_list *d;

	if (list_empty(&nex_dev.head)) {
		return;
	}
	d = list_entry(nex_dev.head.prev, struct dev_list, head);
	nex_dev_del_entry(d);
}

static struct dev_list *
nex_dev_get_by_name(const char *ifname)
{
	struct list_head *lh;
	struct dev_list *d;

	list_for_each (lh, &nex_dev.head) {
		d = list_entry(lh, struct dev_list, head);
		if (strncmp(d->ifname, ifname, sizeof(d->ifname)) == 0) {
			return d;
		}
	}
	return NULL;
}



/*
 * veto function and interface functions
 */

/* we need to be aware of interface changes because we are matching against
   ifindex which might change although the device name remains the same
   (e.g. after reload of the device driver) */
static int
dev_veto_event(unsigned long event, struct net_device *dev)
{
	int stat = 0;
	u8 dimid[] = {DIMID_INIFACE <= DIMID_OUTIFACE ?
		      DIMID_INIFACE : DIMID_OUTIFACE,
		      DIMID_INIFACE <= DIMID_OUTIFACE ?
		      DIMID_OUTIFACE : DIMID_INIFACE};
	int len = 1;
	int ifi_org[2], ifi_new[2];
	struct dev_list *d1, *d2;

	switch (event) {
	    case NETDEV_REGISTER:
		    write_lock(&nex_dev_lock);
		    write_lock(&xdev_lock);
		    ifi_new[0] = xdev_add_tail(dev->name, dev->ifindex);
		    if (ifi_new[0] < 0) {
			    stat = ifi_new[0];
			    goto veto_exit;
		    }
		    d1 = nex_dev_get_by_name(dev->name);
		    if (d1 != NULL) {
			    ifi_org[0] = d1->ifindex;
			    HIPAC_LOCK;
			    if (hipac_substitute(dimid, ifi_org,
						 ifi_new, len) < 0) {
				    printk(KERN_WARNING "%s: unable to fix "
					   "ruleset (ruleset is still "
					   "consistent)\n", __FUNCTION__);
				    xdev_del_tail();
				    stat = -1;
			    }
			    HIPAC_UNLOCK;
			    nex_dev_del_entry(d1);
		    }
		    break;

	    case NETDEV_UNREGISTER:
		    write_lock(&nex_dev_lock);
		    write_lock(&xdev_lock);
		    ifi_new[0] = nex_dev_add_tail(dev->name, min_index - 1);
		    if (ifi_new[0] < 0) {
			    stat = ifi_new[0];
			    goto veto_exit;
		    }
		    d1 = xdev_get_by_name(dev->name);
		    if (d1 != NULL) {
			    ifi_org[0] = d1->ifindex;
			    HIPAC_LOCK;
			    if (hipac_substitute(dimid, ifi_org,
						 ifi_new, len) < 0) {
				    printk(KERN_WARNING "%s: unable to fix "
					   "ruleset (ruleset is still "
					   "consistent)\n", __FUNCTION__);
				    nex_dev_del_tail();
				    stat = -1;
			    }
			    HIPAC_UNLOCK;
			    xdev_del_entry(d1);
		    } else {
			    /* this should never happen */
			    printk(KERN_ERR "%s: unable to find existing "
				   "device %s in xdev\n", __FUNCTION__,
				   dev->name);
			    goto veto_exit;
		    }
		    break;

	    case NETDEV_CHANGENAME:
		    write_lock(&nex_dev_lock);
		    write_lock(&xdev_lock);

		    /* get old interface name and move it into nex_dev */
		    ifi_org[0] = dev->ifindex;
		    d1 = xdev_get_by_index(ifi_org[0]);
		    if (d1 != NULL) {
			    ifi_new[0] = nex_dev_add_tail(d1->ifname,
							  min_index - 1);
			    if (ifi_new[0] < 0) {
				    stat = ifi_new[0];
				    goto veto_exit;
			    }
		    } else {
			    printk(KERN_ERR "%s: unable to find existing "
				   "device in xdev\n", __FUNCTION__);
			    stat = -1;
			    goto veto_exit;
		    }

		    /* insert new interface name into xdev; if the new name 
		       is contained in nex_dev it must be deleted there */
		    ifi_new[1] = xdev_add_tail(dev->name, dev->ifindex);
		    if (ifi_new[1] < 0) {
			    nex_dev_del_tail();
			    stat = ifi_new[1];
			    goto veto_exit;
		    }
		    d2 = nex_dev_get_by_name(dev->name);
		    if (d2 != NULL) {
			    len++;
			    ifi_org[1] = d2->ifindex;
		    }
		    
		    HIPAC_LOCK;
		    if (hipac_substitute(dimid, ifi_org, ifi_new, len) < 0) {
			    printk(KERN_WARNING "%s: unable to fix ruleset "
				   "(ruleset is still consistent)\n",
				   __FUNCTION__);
			    nex_dev_del_tail();
			    xdev_del_tail();
			    stat = -1;
			    goto veto_exit;
		    }
		    HIPAC_UNLOCK;
		    xdev_del_entry(d1);
		    if (d2 != NULL) {
			    nex_dev_del_entry(d2);
		    }
		    break;
	    
	    default:
		    return 0;
	}

 veto_exit:
	write_unlock(&nex_dev_lock);
	write_unlock(&xdev_lock);
	return stat;
}

int
hpdev_get_name(int ifindex, char ifname[])
{
	const struct dev_list *dl[] = {&xdev, &nex_dev};
	rwlock_t *lock[] = {&xdev_lock, &nex_dev_lock};
	struct list_head *lh;
	struct dev_list *d;
	int i;

	for (i = 0; i < 2; i++) {
		read_lock(lock[i]);
		list_for_each(lh, &dl[i]->head) {
			d = list_entry(lh, struct dev_list, head);
			if (d->ifindex == ifindex) {
				strncpy(ifname, d->ifname, sizeof(d->ifname));
				read_unlock(lock[i]);
				return 0;
			}
		}
		read_unlock(lock[i]);
	}
	return -1;
}

int
hpdev_get_index(const char *ifname)
{
	const struct dev_list *dl[] = {&xdev, &nex_dev};
	rwlock_t *lock[] = {&xdev_lock, &nex_dev_lock};
	struct list_head *lh;
	struct dev_list *d;
	int i, ifindex;

	/* search xdev and nex_dev */
	for (i = 0; i < 2; i++) {
		read_lock(lock[i]);
		list_for_each(lh, &dl[i]->head) {
			d = list_entry(lh, struct dev_list, head);
			if (strncmp(d->ifname, ifname,
				    sizeof(d->ifname)) == 0) {
				ifindex = d->ifindex;
				read_unlock(lock[i]);
				return ifindex;
			}
		}
		read_unlock(lock[i]);
	}

	if (min_index <= max_index + 1) {
		return NFHE_INDEX;
	}

	/* device does not exist => associate it with a non-existing index */
	write_lock(&nex_dev_lock);
	ifindex = nex_dev_add_tail(ifname, min_index - 1);
	write_unlock(&nex_dev_lock);
	if (ifindex < 0) {
		return ERRNO_TO_NFHE(-ifindex);
	}
	return ifindex;
}

int
hpdev_init(void)
{
	struct net_device *dev;
	struct list_head *lh;
	struct dev_list *d;	
	int stat;

	/* the interface list cannot change until we release the read lock;
	   since we register the veto function during the lock we never
	   miss a relevant event; note that as long as the lock is taken
	   dev_veto_event cannot be called since the callback mechanism 
	   is embedded in a write lock of dev_base_lock */
	read_lock(&dev_base_lock);

	/* register veto function */
	stat = register_netdevice_veto(dev_veto_event);
	if (stat < 0) {
		goto exit_init;
	}

	/* init xdev list */
	write_lock(&xdev_lock);
	for (dev = dev_base; dev != NULL; dev = dev->next) {
		stat = xdev_add_tail(dev->name, dev->ifindex);
		if (stat < 0) {
			for (lh = xdev.head.next; lh != &xdev.head;) {
				d = list_entry(lh, struct dev_list, head);
				lh = lh->next;
				list_del(lh->prev);
				kfree(d);
			}
			min_index = MIN_INDEX_START;
			max_index = MAX_INDEX_START;
			write_unlock(&xdev_lock);
			if (unregister_netdevice_veto(dev_veto_event) < 0) {
				/* this should never happen */
				printk(KERN_ERR "%s: unregister_netdevice_"
				       "veto failed\n", __FUNCTION__);
			}
			goto exit_init;
		}
	}
	write_unlock(&xdev_lock);
	stat = 0;

 exit_init:
	read_unlock(&dev_base_lock);
	return stat;
}

void
hpdev_exit(void)
{
	const struct dev_list *dl[] = {&xdev, &nex_dev};
	rwlock_t *lock[] = {&xdev_lock, &nex_dev_lock};
	struct list_head *lh;
	struct dev_list *d;	
	int i;

	if (unregister_netdevice_veto(dev_veto_event) < 0) {
		/* this should never happen */
		printk(KERN_ERR "%s: unregister_netdevice_veto "
		       "failed\n", __FUNCTION__);
	}
	for (i = 0; i < 2; i++) {
		write_lock(lock[i]);
		for (lh = dl[i]->head.next; lh != &dl[i]->head;) {
			d = list_entry(lh, struct dev_list, head);
			lh = lh->next;
			list_del(lh->prev);
			kfree(d);
		}
		min_index = MIN_INDEX_START;
		max_index = MAX_INDEX_START;
		write_unlock(lock[i]);
	}
}
