/*
 *  linux/fs/supermount/namei.c
 *
 *  Original version:
 *      Copyright (C) 1995
 *      Stephen Tweedie (sct@dcs.ed.ac.uk)
 *
 *      from
 *
 *      linux/fs/minix/namei.c
 *      Copyright (C) 1991, 1992  Linus Torvalds
 *
 *      and
 *
 *      linux/fs/ext2/namei.c
 *      Copyright (C) 1992, 1993, 1994, 1995  Remy Card
 *
 *  Rewriten for kernel 2.2 & 2.4. (C) 1999, 2000 Alexis Mikhailov
 *                                    (alexis@abc.cap.ru)
 *  Rewritten for kernel 2.4.21 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *
 *  $Id: namei.c,v 1.23 2003/07/13 19:23:31 bor Exp $
 */

#define S_DBG_TRACE_CURRENT S_DBG_TRACE_NAMEI
#include "supermount.h"

/**
 * Attach super dentry to sub dentry 
 *
 * @dentry:	super dentry that is being created
 * @subd:	subfs dentry that just has been found
 *
 * It checks whether subfs is still valid using @dentry->d_parent;
 * new supermount_dentry_info is created and set to point to @subd.
 *
 * It is possible that @subd actually has different name (ntfs, vfat or
 * in general any case-insensitive filesystems). Search @dentry->d_parent
 * for a child matching == @subd->d_name. If found, discard @dentry and child
 * (after some validity checks) is returned. Check that child actually points
 * to @subd
 *
 * d_lookup relies on the fact that hash is properly initialized in
 * @subd->d_name and that superfs is using the same compare method as subfs.
 *
 * About ref counting. @subd is dput in supermount_lookup. I.e. in case of
 * error or if we find out it is already connected to superfs the excessive
 * counter gets decremented. @dentry is finally dput in caller of ->lookup
 * if ->lookup returns something != 0.
 */

static struct dentry *
prepare_dentry(struct dentry *dentry, struct dentry *subd)
{
	struct super_block *sb = dentry->d_sb;
	struct dentry *rc;

	ENTER(sb, "dentry=%s subd=%s", dentry->d_name.name, subd->d_name.name);

	subfs_lock(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, !subd);
	SUPERMOUNT_BUG_LOCKED_ON(sb, dentry->d_fsdata);

	rc = ERR_PTR(-ENOMEDIUM);
	if (!subfs_is_mounted(sb))
		goto out;

	rc = ERR_PTR(-ESTALE);
	if (is_dentry_obsolete(dentry->d_parent))
		goto out;

	rc = d_lookup(dentry->d_parent, &subd->d_name);
	if (IS_ERR(rc))
		goto out;

	if (rc) {
		SUPERMOUNT_BUG_LOCKED_ON(sb, !rc->d_fsdata);
		SUPERMOUNT_BUG_LOCKED_ON(sb,
		((struct supermount_dentry_info *)rc->d_fsdata)->dentry != subd);
	} else {
		/*
		 * this is theoretically possible. We cannot garantee full
		 * coherency between subfs and superfs cache; i.e. entry
		 * may have been left in one cache but removed from another
		 */
		rc = ERR_PTR(-ENOMEM);
		if (init_dentry_info(dentry))
			goto out;
		attach_subfs_dentry(dentry, subd);
		d_add(dentry, 0);
		rc = 0;
	}

out:
	subfs_unlock(sb);

	LEAVE(sb, "dentry=%p subd=%p rc=%p", dentry, subd, rc);

	return rc;
	/*
	 * subdent is implicitly freed on return if we skip dget here
	 */
}


/**
 * Attach superfs inode to subfs inode 
 *
 * @dentry:	superfs dentry
 * @subd:	subfs dentry
 *
 * This is expected to be called only in cointext that requires
 * negative dentry.
 *
 * FIXME
 * Holding sbi->sem during iget4 creates deadlock with write_inode -
 * write_inode sets I_LOCK abd calls supermount_write_inode at the
 * same moment as iget4 sleeps waiting for I_LOCK to be cleared. So
 * it first acquires inode and then checks if subfs is still valid.
 */

static int
prepare_inode(struct dentry *dentry, struct dentry *subd)
{
	struct super_block *sb = dentry->d_sb;
	struct inode *inode = dentry->d_inode;
	struct inode *subi = subd->d_inode;
	int rc;

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	SUPERMOUNT_BUG_ON(inode);
	SUPERMOUNT_BUG_ON(!subi);

	rc = -ENOMEM;
	inode = iget(sb, subi->i_ino);
	if (!inode)
		goto out;

	/* inode is marked bad in read_inode if sii allocation failed */
	if (is_bad_inode(inode)) {
		iput(inode);
		goto out;
	}

	rc = 0;

	subfs_lock(sb);
	if (!subfs_is_mounted(sb))
		rc = -ENOMEDIUM;
	else if (is_dentry_obsolete(dentry))
		rc = -ESTALE;
	else {
		attach_subfs_inode(inode, subi);
		d_instantiate(dentry, inode);
	}
	subfs_unlock(sb);

	if (rc)
		iput(inode);
out:
	LEAVE(sb, "dentry=%s inode=%p rc=%d", dentry->d_name.name, inode, rc);

	return rc;
}

struct inode *
get_subfs_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct inode *err;
	struct supermount_inode_info *sii = supermount_i(inode);

	ENTER(sb, "inode=%p", inode);

	subfs_lock(sb);

	err = ERR_PTR(-ENOMEDIUM);
	if (!subfs_is_mounted(sb))
		goto out;

	err = ERR_PTR(-ESTALE);
	if (is_inode_obsolete(inode))
		goto out;

	err = igrab(sii->inode);
	SUPERMOUNT_BUG_LOCKED_ON(sb, !err);

out:
	subfs_unlock(sb);

	LEAVE(sb, "inode=%p subi=%p", inode, err);

	return err;
}

/* inode methods */

static int
supermount_create(struct inode *dir, struct dentry *dentry, int mode)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EACCES;
	if (!subdir->i_op || !subdir->i_op->create)
		goto put_subdir;
	
	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_subdir;

	rc = subfs_get_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = subdir->i_op->create(subdir, subdent, mode);
	subfs_put_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = prepare_inode(dentry, subdent);
	if (rc)
		goto put_subdent;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

put_subdent:
	dput(subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%d", dir, dentry->d_name.name, rc);

	return rc;
}

/**
 * Search directory for a matching name
 *
 * @dir:	directory that is being searched
 * @dentry:	name to search for (in dentry->d_name)
 *
 * This is (currently :) the only method where we do not call subfs
 * directly. The reason is coherency between subfs and superfs dentry
 * caches. It is impossible to ensure it without modifying the very
 * guts of fs/dcache.c; so we check cache before doing actual lookup.
 * lookup_one_len just avoids duplicating of code.
 *
 * Supermount is in exclusive control of subfs so it is garanteed that
 * dentry cannot magically appear in the middle of lookup.
 *
 * There are filesystems that support multiple forms of file name, like
 * case-insensitive or short-long names on NTFS. In this case cache lookup
 * fails but filesystem may return dentry for different name. In this case
 * we check if dentry with matching name exists in parent and reuse it.
 */
static struct dentry *
supermount_lookup(struct inode *dir, struct dentry *dentry)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *rc, *subdent, *subparent;
	struct inode *subdir;
	struct vfsmount *mnt;
	int ret;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = (struct dentry *)mnt;
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = (struct dentry *)subdir;
	if (IS_ERR(subdir))
		goto go_offline;

	subparent = get_subfs_dentry(dentry->d_parent);
	rc = subparent;
	if (IS_ERR(subparent))
		goto put_subdir;

	ret = subfs_get_access(dir, 0);
	rc = ERR_PTR(ret);
	if (ret)
		goto put_subparent;

	SUPERMOUNT_BUG_ON(subparent->d_inode != subdir);

	subdent = lookup_one_len(dentry->d_name.name, subparent,
				 dentry->d_name.len);
	subfs_put_access(dir, 0);
	rc = subdent;
	if (IS_ERR(rc))
		goto put_subparent;

	rc = prepare_dentry(dentry, subdent);
	if (IS_ERR(rc))
		goto put_subdent;

	if (!rc && subdent->d_inode) {
		ret = prepare_inode(dentry, subdent);
		if (ret < 0)
			rc = ERR_PTR(ret);
	}

put_subdent:
	dput(subdent);
put_subparent:
	dput(subparent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%p", dir, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_link(struct dentry *old_dentry, struct inode *dir,
		struct dentry *new_dentry)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *old_subdent, *new_subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "from=%s dir=%p to=%s", old_dentry->d_name.name, dir, new_dentry->d_name.name);

	SUPERMOUNT_BUG_ON(new_dentry->d_inode);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EPERM;
	if (!subdir->i_op || !subdir->i_op->link)
		goto put_subdir;

	old_subdent = get_subfs_dentry(old_dentry);
	rc = PTR_ERR(old_subdent);
	if (IS_ERR(old_subdent))
		goto put_subdir;

	new_subdent = get_subfs_dentry(new_dentry);
	rc = PTR_ERR(new_subdent);
	if (IS_ERR(new_subdent))
		goto put_old_subdent;

	rc = subfs_get_access(dir, 1);
	if (rc)
		goto put_new_subdent;

	rc = subdir->i_op->link(old_subdent, subdir, new_subdent);
	subfs_put_access(dir, 1);
	if (rc)
		goto put_new_subdent;

	rc = prepare_inode(new_dentry, new_subdent);
	if (rc)
		goto put_new_subdent;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

put_new_subdent:
	dput(new_subdent);
put_old_subdent:
	dput(old_subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "from=%s dir=%p to=%s rc=%d", old_dentry->d_name.name, dir, new_dentry->d_name.name, rc);

	return rc;
}

static int
supermount_unlink(struct inode *dir, struct dentry *dentry)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EPERM;
	if (!subdir->i_op || !subdir->i_op->unlink)
		goto put_subdir;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_subdir;

	/*
	 * below is not a typo. We have to mark _deleted_ inode
	 * for possible later delete in clear_inode
	 */
	rc = subfs_get_access(dentry->d_inode, 1);
	if (rc)
		goto put_subdent;

	rc = subdir->i_op->unlink(subdir, subdent);
	if (rc)
		goto put_write_access;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

	dentry->d_inode->i_nlink = subdent->d_inode->i_nlink;
	dentry->d_inode->i_ctime = subdent->d_inode->i_ctime;

put_write_access:
	/*
	 * we can't put write access if there are pending deletes
	 * so we leave it on and let it be put in clear_inode for
	 * i_nlink == 0
	 *
	 * While i_nlink is believed to be atomic here i_count not,
	 * so we cannot check && i_count == 0. It is expected that
	 * deleted files are kept open only rarely.
	 */
	if (dentry->d_inode->i_nlink)
		subfs_put_access(dentry->d_inode, 1);
put_subdent:
	dput(subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%d", dir, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_symlink(struct inode *dir, struct dentry *dentry,
		   const char *symname)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EPERM;
	if (!subdir->i_op || !subdir->i_op->symlink)
		goto put_subdir;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_subdir;

	rc = subfs_get_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = subdir->i_op->symlink(subdir, subdent, symname);
	subfs_put_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = prepare_inode(dentry, subdent);
	if (rc)
		goto put_subdent;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

put_subdent:
	dput(subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%d", dir, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EPERM;
	if (!subdir->i_op || !subdir->i_op->mkdir)
		goto put_subdir;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_subdir;

	rc = subfs_get_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = subdir->i_op->mkdir(subdir, subdent, mode);
	subfs_put_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = prepare_inode(dentry, subdent);
	if (rc)
		goto put_subdent;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

put_subdent:
	dput(subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%d", dir, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_rmdir(struct inode *dir, struct dentry *dentry)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EPERM;
	if (!subdir->i_op || !subdir->i_op->rmdir)
		goto put_subdir;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_subdir;

	/* cf. supermount_unlink */
	rc = subfs_get_access(dentry->d_inode, 1);
	if (rc)
		goto put_subdent;

	rc = subdir->i_op->rmdir(subdir, subdent);
	if (rc)
		goto put_write_access;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

	/* hmm ... hard links to directories are not allowed, are they? */
	dentry->d_inode->i_nlink = subdent->d_inode->i_nlink;
	dentry->d_inode->i_ctime = subdent->d_inode->i_ctime;

put_write_access:
	/* cf. supermount_unlink */
	if (dentry->d_inode->i_nlink)
		subfs_put_access(dentry->d_inode, 1);
put_subdent:
	dput(subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%d", dir, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_mknod(struct inode *dir, struct dentry *dentry, int
		 mode, int dev)
{
	struct super_block *sb = dir->i_sb;
	struct dentry *subdent;
	struct inode *subdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dir=%p dentry=%s", dir, dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdir = get_subfs_inode(dir);
	rc = PTR_ERR(subdir);
	if (IS_ERR(subdir))
		goto go_offline;

	rc = -EPERM;
	if (!subdir->i_op || !subdir->i_op->mknod)
		goto put_subdir;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_subdir;

	rc = subfs_get_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = subdir->i_op->mknod(subdir, subdent, mode, dev);
	subfs_put_access(dir, 1);
	if (rc)
		goto put_subdent;

	rc = prepare_inode(dentry, subdent);
	if (rc)
		goto put_subdent;

	dir->i_mtime = subdir->i_mtime;
	dir->i_ctime = subdir->i_ctime;
	dir->i_nlink = subdir->i_nlink;
	dir->i_size = subdir->i_size;
	dir->i_blocks = subdir->i_blocks;

put_subdent:
	dput(subdent);
put_subdir:
	iput(subdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dir=%p dentry=%s rc=%d", dir, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_rename(struct inode *olddir, struct dentry *olddentry,
		  struct inode *newdir, struct dentry *newdentry)
{
	struct super_block *sb = olddir->i_sb;
	struct dentry *oldsubdent, *newsubdent;
	struct inode *oldsubdir, *newsubdir;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "olddir=%p olddentry=%s newdir=%p newdentry=%s", olddir, olddentry->d_name.name, newdir, newdentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	oldsubdir = get_subfs_inode(olddir);
	rc = PTR_ERR(oldsubdir);
	if (IS_ERR(oldsubdir))
		goto go_offline;

	rc = -EPERM;
	if (!oldsubdir->i_op || !oldsubdir->i_op->rename)
		goto put_old_subdir;

	oldsubdent = get_subfs_dentry(olddentry);
	rc = PTR_ERR(oldsubdent);
	if (IS_ERR(oldsubdent))
		goto put_old_subdir;

	newsubdir = get_subfs_inode(newdir);
	rc = PTR_ERR(newsubdir);
	if (IS_ERR(newsubdir))
		goto put_old_subdent;

	newsubdent = get_subfs_dentry(newdentry);
	rc = PTR_ERR(newsubdent);
	if (IS_ERR(newsubdent))
		goto put_new_subdir;

	/*
	 * If new file exists it will be implcitly unlinked so
	 * behave like in unlink case.
	 * If it does not exist we have two write accesses - for
	 * both old and new directory, I guess it does not matter
	 * which one is used in this case
	 */
	if (newdentry->d_inode)
		rc = subfs_get_access(newdentry->d_inode, 1);
	else
		rc = subfs_get_access(olddir, 1);
	if (rc)
		goto put_new_subdent;

	rc = oldsubdir->i_op->rename(oldsubdir, oldsubdent,
				     newsubdir, newsubdent);
	if (rc)
		goto put_write_access;

	/*
	 * this is strictly speaking conditional on FS_ODD_RENAME
	 * flag, but as of this writing this flag is set only
	 * for NFS or intermezzo and it hopefully goes away sometimes ...
	 *
	 * Supermount patch adds code to do_kern_mount that checks
	 * for this flag and refuses mounting
	 */
	d_move(oldsubdent, newsubdent);

	olddir->i_mtime = oldsubdir->i_mtime;
	olddir->i_ctime = oldsubdir->i_ctime;
	olddir->i_nlink = oldsubdir->i_nlink;
	olddir->i_size = oldsubdir->i_size;
	olddir->i_blocks = oldsubdir->i_blocks;

	newdir->i_mtime = newsubdir->i_mtime;
	newdir->i_ctime = newsubdir->i_ctime;
	newdir->i_nlink = newsubdir->i_nlink;
	newdir->i_size = newsubdir->i_size;
	newdir->i_blocks = newsubdir->i_blocks;

	olddentry->d_inode->i_nlink = oldsubdent->d_inode->i_nlink;
	olddentry->d_inode->i_ctime = oldsubdent->d_inode->i_ctime;

	if (newdentry->d_inode) {
		newdentry->d_inode->i_nlink = newsubdent->d_inode->i_nlink;
		newdentry->d_inode->i_ctime = newsubdent->d_inode->i_ctime;
	}

put_write_access:
	if (newdentry->d_inode) {
		/* cf. supermount_unlink */
		if (newdentry->d_inode->i_nlink)
			subfs_put_access(newdentry->d_inode, 1);
	} else
		subfs_put_access(olddir, 1);
put_new_subdent:
	dput(newsubdent);
put_new_subdir:
	iput(newsubdir);
put_old_subdent:
	dput(oldsubdent);
put_old_subdir:
	iput(oldsubdir);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "olddentry=%s newdentry=%s rc=%d", olddentry->d_name.name, newdentry->d_name.name, rc);

	return rc;
}

static int 
supermount_readlink(struct dentry *dentry, char *buffer , int buflen)
{
	struct super_block *sb = dentry->d_sb;
	struct inode *inode = dentry->d_inode;
	struct dentry *subdent;
	int write_on = NEED_WRITE_ATIME(inode);
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto go_offline;

	rc = -EINVAL;
	if (!subdent->d_inode->i_op || !subdent->d_inode->i_op->readlink)
		goto put_subdent;

	rc = subfs_get_access(inode, write_on);
	if (rc)
		goto put_subdent;

	UPDATE_ATIME(subdent->d_inode);
	rc = subdent->d_inode->i_op->readlink(subdent, buffer, buflen);
	subfs_put_access(inode, write_on);
	inode->i_atime = subdent->d_inode->i_atime;

put_subdent:
	dput(subdent);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);

	return rc;
}

static int
supermount_follow_link(struct dentry *dentry, struct nameidata *nd)
{
	struct super_block *sb = dentry->d_sb;
	struct inode *inode = dentry->d_inode;
	struct dentry *subdent;
	int write_on = NEED_WRITE_ATIME(inode);
	int rc;
	struct vfsmount *mnt;
	
	ENTER(sb, "dentry=%s", dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto go_offline;

	rc = subfs_get_access(inode, write_on);
	if (rc)
		goto put_subdent;

	UPDATE_ATIME(subdent->d_inode);
	rc = subdent->d_inode->i_op->follow_link(subdent, nd);
	subfs_put_access(inode, write_on);
	inode->i_atime = subdent->d_inode->i_atime;

put_subdent:
	dput(subdent);	
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);

	return rc;
}

static int
supermount_permission(struct inode *inode, int mask)
{
	struct super_block *sb = inode->i_sb;
	struct inode *subi;
	int rc;
	int write_on = !IS_RDONLY(inode) && (mask & MAY_WRITE);
	struct vfsmount *mnt;
	int fake_permissions = 1;

	ENTER(sb, "inode=%p", inode);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subi = get_subfs_inode(inode);
	rc = PTR_ERR(subi);
	if (IS_ERR(subi))
		goto go_offline;

	rc = subfs_get_access(inode, write_on);
	if (rc)
		goto put_subi;

	fake_permissions = 0;
	if (subi->i_op && subi->i_op->permission)
		rc = subi->i_op->permission(subi, mask);
	else
		rc = vfs_permission(subi, mask);

	subfs_put_access(inode, write_on);

put_subi:
	iput(subi);
go_offline:
	subfs_go_offline(sb, mnt);
out:
 	if (fake_permissions && inode == sb->s_root->d_inode) {
 		/* cf. file.c:supermount_open() */
 		rc = vfs_permission(inode, mask);
 	}
 
 	LEAVE(sb, "inode=0x%p rc=%d fake=%d", inode, rc, fake_permissions);

	return rc;
}

static int
supermount_setattr(struct dentry *dentry, struct iattr *attr)
{
	struct super_block *sb = dentry->d_sb;
	struct inode *subi;
	struct dentry *subdent;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	mnt = subfs_go_online(sb);
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	subdent = get_subfs_dentry(dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto go_offline;

	rc = subfs_get_access(dentry->d_inode, 1);
	if (rc)
		goto put_subdent;

	subi = subdent->d_inode;
	if (subi->i_op && subi->i_op->setattr)
		rc = subi->i_op->setattr(subdent, attr);
	else {
		rc = inode_change_ok(subi, attr);
		/*
		 * FIXME
		 * What to do with quota?
		 */
		if (!rc)
			rc = inode_setattr(subi, attr);
	}
	subfs_put_access(dentry->d_inode, 1);
	if (rc)
		goto put_subdent;

	/* 
	 * If it worked, then we need to mark the modification
	 * to the subfs, and we also need to propogate the
	 * change up to the shadowing inode.  
	 */
	attr->ia_mode = subi->i_mode;
	attr->ia_uid = subi->i_uid;
	attr->ia_gid = subi->i_gid;
	attr->ia_size = subi->i_size;
	attr->ia_atime = subi->i_atime;
	attr->ia_mtime = subi->i_mtime;
	attr->ia_ctime = subi->i_ctime;
	attr->ia_valid =
	    ATTR_UID | ATTR_GID | ATTR_MODE | ATTR_SIZE |
	    ATTR_ATIME | ATTR_MTIME | ATTR_CTIME;
	inode_setattr(dentry->d_inode, attr);

put_subdent:
	dput(subdent);
go_offline:
	subfs_go_offline(sb, mnt);
out:
	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);

	return rc;
}

/*
 * directories can handle most operations...  supermount/namei.c just
 * passes them through to the underlying subfs, except for lookup().
 */

/* truncate: is not necesary, handled with setattr
 * revalidate: only needed by nfs
 * ->getattr:	FIXME is not appeared to be used anywhere in kernel; so I am
 *  		not sure how to implement it or what to return
 * FIXME: implement accl functions
 */

struct inode_operations supermount_dir_iops = {
	.create		= supermount_create,
	.lookup		= supermount_lookup,
	.link		= supermount_link,
	.unlink		= supermount_unlink,
	.symlink	= supermount_symlink,
	.mkdir		= supermount_mkdir,
	.rmdir		= supermount_rmdir,
	.mknod		= supermount_mknod,
	.rename		= supermount_rename,
	.permission	= supermount_permission,
	.setattr	= supermount_setattr,
};

struct inode_operations supermount_symlink_iops = {
	.readlink	= supermount_readlink,
	.follow_link	= supermount_follow_link,
	.setattr	= supermount_setattr,
};

struct inode_operations supermount_file_iops = {
	.setattr	= supermount_setattr,
};
