/*
 * WL24xx Wireless LAN Card Common APIs
 *
 * Design Issues:
 *
 *   1. Prevent using O/S dependent routines as possible.
 *   2. Fully utilize Rx/Tx queueing rings.
 *   3. Avoid additional memcpy between O/S and card if its size is large.
 *      (But if the overhead is little, I prefer to make the program clear.)
 *
 * References:
 *
 *   1. WL24xx packet drivers (tooasm.asm)
 *   2. Access Point Firmware Interface Specification for IEEE 802.11 SUTRO
 *   3. IEEE 802.11
 *
 * Additional Information:
 *
 *   1. To prevent forward declaration, I put primitive functions ahead.
 *   2. All veriables are named with type, with form <type><Name> e.g.:
 *      cTmpVar -> type is 'c', and Name is 'TmpVar'
 *
 *      Type List:
 *          c   - unsigned char (BYTE)
 *          n   - int           (DWORD)
 *          b   - boolean int   (BOOL, DWORD)
 *          u   - unsigned int  (WORD)
 *          ul  - unsigned long (DWORD)
 *          p   - pointer       (DWORD)
 *          psz - pointer to string ended with zero
 *          cpsz- const psz
 *          q   - queue (structure)
 *      Others are defined by Windows programming convensions.
 *
 * To do:
 *
 *   1. Fully compatiable with IEEE 802.11 (SUTRO need to be refined too)
 *   2. Process Broadcast Queue , PsPoll Queue, and CfPoll Queue
 *
 * now works on 2.4.18 Tobias Hintze <th@delta.boerde.de>
 */
 
#include "wlapi.h"

WHEN_WLAPI_DEBUG(DWORD WL_DbFlag = 0x00000000L);

static BYTE WL_FPAGE[] = { BSS_FPAGE0, BSS_FPAGE1, BSS_FPAGE2, BSS_FPAGE3 };

static const BYTE WL_SIGNAL_TABLE[2] = { 
    0x0A, 
    0x14   /* QPSK */
};

static const BYTE WL_ROAMING_TABLE[] = {
    0x00,   /* 000000 */
    0x09,   /* 000001 */
    0x14,   /* 000010 */
    0xff,   /* 000011 */
    0x14,   /* 000100 */
    0x14,   /* 000101 */
    0x14,   /* 000110 */
    0xff,   /* 000111 */
    0x04,   /* 001000 */
    0x04,   /* 001001 */
    0x04,   /* 001010 */
    0xff,   /* 001011 */
    0x04,   /* 001100 */
    0x04,   /* 001101 */
    0x04,   /* 001110 */
    0xff,   /* 001111 */
    0x00,   /* 010000 */
    0x09,   /* 010001 */
    0x15,   /* 010010 */
    0xff,   /* 010011 */
    0x15,   /* 010100 */
    0x15,   /* 010101 */
    0x15,   /* 010110 */
    0xff,   /* 010111 */
    0x05,   /* 011000 */
    0x05,   /* 011001 */
    0x05,   /* 011010 */
    0xff,   /* 011011 */
    0x05,   /* 011100 */
    0x05,   /* 011101 */
    0x05,   /* 011110 */
    0xff,   /* 011111 */
    0x00,   /* 100000 */
    0x01,   /* 100001 */
    0x02,   /* 100010 */
    0xff,   /* 100011 */
    0x25,   /* 100100 */
    0x25,   /* 100101 */
    0x25,   /* 100110 */
    0xff,   /* 100111 */
    0x06,   /* 101000 */
    0x06,   /* 101001 */
    0x06,   /* 101010 */
    0xff,   /* 101011 */
    0x06,   /* 101100 */
    0x06,   /* 101101 */
    0x06,   /* 101110 */
    0xff,   /* 101111 */
    0xff,   /* 110000 */ /* 16 dup(0ffh) */
    0xff,   /* 110001 */
    0xff,   /* 110010 */
    0xff,   /* 110011 */
    0xff,   /* 110100 */
    0xff,   /* 110101 */
    0xff,   /* 110110 */
    0xff,   /* 110111 */
    0xff,   /* 111000 */
    0xff,   /* 111001 */
    0xff,   /* 111010 */
    0xff,   /* 111011 */
    0xff,   /* 111100 */
    0xff,   /* 111101 */
    0xff,   /* 111110 */
    0xff,   /* 111111 */
};

/*
 * For debugging
 */
int  TraceNothing(const char *cpszFormat, ...)
{
    return 0;
}

void TraceBuf(void *pBuf, int nSize)
{
    int     i;
    BYTE    *pCh = (BYTE *) pBuf;
    
    for (i = 0; i < nSize; i++)
        printk("%2.2x ", pCh[i]);
        
    printk("\n");
}

/*
 * Get Ethernet MAC addresss.
 *
 * WARNING: We switch to FPAGE0 and switc back again.
 *          Making sure there is no other WL function beening called by ISR.
 */
BOOL WL_GetFlashMacAddress(WL_Adaptor *pAdaptor)
{
    int   nBaseAddr = pAdaptor->nBaseAddr;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_TRIVIAL_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_FLASH_READ);
    
    /* get MAC Address */
    outb_p(BSS_FPAGE0, nBaseAddr + NIC_BSS);  /* BSS */
    outb_p(0x00, nBaseAddr + NIC_LMAL);       /* LMAL */
    outb_p(0x40, nBaseAddr + NIC_LMAH);       /* LMAH */

    /* wait for reading EEPROM */
    NOPLOOP(100);
    pAdaptor->MacAddress.b0 = InB(nBaseAddr + NIC_IODPA);
    NOPLOOP(100);
    pAdaptor->MacAddress.b1 = InB(nBaseAddr + NIC_IODPA);
    NOPLOOP(100);
    pAdaptor->MacAddress.b2 = InB(nBaseAddr + NIC_IODPA);
    NOPLOOP(100);
    pAdaptor->MacAddress.b3 = InB(nBaseAddr + NIC_IODPA);
    NOPLOOP(100);
    pAdaptor->MacAddress.b4 = InB(nBaseAddr + NIC_IODPA);
    NOPLOOP(100);
    pAdaptor->MacAddress.b5 = InB(nBaseAddr + NIC_IODPA);
    
    /* switch to SRAM Page 0 (for safety) */
    outb(BSS_SPAGE0, pAdaptor->nBaseAddr + NIC_BSS);
    
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_FLASH_READ);

    /* Z-Com's MAC Address should be 00:60:... */
    return pAdaptor->MacAddress.b0 == 0x00 && pAdaptor->MacAddress.b1 == 0x60;
}

void WL_FlashOutB(WL_Adaptor *pAdaptor, WORD uPage, WORD uAddr, BYTE cData)
{
    /* switch to Flash RAM Page uPage */
    outb(WL_FPAGE[uPage], pAdaptor->nBaseAddr + NIC_BSS);
    
    /* set LMAL and LMAH */
    outb(uAddr & 0xff, pAdaptor->nBaseAddr + NIC_LMAL);
    outb(uAddr >> 8,   pAdaptor->nBaseAddr + NIC_LMAH);
    
    /* out data to Port A */
    outb(cData, pAdaptor->nBaseAddr + NIC_IODPA);
}

BYTE WL_FlashInB(WL_Adaptor *pAdaptor, WORD uPage, WORD uAddr)
{
    /* switch to Flash RAM Page uPage */
    outb(WL_FPAGE[uPage], pAdaptor->nBaseAddr + NIC_BSS);
    
    /* set LMAL and LMAH */
    outb(uAddr & 0xff, pAdaptor->nBaseAddr + NIC_LMAL);
    outb(uAddr >> 8,   pAdaptor->nBaseAddr + NIC_LMAH);
    
    /* out data to Port A */
    return InB(pAdaptor->nBaseAddr + NIC_IODPA);
}

void WL_FlashInSB(WL_Adaptor *pAdaptor, WORD uPage, WORD uSrc, void *pDest, int nSize)
{
    /* switch to SRAM Page 0 */
    outb(WL_FPAGE[uPage], pAdaptor->nBaseAddr + NIC_BSS);
    
    /* set LMAL and LMAH */
    outb(uSrc & 0xff, pAdaptor->nBaseAddr + NIC_LMAL);
    outb(uSrc >> 8,   pAdaptor->nBaseAddr + NIC_LMAH);
    
    /* rep get from Port A */
    insb(pAdaptor->nBaseAddr + NIC_IODPA, pDest, nSize);
}

/*
 * When calling this function, must hold SUTRO first.
 */
WORD WL_GetFlashID(WL_Adaptor *pAdaptor)
{
    BYTE cByte0, cByte1;
    WORD uID;
    
    /* Autoselect command */
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0x90);
    NOPLOOP(10000);
    
    cByte0 = WL_FlashInB(pAdaptor, 0, 0x0);
    cByte1 = WL_FlashInB(pAdaptor, 0, 0x1);

    /* Reset command */
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xF0);
    NOPLOOP(10000);
    
    uID = (cByte0 << 8) | cByte1;
    
    printk("Flash ROM ID = 0x%x\n", uID);
    return uID;
}

/*
 * Polling if Erase/Programming command is completed
 * Note: IF a==b THEN XOR(a,b) = 0 
 *
 * When calling this function, must hold SUTRO first.
 */
BOOL WL_FlashWriteOk(WL_Adaptor *pAdaptor)
{
    BYTE cByte0, cByte1;

    /* Check 'Toggle Bit' (DQ6) to see if completed */
    do {
        cByte0 = WL_FlashInB(pAdaptor, 0, 0x0);
        cByte1 = WL_FlashInB(pAdaptor, 0, 0x0);
        
        /* Test if exceeded Time Limits (DQ5) */
        if (cByte1 & 0x20) {
            /* Must test DQ6 again before return FALSE */
            cByte0 = WL_FlashInB(pAdaptor, 0, 0x0);
            cByte1 = WL_FlashInB(pAdaptor, 0, 0x0);
    
            return ((cByte0 ^ cByte1) & 0x40) == 0;
        }
            
    } while ((cByte0 ^ cByte1) & 0x40);
    
    return TRUE;
}

BOOL WL_FlashWriteOkAT29C010A(WL_Adaptor *pAdaptor)
{
    BYTE cByte0, cByte1;

    /* Check 'Toggle Bit' (DQ6) to see if completed */
    do {
        cByte0 = WL_FlashInB(pAdaptor, 0, 0x0);
        cByte1 = WL_FlashInB(pAdaptor, 0, 0x0);
        
    } while ((cByte0 ^ cByte1) & 0x40);
    
    return TRUE;
}

/*
 * When calling this function, must hold SUTRO first.
 */
BOOL WL_FlashEraseSector(WL_Adaptor *pAdaptor, WORD uSector)
{
    WORD uPage, uAddr;

    uPage = uSector / 2;
    uAddr = (uSector & 1) ? 0x4000 : 0x0;
    
    /* Sector Erase command (6 commands must within 100 uS) */
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0x80);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, uPage, uAddr, 0x30);

    return WL_FlashWriteOk(pAdaptor);
}

/*
 * When calling this function, must hold SUTRO first.
 */
BOOL WL_FlashWriteByte(WL_Adaptor *pAdaptor, WORD uPage, WORD uAddr, BYTE cData)
{
    /* Autoselect command */
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xA0);

    WL_FlashOutB(pAdaptor, uPage, uAddr, cData);

    return WL_FlashWriteOk(pAdaptor);
}

BOOL WL_FlashWriteBufAT29C010A(WL_Adaptor *pAdaptor, WORD uPage, WORD uAddr, BYTE * buf, WORD size)
{
    int i;
    /* Autoselect command */
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xA0);

    for (i = 0; i < size; i++) 
    { //write all bytes at once - this is different to the AM29F010
       WL_FlashOutB(pAdaptor, uPage, uAddr+i, buf[i]);
    }

    return WL_FlashWriteOkAT29C010A(pAdaptor);
}

/*
 * Write pAdaptor->mibExtra into Flash ROM
 *
 * In fact, only first 29 bytes are used. (not all of extra MIB)
 * To prevent alter other data will be used by future versions, we preserve
 * 256 bytes.
 */
BOOL WL_SetMibInFlash(WL_Adaptor *pAdaptor)
{
    int  i;
    WORD uFlashID;
    BYTE buf[128];//only 128 bytes can be written once to the AT29C010! 
    BOOL bRunning;
    
    bRunning = WL_HoldSutro(pAdaptor);
    
    /* Read original data from flash ROM */
    WL_FlashInSB(pAdaptor, 0, 0x0, buf, sizeof(buf));
    /* TraceBuf(buf, IDX_FREQDOMAIN); */
    
    /* Replace only first 27 bytes (just include Speed+1 (FreqDoamin!:-) */
    memcpy(buf, &pAdaptor->mibExtra, IDX_FREQDOMAIN+1);
    
    uFlashID = WL_GetFlashID(pAdaptor);
    
    if (uFlashID == 0x0120) 
    {
        /* It's AMD AM29F010, must be erase before programming */
        /* Erase 1st 16 Kbytes within Page 0 */
        if (!WL_FlashEraseSector(pAdaptor, 0)) 
        {
            printk("wlioctl.c: WL_FlashEraseSector(0) failed!\n");
            if (bRunning)
                WL_UnHoldSutro(pAdaptor);
            return FALSE;
        }
    	/* Programming flash ROM byte by byte */
    	for (i = 0; i < sizeof(buf); i++) 
    	{
          if (!WL_FlashWriteByte(pAdaptor, 0, i, buf[i])) 
          {
            printk("wlioctl.c: WL_FlashWriteByte(buf[%d]) failed!\n", i);
            if (bRunning)
                WL_UnHoldSutro(pAdaptor);
            return FALSE;
          }  
        }
    }
    else if (uFlashID == 0x1fd5)
    {
        /* It's a AT29C010A */
        if (!WL_FlashWriteBufAT29C010A(pAdaptor, 0, 0, buf, sizeof(buf))) 
        {
            printk("wlioctl.c: WL_FlashWriteBufAT29C010A failed!\n");
            if (bRunning)
                WL_UnHoldSutro(pAdaptor);
            return FALSE;
        }
    }
    else 
    { //orig:0x1df5 is wrong, 0xf51d of v2.0 driver is wrong too, 
        /* It's not AT29C010A */
        printk("wlioctl.c: Flash ROM type (0x%x) is unknown!\n", uFlashID);
        if (bRunning)
            WL_UnHoldSutro(pAdaptor);
        return FALSE;
    }
    
    /* Reset command */
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xAA);
    WL_FlashOutB(pAdaptor, 0, 0x2AAA, 0x55);
    WL_FlashOutB(pAdaptor, 0, 0x5555, 0xF0);
    NOPLOOP(10000);
    
    if (bRunning)
        WL_UnHoldSutro(pAdaptor);
        
    return TRUE;
}

/*
 * Mask interrupt from SUTRO. (i.e. SUTRO cannot interrupt the HOST)
 * Return: TRUE if interrupt is originally enabled
 */
BOOL WL_BlockInterrupt(WL_Adaptor *pAdaptor)
{
    BYTE    cOld, cNew;
    
    /* ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_TRIVIAL_FUNC) == 0); */
    /* WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_BLK_INTR); */
    
    cOld = InB(pAdaptor->nBaseAddr + NIC_GCR);
    cNew = cOld & (~(GCR_ECINT | GCR_INT2EC | GCR_ENECINT));
    outb(cNew, pAdaptor->nBaseAddr + NIC_GCR);
    
    /* WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_BLK_INTR); */
    return (cOld & GCR_ENECINT);
}

/*
 * Enable interrupt from SUTRO. (i.e. SUTRO can interrupt the HOST)
 * Return: TRUE if interrupt is originally enabled
 */
BOOL WL_UnBlockInterrupt(WL_Adaptor *pAdaptor)
{
    BYTE    cOld, cNew;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_TRIVIAL_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_UNBLK_INTR);
    
    cOld = InB(pAdaptor->nBaseAddr + NIC_GCR);
    cNew = (cOld  & ~(GCR_ECINT | GCR_INT2EC)) | GCR_ENECINT;
    outb(cNew, pAdaptor->nBaseAddr + NIC_GCR);
    
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_UNBLK_INTR);
    return (cOld & GCR_ENECINT);
}

/*
 * Hold SUTRO. (i.e. make SUTRO stop)
 * Return: TRUE if SUTRO is originally running
 */
BOOL WL_HoldSutro(WL_Adaptor *pAdaptor)
{
    BYTE    cOld, cNew;
    
    cOld = InB(pAdaptor->nBaseAddr + NIC_GCR);
    cNew = (cOld & ~(GCR_ECINT | GCR_INT2EC)) | GCR_ECWAIT;
    outb(cNew, pAdaptor->nBaseAddr + NIC_GCR);

    return (cOld & GCR_ECWAIT) == 0;
}

/*
 * UnHold SUTRO. (i.e. make SUTRO running)
 * Return: TRUE if SUTRO is originally running
 */
BOOL WL_UnHoldSutro(WL_Adaptor *pAdaptor)
{
    BYTE    cOld, cNew;
    
    cOld = InB(pAdaptor->nBaseAddr + NIC_GCR);
    cNew = cOld  & (~(GCR_ECINT | GCR_INT2EC  | GCR_ECWAIT));
    outb(cNew, pAdaptor->nBaseAddr + NIC_GCR);

    return (cOld & GCR_ECWAIT) == 0;
}

/*
 * Info card to execute a command
 */
void WL_InteruptWla(WL_Adaptor *pAdaptor)
{
    BYTE    cTmp;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_TRIVIAL_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_INT_WLA);
    
    cTmp = InB(pAdaptor->nBaseAddr + NIC_GCR);
    cTmp = (cTmp | GCR_INT2EC) &  (~GCR_ECINT);
    outb(cTmp, pAdaptor->nBaseAddr + NIC_GCR);

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INT_WLA);
}

/*
 *  Move 'nSize' bytes from PC to card. (Shouldn't be interrupted)
 *
 *  uDest = Card addressing space
 *  pSrc  = PC addressing space
 *  nSize = Bytes to move
 */
void WL_SetToWla(WL_Adaptor *pAdaptor, WORD uDest, void *pSrc, int nSize)
{
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_TRIVIAL_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_SET_WLA);
    
    /* switch to SRAM Page 0 */
    outb(BSS_SPAGE0, pAdaptor->nBaseAddr + NIC_BSS);
    
    /* set LMAL and LMAH */
    outb(uDest & 0xff, pAdaptor->nBaseAddr + NIC_LMAL);
    outb(uDest >> 8,   pAdaptor->nBaseAddr + NIC_LMAH);
    
    /* rep out to Port A */
    outsb(pAdaptor->nBaseAddr + NIC_IODPA, pSrc, nSize);

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_SET_WLA);
}

/*
 *  Move 'nSize' bytes from card to PC. (Shouldn't be interrupted)
 *
 *  pSrc  = Card addressing space
 *  uDest = PC addressing space
 *  nSize = Bytes to move
 */
void WL_GetFromWla(WL_Adaptor *pAdaptor, WORD uSrc, void *pDest, int nSize)
{
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_TRIVIAL_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_GET_WLA);
    
    /* switch to SRAM Page 0 */
    outb(BSS_SPAGE0, pAdaptor->nBaseAddr + NIC_BSS);
    
    /* set LMAL and LMAH */
    outb(uSrc & 0xff, pAdaptor->nBaseAddr + NIC_LMAL);
    outb(uSrc >> 8,   pAdaptor->nBaseAddr + NIC_LMAH);
    
    /* rep get from Port A */
    insb(pAdaptor->nBaseAddr + NIC_IODPA, pDest, nSize);

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_GET_WLA);
}

/*
 * Set the uDesc's Next_Descriptor to uNew. Faster!
 * Only change the Next_Descriptor field for nearly atomic operation.
 * Suitable for both Tx and Rx Descriptor.
 */
void WL_SetDescNext(WL_Adaptor *pAdaptor, WORD uDesc, DWORD ulNext)
{
    WL_SetToWla(pAdaptor, uDesc + TXD_Next_Descriptor, &ulNext, sizeof(ulNext));
}

DWORD WL_GetDescNext(WL_Adaptor *pAdaptor, WORD uDesc)
{
    DWORD   ulNext;
    
    WL_GetFromWla(pAdaptor, uDesc + TXD_Next_Descriptor, &ulNext, sizeof(ulNext));
    
    return ulNext;
}

/*
 * Get/Allocate a free Tx Descriptor
 */
WORD WL_GetTxDesc(WL_Adaptor *pAdaptor)
{
    WORD  uRet = pAdaptor->uTxFreeDesc;
    
    if (uRet == 0)
        return 0;
        
    pAdaptor->uTxFreeDesc = WL_GetDescNext(pAdaptor, uRet);
    
    return uRet;
}

/*
 * Free an allocated Tx Descriptor
 */
void WL_FreeTxDesc(WL_Adaptor *pAdaptor, WORD uDesc)
{
    /* Link this descriptor in front of free list */
    WL_SetDescNext(pAdaptor, uDesc, pAdaptor->uTxFreeDesc);
    
    /* Re-point the TxFreeDesc */
    pAdaptor->uTxFreeDesc = uDesc;
}

/*
 * Lock out the SUTRO (e.g. Card), give the host (e.g. PC) the premission of
 * accessing Tx_Q and Rx_Q
 *
 * Algorithm: e.g. Two bedrooms share a bathroom with 2 doors.
 *
 *         HOST --> [lockhost] ..... [locksutro] <-- SUTRO
 *
 * 1. HOST check lockhost first, if lockhost=0, then it can enter the bathroom
 * 2. HOST set locksutro=1.
 * 3. HOST check lockhost again.
 * 3-1 If lockhost=1, then sorry! Both in bathroom. Go back and try again. 
 *     (set locksutro=0)
 * 3-2 If lockhost=0, success! HOST lock out SUTRO from now on.
 *
 * The algorithm above is not good enough. It doesn't guarantee limited waiting.
 *-----------------------------------------------------------------------------
 * A better algorithm should provide 1.exclusive 2.progress 3.limited waiting.
 *
 * We need 3 variables: host, sutro, turn
 *
 * typedef enum turn_state { HOST, SUTRO } turn_t;
 *
 * Initial:
 *                      int host = 0;
 *                      int sutro = 0;
 *                      turn_t turn = HOST; (or SUTRO)
 *
 *
 * <Host Side>                              <Sutro Side>
 *
 * LockSutro()                              LockHost()
 * {                                        {
 *      host = 1;                               sutro = 1;
 *      turn = SUTRO;                           turn  = HOST;
 *
 *      while (sutro && turn == SUTRO);         while (host && turn == HOST);
 *
 *      ...critical session begin...            ...critical session begin...
 * }                                        }
 * 
 * UnLockSutro()                            UnLockHost()
 * {                                        {
 *      host = 0;                               sutro = 0;
 * }                                        }
 *
 * All operations needn't be atomic.
 *
 * 1. Exclusive: if not, then host=1, sutro=1, and turn=HOST & turn=SUTRO.
 *               It's impossible, so Sutro and Host shall not both enter CS.
 * 2. Progess:   if UnLockSutro(), Sutro can enter CS (critical session).
 *               No dead lock.
 * 3. Limited Waiting: If host had entered CS, Sutro waits on while-loop. 
 *               After host leaving CS, even if it tries to enter CS again,
 *               it will wait on while-loop. Because sutro=1, and turn=SUTRO.
 *               On the other hand, Sutro can enter CS, since turn=SUTRO.
 *               (Even Sutro checking host slowly, cannot catch host=0, and
 *                Host enter LockSutro again.)
 *               No matter how fast host or sutro runs, one must be able to
 *               enter CS after the other one leaving.
 *
 * Notice: compiler shouldn't optimize 'turn' assignment and comparison.
 * In this case, WL_GetFromWla() is called every time. No problem.
 */
BOOL WL_LockSutro(WL_Adaptor *pAdaptor)
{
    int     i;
    BYTE    cLockHost;
    BYTE    cLockSutro;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_LOCK_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_LOCK_SUTRO);
    
    for (i = 0; i < 5000; i++) {
        WL_GetFromWla(pAdaptor, CSB_Lockout_Host, &cLockHost, sizeof(cLockHost));

        if (cLockHost == 0) {
            /* Ok! Sutro does not lock host now. Enter! */
            cLockSutro = 1;
            
            /* Set LockSutro = 1 */
            WL_SetToWla(pAdaptor, CSB_Lockout_SUTRO, &cLockSutro, sizeof(cLockSutro));
            
            /* Now, check LockHost flag again */
            WL_GetFromWla(pAdaptor, CSB_Lockout_Host, &cLockHost, sizeof(cLockHost));
            
            /* Check if still not lock host */
            if (cLockHost == 0) {
                WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_LOCK_SUTRO);
                return TRUE;    /* Good! We lock out SUTRO */
            }
            
            /* What a pity! SUTRO try to lock host on the same time */
            /* Go back and try again */
            cLockSutro = 0;
            WL_SetToWla(pAdaptor, CSB_Lockout_SUTRO, &cLockSutro, sizeof(cLockSutro));
        }
    }

    /* Try too many times. Give up! */
    printk("wlapi.c: Try too many times to lock SUTRO, but failed.\n");
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_LOCK_SUTRO);
    return FALSE;
}

/*
 * Unlock out SUTRO. Let SUTRO may access queues
 */
void WL_UnlockSutro(WL_Adaptor *pAdaptor)
{
    BYTE    cLockSutro = 0;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_LOCK_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_UNLOCK_SUTRO);
    
    WL_SetToWla(pAdaptor, CSB_Lockout_SUTRO, &cLockSutro, sizeof(cLockSutro));

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_UNLOCK_SUTRO);
}
 
/*
 * Get/Allocate a free Tx Data Buffer
 *
 *  *--------------*-----------------*----------------------------------*
 *  |    PLCP      |    MAC Header   |  DST  SRC         Data ...       |
 *  |  (24 bytes)  |    (30 bytes)   |  (6)  (6)  (Ethernet Row Data)   |
 *  *--------------*-----------------*----------------------------------*
 *  \               \- IEEE 802.11 -/ \------------ uDataLen -----------/
 *    \-------- TX_HEADER --------/    \-------- Ethernet Frame -------/
 *
 * Return = Postion in Card
 */
WORD WL_GetTxBuffer(WL_Adaptor *pAdaptor, WORD uDataLen)
{
    WORD  uRet;
    TX_BUFFER_LINK  tag;

    WORD  uFullLen = sizeof(tag) + sizeof(TX_HEADER) + uDataLen;
    WORD  uEnd = pAdaptor->uTxBufferHead + uFullLen;
    
    if (pAdaptor->uTxBufferHead >= pAdaptor->uTxBufferTail) {
        /* Free memory is from Head to End, and from Begin to Tail */
        /* Check if have enough space for the data and next link */
        if (uEnd > pAdaptor->uTxBufferEnd - sizeof(tag)) {
            /* Sorry! The part from Head to End is too small, give up */
            uEnd = pAdaptor->uTxBufferBegin + uFullLen; 

            if (uEnd >= pAdaptor->uTxBufferTail) {
                /* The part from Begin to Tail is still too small! */
                return 0;
            }
            tag.ulNextBuf   = pAdaptor->uTxBufferBegin;
            tag.ulConsumed  = 1;    /* allocated but belongs nobody */
            
            WL_SetToWla(pAdaptor, pAdaptor->uTxBufferHead, &tag, sizeof(tag));
            uRet = pAdaptor->uTxBufferBegin;
        }
        else {
            uRet = pAdaptor->uTxBufferHead;
        }
    }
    else {
        /* Free space is between Head & Tail */
        if (uEnd >= pAdaptor->uTxBufferTail) {
            /* No memory availabe */
            return 0;
        }
            
        uRet = pAdaptor->uTxBufferHead;
    }

    /* Set up new Head */
    pAdaptor->uTxBufferHead = uRet + uFullLen;
    tag.ulNextBuf   = pAdaptor->uTxBufferHead;
    tag.ulConsumed  = 0;

    /* Write Mark on Card */    
    WL_SetToWla(pAdaptor, uRet, &tag, sizeof(tag));
    
    return uRet + sizeof(tag);
}

/*
 * Free an allocated Tx Buffer. uPtr must be correct position.
 *
 */
void WL_FreeTxBuffer(WL_Adaptor *pAdaptor, WORD uPtr)
{
    TX_BUFFER_LINK  tag;
    WORD            uStart = uPtr - sizeof(tag);

    /* check if all space is free */
    if (pAdaptor->uTxBufferHead == pAdaptor->uTxBufferTail)    
        return;

    WL_GetFromWla(pAdaptor, uStart, &tag, sizeof(tag));
    
    if (tag.ulConsumed != 0 && tag.ulConsumed != 1) {
        /* not a valid flag ?? uPtr is incorrect ?? */
        printk("wlapi.c: WL_FreeTxBuffer uPtr is incorrect\n");
        return;
    }
    
    tag.ulConsumed = 1;     /* mark it consumed */
    WL_SetToWla(pAdaptor, uStart, &tag, sizeof(tag));
    
    if (pAdaptor->uTxBufferTail != uStart) {
        /* This buffer isn't the last one. Nothing to do */
        /* It's possible because there are many kinds of TxQueue */
        return;
    }
    
    /* It's the last one but consumed. Try to adjust Tail to last allocated */
    do {
        WL_GetFromWla(pAdaptor, pAdaptor->uTxBufferTail, &tag, sizeof(tag));
        if (tag.ulConsumed == 0)    /* This buffer is consumed ? */
            break;
            
        pAdaptor->uTxBufferTail = tag.ulNextBuf;
    } while (pAdaptor->uTxBufferTail != pAdaptor->uTxBufferHead);
}

/*
 * Purge Tx Queue
 *
 * Sent descriptor and buffer will be recycled
 */
void WL_PurgeTxQueue(WL_Adaptor *pAdaptor, TX_Q *pTxQ)
{
    TX_DESC desc;

    /* In face, we may use while(1) instead, since there is always at least */
    /* one node left in queue. But do double check for ensurence            */
    while (pTxQ->TxQHead != 0 && pTxQ->TxQHead != NULL_DESC) {
        WL_GetFromWla(pAdaptor, pTxQ->TxQHead, &desc, sizeof(desc));
        
        if (desc.Next_Descriptor & NULL_DESC)
            return;        /* This is the last one node */
        
        desc.Tx_State &= TX_STATE_OWN_BY_SUTRO | TX_STATE_DONE;
        
        if (desc.Tx_State != TX_STATE_DONE)
            return;
            
        /* Owned by the card and state is TX_STATE_DOWN */
        if (desc.Tx_Start_of_Frame != 0) {
            /* have some frames with the descriptor */
            WL_FreeTxBuffer(pAdaptor, desc.Tx_Start_of_Frame);
        }
        
        WL_FreeTxDesc(pAdaptor, pTxQ->TxQHead);
        pTxQ->TxQHead = desc.Next_Descriptor;
    }
}

/*
 * Enable card receiver
 */
BOOL WL_EnableReceiver(WL_Adaptor *pAdaptor)
{
    int         i;
    CMD_BLOCK   CB;
    
    CB.Command                  = CMD_Enable_Receiver;
    CB.Command_Status           = 0x00;

    /* copy CB to card */
    WL_SetToWla(pAdaptor, Command_Block, &CB, sizeof(CB));
    
    /* INT2EC */
    WL_InteruptWla(pAdaptor);

    /* Polling for command complete */
    for (i = 0; i < 10000; i++) {
        /* Get command status */
        WL_GetFromWla(pAdaptor, Command_Block, &CB, sizeof(CB));
        
        if (CB.Command_Status != 0) {    /* Command success */
            DWORD *plParam      = (DWORD *) &CB.Command_Parameters[0];
            
            /* Clear 'Command' and 'Command_Status', only 2 bytes */
            CB.Command          = 0;
            CB.Command_Status   = 0;
            WL_SetToWla(pAdaptor, Command_Block, &CB, CB_Error_Offset - CB_Command);

            pAdaptor->uRxDataQueue        = *plParam++;
            pAdaptor->uRxManagementQueue  = *plParam;
            
            return TRUE;
        }
    }
    
    return FALSE;
}
 
/* 
 * Enable Card Transmiter
 */
BOOL WL_InitTx(WL_Adaptor *pAdaptor)
{
    int         i;
    CMD_BLOCK   CB;
    DWORD       *plParam    = (DWORD *) &CB.Command_Parameters[0];
    
    CB.Command              = CMD_Init_Tx;
    CB.Command_Status       = 0x00;
    *plParam++              = pAdaptor->qData.TxQHead;
    *plParam++              = pAdaptor->qManagement.TxQHead;
    *plParam++              = 0 /* pAdaptor->qBroadcast.TxQHead */;
    *plParam++              = 0 /* pAdaptor->qPsPoll.TxQHead */;
    *plParam                = 0 /* pAdaptor->qCfPoll.TxQHead */;
    
    /* copy CB to card */
    WL_SetToWla(pAdaptor, Command_Block, &CB, sizeof(CB));
    
    /* INT2EC */
    WL_InteruptWla(pAdaptor);

    /* Polling for command complete */
    for (i = 0; i < 10000; i++) {
        /* Get command status */
        WL_GetFromWla(pAdaptor, Command_Block, &CB, sizeof(CB));
        
        if (CB.Command_Status != 0) {    /* Command success */
            /* Clear 'Command' and 'Command_Status', only 2 bytes */
            CB.Command          = 0;
            CB.Command_Status   = 0;
            WL_SetToWla(pAdaptor, Command_Block, &CB, CB_Error_Offset - CB_Command);

            return TRUE;
        }
    }
    
    return FALSE;
}

BOOL WL_GetMibValue(WL_Adaptor *pAdaptor, BYTE cGroup, BYTE cIndex, void *pDest, BYTE cSize)
{
    int         i;
    CMD_BLOCK   CB;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_MIB_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_GET_MIB);
    
    CB.Command                  = CMD_Get_MIB_Variables;
    CB.Command_Status           = 0x00;
    CB.Command_Parameters[0]    = cGroup;
    CB.Command_Parameters[1]    = cSize;
    CB.Command_Parameters[2]    = cIndex;
    CB.Command_Parameters[3]    = 0x00;     /* Reserved */
    
    /* copy CB to card */
    WL_SetToWla(pAdaptor, Command_Block, &CB, sizeof(CB));
    
    /* INT2EC */
    WL_InteruptWla(pAdaptor);

    /* Polling for command complete */
    for (i = 0; i < 10000; i++) {
        /* Get command status */
        WL_GetFromWla(pAdaptor, Command_Block, &CB, sizeof(CB));
        
        if (CB.Command_Status == 1) {    /* Command success */
            /* Clear 'Command' and 'Command_Status', only 2 bytes */
            CB.Command          = 0;
            CB.Command_Status   = 0;
            WL_SetToWla(pAdaptor, Command_Block, &CB, CB_Error_Offset - CB_Command);

            /* Copy command parameters to data buffer */
            memcpy(pDest, &(CB.Command_Parameters[4]), cSize);

            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_GET_MIB);
            return TRUE;
        }
        else if (CB.Command_Status == 3){
            /* Clear 'Command' and 'Command_Status', only 2 bytes */
            CB.Command          = 0;
            CB.Command_Status   = 0;
            WL_SetToWla(pAdaptor, Command_Block, &CB, CB_Error_Offset - CB_Command);
        
            /* Command Rejected (Invalid parameter) */
            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_GET_MIB);
            return FALSE;
        }
    }
    
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_GET_MIB);
    return FALSE;
}

BOOL WL_SetMibValue(WL_Adaptor *pAdaptor, BYTE cGroup, BYTE cIndex, void *pSrc, BYTE cSize)
{
    int         i;
    CMD_BLOCK   CB;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_MIB_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_SET_MIB);
    
    CB.Command                  = CMD_Set_MIB_Variables;
    CB.Command_Status           = 0x00;
    CB.Command_Parameters[0]    = cGroup;
    CB.Command_Parameters[1]    = cSize;
    CB.Command_Parameters[2]    = cIndex;

    /* copy additional parameters */
    memcpy(&CB.Command_Parameters[4], pSrc, cSize);
        
    /* copy CB to card */
    WL_SetToWla(pAdaptor, Command_Block, &CB, sizeof(CB));
    
    /* INT2EC */
    WL_InteruptWla(pAdaptor);

    /* Polling for command complete */
    for (i = 0; i < 10000; i++) {
        /* Get command status */
        WL_GetFromWla(pAdaptor, Command_Block, &CB, sizeof(CB));
        
        if (CB.Command_Status == 1) {    /* Command success */
            /* Clear 'Command' and 'Command_Status', only 2 bytes */
            CB.Command          = 0;
            CB.Command_Status   = 0;
            WL_SetToWla(pAdaptor, Command_Block, &CB, CB_Error_Offset - CB_Command);

            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_SET_MIB);
            return TRUE;
        }
        else if (CB.Command_Status == 3){
            /* Clear 'Command' and 'Command_Status', only 2 bytes */
            CB.Command          = 0;
            CB.Command_Status   = 0;
            WL_SetToWla(pAdaptor, Command_Block, &CB, CB_Error_Offset - CB_Command);
        
            /* Command Rejected (Invalid parameter) */
            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_SET_MIB);
            return FALSE;
        }
    }
    
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_SET_MIB);
    return FALSE;
}

BOOL WL_ResetBoard(WL_Adaptor *pAdaptor)
{
    BYTE    cTmp;
    DWORD   i;
    
    /* Coreset */
    outb_p(GCR_CORESET, pAdaptor->nBaseAddr + NIC_GCR);
    outb_p(0,           pAdaptor->nBaseAddr + NIC_GCR);
    outb_p(GCR_CORESET, pAdaptor->nBaseAddr + NIC_GCR);

#if 0
    /*** Clean SRAM ***/
    /* switch to SRAM Page 0 */
    OutB(BSS_SPAGE0, pAdaptor->nBaseAddr + NIC_BSS);
    
    /* set LMAL and LMAH */
    OutB(0, pAdaptor->nBaseAddr + NIC_LMAL);
    OutB(0, pAdaptor->nBaseAddr + NIC_LMAH);
    
    /* rep out to Port A */
    for (i = 0; i < 0x8000; i++)
        OutB(0, pAdaptor->nBaseAddr + NIC_IODPA);
#endif

    /* Set Self_Test_Status = 0x00 (Inital State) */
    cTmp = 0;
    WL_SetToWla(pAdaptor, CSB_Self_Test_Status, &cTmp, sizeof(cTmp));
    
    /* Start up */
    outb_p(0,           pAdaptor->nBaseAddr + NIC_GCR);
    
    /* NOP */
    NOPLOOP(1024*50);

    /* WL_UnBlockInterrupt(pAdaptor); */

    /* Polling Self_Test_Status */    
    for (i = 0; i < 10000; i++) {
        WL_GetFromWla(pAdaptor, CSB_Self_Test_Status, &cTmp, sizeof(cTmp));
        
        if (cTmp == 0xFF) {
            /* firmware complete all test successfully */
            return TRUE;
        }
        NOPLOOP(10);
    }
    
    return FALSE;
}

BOOL WL_InitFirmware(WL_Adaptor *pAdaptor)
{
    int     i;
    WORD    uNext;
    WORD    uDescAddr;
    BYTE    cMask;
    TX_DESC desc;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_API_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_INIT);
    
    if (!WL_ResetBoard(pAdaptor)) {
        TRACE_INIT("wlapi.c: WL_ResetBoard failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }
        
    /* Switch to SRAM Page 0 */
    outb(BSS_SPAGE0, pAdaptor->nBaseAddr + NIC_BSS);

    /* Set up SUTRO Interrupt Mask */
    cMask = IMR_Cmnd;
    WL_SetToWla(pAdaptor, CSB_Interrupt_Mask, &cMask, sizeof(cMask));
    
    /* Get Tx block offset */
    if (!WL_GetMibValue(pAdaptor, TYPE_LOCAL_MIB, IDX_TX_BUFFER_OFFSET, 
        &pAdaptor->ulTxBlockOffset, sizeof(pAdaptor->ulTxBlockOffset))) {
        TRACE_INIT("wlapi.c: Get Tx block offset failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }
    /* Get Tx buffer size */
    if (!WL_GetMibValue(pAdaptor, TYPE_LOCAL_MIB, IDX_TX_BUFFER_SIZE, 
        &pAdaptor->ulTxBlockSize, sizeof(pAdaptor->ulTxBlockSize))) {
        TRACE_INIT("wlapi.c: Get Tx buffer size failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }
        
    /* Get Mac Address */
    if (!WL_GetMibValue(pAdaptor, TYPE_MAC_ADDRESS_STATUS_GRP, IDX_MAC_ADDR,
        &pAdaptor->MacAddress, sizeof(pAdaptor->MacAddress))) {
        TRACE_INIT("wlapi.c: Get Mac Address failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }
        
    /* Get full Extra MIB data */
    if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
        &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
        TRACE_INIT("wlapi.c: Get Extra MIB failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }
    
    /* Initial TxDesc Link structure */
    pAdaptor->uTxFreeDesc   = pAdaptor->ulTxBlockOffset;
    
    pAdaptor->uTxBufferBegin = pAdaptor->uTxFreeDesc + sizeof(TX_DESC) * MAX_TX_DESC_RESERVED;
    pAdaptor->uTxBufferHead = pAdaptor->uTxBufferBegin;
    pAdaptor->uTxBufferTail = pAdaptor->uTxBufferHead;
    
    pAdaptor->uTxBufferEnd  = pAdaptor->ulTxBlockOffset + pAdaptor->ulTxBlockSize - 1;

    /* Link all Free-Desc together, from tail to head */
    memset(&desc, 0 , sizeof(desc));
    
    uNext = 0;
    for (i = MAX_TX_DESC_RESERVED - 1; i >= 0; i--) {
        WORD  uAddr;
        
        desc.Next_Descriptor    = uNext;
        desc.Tx_Start_of_Frame  = 0;

        uAddr = pAdaptor->uTxFreeDesc + i * sizeof(TX_DESC);
        WL_SetToWla(pAdaptor, uAddr, &desc, sizeof(desc));
        
        uNext = uAddr;
    }
    
    /* Initial 5 TxQueue pointers */
    memset(&desc, 0, sizeof(desc));

    /* The initial descriptor is an empty one */    
    desc.Tx_State           = TX_STATE_OWN_BY_SUTRO | TX_STATE_DONE;
    desc.Next_Descriptor    = NULL_DESC;
    desc.Tx_Start_of_Frame  = 0;

    /* Data Queue */    
    uDescAddr = WL_GetTxDesc(pAdaptor);
    WL_SetToWla(pAdaptor, uDescAddr, &desc, sizeof(desc));
    pAdaptor->qData.TxQHead = uDescAddr;
    pAdaptor->qData.TxQTail = uDescAddr;
    
    /* Management Queue */    
    uDescAddr = WL_GetTxDesc(pAdaptor);
    WL_SetToWla(pAdaptor, uDescAddr, &desc, sizeof(desc));
    pAdaptor->qManagement.TxQHead = uDescAddr;
    pAdaptor->qManagement.TxQTail = uDescAddr;
    
    /* Broadcast Queue */    
    uDescAddr = WL_GetTxDesc(pAdaptor);
    WL_SetToWla(pAdaptor, uDescAddr, &desc, sizeof(desc));
    pAdaptor->qBroadcast.TxQHead = uDescAddr;
    pAdaptor->qBroadcast.TxQTail = uDescAddr;
    
    /* PsPoll Queue */    
    uDescAddr = WL_GetTxDesc(pAdaptor);
    WL_SetToWla(pAdaptor, uDescAddr, &desc, sizeof(desc));
    pAdaptor->qPsPoll.TxQHead = uDescAddr;
    pAdaptor->qPsPoll.TxQTail = uDescAddr;
    
    /* CfPoll Queue */    
    uDescAddr = WL_GetTxDesc(pAdaptor);
    WL_SetToWla(pAdaptor, uDescAddr, &desc, sizeof(desc));
    pAdaptor->qCfPoll.TxQHead = uDescAddr;
    pAdaptor->qCfPoll.TxQTail = uDescAddr;

    /* Variables */
    pAdaptor->uTxSequence = 0;
    memset(&pAdaptor->CurrentBSSID, 0, sizeof(pAdaptor->CurrentBSSID));
    pAdaptor->ulLastBeaconTime = 0;
    pAdaptor->uRoamingState    = 2;
    
    /* Enable Card Receiver */
    if (!WL_EnableReceiver(pAdaptor)) {
        TRACE_INIT("wlapi.c: WL_EnableReceiver failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }
    
    /* Enable Card Transmiter */
    if (!WL_InitTx(pAdaptor)) {
        TRACE_INIT("wlapi.c: WL_InitTx failed!\n");
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
        return FALSE;
    }

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_INIT);
    return TRUE;    
}

/*
 * Send a packet.
 *
 * pData = Ethernet raw frame.  (e.g. pData[0] - pData[5] is Dest MAC Addr, 
 *                                    pData[6] - pData[11] is Src MAC Addr)
 * cRate = 0 : Default (Depends on MIB)
 *         1 : 1 M bits
 *         2 : 2 M bits
 *
 * Ref: IEEE 802.11
 */
BOOL WL_SendPkt(WL_Adaptor *pAdaptor, BYTE *pData, WORD uLen, BYTE cRate)
{
    WORD        uDesc;
    WORD        uBuf;
    BYTE        cCCR;

    TX_DESC     desc;
    TX_HEADER   hdr;

    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_API_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_TX_PKT);
    
    uDesc = WL_GetTxDesc(pAdaptor);

    if (uDesc == 0) {   /* No free descriptor available */
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_TX_PKT);
        return FALSE;
    }
    
    uBuf = WL_GetTxBuffer(pAdaptor, uLen);

    if (uBuf == 0)  {
        /* No free data buffer available */
        WL_FreeTxDesc(pAdaptor, uDesc);
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_TX_PKT);
        return FALSE;
    }

    if (cRate != 1 && cRate != 2) { 
        /* Not valid rate setup, get from MIB */
        cRate = pAdaptor->mibExtra.Speed;
        
        if (cRate != 1 && cRate != 2) {
            printk("wlapi.c: Speed setup in MIB is incorrect. Use QPSK instead.\n");
            cRate = 2;
        }
    }

    /* Set up Descriptor */    
    desc.Tx_Start_of_Frame  = uBuf;         /* point to Tx buffer */
    desc.Next_Descriptor    = NULL_DESC;    /* is the last in Tx queue */
    desc.Tx_Length          = uLen + sizeof(DATA_MAC_HDR); /* include MPDU */
    desc.Tx_State           = TX_STATE_OWN_BY_SUTRO;
    desc.Tx_Rate            = cRate - 1;    /* 0: 1M, 1: 2M */
    desc.Driver             = 0;            /* not necessary */
    desc.Reserved1          = 0;            /* not necessary */
    desc.Num_RTS_Attempts   = 0;
    desc.Num_Data_Attempts  = 0;
    desc.Control            = 0;
    desc.Reserved2          = 0;            /* not necessary */
    
    /* Set up PLCP of TX_HEADER */
    memset(hdr.PlcpHdr.SYNC, 0xFF, sizeof(hdr.PlcpHdr.SYNC));
        
    hdr.PlcpHdr.SFD         = 0xF3A0;
    hdr.PlcpHdr.SIGNAL      = WL_SIGNAL_TABLE[cRate - 1];  /* FSK or QPSK */
    hdr.PlcpHdr.SERVICE     = 0;
    hdr.PlcpHdr.LENGTH      = (uLen + sizeof(DATA_MAC_HDR) + 4) << 3; /* why ? */
    
    /* hdr.PlcpHdr.CRC16       = ?? */
    
    /* Setup MPDU of TX_HEADER */
    hdr.MacHdr.SequenceControl     = ++pAdaptor->uTxSequence;
    
    cCCR = pAdaptor->mibExtra.CCR;
    TRACE_API("wlapi.c: cCCR = %d\n", (int) cCCR);
    
    if (cCCR == CCR_INFRA) {
        /* Infrastructure Mode */
        hdr.MacHdr.FrameControl    = FRAME_CTRL_INFRA;
        hdr.MacHdr.DurationID      = 0;
        
        /* Address1 */
        if (pAdaptor->mibExtra.RoamingEnable) {
            /* Use current BSS ID */
            hdr.MacHdr.Address1 = pAdaptor->CurrentBSSID;
        }
        else {
            /* Use BSS ID set in MIB */
            hdr.MacHdr.Address1 = pAdaptor->mibExtra.BSS_ID;
        }
        
        /* Address 2 = Src MAC Address */
        hdr.MacHdr.Address2 = *((MADDR *) &pData[6]);
        
        /* Address 3 = Dest MAC Address */
        hdr.MacHdr.Address3 = *((MADDR *) pData);
    }
    else if (cCCR == CCR_ADHOC) {
        /* Ad-HOC Mode */
        hdr.MacHdr.FrameControl    = FRAME_CTRL_ADHOC;
        hdr.MacHdr.DurationID      = 0;
        
        /* Address 1 = Dest MAC Address */
        hdr.MacHdr.Address1 = *((MADDR *) pData);
        
        /* Address 2 = Src MAC Address */
        hdr.MacHdr.Address2 = *((MADDR *) &pData[6]);
    }
    else if (cCCR == CCR_AP) {
        /* Access Point Mode */
        hdr.MacHdr.FrameControl    = FRAME_CTRL_AP;
        hdr.MacHdr.DurationID      = 0;

        /* Address 2 = My MAC Address */
        hdr.MacHdr.Address2 = pAdaptor->MacAddress;
        
        /* Address 1 = Dest MAC Address */
        hdr.MacHdr.Address1 = *((MADDR *) pData);
        
        /* Address 3 = Src MAC Address */
        hdr.MacHdr.Address3 = *((MADDR *) &pData[6]);
    }
    else if (cCCR == CCR_IAP) {
        /* Inter-AP Mode */
        hdr.MacHdr.FrameControl    = FRAME_CTRL_IAP;
        hdr.MacHdr.DurationID      = 0;
        
        /* Address 2 = My MAC Address */
        hdr.MacHdr.Address2 = pAdaptor->MacAddress;
        
        /* Address 1 = 0xFF 0xFF 0xFF 0xFF 0xFF 0xFF */
        memset(&hdr.MacHdr.Address1, 0xFF, sizeof(hdr.MacHdr.Address1));
        
        /* Address 3 = Dest MAC Address */
        hdr.MacHdr.Address3 = *((MADDR *) pData);
        
        /* Address 4 = Src MAC Address */
        hdr.MacHdr.Address4 = *((MADDR *) &pData[6]);
    }
    else {
        /* any other modes ?? */
        printk("wlapi.c: CCR = %u ?? Check your card setup!\n", (unsigned) cCCR);
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_TX_PKT);
        return FALSE;
    }
    
    /* Write TX_HEADER to Card */
    WL_SetToWla(pAdaptor, uBuf, &hdr, sizeof(hdr));
    
    /* Write the data to card */
    WL_SetToWla(pAdaptor, uBuf + sizeof(hdr), pData, uLen);
    
    /* Update the allocated descriptor */
    WL_SetToWla(pAdaptor, uDesc, &desc, sizeof(desc));
    
    /* Link the descriptor behind the last in data descriptor queue         */
    /* There is at least one descriptor in Queue. (Ref. WL_InitFirmware()   */
    /* This make SUTRO to process this node. Note: MSB bit is written last, */
    /* and ensure 4 bytes are completly updated.                            */
    WL_SetDescNext(pAdaptor, pAdaptor->qData.TxQTail, uDesc);
    
    /* Change Data Descriptor Queue tail pointer */
    pAdaptor->qData.TxQTail = uDesc;

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_TX_PKT);
    return TRUE;    
}

/*
 * Check Receive Queue, and return packet length
 *
 * uQueueType = DATA_Q : Check data queue
 *            = MGNT_Q : Check management queue
 *
 * Return = data length in the first queue, or -1 if no data.
 * Note: if return 0, you must still call WL_ReceiveDown too.
 */
int WL_ReceiveLook(WL_Adaptor *pAdaptor, RX_Q_TYPE uQueueType)
{
    BYTE    uRxState;
    WORD    *puRxQueue;
    WORD    uRxDesc;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_API_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_RX_LOOK);
    
    puRxQueue = uQueueType == DATA_Q ? 
                &pAdaptor->uRxDataQueue : &pAdaptor->uRxManagementQueue;
                
    uRxDesc = *puRxQueue;
                     
    WL_GetFromWla(pAdaptor, uRxDesc + RXD_Rx_State, &uRxState, sizeof(uRxState));
    
    while ((uRxState & RX_STATE_OWN_BY_SUTRO) == 0) {
        DWORD ulNextRxDesc;
        
        /* the descriptor doesn't belong to SUTRO (Ethernet card) */
        if ((uRxState & RX_STATE_CONSUMED_BY_DRIVER) == 0) {
            /* not consumed by driver, take this one */
            WORD  uRxLen;
            
            /* Get the RxLength */
            WL_GetFromWla(pAdaptor, uRxDesc + RXD_Rx_Length, &uRxLen, sizeof(uRxLen));

            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_LOOK);
            return  uRxLen;
        }
        
        /* This descriptor is consumed by driver last time */
        /* Check if there is any next descriptor */
        ulNextRxDesc = WL_GetDescNext(pAdaptor, uRxDesc);
        TRACE_API("ulNextRxDesc=%lX\n", ulNextRxDesc);
        if (ulNextRxDesc & NULL_DESC) {  /* Test MSB (bit 31) */
            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_LOOK);
            return -1;   /* This is the last one but consumed */
        }

        /* ------------------------ CAUTION ---------------------------*/            
        /* We must reload Next Descriptor again although MSB is clear, */
        /* because maybe some of the bytes are partially updated.      */
        /* Those bytes doesn't fix untill MSB is clear.                */
        ulNextRxDesc = WL_GetDescNext(pAdaptor, uRxDesc);
        
        if (ulNextRxDesc == 0) {
            WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_LOOK);
            return -1;   /* ulNextRxDesc = 0 ?? */
        }
        
        /* Since it has a next descriptor, and consumed, give back to SUTRO */
        TRACE_API("WL_ReceiveLook: return the desc %x to SUTRO\n", uRxDesc);
        uRxState |= RX_STATE_OWN_BY_SUTRO;
        WL_SetToWla(pAdaptor, uRxDesc + RXD_Rx_State, &uRxState, sizeof(uRxState));
        
        /* Set up new pointer. puRxQueue -> Data or Mgmt queue */
        *puRxQueue = ulNextRxDesc;

        /* Test next descriptor */
        uRxDesc = *puRxQueue;
                     
        WL_GetFromWla(pAdaptor, uRxDesc + RXD_Rx_State, &uRxState, sizeof(uRxState));
    }

    /* This descriptor is owned by SUTRO, means it is beening processing */
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_LOOK);
    return -1;
}

/*
 * Receive data from Receive Queue.
 *
 * uQueueType = DATA_Q : Check data queue
 *            = MGNT_Q : Check management queue
 * uOffset = where to begin (0: first byte in Rx Buffer)
 * pBuf = address of host 
 * uSize = size of buffer.
 *
 * Scenario:
 *
 *  WORD uQueueType = DATA_Q;
 *  int  nSize = WL_ReceiveLook(pAdaptor, uQueueType);
 *  if (nSize != -1) {
 *      DWORD  ulTimeStamp;
 *      WORD uRSSI = WL_Receive(pAdaptor, uQueueType, 0, pBuf, uSize, &ulTimeStamp);
 *      WL_ReceiveDone(pAdaptor, uQueueType);
 *  }
 *
 * Return: RSSI or 0 if failed
 */
WORD WL_Receive(WL_Adaptor *pAdaptor, RX_Q_TYPE uQueueType, WORD uOffset, 
    BYTE *pBuf, WORD uSize, DWORD *plTimeStamp)
{
    BYTE    cRSSI;
    WORD    uRxDesc;
    RX_DESC desc;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_API_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_RX_RECV);
    
    uRxDesc = uQueueType == DATA_Q ? 
              pAdaptor->uRxDataQueue : pAdaptor->uRxManagementQueue;
                
    WL_GetFromWla(pAdaptor, uRxDesc, &desc, sizeof(desc));
    
    if (desc.Rx_State & RX_STATE_OWN_BY_SUTRO) {
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_RECV);
        return 0;
    }
    
    if (desc.Rx_State & RX_STATE_CONSUMED_BY_DRIVER) {
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_RECV);
        return 0;
    }

    TRACE_API("desc.Rx_Start_Of_Frame = 0x%lX\n", desc.Rx_Start_Of_Frame);
    
    /* Owner is me, and not consumed, load it to pBuf */
    WL_GetFromWla(pAdaptor, desc.Rx_Start_Of_Frame + uOffset, pBuf, uSize);
    
    /* Get RSSI, why offset = -12 ? */
    WL_GetFromWla(pAdaptor, desc.Rx_Start_Of_Frame - 12, &cRSSI, sizeof(cRSSI));
    
    /* Get Timestamp, why offset = -11 ? */
    WL_GetFromWla(pAdaptor, desc.Rx_Start_Of_Frame - 11, plTimeStamp, sizeof(DWORD));
    
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_RECV);
    return cRSSI;
}

/*
 * Clean current packet, and return 
 *
 * 0 : no more packet, 1: have more packets
 *
 * uQueueType = DATA_Q : data queue
 *            = MGNT_Q : management queue
 *
 */
BOOL WL_ReceiveDone(WL_Adaptor *pAdaptor, RX_Q_TYPE uQueueType)
{
    BYTE    uRxState;
    WORD    *puRxQueue;
    WORD    uRxDesc;
    BOOL    bWithNext;
    DWORD   ulNextRxDesc;
    
    ASSERT_STAT((pAdaptor->ulDbgStat & WL_STAT_API_FUNC) == 0);
    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat |= WL_STAT_RX_DONE);
    
    puRxQueue = uQueueType == DATA_Q ? 
                &pAdaptor->uRxDataQueue : &pAdaptor->uRxManagementQueue;
                
    uRxDesc = *puRxQueue;
                     
    WL_GetFromWla(pAdaptor, uRxDesc + RXD_Rx_State, &uRxState, sizeof(uRxState));
    
    TRACE_API("WL_ReceiveDone(): uRxDesc=%x, uRxState=%x\n", uRxDesc, (int) uRxState); 
    if (uRxState & RX_STATE_OWN_BY_SUTRO) {
        WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_DONE);
        return FALSE;
    }
    
    uRxState |= RX_STATE_CONSUMED_BY_DRIVER;

    bWithNext  = FALSE;
    
    ulNextRxDesc = WL_GetDescNext(pAdaptor, uRxDesc);
    TRACE_API("WL_ReceiveDone(): ulNextRxDesc=%lx\n", ulNextRxDesc); 
    if ((ulNextRxDesc & NULL_DESC) == 0) {
        /* Have next descriptor */
        /* ------------------------ CAUTION ---------------------------*/            
        /* We must reload Next Descriptor again although MSB is clear, */
        /* because maybe some of the bytes are partially updated.      */
        /* Those bytes doesn't fix untill MSB is clear.                */
        ulNextRxDesc = WL_GetDescNext(pAdaptor, uRxDesc);

        if (ulNextRxDesc != 0) {
            TRACE_API("WL_ReceiveDone: return the desc %x to SUTRO\n", uRxDesc);
            /* Move to next one, return this one to SUTRO */
            uRxState |= RX_STATE_OWN_BY_SUTRO;
            *puRxQueue = ulNextRxDesc;

            bWithNext = TRUE;
        }
    }
    
    /* After everything is done, write uRxState to SUTRO */
    WL_SetToWla(pAdaptor, uRxDesc + RXD_Rx_State, &uRxState, sizeof(uRxState));

    WHEN_WLAPI_DEBUG(pAdaptor->ulDbgStat &= ~WL_STAT_RX_DONE);
    return bWithNext;    
}

/*
 * Check Data Receive Queue, and return packet size.
 * (DATA_MAC_HDR not included). Skip invalid packet automatically.
 * 
 * return:  -1 : no more packet.
 *         > 0 : packet size.
 *         = 0 : SHOULD NOT HAPPEN
 */
int WL_ReceiveDataLook(WL_Adaptor *pAdaptor) 
{
    int  nRawSize;
    
    do {
        nRawSize = WL_ReceiveLook(pAdaptor, DATA_Q);

        /* Normal ethernet pkt. Notice: must cast to int */
        if (nRawSize > (int) sizeof(DATA_MAC_HDR))
            return nRawSize - sizeof(DATA_MAC_HDR);
        
        /* No data packet in queue */        
        if (nRawSize == -1)
            return -1;
        
        /* Invalid packet, ether packet size = 0 ? strange! */
        /* It does happen sometimes, makes the caller confused */
        printk("wlapi.c: nRawsize=%d <= %d, Skip it\n", nRawSize, sizeof(DATA_MAC_HDR));
        
        /* Must skip it, and test next one */
    } while (WL_ReceiveDone(pAdaptor, DATA_Q));
            
    /* no next */
    return -1;        
}

/*
 * Receive ethernet data from Receive Queue.
 * It is similar with WL_Receive() except that it only processes DATA_Q.
 * Return RSSI if success, or 0 if failed
 */
WORD WL_ReceiveData(WL_Adaptor *pAdaptor, WORD uOffset, BYTE *pBuf, WORD uSize)
{
    DWORD   dwTimeStamp;

    ASSERT(uOffset <= ETHER_MAX_LEN && (uOffset + uSize) < ETHER_MAX_LEN);
    return WL_Receive(pAdaptor, DATA_Q, uOffset + sizeof(DATA_MAC_HDR), 
                      pBuf, uSize, &dwTimeStamp);
}

BOOL WL_ReceiveDataDone(WL_Adaptor *pAdaptor)
{
    return WL_ReceiveDone(pAdaptor, DATA_Q);
}

void WL_Roaming(WL_Adaptor *pAdaptor)
{
    int     nSize;
    BOOL    bMorePkts;

    if (pAdaptor->mibExtra.CCR != CCR_INFRA)
        return;
    
    /* Check for management queue */
    bMorePkts = TRUE;
    while (bMorePkts && (nSize = WL_ReceiveLook(pAdaptor, MGNT_Q)) != -1) {
        WORD        uRSSI;
        DWORD       ulTimeStamp;
        BEACON_PKT  pkt;
        BYTE        cIndex;
        
        TRACE_ROAM("Rx ....> Management\n");
        uRSSI = WL_Receive(pAdaptor, MGNT_Q, 0, (BYTE *) &pkt, sizeof(pkt), 
                           &ulTimeStamp);
        
        TRACE_ROAM("Rx RSSI = %u, BSSID = %02x %02x %02x %02x %02x %02x\n",
            uRSSI, pkt.MacHdr.BSSID.b0, pkt.MacHdr.BSSID.b1, pkt.MacHdr.BSSID.b2,
            pkt.MacHdr.BSSID.b3, pkt.MacHdr.BSSID.b4, pkt.MacHdr.BSSID.b5);
            
        if (uRSSI == 0)
            return;
            
        bMorePkts = WL_ReceiveDone(pAdaptor, MGNT_Q);

        /* First byte should be 0x80, refer 802.11 */        
        if ((pkt.MacHdr.FrameControl & 0xFF) != FRAME_CTRL_BEACON_LO)
            return;
        
        /* Verify if MIB enable roaming */
        if (pAdaptor->mibExtra.RoamingEnable != 1) {
            /* Disable roaming */
            /* Keep Beacon time, and use BSS ID in MIB as current BSS ID */
            pAdaptor->ulLastBeaconTime  = ulTimeStamp;
            pAdaptor->CurrentBSSID      = pAdaptor->mibExtra.BSS_ID;
            return;
        }
        
        if (pkt.Frame.DS_ElementID == DS_ELEMENT_ID) {
            /* Check if channel is correct */
            if (pkt.Frame.DS_CurrentChannel != pAdaptor->mibExtra.Channel)
                return;
        }
        
        /* Verify if ESS ID is correct */
        if (!MA_ISEQUAL(pkt.Frame.SSID, pAdaptor->mibExtra.ESS_ID))
            return;
        
        cIndex = 0;
        
        /* Check if BSS ID is correct */
        if (MA_ISEQUAL(pkt.MacHdr.BSSID, pAdaptor->CurrentBSSID))
            cIndex |= 0x08;
        
        /* Check if over time */
        if (ulTimeStamp > pAdaptor->ulLastBeaconTime + BEACON_TIMEOUT)
            cIndex |= 0x04;
        
        cIndex |= pAdaptor->uRoamingState & 0x03;
        
        if (uRSSI < 75) {
            if (uRSSI >= 68)
                cIndex |= 0x10;     /* 75 > uRSSI >= 68 */
            else
                cIndex |= 0x20;     /* uRSSI < 68 */
        }
        
        cIndex = WL_ROAMING_TABLE[cIndex];
        if (cIndex == 0xFF)         /* Skip if invalid value */
            return;

        /* Keep RSSI level */
        pAdaptor->uRoamingState = cIndex & (0x01 | 0x02);
        
        if (cIndex & 0x04)          /* Update Time */
            pAdaptor->ulLastBeaconTime = ulTimeStamp;

        if (cIndex & 0x08)          /* Keep BSS ID */
            pAdaptor->KeepBSSID = pkt.MacHdr.BSSID;
        
        if (cIndex & 0x10) {        /* New BSS ID */
            /* current BSS ID = pkt.MacHdr.BSSID */
            pAdaptor->CurrentBSSID = pkt.MacHdr.BSSID;
            /* Update Extra MIB */
            WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_BSSID, 
                &pAdaptor->CurrentBSSID, sizeof(pAdaptor->CurrentBSSID));
        }
        
        if (cIndex & 0x20) {
            if (MA_ISZERO(pAdaptor->KeepBSSID)) {
                /* CurrentBSSID = pkt.MacHdr.BSSID */
                pAdaptor->CurrentBSSID = pkt.MacHdr.BSSID;
            }
            else {
                /* CurrentBSSID = KeepBSSID */
                pAdaptor->CurrentBSSID = pAdaptor->KeepBSSID;
            }
            /* Update Extra MIB */
            WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_BSSID, 
                &pAdaptor->CurrentBSSID, sizeof(pAdaptor->CurrentBSSID));
        }   /* end of if 0x20 */
        
    }   /* end of while */
}

