/*
 * lcd.c -- code to handle LCD brightness
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Maciek Grniak <mago@acn.waw.pl>, 2002
 * Modified by Sos Pter <sp@osb.hu>, 2002-2003
 * Some minor fixes by Bernhard Kaindl, SuSE
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#ifdef CONFIG_VT
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct proc_dir_entry *proc_brightness;

int omnibook_get_lcd_brightness(void)
{
	int retval;
	u8 brgt;

	switch (omnibook_ectype) {
	case XE3GF:
		if ((retval = omnibook_ec_read(XE3GF_BRTS, &brgt)))
			return retval;
		retval = brgt & XE3GF_BRTS_MASK;
		break;
	case XE3GC:
		if ((retval = omnibook_ec_read(XE3GC_BTVL, &brgt)))
			return retval;
		retval = brgt & XE3GC_BTVL_MASK;
		break;
	default:
		printk(KERN_INFO "%s: LCD brightness handling is " \
			"unsupported on this machine.\n", MODULE_NAME);
		retval = -ENODEV;
	}
	return retval;
}

int omnibook_set_lcd_brightness(u8 brgt)
{
	int retval;
	
	brgt = (brgt > 10) ? 10 : brgt;

	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_write(XE3GF_BRTS, brgt);
		if (retval)
			return retval;
	case XE3GC:
		retval = omnibook_ec_write(XE3GC_BTVL, brgt);
		if (retval)
			return retval;
		break;
	default:
		printk(KERN_INFO "%s: LCD brightness handling is unsupported on this machine.\n", MODULE_NAME);
		retval = -ENODEV;
	}
	return retval;	
}

static int omnibook_proc_brightness_status(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;
	int brgt;

	brgt = omnibook_get_lcd_brightness();

	b += sprintf(b, "LCD brightness: %2d\n", brgt);
	
	len = b - buffer;
	if (len <= off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

static int omnibook_proc_brightness_set(struct file *file, const char *buffer, unsigned long count, void *data)
{
	char value[4+1];
	int brgt = 0;
	int i;
	unsigned int len;
	
	len = sizeof(value) - 1;
	if (count < len)
		len = count;
	if (copy_from_user(value, buffer, len))
		return -EFAULT;
	value[len] = 0;

	if (strcmp(value, "off\n") == 0)
		omnibook_lcd_blank(1);
	else if (strcmp(value, "on\n") == 0)
		omnibook_lcd_blank(0);
	else
	{
		for (i = 0; (value[i] != '\0') && (value[i] != '\n'); i++)
			if ((value[i] >= '0') && (value[i] <= '9'))
				brgt = brgt * 10 + value[i] - '0';
			else
				return -EINVAL;

		if ((brgt >= 0) && (brgt <= 10))
			omnibook_set_lcd_brightness(brgt);
		else
			return -EINVAL;
	}

	return count;
}
#endif // (if defined CONFIG_VT)

int __init omnibook_brightness_init(void)
{
#ifdef CONFIG_VT
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
		pmode = S_IFREG | S_IWUSR | S_IRUGO;
		if (omnibook_userset)
			pmode = pmode | S_IWUGO;
		proc_brightness = create_proc_entry("lcd", pmode, omnibook_proc_root);
		break;
	default:
		printk(KERN_INFO "%s: LCD brightness handling is unsupported on this machine.\n", MODULE_NAME);
		return 0;
	}
	if (! proc_brightness) {
		printk(KERN_ERR "%s: Unable to create /proc/%s/lcd.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	} else {
		proc_brightness->read_proc = omnibook_proc_brightness_status;
		proc_brightness->write_proc = omnibook_proc_brightness_set;
	}
	printk(KERN_INFO "%s: LCD brightness handling is enabled.\n", MODULE_NAME);
	return 0;
#else
	printk(KERN_ERR "%s: Virtual terminal support is not compiled into your kernel.\n", MODULE_NAME);
	return -ENODEV;
#endif
}

#ifdef CONFIG_VT
void __exit omnibook_brightness_cleanup(void)
{
	if (proc_brightness)
		remove_proc_entry("lcd", omnibook_proc_root);
}
#endif
