/* g10.c - The GnuPG utility (main for gpg)
 *	Copyright (C) 1998, 1999 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

//#include <config.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

/* <IMMUNIX> */
#include <elf.h>
#include "cm_md5.h"
#include "cm.h"
/* </IMMUNIX> */

//#include "packet.h"
//#include "iobuf.h"
//#include "memory.h"
#include "util.h"
#include "main.h"
#include "options.h"
//#include "keydb.h"
#include "trustdb.h"
//#include "mpi.h"
//#include "cipher.h"
#include "filter.h"
//#include "ttyio.h"
//#include "i18n.h"
//#include "status.h"
//#include "g10defs.h"
//#include "hkp.h"

/* <IMMUNIX> */
#define NODUP
#define SECTION_STRTAB_ID ".shstrtab"
//static char digest_section_name[] = ".cm_sig";
static char digest_section_name[] = ".note.CM_digest";
/* </IMMUNIX> */

// static int utf8_strings = 0;
static int maybe_setuid = 1;

// static void print_hex( byte *p, size_t n );
// static void print_mds( const char *fname, int algo );
// static void add_notation_data( const char *string );
// static int  check_policy_url( const char *s );

static void
i18n_init(void)
{
  #ifdef USE_SIMPLE_GETTEXT
    set_gettext_file( PACKAGE );
  #else
  #ifdef ENABLE_NLS
    #ifdef HAVE_LC_MESSAGES
       setlocale( LC_TIME, "" );
       setlocale( LC_MESSAGES, "" );
    #else
       setlocale( LC_ALL, "" );
    #endif
    bindtextdomain( PACKAGE, G10_LOCALEDIR );
    textdomain( PACKAGE );
  #endif
  #endif
}

static void
set_debug(void)
{
    if( opt.debug & DBG_MEMORY_VALUE )
	memory_debug_mode = 1;
    if( opt.debug & DBG_MEMSTAT_VALUE )
	memory_stat_debug_mode = 1;
    if( opt.debug & DBG_MPI_VALUE )
	mpi_debug_mode = 1;
    if( opt.debug & DBG_CIPHER_VALUE )
	g10c_debug_mode = 1;
    if( opt.debug & DBG_IOBUF_VALUE )
	iobuf_debug_mode = 1;

}

static void
debug_digest (unsigned char digest[])
{
    log_debug ( "digest 0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n", 
		    digest[0], digest[1], digest[2], digest[3], digest[4],
		    digest[5], digest[6], digest[7], digest[8], digest[9],
		    digest[10], digest[11], digest[12], digest[13], digest[14],
		    digest[15]);
}

int 
add_section_name (FILE * file, Elf32_Shdr * elf_shdr)
{
	int retval;
	int i;
	char * strtab;
	char * entry;
	char * c;
	int is_shdr_strtab = FALSE;

	retval = read_string_table (&strtab, elf_shdr, file);
	if (retval != 0)
		 goto out;

	c = strtab;
	/* First byte is *supposed* to be NULL */
	c++;
	entry = c;
	for (i = 0, entry = c; (*entry) != '\0'; entry = c, i++)
	{
		if (strncmp(entry, SECTION_STRTAB_ID, sizeof(SECTION_STRTAB_ID) + 1) == 0)
		{
			is_shdr_strtab = TRUE;
			if (entry - strtab != elf_shdr->sh_name)
				log_debug ("offset: 0x%x name index: 0x%x\n", entry - strtab,
						elf_shdr->sh_name);
		}

		while ((*c) != '\0')
			c++;
		/* advance past the null terminator */
		c++;
	}
	retval = 0;

	if (is_shdr_strtab)
	{
		/* log_debug ("Found section header string table.\n");
		   log_debug ("elf header offset: 0x%x size: 0x%x offset+size: 0x%x\n", 
				elf_shdr->sh_offset, elf_shdr->sh_size, 
				elf_shdr->sh_offset + elf_shdr->sh_size);
		   log_debug ("computed offset from beginning of section: %x\n",
				entry - strtab); */
		shift_and_write (digest_section_name, sizeof (digest_section_name), 
				elf_shdr->sh_offset + elf_shdr->sh_size, file);

		retval = sizeof (digest_section_name);
	}
	
out:
	free (strtab);
	return retval;
}

int 
find_previous_note (Elf32_Ehdr * elf_ex, FILE * file, unsigned long *offset, 
		    unsigned long *descsz)
{
	Elf32_Shdr * elf_psnt, * elf_shdata = NULL;
	int retval;
	int i;

	*offset = 0;
	*descsz = 0;

#ifdef NODUP
	/* We're looking for a previous signature */
	retval = read_section_hdr (&elf_shdata, elf_ex, file, FALSE);
	if (retval != 0)
		goto out;

	for (i = 0, elf_psnt = elf_shdata; i < elf_ex->e_shnum; i++, elf_psnt++)
		if (elf_psnt->sh_type == SHT_NOTE) 
		{
			Elf32_Nhdr note;

			retval = read_elf_note (&note, elf_psnt, file);
			if (retval != 0)
				goto out;

			if (is_note_a_signature (&note, elf_psnt, file))
			{
				*offset = elf_psnt->sh_offset + 
					  sizeof (Elf32_Nhdr) + 
					  round_up_word (note.n_namesz);
				*descsz = note.n_descsz;
			}
		}
#endif			

	retval = 0;
out:
	if (elf_shdata)
		free (elf_shdata);
	return retval;
}

int 
add_section_hdr (unsigned long *checkoff, int * name_index, Elf32_Ehdr * elf_ex, FILE *file) 
{
	Elf32_Shdr * elf_psnt, * elf_shdata = NULL;
	int i;
	int retval = 0;
	int section_hdr_changed = FALSE;
	int name_offset = 0;
	int shstrtab_offset = 0;

	/* log_debug (__FUNCTION__ "\n"); */

	/* This is the section header area, the same process is done
	 * here looking for the end of the offsets but no MD5
	 * encoding is done */

	retval = read_section_hdr (&elf_shdata, elf_ex, file, TRUE);
	if (retval != 0)
		goto out;

	/* initialize starting search position for location (offset into
	 * file) to place the note to the end of the section header, adding
	 * one for the additional section header we'll be adding. */
	(*checkoff) = elf_ex->e_shoff + ((elf_ex->e_shnum + 1) * elf_ex->e_shentsize);
		
	/* Update each of the section header offsets by an additional section
	 * header. We'll have to move everything later to make room for the
	 * additional section header we're adding. -SMB */

	for(i = 0, elf_psnt = elf_shdata; i < elf_ex->e_shnum; i++, elf_psnt++) 
	{
		/* log_debug ("Section header %d type: %x\n", i,elf_psnt->sh_type); */
		switch (elf_psnt->sh_type) 
		{
		    case SHT_NOTE: 
		    case SHT_NULL: 
		    case SHT_SYMTAB:
		    case SHT_PROGBITS: 
		    case SHT_RELA: 
		    case SHT_HASH: 
		    case SHT_DYNAMIC:
		    case SHT_NOBITS: 
		    case SHT_SHLIB: 
		    case SHT_DYNSYM: 
		    case SHT_REL: 
		    case SHT_LOPROC:
		    case SHT_HIPROC: 
		    case SHT_LOUSER: 
		    case SHT_HIUSER: 
		    /* GNU (OS) specific section types
		     * (see /usr/include/elf.h) */ 
		    case SHT_GNU_verdef: 
		    case SHT_GNU_verneed:
		    case SHT_GNU_versym:
			break; 
		    case SHT_STRTAB: 
			/* COuld be the section name string table */
			{
			    retval = add_section_name (file, elf_psnt);
			    if (retval < 0)
				/* An error occurred */
				goto out;

			    if (retval > 0)
			    {
				name_offset = retval;

				/* Save the location of our added name */
				*name_index = elf_psnt->sh_size;

				/* increase the size by the amount we're shifting */
				elf_psnt->sh_size += name_offset;
				section_hdr_changed = TRUE;
				shstrtab_offset = elf_psnt->sh_offset;
			    }
			    break;
			}
		    default:
			{
			    printf (__FUNCTION__ ": Unknown section header %x\n",
					elf_psnt->sh_type);
			    retval = -1; 
			    goto out;
			    break;
			}
		}

		/* Update each section entry offset */
		if (elf_psnt->sh_offset > elf_ex->e_shoff)
		{
			elf_psnt->sh_offset += elf_ex->e_shentsize;
			section_hdr_changed = TRUE;
		}
		if ((name_offset > 0) && (elf_psnt->sh_offset > shstrtab_offset))
		{
			elf_psnt->sh_offset += name_offset;
			section_hdr_changed = TRUE;
		}
		if (*checkoff <= elf_psnt->sh_offset)
			*checkoff = elf_psnt->sh_offset + elf_psnt->sh_size;

	}

	/* Lateef - write all file sections to disk, this could be saved until
	 * all notes are completed but I put it here because the following
	 * sections don't really deal with these file sections anylonger */

	if (section_hdr_changed)
	{
		if ((name_offset > 0) && (elf_ex->e_shoff > shstrtab_offset))
		{
			/* The section header has been moved, better update the elf header */
			elf_ex->e_shoff += name_offset;
			retval = write_elf_hdr (elf_ex, file);
			if (retval != 0)
				goto out;
		}

		retval = write_section_hdr (elf_shdata, elf_ex, file);
		if (retval != 0)
			goto out;
	}

	retval = 0;
out:
	if (elf_shdata)
		free (elf_shdata);
	return retval;
}

int
do_encode_elf_binary(FILE **file, Elf32_Ehdr * elf_ex, unsigned char *filename, 
		     unsigned long * checkoff,  unsigned char digest[], 
		     unsigned long * size, unsigned long * offset, int * name_index)
{
	int retval;

	/* OPEN FILE */
	*file = fopen(filename, "r+");
	if (!file) 
	{
		log_error (__FUNCTION__ ": unable to open %s\n", filename);
		return -1;
	}

	retval = read_elf_hdr(elf_ex, *file);
	if (retval != 0) 
		return retval;

	/* Because part of the elf header is encoded in the MD5
	 * signature the header is updated and rewritten to the file
	 * here then decrimented for further calculations, thus the need
	 * for the seemingly gratuitous fwrite()s. */

#ifdef NODUP
	if (find_previous_note (elf_ex, *file, offset, size) != 0)
		return -1;

	if (*size == 0)
	{
#endif /* NODUP */
		elf_ex->e_shnum += 1;

		retval = write_elf_hdr(elf_ex, *file);
		if (retval != 0)
			return retval;
#ifdef NODUP
	}
#endif /* NODUP */

	/* Add the section header, and get the index of the name. This can
	 * modify the elf header which is part of the hashed data, so it
	 * MUST be performed before the hashing in compute_phdr_digest. */
	retval = add_section_hdr(checkoff, name_index, elf_ex, *file);
    	if (retval != 0)
	    	return retval;

	retval = compute_phdr_digest (checkoff, digest, elf_ex, *file);
	if (retval != 0)
		return retval;

	return 0;
}

int 
add_note (FILE * file, Elf32_Ehdr * elf_ex, long checkoff, int name_index,
	     unsigned char * description, long desc_len)
{
    	void * write_this;
	Elf32_Shdr * shdr;
	Elf32_Nhdr * note;
	unsigned char * name;
	unsigned char * desc;
	int retval = 0;

	/* create a note */
	note = (Elf32_Nhdr *) malloc (sizeof (Elf32_Nhdr));
	if (!note) {
		retval = -1;
		goto exit;
	}
	note->n_namesz = strlen (CRYPTOMARK_NAME);
	note->n_descsz = desc_len;
	note->n_type = 0;

	name = (unsigned char *) malloc (round_up_word (note->n_namesz)); 
	if (!name) {
		retval = -1;
		goto cleanup_note;
	}
	memset (name, 0, round_up_word (note->n_namesz));
	memcpy (name, CRYPTOMARK_NAME, note->n_namesz);
	
	desc = (unsigned char *) malloc (round_up_word (note->n_descsz));
	if (!desc) {
		retval = -1;
		goto cleanup_name;
	}
	memset (desc, 0, round_up_word (note->n_descsz));
	memcpy (desc, description, note->n_descsz);
	
	/* Constructing Section Header */
	shdr = (Elf32_Shdr *) malloc (sizeof (Elf32_Shdr));
	if (!shdr)
	{
		retval = -ENOMEM;
		goto cleanup_desc;
	}

	/* Lateef - This may be a problem, the .note name offset may not
	 * always be at 178, is it needed, probably not.  I am considering
	 * whether it should be 0, the null name */
	/* Actually, the correct the to do if we don't want to use a NULL name here would
	 * be to add the name to the section header symbol table. But I'm too lazy to do
	 * that yet. -SMB */
	//shdr->sh_name = 178;
	//shdr->sh_name = 0;
	shdr->sh_name = name_index;
	shdr->sh_type = SHT_NOTE;
	shdr->sh_flags = 0;
	shdr->sh_addr = 0;
	shdr->sh_offset = checkoff;
	shdr->sh_size = round_up_word (note->n_namesz) + 
	                round_up_word (note->n_descsz) + sizeof (Elf32_Nhdr);
	shdr->sh_link = SHN_UNDEF;
	shdr->sh_info = 0;
	shdr->sh_addralign = 1;
	shdr->sh_entsize = 0;

	shift_and_write ((void *) shdr, sizeof (Elf32_Shdr), 
	       elf_ex->e_shoff + ((elf_ex->e_shnum - 1) * elf_ex->e_shentsize),
	       file);

	write_this = malloc (sizeof (Elf32_Nhdr) + round_up_word (note->n_namesz)
		             + round_up_word (note->n_descsz));
	if (!write_this) {
		retval = -ENOMEM;
		goto cleanup_shdr;
	}	
	memcpy (write_this, note, sizeof (Elf32_Nhdr));
	memcpy (write_this + sizeof (Elf32_Nhdr), name, 
		round_up_word (note->n_namesz));
	memcpy (write_this + sizeof (Elf32_Nhdr) + round_up_word (note->n_namesz),
		desc, round_up_word (note->n_descsz));

	shift_and_write ((void *) write_this, 
		    sizeof (Elf32_Nhdr) + round_up_word (note->n_namesz) +
		    round_up_word (note->n_descsz), checkoff, file);

	free (write_this);
cleanup_shdr:
	free (shdr);
cleanup_desc:
	free (desc);
cleanup_name:
	free (name);
cleanup_note:
	free (note);
exit:
	return retval;
}

/* This function is for strictly rewriting the elf note section,
 * specifically the description which contains the md5. */
static int
quick_write (FILE * file, unsigned long offset, int descsz, 
	     unsigned char * description)
{
	unsigned char * desc;
	int retval = 0;

	desc = (unsigned char *) malloc (round_up_word (descsz));
	if (!desc) {
		retval = -1;
		goto exit;
	}
	memset (desc, 0, round_up_word (descsz));
	memcpy (desc, description, descsz);

	retval = fseek (file, offset, SEEK_SET);
	if (retval != 0)
		goto exit;

	retval = fwrite ((void *) desc, round_up_word(descsz), 1, file);
	if (retval != 1)
	{
		retval = -1;
		goto exit;
	}

	retval = 0;
exit:
	return retval;
}

int
main( int argc, char **argv )
{
    /* <IMMUNIX> */
    long checkoff;
    unsigned long desc_len;
    unsigned char *Desc=NULL;
    Elf32_Ehdr elf_ex;
    unsigned char digest[17];
    FILE * file;
    unsigned long descsz;
    unsigned long offset;
    int retval;
    int name_index;
    /* </IMMUNIX> */

    STRLIST sl;
    STRLIST nrings=NULL, sec_nrings=NULL;
    FILE *configfp = NULL;
    char *configname = NULL;
    unsigned configlineno;
    int parse_debug = 0;
    int default_config =1;
    int default_keyring = 1;

  #ifdef USE_SHM_COPROCESSING
    ulong requested_shm_size=0;
  #endif

    trap_unaligned();
    secmem_set_flags( secmem_get_flags() | 2 ); /* suspend warnings */
    /* Please note that we may running SUID(ROOT), so be very CAREFUL
     * when adding any stuff between here and the call to
     * secmem_init()  somewhere after the option parsing
     */
    log_set_name("cm_insert");
    secure_random_alloc(); /* put random number into secure memory */
    disable_core_dumps();
    init_signals();
    create_dotlock(NULL); /* register locking cleanup */
    i18n_init();
    opt.compress = -1; /* defaults to standard compress level */
    /* note: if you change these lines, look at oOpenPGP */
    opt.def_cipher_algo = 0;
    opt.def_digest_algo = 0;
    opt.def_compress_algo = 2;
    opt.s2k_mode = 3; /* iterated+salted */
    opt.s2k_digest_algo = DIGEST_ALGO_RMD160;
    opt.s2k_cipher_algo = CIPHER_ALGO_BLOWFISH;
    opt.completes_needed = 1;
    opt.marginals_needed = 3;
    opt.max_cert_depth = 5;
  #ifdef __MINGW32__
    opt.homedir = read_w32_registry_string( NULL, "Software\\GNU\\GnuPG", "HomeDir" );
  #else
    opt.homedir = getenv("GNUPGHOME");
  #endif
    if( !opt.homedir || !*opt.homedir ) {
      #ifdef HAVE_DRIVE_LETTERS
	opt.homedir = "c:/cryptomark";
      #else
	opt.homedir = "/etc/cryptomark";
      #endif
    }

    /* initialize the secure memory. */
    secmem_init( 16384 );
    maybe_setuid = 0;
    /* Okay, we are now working under our real uid */

    //if( default_config )
	configname = make_filename(opt.homedir, "options", NULL );

    if( configname ) {
	configlineno = 0;
	configfp = fopen( configname, "r" );
	if( !configfp ) {
	    log_error(("option file `%s': %s\n"),
				    configname, strerror(errno) );
	    g10_exit(2);
	    m_free(configname); configname = NULL;
	}
	if( parse_debug && configname )
	    log_info(("reading options from `%s'\n"), configname );
	default_config = 0;
    }

    /* <IMMUNIX> */
    opt.force_v3_sigs = 1;
    opt.escape_from = 1;
    opt.lock_once = 1;
    /* </IMMUNIX> */

    if( configfp ) {
	fclose( configfp );
	configfp = NULL;
	m_free(configname); configname = NULL;
    }
    m_free( configname ); configname = NULL;

    secmem_set_flags( secmem_get_flags() & ~2 ); /* resume warnings */

    set_debug();

    if( !sec_nrings && default_keyring )  /* add default secret rings */
	    add_keyblock_resource("secring.gpg", 0, 1);
    for(sl = sec_nrings; sl; sl = sl->next )
	    add_keyblock_resource( sl->d, 0, 1 );
    if( !nrings && default_keyring )  /* add default ring */
	    add_keyblock_resource("pubring.gpg", 0, 0);
    for(sl = nrings; sl; sl = sl->next )
	    add_keyblock_resource( sl->d, 0, 0 );
    FREE_STRLIST(nrings);
    FREE_STRLIST(sec_nrings);

/* <IMMUNIX> */
    if (argc == 2) 
    {
    	retval = do_encode_elf_binary(&file, &elf_ex, argv[1], &checkoff, 
			              digest, &descsz, &offset, &name_index);
    	if (retval != 0)
	{
	    log_error ("0 an error occurred\n");
	    return retval;
	}
    }	
    else
    {
	    printf ("Invalid # of arguments\n");
	    return -1;
    }

    debug_digest(digest);
    cm_sign_data (digest, 16, &Desc, &desc_len, NULL);
    if (retval != 0)
    {
	log_error ("1 an error occurred\n");
	return retval;
    }

#ifdef NODUP
    if ((descsz != 0) &&  (descsz != desc_len)) {
	printf ("Descsz %ld != desc_len %ld\n", descsz, desc_len);
	return -1;
    }
    if ((descsz == 0)/* || (descsz != desclength)*/) {
#endif /* NODUP */
	retval = add_note (file, &elf_ex, checkoff, name_index, Desc, desc_len);
    	if (retval != 0)
	{
	    log_error ("2 an error occurred\n");
	    return retval;
	}
#ifdef NODUP
    }
    else
    {
	retval = quick_write (file, offset, descsz, Desc);
    	if (retval != 0)
	{
	    log_error ("an error occurred\n");
	    return retval;
	}
    }
#endif /* NODUP */

    fclose (file);

    return 0;
/* </IMMUNIX> */
}

