/* valaversion.vala
 *
 * Copyright (C) 2018  Rico Tzschichholz
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Rico Tzschichholz <ricotz@ubuntu.com>
 */

namespace Vala {
	/**
	 * Like get_major_version, but from the headers used at application compile time,
	 * rather than from the library linked against at application run time
	 */
	public const int MAJOR_VERSION = 0;
	/**
	 * Like get_minor_version, but from the headers used at application compile time,
	 * rather than from the library linked against at application run time
	 */
	public const int MINOR_VERSION = 54;
	/**
	 * Like get_micro_version, but from the headers used at application compile time,
	 * rather than from the library linked against at application run time
	 */
	public const int MICRO_VERSION = 6;

	/**
	 * The API version string
	 */
	public const string API_VERSION = "0.54";

	/**
	 * The full build-version string generated by the build-system
	 */
	public const string BUILD_VERSION = "0.54.6";

	/**
	 * Returns the major version number of the vala library.
	 *
	 * This function is in the library, so it represents the GTK+
	 * library your code is running against.
	 *
	 * @return the major version number of the vala library
	 */
	public uint get_major_version () {
		return MAJOR_VERSION;
	}

	/**
	 * Returns the minor version number of the vala library.
	 *
	 * This function is in the library, so it represents the vala
	 * library your code is are running against.
	 *
	 * @return the minor version number of the vala library
	 */
	public uint get_minor_version () {
		return MINOR_VERSION;
	}

	/**
	 * Returns the micro version number of the vala library.
	 *
	 * This function is in the library, so it represents the vala
	 * library your code is running against.
	 *
	 * @return the micro version number of the vala library
	 */
	public uint get_micro_version () {
		return MICRO_VERSION;
	}

	/**
	 * Returns the full build-version string of the vala library.
	 *
	 * This function is in the library, so it represents the vala
	 * library your code is running against.
	 *
	 * @return the full build-version string of the vala library
	 */
	public unowned string get_build_version () {
		return BUILD_VERSION;
	}

	/**
	 * Checks that the vala library in use is compatible with the given version.
	 *
	 * This function is in the library, so it represents the vala
	 * library your code is running against.
	 *
	 * @param required_major the required major version
	 * @param required_minor the required minor version
	 * @param required_micro the required micro version
	 * @return null if the vala library is compatible with the given version,
	 * or a string describing the version mismatch.
	 */
	public unowned string? check_version (uint required_major, uint required_minor, uint required_micro)
	{
		uint effective_micro = 100 * MINOR_VERSION + MICRO_VERSION;
		uint required_effective_micro = 100 * required_minor + required_micro;

		if (required_major > MAJOR_VERSION)
			return "vala version too old (major mismatch)";

		if (required_effective_micro > effective_micro)
			return "vala version too old (micro mismatch)";

		return null;
	}
}
