/*
 * Copyright (C) 2002-2003 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: skin_window.c,v 1.5 2003/03/14 20:47:05 guenter Exp $
 *
 * cool fully-user-skinned windows, generate from xml/js code
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

#include "skin_window.h"
#include "utils.h"
#include "actions.h"
#include "xmlparser.h"

typedef struct view_s view_t;

struct view_s {

  char       *basename;

  char       *id;
  int         w, h;

  gint        x, y;
  gboolean    is_moving;
  gboolean    repaint_needed;

  GList      *widgets;

  GtkWidget  *window;
  GdkPixmap  *bg;
  GdkBitmap  *mask;
};

#define WT_SUBVIEW     0
#define WT_BUTTON      1
#define WT_BUTTONGROUP 2
#define WT_SLIDER      3

typedef struct skin_widget_s skin_widget_t;

struct skin_widget_s {

  view_t    *parent;

  char      *id;

  char      *xscript, *yscript;
  int        x, y, w, h;
  int        zIndex;
  int        type;

  void (*layout)             (skin_widget_t *w, int xo, int yo);
  void (*paint)              (skin_widget_t *w);
  void (*handle_motion)      (skin_widget_t *w, int px, int py);
  gboolean (*handle_button_press) (skin_widget_t *w, int px, int py);
  void (*handle_button_release)   (skin_widget_t *w, int px, int py);
};

typedef struct subview_s subview_t;

struct subview_s {

  skin_widget_t widget;

  gboolean   tiled;
  char      *halign, *valign;
  /* tiled borders limits: */
  char      *h_max_script, *v_max_script;
  int        h_max, v_max;

  GdkPixbuf *pix;

  GList     *widgets;
};

typedef struct buttongroup_s buttongroup_t;

struct buttongroup_s {

  skin_widget_t  widget;

  GdkPixbuf     *pix, *pix_hover, *pix_down, *pix_hover_down;
  GdkPixbuf     *pix_map;
  
  GList         *buttons;
};

typedef struct {

  buttongroup_t *parent;

  uint32_t       color;
  char          *id;
  char          *script;

  GdkPixbuf     *pix, *pix_hover, *pix_down, *pix_hover_down;

  gboolean       pressed, hovered;

} button_element_t;

#define MAXPATH 1024

static int script_view_cb (void *view_gen, char *prop) {

  view_t *view = (view_t *) view_gen;

  if (!strcasecmp (prop, "width")) {
    return view->w;
  } else if (!strcasecmp (prop, "height")) {
    return view->h;
  }

  printf ("skin_window: error, unknown prop %s\n", prop);
  return -1;
}

static int se_eval_int (char *script, view_t *view) {

  if (!script)
    return 0;

  if (!strncasecmp (script, "jscript:", 8)) 
    return action_eval_view (&script[8], script_view_cb, view);
  else 
    return atoi (script);
}

gint widgetzcomp (gconstpointer a, gconstpointer b) {

  skin_widget_t *wa, *wb;

  wa = (skin_widget_t *) a;
  wb = (skin_widget_t *) b;

  return wa->zIndex - wb->zIndex;
}

static void my_gdk_pixbuf_render_threshold_alpha (GdkPixbuf *pixbuf,
						  GdkBitmap *bitmap,
						  int src_x,  int src_y,
						  int dest_x, int dest_y,
						  int width,  int height,
						  int alpha_threshold) {
  GdkGC    *gc;
  GdkColor  color;
  int       x, y;
  guchar   *p;
  int       start, start_status;
  int       status;

  if (width == -1) 
    width = gdk_pixbuf_get_width(pixbuf);
  if (height == -1)
    height = gdk_pixbuf_get_height(pixbuf);

  gc = gdk_gc_new (bitmap);

  if (!gdk_pixbuf_get_has_alpha (pixbuf)) {
    color.pixel = (alpha_threshold == 255) ? 0 : 1;
    gdk_gc_set_foreground (gc, &color);
    gdk_draw_rectangle (bitmap, gc, TRUE, dest_x, dest_y, width, height);
    g_object_unref (gc);
    return;
  }

  color.pixel = 0;
  gdk_gc_set_foreground (gc, &color);

  color.pixel = 1;
  gdk_gc_set_foreground (gc, &color);

  for (y = 0; y < height; y++) {
    p = (gdk_pixbuf_get_pixels(pixbuf) 
	 + (y + src_y) * gdk_pixbuf_get_rowstride(pixbuf) 
	 + src_x * gdk_pixbuf_get_n_channels(pixbuf)
	 + gdk_pixbuf_get_n_channels(pixbuf) - 1);
    
    start = 0;
    start_status = *p < alpha_threshold;
    
    for (x = 0; x < width; x++) {
      status = *p < alpha_threshold;
      
      if (status != start_status) {
	if (!start_status)
	  gdk_draw_line (bitmap, gc,
			 start + dest_x, y + dest_y,
			 x - 1 + dest_x, y + dest_y);
	
	start = x;
	start_status = status;
      }
      
      p += gdk_pixbuf_get_n_channels(pixbuf);
    }
    
    if (!start_status)
      gdk_draw_line (bitmap, gc,
		     start + dest_x, y + dest_y,
		     x - 1 + dest_x, y + dest_y);
  }
  
  g_object_unref (gc);
}

static GdkPixbuf *load_pix_trans (xml_node_t *xml, char *attr, char *basename) {

  char      *img_name;
  GdkPixbuf *pix;
  char       filename[MAXPATH];

  img_name = xml_parser_get_property (xml, attr);
  if (!img_name)
    return NULL;

  snprintf (filename, MAXPATH, "%s/%s", basename, img_name);

  pix = gdk_pixbuf_new_from_file (filename, NULL);

  if (!pix) 
    return NULL;
  else {
    char     *transparency_color_str;
    uint32_t  color;

    transparency_color_str = xml_parser_get_property (xml, "transparencyColor");
    if ( transparency_color_str 
	 && (sscanf (transparency_color_str, "#%x", &color) == 1) ) {
      int r, g, b;
      
      r = (color & 0xff0000) >> 16;
      g = (color & 0x00ff00) >>  8;
      b =  color & 0x0000ff ;
      
      pix = gdk_pixbuf_add_alpha (pix, TRUE, r, g, b);
    } 
  }
  return pix;
}

static GdkPixbuf *load_pix (xml_node_t *xml, char *attr, char *basename) {

  char      *img_name;
  GdkPixbuf *pix;
  char       filename[MAXPATH];

  img_name = xml_parser_get_property (xml, attr);
  if (!img_name)
    return NULL;

  snprintf (filename, MAXPATH, "%s/%s", basename, img_name);

  pix = gdk_pixbuf_new_from_file (filename, NULL);

  return pix;
}

static int is_inside (int px, int py, int x, int y, int w, int h) {

  return ( (px>=x) && (py>=y) && (px<=(x+w)) && (py<=(y+h)) );
}

static int is_inside_map (GdkPixbuf *map, uint32_t color, int px, int py) {

  uint8_t *p;
  uint32_t c;

  p = (gdk_pixbuf_get_pixels (map) 
       + py * gdk_pixbuf_get_rowstride (map)
       + px * gdk_pixbuf_get_n_channels (map)) ;

  c = *(p+2) | (*(p+1) << 8) | (*p) << 16;

  return (c==color);
}

static GdkPixbuf *gdk_pixmap_extract_map (GdkPixbuf *pix_src, GdkPixbuf *map,
					  uint32_t color) {

  GdkPixbuf *pix;
  int        x,y,width,height,n_channels, nc;
  uint8_t   *ps, *pd, *pm;
  
  if (!pix_src)
    return NULL;
  if (!map)
    return NULL;

  width      = gdk_pixbuf_get_width      (pix_src);
  height     = gdk_pixbuf_get_height     (pix_src);
  n_channels = gdk_pixbuf_get_n_channels (pix_src);

  pix = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (pix_src),
			TRUE, /* alpha */
			gdk_pixbuf_get_bits_per_sample (pix_src),
			width, height);

  for (y = 0; y < height; y++) {
    int map_color;

    ps = (gdk_pixbuf_get_pixels (pix_src) 
	  + y * gdk_pixbuf_get_rowstride (pix_src)) ;
    pd = (gdk_pixbuf_get_pixels (pix) 
	  + y * gdk_pixbuf_get_rowstride (pix)) ;
    pm = (gdk_pixbuf_get_pixels (map) 
	  + y * gdk_pixbuf_get_rowstride (map)) ;

    for (x = 0; x < width; x++) {

      map_color = *(pm+2) | (*(pm+1) << 8) | (*pm) << 16;
    
      pm += gdk_pixbuf_get_n_channels (map);
    
      for (nc = 0; nc<n_channels; nc++) {
	if (map_color == color)
	  *pd = *ps;
	else
	  *pd = 0x00;
      
	pd++; ps++;
      }

      if (!gdk_pixbuf_get_has_alpha (pix_src)) {
	if (map_color == color)
	  *pd = 0xff;
	else
	  *pd = 0x00;
	pd++; 
      }
    }
  }

  return pix;
}

/*
 *
 * button element stuff
 *
 */

static button_element_t *create_button_element (xml_node_t *xbutton_element, 
						buttongroup_t *group) {

  button_element_t *button_element;
  char             *color_str;

  button_element = (button_element_t *) malloc (sizeof (button_element_t));

  button_element->parent  = group;
  button_element->id      = xml_parser_get_property     (xbutton_element, "id");
  button_element->script  = xml_parser_get_property     (xbutton_element, "onClick");
  
  color_str = xml_parser_get_property (xbutton_element, "mappingColor");

  button_element->color = 0;

  if ( color_str )
    sscanf (color_str, "#%x", &button_element->color);

  button_element->pix = gdk_pixmap_extract_map (group->pix, group->pix_map,
						button_element->color);
  button_element->pix_hover = gdk_pixmap_extract_map (group->pix_hover, group->pix_map,
						button_element->color);
  button_element->pix_down = gdk_pixmap_extract_map (group->pix_down, group->pix_map,
						button_element->color);
  button_element->pix_hover_down = gdk_pixmap_extract_map (group->pix_hover_down, 
							   group->pix_map,
							   button_element->color);

  button_element->pressed = FALSE;
  button_element->hovered = FALSE;

  return button_element;
}


/*
 *
 * buttongroup stuff
 *
 */

static void buttongroup_paint (skin_widget_t *w) {
  buttongroup_t *bg = (buttongroup_t *) w;
  view_t    *view = bg->widget.parent;

  if (bg->pix) {

    GList *n;
    
    n = g_list_first (bg->buttons);
    while (n) {
      button_element_t *b;

      b = (button_element_t *) n->data;

      if (b->hovered) {
	
	if (b->pressed) {
#if 0 /* FIXME */
	  if (b->pix_hover_down) {
	    printf ("using hover down image\n");
	    gdk_pixbuf_render_to_drawable (b->pix_hover_down, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  } else 
#endif
	    if (b->pix_down)
	    gdk_pixbuf_render_to_drawable (b->pix_down, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  else if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	} else { /* !pressed */

	  if (b->pix_hover)
	    gdk_pixbuf_render_to_drawable (b->pix_hover, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  else if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	}

      } else { /* !hovered */

	if (b->pressed) {
	  if (b->pix_down)
	    gdk_pixbuf_render_to_drawable (b->pix_down, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  else if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	  
	} else { /* !pressed */

	  if (b->pix)
	    gdk_pixbuf_render_to_drawable (b->pix, view->bg, NULL, 0, 0,
					   bg->widget.x, 
					   bg->widget.y,
					   -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
	}

      }

      n = g_list_next (n);
    }
  }
}

static void buttongroup_layout (skin_widget_t *w, int xo, int yo) {
  buttongroup_t *bg = (buttongroup_t *) w;

  if (bg->pix) {
    bg->widget.x  = se_eval_int (bg->widget.xscript, bg->widget.parent) + xo;
    bg->widget.y  = se_eval_int (bg->widget.yscript, bg->widget.parent) + yo;
  } 
}

static gboolean buttongroup_handle_button_press (skin_widget_t *w, int px, int py) {

  buttongroup_t *bg = (buttongroup_t *) w;
  gboolean       is_inside_group;
  GList         *n;

  is_inside_group = is_inside (px, py, w->x, w->y, w->w, w->h);

  n = g_list_first (bg->buttons);
  while (n) {
    button_element_t *b;
    
    b = (button_element_t *) n->data;
    
    if (is_inside_group && is_inside_map (bg->pix_map, b->color, px-w->x, py-w->y)) {

      if (!b->pressed) {
	b->pressed = TRUE;
	bg->widget.parent->repaint_needed = TRUE;
      }

    } else {
      
      if (b->pressed) {
	b->pressed = FALSE;
	bg->widget.parent->repaint_needed = TRUE;
      }
    }
    
    n = g_list_next (n);
  }

  return is_inside_group;
}

static void buttongroup_handle_button_release (skin_widget_t *w, int px, int py) {

  buttongroup_t *bg = (buttongroup_t *) w;
  gboolean       is_inside_group;
  GList         *n;

  is_inside_group = is_inside (px, py, w->x, w->y, w->w, w->h);

  n = g_list_first (bg->buttons);
  while (n) {
    button_element_t *b;
    
    b = (button_element_t *) n->data;
    
    if (is_inside_group && is_inside_map (bg->pix_map, b->color, px-w->x, py-w->y)) {

      printf ("skin_window: action: %s\n", b->script);

      action_exec (b->script, NULL, NULL);

    }

    if (b->pressed) {
      b->pressed = FALSE;
      bg->widget.parent->repaint_needed = TRUE;
    }
    
    n = g_list_next (n);
  }
}

static void buttongroup_handle_motion (skin_widget_t *w, int px, int py) {

  buttongroup_t *bg = (buttongroup_t *) w;
  gboolean       is_inside_group;
  GList         *n;

  is_inside_group= is_inside (px, py, w->x, w->y, w->w, w->h);

  n = g_list_first (bg->buttons);
  while (n) {
    button_element_t *b;
    
    b = (button_element_t *) n->data;
    
    if (is_inside_group && is_inside_map (bg->pix_map, b->color, px-w->x, py-w->y)) {

      if (!b->hovered) {
	b->hovered = TRUE;
	bg->widget.parent->repaint_needed = TRUE;
      }
      
    } else {
      
      if (b->hovered) {
	b->hovered = FALSE;
	bg->widget.parent->repaint_needed = TRUE;
      }
      
    }
    
    n = g_list_next (n);
  }
}

static buttongroup_t *create_buttongroup (xml_node_t *xbuttongroup, view_t *view) {

  buttongroup_t *buttongroup;
  xml_node_t    *node;

  buttongroup = (buttongroup_t *) malloc (sizeof (buttongroup_t));

  buttongroup->widget.parent  = view;
  buttongroup->widget.id      = xml_parser_get_property     (xbuttongroup, "id");
  buttongroup->widget.zIndex  = xml_parser_get_property_int (xbuttongroup, "zIndex", 0);

  buttongroup->widget.xscript = xml_parser_get_property     (xbuttongroup, "left");
  buttongroup->widget.yscript = xml_parser_get_property     (xbuttongroup, "top");

  buttongroup->widget.layout                = buttongroup_layout;
  buttongroup->widget.paint                 = buttongroup_paint;
  buttongroup->widget.handle_motion         = buttongroup_handle_motion;
  buttongroup->widget.handle_button_press   = buttongroup_handle_button_press;
  buttongroup->widget.handle_button_release = buttongroup_handle_button_release;

  buttongroup->pix       = load_pix_trans (xbuttongroup, "image", view->basename);
  buttongroup->pix_hover = load_pix_trans (xbuttongroup, "hoverImage", view->basename);
  buttongroup->pix_down  = load_pix_trans (xbuttongroup, "downImage", view->basename);
  buttongroup->pix_hover_down = load_pix_trans (xbuttongroup, "hoverDownImage", 
						view->basename);
  buttongroup->pix_map   = load_pix (xbuttongroup, "mappingImage", view->basename);

  if (buttongroup->pix) {
    buttongroup->widget.w = gdk_pixbuf_get_width  (buttongroup->pix);
    buttongroup->widget.h = gdk_pixbuf_get_height (buttongroup->pix);
  } else {
    buttongroup->widget.w = 0;
    buttongroup->widget.h = 0;
  }

  buttongroup->buttons = NULL;

  node = xbuttongroup->child;
  while (node) {

    if (!strcasecmp (node->name, "buttonelement")) {

      button_element_t *button_element;

      button_element = create_button_element (node, buttongroup);

      buttongroup->buttons = g_list_append (buttongroup->buttons, button_element);
    } 

    node = node->next;
  }


  return buttongroup;
}

/*
 *
 * subview stuff
 *
 */

static void subview_layout (skin_widget_t *w, int xo, int yo) {
  subview_t  *s = (subview_t *) w;
  GList      *n;

  s->widget.x  = se_eval_int (s->widget.xscript, s->widget.parent) + xo;
  s->widget.y  = se_eval_int (s->widget.yscript, s->widget.parent) + yo;

  if (s->pix) {
    s->h_max     = se_eval_int (s->h_max_script, s->widget.parent);
    s->v_max     = se_eval_int (s->v_max_script, s->widget.parent);
  }

  /*
   * layout children
   */

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->layout (w, s->widget.x, s->widget.y);

    n = g_list_next (n);
  }
}


static void subview_paint (skin_widget_t *w) {
  subview_t *s = (subview_t *) w;
  view_t    *view = s->widget.parent;
  GList      *n;

  if (s->pix) {
    gboolean finished;
    gint     xoff, yoff;
    
    finished = 0; xoff = 0; yoff = 0;
    do {
      gdk_pixbuf_render_to_drawable (s->pix, view->bg, NULL, 0, 0,
				     s->widget.x+xoff, 
				     s->widget.y+yoff,
				     -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
      
      my_gdk_pixbuf_render_threshold_alpha (s->pix, view->mask,
					    0, 0, 
					    s->widget.x+xoff, 
					    s->widget.y+yoff,
					    -1, -1, 1);
      if (s->tiled) {
	if (s->halign && !strcasecmp (s->halign, "stretch")) {
	  xoff += gdk_pixbuf_get_width (s->pix);
	  
	  if ((xoff+s->widget.x) > s->h_max)
	    finished = TRUE;
	} else if (s->valign && !strcasecmp (s->valign, "stretch")) {
	  yoff += gdk_pixbuf_get_height (s->pix);
	  
	  if ((yoff+s->widget.y) > s->v_max) 
	    finished = TRUE;
	} else
	  finished = TRUE;
      } else
	finished = TRUE;
    } while (!finished);
  }

  /*
   * paint children
   */

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->paint (w);

    n = g_list_next (n);
  }
}

static void subview_handle_button_release (skin_widget_t *w, int px, int py) {

  subview_t *s = (subview_t *) w;
  GList     *n;

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->handle_button_release (w, px, py);

    n = g_list_next (n);
  }
  return FALSE;
}

static gboolean subview_handle_button_press (skin_widget_t *w, int px, int py) {

  subview_t *s = (subview_t *) w;
  GList     *n;

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    if (w->handle_button_press (w, px, py))
      return TRUE;

    n = g_list_next (n);
  }
  return FALSE;
}

static void subview_handle_motion (skin_widget_t *w, int px, int py) {
  subview_t *s = (subview_t *) w;
  GList     *n;

  n = g_list_first (s->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->handle_motion (w, px, py);

    n = g_list_next (n);
  }
}

static subview_t *create_subview (xml_node_t *xsubview, view_t *view) {

  subview_t  *subview;
  xml_node_t *node;

  subview = (subview_t *) malloc (sizeof (subview_t));

  subview->widget.parent  = view;
  subview->widget.id      = xml_parser_get_property     (xsubview, "id");
  subview->widget.zIndex  = xml_parser_get_property_int (xsubview, "zIndex", 0);

  subview->widget.xscript = xml_parser_get_property     (xsubview, "left");
  subview->widget.yscript = xml_parser_get_property     (xsubview, "top");

  subview->widget.layout                = subview_layout;
  subview->widget.paint                 = subview_paint;
  subview->widget.handle_motion         = subview_handle_motion;
  subview->widget.handle_button_press   = subview_handle_button_press;
  subview->widget.handle_button_release = subview_handle_button_release;

  subview->tiled   = xml_parser_get_property_bool(xsubview, "backgroundTiled", FALSE);
  subview->halign  = xml_parser_get_property     (xsubview, "horizontalAlignment");
  subview->valign  = xml_parser_get_property     (xsubview, "verticalAlignment");
  subview->h_max_script = xml_parser_get_property(xsubview, "horizontalLimit");
  subview->v_max_script = xml_parser_get_property(xsubview, "verticalLimit");

  subview->pix = load_pix_trans (xsubview, "backgroundImage", view->basename);

  subview->widgets = NULL;

  node = xsubview->child;
  while (node) {

    if (!strcasecmp (node->name, "subview")) {

      subview_t *subview;

      subview = create_subview (node, view);

      subview->widgets = g_list_insert_sorted (subview->widgets, subview, widgetzcomp);
    } else if (!strcasecmp (node->name, "buttongroup")) {
      buttongroup_t *buttongroup;

      buttongroup = create_buttongroup (node, view);

      subview->widgets = g_list_insert_sorted (subview->widgets, buttongroup, widgetzcomp);
      
    }

    node = node->next;
  }


  return subview;
}

static void paint_view (view_t *view) {

  GList      *n;

  /*
   * compute widget positions
   */

  n = g_list_first (view->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->layout (w, 0, 0);

    n = g_list_next (n);
  }

  /*
   * render widgets
   */

  n = g_list_first (view->widgets);
  while (n) {
    skin_widget_t *w;

    w = (skin_widget_t *) n->data;

    w->paint (w);

    n = g_list_next (n);
  }
}

static void view_motion_cb (GtkWidget *widget, GdkEventMotion *event, 
			    gpointer callback_data) {
  view_t    *view = (view_t *) callback_data;
  XEvent     ev; 
  gint       i = 0;

  XSync (gdk_display, False);
  
  while (XCheckTypedEvent (GDK_DISPLAY(), MotionNotify, &ev)) {
    event->x = ev.xmotion.x;
    event->y = ev.xmotion.y;
    i++;
  }
  
  if (view->is_moving) {
    gint mx, my, newx, newy;
    GdkModifierType modmask;
    
    gdk_window_get_pointer (NULL, &mx, &my, &modmask);
    
    newx = mx - view->x;
    newy = my - view->y;
    gdk_window_move (view->window->window, newx, newy);

  } else {
    GList *n;
    gint mx, my;

    view->repaint_needed = FALSE;

    mx = event->x; my = event->y;
    
    n = g_list_first (view->widgets);
    while (n) {
      skin_widget_t *w;

      w = (skin_widget_t *) n->data;
      
      w->handle_motion (w, mx, my);
      
      n = g_list_next (n);
    }

    if (view->repaint_needed) {
      paint_view (view);

      gdk_window_set_back_pixmap (view->window->window, view->bg, 0);
      gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 
      gdk_window_clear (view->window->window);
    }
  }
  
  gdk_flush();
}

static void view_button_release_cb (GtkWidget * widget, GdkEventButton * event, 
				    gpointer callback_data) {

  view_t    *view = (view_t *) callback_data;
  GList    *n;
  gint      mx, my;

  view->repaint_needed = FALSE;

  mx = event->x; my = event->y; 
  
  n = g_list_first (view->widgets);
  while (n) {
    skin_widget_t *w;
    
    w = (skin_widget_t *) n->data;
      
    w->handle_button_release (w, mx, my);
      
    n = g_list_next (n);
  }

  if (view->repaint_needed) {
    paint_view (view);
    
    gdk_window_set_back_pixmap (view->window->window, view->bg, 0);
    gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 
    gdk_window_clear (view->window->window);
  }

  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  gdk_flush ();
  
  view->is_moving = FALSE;
}

static gboolean view_button_press_cb (GtkWidget *widget, GdkEventButton *event, 
				      gpointer data) {

  view_t *view = (view_t *) data;

  if (event && (event->type==GDK_BUTTON_PRESS)) {
    
    if (event->button == 1) {

      GList    *n;
      gint      mx, my;
      gboolean  handled;

      view->repaint_needed = FALSE;

      mx = event->x; my = event->y; handled = FALSE;
    
      n = g_list_first (view->widgets);
      while (n) {
	skin_widget_t *w;
	
	w = (skin_widget_t *) n->data;
      
	handled |= w->handle_button_press (w, mx, my);
      
	n = g_list_next (n);
      }

      if (view->repaint_needed) {
	paint_view (view);
	
	gdk_window_set_back_pixmap (view->window->window, view->bg, 0);
	gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 
	gdk_window_clear (view->window->window);
      }
      
      if (!handled) {
	view->is_moving = TRUE;
	view->x = event->x;
	view->y = event->y;
      }
    }
  }

  return TRUE;
}

static view_t *create_view (xml_node_t *xview, const char *basename,
			    int view_num) {

  xml_node_t *node;
  view_t     *view;

  view = (view_t *) malloc (sizeof (view_t));

  view->w         = xml_parser_get_property_int (xview, "width", 0);
  view->h         = xml_parser_get_property_int (xview, "height", 0);
  view->id        = xml_parser_get_property     (xview, "id");
  view->is_moving = FALSE;

  if (!view->w || !view->h) {
    free (view);
    return NULL;
  }

  view->basename = strdup (basename);
  view->widgets  = NULL;

  node = xview->child;
  while (node) {

    if (!strcasecmp (node->name, "subview")) {

      subview_t *subview;

      subview = create_subview (node, view);

      view->widgets = g_list_insert_sorted (view->widgets, subview, widgetzcomp);
    } else if (!strcasecmp (node->name, "buttongroup")) {
      buttongroup_t *buttongroup;

      buttongroup = create_buttongroup (node, view);

      view->widgets = g_list_insert_sorted (view->widgets, buttongroup, widgetzcomp);
      
    }

    node = node->next;
  }

  view->window =  gtk_window_new(GTK_WINDOW_TOPLEVEL);

  gtk_widget_set_app_paintable (view->window, TRUE);

  gtk_widget_set_usize (view->window, view->w, view->h);

  gtk_widget_add_events (view->window, GDK_POINTER_MOTION_MASK 
			 | GDK_BUTTON_MOTION_MASK    
			 | GDK_BUTTON1_MOTION_MASK
			 | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK 
			 | GDK_KEY_PRESS_MASK);

  g_signal_connect (G_OBJECT (view->window), "button_press_event",
		    G_CALLBACK (view_button_press_cb), view);
  g_signal_connect (G_OBJECT (view->window), "button_release_event",
		    G_CALLBACK (view_button_release_cb), view);
  g_signal_connect (G_OBJECT (view->window), "motion_notify_event",
		    G_CALLBACK (view_motion_cb), view);

  gtk_widget_realize (view->window);

  view->bg   = gdk_pixmap_new (view->window->window, view->w, view->h, -1);
  view->mask = gdk_pixmap_new (view->window->window, view->w, view->h, 1);

  {
    GdkGC *gc;
    
    gc = gdk_gc_new (view->mask);
    
    gdk_draw_rectangle (view->mask, gc, TRUE, 0, 0, view->w, view->h);
  }

  paint_view (view);

  gdk_window_set_back_pixmap (view->window->window, view->bg, 0);

  gdk_window_shape_combine_mask (view->window->window, view->mask, 0, 0); 

  gdk_window_set_decorations (view->window->window, 0);
  gtk_window_set_policy (GTK_WINDOW (view->window), FALSE, FALSE, FALSE);

  /* if (!view_num) */
    gtk_widget_show_all (view->window);

  return view;
}

static void create_theme (xml_node_t *theme, char *basename) {

  xml_node_t *node;
  int         view_num;

  node = theme->child; view_num = 0;
  while (node) {

    if (!strcasecmp (node->name, "view")) {
      create_view (node, basename, view_num++);
    }

    node = node->next;
  }
}

void create_skin_window (void) {

  /* char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/plus/bionic_d.wms";  */
  /* char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/xbox_live/xlive.wms";  */
   /* char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/kenwood/kenwood.wms"; */
  char       *skinfile = "/home/guenter/projects/video/gnome-xine/skins/media_center/skin.gxs";
  char       *xml_data;
  xml_node_t *tree, *node;

  printf ("skin_window: loading %s...\n", skinfile);

  xml_data = read_entire_file_ascii (skinfile);

  printf ("skin_window: ascii: %s\n", xml_data); 

  printf ("skin_window: parsing...\n");

  xml_parser_init (xml_data, strlen(xml_data), XML_PARSER_CASE_INSENSITIVE);

  if (xml_parser_build_tree (&tree)<0) {
    printf ("skin_window: error: xml parser failed\n");
    return;
  }

  node = tree;
  while (node) {

    if (!strcasecmp (node->name, "theme")) {
      gchar *skin_dir, *dir_sep;

      skin_dir = strdup (skinfile);
      dir_sep  = strrchr (skin_dir, '/');
      if (dir_sep)
	*dir_sep = '\0';

      create_theme (node, skin_dir);
      break;
    }

    node = node->next;
  }

}
