/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include "exrinput.h"
#include "exroutput.h"

/**
 *              [  0.7328  0.4296  -0.1624 ]
 *    M_CAT02 = [ -0.7036  1.6975   0.0061 ]
 *              [  0.0030  0.0136   0.9834 ]
 *
 *              [  1.096124 -0.278869 0.182745 ]
 * M^-1_CAT02 = [  0.454369  0.473533 0.072098 ]
 *              [ -0.009628 -0.005698 1.015326 ]
 */
static void xyz_to_cat02( double *r, double *g, double *b,
                          double x, double y, double z )
{
    *r = ( 0.7328 * x) + (0.4296 * y) - (0.1624 * z);
    *g = (-0.7036 * x) + (1.6975 * y) + (0.0061 * z);
    *b = ( 0.0030 * x) + (0.0136 * y) + (0.9834 * z);
}

static void cat02_to_xyz( double *x, double *y, double *z,
                          double r, double g, double b )
{
    *x = ( 1.096124 * r) - (0.278869 * g) + (0.182745 * b);
    *y = ( 0.454369 * r) + (0.473533 * g) + (0.072098 * b);
    *z = (-0.009628 * r) - (0.005698 * g) + (1.015326 * b);
}

static void xyz_to_srgb( double *r, double *g, double *b,
                         double  x, double  y, double  z )
{
    *r = (  3.2406 * x ) + ( -1.5372 * y ) + ( -0.4986 * z );
    *g = ( -0.9689 * x ) + (  1.8758 * y ) + (  0.0415 * z );
    *b = (  0.0557 * x ) + ( -0.2040 * y ) + (  1.0570 * z );
}

static void srgb_to_xyz( double *x, double *y, double *z,
                         double  r, double  g, double  b )
{
    *x = ( 0.4124 * r ) + ( 0.3576 * g ) + ( 0.1805 * b );
    *y = ( 0.2126 * r ) + ( 0.7152 * g ) + ( 0.0722 * b );
    *z = ( 0.0193 * r ) + ( 0.1192 * g ) + ( 0.9505 * b );
}

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    exrinput_t *exrin;
    exrinput_t *whitein;
    double *scanline;
    int width, height;
    double d;
    int y;

    if( argc < 5 ) {
        fprintf( stderr, "Usage: exrchr input.exr blur.exr output.exr degree_of_adaptation\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exrchr: Cannot open %s.\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exrchr: %s [%dx%d]\n", argv[ 1 ], width, height );

    scanline = malloc( width * 4 * sizeof( double ) );
    if( !scanline ) {
        fprintf( stderr, "exrchr: Cannot allocate memory.\n" );
        exrinput_delete( exrin );
        return 1;
    }

    whitein = exrinput_new( argv[ 2 ] );
    if( !whitein ) {
        fprintf( stderr, "exrchr: Cannot open %s.\n", argv[ 2 ] );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    if( exrinput_get_width( whitein ) != width || exrinput_get_height( whitein ) != height ) {
        fprintf( stderr, "exrchr: Whitepoint image (%dx%d) does not match input image (%dx%d).\n",
                 exrinput_get_width( whitein ), exrinput_get_height( whitein ), width, height );
        exrinput_delete( whitein );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    exrout = exroutput_new( argv[ 3 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "exrchr: Cannot open %s.\n", argv[ 3 ] );
        exrinput_delete( whitein );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    d = atof( argv[ 4 ] );
    fprintf( stderr, "exrchr: Degree of adaptation %.2f\n", d );

    for( y = 0; y < height; y++ ) {
        double *input = exrinput_get_scanline( exrin, y );
        double *whiteinput = exrinput_get_scanline( whitein, y );
        int x;

        fprintf( stderr, "exrchr: Correcting scanline %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r, g, b;
            double rw, gw, bw;
            double rc, gc, bc;
            double X, Y, Z;
            double yw;

            r = whiteinput[ (x * 4) + 0 ];
            g = whiteinput[ (x * 4) + 1 ];
            b = whiteinput[ (x * 4) + 2 ];
            srgb_to_xyz( &X, &Y, &Z, r, g, b );
            xyz_to_cat02( &rw, &gw, &bw, X, Y, Z );
            yw = Y;

            r = input[ (x * 4) + 0 ];
            g = input[ (x * 4) + 1 ];
            b = input[ (x * 4) + 2 ];

            srgb_to_xyz( &X, &Y, &Z, r, g, b );
            xyz_to_cat02( &r, &g, &b, X, Y, Z );

            rc = (((yw * d) / rw) + (1.0 - d)) * r;
            gc = (((yw * d) / gw) + (1.0 - d)) * g;
            bc = (((yw * d) / bw) + (1.0 - d)) * b;

            cat02_to_xyz( &X, &Y, &Z, rc, gc, bc );
            xyz_to_srgb( &r, &g, &b, X, Y, Z );

            scanline[ (x * 4) + 0 ] = r;
            scanline[ (x * 4) + 1 ] = g;
            scanline[ (x * 4) + 2 ] = b;
            scanline[ (x * 4) + 3 ] = input[ (x * 4) + 3 ];
        }
        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\nexrchr: Done.\n" );

    exroutput_delete( exrout );
    exrinput_delete( exrin );
    free( scanline );
    return 0;
}

