/* Copyright (c) 2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */


/*
[SENSOR]
Sensor Model:   MI1040
Camera Module:
Lens Model:
Driver IC:
PV Size         = 1280 x 960
Cap Size        = 1280 x 960
Output Format   = YCbCr
MCLK Speed      = 24M
PV Frame Rate   = 30fps
Cap Frame Rate  = 30fps
I2C Slave ID    = 0x90
I2C Mode        = 16Addr, 16Data
*/

#ifndef CAMSENSOR_MI1040
#define CAMSENSOR_MI1040

/* Sysctl registers */
#define MI1040_COMMAND_REGISTER                0x0080
#define MI1040_COMMAND_REGISTER_APPLY_PATCH    (1 << 0)
#define MI1040_COMMAND_REGISTER_SET_STATE      (1 << 1)
#define MI1040_COMMAND_REGISTER_REFRESH        (1 << 2)
#define MI1040_COMMAND_REGISTER_WAIT_FOR_EVENT (1 << 3)
#define MI1040_COMMAND_REGISTER_OK             (1 << 15)

extern struct proc_dir_entry proc_root;

enum Vendor_ID {
	A,
	B,
};

static uint16_t mi1040_flicker_detect_table[] = {
0xdc10, 0xc0f1, 0x0cda, 0x0580, 0x76cf, 0xff00, 0x2184,
0x9624, 0x218c, 0x8fc3, 0x75cf, 0xffff, 0xe058, 0xf686,
0x1550, 0x1080, 0xe001, 0x1d50, 0x1002, 0x1552, 0x1100,
0x6038, 0x1d52, 0x1004, 0x1540,
0xdc40, 0x1080, 0x081b, 0x00d1, 0x8512, 0x1000, 0x00c0,
0x7822, 0x2089, 0x0fc1, 0x2008, 0x0f81, 0xffff, 0xff80,
0x8512, 0x1801, 0x0052, 0xa512, 0x1544, 0x1080, 0xb861,
0x262f, 0xf007, 0x1d44, 0x1002,
0xdc70, 0x20ca, 0x0021, 0x20cf, 0x04e1, 0x0850, 0x04a1,
0x21ca, 0x0021, 0x1542, 0x1140, 0x8d2c, 0x6038, 0x1d42,
0x1004, 0x1542, 0x1140, 0xb601, 0x046d, 0x0580, 0x78e0,
0xd800, 0xb893, 0x002d, 0x04a0,
0xdca0, 0xd900, 0x78e0, 0x72cf, 0xffff, 0xe058, 0x2240,
0x0340, 0xa212, 0x208a, 0x0fff, 0x1a42, 0x0004, 0xd830,
0x1a44, 0x0002, 0xd800, 0x1a50, 0x0002, 0x1a52, 0x0004,
0x1242, 0x0140, 0x8a2c, 0x6038,
0xdcd0, 0x1a42, 0x0004, 0x1242, 0x0141, 0x70cf, 0xff00,
0x2184, 0xb021, 0xd800, 0xb893, 0x07e5, 0x0460, 0xd901,
0x78e0, 0xc0f1, 0x0bfa, 0x05a0, 0x216f, 0x0043, 0xc1a4,
0x220a, 0x1f80, 0xffff, 0xe058,
0xdd00, 0x2240, 0x134f, 0x1a48, 0x13c0, 0x1248, 0x1002,
0x70cf, 0x7fff, 0xffff, 0xe230, 0xc240, 0xda00, 0xf00c,
0x1248, 0x1003, 0x1301, 0x04cb, 0x7261, 0x2108, 0x0081,
0x2009, 0x0080, 0x1a48, 0x10c0,
0xdd30, 0x1248, 0x100b, 0xc300, 0x0be7, 0x90c4, 0x2102,
0x0003, 0x238c, 0x8fc3, 0xf6c7, 0xdaff, 0x1a05, 0x1082,
0xc241, 0xf005, 0x7a6f, 0xc241, 0x1a05, 0x10c2, 0x2000,
0x8040, 0xda00, 0x20c0, 0x0064,
0xdd60, 0x781c, 0xc042, 0x1c0e, 0x3082, 0x1a48, 0x13c0,
0x7548, 0x7348, 0x7148, 0x7648, 0xf002, 0x7608, 0x1248,
0x1000, 0x1400, 0x300b, 0x084d, 0x02c5, 0x1248, 0x1000,
0xe101, 0x1001, 0x04cb, 0x1a48,
0xdd90, 0x1000, 0x7361, 0x1408, 0x300b, 0x2302, 0x02c0,
0x780d, 0x2607, 0x903e, 0x07d6, 0xffe3, 0x792f, 0x09cf,
0x8152, 0x1248, 0x100e, 0x2400, 0x334b, 0xe501, 0x7ee2,
0x0dbf, 0x90f2, 0x1b0c, 0x1382,
0xddc0, 0xc123, 0x140e, 0x3080, 0x7822, 0x1a07, 0x1002,
0x124c, 0x1000, 0x120b, 0x1081, 0x1207, 0x1083, 0x2142,
0x004b, 0x781b, 0x0b21, 0x02e2, 0x1a4c, 0x1000, 0xe101,
0x0915, 0x00c2, 0xc101, 0x1204,
0xddf0, 0x1083, 0x090d, 0x00c2, 0xe001, 0x1a4c, 0x1000,
0x1a06, 0x1002, 0x234a, 0x1000, 0x7169, 0xf008, 0x2053,
0x0003, 0x6179, 0x781c, 0x2340, 0x104b, 0x1203, 0x1083,
0x0bf1, 0x90c2, 0x1202, 0x1080,
0xde20, 0x091d, 0x0004, 0x70cf, 0xffff, 0xc644, 0x881b,
0xe0b2, 0xd83c, 0x20ca, 0x0ca2, 0x1a01, 0x1002, 0x1a4c,
0x1080, 0x02b9, 0x05a0, 0xc0a4, 0x78e0, 0xc0f1, 0xff95,
0xd800, 0x71cf, 0xff00, 0x1fe0,
0xde50, 0x19d0, 0x001c, 0x19d1, 0x001c, 0x70cf, 0xffff,
0xe058, 0x901f, 0xb861, 0x19d2, 0x001c, 0xc0d1, 0x7ee0,
0x78e0, 0xc0f1, 0x0a7a, 0x0580, 0x70cf, 0xffff, 0xc5d4,
0x9041, 0x9023, 0x75cf, 0xffff,
0xde80, 0xe058, 0x7942, 0xb967, 0x7f30, 0xb53f, 0x71cf,
0xffff, 0xc84c, 0x91d3, 0x108b, 0x0081, 0x2615, 0x1380,
0x090f, 0x0c91, 0x0a8e, 0x05a0, 0xd906, 0x7e10, 0x2615,
0x1380, 0x0a82, 0x05a0, 0xd960,
0xdeb0, 0x790f, 0x090d, 0x0133, 0xad0c, 0xd904, 0xad2c,
0x79ec, 0x2941, 0x7402, 0x71cf, 0xff00, 0x2184, 0xb142,
0x1906, 0x0e44, 0xffde, 0x70c9, 0x0a5a, 0x05a0, 0x8d2c,
0xad0b, 0xd800, 0xad01, 0x0219,
0xdee0, 0x05a0, 0xa513, 0xc0f1, 0x71cf, 0xffff, 0xc644,
0xa91b, 0xd902, 0x70cf, 0xffff, 0xc84c, 0x093e, 0x03a0,
0xa826, 0xffdc, 0xf1b5, 0xc0f1, 0x09ea, 0x0580, 0x75cf,
0xffff, 0xe058, 0x1540, 0x1080,
0xdf10, 0x08a7, 0x0010, 0x8d00, 0x0813, 0x009e, 0x1540,
0x1081, 0xe181, 0x20ca, 0x00a1, 0xf24b, 0x1540, 0x1081,
0x090f, 0x0050, 0x1540, 0x1081, 0x0927, 0x0091, 0x1550,
0x1081, 0xde00, 0xad2a, 0x1d50,
0xdf40, 0x1382, 0x1552, 0x1101, 0x1d52, 0x1384, 0xb524,
0x082d, 0x015f, 0xff55, 0xd803, 0xf033, 0x1540, 0x1081,
0x0967, 0x00d1, 0x1550, 0x1081, 0xde00, 0xad2a, 0x1d50,
0x1382, 0x1552, 0x1101, 0x1d52,
0xdf70, 0x1384, 0xb524, 0x0811, 0x019e, 0xb8a0, 0xad00,
0xff47, 0x1d40, 0x1382, 0xf01f, 0xff5a, 0x8d01, 0x8d40,
0xe812, 0x71cf, 0xffff, 0xc644, 0x893b, 0x7030, 0x22d1,
0x8062, 0xf20a, 0x0a0f, 0x009e,
0xdfa0, 0x71cf, 0xffff, 0xc84c, 0x893b, 0xe902, 0xffcf,
0x8d00, 0xb8e7, 0x26ca, 0x1022, 0xf5e2, 0xff3c, 0xd801,
0x1d40, 0x1002, 0x0141, 0x0580, 0x78e0, 0xc0f1, 0xc5e1,
0xff34, 0xdd00, 0x70cf, 0xffff,
0xdfd0, 0xe090, 0xa8a8, 0xd800, 0xb893, 0x0c8a, 0x0460,
0xd901, 0x71cf, 0xffff, 0xdc10, 0xd813, 0x0b96, 0x0460,
0x72a9, 0x0119, 0x0580, 0xc0f1, 0x71cf, 0x0000, 0x5bae,
0x7940, 0xff9d, 0xf135, 0x78e0,
0xe000, 0xc0f1, 0x70cf, 0x0000, 0x5cba, 0x7840, 0x70cf,
0xffff, 0xe058, 0x8800, 0x0815, 0x001e, 0x70cf, 0xffff,
0xc84c, 0x881a, 0xe080, 0x0ee0, 0xffc1, 0xf121, 0x78e0,
0xc0f1, 0xd900, 0xf009, 0x70cf,
0xe030, 0xffff, 0xe0ac, 0x7835, 0x8041, 0x8000, 0xe102,
0xa040, 0x09f3, 0x8114, 0x71cf, 0xffff, 0xe058, 0x70cf,
0xffff, 0xc594, 0xb030, 0xffdd, 0xd800, 0xf109, 0x0000,
0x0300, 0x0204, 0x0700, 0x0000,
0xe060, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
0x0000, 0x0000, 0x0000, 0x0000,
0xe090, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
0x0000, 0xffff, 0xcb68, 0xffff, 0xdff0, 0xffff, 0xcb6c,
0xffff, 0xe000
};

static struct msm_camera_i2c_reg_conf mi1040_960p_settings[] = {
	{0xdc00, 0x50, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_SET_STATE), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{0xDC01, 0x52, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_POLL},

	{0x098E, 0, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC868, 0x0500,},
	{0xC86A, 0x03C0,},
	{0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC854, 0x0000,},
	{0xC856, 0x0000,},
	{0xC858, 0x0500,},
	{0xC85A, 0x03C0,},
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF,},
	{0xC91A, 0x03BF,},
	{0xC91C, 0x0000,},
	{0xC91E, 0x0000,},
	{0xC920, 0x00FF,},
	{0xC922, 0x00BF,},
};

static struct msm_camera_i2c_reg_conf mi1040_recommend_settings[] = {
	{0x301A, 0x0234, MSM_CAMERA_I2C_SET_WORD_MASK},
	{0x098E, 0, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_enable = 1*/
	{0xC97E, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_divider_m_n = 288*/
	{0xC980, 0x0120,},
	/*cam_sysctl_pll_divider_p = 1792*/
	{0xC982, 0x0700,},

	{0x98E, 0x071D ,},
	{0xC800, 0x0004,},
	{0xC802, 0x0004,},
	{0xC804, 0x03CB,},
	{0xC806, 0x050B,},
	/* data length is double words */
	{0xC808, 0x02DC,},
	{0xC80A, 0x6C00,},
	{0xC80C, 0x0001,},
	{0xC80E, 0x00DB,},
	{0xC810, 0x05B3,},
	{0xC812, 0x03EE,},
	{0xC814, 0x0636,},
	{0xC816, 0x0060,},
	{0xC818, 0x03C3,},
	{0xC834, 0x0000,},
	{0xC854, 0x0000,},
	{0xC856, 0x0000,},
	{0xC858, 0x0500,},
	{0xC85A, 0x03C0,},
	{0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC868, 0x0500,},
	{0xC86A, 0x03C0,},
	{0xC88C, 0x1E02,},
	/* Changed to varied frame rate to 7.5~30fps */
	{0xC88E, 0x0780,},
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF,},
	{0xC91A, 0x03BF,},
	{0xC91C, 0x0000,},
	{0xC91E, 0x0000,},
	{0xC920, 0x00FF,},
	{0xC922, 0x00BF,},
	{0xE801, 0x00, MSM_CAMERA_I2C_BYTE_DATA ,},

	/*Sensor optimization*/
	{0x316A, 0x8270,},
	{0x316C, 0x8270,},
	{0x3ED0, 0x2305,},
	{0x3ED2, 0x77CF,},
	{0x316E, 0x8202,},
	{0x3180, 0x87FF,},
	{0x30D4, 0x6080,},
	{0xA802, 0x0008,},
	{0x3E14, 0xFF39,},

	/* PGA parameter and APGA */
	{0x098E, 0x495E,},
	{0xC95E, 0x0002,},
	{0x3640, 0x02D0,},
	{0x3642, 0xE0AA,},
	{0x3644, 0x4570,},
	{0x3646, 0x2C2E,},
	{0x3648, 0x11F1,},
	{0x364A, 0x0150,},
	{0x364C, 0x8B8C,},
	{0x364E, 0x5130,},
	{0x3650, 0x70EE,},
	{0x3652, 0x47B1,},
	{0x3654, 0x01B0,},
	{0x3656, 0x5BCA,},
	{0x3658, 0x0D30,},
	{0x365A, 0x042F,},
	{0x365C, 0x1EF1,},
	{0x365E, 0x0130,},
	{0x3660, 0xF92B,},
	{0x3662, 0x5310,},
	{0x3664, 0x0CAE,},
	{0x3666, 0x0151,},
	{0x3680, 0x686C,},
	{0x3682, 0xA2AE,},
	{0x3684, 0x88CF,},
	{0x3686, 0x1430,},
	{0x3688, 0x506F,},
	{0x368A, 0x76CB,},
	{0x368C, 0x9BEE,},
	{0x368E, 0xD16E,},
	{0x3690, 0x0210,},
	{0x3692, 0x00EF,},
	{0x3694, 0x0C2D,},
	{0x3696, 0xAEEE,},
	{0x3698, 0xD3CF,},
	{0x369A, 0x7F6F,},
	{0x369C, 0x26AF,},
	{0x369E, 0x11AD,},
	{0x36A0, 0xF0AD,},
	{0x36A2, 0xE40F,},
	{0x36A4, 0x03B0,},
	{0x36A6, 0x71AE,},
	{0x36C0, 0x5490,},
	{0x36C2, 0x2A90,},
	{0x36C4, 0x1133,},
	{0x36C6, 0xBDD1,},
	{0x36C8, 0xEEF2,},
	{0x36CA, 0x73D0,},
	{0x36CC, 0x3F90,},
	{0x36CE, 0x3BF3,},
	{0x36D0, 0xDC51,},
	{0x36D2, 0xCD93,},
	{0x36D4, 0x25D0,},
	{0x36D6, 0x2750,},
	{0x36D8, 0x0DF3,},
	{0x36DA, 0xAA31,},
	{0x36DC, 0xF832,},
	{0x36DE, 0x5CB0,},
	{0x36E0, 0x2570,},
	{0x36E2, 0x07D3,},
	{0x36E4, 0xC831,},
	{0x36E6, 0xC5D2,},
	{0x3700, 0xB0AF,},
	{0x3702, 0x2210,},
	{0x3704, 0x44EF,},
	{0x3706, 0x8332,},
	{0x3708, 0x550E,},
	{0x370A, 0xEE0E,},
	{0x370C, 0x25D0,},
	{0x370E, 0xC32E,},
	{0x3710, 0xEB31,},
	{0x3712, 0x07F2,},
	{0x3714, 0xD72C,},
	{0x3716, 0x4170,},
	{0x3718, 0xCCAD,},
	{0x371A, 0xF031,},
	{0x371C, 0x2552,},
	{0x371E, 0xEB4E,},
	{0x3720, 0x2390,},
	{0x3722, 0xA750,},
	{0x3724, 0xFC91,},
	{0x3726, 0x5392,},
	{0x3740, 0x7B30,},
	{0x3742, 0xF151,},
	{0x3744, 0x60B1,},
	{0x3746, 0x1E73,},
	{0x3748, 0x98F6,},
	{0x374A, 0x4F91,},
	{0x374C, 0x82B2,},
	{0x374E, 0x6331,},
	{0x3750, 0x4B73,},
	{0x3752, 0xB3F6,},
	{0x3754, 0x2911,},
	{0x3756, 0xB4B1,},
	{0x3758, 0x7510,},
	{0x375A, 0x0293,},
	{0x375C, 0x8196,},
	{0x375E, 0x5890,},
	{0x3760, 0xF6F1,},
	{0x3762, 0x3C32,},
	{0x3764, 0x3613,},
	{0x3766, 0xA136,},
	{0x3784, 0x0288,},
	{0x3782, 0x01E8,},
	{0x37C0, 0x920B,},
	{0x37C2, 0xD44A,},
	{0x37C4, 0xA50B,},
	{0x37C6, 0x928B,},
	{0x098E, 0x0000,},

	/* -- 0518, Alias */
	{0xC960, 0x0AF0,},
	{0xC962, 0x7746,},
	{0xC964, 0x5EB8,},
	{0xC966, 0x7656,},
	{0xC968, 0x71CA,},
	{0xC96A, 0x0FA0,},
	{0xC96C, 0x7DB8,},
	{0xC96E, 0x7E58,},
	{0xC970, 0x7DB6,},
	{0xC972, 0x7D6C,},
	{0xC974, 0x1964,},
	{0xC976, 0x7BC9,},
	{0xC978, 0x6F08,},
	{0xC97A, 0x7D51,},
	{0xC97C, 0x74B2,},
	{0xC95E, 0x0003,},


	/*[CCM]*/
	{0xC892, 0x0267,},
	{0xC894, 0xFF1A,},
	{0xC896, 0xFFB3,},
	{0xC898, 0xFF80,},
	{0xC89A, 0x0166,},
	{0xC89C, 0x0003,},
	{0xC89E, 0xFF9A,},
	{0xC8A0, 0xFEB4,},
	{0xC8A2, 0x024D,},
	{0xC8A4, 0x01BF,},
	{0xC8A6, 0xFF01,},
	{0xC8A8, 0xFFF3,},
	{0xC8AA, 0xFF75,},
	{0xC8AC, 0x0198,},
	{0xC8AE, 0xFFFD,},
	{0xC8B0, 0xFF9A,},
	{0xC8B2, 0xFEE7,},
	{0xC8B4, 0x02A8,},

	{0xC8B6, 0x0100,},
	{0xC8B8, 0xFF72,},
	{0xC8BA, 0xFFF8,},
	{0xC8BC, 0xFFA8,},
	{0xC8BE, 0x011D,},
	{0xC8C0, 0xFFFB,},
	{0xC8C2, 0xFFE3,},
	{0xC8C4, 0xFE48,},
	{0xC8C6, 0x0356,},

	{0xC8DA, 0x004D,},
	{0xC8DC, 0x0096,},
	{0xC8DE, 0x001D,},
	{0xC8E0, 0x004D,},
	{0xC8E2, 0x0096,},
	{0xC8E4, 0x001D,},
	{0xC8E6, 0x004D,},
	{0xC8E8, 0x0096,},
	{0xC8EA, 0x001D,},
	{0xC8C8, 0x0075,},
	{0xC8CA, 0x011C,},
	{0xC8CC, 0x009A,},
	{0xC8CE, 0x0105,},
	{0xC8D0, 0x00A4,},
	{0xC8D2, 0x00AC,},
	{0xC8D4, 0x0A8C,},
	{0xC8D6, 0x0F0A,},
	{0xC8D8, 0x1964,},

	/*[AWB]*/
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF,},
	{0xC91A, 0x03BF,},
	{0xC8F2, 0x04, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC8F3, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC904, 0x0036,},
	{0xC906, 0x0040,},
	{0xC8F4, 0x0000,},
	{0xC8F6, 0x0000,},
	{0xC8F8, 0x0000,},
	{0xC8FA, 0xE724,},
	{0xC8FC, 0x1583,},
	{0xC8FE, 0x2045,},
	{0xC900, 0x05DC,},
	{0xC902, 0x007C,},
	{0xC90A, 0x1388,},
	{0xC90C, 0x8D, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC90D, 0x9A, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC90E, 0x8E, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC90F, 0x80, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC910, 0x88, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC911, 0x7C, MSM_CAMERA_I2C_BYTE_DATA},
	{0x098E, 0xAC06,},
	{0xAC06, 0x62, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC07, 0x66, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC08, 0x64, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC09, 0x66, MSM_CAMERA_I2C_BYTE_DATA},

	/*[Step7-CPIPE_Preference]*/
	{0xC926, 0x0060,},
	{0xC928, 0x009A,},
	{0xC946, 0x0070,},
	{0xC948, 0x00F3,},
	{0xC952, 0x0060,},
	{0xC954, 0x009A,},
	{0xC95A, 0x04, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92A, 0x5A, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92B, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92C, 0x00, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92D, 0xFF, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92E, 0x32, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92F, 0x04, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC930, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC931, 0x78, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC932, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC933, 0x09, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC934, 0x32, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC935, 0x14, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC936, 0x32, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC937, 0x14, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC938, 0x50, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC939, 0x50, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC93A, 0x50, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC93B, 0x50, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC93C, 0x0005 ,},
	{0xC93E, 0x0358 ,},
	{0xC940, 0x00DC ,},
	/*CAM_LL_START_CONTRAST_GRADIENT*/
	{0xC942, 0x4B, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC943, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC944, 0x22, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC945, 0x19, MSM_CAMERA_I2C_BYTE_DATA},
	{0x098E, 0xC942,},
	{0x098E, 0xC92A,},
	{0xC94A, 0x00F0,},
	{0xC94C, 0x0010,},
	{0xC94E, 0x01CD,},
	{0xC950, 0x05, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC951, 0x40, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC87A, 0x35, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC87B, 0x32, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC878, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC890, 0x0080 ,},
	{0xC882, 0x0100 ,},
	{0xC886, 0x0100 ,},
	{0xA404, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	/*AE_TRACK_AE_TRACKING_DAMPENING*/
	{0xC87C, 0x0016,},
	{0xB42A, 0x05, MSM_CAMERA_I2C_BYTE_DATA},
	{0xA80A, 0x10, MSM_CAMERA_I2C_BYTE_DATA},
	{0x326C, 0x0A08,},

	/*Step8-Features*/
	{0x098E, 0x0000,},
	{0xC984, 0x8040,},
	{0x001E, 0x0777,},

	/* LOAD=MIPI setting for SOC1040 */
	{0xC984, 0x8041,},
	{0xC988, 0x0F00,},
	{0xC98A, 0x0B07,},
	{0xC98C, 0x0D01,},
	{0xC98E, 0x071D,},
	{0xC990, 0x0006,},
	{0xC992, 0x0A0C,},
	{0x3C5A, 0x0009,},

	/* Improve AE Dampening speed */
	/* [Speed up AE/AWB] */
	{0x098E, 0x2802,},
	{0xA802, 0x0008,},
	{0xC908, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC879, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC909, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xA80A, 0x18, MSM_CAMERA_I2C_BYTE_DATA},
	{0xA80B, 0x18, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC16, 0x18, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC878, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},

	{0x31E0, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
};

/*static void mi1040_stop_stream(struct msm_sensor_ctrl_t *s_ctrl) {}*/

static struct msm_camera_i2c_reg_conf mi1040_config_change_settings[] = {
	{0xdc00, 0x28, MSM_CAMERA_I2C_BYTE_DATA, MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_SET_STATE), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{0xDC01, 0x31, MSM_CAMERA_I2C_BYTE_DATA},
};

/*
static int mi1040_saturation_enum_map[] = {
	MSM_V4L2_SATURATION_L0,
	MSM_V4L2_SATURATION_L1,
	MSM_V4L2_SATURATION_L2,
	MSM_V4L2_SATURATION_L3,
	MSM_V4L2_SATURATION_L4,
	MSM_V4L2_SATURATION_L5,
	MSM_V4L2_SATURATION_L6,
	MSM_V4L2_SATURATION_L7,
	MSM_V4L2_SATURATION_L8,
	MSM_V4L2_SATURATION_L9,
	MSM_V4L2_SATURATION_L10,
};

static struct msm_camera_i2c_reg_conf mi1040_saturation[][1] = {
	{{0xCC12, 0x00},},
	{{0xCC12, 0x1A},},
	{{0xCC12, 0x34},},
	{{0xCC12, 0x4E},},
	{{0xCC12, 0x68},},
	{{0xCC12, 0x80},},
	{{0xCC12, 0x9A},},
	{{0xCC12, 0xB4},},
	{{0xCC12, 0xCE},},
	{{0xCC12, 0xE8},},
	{{0xCC12, 0xFF},},
};

static struct msm_camera_i2c_reg_conf mi1040_refresh[] = {
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_REFRESH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, (MI1040_COMMAND_REGISTER_OK |
		MI1040_COMMAND_REGISTER_REFRESH), MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_WRITE},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_REFRESH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_OK,
		MSM_CAMERA_I2C_SET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
};
*/

static struct msm_camera_i2c_reg_conf ColorEffect_None[] = {
	/* [2.1 Normal -- default] */
	{0x098E, 0xC874},
	{0xC874, 0x00, MSM_CAMERA_I2C_BYTE_DATA},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA},
	{0x0080, 0x8004},
};

static struct msm_camera_i2c_reg_conf ColorEffect_Mono[] = {
	/* [2.2 Mono color effect] */
	{0x098E, 0xC874},
	{0xC874, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA},
	{0x0080, 0x8004},
};

static struct msm_camera_i2c_reg_conf ColorEffect_Sepia[] = {
	/* [2.3 Sepia effect] */
	{0x098E, 0xC874},
	{0xC874, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA},
	{0x0080, 0x8004},
};

static struct msm_camera_i2c_reg_conf ColorEffect_Negative[] = {
	/* [2.4 Negative effect] */
	{0x098E, 0xC874},
	{0xC874, 0x03, MSM_CAMERA_I2C_BYTE_DATA},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA},
	{0x0080, 0x8004},
};

static struct msm_camera_i2c_reg_conf ColorEffect_Solarize[] = {
	/* [2.5 Solarize 1] */
	{0x098E, 0xC874},
	{0xC874, 0x04, MSM_CAMERA_I2C_BYTE_DATA},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA},
	{0x0080, 0x8004},
};

static struct msm_camera_i2c_reg_conf Whitebalance_Auto[] = {
	{0x098E, 0xC909},
	{0xC909, 0x03, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC04, 0x0288},
};

static struct msm_camera_i2c_reg_conf Whitebalance_Incandescent[] = {
	{0x098E, 0xC909},
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC04, 0x0288},
	{0xC8F0, 0x09C4},
};

static struct msm_camera_i2c_reg_conf Whitebalance_Daylight[] = {
	{0x098E, 0xC909},
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC04, 0x0288},
	{0xC8F0, 0x1964},
};

static struct msm_camera_i2c_reg_conf Whitebalance_Fluorescent[] = {
	{0x098E, 0xC909},
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC04, 0x0288},
	{0xC8F0, 0x0D67},
};

static struct msm_camera_i2c_reg_conf Whitebalance_Cloudy[] = {
	{0x098E, 0xC909},
	{0xC909, 0x00, MSM_CAMERA_I2C_BYTE_DATA} ,
	{0xAC04, 0x0208},
	{0xC8F0, 0x1964},
	{0xAC12, 0x00B4},
	{0xAC14, 0x0080},
};

static struct msm_camera_i2c_reg_conf EV_zero[] = {
	/* [3.3 EV0: 128 -- default] */
	{0x098E, 0x4C0A},
	{0xCC0A, 0x0037},
};

static struct msm_camera_i2c_reg_conf EV_plus_1[] = {
	/* [3.4 EV+1: 138] */
	{0x098E, 0x4C0A},
	{0xCC0A, 0x004E},
};

static struct msm_camera_i2c_reg_conf EV_plus_2[] = {
	/* [3.5 EV+2: 148] */
	{0x098E, 0x4C0A},
	{0xCC0A, 0x006E},
};

static struct msm_camera_i2c_reg_conf EV_minus_1[] = {
	/* [3.2 EV-1: 118] */
	{0x098E, 0x4C0A},
	{0xCC0A, 0x002A},
};

static struct msm_camera_i2c_reg_conf EV_minus_2[] = {
	/* [3.1 EV-2: 108] */
	{0x098E, 0x4C0A},
	{0xCC0A, 0x001F},
};

static struct msm_camera_i2c_reg_conf Fix_30_fps[] = {
	/* [fix 30 fps] */
	{0x098E, 0x1000,},
	{0xC810, 0x05B3,},
	{0xC812, 0x03EE,},
	{0xC814, 0x0636,},
	{0xC88C, 0x1E02,},
	{0xC88E, 0x1E02,},
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
};

static struct msm_camera_i2c_reg_conf Fix_25_fps[] = {
	/* [fix 25 fps] */
	{0x098E, 0x1000,},
	{0xC810, 0x05BD,},
	{0xC812, 0x04B0,},
	{0xC814, 0x0640,},
	{0xC88C, 0x1900,},
	{0xC88E, 0x1900,},
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
};

static struct msm_camera_i2c_reg_conf Fix_24_fps[] = {
	/* [fix 24 fps] */
	{0x098E, 0x1000,},
	{0xC810, 0x05BD,},
	{0xC812, 0x04E2,},
	{0xC814, 0x0640,},
	{0xC88C, 0x1800,},
	{0xC88E, 0x1800,},
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
};

static struct msm_camera_i2c_reg_conf Fix_20_fps[] = {
	/* [fix 20 fps] */
	{0x098E, 0x1000,},
	{0xC810, 0x05BD,},
	{0xC812, 0x05DC,},
	{0xC814, 0x0640,},
	{0xC88C, 0x1400,},
	{0xC88E, 0x1400,},
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
};

static struct msm_camera_i2c_reg_conf Fix_15_fps[] = {
	/* [fix 15 fps] */
	{0x098E, 0x1000,},
	{0xC810, 0x05BD,},
	{0xC812, 0x07D0,},
	{0xC814, 0x0640,},
	{0xC88C, 0x0F00,},
	{0xC88E, 0x0F00,},
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
};

static struct msm_camera_i2c_reg_conf auto_30_fps[] = {
	/* [auto 30 fps] */
	{0x098E, 0x1000,},
	{0xC810, 0x05B3,},
	{0xC812, 0x03EE,},
	{0xC814, 0x0636,},
	{0xC88C, 0x1E02,},
	{0xC88E, 0x0F00,},
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA,},
	{0x0080, 0x8002,},
};

static struct msm_camera_i2c_reg_conf mi1040_B_recommend_settings[] = {
	{0x098E, 0x1000, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_enable = 1*/
	{0xC97E, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	/*cam_sysctl_pll_divider_m_n = 288*/
	{0xC980, 0x0120,},
	/*cam_sysctl_pll_divider_p = 1792*/
	{0xC982, 0x0700,},

	{0xC800, 0x0004,},
	{0xC802, 0x0004,},
	{0xC804, 0x03CB,},
	{0xC806, 0x050B,},
	/* data length is double words */
	{0xC808, 0x02DC,},/*pixclk = 48000000*/
	{0xC80A, 0x6C00,},/*pixclk = 48000000*/
	{0xC80C, 0x0001,},
	{0xC80E, 0x00DB,},
	{0xC810, 0x05C1,},
	{0xC812, 0x03F3,},
	{0xC814, 0x0644,},
	{0xC816, 0x0060,},
	{0xC818, 0x03C3,},
	{0xC826, 0x0020,},
	{0xC834, 0x0000,},
	{0xC854, 0x0000,},
	{0xC856, 0x0000,},
	{0xC858, 0x0500,},
	{0xC85A, 0x03C0,},
	{0xC85C, 0x03, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC868, 0x0500,},
	{0xC86A, 0x03C0,},
	{0xC878, 0x00, MSM_CAMERA_I2C_BYTE_DATA,},
	{0xC88C, 0x1D99,},
	/* Changed to varied frame rate to 7.5~30fps */
	{0xC88E, 0x0766,},
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF,},
	{0xC91A, 0x03BF,},
	{0xC91C, 0x0000,},
	{0xC91E, 0x0000,},
	{0xC920, 0x00FF,},
	{0xC922, 0x00BF,},
	{0xE801, 0x00, MSM_CAMERA_I2C_BYTE_DATA ,},

	/* Enable FW patch of anti-flicker */
	{0x098E, 0xDC00,},
	{0xDC00, 0x28, MSM_CAMERA_I2C_BYTE_DATA ,},
	{0x0080, 0x8002,},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_SET_STATE,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},

	/*Sensor optimization*/
	{0x316A, 0x8270,},
	{0x316C, 0x8270,},
	{0x3ED0, 0x2305,},
	{0x3ED2, 0x77CF,},
	{0x316E, 0x8202,},
	{0x3180, 0x87FF,},
	{0x30D4, 0x6080,},
	{0xA802, 0x0008,},
	{0x3E14, 0xFF39,},
	{0x31E0, 0x0001,},

	/* Flicker Detect */
	{ 0x0982, 0x0001,},
	{ 0xC98A, 0x5C10,},
	{ 0x5C10, 0xDC10, MSM_CAMERA_I2C_WORD_DATA,
		MSM_CAMERA_I2C_CMD_SEQ_WRITE,
		0, mi1040_flicker_detect_table,
		ARRAY_SIZE(mi1040_flicker_detect_table)},
	{ 0x098E, 0x0000,},
	{ 0xE000, 0x1028,},
	{ 0xE002, 0x1004,},
	{ 0xE004, 0x4103,},
	{ 0xE006, 0x0202,},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_APPLY_PATCH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},
	{MI1040_COMMAND_REGISTER, MI1040_COMMAND_REGISTER_APPLY_PATCH,
		MSM_CAMERA_I2C_UNSET_WORD_MASK, MSM_CAMERA_I2C_CMD_POLL},

	/* PGA parameter and APGA */
	{ 0x3640, 0x0290,},
	{ 0x3642, 0xAFEA,},
	{ 0x3644, 0x04D1,},
	{ 0x3646, 0x85CD,},
	{ 0x3648, 0xA46E,},
	{ 0x364A, 0x00D0,},
	{ 0x364C, 0x8EEB,},
	{ 0x364E, 0x1C71,},
	{ 0x3650, 0xAF8C,},
	{ 0x3652, 0xF40D,},
	{ 0x3654, 0x0170,},
	{ 0x3656, 0x16AB,},
	{ 0x3658, 0x4870,},
	{ 0x365A, 0xFC2B,},
	{ 0x365C, 0x8DED,},
	{ 0x365E, 0x0130,},
	{ 0x3660, 0xBFAB,},
	{ 0x3662, 0x0CB1,},
	{ 0x3664, 0xA0CD,},
	{ 0x3666, 0x97AF,},
	{ 0x3680, 0xA0AB,},
	{ 0x3682, 0x5707,},
	{ 0x3684, 0x108B,},
	{ 0x3686, 0x2F8D,},
	{ 0x3688, 0x1BED,},
	{ 0x368A, 0x956C,},
	{ 0x368C, 0x3C89,},
	{ 0x368E, 0x91CB,},
	{ 0x3690, 0x79AB,},
	{ 0x3692, 0x7F6C,},
	{ 0x3694, 0x6266,},
	{ 0x3696, 0xA32B,},
	{ 0x3698, 0x84CF,},
	{ 0x369A, 0x8E2C,},
	{ 0x369C, 0x6BAF,},
	{ 0x369E, 0xE109,},
	{ 0x36A0, 0x47EC,},
	{ 0x36A2, 0xA98F,},
	{ 0x36A4, 0x944C,},
	{ 0x36A6, 0x4A2F,},
	{ 0x36C0, 0x11D1,},
	{ 0x36C2, 0x17CD,},
	{ 0x36C4, 0x8812,},
	{ 0x36C6, 0xE56F,},
	{ 0x36C8, 0x5073,},
	{ 0x36CA, 0x36F1,},
	{ 0x36CC, 0x022E,},
	{ 0x36CE, 0x8A31,},
	{ 0x36D0, 0xC96F,},
	{ 0x36D2, 0x6272,},
	{ 0x36D4, 0x0151,},
	{ 0x36D6, 0x32AE,},
	{ 0x36D8, 0x8992,},
	{ 0x36DA, 0xD90F,},
	{ 0x36DC, 0x69F3,},
	{ 0x36DE, 0x1271,},
	{ 0x36E0, 0x6BCC,},
	{ 0x36E2, 0x8692,},
	{ 0x36E4, 0xA0CF,},
	{ 0x36E6, 0x4933,},
	{ 0x3700, 0xFA2E,},
	{ 0x3702, 0xCF8C,},
	{ 0x3704, 0x1850,},
	{ 0x3706, 0xCF4F,},
	{ 0x3708, 0xE671,},
	{ 0x370A, 0xF00E,},
	{ 0x370C, 0xFF6D,},
	{ 0x370E, 0x7BCE,},
	{ 0x3710, 0xA96D,},
	{ 0x3712, 0xCA70,},
	{ 0x3714, 0xD70C,},
	{ 0x3716, 0x1B0C,},
	{ 0x3718, 0x716F,},
	{ 0x371A, 0x8E2B,},
	{ 0x371C, 0xCC51,},
	{ 0x371E, 0xD6EE,},
	{ 0x3720, 0x872E,},
	{ 0x3722, 0x0D90,},
	{ 0x3724, 0xFAEC,},
	{ 0x3726, 0xE710,},
	{ 0x3740, 0xB34F,},
	{ 0x3742, 0x87AE,},
	{ 0x3744, 0x08F4,},
	{ 0x3746, 0x0892,},
	{ 0x3748, 0x9CB5,},
	{ 0x374A, 0x8E50,},
	{ 0x374C, 0xF58E,},
	{ 0x374E, 0x7132,},
	{ 0x3750, 0x5C91,},
	{ 0x3752, 0x9094,},
	{ 0x3754, 0xE090,},
	{ 0x3756, 0xAD30,},
	{ 0x3758, 0x0E34,},
	{ 0x375A, 0x0092,},
	{ 0x375C, 0xA315,},
	{ 0x375E, 0xAD2F,},
	{ 0x3760, 0x09AB,},
	{ 0x3762, 0x0154,},
	{ 0x3764, 0x32F1,},
	{ 0x3766, 0x8CD5,},
	{ 0x3784, 0x0278,},
	{ 0x3782, 0x01EC,},
	{ 0x37C0, 0xCA47,},
	{ 0x37C2, 0x49C5,},
	{ 0x37C4, 0x9AEA,},
	{ 0x37C6, 0xE348,},
	{ 0x098E, 0x0000,},
	{ 0xC960, 0x0AF0,},
	{ 0xC962, 0x79CF,},
	{ 0xC964, 0x6A40,},
	{ 0xC966, 0x790F,},
	{ 0xC968, 0x78C6,},
	{ 0xC96A, 0x0FA0,},
	{ 0xC96C, 0x7FCD,},
	{ 0xC96E, 0x800D,},
	{ 0xC970, 0x7F8E,},
	{ 0xC972, 0x7ECA,},
	{ 0xC974, 0x1964,},
	{ 0xC976, 0x7CFB,},
	{ 0xC978, 0x70C6,},
	{ 0xC97A, 0x7EA8,},
	{ 0xC97C, 0x784F,},
	{ 0xC95E, 0x0003,},
	{ 0x3786, 0x0004,},
	{ 0x3788, 0x03CB,},
	{ 0x378A, 0x0004,},
	{ 0x378C, 0x050B,},

	/*[CCM]*/
	{0xC892, 0x0267,},
	{0xC894, 0xFF1A,},
	{0xC896, 0xFFB3,},
	{0xC898, 0xFF80,},
	{0xC89A, 0x0166,},
	{0xC89C, 0x0003,},
	{0xC89E, 0xFF9A,},
	{0xC8A0, 0xFEB4,},
	{0xC8A2, 0x024D,},
	{0xC8A4, 0x01BF,},
	{0xC8A6, 0xFF01,},
	{0xC8A8, 0xFFF3,},
	{0xC8AA, 0xFF75,},
	{0xC8AC, 0x0198,},
	{0xC8AE, 0xFFFD,},
	{0xC8B0, 0xFF9A,},
	{0xC8B2, 0xFEE7,},
	{0xC8B4, 0x02A8,},

	{0xC8B6, 0x0100,},
	{0xC8B8, 0xFF84,},
	{0xC8BA, 0xFFEB,},
	{0xC8BC, 0xFFBD,},
	{0xC8BE, 0x0105,},
	{0xC8C0, 0xFFDB,},
	{0xC8C2, 0xFFCC,},
	{0xC8C4, 0xFE2A,},
	{0xC8C6, 0x03AB,},

	{0xC8DA, 0x004D,},
	{0xC8DC, 0x0096,},
	{0xC8DE, 0x001D,},
	{0xC8E0, 0x004D,},
	{0xC8E2, 0x0096,},
	{0xC8E4, 0x001D,},
	{0xC8E6, 0x004D,},
	{0xC8E8, 0x0096,},
	{0xC8EA, 0x001D,},

	/*[AWB Gain Setting]*/
	{0xC8C8, 0x0075,},
	{0xC8CA, 0x011C,},
	{0xC8CC, 0x00A0,},
	{0xC8CE, 0x00F3,},
	{0xC8D0, 0x00B3,},
	{0xC8D2, 0x009D,},
	{0xC8D4, 0x0A8C,},
	{0xC8D6, 0x1004,},
	{0xC8D8, 0x1964,},

	/*[AWB]*/
	{0xC914, 0x0000,},
	{0xC916, 0x0000,},
	{0xC918, 0x04FF,},
	{0xC91A, 0x03BF,},
	{0xC8F2, 0x04, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC8F3, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC904, 0x0037,},
	{0xC906, 0x003C,},
	{0xC8F4, 0x0000,},
	{0xC8F6, 0x0000,},
	{0xC8F8, 0x0000,},
	{0xC8FA, 0xE724,},
	{0xC8FC, 0x1583,},
	{0xC8FE, 0x2045,},
	{0xC900, 0x05DC,},
	{0xC902, 0x007C,},
	{0xC90A, 0x1004,},
	{0xC90C, 0x78, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC90D, 0x80, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC90E, 0x88, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC90F, 0x76, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC910, 0x83, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC911, 0x78, MSM_CAMERA_I2C_BYTE_DATA},
	{0x098E, 0xAC06,},
	{0xAC06, 0x63, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC07, 0x65, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC08, 0x63, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC09, 0x65, MSM_CAMERA_I2C_BYTE_DATA},

	/*[Step7-CPIPE_Preference]*/
	{0xC926, 0x0020,},
	{0xC928, 0x009A,},
	{0xC946, 0x0070,},
	{0xC948, 0x00F3,},
	{0xC952, 0x0090,},
	{0xC954, 0x02C0,},
	{0xC95A, 0x04, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92A, 0x50, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92B, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92C, 0x00, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92D, 0xFF, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92E, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC92F, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC930, 0x06, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC931, 0x64, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC932, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC933, 0x0C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC934, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC935, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC936, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC937, 0x0F, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC938, 0x64, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC939, 0x64, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC93A, 0x64, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC93B, 0x32, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC93C, 0x0005 ,},
	{0xC93E, 0x0358 ,},
	{0xC940, 0x00DC ,},
	/*CAM_LL_START_CONTRAST_GRADIENT*/
	{0xC942, 0x3C, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC943, 0x30, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC944, 0x22, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC945, 0x19, MSM_CAMERA_I2C_BYTE_DATA},
	{0x098E, 0xC942,},
	{0x098E, 0xC92A,},
	{0xC94A, 0x00F0,},
	{0xC94C, 0x0010,},
	{0xC94E, 0x01CD,},
	{0xC950, 0x05, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC951, 0x40, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC87A, 0x37, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC87B, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC878, 0x0E, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC890, 0x0040 ,},
	{0xC882, 0x0100 ,},
	{0xC886, 0x0100 ,},
	{0xA404, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	/*AE_TRACK_AE_TRACKING_DAMPENING*/
	{0xC87C, 0x0016,},
	{0xB42A, 0x05, MSM_CAMERA_I2C_BYTE_DATA},
	{0xA80A, 0x18, MSM_CAMERA_I2C_BYTE_DATA},
	{0x326C, 0x1706,},

	/*Step8-Features*/
	{0x098E, 0x0000,},
	{0xC984, 0x8040,},
	{0x001E, 0x0777,},

	/* LOAD=MIPI setting for SOC1040 */
	{0xC984, 0x8001,},
	{0xC988, 0x0F00,},
	{0xC98A, 0x0B07,},
	{0xC98C, 0x0D01,},
	{0xC98E, 0x071D,},
	{0xC990, 0x0006,},
	{0xC992, 0x0A0C,},

	/* Improve AE Dampening speed */
	/* [Speed up AE/AWB] */
	{0x098E, 0x2802,},
	{0xC908, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC879, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
	{0xC909, 0x02, MSM_CAMERA_I2C_BYTE_DATA},
	{0xA80B, 0x18, MSM_CAMERA_I2C_BYTE_DATA},
	{0xAC16, 0x18, MSM_CAMERA_I2C_BYTE_DATA},
	{0x31E0, 0x01, MSM_CAMERA_I2C_BYTE_DATA},
};
#endif
