/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>

#if ENABLE_NLS
#  include <libintl.h>
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <string.h>

#define HFS_SIGNATURE	0x4244

static PedGeometry*
hfs_probe (PedGeometry* geom)
{
	char	buf[512];

	if (geom->length < 2)
		return 0;
	if (!ped_geometry_read (geom, buf, 2, 1))
		return 0;

	if (PED_BE16_TO_CPU (*(uint16_t*) buf) == HFS_SIGNATURE)
		return ped_geometry_duplicate (geom);
	else
		return NULL;
}

#ifndef DISCOVER_ONLY
static int
hfs_clobber (PedGeometry* geom)
{
	char	buf[512];

	memset (buf, 0, 512);
	return ped_geometry_write (geom, buf, 2, 1);
}
#endif /* !DISCOVER_ONLY */

static PedFileSystemOps hfs_ops = {
	probe:		hfs_probe,
#ifndef DISCOVER_ONLY
	clobber:	hfs_clobber,
#else
	clobber:	NULL,
#endif
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_create_constraint:	NULL,
	get_resize_constraint:	NULL,
	get_copy_constraint:	NULL,
};

static PedFileSystemType hfs_type = {
	next:	NULL,
	ops:	&hfs_ops,
	name:	"hfs"
};

void
ped_file_system_hfs_init ()
{
	ped_file_system_type_register (&hfs_type);
}

void
ped_file_system_hfs_done ()
{
	ped_file_system_type_unregister (&hfs_type);
}

