/*	Copyright (C) 1995, 1996 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


typedef struct SCM_PTHREAD_INFO {
  queue q;		     /* the dequeue on which this structure exists */
  pthread_t thread;	     /* the corresponding thread structure */
  void *stack_top;	     /* the highest address in this thread's stack */
  void *(*launch)(void *);   /* the thread's real start routine */
  void *arg;		     /* the thread's real start routine argument */
  SCM *prots;                /* prots for this thread */
} scm_pthread_info;

#define struct_base(PTR, FLD, TYP) ((TYP*)((char*)(PTR)-(char*)(&(((TYP*)0)->FLD))))

pthread_mutex_t scm_critical_section_mutex;
pthread_t scm_critical_section_owner;

static queue infos = { &infos, &infos };  /* the dequeue of info structures */

static struct gscm_type scm_thread_type;
static struct gscm_type scm_mutex_type;
static struct gscm_type scm_condition_variable_type;

static queue infos;

/* Key to thread specific data */
pthread_key_t info_key;

/* This mutex is used to synchronize thread creation */
static pthread_mutex_t scm_pthread_create_mutex;

#ifdef __STDC__
int
gscm_threads_thread_equal (SCM t1, SCM t2)
#else
int
gscm_threads_thread_equal ()
     SCM t1, t2;
#endif
{
  return (pthread_equal(* (pthread_t *) gscm_unwrap_obj (&scm_thread_type, &t1),
			* (pthread_t *) gscm_unwrap_obj (&scm_thread_type, &t2)));
}

#ifdef __STDC__
void
gscm_threads_thread_die ()
#else
void
gscm_threads_thread_die ()
#endif
{
  /* He's dead, Jim */
}

#ifdef __STDC__
void
gscm_threads_mutex_die (SCM m)
#else
void
gscm_threads_scm_mutex_die ()
     SCM m;
#endif
{
  pthread_mutex_t *mutex = (pthread_mutex_t *) 
    gscm_unwrap_obj (&scm_mutex_type, &m);

  pthread_mutex_destroy(mutex);

  /* He's dead, Jim */
}

#ifdef __STDC__
void
gscm_threads_condition_variable_die (SCM c)
#else
void
gscm_threads_condition_variable_die (c)
     SCM c;
#endif
{
  pthread_cond_t *cond = (pthread_cond_t *) 
    gscm_unwrap_obj (&scm_condition_variable_type, &c);

  pthread_cond_destroy(cond);

  /* He's dead, Jim */
}

#ifdef __STDC__
void
gscm_threads_init ()
#else
void
gscm_threads_init ()
#endif
{
}

/* cleanup for info structure
 */
#ifdef __STDC__
static void
scm_pthread_delete_info (void *ptr)
#else
static void
scm_pthread_delete_info ()
     void *ptr;
#endif
{
  scm_pthread_info *info= (scm_pthread_info *)ptr;
  info->q.blink->flink= info->q.flink;
  info->q.flink->blink= info->q.blink;
  free(info);
}

#ifdef __STDC__
void
gscm_threads_init_mit_pthreads ()
#else
void
gscm_threads_init_mit_pthreads ()
#endif
{
  SCM *prots;

  prots = (SCM *)malloc (sizeof (SCM) * scm_num_thread_local_protects);
  pthread_attr_setcleanup (&pthread_self()->attr, free, prots);

  /* Initialize the root thread specific data pointer. All new threads
     get a copy of this buffer. 
  scm_root_prots = prots; */
  
  /*
   * each info structure is made thread-specific, so that the cleanup
   * mechanism can be used to reclaim the space in a timely fashion.
   */
  pthread_key_create(&info_key, scm_pthread_delete_info);

  /* initialize various mutex variables */
  pthread_mutex_init(&scm_critical_section_mutex, NULL);

  /*
   * create an info structure for the initial thread and push it onto
   * the info dequeue
   */
  {
    scm_pthread_info *info = 
      (scm_pthread_info *)malloc(sizeof(scm_pthread_info));
    infos.flink= infos.blink= &info->q;
    info->q.flink= info->q.blink= &infos;
    info->thread= pthread_initial;
    /* this is heuristic, but is unlikely to ever fail */
    info->stack_top= (void *)(((unsigned long)&info+pthread_pagesize)
			      & ~(pthread_pagesize-1));
    info->prots = prots;
    pthread_setspecific(info_key, info);
  }
}

/* given some thread, find the corresponding info
 */
static scm_pthread_info *pthreads_find_info(pthread_t target)
{
  queue *ptr= infos.flink;

  while (ptr != &infos)
    {
      scm_pthread_info *info= struct_base(ptr, q, scm_pthread_info);

      if (info->thread == target)
        {
          return (info);
        }
      ptr= ptr->flink;
    }
}

#ifdef __STDC__
void
gscm_threads_mark_stacks ()
#else
void
gscm_threads_mark_stacks ()
#endif
{
  int j;
  jmp_buf scm_save_regs_gc_mark;
  pthread_t thread;
  
  for (thread= pthread_link_list; thread; thread= thread->pll)
    {
      if (thread->state != PS_DEAD)
	{
	  scm_pthread_info *info = pthreads_find_info(thread);
	  
	  if (thread == pthread_run)
	    {
	      /* Active thread */
	      /* stack_len is long rather than sizet in order to guarantee
		 that &stack_len is long aligned */
#ifdef STACK_GROWS_UP
	      long stack_len = (STACKITEM *) (&thread) -
		(STACKITEM *) info->stack_top;
	      
	      scm_mark_locations (((size_t)info->stack_top,
				   (sizet) stack_len, BOOL_T));
#else
	      long stack_len = (STACKITEM *) info->stack_top -
		(STACKITEM *) (&thread);
	      
	      /* Protect from the C stack.  This must be the first marking
	       * done because it provides information about what objects
	       * are "in-use" by the C code.   "in-use" objects are  those
	       * for which the values from LENGTH and CHARS must remain
	       * usable.   This requirement is stricter than a liveness
	       * requirement -- in particular, it constrains the implementation
	       * of scm_resizuve.
	       */
	      FLUSH_REGISTER_WINDOWS;
	      /* This assumes that all registers are saved into the jmp_buf */
	      setjmp (scm_save_regs_gc_mark);
	      scm_mark_locations ((STACKITEM *) scm_save_regs_gc_mark,
				  ((sizet) sizeof scm_save_regs_gc_mark
				   / sizeof (STACKITEM)), BOOL_T);
	      
	      scm_mark_locations ((STACKITEM *) &thread,
				  stack_len, BOOL_T);
#endif
	    }
	  else
	    {
	      /* Suspended thread */
#ifdef STACK_GROWS_UP
	      long stack_len = (STACKITEM *) (thread->THREAD_SP) -
		(STACKITEM *) info->stack_top;
	      
	      scm_mark_locations (((size_t)info->stack_top,
				   (sizet) stack_len, BOOL_T));
#else
	      long stack_len = (STACKITEM *) info->stack_top -
		(STACKITEM *) (thread->THREAD_SP);
	      
	      scm_mark_locations((STACKITEM *) thread->machdep_data.machdep_state,
				 ((sizet)sizeof(*thread->machdep_data.machdep_state)
				  / sizeof (STACKITEM)),
				 BOOL_T);
	      scm_mark_locations ((STACKITEM *) (size_t)thread->THREAD_SP,
				  stack_len, BOOL_T);
#endif
	    }
	  
	  /* Mark all the of this thread's thread-local protects */
	  for (j = scm_num_thread_local_protects-1; j >= 0; j--)
	    {
	      scm_gc_mark (info->prots[j], BOOL_F);
	    }
	}
      else /* It's dead, Jim */
	{
	  /* fprintf(stderr, "not marking stack - thread is dead\n"); */
	}
    }
}

#ifdef __STDC__
void *
launch_thread (void *p)
#else
void *
launch_thread ()
     void *p;
#endif
{
  SCM thunk = scm_pthread_create_info.thunk;
  SCM error = scm_pthread_create_info.error;
  SCM *prots = scm_pthread_create_info.prots;

  pthread_mutex_unlock(&scm_pthread_create_mutex);

  pthread_attr_setcleanup (&pthread_self()->attr, free, prots);

  /* dynwinds must be set to BOOL_F for each new thread
     (it is a thread-local variable) */
  dynwinds = BOOL_F;
  
  scm_with_dynamic_root (thunk, error);

  return NULL;
}

#ifdef __STDC__
SCM
gscm_threads_with_new_thread (SCM thunk, SCM error_thunk)
#else
SCM
gscm_threads_with_new_thread (thunk, error_thunk)
     SCM thunk;
     SCM error_thunk;
#endif
{
  int rc;
  pthread_attr_t attr;
  SCM t = gscm_alloc (&scm_thread_type, sizeof(pthread_t));

  pthread_t *pt = (pthread_t *) gscm_unwrap_obj (&scm_thread_type, &t);

  int status;
  scm_pthread_info *info = 
    (scm_pthread_info *)malloc(sizeof(scm_pthread_info));

  /* Rather than allocate space to hold fn and arg, a mutex is used
     to serialize thread creation. */
  pthread_mutex_lock(&scm_pthread_create_mutex); 
  
  /* thread mustn't start until we've built the info struct */
  pthread_kernel_lock++;

  /* this data is passed to the newly created thread */
  scm_pthread_create_info.thunk = thunk;
  scm_pthread_create_info.error = error_thunk;
  
  /* initialize and create the thread. */
  pthread_attr_init(&attr);
  pthread_attr_setschedpolicy(&attr, SCHED_RR);
  
  pthread_create(pt, &attr, launch_thread, &scm_pthread_create_info);
  pthread_attr_destroy(&attr);

  /* push the info onto the dequeue */
  info->q.flink= infos.flink;
  info->q.blink= &infos;
  infos.flink->blink= &info->q;
  infos.flink= &info->q;
  /* fill it in the blanks */
  info->launch= launch_thread;
  info->arg= &scm_pthread_create_info;
  /* pthread_create filled in the initial SP -- profitons-en ! */
  info->stack_top= (void *)((*pt)->THREAD_SP);
  info->thread = *pt;

  {
    SCM * prots;

    prots = (SCM *)malloc (sizeof (SCM) * scm_num_thread_local_protects);

    info->prots = prots;
    scm_pthread_create_info.prots = prots;

    /* Copy root thread specific data over */
    memcpy(prots, (SCM*)pthread_self()->attr.arg_attr, 
	   sizeof (SCM) * scm_num_thread_local_protects);

    prots[SCM_THREAD_T] = t;
    prots[SCM_THREAD_THUNK] = thunk;
    prots[SCM_THREAD_ERROR] = error_thunk;
  }

  /* we're now ready for the thread to begin */
  pthread_kernel_lock--;

  return t;
}

#ifdef __STDC__
SCM
gscm_threads_make_mutex ()
#else
SCM
gscm_threads_make_mutex ()
#endif
{
  SCM t = gscm_alloc (&scm_mutex_type, sizeof(pthread_mutex_t));

  pthread_mutex_t *m = (pthread_mutex_t *) 
    gscm_unwrap_obj (&scm_mutex_type, &t);

  pthread_mutex_init(m, NULL);

  return t;
}

#ifdef __STDC__
SCM
gscm_threads_lock_mutex (SCM m)
#else
SCM
gscm_threads_lock_mutex ()
     SCM m;
#endif
{
  pthread_mutex_t *mutex = (pthread_mutex_t *) 
    gscm_unwrap_obj (&scm_mutex_type, &m);

  pthread_mutex_lock(mutex);

  return SCM_BOOL_T;
}

#ifdef __STDC__
SCM
gscm_threads_unlock_mutex (SCM m)
#else
SCM
gscm_threads_unlock_mutex ()
     SCM m;
#endif
{
  pthread_mutex_t *mutex = (pthread_mutex_t *) 
    gscm_unwrap_obj (&scm_mutex_type, &m);

  pthread_mutex_unlock(mutex);

  return SCM_BOOL_T;
}

#ifdef __STDC__
SCM
gscm_threads_yield ()
#else
SCM
gscm_threads_yield ()
#endif
{
  return BOOL_F;
}

#ifdef __STDC__
SCM
gscm_threads_join_thread (SCM t)
#else
SCM
gscm_threads_join_thread ()
     SCM t;
#endif
{
  void *value;
  pthread_join(*(pthread_t *) gscm_unwrap_obj (&scm_thread_type, &t), &value);
  return BOOL_T;
}

#ifdef __STDC__
SCM
gscm_threads_make_condition_variable ()
#else
SCM
gscm_threads_make_condition_variable ()
#endif
{
  return SCM_BOOL_F;
}

#ifdef __STDC__
SCM
gscm_threads_condition_variable_wait (SCM c, SCM m)
#else
SCM
gscm_threads_condition_variable_wait (c, m)
     SCM c;
     SCM m;
#endif
{
  return SCM_BOOL_T;
}

#ifdef __STDC__
SCM
gscm_threads_condition_variable_signal (SCM c)
#else
SCM
gscm_threads_condition_variable_signal (c)
     SCM c;
#endif
{
  return SCM_BOOL_T;
}
