/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2020 Yegor Bugayenko
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.cactoos.text;

import org.junit.jupiter.api.Test;
import org.llorllale.cactoos.matchers.Assertion;
import org.llorllale.cactoos.matchers.TextHasString;

/**
 * Test case for {@link Abbreviated}.
 * @since 0.29
 * @checkstyle JavadocMethodCheck (500 lines)
 */
@SuppressWarnings("PMD.TooManyMethods")
final class AbbreviatedTest {

    @Test
    void abbreviatesAnEmptyText() {
        final String msg = "";
        new Assertion<>(
            "Can't abbreviate an msg text",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated(msg, 8),
            new TextHasString(msg)
        ).affirm();
    }

    @Test
    void abbreviatesText() {
        new Assertion<>(
            "Can't abbreviate a text",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated("hello world", 8),
            new TextHasString("hello...")
        ).affirm();
    }

    @Test
    void abbreviatesTextOneCharSmaller() {
        new Assertion<>(
            "Can't abbreviate a text one char smaller",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated("oo programming", 10),
            new TextHasString("oo prog...")
        ).affirm();
    }

    @Test
    void abbreviatesTextWithSameLength() {
        final String msg = "elegant objects";
        new Assertion<>(
            "Can't abbreviate a text with same length",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated(msg, 15),
            new TextHasString(msg)
        ).affirm();
    }

    @Test
    void abbreviatesTextOneCharBigger() {
        final String msg = "the old mcdonald";
        new Assertion<>(
            "Can't abbreviate a text one char bigger",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated(msg, 17),
            new TextHasString(msg)
        ).affirm();
    }

    @Test
    void abbreviatesTextTwoCharsBigger() {
        final String msg = "hi everybody!";
        new Assertion<>(
            "Can't abbreviate a text two chars bigger",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated(msg, 15),
            new TextHasString(msg)
        ).affirm();
    }

    @Test
    void abbreviatesTextWithWidthBiggerThanLength() {
        final String msg = "cactoos framework";
        new Assertion<>(
            "Can't abbreviate a text with width bigger than length",
            // @checkstyle MagicNumber (1 line)
            new Abbreviated(msg, 50),
            new TextHasString(msg)
        ).affirm();
    }

    @Test
    void abbreviatesTextBiggerThanDefaultMaxWidth() {
        // @checkstyle LineLengthCheck (10 line)
        new Assertion<>(
            "Can't abbreviate a text bigger than default max width",
            new Abbreviated(
                "The quick brown fox jumps over the lazy black dog and after that returned to the cave"
            ),
            new TextHasString(
                "The quick brown fox jumps over the lazy black dog and after that returned to ..."
            )
        ).affirm();
    }

}
