/****************************************************************************** 
 *
 * File:        midiprint.c
 * Version:     $Id: midiprint.c,v 1.10 1995/07/30 01:46:11 burgaard Exp $
 *              $Version: 0.2$
 *
 * Purpose:     Standard MIDI File Pretty Printing
 *
 * Project:     MIDI/Sequencer library.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Compile and see ../doc/libmidifile.tex for documentation on this file.
 *
 * This file contains pretty printing functions.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <stdio.h>

#include "miditypes.h"
#include "midiconst.h"
#include "midiqueue.h"
#include "midiprint.h"

/*** CONSTANTS ****************************************************************/

/* This array is indexed after the most significant 4 BITS of a CHANNEL MESSAGE
 *                                         0 1 2 3 4 5 6 7 8 9 A B C D E F */
/*
static const char MIDI_CHMSG_ARGS[0x10] = {0,0,0,0,0,0,0,0,2,2,2,2,1,1,2,0};
*/

const char *MIDI_META_SMPTE_TXT[5] = 
{
  "Hour",
  "Minutte",
  "Second",
  "Frame",
  "Frame Fraction"
};

const char MIDI_META_KEY_SIG_OFFSET = 7; /* to get C/Am */
const char *MIDI_META_KEY_SIG_MAJOR[15] =
{
  "Cb"
  "Gb",
  "Db",
  "Ab",
  "Eb",
  "Bb",
  "F",
  "C",
  "G",
  "D",
  "A",
  "E",
  "H",
  "F#",
  "C#"
};
const char *MIDI_META_KEY_SIG_MINOR[15] =
{
  "Gbm",
  "Bbm",
  "Fm",
  "Cm",
  "Gm",
  "Dm",
  "Am",
  "Em",
  "Hm",
  "F#m",
  "C#m",
  "G#m",
  "D#m",
  "A#m"
};

/*** MIDI FILE PRETTY PRINTING ************************************************/

void midi_printdata(int size, byte * data)
{
  int i = 0;
  printf("[ ");
  for (i = 0; i < size; i++) printf("%02hX ", data[i]&0xff);
  printf("]");
}

void midi_printtext(int size, byte * data)
{
  int i = 0;
  printf("\"");
  for (i = 0; i < size; i++) printf("%c", data[i]&0xff);
  printf("\"");
}

void midi_printunknown(midi_event *event, enum MIDI_PRINT level)
{
  if (!level) return;
  printf("UNKNOWN ");
  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };

  if (level > MIDI_PRINT_META)

  printf(" FORMAT: ");
  midi_printdata(event->size, event->data);
  printf("\n");
}

void midi_printmeta(midi_event *event, enum MIDI_PRINT level)
{
  long i = 0;
  word j = 0;

  if (!level) return;
  if (level > MIDI_PRINT_SPARSE) printf("META  ");

  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };

  if (!event->data || !event->size)
    {
      printf("NULL DATA!\n");
      return;
    };

  switch (event->data[0]&0xff)
    {
    case MID_META_SEQ_NUMBER: 
      if (level < MIDI_PRINT_META) break;
      printf("SEQUENCE #       ");
      if (event->size == 3)
	{
	  i = ((event->data[1]&0xff) << 8) | (event->data[2]&0xff);
	  printf("%ld", i);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);

      break;
      
    case MID_META_TEXT:
      printf("TEXT:            ");
      midi_printtext(event->size - 1, &event->data[1]);
      level++;
      break;
    case MID_META_COPYRIGHT:
      printf("COPYRIGHT:       ");
      midi_printtext(event->size - 1, &event->data[1]);
      level++;
      break;
    case MID_META_SEQ_NAME:
      printf("SEQUENCE NAME:   ");
      midi_printtext(event->size - 1, &event->data[1]);
      level++;
      break;
    case MID_META_INST_NAME:
      if (level < MIDI_PRINT_META) break;
      printf("INSTRUMENT NAME: ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
    case MID_META_LYRIC:
      if (level < MIDI_PRINT_META) break;
      printf("LYRIC:           ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
    case MID_META_MARKER:
      if (level < MIDI_PRINT_META) break;
      printf("MARKER:          ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
    case MID_META_CUE_POINT:
      if (level < MIDI_PRINT_META) break;
      printf("CUE POINT:       ");
      midi_printtext(event->size - 1, &event->data[1]);
      break;
      
    case MID_META_CH_PREFIX:
      if (level < MIDI_PRINT_META) break;
      printf("CHANNEL PREFIX # ");
      if (event->size == 2)
	printf("%hd", event->data[1]&0xff);
      else
	midi_printdata(event->size - 1, &event->data[1]);

      break;
    case MID_META_EOT:
      if (level < MIDI_PRINT_META) break;
      printf("END OF TRACK");
      break;
    case MID_META_SET_TEMPO:
      if (level < MIDI_PRINT_META) break;
      printf("TEMPO            ");
      if (event->size == 4)
	{
	  i  = (event->data[1]&0xff) << 16;
	  i |= (event->data[2]&0xff) << 8;
	  i |= (event->data[3]&0xff);

	  i = ( 60.0 * 1000000.0 / (float)i );
	  printf("%03ld BPM", i);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);
      break;
    case MID_META_SMPTE_OFFSET:
      if (level < MIDI_PRINT_META) break;
      printf("SMPTE OFFSET     ");
      if (event->size == 6)
	{
	  for (j = 1; j < event->size; j++) 
	    printf("%s: %02hu ", MIDI_META_SMPTE_TXT[j-1], event->data[j]&0xff);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);

      break;
    case MID_META_TIME_SIG:
      if (level < MIDI_PRINT_META) break;
      printf("TIME SIGNATURE   ");
      if (event->size == 5)
	printf("%d/%d", event->data[1]&0xff, 1<<(event->data[2]&0xff));
      else
	midi_printdata(event->size - 1, &event->data[1]);
      break;
    case MID_META_KEY_SIG:
      if (level < MIDI_PRINT_META) break;
      printf("KEY SIGNATURE    ");
      if (event->size == 3)
	{
	  if (event->data[2]&0xff)
	    {
	      i = (signed long)(event->data[1]&0xff) + MIDI_META_KEY_SIG_OFFSET;
	      printf("%s", MIDI_META_KEY_SIG_MINOR[i]);
	    }
	  else
	    {
	      i = (signed long)(event->data[1]&0xff) + MIDI_META_KEY_SIG_OFFSET;
	      printf("%s", MIDI_META_KEY_SIG_MAJOR[i]);
	    };

	  if ((signed char)event->data[1]&0xff < 0)
	    printf(" (%hd b)", -event->data[1]);
	  else if ((signed char)event->data[1]&0xff > 0)
	    printf(" (%hd #)", event->data[1]);
	}
      else
	midi_printdata(event->size - 1, &event->data[1]);
      break;
    case MID_META_SEQ_SPECIFIC:
      if (level < MIDI_PRINT_META) break;
      printf("SEQ. SPECIFIC    ");
      midi_printdata(event->size - 1, &event->data[1]);
      break;

    default:
      if (level < MIDI_PRINT_META) break;
      printf("UNKNOWN          ");
      midi_printdata(event->size, &event->data[0]);
      break;
    };
  if (level > MIDI_PRINT_SPARSE) printf("\n");
}

void midi_printsysex(midi_event *event, enum MIDI_PRINT level)
{
  if (!level) return;
  if (level > MIDI_PRINT_SPARSE) printf("SYSEX ");
  
  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };
  
  if (!event->data || !event->size)
    {
      printf("NULL DATA!\n");
      return;
    };
  
  if (level > MIDI_PRINT_META)
    {
      switch (event->data[0]&0xff)
	{
	case MID_SYS_SYSEX:
	  midi_printdata(event->size, &event->data[0]);
	  break;
	case MID_EVENT_OTHER:
	  if (level < MIDI_PRINT_META) break;
	  midi_printdata(event->size, &event->data[0]);
	  break;
	default:
	  if (level < MIDI_PRINT_META) break;
	  midi_printdata(event->size - 1, &event->data[1]);
	  break;
	};
    };

  if (level > MIDI_PRINT_SPARSE) printf("\n");
}

void midi_printvoice(midi_event *event, enum MIDI_PRINT level)
{
  static int bank_sel_run = 0;
  static int bank_sel_mm = 0;
  static int bank_sel_ll = 0;
  static int data_mm = 0;
  static int data_ll = 0;
  static int nrpn_run = 0;
  static int nrpn_mm = 0;
  static int nrpn_ll = 0;
  static int rpn_run = 0;
  static int rpn_mm = 0;
  static int rpn_ll = 0;
  static int cmd = 0;
  int j = 0;
  int i = 0;

  if (!level) return;
  if (level > MIDI_PRINT_META) printf("VOICE ");

  if (!event)
    {
      printf("NULL EVENT!\n");
      return;
    };

  if (!event->data || !event->size)
    {
      printf("NULL DATA!\n");
      return;
    };

  if (level > MIDI_PRINT_META)
    {
      if ((cmd =  event->data[0]&0xff) < MID_EVENT_CHMSG_FIRST)
	{
	  fprintf(stderr, "Unexpected RUNNING status in voice command\n");
	  fprintf(stderr, "Running voice commands not allowed\n");
	  return;
	};

      printf("CHANNEL %02d ", (cmd & 0x0f) + 1);
      
      switch (cmd & 0xf0)
	{
	case MID_EVENT_NOTE_OFF:
	  printf("NOTE #%03hd    OFF  VELOCITY   %03hd", event->data[1]&0xff,
		 event->data[2]&0xff);
	  break;
	case MID_EVENT_NOTE_ON:
	  printf("NOTE #%03hd    ON   VELOCITY   %03hd", event->data[1]&0xff,
		 event->data[2]&0xff);
	  break;
	case MID_EVENT_POLY_PRESSURE:
	  printf("NOTE #%03hd    POLY AFTERTOUCH %hd", event->data[1]&0xff,
		 event->data[2]&0xff);
	  break;
	case MID_EVENT_CONTROL:
	  switch (event->data[1]&0xff)
	    {
	    case MID_CTRL_BANK_MSB:
	      bank_sel_mm = event->data[2]&0xff;
	      if (bank_sel_run)
		{
		  printf("BANK SELECT  %d", (bank_sel_mm << 7) | bank_sel_ll);
		  bank_sel_run = 0;
		}
	      else
		{
		  printf("(bank select running MSB)");
		  bank_sel_run = 1;
		};
	      break;
	    case MID_CTRL_BANK_LSB:
	      bank_sel_ll = event->data[2]&0xff;
	      if (bank_sel_run)
		{
		  printf("BANK SELECT  %d", (bank_sel_mm << 7) | bank_sel_ll);
		  bank_sel_run = 0;
		}
	      else
		{
		  printf("(bank select running LSB)");
		  bank_sel_run = 1;
		};
	      break;
	    case MID_CTRL_MODULATION:
	      printf("MODULATION   %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_PORTAMENTO_TIME:
	      printf("PORTA TIME   %hd", event->data[2]&0xff);
	      break;
	      
	    case MID_CTRL_DATA_MSB:
	      data_mm = event->data[2]&0xff;
	      if (rpn_run == 2)
		{
		  rpn_run = 0;
		  switch ( (rpn_mm << 7) | rpn_ll )
		    {
		    case MID_RPN_PITCH_BEND_SENSE:
		      printf("PITCH BEND SENSE     %hd", data_mm);
		      break;
		    case MID_RPN_MASTER_FINE_TUNE:
		      j = (data_ll&0x7f)|((data_mm&0x7f) << 7);
		      printf("MASTER FINE TUNING   %d", j - 0x2000);
		      break;
		    case MID_RPN_MASTER_COARSE_TUNE:
		      printf("MASTER COARSE TUNING %d semitones", data_mm - 0x40);
		      break;
		    case MID_RPN_RPN_RESET:
		      printf("RPN RESET");
		      break;
		    default:
		      printf("unknown RPN message");
		    };
		}
	      else if (nrpn_run)
		{
		  nrpn_run = 0;
		  switch ((nrpn_ll&0xff)|((nrpn_mm << 8)&0xff00))
		    {
		    case MID_NRPN_GS_VIBRA_RATE:
		      printf("VIBRA RATE   %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_VIBRA_DEPTH:
		      printf("VIBRA DEPTH  %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_VIBRA_DELAY:
		      printf("VIBRA DELAY  %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_TVF_CUTOFF:
		      printf("TVF CUTOFF   %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_TVF_RESONANCE:
		      printf("TVF RESONAN  %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_ENV_ATTACK:
		      printf("ENV ATTACK   %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_ENV_DECAY:
		      printf("ENV DECAY    %hd", data_mm - 0x40);
		      break;
		    case MID_NRPN_GS_ENV_RELEASE:
		      printf("ENV RELEASE  %hd", data_mm - 0x40);
		      break;
		    default:
		      switch (nrpn_mm)
			{
			case MID_NRPN_GS_DRUM_PITCH:
			  printf("PITCH COARSE %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			case MID_NRPN_GS_DRUM_VOLUME:
			  printf("TVA LEVEL    %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			case MID_NRPN_GS_DRUM_PAN_POT:
			  printf("PANPOT       %hd (drum instrument key %hd)", 
				 data_mm - 0x40, nrpn_ll);
			  break;
			case MID_NRPN_GS_DRUM_REVERB:
			  printf("REVERB LEVEL %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			case MID_NRPN_GS_DRUM_CHORUS:
			  printf("CHORUS LEVEL %hd (drum instrument key %hd)", 
				 data_mm, nrpn_ll);
			  break;
			default:
			  printf("unknown NRPN message. Probably not Roland GS");
			};
		    };
		}
	      else
		printf("(data running MSB)");

	      break;
	    case MID_CTRL_DATA_LSB:
	      data_ll = event->data[i]&0xff;
	      printf("(data running LSB)");
	      break;
	    case MID_CTRL_VOLUME:
	      printf("VOLUME       %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_PAN_POT:
	      printf("PANPOT       %hd", (event->data[2]&0xff) - 0x40);
	      break;
	    case MID_CTRL_EXPRESSION:
	      printf("EXPRESSION   %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_HOLD_1:
	      printf("HOLD 1       %s", (event->data[2]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case MID_CTRL_PORTAMENTO:
	      printf("PORTAMENTO   %s", (event->data[2]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case MID_CTRL_SOSTENUTO:
	      printf("SOSTENUTO    %s", (event->data[2]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case MID_CTRL_SOFT:
	      printf("SOFT         %s", (event->data[2]&0xff < 64) ? "OFF" : "ON");
	      break;
	    case MID_CTRL_GENERAL_4:
	      printf("GENERAL CTL4 %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_HOLD_2:
	      printf("HOLD 2       %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_GENERAL_5:
	      printf("GENERAL CTL5 %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_GENERAL_6:
	      printf("GENERAL CTL6 %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_GENERAL_7:
	      printf("GENERAL CTL7 %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_GENERAL_8:
	      printf("GENERAL CTL8 %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_PORTAMENTO_CTRL:
	      printf("PORTA CONTRL %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_REVERB_LEVEL:
	      printf("REVERB LEVEL %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_CHORUS_LEVEL:
	      printf("CHORUS LEVEL %hd", event->data[2]&0xff);
	      break;
	    case MID_CTRL_NON_REG_LSB:
	      nrpn_ll = event->data[2]&0xff;
	      printf("(NRPN running LSB)");
	      nrpn_run++;
	      if (nrpn_run > 2) printf(" -- warning: Too many running NRPN MSB bytes!!!"); 
	      break;
	    case MID_CTRL_NON_REG_MSB:
	      nrpn_mm = event->data[2]&0xff;
	      printf("(NRPN running MSB)");
	      nrpn_run++;
	      if (nrpn_run > 2) printf(" -- warning: Too many running NRPN MSB bytes!!!"); 
	      break;
	    case MID_CTRL_REG_MSB:
	      rpn_mm = event->data[2]&0xff;
	      printf("(RPN running MSB)");
	      rpn_run++;
	      if (rpn_run > 2) printf(" -- warning: Too many running RPN MSB bytes!!!"); 
	      break;
	    case MID_CTRL_REG_LSB:
	      rpn_ll = event->data[2]&0xff;
	      printf("(RPN running LSB)");
	      if (rpn_run > 2) printf(" -- warning: Too many running RPN LSB bytes!!!"); 
	      rpn_run++;
	      break;
	    case MID_CTRL_ALL_SOUNDS_OFF:
	      printf("ALL SOUNDS OFF");
	      break;
	    case MID_CTRL_RESET_ALL_CTRL:
	      printf("RESET ALL CONTROLLERS");
	      break;
	    case MID_CTRL_ALL_NOTES_OFF:
	      printf("ALL NOTES OFF");
	      break;
	    case MID_CTRL_OMNI_OFF:
	      printf("OMNI OFF");
	      break;
	    case MID_CTRL_OMNI_ON:
	      printf("OMNI ON");
	      break;
	    case MID_CTRL_POLY:
	      printf("POLYPHONIC MODE");
	      break;
	    case MID_CTRL_MONO:
	      printf("MONOPHONIC MODE");
	      break;
	    default:
	      printf("CTRL CHANGE  0x%02x 0x%02hx", event->data[1]&0xff, event->data[2]&0xff);
	    };
	  break;
	case MID_EVENT_PROG_CHG:
	  printf("PROG CHANGE  %hd", event->data[1]&0xff);
	  break;
	case MID_EVENT_CH_PRESSURE:
	  printf("AFTERTOUCH   %hd", event->data[1]&0xff);
	  break;
	case MID_EVENT_PITCH_BEND:
	  j = (event->data[1]&0xff)|((event->data[2]&0xff) << 7);
	  printf("PITCH BEND   %d", j - 0x2000);
	  break;
	};
      printf("\n");
    };
}

void midi_printevent(midi_event *event, enum MIDI_PRINT level)
{
  if (!event || !level) return;
  if (level > MIDI_PRINT_META) printf("TIME %08lx, SIZE %08x, ", event->time, event->size);

  switch (event->type&0xff)
    {
    case MIDI_TYP_VOICE:
      midi_printvoice(event, level);
      break;
    case MIDI_TYP_SYSEX:
      midi_printsysex(event, level);
      break;
    case MIDI_TYP_META:
      midi_printmeta(event, level);
      break;
    default:
      midi_printunknown(event, level);
      break;
    };
}

void midi_printrawevent(byte type, unsigned long time, word size, byte * data, enum MIDI_PRINT level)
{
  midi_event event;
  event.prev = event.next = NULL;
  event.type = type;
  event.time = time;
  event.size = size;
  event.data = data;
  midi_printevent(&event, level);
}

/*** END OF FILE **************************************************************/
