/****************************************************************************** 
 *
 * File:        mpuinfo.c
 * Version:     $Id: mpuinfo.c,v 1.32 1995/07/27 23:46:51 burgaard Exp $
 *              $Version: 1.3$
 *
 * Purpose:     Get MPU-401 information
 *
 * Project:     Roland MPU-401 Device driver for Linux 1.1.64 and higher.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994, 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/
/*** INCLUDES & DEFINES *******************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <ncurses/curses.h>

#include "mpuioctl.h"

/*** CONSTANTS ****************************************************************/

const char *title1 = "mpuinfo version 1.3";
const char *title2 = "Copyright (c) 1994, 1995 Kim Burgaard";

/*** GLOBAL VARIABLES *********************************************************/

int setdebug = -1;
int getdebug = 0;
int follow = 0;

int mpudev = -1;
mpu_status_t ms;

/*** ARGUMENT PARSING *********************************************************/

void usage(int exitcode)
{
  printf("%s\n", title1);
  printf("%s\n", title2);
  printf("\n");
  printf("Usage:   mpuinfo [options]\n");
  printf("\n");
  printf("   -s --setdebug <n> Set debug information level to n, where\n");
  printf("                     n is between 0 (disable) and 3 (full).\n");
  printf("   -g --getdebug     Display current debug level\n");
  printf("   -f --follow       Display detailed debug information continuously\n");
  printf("\n");
  printf("   -h --help         This help text.\n");
  exit(exitcode);
}

inline char isoption(char *s)
{
  return ( s && strlen(s) > 1 && s[0] == '-' );
}

inline char islongoption(char *s)
{
  return ( s && strlen(s) > 2 && s[0] == '-' && s[1] == '-' );
}

int parsearg(int argc, char *argv[])
{
  int i;
  word j;
  char doskip;

  i = 1;
  while (i < argc)
    {
      if ( islongoption(argv[i]) )
	{
	  if ( !strcmp("--setdebug", argv[i]) )
	    {
	      i++;
	      if (!(i < argc))
		{
		  fprintf(stderr, "Argument expected\n");
		  exit(1);
		}
	      else if (sscanf(argv[i], "%i", &setdebug))
		{
		  if (setdebug < 0 || setdebug > 3)
		    {
		      fprintf(stderr, "Argument out of range (0 to 3)\n");
		      exit(1);
		    };
		}
	      else
		{
		  fprintf(stderr, "Invalid argument %s\n", argv[i]);
		  exit(1);
		};
	    }
	  else if ( !strcmp("--follow", argv[i]) )
	    follow = 1;
	  else if ( !strcmp("--getdebug", argv[i]) )
	    getdebug = 1;
	  else if ( !strcmp("--help", argv[i]) )
	    usage(0);
	  else
	    {
	      fprintf(stderr, "Unknown option `%s'\n", argv[i]);
	      fprintf(stderr, "Try `mpuinfo --help' for more information\n");
	      exit(1);
	    };
	}
      else if ( isoption(argv[i]) )
	{
	  doskip = 0;
	  for (j = 1; j < strlen(argv[i]); j++)
	    {
	      switch (argv[i][j])
		{
		case 's':
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Argument expected\n");
		      exit(1);
		    }
		  else if (sscanf(argv[i], "%i", &setdebug))
		    {
		      if (setdebug < 0 || setdebug > 6)
			{
			  fprintf(stderr, "Argument out of range\n");
			  exit(1);
			};
		    }
		  else
		    {
		      fprintf(stderr, "Invalid argument %s\n", argv[i]);
		      exit(1);
		    };
		  doskip = 1;
		  i--;
		  break;
		case 'g':
		  getdebug = 1;
		  break;
		case 'f':
		  follow = 1;
		  break;
		case 'h':
		  usage(0);
		  break;
		default:
		  fprintf(stderr, "Unknown option `-%c'\n", argv[i][j]);
		  fprintf(stderr, "Try `mpuinfo --help' for more information\n");
		  exit(1);
		};
	    };
	  if (doskip) i++;
	};
      i++;
    };
  return 0;
}

/*** DRIVER INFORMATION *******************************************************/

int get_driver_info(void)
{
  mpudev = open("/dev/mpu401stat", O_RDWR);
  if (mpudev != -1)
    {
      ioctl(mpudev, MPUIOC_GET_STATUS, &ms);
      close(mpudev);
      return 1;
    };
  return 0;
}

int set_debug_lvl(int i)
{
    mpudev = open("/dev/mpu401stat", O_RDWR);
    if (mpudev != -1)
    {
	ioctl(mpudev, MPUIOC_SET_DEBUG_LVL, &i);
	close(mpudev);
	return 1;
    };
    return 0;
}

/* Horisontal spacing */
int MAR1 = 2;
int MAR2 = 16;
int MAR3 = 10;

/* Vertical spacing */
int VSP1 = 1;
int VSP2 = 0;
int VSP3 = 1;
int VSP4 = 1;

/* ``tabulator'' stops */
int COL1 = 2;
int COL2 = 2+16;
int COL3 = 18+10;
int COL4 = 28+16;
int COL5 = 44+10;
int COL6 = 54+16;

void print_header(long *WC, char *s)
{
  int i = 0, j = 0;
  *WC += VSP2;
  if (s)
    {
      attrset(COLOR_PAIR(5));
      for (j = 0; j < COL1; j++) mvprintw(*WC, j, " ");
      mvprintw(*WC, COL1, "%s%n", s, &i);
      for (j = i+COL1; j < COLS; j++) mvprintw(*WC, j, " ");
    };
  *WC += VSP3;
}

void header_skip(long *WC)
{
  *WC += (VSP2 + VSP3);
}

void print_topics(long *WC, char *s1, char *s2, char *s3)
{
  attrset(COLOR_PAIR(3));
  if (s1) mvprintw(*WC, COL1, "%s:", s1);
  if (s2) mvprintw(*WC, COL3, "%s:", s2);
  if (s3) mvprintw(*WC, COL5, "%s:", s3);
  *WC += VSP4;
}

inline void topic_skip(long *WC)
{
  *WC += VSP4;
}

void build_display(void)
{
  long WC = 0;
  int i, j;

  /* setup columns etc. */
  if ((COLS < 80) || (LINES < 25))
    {
      endwin();
      resetty();
      fprintf(stderr, "Sorry, this option requieres minimum a 80x25 terminal\n");
      exit(1);
    };
  if (LINES >= (25+7)) VSP2++;
  if (LINES >= (25+14)) VSP3++;
  if (LINES >= (25+21)) VSP2++;
  if (LINES >= (25+28)) VSP3++;
  if (COLS > 80)
    {
      MAR2 = ((COLS-(3*MAR3))/3)-1;
      MAR1 = COLS-(3*MAR2)-(3*MAR3)-1;
    };

  COL1 = MAR1;
  COL2 = COL1+MAR2;
  COL3 = COL2+MAR3;
  COL4 = COL3+MAR2;
  COL5 = COL4+MAR3;
  COL6 = COL5+MAR2;

  /* make a blue background */
  attrset(COLOR_PAIR(1));
  for (i = 0; i < LINES; i++)
    for (j = 0; j < COLS; j++)
      mvprintw(i, j, " ");

  /* print text */
  attrset(COLOR_PAIR(1)|A_BOLD);
  mvprintw(WC,0,"MODULAR MPU-401 DEVICE DRIVER");
  attrset(COLOR_PAIR(1));
  mvprintw(WC,COLS-strlen(title1),"%s", title1);
  WC++;
  attrset(COLOR_PAIR(1)|A_BOLD);
  mvprintw(WC,0,"MPU-401 INTERFACE");
  attrset(COLOR_PAIR(1));
  mvprintw(WC,COLS-strlen(title2),"%s", title2);
  WC += VSP1;

  print_header(&WC, "General information");
  print_topics(&WC, "Debug level", "Status", "IRQ/IO Port");
  print_header(&WC, "Memory");
  print_topics(&WC, "kmalloc()", "kfree()", "balance");
  print_header(&WC, "Interrupts and ports");
  print_topics(&WC, "Real IRQ", "DRR time out", "DSR time out");
  print_topics(&WC, "Fake IRQ", "DRR max loop", "DSR max loop");
  print_header(&WC, "Blocked reads and writes");
  print_topics(&WC, "Demand voice", "Demand control", "Demand record");
  print_topics(&WC, "Demand on sync", "Demand", "Sync");
  print_topics(&WC, "Play delay", "Play end", NULL);
  print_header(&WC, "Queue statistic");
  print_topics(&WC, "Voice count", "Control count", "Record count");
  print_topics(&WC, "Voice time", "Control time", "Record time");
  print_header(&WC, "Event statistic");
  print_topics(&WC, "Voice send", "SysEx send", "Meta send");
  print_topics(&WC, "Running voice", "NOP events", NULL);
  print_header(&WC, "Metronome");
  print_topics(&WC, "Status", "Microsec/beat", "BPM");
  print_topics(&WC, "Time", "Timebase", "MIDI CLK/metro");
  print_topics(&WC, "Numerator", "Denominator", NULL);
  refresh();
}

void display_info(void)
{
  long membalance = ms.queues.kmallocs-ms.queues.kfrees;
  long WC = 0, i;
  int j;
  char * status = NULL;

  if (!get_driver_info())
    {
      attrset(COLOR_PAIR(4)|A_BOLD);
      mvprintw(LINES-1, 0, "*** ERROR: %s ***%n", strerror(errno), &j);
      for (i = j; i < COLS; i++) mvprintw(LINES-1, i, " ");
      return;
    }
  else
    {
      attrset(COLOR_PAIR(2));
      for (i = 0; i < COLS; i++) mvprintw(LINES-1, i, " ");
    };

  switch (ms.status)
    {
    case MPU_STA_NONE:
      status = "CLOSED";
      break;
    case MPU_STA_OPEN:
      status = "OPEN";
      break;
    case MPU_STA_IS_PLAY:
      status = "PLAY";
      break;
    case MPU_STA_IS_REC:
      status = "RECORD";
      break;
    case MPU_STA_IS_OVERDUB:
      status = "OVERDUB";
      break;
    case MPU_STA_IS_STEPREC:
      status = "STEP REC";
      break;
    default:
      status = "UNKNOWN";
      break;
    };
  
  WC = 0;
  attrset(COLOR_PAIR(2)|A_BOLD);
  mvprintw(WC, 30,"v%hd.%hd.%s",ms.drv_ver.version,ms.drv_ver.subversion,ms.drv_ver.revision);WC++;
  mvprintw(WC, 30,"v%hd.%hd%s",ms.mpu_ver.version,ms.mpu_ver.subversion,ms.mpu_ver.revision);
  WC+=VSP1;
  header_skip(&WC);
  mvprintw(WC, COL2, "%8d", ms.debug);
  mvprintw(WC, COL4, "%8s", status);
  mvprintw(WC, COL6, "%02hd/0x%03x", ms.hardware.irq, ms.hardware.ioport);
  topic_skip(&WC);
  header_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.queues.kmallocs);
  mvprintw(WC, COL4, "%8ld", ms.queues.kfrees);
  mvprintw(WC, COL6, "%8ld", membalance);
  topic_skip(&WC);
  header_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.hardware.real_irq_cnt);
  mvprintw(WC, COL4, "%8ld", ms.hardware.timeout_drr);
  mvprintw(WC, COL6, "%8ld", ms.hardware.timeout_dsr);
  topic_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.hardware.fake_irq_cnt);
  mvprintw(WC, COL4, "%8ld", ms.hardware.loops_drr);
  mvprintw(WC, COL6, "%8ld", ms.hardware.loops_dsr);
  topic_skip(&WC);
  header_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.blocking.bias.voice);
  mvprintw(WC, COL4, "%8ld", ms.blocking.bias.control);
  mvprintw(WC, COL6, "%8ld", ms.blocking.bias.record);
  topic_skip(&WC);
  mvprintw(WC, COL2, "%8s", (ms.blocking.bias.sync) ? "On" : "Off");
  mvprintw(WC, COL4, "%8s", (ms.blocking.demand) ? "Yes" : "No");
  mvprintw(WC, COL6, "%8s", (ms.blocking.sync) ? "Yes" : "No");
  topic_skip(&WC);
  mvprintw(WC, COL2, "%8s", (ms.blocking.play_delay) ? "Yes" : "No");
  mvprintw(WC, COL4, "%8s", (ms.blocking.play_end) ? "Yes" : "No");
  topic_skip(&WC);
  header_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.queues.voice.count);
  mvprintw(WC, COL4, "%8ld", ms.queues.control.count);
  mvprintw(WC, COL6, "%8ld", ms.queues.record.count);
  topic_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.queues.voice_time);
  mvprintw(WC, COL4, "%8ld", ms.queues.control_time);
  mvprintw(WC, COL6, "%8ld", ms.queues.record_time);
  topic_skip(&WC);
  header_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.queues.voice_cnt);
  mvprintw(WC, COL4, "%8ld", ms.queues.sysex_cnt);
  mvprintw(WC, COL6, "%8ld", ms.queues.meta_cnt);
  topic_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.queues.voice_hits);
  mvprintw(WC, COL4, "%8ld", ms.queues.nop_cnt);
  topic_skip(&WC);
  header_skip(&WC);
  mvprintw(WC, COL2, "%8s", (ms.metronome.status) ? "On" : "Off");
  mvprintw(WC, COL4, "%8ld", ms.metronome.mysectempo);
  mvprintw(WC, COL6, "%8ld", ms.metronome.bpm);
  topic_skip(&WC); 
  mvprintw(WC, COL2, "%8ld", ms.metronome.time);
  mvprintw(WC, COL4, "%8ld", ms.metronome.timebase);
  mvprintw(WC, COL6, "%8ld", ms.metronome.midi_metro);
  topic_skip(&WC);
  mvprintw(WC, COL2, "%8ld", ms.metronome.setup.numerator);
  mvprintw(WC, COL4, "%8ld", ms.metronome.setup.denominator);
  move(LINES-1, 0);
  refresh();
}

void setterminal(void)
{
  /* ncurses initialization */
  initscr();
  if (has_colors()) start_color();
  savetty();
  cbreak();
  noecho();
  nonl();
  nodelay(stdscr, TRUE);
  leaveok(stdscr, TRUE);

  standend();
  standout();

  /* setup colors */
  init_pair(1,COLOR_YELLOW,COLOR_BLUE);
  init_pair(2,COLOR_WHITE,COLOR_BLUE);
  init_pair(3,COLOR_GREEN,COLOR_BLUE);
  init_pair(4,COLOR_WHITE,COLOR_RED);
  init_pair(5,COLOR_BLUE,COLOR_WHITE);
}

void follow_info(void)
{
  char alive = 1;

  setterminal();
  build_display();
  while(alive)
    {
      display_info();
      switch(getch())
	{
	case 'q':
	case 'Q':
	    alive = 0;
	    break;
	case '0':
	    set_debug_lvl(0);
	    break;
	case '1':
	    set_debug_lvl(1);
	    break;
	case '2':	
	    set_debug_lvl(2);
	    break;
	case '3':
	    set_debug_lvl(3);
	    break;
	};
      usleep(50000);
    };

  /* exit gracefully */
  refresh();
  endwin();
  resetty();
}

/*** MAIN *********************************************************************/

int main(int argc, char *argv[])
{
  if (parsearg(argc, argv)) return 1;
  if (follow)
    {
      follow_info();
      return 0;
    };

  if (!get_driver_info())
    {
      fprintf(stderr, "MPU-401 Device driver not found!\n");
      return 1;
    };

  printf("MPU-401 device driver v%hd.%hd.%s\n",
	 ms.drv_ver.version, ms.drv_ver.subversion, ms.drv_ver.revision);
  printf("MPU-401 version %hd.%hd%s detected at IRQ %hd I/O port 0x%03x\n",
	 ms.mpu_ver.version, ms.mpu_ver.subversion, ms.mpu_ver.revision,
	 ms.hardware.irq, ms.hardware.ioport);

  if ((setdebug != -1) || (getdebug))
    {
      mpudev = open("/dev/mpu401stat", O_RDWR);
      if (mpudev != -1)
	{
	  if (setdebug != -1)
	    {
	      ioctl(mpudev, MPUIOC_SET_DEBUG_LVL, &setdebug);
	      printf("Driver debug level set to %d = ", setdebug);
	      switch (setdebug)
	      {
	      case MPU_DBG_NONE:
		printf("None\n");
		break;
	      case MPU_DBG_SPARSE:
		printf("Sparse\n");
		break;
	      case MPU_DBG_VERBOSE:
		printf("Verbose\n");
		break;
	      case MPU_DBG_ALL:
		printf("All\n");
		break;
	      };
	    };
	  if (getdebug)
	    {
	      ioctl(mpudev, MPUIOC_GET_DEBUG_LVL, &getdebug);
	      printf("Current driver debug level is %d = ", getdebug);
	      switch (getdebug)
	      {
	      case MPU_DBG_NONE:
		printf("None\n");
		break;
	      case MPU_DBG_SPARSE:
		printf("Sparse\n");
		break;
	      case MPU_DBG_VERBOSE:
		printf("Verbose\n");
		break;
	      case MPU_DBG_ALL:
		printf("All\n");
		break;
	      };
	    };
	}
      else
	{
	  fprintf(stderr, "Unable to get driver information.\n");
	  perror("mpuinfo: ");
	  return 1;
	};
      close(mpudev);
    };
  return 0;
}

/*** END OF FILE **************************************************************/
