/*
 * ibmccae.c: Linux driver for the IBM Credit Card Adapter for Ethernet
 *
 * written January 1994 by Keith Moore <moore@cs.utk.edu>
 * with valuable help from Ken Key <key@cs.utk.edu>
 * heavily based on the Linux wd.c and ne.c drivers by Donald Becker.
 *
 * Copyright (c) 1994 by Keith Moore.  This software may be used and
 * distributed according to the terms of the GNU Public License.
 *
 * THERE IS NO WARRANTY ON THIS CODE, NOT EVEN FOR MERCHANTABILITY
 * NOR FITNESS FOR ANY PARTICULAR PURPOSE.  IF YOU INSTALL THIS
 * CODE IN A SYSTEM, YOU ASSUME ALL RESPONSIBILITY FOR ANYTHING
 * THAT HAPPENS AS A RESULT.
 */

static char *version =
    "ibmccae.c: 940128 Keith Moore <moore@cs.utk.edu>\n";

/*
 * Sources:
 *
 * [1] _Local Area Network Credit Card Adapters Technical Reference_,
 *     IBM Corp., SC30-3585-00, part # 33G9243.
 * [2] "pre-alpha" PCMCIA support code for Linux by Barry Jaspan.
 * [3] Intel 82536SL PC Card Interface Controller Data Sheet, Intel
 *     Order Number 290423-002
 * [4] National Semiconductor DP83902A ST-NIC (tm) Serial Network
 *     Interface Controller for Twisted Pair data sheet.
 */

#ifdef TEST
#include <stdio.h>
#include <sys/types.h>
#include <asm/io.h>
#include "82365.h"
#else
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <asm/system.h>
#include <asm/io.h>
#include <memory.h>

#include "dev.h"
#include "8390.h"
#include "82365.h"
#endif


/*****************************************************************************
 *                 pcmcia controller chip (PCIC) support                     *
 *               (eventually, move this to a separate file)                  *
 *****************************************************************************/

/*
 * Each PCIC chip (82365SL or clone) can handle two card slots, and there
 * can be up to four PCICs in a system.  (On some machines, not all of the
 * address lines are decoded, so a card may appear to be in more than one 
 * slot.)
 */
#define MAXSLOT 8

/*
 * To access a register on the PCIC for a particular slot, you
 * first write the correct OFFSET value for that slot in the
 * INDEX register for the PCIC controller.  You then read or write
 * the value from or to the DATA register for that controller.
 *
 * The first pair of chips shares I/O addresss for DATA and INDEX,
 * as does the second pair.   (To the programmer, it looks like each 
 * pair is a single chip.)  The i/o port addresses are hard-wired 
 * into the PCIC; so the following addresses should be valid for
 * any machine that uses this chip.
 */

#define PCIC_INDEX_0	0x3E0	/* index reg, chips 0 and 1 */
#define PCIC_DATA_0	0x3E1	/* data register, chips 0 and 1 */
#define PCIC_INDEX_1	0x3E2	/* index reg, chips 1 and 2 */
#define PCIC_DATA_1	0x3E3	/* data register, chips 1 and 2 */

/*
 * Given a slot number, calculate the INDEX and DATA registers
 * to talk to that slot.  OFFSET is added to the register number
 * to address the registers for a particular slot.
 */
#define INDEX(slot) ((slot) < 4 ? PCIC_INDEX_0 : PCIC_INDEX_1)
#define DATA(slot) ((slot) < 4 ? PCIC_DATA_0 : PCIC_DATA_1)
#define OFFSET(slot) ((slot) % 4 * 0x40)

/*
 * There are 5 sets (windows) of memory mapping registers on the PCIC chip
 * for each slot, numbered 0..4.
 *
 * They start at 10/50 hex within the chip's register space (not system
 * I/O space), and are eight addresses apart.  These are actually pairs of
 * 8-bit-wide registers (low byte first, then high byte) since the
 * address fields are actually 12 bits long.  The upper bits are used
 * for other things like 8/16-bit select and wait states.
 *
 * Memory mapping registers include start/stop addresses to define the
 * region to be mapped (in terms of system memory addresses), and
 * an offset register to allow for translation from system space
 * to card space.  The lower 12 bits aren't included in these, so memory is
 * mapped in 4K chunks.
 */
#define MEM_START_ADDR(window) (((window) * 0x08) + 0x10)
#define MEM_STOP_ADDR(window) (((window) * 0x08) + 0x12)
#define MEM_OFFSET(window) (((window) * 0x08) + 0x14)
/*
 * this bit gets set in the address window enable register (PCIC_ADDRWINE)
 * to enable a particular address window.
 */
#define MEM_ENABLE_BIT(window) ((1) << (window))

/*
 * There are two i/o port addressing windows.  I/O ports cannot be
 * relocated within system i/o space (unless the card doesn't decode
 * all of the address bits); unlike card memory, there is no address
 * translation offset.
 */
#define IO_START_ADDR(window) ((window) ? PCIC_IO1_STL : PCIC_IO0_STL)
#define IO_STOP_ADDR(window) ((window) ? PCIC_IO1_SPL : PCIC_IO0_SPL)
#define IO_ENABLE_BIT(window) ((window) ? PCIC_IO1_EN : PCIC_IO0_EN)
#define IO_CS16_BIT(window) ((window) ? PCIC_IO1_CS16 : PCIC_IO0_CS16)

/*
 * read a byte from a pcic register for a particular slot
 */
static inline unsigned char
pcic_getb (int slot, int reg)
{
    outb_p (OFFSET (slot) + reg, INDEX (slot));
    return inb_p (DATA (slot));
}

/*
 * write a byte to a pcic register for a particular slot
 */
static inline void
pcic_putb (int slot, int reg, unsigned char val)
{
    outb_p (OFFSET (slot) + reg, INDEX (slot));
    outb_p (val, DATA (slot));
}

/*
 * read a word from a pcic register for a particular slot
 */
static inline unsigned short
pcic_getw (int slot, int reg)
{
    return pcic_getb (slot, reg) | (pcic_getb (slot, reg+1) << 8);
}

/*
 * write a word to a pcic register at a particular slot
 */
static inline void
pcic_putw (int slot, int reg, unsigned short val)
{
    pcic_putb (slot, reg, val & 0xff);
    pcic_putb (slot, reg + 1, (val >> 8) & 0xff);
}

static void
pcic_print_regs (int slot)
{
    int i, j;

    for (i = 0; i < 0x40; i += 16) {
	for (j = 0; j < 16; ++j)
	    printk ("%02x ", pcic_getb (slot, i + j));
	printk ("\n");
    }
}

/*
 * map a portion of the card's memory space into system memory
 * space.
 *
 * slot = # of the slot the card is plugged into
 * window = which pcic memory map registers to use (0..4)
 * sys_addr = base system memory address where we want it.  must
 *	      be on an appropriate boundary (lower 12 bits are zero).
 * card_addr = the base address of the card's memory to correspond
 *             to sys_addr
 * length = length of the segment to map (may be rounded up as necessary)
 * type = which card memory space to map (attribute or shared)
 * width = 1 for byte-wide mapping; 2 for word (16-bit) mapping.
 */

enum memtype { COMMON, ATTRIBUTE };

static void
pcic_map_memory (int slot, int window, unsigned long sys_addr,
		 unsigned long card_addr, unsigned long length,
		 enum memtype type, int width)
{
    unsigned short offset;
    unsigned short mem_start_addr;
    unsigned short mem_stop_addr;

    sys_addr >>= 12;
    card_addr >>= 12;
    length >>= 12;
    /*
     * compute an offset for the chip such that
     * (sys_addr + offset) = card_addr
     * but the arithmetic is done modulo 2^14
     */
    offset = (card_addr - sys_addr) & 0x3FFF;
    /*
     * now OR in the bit for "attribute memory" if necessary
     */
    if (type == ATTRIBUTE) {
	offset |= (PCIC_REG << 8);
	/* REG == "region active" pin on card */
    }
    /*
     * okay, set up the chip memory mapping registers, and turn
     * on the enable bit for this window.
     * if we are doing 16-bit wide accesses (width == 2),
     * turn on the appropriate bit.
     *
     * XXX for now, we set all of the wait state bits to zero.
     * Not really sure how they should be set.
     */
    mem_start_addr = sys_addr & 0xFFF;
    if (width == 2)
	mem_start_addr |= (PCIC_DATA16 << 8);
    mem_stop_addr = (sys_addr + length) & 0xFFF;

    pcic_putw (slot, MEM_START_ADDR(window), mem_start_addr);
    pcic_putw (slot, MEM_STOP_ADDR(window), mem_stop_addr);
    pcic_putw (slot, MEM_OFFSET(window), offset);
    /*
     * Assert the bit (PCIC_MEMCS16) that says to decode all of
     * the address lines.
     */
    pcic_putb (slot, PCIC_ADDRWINE,
	       pcic_getb (slot, PCIC_ADDRWINE) |
	       MEM_ENABLE_BIT(window) | PCIC_MEMCS16);
}

static void
pcic_unmap_memory (int slot, int window)
{
    /*
     * seems like we need to turn off the enable bit first, after which
     * we can clear the registers out just to be sure.
     */
    pcic_putb (slot, PCIC_ADDRWINE,
	       pcic_getb (slot, PCIC_ADDRWINE) & ~MEM_ENABLE_BIT(window));
    pcic_putw (slot, MEM_START_ADDR(window), 0);
    pcic_putw (slot, MEM_STOP_ADDR(window), 0);
    pcic_putw (slot, MEM_OFFSET(window), 0);
}

/*
 * map a range of addresses into system i/o space
 * (no translation of i/o addresses is possible)
 *
 * 'width' is:
 * + 0 to tell the PCIC to generate the ISA IOCS16* signal from
 *   the PCMCIA IOIS16* signal.
 * + 1 to select 8-bit width
 * + 2 to select 16-bit width
 */

static void
pcic_map_io (int slot, int window, unsigned short base, unsigned short length,
	     unsigned short width)
{
    unsigned char x;

    pcic_putw (slot, IO_START_ADDR(window), base);
    pcic_putw (slot, IO_STOP_ADDR(window), base+length-1);
    /*
     * select the bits that determine whether
     * an i/o operation is 8 or 16 bits wide
     */
    x = pcic_getb (slot, PCIC_IOCTL);
    switch (width) {
    case 0:			/* PCMCIA card decides */
	if (window)
	    x = (x & 0xf0) | PCIC_IO1_CS16;
	else
	    x = (x & 0x0f) | PCIC_IO0_CS16;
	break;
    case 1:			/* 8 bits wide */
	break;
    case 2:			/* 16 bits wide */
	if (window)
	    x = (x & 0xf0) | PCIC_IO1_16BIT;
	else
	    x = (x & 0x0f) | PCIC_IO0_16BIT;
	break;
    }
    pcic_putb (slot, PCIC_IOCTL, x);
    pcic_putb (slot, PCIC_ADDRWINE,
	       pcic_getb (slot, PCIC_ADDRWINE) | IO_ENABLE_BIT(window));
}

#ifdef TEST
static void
pcic_unmap_io (int slot, int window)
{
    pcic_putb (slot, PCIC_ADDRWINE,
	       pcic_getb (slot, PCIC_ADDRWINE) & ~IO_ENABLE_BIT(window));
    pcic_putw (slot, IO_START_ADDR(window), 0);
    pcic_putw (slot, IO_STOP_ADDR(window), 0);
}
#endif

/*
 * tell the PCIC which irq we want to use.  only the following are legal:
 * 3, 4, 5, 7, 9, 10, 11, 12, 14, 15
 * ...but this routine doesn't check.
 */

static void
pcic_map_irq (int slot, int irq)
{
    pcic_putb (slot, PCIC_INT_GEN,
	       pcic_getb (slot, PCIC_INT_GEN) | (irq & 0x0F));
}

static void
delay(int len)
{
    sti();
    {
        int start = jiffies;
	int boguscount = 150000;
	while(jiffies - start < len)
	    if (boguscount-- < 0) {
		printk("jiffy failure (t=%d)...", jiffies);
		break;
	    }	
    }	
}

static void
pcic_power_on (int slot)
{
    pcic_putb (slot, PCIC_POWER,
	       pcic_getb (slot, PCIC_POWER) | PCIC_DISRST | PCIC_PCPWRE);
    delay (2);
    pcic_putb (slot, PCIC_POWER,
	       pcic_getb (slot, PCIC_POWER) | PCIC_OUTENA);
}

static void
pcic_reset (int slot)
{
    /* assert RESET (by clearing a bit!), wait a bit, and de-assert it */
    pcic_putb (slot, PCIC_INT_GEN,
	       pcic_getb (slot, PCIC_INT_GEN) & ~PCIC_CARDRESET);
    delay (2);
    pcic_putb (slot, PCIC_INT_GEN,
	       pcic_getb (slot, PCIC_INT_GEN) | PCIC_CARDRESET);
}


/*****************************************************************************
 *                       Driver for Ethernet Adapter                         *
 *****************************************************************************/
#define IBMCCAE_START_PAGE	0x40	/* first page of shared buffer */
#define IBMCCAE_STOP_PAGE	0x80	/* last page+1 of shared buffer */

#define IBMCCAE_RESET		0x1F	/* offset of reset reg in i/o space  */
#define IBMCCAE_MISC		0x18	/* offset of misc reg in i/o space */

static unsigned char card_info[255];
static unsigned char enet_addr[6];

#define CARD_INFO  "IBM Corp./Ethernet/0933495"

static void ibmccae_reset_8390 (struct device *dev);
static int ibmccae_block_input(struct device *dev, int count, char *buf,
			       int ring_offset);
static void ibmccae_block_output(struct device *dev, int count,
				 const unsigned char *buf,
				 int start_page);
static int ibmccae_close(struct device *dev);


/*
 * scan the card information structure looking for the version/product info
 * tuple.  when we find it, compare it to the string we are looking for.
 * return 1 if we find it, 0 otherwise.
 */

static int
ibmccae_check_cis (unsigned char *scratch)
{
    int i,j,k;


    card_info[0] = '\0';
    i = 0;
    while (scratch[i] != 0xff && i < 1024) {
	unsigned char link = scratch[i+2];

#if 0
	printk ("[%02x] %02x ", i, link);
	for (j = 4; j < 2 * link + 4 && j < 32; j += 2)
	    printk ("%02x ", scratch[j + i]);
	printk ("\n");
#endif

	if (scratch[i] == 0x15) {
	    /*
	     * level 1 version/product info
	     * copy to card_info, translating '\0' to '/'
	     */
	    k = 0;
	    for (j = i+8; scratch[j] != 0xff; j += 2)
		card_info[k++] = scratch[j] == '\0' ? '/' : scratch[j];
	    card_info[k++] = '\0';
	    return (memcmp (card_info, CARD_INFO, sizeof(CARD_INFO)-1) == 0);
	}
	i += 4 + 2 * link;
    }
    return 0;
}

/*
 * Probe each slot looking for an IBM Credit Card Adapter for Ethernet
 * For each card that we find, map its card information structure
 * into system memory at 'scratch' and see whether it's one of ours.
 * Return the slot number if we find a card, or -1 otherwise. 
 *
 * Side effects:
 * + On success, leaves CIS mapped into memory at 'scratch';
 *   caller must free it.
 * + On success, leaves ethernet address in enet_addr.
 * + Leaves product/vendor id of last card probed in 'card_info'
 */

static int
ibmccae_find_adapter (unsigned char *scratch)
{
    int slot;

    for (slot = 0; slot < MAXSLOT; ++slot) {
	/*
	 * see if there's a PCMCIA controller here
	 * Intel PCMCIA controllers use 0x82 and 0x83
	 * IBM clone chips use 0x88 and 0x89, apparently
	 */
	unsigned char idbyte = pcic_getb (slot, PCIC_ID_REV);

	if (idbyte != 0x82 && idbyte != 0x83 &&
	    idbyte != 0x88 && idbyte != 0x89) {
#if 0
	    printk ("ibmccae: pcic slot %d: wierd id/rev code 0x%02x\n",
		    slot, idbyte);
#endif
	    continue;
	}
	if ((pcic_getb (slot, PCIC_STATUS) & PCIC_CD) != PCIC_CD) {
	    printk ("ibmccae: slot %d: no card in slot\n", slot);
	    /* no card in slot */
	    continue;
	}
	pcic_power_on (slot);
	pcic_reset (slot);
	/*
	 * map the card's attribute memory and examine its 
	 * card information structure tuples for something
	 * we recognize.
	 */
	pcic_map_memory (slot, 0, (unsigned long) scratch, 0L,
			 0xFFFL, ATTRIBUTE, 1);
	
	if ((ibmccae_check_cis (scratch)) > 0) {
	    /* found it */
	    printk ("ibmccae: found card in slot %d\n", slot);
	    return slot;
	}
	else
	    printk ("ibmccae: slot %d: %s\n", slot, card_info);
	pcic_unmap_memory (slot, 0);
    }
    return -1;
}

/*
 * Probe for the Credit Card Adapter. 
 * The best way to do this is to examine each socket and look for the
 * card identification string, and see if it matches the adapter
 * we are looking for.  If we find one, assign it some i/o and memory
 * space, and set up the interrupt handler.
 */

/*
 * macros to handle casting unsigned long to (char *) so we can
 * read/write into physical memory space.
 */

#define PEEK(addr) (*((unsigned char *)(addr)))
#define POKE(addr,val) do { PEEK(addr) = (val); } while (0)

int
ibmccae_probe (struct device *dev)
{
    int slot;

    /*
     * first find a place to map the card into memory.  we need this
     * to probe for the enet card.
     */
    if (dev->mem_start == 0) {
#if defined(IBMCCAE_SHMEM)
	if ((IBMCCAE_SHMEM & 0x3fff) == 0)
	    dev->mem_start = IBMCCAE_SHMEM;
	else {
	    dev->mem_start = 0xd4000;
	    printk ("ibmccae: Warning: compiled-in IBMCCAE_SHMEM (0x%x) not on 16K boundary\n",
		    IBMCCAE_SHMEM);
	    printk ("         using 0x%x instead.\n", dev->mem_start);
	}
#else	
	dev->mem_start = 0xd4000;
#endif
    }
    /*
     * now map each card's information structure (CIS) into system
     * memory space, and see if we find the ethernet card.
     */
    if ((slot = ibmccae_find_adapter ((unsigned char *) dev->mem_start)) < 0)
	return ENODEV;
    /*
     * okay, we found a card, so set it up
     */
    /*
     * Inhibit 16 bit memory delay.
     * POINTETH.SYS apparently does this, for what reason I don't know.
     */
    pcic_putb (slot, PCIC_CDGC,
	       pcic_getb (slot, PCIC_CDGC) | PCIC_16_DL_INH);
    /*
     * things to map (and maybe allocate from Linux)
     * (1) card's EEPROM is already mapped by the find_adapter routine
     *     but we still need to get the card's ethernet address.
     *     after that we unmap that part of attribute memory.
     * (2) card configuration registers need to be mapped in so we
     *     can set the configuration and socket # registers.
     * (3) shared memory packet buffer
     * (4) i/o ports
     * (5) IRQ
     */
    /*
     * Sigh.  Location of the ethernet address isn't documented in [1].
     * It was derived by doing a hex dump of all of attribute memory
     * and looking for the IBM vendor prefix.
     */
    enet_addr[0] = PEEK(dev->mem_start+0xff0);
    enet_addr[1] = PEEK(dev->mem_start+0xff2);
    enet_addr[2] = PEEK(dev->mem_start+0xff4);
    enet_addr[3] = PEEK(dev->mem_start+0xff6);
    enet_addr[4] = PEEK(dev->mem_start+0xff8);
    enet_addr[5] = PEEK(dev->mem_start+0xffa);
    pcic_unmap_memory (slot, 0);

    /*
     * (2) map card configuration registers.  these are offset
     * in card memory space by 0x20000.  normally we could get
     * this offset from the card information structure, but I'm
     * too lazy and am not quite sure if I understand the CIS anyway.
     */
    pcic_map_memory (slot, 0, dev->mem_start, 0x20000, 8L, ATTRIBUTE, 1);
    POKE(dev->mem_start, 0x80);	/* reset the card (how long?) */
    delay (100);
    /*
     * Set the configuration index.  According to [1], the adapter won't
     * respond to any i/o signals until we do this; it uses the
     * Memory Only interface (whatever that is; it's not documented).
     * Also turn on "level" (not pulse) interrupts.
     *
     * XXX probably should init the socket and copy register also,
     * so that we can deal with multiple instances of the same card.
     */
    POKE(dev->mem_start, 0x41);
    pcic_unmap_memory (slot, 0);

    /*
     * (3) now map in the shared memory buffer.  This has to be mapped
     * as words, not bytes, and on a 16k boundary.  The offset value
     * was derived by installing IBM's POINTETH.SYS under DOS and
     * looking at the PCIC registers; it's not documented in IBM's
     * tech ref manual ([1]).
     */
    pcic_map_memory (slot, 0, dev->mem_start, 0x4000L, 0x4000L, COMMON, 2);

    /*
     * (4) map i/o ports.  If we are called are called with a port
     * already in mind, use that one;  else, use the first unreserved
     * range of ports in our list.
     */
    if (dev->base_addr == 0) {
	/*
	 * the i/o ports on the pcmcia card can apparently go anywhere
	 * in i/o space, but the port numbers in the list below are
	 * commonly used by Ethernet cards.  So we look for one of these
	 * that's not already used, and locate the card there.
	 */
	unsigned short ports[] = { 0x300, 0x280, 0x380, 0x240, 0 };
	int i;

	for (i = 0; ports[i] != 0; ++i) {
	    if (check_region (ports[i], 32) == 0) {
		dev->base_addr = ports[i];
		break;
	    }
	}
	if (dev->base_addr == 0) {
	    printk ("ibmccae: (slot %d): device found, can't alloc i/o ports!\n",
		    slot);
	    return ENODEV;
	}
    }
    else if (check_region (dev->base_addr, 32) != 0) {
	printk ("ibmccae: i/o address 0x%x is already in use\n",
		dev->base_addr);
	return ENODEV;
    }
    snarf_region (dev->base_addr, 32);
    /*
     * At least one PCMCIA device driver I'v seen maps a block
     * of 32 consecutive i/o ports as two windows of 16 ports each.
     * Maybe some other pcic chips are restricted to 16-port windows;
     * the 82365SL doesn't seem to have that problem.  But since
     * we have an extra window anyway...
     */
#ifdef SHARED_MEMORY
    pcic_map_io (slot, 0, dev->base_addr, 32, 1);
#else
    pcic_map_io (slot, 0, dev->base_addr, 16, 1);
    pcic_map_io (slot, 1, dev->base_addr+16, 16, 2);
#endif
    /*
     * (5) allocate an interrupt and assign it to the device
     */
    if (dev->irq == 0) {
	/* pick the first available interrupt from the list below */
        static int irqs[] = { 5, 7, 9, 10, 11, 12, 14, 15, 0 };
	int i;

	for (i = 0; irqs[i]; ++i) {
	    if (irqaction (irqs[i], &ei_sigaction) == 0) {
		/* got it */
		dev->irq = irqs[i];
		break;
	    }
	}
	if (irqs[i] == 0) {
	    printk ("ibmccae: couldn't allocate an interrupt!\n");
	    return 0;
	}
    }
    else {
	printk ("using pre-assigned irq %d\n", dev->irq);
	if (irqaction (dev->irq, &ei_sigaction) != 0) {
	    printk ("ibmccae: unable to use interrupt %d\n", dev->irq);
	    return 0;
	}
    }
    pcic_map_irq (slot, dev->irq);
    /* tell the PCIC that this is an I/O card (not memory) */
    pcic_putb (slot, PCIC_INT_GEN,
	       pcic_getb (slot, PCIC_INT_GEN) | PCIC_CARDTYPE);

#if 0
    pcic_print_regs (slot);
#endif

    /*
     * okay, setup the device
     */
    if (ei_debug > 0)
	printk ("%s", version);

    ethdev_init (dev);
    ei_status.name = "ibmccae";
    ei_status.word16 = 1;
    ei_status.tx_start_page = IBMCCAE_START_PAGE;
    ei_status.rx_start_page = IBMCCAE_START_PAGE + TX_PAGES;
    ei_status.stop_page = IBMCCAE_STOP_PAGE;
    ei_status.reset_8390 = &ibmccae_reset_8390;
    ei_status.block_input = &ibmccae_block_input;
    ei_status.block_output = &ibmccae_block_output;

    dev->rmem_start = dev->mem_start + TX_PAGES * 256;
    dev->mem_end = dev->rmem_end =
	dev->mem_start + (ei_status.stop_page - IBMCCAE_START_PAGE)*256;
    dev->stop = &ibmccae_close;
    dev->dev_addr[0] = enet_addr[0];
    dev->dev_addr[1] = enet_addr[1];
    dev->dev_addr[2] = enet_addr[2];
    dev->dev_addr[3] = enet_addr[3];
    dev->dev_addr[4] = enet_addr[4];
    dev->dev_addr[5] = enet_addr[5];

    /*
     *	finally, announce that we found the device
     */
    printk ("ibmccae: slot %d io 0x%x mem 0x%x-0x%x irq %d mac %x:%x:%x:%x:%x:%x\n",
	    slot, dev->base_addr, dev->mem_start, dev->mem_end, dev->irq,
	    enet_addr[0], enet_addr[1], enet_addr[2],
	    enet_addr[3], enet_addr[4], enet_addr[5]);
    return 0;
}

static void
ibmccae_reset_8390 (struct device *dev)
{
    int tmp = inb_p (dev->base_addr + IBMCCAE_RESET);
    int reset_start_time = jiffies;

    ei_status.txing = 0;
    outb_p (tmp, dev->base_addr + IBMCCAE_RESET);
    /* This check _should_not_ be necessary, omit eventually. */
    while ((inb_p (dev->base_addr + EN0_ISR) & ENISR_RESET) == 0) {
	if (jiffies - reset_start_time > 2) {
	    printk ("%s: ibmccae_reset_8390() did not complete.\n",
		    dev->name);
	    break;
	}
    }
    /*
     * set MAM bit in misc register for 10base2
     * XXX is this necessary?
     */
    tmp = inb_p (dev->base_addr + IBMCCAE_MISC);
    outb_p (tmp | 0x9, dev->base_addr + IBMCCAE_MISC);
}

/*
 * [1] says we need to read from the shared memory buffer with only
 * aligned move word instructions.  So we don't trust memcpy.
 * src should always be an even address.
 *
 * XXX should do this in assembler.
 */

static void
ibmccae_copyin (unsigned char *dest, unsigned char *src, int count)
{
    /*
     * make copies of the pointers in registers.
     * it's okay for count to be 16 bits since ether packets
     * are limited to 1500 bytes in size...except in Utah.
     */
    register unsigned short *d = (unsigned short *) dest;
    register unsigned short *s = (unsigned short *) src;
    register unsigned short c = count;
    register int odd;
    
    if (c <= 0)
	return;
    odd = (c & 01);
    c >>= 1;

    /* 
     * do copy as fast as possible by putting the test at the end.
     * compiler may optimize this into rep movsw...I guess that's okay.
     * if not, guess we could unroll the loop once or twice.
     */
    if (c) {
	do {	
	    *d++ = *s++;	     
	} while (--c);
    }
    /* get last byte by fetching a word and masking */
    if (odd)
	*((unsigned char *) d) = *s & 0xff;
}

#ifdef SHARED_MEMORY
/* Block input and output are easy on shared memory ethercards, and trivial
   on this particular card where there is no choice of how to do it.
   The only complication is if the ring buffer wraps. */

static int
ibmccae_block_input(struct device *dev, int count, char *buf, int ring_offset)
{
    void *xfer_start = (void *)(dev->mem_start + ring_offset
				- (IBMCCAE_START_PAGE<<8));
    int i, j;
    
#if 0
    printk ("block_input (count = %d, ring_offset = %x)\n",
	    count, ring_offset);
    for (i = 0; i < 0x40; ++i) {
	printk ("%02x:", i);
	for (j = 0; j < 16; ++j)
	    printk ("%02x ", PEEK(dev->mem_start + (i << 8) + j));
	printk ("\n");
    }
#endif

    if (xfer_start + count > (void*) dev->rmem_end) {
	/* We must wrap the input move. */
	int semi_count = (void*)dev->rmem_end - xfer_start;
#if 0
	ibmccae_copyin (buf, xfer_start, semi_count);
#else
	memcpy (buf, xfer_start, semi_count);
#endif
	count -= semi_count;
#if 0
	ibmccae_copyin (buf + semi_count, (char *)dev->rmem_start, count);
#else
	memcpy (buf + semi_count, (char *) dev->rmem_start, count);
#endif
	return dev->rmem_start + count;
    }
#if 0
    ibmccae_copyin (buf, xfer_start, count);
#else
    memcpy (buf, xfer_start, count);
#endif
    return ring_offset + count;
}
#else
#ifndef port_read
#include "iow.h"
#endif

static int
ibmccae_block_input(struct device *dev, int count, char *buf, int ring_offset)
{
    int xfer_count = count;
    int e8390_base = dev->base_addr;

    if (ei_status.dmaing) {
	if (ei_debug > 0)
	    printk("%s: DMAing conflict in ibmccae_block_input."
		   "[DMAstat:%1x][irqlock:%1x]\n",
		   dev->name, ei_status.dmaing, ei_status.irqlock);
	return 0;
    }
    ei_status.dmaing |= 0x01;
    outb_p(E8390_NODMA+E8390_PAGE0+E8390_START, e8390_base+ E8390_CMD);
    outb_p(count & 0xff, e8390_base + EN0_RCNTLO);
    outb_p(count >> 8, e8390_base + EN0_RCNTHI);
    outb_p(ring_offset & 0xff, e8390_base + EN0_RSARLO);
    outb_p(ring_offset >> 8, e8390_base + EN0_RSARHI);
    outb_p(E8390_RREAD+E8390_START, e8390_base + E8390_CMD);
    if (ei_status.word16) {
	port_read(e8390_base + 0x10 ,buf,count>>1);
	if (count & 0x01)
	    buf[count-1] = inb(e8390_base + 0x10), xfer_count++;
    } else {
	port_read_b(e8390_base + 0x10, buf, count);
    }

    /* This was for the ALPHA version only, but enough people have
       encountering problems that it is still here.  If you see
       this message you either 1) have an slightly imcompatible clone
       or 2) have noise/speed problems with your bus. */
    if (ei_debug > 1) {		/* DMA termination address check... */
	int addr, tries = 20;
	do {
	    /* DON'T check for 'inb_p(EN0_ISR) & ENISR_RDC' here
	       -- it's broken! Check the "DMA" address instead. */
	    int high = inb_p(e8390_base + EN0_RSARHI);
	    int low = inb_p(e8390_base + EN0_RSARLO);
	    addr = (high << 8) + low;
	    if (((ring_offset + xfer_count) & 0xff) == low)
		break;
	} while (--tries > 0);
	if (tries <= 0)
	    printk("%s: RX transfer address mismatch,"
		   "%#4.4x (expected) vs. %#4.4x (actual).\n",
		   dev->name, ring_offset + xfer_count, addr);
    }
    ei_status.dmaing &= ~0x01;
    return ring_offset + count;
}
#endif

/*
 * [1] says we need to write to the shared memory buffer with only
 * aligned move word instructions.  So we don't trust memcpy.
 * dest should always be an even address.
 *
 * XXX should do this in assembler.
 */

static void
ibmccae_copyout (unsigned char *dest, unsigned char *src, int count)
{
    /*
     * make copies of the pointers in registers.
     * it's okay for count to be 16 bits since ether packets
     * are limited to 1500 bytes in size...except in Utah.
     */
    register unsigned short *d = (unsigned short *) dest;
    register unsigned short *s = (unsigned short *) src;
    register unsigned short c = count;
    register int odd;
    
    if (c <= 0)
	return;
    odd = (c & 01);
    c >>= 1;

    /* 
     * do copy as fast as possible by putting the test at the end.
     * compiler may optimize this into rep movsw...I guess that's okay.
     * if not, guess we could unroll the loop once or twice.
     */
    if (c) {
	do {	
	    *d++ = *s++;	     
	} while (--c);
    }
    /* copy last byte doing a read-modify-write */
    if (odd)
	*d = (*d & 0xff00) | *((unsigned char *) s);
}

#ifdef SHARED_MEMORY
static void
ibmccae_block_output(struct device *dev, int count, const unsigned char *buf,
		     int start_page)
{
    unsigned char *shmem
	= (unsigned char *)dev->mem_start +
	    ((start_page - IBMCCAE_START_PAGE)<<8);
#if 0
    ibmccae_copyout (shmem, buf, count);
#else
    memcpy (shmem, buf, count);
#endif
}
#else
static void
ibmccae_block_output(struct device *dev, int count,
		const unsigned char *buf, const int start_page)
{
    int retries = 0;
    int e8390_base = dev->base_addr;

    /* Round the count up for word writes.  Do we need to do this?
       What effect will an odd byte count have on the 8390?
       I should check someday. */
    if (ei_status.word16 && (count & 0x01))
      count++;
    if (ei_status.dmaing) {
	if (ei_debug > 0)
	    printk("%s: DMAing conflict in ne_block_output."
		   "[DMAstat:%1x][irqlock:%1x]\n",
		   dev->name, ei_status.dmaing, ei_status.irqlock);
	return;
    }
    ei_status.dmaing |= 0x02;
    /* We should already be in page 0, but to be safe... */
    outb_p(E8390_PAGE0+E8390_START+E8390_NODMA, e8390_base + E8390_CMD);

 retry:
#if defined(rw_bugfix)
    /* Handle the read-before-write bug the same way as the
       Crynwr packet driver -- the NatSemi method doesn't work.
       Actually this doesn't aways work either, but if you have
       problems with your NEx000 this is better than nothing! */
    outb_p(0x42, e8390_base + EN0_RCNTLO);
    outb_p(0x00, e8390_base + EN0_RCNTHI);
    outb_p(0x42, e8390_base + EN0_RSARLO);
    outb_p(0x00, e8390_base + EN0_RSARHI);
    outb_p(E8390_RREAD+E8390_START, e8390_base + NE_CMD);
    /* Make certain that the dummy read has occured. */
    SLOW_DOWN_IO;
    SLOW_DOWN_IO;
    SLOW_DOWN_IO;
#endif  /* rw_bugfix */

    /* Now the normal output. */
    outb_p(count & 0xff, e8390_base + EN0_RCNTLO);
    outb_p(count >> 8,   e8390_base + EN0_RCNTHI);
    outb_p(0x00, e8390_base + EN0_RSARLO);
    outb_p(start_page, e8390_base + EN0_RSARHI);

    outb_p(E8390_RWRITE+E8390_START, e8390_base + E8390_CMD);
    if (ei_status.word16) {
	port_write(e8390_base + 0x10, buf, count>>1);
    } else {
	port_write_b(e8390_base + 0x10, buf, count);
    }

    /* This was for the ALPHA version only, but enough people have
       encountering problems that it is still here. */
    if (ei_debug > 1) {		/* DMA termination address check... */
	int addr, tries = 20;
	do {
	    /* DON'T check for 'inb_p(EN0_ISR) & ENISR_RDC' here
	       -- it's broken! Check the "DMA" address instead. */
	    int high = inb_p(e8390_base + EN0_RSARHI);
	    int low = inb_p(e8390_base + EN0_RSARLO);
	    addr = (high << 8) + low;
	    if ((start_page << 8) + count == addr)
		break;
	} while (--tries > 0);
	if (tries <= 0) {
	    printk("%s: Tx packet transfer address mismatch,"
		   "%#4.4x (expected) vs. %#4.4x (actual).\n",
		   dev->name, (start_page << 8) + count, addr);
	    if (retries++ == 0)
		goto retry;
	}
    }
    ei_status.dmaing &= ~0x02;
    return;
}
#endif

/* This function resets the ethercard if something screws up. */
static int
ibmccae_close (struct device *dev)
{
    if (ei_debug > 1)
	printk("%s: Shutting down ethercard.\n", dev->name);
    NS8390_init(dev, 0);
    return 0;
}

#if 0
main (argc, argv)
int argc;
char **argv;
{
    int page;
    int mem_fd;
    unsigned char buf[0x4000];
    unsigned short wbuf[8192];
    int i, j;

    if (ioperm (0x3e0, 4, 1) || ioperm (0x80, 1, 1)) {
	perror ("ioperm 0x3e0");
	exit (1);
    }
    if (ioperm (0x300, 32, 1)) {
	perror ("ioperm 0x300");
	exit (1);
    }
    if ((mem_fd = open ("/dev/mem", 2, 0)) < 0) {
	perror ("/dev/mem");
	exit (1);
    }
    
    pcic_power_on (0);
    pcic_reset (0);

    /*
     * map card information structure into 0xd0000...0xd0fff
     * mapping as bytes for now, even though we only use
     * every other one.
     */
    pcic_map_memory (0, 4, 0xD0000L, 0L, 0xFFFL, ATTRIBUTE, 1);

    /* copy this space into local memory and dump it in hex */
    lseek (mem_fd, 0xD0000L, 0);
    read (mem_fd, buf, 4*1024);
#define FOO(x) (((x) >= ' ' && (x) <= '~') ? (x) : '.')
    for (i = 0; i < 1024; i += 32) {
	for (j = 0; j < 32; j += 2)
	    printf ("%02x ", buf[i+j] & 0xff);
	for (j = 0; j < 32; j += 2)
	    printf ("%c", FOO(buf[i+j] & 0xff));
	printf ("\n");
    }
    i = 0;
    while (buf[i] != (unsigned char) 0xff) {
	int link = buf[i+2];
	switch (buf[i]) {
	case 0x15:
	    /* level 1 version/product info */
	    printf ("product info:\n");
	    for (j = i + 4; buf[j] != 0xff; j += 2)
		printf ("%c", buf[j] ? buf[j] : '/');
	    printf ("\n");
	    break;
	}
	i += 2 * link + 4;
    }
    printf ("MAC address: %x:%x:%x:%x:%x:%x\n",
	    buf[0xff0], buf[0xff2], buf[0xff4],
	    buf[0xff6], buf[0xff8], buf[0xffa]);
    /*
     * print out card configuration registers.  in real life
     * we would have gotten the offset of 0x20000 from the
     * configuration tuples in the card information structure.
     * (mapping as bytes now, maybe should be words?)
     */
    pcic_map_memory (0, 3, 0xD1000L, 0x20000L, 0x8L, ATTRIBUTE, 1);
    if (lseek (mem_fd, 0xD1000L, 0) < 0) {
	perror ("lseek 0xd1000");
	exit (1);
    }
    if (read (mem_fd, buf, 8) < 0) {
	perror ("read 0xd1000");
	exit (1);
    }

    printf ("configuration regs:\n");
    for (j = 0; j < 8; j += 2)
	printf ("%05x: %02x\n", 0x20000 + j, buf[j] & 0xff);
    buf[0] = (buf[0] & 0xc0) | 1; /* config index = 1 */
    buf[6] = 0x10;
    if (lseek (mem_fd, 0xD1000L, 0) < 0) {
	perror ("lseek 0xd1000");
	exit (1);
    }
    if (write (mem_fd, buf, 8) < 0) {
	perror ("write 0xd1000");
	exit (1);
    }
    printf ("after modification\n");
    for (j = 0; j < 8; j += 2)
	printf ("%05x: %02x\n", 0x20000 + j, buf[j] & 0xff);

    for (page = 0; page < 8; ++page) {
	pcic_unmap_memory (0, 4);
	pcic_map_memory (0, 4, 0xD2000L, page * 0x4000, 0x4000,
			 ATTRIBUTE, 1);
	if (lseek (mem_fd, 0xD2000L, 0) < 0) {
	    perror ("lseek 0xd2000");
	    exit (1);
	}
	if (read (mem_fd, buf, sizeof (buf)) < 0) {
	    perror ("read 0xd2000");
	    exit (1);
	}
	for (i = 0; i < 0x4000; i += 16) {
	    int notff = 0;
	    for (j = 0; j < 16; j += 1)
	        if ((buf[i+j] & 0xff)  != 0xff)
		    notff = 1;
	    if (notff == 0)
	        continue;
	    printf ("%x: ", page * 0x4000 + i);
	    for (j = 0; j < 16; j += 1)
	      printf ("%02x ", buf[i+j] & 0xff);
	    for (j = 0; j < 16; j += 1)
	      printf ("%c", FOO(buf[i+j] & 0xff));
	    printf ("\n");
	}
    }



    /* map regs into 0x300 ... */
    pcic_map_io (0, 0, 0x300, 32, 1);

    printf ("page 0\n");
    for (j = 0; j < 32; ++j)
	printf ("%02x: %02x\n", j + 0x300, inb_p (j + 0x300) & 0xff);

    outb_p ((inb_p (0x300) & 0x3f) | 0x40, 0x300);
    printf ("page 1\n");
    for (j = 0; j < 32; ++j)
	printf ("%02x: %02x\n", j + 0x300, inb_p (j + 0x300) & 0xff);

    outb_p (inb_p (0x300) & 0x3f, 0x300);

    /* clear buf */
    for (i=0;  i < 0x400; i++)
	buf[i] = 0x00;

    pcic_unmap_memory (0, 4);

    /* map shared ram (as WORDS) into system memory space */
    pcic_map_memory (0, 0, 0xd4000L, 0x4000L, 0x3000L, COMMON, 2);
    printf ("pcic regs:\n");
    for (j = 0; j < 0x40; ++j)
	printf ("%02x: %02x\n", j, pcic_getb (0, j) & 0xff);


    printf("Reading Ram pass 1\n");
    if (lseek (mem_fd, 0xd4000L, 0) < 0) {
	perror ("lseek 0xd4000");
	exit (1);
    }
    /* Snarf RAM */
    for (i = 0; i < 8192; ++i) {
	read (mem_fd, wbuf + i, sizeof (unsigned short));
    }
    for (i = 0; i < 0x100; i += 16) {
	printf ("%x: ",  i);
	for (j = 0; j < 16; j += 1)
	    printf ("%02x ", ((char *) wbuf)[i+j] & 0xff);
	for (j = 0; j < 16; j += 1)
	    printf ("%c", FOO(((char *) wbuf)[i+j] & 0xff));
	printf ("\n");
    }

    /* Scribble into buffer */

    wbuf[0] = 0x1234;

    printf("Writing Ram pass 1\n");
    if (lseek (mem_fd, 0xd4000L, 0) < 0) {
	perror ("lseek 0xd4000");
	exit (1);
    }

    write (mem_fd, wbuf, 2);
#if 0
    for (i = 0; i < 0x8192; i++) {
	write (mem_fd, wbuf + i, sizeof (unsigned short));
    }
#endif
    
    /* clear buf */
    for (i=0;  i < 0x100; i++)
	wbuf[i] = 0x00;

    printf("Reading Ram pass 2\n");
    /* Snarf RAM */
    if (lseek (mem_fd, 0xd4000L, 0) < 0) {
	perror ("lseek 0xd4000");
	exit (1);
    }
    for (i = 0; i < 0x100; i++) {
	read (mem_fd, wbuf + i, sizeof (unsigned short));
    }

    for (i = 0; i < 0x400; i += 16) {
	printf ("%x: ",  i);
	for (j = 0; j < 16; j += 1)
	    printf ("%02x ", ((char *) wbuf)[i+j] & 0xff);
	for (j = 0; j < 16; j += 1)
	    printf ("%c", FOO(((char *) wbuf)[i+j] & 0xff));
	printf ("\n");
    }

    pcic_unmap_io (0, 0);
    pcic_unmap_memory (0, 0);
}

#endif

/*
 * Local variables:
 * compile-command: "gcc -D__KERNEL__ -Wall -O6 -fomit-frame-pointer -I/usr/src/linux/net/tcp -c ibmccae.c"
 * version-control: t
 * kept-new-versions: 5
 * End:
 */
