package Process::Child::Leash;
$Process::Child::Leash::VERSION = '000.002';
=head1 NAME

Process::Child::Leash - to make sure the child process wont get lost with their parent.

=head1 DESCRIPTION

Here is the issue. The parent process is a wrapping bash script around the real process (child).
If we stopped the wrapper script. The real process ( child ) will be still remained and running as normal.

How to terminal the parent process and the child process would be stopped as a chain reaction?

 +
 |--run.sh
   |
   |-- perl script.pl

This module will keep an eye on the parent process. When the parent is gone. It will remove the
the real process ( child ).

=head1 USAGE

=head1 ATTACH IT WITHIN THE CODE

 #!/usr/bin/perl
 use strict;
 use warnings;
 use Process::Child::Leash;

 ... start of the script ...

=head1 USE IT OUTSIDE THE CODE

 #!/bin/bash
 export SOMETHING=FOOBAR
 perl -MProcess::Child::Leash script.pl

=head1 TIMEOUT THE HANGING PROCESS

Timeout after 10 seconds running.

 #!/bin/bash
 export CHILD_LEASH_TIMEOUT=10
 export DBIC_TRACE=1
 perl -MProcess::Child::Leash script.pl

 OR

 perl -MProcess::Child::Leash=timeout,10 script.pl

=cut

use strict;
use warnings;
use Unix::PID;
use Async;
use Mouse;

has _started_time => (
    is      => "ro",
    isa     => "Int",
    builder => "_build__started_time",
);

sub _build__started_time { time }

has _child_pid => (
    is      => "ro",
    isa     => "Int",
    builder => "_build__child_process",
);

sub _build__child_process { $$ }

has _parent_pid => (
    is      => "ro",
    isa     => "Int",
    builder => "_build__parent_pid",
);

sub _build__parent_pid { getppid() }

has timeout => (
    is      => "ro",
    isa     => "Int",
    builder => "_build_timeout",
);

sub _build_timeout {
         $ENV{CHILD_LEASH_TIMEOUT}
      || $ENV{ZATON_PROCESS_TIMEOUT}
      || $ENV{MUNNER_TIMEOUT}
      || 0;
}

no Mouse;

sub import {
    my $class = shift;
    my %args  = @_;
    return
      if $args{test};
    my $leash = $class->new(%args)->tieup;
}

sub tieup {
    my $self = shift;
    my $hook = Async->new( sub { $self->_hook } );
    while ( !$hook->ready ) {
        sleep 1;
    }
    exit;
}

sub _hook {
    my $self       = shift;
    my $parent_pid = $self->_parent_pid;
    my $child_pid  = $self->_child_pid;
    my $start_at   = $self->_started_time;
    my $timeout    = $self->timeout;

  CHECK: while (1) {
        if ( $self->_is_timeout && $self->_kill_child_process ) {
            return "timeout. killed child process";
        }
        elsif ( !$self->_is_child_still_running ) {
            return "child is gone. finish checking";
        }
        elsif ( $self->_is_parent_still_running ) {
            sleep 1;
            next CHECK;
        }
        elsif ( $self->_kill_child_process ) {
            return "parent is gone. killed child process";
        }
    }
}

{
    my $RETRY = 0;

    sub _kill_child_process {
        my $self = shift;

        my $pid = $self->_child_pid;

        $self->_kill_process($pid);

        if ( $RETRY++ < 10 ) {
            return $self->_is_process_still_running($pid) ? 0 : 1;
        }
        else {
            warn ">> Cannot kill child process $pid.\n";
            return 1;
        }
    }
}

sub _kill_process { kill -9, shift }

sub _is_timeout {
    my $self    = shift;
    my $timeout = $self->timeout
      or return;
    my $started_time = $self->_started_time;
    my $used_time    = time - $started_time;
    return ( $used_time > $self->timeout ) ? 1 : 0;
}

sub _is_process_still_running {
    my $self = shift;
    my $pid  = shift;
    return Unix::PID->new->pid_info($pid) ? 1 : 0;
}

sub _is_parent_still_running {
    my $self = shift;
    $self->_is_process_still_running($self->_parent_pid);
}

sub _is_child_still_running {
    my $self = shift;
    $self->_is_process_still_running($self->_child_pid);
}

1;
