package Lingua::FI::Kontti;

=pod

=head1 NAME

Lingua::FI::Kontti - Finnish Pig Latin (kontinkieli)

=head1 NIMI

Lingua::FI::Kontti - kontinkieli

=head1 SYNOPSIS

    use Lingua::FI::Kontti qw(kontita);

    print kontita("on meill hauska tti"), "\n";
    # will print "kon ontti keill montti kauska hontti koti tntti\n";

    print kontita("on meill hauska tti", "tunkki"), "\n";
    # will print "tun onkki teill munkki tauska hunkki tuti tnkki\n";

=head1 KYTT

    use Lingua::FI::Kontti qw(kontita);

    print kontita("on meill hauska tti"), "\n";
    # tulostaa "kon ontti keill montti kauska hontti koti tntti\n";

    print kontita("on meill hauska tti", "tunkki"), "\n";
    # tulostaa "tun onkki teill munkki tauska hunkki tuti tnkki\n";

=head1 DESCRIPTION

Similar to Pig Latin of English

    English    We're gonna rock around the clock tonight
    Pig Latin  E'reway onnagay ockray aroundway ethay ockclay onighttay

there's a tongue-and-brain-twisting "secret" kids' language for
Finnish, called "kontinkieli" ("kontti speak").  In principle the
transformation is simple: the beginning of the word you want to
translate is switched with the beginning of the word "kontti".  In
practice it's a little bit more complicated that that because one has
to know the Finnish syllable division and vowel harmony rules.

With this module you can converse like a pro with Finnish kids.

In addition to the standard "secret key" I<kontti> you can use
any other word that according to Finnish syllable division rules
starts with CVCC (consonant-vowel-consonant-constant) syllable,
like for example I<kirppu>, I<linssi>, I<portti>, I<salkku>, I<turkki>.
Give the keyword as the second argument.

=head1 KUVAUS

Tll modulilla voit knt suomea kontiksi.

"Salaisen avaimen" I<kontti> sijasta voit kytt mit tahansa sanaa joka
suomeksi tavutettuna alkaa KVKK-tavulla (konsonantti-vokaali-konsonantti-
konsonantti), kuten esimerkiksi I<kirppu>, I<linssi>, I<portti>, I<salkku>,
I<turkki>.  Anna avainsana toisena argumenttina.

Englannin puhujilla on samankaltainen lastenkieli, "sikalatina" (Pig Latin),
yll esimerkki.

=head1 ACKNOWLEDGEMENTS

=over 4

=item *

Pig Latin translator

http://www.snowcrest.net/donnelly/piglatin.html

=item *

Rock Around the Clock

Bill Haley and the Comets

=back

=head1 KIITOKSET

=over 4

=item *

Sikalatinaknnin

http://www.snowcrest.net/donnelly/piglatin.html

=item *

Rock Around the Clock

Bill Haley and the Comets

=back

=head1 AUTHOR

Jarkko Hietaniemi <jhi@iki.fi>

=head1 COPYRIGHT

Copyright 2001 Jarkko Hietaniemi

=head1 LICENSE

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself. 

=head1 TEKIJ

Jarkko Hietaniemi <jhi@iki.fi>

=head1 TEKIJNOIKEUS

Copyright 2001 Jarkko Hietaniemi

=head1 LISENSSI

Tm kirjastomoduli on vapaa; voit jakaa ja/tai muuttaa sit samojen
ehtojen mukaisesti kuin Perli itsen.

=cut

use strict;

use vars qw($VERSION @ISA @EXPORT_OK);

$VERSION = 0.02;

require Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw(kontita);

use Lingua::FI::Hyphenate 'tavuta';

my $vp = "aeiouy";
my $vI = "AEIOUY";
my $v  = "$vp$vI";
my $kp = "bcdfghjklmnpqrstvwxz";
my $kI = "BCDFGHJKLMNPQRSTVWXZ";
my $k  = "$kp$kI";
my $V  = "[$v]";
my $K  = "[$k]";
my $p  = "$vp$kp";
my $I  = "$vI$kI";

sub kontita {
    my $s = shift;

    my $kontti = @_ ? shift : "kontti";

    my @ak = tavuta($kontti);

    die "kontita: '$kontti' ei ala KVKK\n"
	unless $ak[0] =~ /^($K)($V)($K)($K)/;

    my $k1 = substr($kontti, 0, 1);
    my $v1 = substr($kontti, 1, 1);
    my $k2 = substr($kontti, 2, 1);
    my $k3 = substr($kontti, 3, 1);

    my $kontitettu = '';

    for my $s (split(/([$v$k]+)/, $s)) {
	$kontitettu .= $s, next unless $s =~ /[$v$k]/;
	my @k = @ak;
	my @t = tavuta($s);
	if ($t[0] =~ /^($K)($V)$/) {				# talo
	    $t[0] = $k1 . $v1 ;					# kolo
	    $k[0] = $1  . $2 . $k2 . $k3;			# tantti
	} elsif ($t[0] =~ /^($K)($V)($K)$/) {			# marras
	    $t[0] = $k1 . $v1  . $3;				# korras
	    $k[0] = $1  . $2 . $k2 . $k3;			# mantti
	} elsif ($t[0] =~ /^($K)($V)($V)$/) {
	    if ($2 eq $3) {					# saari
		$t[0] = $k1 . $v1 . $v1;			# koori
		$k[0] = $1  . $2  . $k2 . $k3;			# santti
	    } elsif ($3 eq $v1) {				# huomenta
		$t[0] = $k1 . $v1 . $2;				# koumenta
		$k[0] = $1  . $2  . $k2 . $k3;			# huntti
	    } else {						# taivas
		$t[0] = $k1 . $v1 . $3;				# koivas
		$k[0] = $1  . $2  . $k2 . $k3;			# tantti
	    }
	} elsif ($t[0] =~ /^($K)($V)($V)($K)$/) {
	    if ($2 eq $3) {					# saarni
		$t[0] = $k1 . $v1 . $v1 . $4;			# koorni
		$k[0] = $1  . $2  . $k2 . $k3;			# kantti
	    } else {						# hiekka
		$t[0] = $k1 . $2  . $3  . $4;			# kiekka
		$k[0] = $1  . $v1 . $k2 . $k3;			# hontti
	    }
	} elsif ($t[0] =~ /^($V)($K)$/) {			# alku
	    $t[0] = $k1 . $v1 . $2;				# kolku
	    $k[0] = $1  . $k2 . $k3;				# antti
	} elsif ($t[0] =~ /^($V)$/) {				# ase
	    $t[0] = $k1 . $v1;					# kose
	    $k[0] = $1  . $k2 . $k3;				# antti
	} elsif ($t[0] =~ /^($V)($V)$/) {
	    if ($1 eq $2) {					# aari
		$t[0] = $k1 . $v1 . $v1;			# koori
		$k[0] = $1  . $k2 . $k3;			# antti
	    } elsif ($2 eq '') {				# y
		$t[0] = $k1 . $2  . $1;				# ky
		$k[0] = $1  . $k2 . $k3;			# yntti
	    } else {						# autio
		$t[0] = $k1 . $v1 . $2;				# koutio
		$k[0] = $1  . $k2 . $k3;			# antti
	    }
	} elsif ($t[0] =~ /^($K)($V)($K)($K)$/) {		# tausta
	    $t[0] = $k1 . $v1  . $3  . $4;			# kousta
	    $k[0] = $1  . $2   . $k2 . $k3;			# tantti
	} elsif ($t[0] =~ /^($V)($V)($K)$/) {
	    if ($1 eq $2) {					# aarni
		$t[0] = $k1 . $v1 . $v1 . $3;			# koorni
		$k[0] = $1  . $k2 . $k3;			# antti
	    } else {						# aukko
		$t[0] = $k1 . $v1 . $2  . $3;			# koukko
		$k[0] = $1  . $k2 . $k3;			# antti
	    }
	} elsif ($t[0] =~ /^($V)($K)($K)$/) {			# arkku
	    $t[0] = $k1 . $v1  . $2  . $3;			# korkku
	    $k[0] = $1  . $k2 . $k3;				# antti
	} elsif ($t[0] =~ /^($K)($K)($V)$/) {			# trapetsi
	    $t[0] = $k1 . $v1;					# kopetsi
	    $k[0] = $1  . $2  . $3 . $k2  . $k3;		# trantti
	} elsif ($t[0] =~ /^($K)($K)($V)($K)$/) {		# traktori
	    $t[0] = $k1 . $2  . $v1 . $4;			# kroktori
	    $k[0] = $1  . $3  . $k2 . $k3;			# tantti
	} elsif ($t[0] =~ /^($K)($K)($V)($V)$/) {
	    if ($3 eq $4) {					# traani
		$t[0] = $k1 . $v1 . $v1;			# kooni
		$k[0] = $1  . $2  . $3  .  $k2 . $k3;		# trantti
	    } else {						# trauma
		$t[0] = $k1 . $v1 . $4;				# kouma
		$k[0] = $1  . $2  . $3  .  $k2 . $k3;		# trantti
	    }
	} elsif ($t[0] =~ /^($K)($K)($V)($V)($K)$/) {		# truantti
	    $t[0] = $k1 . $v1 . $4  . $5;			# koantti
	    $k[0] = $1  . $2 . $3  . $k2 . $k3;			# trantti
	} elsif ($t[0] =~ /^($K)($K)($V)($K)($K)$/) {		# transsi
	    $t[0] = $k1 . $v1 . $4  . $5;			# konssi
	    $k[0] = $1  . $2  . $3  . $k2 . $k3;		# trantti
	}
	
	# vokaalisointu

	@t = map { tr/aouAOU/yY/; $_ } @t if grep { /[yY]/ } @t;
	@k = map { tr/aouAOU/yY/; $_ } @k if grep { /[yY]/ } @k;

	# Iso alkukirjain
	my $a = substr($t[0], 0, 1);
	my $b = substr($k[0], 0, 1);
	if ($a =~ /^[$p]/ && $b =~ /^[$I]/) {
	    substr($t[0], 0, 1) = substr($I, index($p, $a), 1);
	    substr($k[0], 0, 1) = substr($p, index($I, $b), 1);
	}

	$kontitettu .= join("", @t) . " " . join("", @k)
    }

    return $kontitettu;
}

1;
