#! /usr/bin/env false

use v6.d;

use Config;
use Grammar::DiceRolls;
use Grammar::DiceRolls::CountActions;
use Grammar::DiceRolls::ListActions;
use IRC::Client;
use X::Grammar::DiceRolls::TooManyDice;
use X::Grammar::DiceRolls::TooManySides;

unit class IRC::Client::Plugin::DiceRolls does IRC::Client::Plugin;

#| A Config instance to hold all configuration.
has Config $.config;

#| The calculated prefix for the dice roll command.
has Str $!prefix;

submethod TWEAK
{
	$!prefix = [~]
		$!config.get('irc.command-prefix', '.'),
		$!config.get('irc.plugins.dicerolls.command-word', 'roll'),
		;
}

method reload-config (
	Config $config,
) {
	$!config = $config;

	self.TWEAK;
}

multi method irc-privmsg (
	$event where { $event.text.starts-with($!prefix) }
) {
	my $roll = $event.text.substr($!prefix.chars).trim || '1d20';

	.debug("Rolling for $roll") with $*LOG;

	given ($!config.get('irc.plugins.dicerolls.style', 'list').fc) {
		when 'list' {
			my @outcomes = Grammar::DiceRolls.parse(
				$roll,
				:actions(Grammar::DiceRolls::ListActions.new(
					limit-dice => $!config.get('irc.plugins.dicerolls.limit-dice', 30),
					limit-sides => $!config.get('irc.plugins.dicerolls.limit-sides', 1_000_000_000),
				))
			).made;

			return "Invalid roll ($roll)" unless @outcomes;
			return "$roll = {@outcomes.first}" if @outcomes.elems < 2;
			return "$roll = {@outcomes.join(' + ')} = {@outcomes.sum}";
		}
		when 'count' {
			my $outcome = Grammar::DiceRolls.parse(
				$roll,
				:actions(Grammar::DiceRolls::CountActions)
			).made;

			return "Invalid roll ($roll)" unless $outcome;
			return "$roll = $outcome";
		}
		default {
			.alert('Invalid rolling style for irc.plugins.dicerolls.style') with $*LOG;
		}
	}

	CATCH {
		when X::Grammar::DiceRolls::TooManyDice {
			.error("Too many dice in $roll") with $*LOG;
			return 'No dice';
		}
		when X::Grammar::DiceRolls::TooManySides {
			.error("Too many sides in $roll") with $*LOG;
			return 'No dice';
		}
	}
}

=begin pod

=NAME    IRC::Client::Plugin::DiceRolls
=VERSION 0.2.1
=AUTHOR  Patrick Spek <p.spek@tyil.nl>

=begin LICENSE
Copyright © 2020

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU Affero General Public License as published by the Free
Software Foundation, either version 3 of the License, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
details.

You should have received a copy of the GNU Affero General Public License along
with this program.  If not, see http://www.gnu.org/licenses/.
=end LICENSE

=end pod

# vim: ft=raku noet sw=8 ts=8
