#! /usr/bin/env false

use v6.d;

use HTTP::UserAgent;
use JSON::Fast;
use URL;

unit class HTTP::API::MusicBrainz;

#| The base URL object to create all URLs from.
has URL $!root-url;

#| The user agent which will be used internally for handling HTTP requests.
has $!user-agent;

has Bool:D $.rate-limit-retry is rw = True;

submethod TWEAK ()
{
	$!user-agent = HTTP::UserAgent.new(
		useragent => 'HTTP::API::MusicBrainz/0.0.1',
	);

	if (!$!root-url) {
		$!root-url = URL.new(
			scheme => 'https',
			hostname => 'musicbrainz.org',
			path => < ws 2 >,
		);
	}
}

submethod !browse ($entity, $mbid, $limit, $offset, @inc)
{
}

submethod !lookup (
	Str:D $entity,
	Str() $mbid,
	@inc = [],
) {
	my $url = $!root-url
		.add-path(
			$entity,
			~$mbid,
		)
		.add-query(
			inc => @inc.join('+'),
		)
		;

	self!http-request($url);
}

#| Private method to perform search requests.
method !search (
	#| The name of the entity to search for.
	Str:D $entity,

	#| The query to search for.
	Str:D $query,
	
	#| The number of results to fetch.
	Int $limit = 30,

	#| The offset from the start.
	Int $offset = 0,
) {
	my $url = $!root-url
		.add-path(
			$entity,
		)
		.add-query(
			:$query,
			:$limit,
			:$offset,
		)
		;

	self!http-request($url);
}

#| Private method to handle the actual HTTP requests.
method !http-request (
	#| The URL to make the request to.
	URL:D $url,

	--> Associative
) {
	my $response = $!user-agent.get($url.add-query(fmt => 'json').Str);

	if (!$response.is-success) {
		# Check for rate limiting
		if ($!rate-limit-retry && $response.code == 503 && $response.header.field('Retry-After')) {
			my $timeout = $response.header.field('Retry-After').values.first.Int;
			note "Retry-After: $timeout";
			sleep($timeout + 1);
			return self!http-request($url);
		}

		return fail($response);
	}

	from-json($response.content);
}

method lookup-area (
	Str() $mbid,
) {
	self!lookup('area', $mbid)
}

method lookup-artist (
	Str() $mbid,
	Bool:D :$recordings = False,
	Bool:D :$releases = False,
	Bool:D :$release-groups = False,
	Bool:D :$works = False,
) {
	my @inc;

	@inc.append('recordings') if $recordings;
	@inc.append('releases') if $releases;
	@inc.append('release-groups') if $release-groups;
	@inc.append('works') if $works;

	self!lookup('artist', $mbid, @inc)
}

method lookup-collection (
	Str() $mbid,
	Bool:D :$user-collections = False,
) {
	my @inc;

	@inc.append('user-collections') if $user-collections;

	self!lookup('collections', $mbid, @inc)
}

method lookup-event (
	Str() $mbid,
) {
	self!lookup('event', $mbid)
}

method lookup-instrument (
	Str() $mbid,
) {
	self!lookup('instrument', $mbid)
}

method lookup-label (
	Str() $mbid,
	Bool:D :$releases = False,
) {
	my @inc;

	@inc.append('releases') if $releases;

	self!lookup('label', $mbid, @inc)
}

method lookup-place (
	Str() $mbid,
) {
	self!lookup('place', $mbid)
}

method lookup-recording (
	Str() $mbid,
	Bool:D :$artists = False,
	Bool:D :$releases = False,
	Bool:D :$isrcs = False,
	Bool:D :$url-rels = False,
) {
	my @inc;

	@inc.append('artists') if $artists;
	@inc.append('releases') if $releases;
	@inc.append('isrcs') if $isrcs;
	@inc.append('url-rels') if $url-rels;

	self!lookup('recording', $mbid, @inc)
}

method lookup-release (
	Str() $mbid,
	Bool:D :$artists = False,
	Bool:D :$collections = False,
	Bool:D :$labels = False,
	Bool:D :$recordings = False,
	Bool:D :$release-groups = False,
) {
	my @inc;

	@inc.append('artists') if $artists;
	@inc.append('collections') if $collections;
	@inc.append('labels') if $labels;
	@inc.append('recordings') if $recordings;
	@inc.append('release-groups') if $release-groups;

	self!lookup('release', $mbid, @inc)
}

method lookup-release-group (
	Str() $mbid,
	Bool:D :$artists = False,
	Bool:D :$releases = False,
) {
	my @inc;

	@inc.append('artists') if $artists;
	@inc.append('releases') if $releases;

	self!lookup('release-group', $mbid, @inc)
}

method lookup-series (
	Str() $mbid,
) {
	self!lookup('series', $mbid)
}

method lookup-work (
	Str() $mbid,
) {
	self!lookup('work', $mbid)
}

method lookup-url (
	Str() $mbid,
) {
	self!lookup('url', $mbid)
}

method search-artist (
	#| A string to search for.
	Str() $query,

	#| The maximum number of results to fetch.
	Int $limit = Int,

	#| An offset from the start of the result set.
	Int $offset = Int,

	--> Iterable
) {
	self!search('artist', $query, $limit, $offset)<artists>.list
}

=begin pod

=NAME    HTTP::API::MusicBrainz
=VERSION 0.0.1
=AUTHOR  Patrick Spek <p.spek@tyil.nl>

=begin LICENSE
Copyright © 2020

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU Affero General Public License as published by the Free
Software Foundation, either version 3 of the License, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
details.

You should have received a copy of the GNU Affero General Public License along
with this program.  If not, see http://www.gnu.org/licenses/.
=end LICENSE

=end pod

# vim: ft=raku noet sw=8 ts=8
