{
# GENERATE RECURSIVE DESCENT PARSER OBJECTS FROM A GRAMMAR

use 5.006;
use strict;

package Parse::RecDescent::_Runtime;

use Text::Balanced qw ( extract_codeblock extract_bracketed extract_quotelike extract_delimited );

use vars qw ( $skip );

   *defskip  = \ '\s*'; # DEFAULT SEPARATOR IS OPTIONAL WHITESPACE
   $skip  = '\s*';      # UNIVERSAL SEPARATOR IS OPTIONAL WHITESPACE
my $MAXREP  = 100_000_000;  # REPETITIONS MATCH AT MOST 100,000,000 TIMES



package Parse::RecDescent::_Runtime::LineCounter;


sub TIESCALAR   # ($classname, \$text, $thisparser, $prevflag)
{
    bless {
        text    => $_[1],
        parser  => $_[2],
        prev    => $_[3]?1:0,
          }, $_[0];
}

sub FETCH
{
    my $parser = $_[0]->{parser};
    my $cache = $parser->{linecounter_cache};
    my $from = $parser->{fulltextlen}-length(${$_[0]->{text}})-$_[0]->{prev}
;

    unless (exists $cache->{$from})
    {
        $parser->{lastlinenum} = $parser->{offsetlinenum}
          - Parse::RecDescent::_Runtime::_linecount(substr($parser->{fulltext},$from))
          + 1;
        $cache->{$from} = $parser->{lastlinenum};
    }
    return $cache->{$from};
}

sub STORE
{
    my $parser = $_[0]->{parser};
    $parser->{offsetlinenum} -= $parser->{lastlinenum} - $_[1];
    return undef;
}

sub resync   # ($linecounter)
{
    my $self = tied($_[0]);
    die "Tried to alter something other than a LineCounter\n"
        unless $self =~ /Parse::RecDescent::_Runtime::LineCounter/;

    my $parser = $self->{parser};
    my $apparently = $parser->{offsetlinenum}
             - Parse::RecDescent::_Runtime::_linecount(${$self->{text}})
             + 1;

    $parser->{offsetlinenum} += $parser->{lastlinenum} - $apparently;
    return 1;
}

package Parse::RecDescent::_Runtime::ColCounter;

sub TIESCALAR   # ($classname, \$text, $thisparser, $prevflag)
{
    bless {
        text    => $_[1],
        parser  => $_[2],
        prev    => $_[3]?1:0,
          }, $_[0];
}

sub FETCH
{
    my $parser = $_[0]->{parser};
    my $missing = $parser->{fulltextlen}-length(${$_[0]->{text}})-$_[0]->{prev}+1;
    substr($parser->{fulltext},0,$missing) =~ m/^(.*)\Z/m;
    return length($1);
}

sub STORE
{
    die "Can't set column number via \$thiscolumn\n";
}


package Parse::RecDescent::_Runtime::OffsetCounter;

sub TIESCALAR   # ($classname, \$text, $thisparser, $prev)
{
    bless {
        text    => $_[1],
        parser  => $_[2],
        prev    => $_[3]?-1:0,
          }, $_[0];
}

sub FETCH
{
    my $parser = $_[0]->{parser};
    return $parser->{fulltextlen}-length(${$_[0]->{text}})+$_[0]->{prev};
}

sub STORE
{
    die "Can't set current offset via \$thisoffset or \$prevoffset\n";
}



package Parse::RecDescent::_Runtime::Rule;

sub new ($$$$$)
{
    my $class = ref($_[0]) || $_[0];
    my $name  = $_[1];
    my $owner = $_[2];
    my $line  = $_[3];
    my $replace = $_[4];

    if (defined $owner->{"rules"}{$name})
    {
        my $self = $owner->{"rules"}{$name};
        if ($replace && !$self->{"changed"})
        {
            $self->reset;
        }
        return $self;
    }
    else
    {
        return $owner->{"rules"}{$name} =
            bless
            {
                "name"     => $name,
                "prods"    => [],
                "calls"    => [],
                "changed"  => 0,
                "line"     => $line,
                "impcount" => 0,
                "opcount"  => 0,
                "vars"     => "",
            }, $class;
    }
}

sub reset($)
{
    @{$_[0]->{"prods"}} = ();
    @{$_[0]->{"calls"}} = ();
    $_[0]->{"changed"}  = 0;
    $_[0]->{"impcount"}  = 0;
    $_[0]->{"opcount"}  = 0;
    $_[0]->{"vars"}  = "";
}

sub DESTROY {}

sub hasleftmost($$)
{
    my ($self, $ref) = @_;

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        return 1 if $prod->hasleftmost($ref);
    }

    return 0;
}

sub leftmostsubrules($)
{
    my $self = shift;
    my @subrules = ();

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        push @subrules, $prod->leftmostsubrule();
    }

    return @subrules;
}

sub expected($)
{
    my $self = shift;
    my @expected = ();

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        my $next = $prod->expected();
        unless (! $next or _contains($next,@expected) )
        {
            push @expected, $next;
        }
    }

    return join ', or ', @expected;
}

sub _contains($@)
{
    my $target = shift;
    my $item;
    foreach $item ( @_ ) { return 1 if $target eq $item; }
    return 0;
}

sub addcall($$)
{
    my ( $self, $subrule ) = @_;
    unless ( _contains($subrule, @{$self->{"calls"}}) )
    {
        push @{$self->{"calls"}}, $subrule;
    }
}

sub addprod($$)
{
    my ( $self, $prod ) = @_;
    push @{$self->{"prods"}}, $prod;
    $self->{"changed"} = 1;
    $self->{"impcount"} = 0;
    $self->{"opcount"} = 0;
    $prod->{"number"} = $#{$self->{"prods"}};
    return $prod;
}

sub addvar
{
    my ( $self, $var, $parser ) = @_;
    if ($var =~ /\A\s*local\s+([%@\$]\w+)/)
    {
        $parser->{localvars} .= " $1";
        $self->{"vars"} .= "$var;\n" }
    else
        { $self->{"vars"} .= "my $var;\n" }
    $self->{"changed"} = 1;
    return 1;
}

sub addautoscore
{
    my ( $self, $code ) = @_;
    $self->{"autoscore"} = $code;
    $self->{"changed"} = 1;
    return 1;
}

sub nextoperator($)
{
    my $self = shift;
    my $prodcount = scalar @{$self->{"prods"}};
    my $opcount = ++$self->{"opcount"};
    return "_operator_${opcount}_of_production_${prodcount}_of_rule_$self->{name}";
}

sub nextimplicit($)
{
    my $self = shift;
    my $prodcount = scalar @{$self->{"prods"}};
    my $impcount = ++$self->{"impcount"};
    return "_alternation_${impcount}_of_production_${prodcount}_of_rule_$self->{name}";
}


sub code
{
    my ($self, $namespace, $parser, $check) = @_;

eval 'undef &' . $namespace . '::' . $self->{"name"} unless $parser->{saving};

    my $code =
'
# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub ' . $namespace . '::' . $self->{"name"} .  '
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"' . $self->{"name"} . '"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [' . $self->{"name"} . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{' . $self->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;

    ' . ($parser->{deferrable}
        ? 'my $def_at = @{$thisparser->{deferred}};'
        : '') .
    '
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{' . $self->expected() . '});
    $expectation->at($_[1]);
    '. ($parser->{_check}{thisoffset}?'
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    ':'') . ($parser->{_check}{prevoffset}?'
    my $prevoffset;
    tie $prevoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser, 1;
    ':'') . ($parser->{_check}{thiscolumn}?'
    my $thiscolumn;
    tie $thiscolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser;
    ':'') . ($parser->{_check}{prevcolumn}?'
    my $prevcolumn;
    tie $prevcolumn, q{Parse::RecDescent::_Runtime::ColCounter}, \$text, $thisparser, 1;
    ':'') . ($parser->{_check}{prevline}?'
    my $prevline;
    tie $prevline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser, 1;
    ':'') . '
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    '. $self->{vars} .'
';

    my $prod;
    foreach $prod ( @{$self->{"prods"}} )
    {
        $prod->addscore($self->{autoscore},0,0) if $self->{autoscore};
        next unless $prod->checkleftmost();
        $code .= $prod->code($namespace,$self,$parser);

        $code .= $parser->{deferrable}
                ? '     splice
                @{$thisparser->{deferred}}, $def_at unless $_matched;
                  '
                : '';
    }

    $code .=
'
    unless ( $_matched || defined($score) )
    {
        ' .($parser->{deferrable}
            ? '     splice @{$thisparser->{deferred}}, $def_at;
              '
            : '') . '

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{' . $self->{"name"} .'},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{' . $self->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' rule<< (return value: [} .
                      $return . q{])}, "",
                      q{' . $self->{"name"} .'},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{' . $self->{"name"} .'},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}
';

    return $code;
}

my @left;
sub isleftrec($$)
{
    my ($self, $rules) = @_;
    my $root = $self->{"name"};
    @left = $self->leftmostsubrules();
    my $next;
    foreach $next ( @left )
    {
        next unless defined $rules->{$next}; # SKIP NON-EXISTENT RULES
        return 1 if $next eq $root;
        my $child;
        foreach $child ( $rules->{$next}->leftmostsubrules() )
        {
            push(@left, $child)
            if ! _contains($child, @left) ;
        }
    }
    return 0;
}

package Parse::RecDescent::_Runtime::Production;

sub describe ($;$)
{
    return join ' ', map { $_->describe($_[1]) or () } @{$_[0]->{items}};
}

sub new ($$;$$)
{
    my ($self, $line, $uncommit, $error) = @_;
    my $class = ref($self) || $self;

    bless
    {
        "items"    => [],
        "uncommit" => $uncommit,
        "error"    => $error,
        "line"     => $line,
        strcount   => 0,
        patcount   => 0,
        dircount   => 0,
        actcount   => 0,
    }, $class;
}

sub expected ($)
{
    my $itemcount = scalar @{$_[0]->{"items"}};
    return ($itemcount) ? $_[0]->{"items"}[0]->describe(1) : '';
}

sub hasleftmost ($$)
{
    my ($self, $ref) = @_;
    return ${$self->{"items"}}[0] eq $ref  if scalar @{$self->{"items"}};
    return 0;
}

sub isempty($)
{
    my $self = shift;
    return 0 == @{$self->{"items"}};
}

sub leftmostsubrule($)
{
    my $self = shift;

    if ( $#{$self->{"items"}} >= 0 )
    {
        my $subrule = $self->{"items"}[0]->issubrule();
        return $subrule if defined $subrule;
    }

    return ();
}

sub checkleftmost($)
{
    my @items = @{$_[0]->{"items"}};
    if (@items==1 && ref($items[0]) =~ /\AParse::RecDescent::_Runtime::Error/
        && $items[0]->{commitonly} )
    {
        Parse::RecDescent::_Runtime::_warn(2,"Lone <error?> in production treated
                        as <error?> <reject>");
        Parse::RecDescent::_Runtime::_hint("A production consisting of a single
                      conditional <error?> directive would
                      normally succeed (with the value zero) if the
                      rule is not 'commited' when it is
                      tried. Since you almost certainly wanted
                      '<error?> <reject>' Parse::RecDescent::_Runtime
                      supplied it for you.");
        push @{$_[0]->{items}},
            Parse::RecDescent::_Runtime::UncondReject->new(0,0,'<reject>');
    }
    elsif (@items==1 && ($items[0]->describe||"") =~ /<rulevar|<autoscore/)
    {
        # Do nothing
    }
    elsif (@items &&
        ( ref($items[0]) =~ /\AParse::RecDescent::_Runtime::UncondReject/
        || ($items[0]->describe||"") =~ /<autoscore/
        ))
    {
        Parse::RecDescent::_Runtime::_warn(1,"Optimizing away production: [". $_[0]->describe ."]");
        my $what = $items[0]->describe =~ /<rulevar/
                ? "a <rulevar> (which acts like an unconditional <reject> during parsing)"
             : $items[0]->describe =~ /<autoscore/
                ? "an <autoscore> (which acts like an unconditional <reject> during parsing)"
                : "an unconditional <reject>";
        my $caveat = $items[0]->describe =~ /<rulevar/
                ? " after the specified variable was set up"
                : "";
        my $advice = @items > 1
                ? "However, there were also other (useless) items after the leading "
                  . $items[0]->describe
                  . ", so you may have been expecting some other behaviour."
                : "You can safely ignore this message.";
        Parse::RecDescent::_Runtime::_hint("The production starts with $what. That means that the
                      production can never successfully match, so it was
                      optimized out of the final parser$caveat. $advice");
        return 0;
    }
    return 1;
}

sub changesskip($)
{
    my $item;
    foreach $item (@{$_[0]->{"items"}})
    {
        if (ref($item) =~ /Parse::RecDescent::_Runtime::(Action|Directive)/)
        {
            return 1 if $item->{code} =~ /\$skip\s*=/;
        }
    }
    return 0;
}

sub adddirective
{
    my ( $self, $whichop, $line, $name ) = @_;
    push @{$self->{op}},
        { type=>$whichop, line=>$line, name=>$name,
          offset=> scalar(@{$self->{items}}) };
}

sub addscore
{
    my ( $self, $code, $lookahead, $line ) = @_;
    $self->additem(Parse::RecDescent::_Runtime::Directive->new(
                  "local \$^W;
                   my \$thisscore = do { $code } + 0;
                   if (!defined(\$score) || \$thisscore>\$score)
                    { \$score=\$thisscore; \$score_return=\$item[-1]; }
                   undef;", $lookahead, $line,"<score: $code>") )
        unless $self->{items}[-1]->describe =~ /<score/;
    return 1;
}

sub check_pending
{
    my ( $self, $line ) = @_;
    if ($self->{op})
    {
        while (my $next = pop @{$self->{op}})
        {
        Parse::RecDescent::_Runtime::_error("Incomplete <$next->{type}op:...>.", $line);
        Parse::RecDescent::_Runtime::_hint(
            "The current production ended without completing the
             <$next->{type}op:...> directive that started near line
             $next->{line}. Did you forget the closing '>'?");
        }
    }
    return 1;
}

sub enddirective
{
    my ( $self, $line, $minrep, $maxrep ) = @_;
    unless ($self->{op})
    {
        Parse::RecDescent::_Runtime::_error("Unmatched > found.", $line);
        Parse::RecDescent::_Runtime::_hint(
            "A '>' angle bracket was encountered, which typically
             indicates the end of a directive. However no suitable
             preceding directive was encountered. Typically this
             indicates either a extra '>' in the grammar, or a
             problem inside the previous directive.");
        return;
    }
    my $op = pop @{$self->{op}};
    my $span = @{$self->{items}} - $op->{offset};
    if ($op->{type} =~ /left|right/)
    {
        if ($span != 3)
        {
        Parse::RecDescent::_Runtime::_error(
            "Incorrect <$op->{type}op:...> specification:
             expected 3 args, but found $span instead", $line);
        Parse::RecDescent::_Runtime::_hint(
            "The <$op->{type}op:...> directive requires a
             sequence of exactly three elements. For example:
             <$op->{type}op:leftarg /op/ rightarg>");
        }
        else
        {
        push @{$self->{items}},
            Parse::RecDescent::_Runtime::Operator->new(
                $op->{type}, $minrep, $maxrep, splice(@{$self->{"items"}}, -3));
        $self->{items}[-1]->sethashname($self);
        $self->{items}[-1]{name} = $op->{name};
        }
    }
}

sub prevwasreturn
{
    my ( $self, $line ) = @_;
    unless (@{$self->{items}})
    {
        Parse::RecDescent::_Runtime::_error(
            "Incorrect <return:...> specification:
            expected item missing", $line);
        Parse::RecDescent::_Runtime::_hint(
            "The <return:...> directive requires a
            sequence of at least one item. For example:
            <return: list>");
        return;
    }
    push @{$self->{items}},
        Parse::RecDescent::_Runtime::Result->new();
}

sub additem
{
    my ( $self, $item ) = @_;
    $item->sethashname($self);
    push @{$self->{"items"}}, $item;
    return $item;
}

sub _duplicate_itempos
{
    my ($src) = @_;
    my $dst = {};

    foreach (keys %$src)
    {
        %{$dst->{$_}} = %{$src->{$_}};
    }
    $dst;
}

sub _update_itempos
{
    my ($dst, $src, $typekeys, $poskeys) = @_;

    my @typekeys = 'ARRAY' eq ref $typekeys ?
      @$typekeys :
      keys %$src;

    foreach my $k (keys %$src)
    {
        if ('ARRAY' eq ref $poskeys)
        {
            @{$dst->{$k}}{@$poskeys} = @{$src->{$k}}{@$poskeys};
        }
        else
        {
            %{$dst->{$k}} = %{$src->{$k}};
        }
    }
}

sub preitempos
{
    return q
    {
        push @itempos, {'offset' => {'from'=>$thisoffset, 'to'=>undef},
                        'line'   => {'from'=>$thisline,   'to'=>undef},
                        'column' => {'from'=>$thiscolumn, 'to'=>undef} };
    }
}

sub incitempos
{
    return q
    {
        $itempos[$#itempos]{'offset'}{'from'} += length($lastsep);
        $itempos[$#itempos]{'line'}{'from'}   = $thisline;
        $itempos[$#itempos]{'column'}{'from'} = $thiscolumn;
    }
}

sub unincitempos
{
    # the next incitempos will properly set these two fields, but
    # {'offset'}{'from'} needs to be decreased by length($lastsep)
    # $itempos[$#itempos]{'line'}{'from'}
    # $itempos[$#itempos]{'column'}{'from'}
    return q
    {
        $itempos[$#itempos]{'offset'}{'from'} -= length($lastsep) if defined $lastsep;
    }
}

sub postitempos
{
    return q
    {
        $itempos[$#itempos]{'offset'}{'to'} = $prevoffset;
        $itempos[$#itempos]{'line'}{'to'}   = $prevline;
        $itempos[$#itempos]{'column'}{'to'} = $prevcolumn;
    }
}

sub code($$$$)
{
    my ($self,$namespace,$rule,$parser) = @_;
    my $code =
'
    while (!$_matched'
    . (defined $self->{"uncommit"} ? '' : ' && !$commit')
    . ')
    {
        ' .
        ($self->changesskip()
            ? 'local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;'
            : '') .'
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['
                      . $self->describe . ']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{' . $rule ->{name}. '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[' . $self->{"number"} . '];
        ' . (defined $self->{"error"} ? '' : '$text = $_[1];' ) . '
        my $_savetext;
        @item = (q{' . $rule->{"name"} . '});
        %item = (__RULE__ => q{' . $rule->{"name"} . '});
        my $repcount = 0;

';
    $code .=
'        my @itempos = ({});
'           if $parser->{_check}{itempos};

    my $item;
    my $i;

    for ($i = 0; $i < @{$self->{"items"}}; $i++)
    {
        $item = ${$self->{items}}[$i];

        $code .= preitempos() if $parser->{_check}{itempos};

        $code .= $item->code($namespace,$rule,$parser->{_check});

        $code .= postitempos() if $parser->{_check}{itempos};

    }

    if ($parser->{_AUTOACTION} && defined($item) && !$item->isa("Parse::RecDescent::_Runtime::Action"))
    {
        $code .= $parser->{_AUTOACTION}->code($namespace,$rule);
        Parse::RecDescent::_Runtime::_warn(1,"Autogenerating action in rule
                       \"$rule->{name}\":
                        $parser->{_AUTOACTION}{code}")
        and
        Parse::RecDescent::_Runtime::_hint("The \$::RD_AUTOACTION was defined,
                      so any production not ending in an
                      explicit action has the specified
                      \"auto-action\" automatically
                      appended.");
    }
    elsif ($parser->{_AUTOTREE} && defined($item) && !$item->isa("Parse::RecDescent::_Runtime::Action"))
    {
        if ($i==1 && $item->isterminal)
        {
            $code .= $parser->{_AUTOTREE}{TERMINAL}->code($namespace,$rule);
        }
        else
        {
            $code .= $parser->{_AUTOTREE}{NODE}->code($namespace,$rule);
        }
        Parse::RecDescent::_Runtime::_warn(1,"Autogenerating tree-building action in rule
                       \"$rule->{name}\"")
        and
        Parse::RecDescent::_Runtime::_hint("The directive <autotree> was specified,
                      so any production not ending
                      in an explicit action has
                      some parse-tree building code
                      automatically appended.");
    }

    $code .=
'
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' production: ['
                      . $self->describe . ']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;

' . ( $parser->{_check}{itempos} ? '
        if ( defined($_itempos) )
        {
            Parse::RecDescent::_Runtime::Production::_update_itempos($_itempos, $itempos[ 1], undef, [qw(from)]);
            Parse::RecDescent::_Runtime::Production::_update_itempos($_itempos, $itempos[-1], undef, [qw(to)]);
        }
' : '' ) . '

        $_matched = 1;
        last;
    }

';
    return $code;
}

1;

package Parse::RecDescent::_Runtime::Action;

sub describe { undef }

sub sethashname { $_[0]->{hashname} = '__ACTION' . ++$_[1]->{actcount} .'__'; }

sub new
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "code"      => $_[1],
        "lookahead" => $_[2],
        "line"      => $_[3],
    }, $class;
}

sub issubrule { undef }
sub isterminal { 0 }

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

'
        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'

        $_tok = ($_noactions) ? 0 : do ' . $self->{"code"} . ';
        ' . ($self->{"lookahead"}<0?'if':'unless') . ' (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        ' . ($self->{line}>=0 ? '$item{'. $self->{hashname} .'}=$_tok;' : '' ) .'
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
'
}


1;

package Parse::RecDescent::_Runtime::Directive;

sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }

sub issubrule { undef }
sub isterminal { 0 }
sub describe { $_[1] ? '' : $_[0]->{name} }

sub new ($$$$$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "code"      => $_[1],
        "lookahead" => $_[2],
        "line"      => $_[3],
        "name"      => $_[4],
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

'
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: ['
                    . $self->describe . ']},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE; ' .'
        $_tok = do { ' . $self->{"code"} . ' };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        ' . ($self->{"lookahead"} ? '$text = $_savetext and ' : '' ) .'
        last '
        . ($self->{"lookahead"}<0?'if':'unless') . ' defined $_tok;
        push @item, $item{'.$self->{hashname}.'}=$_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
'
}

1;

package Parse::RecDescent::_Runtime::UncondReject;

sub issubrule { undef }
sub isterminal { 0 }
sub describe { $_[1] ? '' : $_[0]->{name} }
sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }

sub new ($$$;$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "lookahead" => $_[1],
        "line"      => $_[2],
        "name"      => $_[3],
    }, $class;
}

# MARK, YOU MAY WANT TO OPTIMIZE THIS.


sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

'
        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found '
                     . $self->describe . ')},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'

        $_tok = undef;
        ' . ($self->{"lookahead"} ? '$text = $_savetext and ' : '' ) .'
        last '
        . ($self->{"lookahead"}<0?'if':'unless') . ' defined $_tok;
'
}

1;

package Parse::RecDescent::_Runtime::Error;

sub issubrule { undef }
sub isterminal { 0 }
sub describe { $_[1] ? '' : $_[0]->{commitonly} ? '<error?:...>' : '<error...>' }
sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }

sub new ($$$$$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "msg"        => $_[1],
        "lookahead"  => $_[2],
        "commitonly" => $_[3],
        "line"       => $_[4],
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

    my $action = '';

    if ($self->{"msg"})  # ERROR MESSAGE SUPPLIED
    {
        #WAS: $action .= "Parse::RecDescent::_Runtime::_error(qq{$self->{msg}}" .  ',$thisline);';
        $action .= 'push @{$thisparser->{errors}}, [qq{'.$self->{msg}.'},$thisline];';

    }
    else      # GENERATE ERROR MESSAGE DURING PARSE
    {
        $action .= '
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        ';
    }

    my $dir =
          new Parse::RecDescent::_Runtime::Directive('if (' .
        ($self->{"commitonly"} ? '$commit' : '1') .
        ") { do {$action} unless ".' $_noactions; undef } else {0}',
                    $self->{"lookahead"},0,$self->describe);
    $dir->{hashname} = $self->{hashname};
    return $dir->code($namespace, $rule, 0);
}

1;

package Parse::RecDescent::_Runtime::Token;

sub sethashname { $_[0]->{hashname} = '__PATTERN' . ++$_[1]->{patcount} . '__'; }

sub issubrule { undef }
sub isterminal { 1 }
sub describe ($) { shift->{'description'}}


# ARGS ARE: $self, $pattern, $left_delim, $modifiers, $lookahead, $linenum
sub new ($$$$$$)
{
    my $class = ref($_[0]) || $_[0];
    my $pattern = $_[1];
    my $pat = $_[1];
    my $ldel = $_[2];
    my $rdel = $ldel;
    $rdel =~ tr/{[(</}])>/;

    my $mod = $_[3];

    my $desc;

    if ($ldel eq '/') { $desc = "$ldel$pattern$rdel$mod" }
    else          { $desc = "m$ldel$pattern$rdel$mod" }
    $desc =~ s/\\/\\\\/g;
    $desc =~ s/\$$/\\\$/g;
    $desc =~ s/}/\\}/g;
    $desc =~ s/{/\\{/g;

    if (!eval "no strict;
           local \$SIG{__WARN__} = sub {0};
           '' =~ m$ldel$pattern$rdel$mod" and $@)
    {
        Parse::RecDescent::_Runtime::_warn(3, "Token pattern \"m$ldel$pattern$rdel$mod\"
                         may not be a valid regular expression",
                       $_[5]);
        $@ =~ s/ at \(eval.*/./;
        Parse::RecDescent::_Runtime::_hint($@);
    }

    # QUIETLY PREVENT (WELL-INTENTIONED) CALAMITY
    $mod =~ s/[gc]//g;
    $pattern =~ s/(\A|[^\\])\\G/$1/g;

    bless
    {
        "pattern"   => $pattern,
        "ldelim"      => $ldel,
        "rdelim"      => $rdel,
        "mod"         => $mod,
        "lookahead"   => $_[4],
        "line"        => $_[5],
        "description" => $desc,
    }, $class;
}


sub code($$$$$)
{
    my ($self, $namespace, $rule, $check) = @_;
    my $ldel = $self->{"ldelim"};
    my $rdel = $self->{"rdelim"};
    my $sdel = $ldel;
    my $mod  = $self->{"mod"};

    $sdel =~ s/[[{(<]/{}/;

my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [' . $self->describe
                      . ']}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{' . ($rule->hasleftmost($self) ? ''
                : $self->describe ) . '})->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) . '

        ' . ($self->{"lookahead"}<0?'if':'unless')
        . ' ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and '
        . ($check->{itempos}? 'do {'.Parse::RecDescent::_Runtime::Production::incitempos().' 1} and ' : '')
        . '  $text =~ m' . $ldel . '\A(?:' . $self->{"pattern"} . ')' . $rdel . $mod . ')
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '$text = $lastsep . $text if defined $lastsep;') .
            ($check->{itempos} ? Parse::RecDescent::_Runtime::Production::unincitempos() : '') . '
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn\'t match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{'.$self->{hashname}.'}=$current_match;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
';

    return $code;
}

1;

package Parse::RecDescent::_Runtime::Literal;

sub sethashname { $_[0]->{hashname} = '__STRING' . ++$_[1]->{strcount} . '__'; }

sub issubrule { undef }
sub isterminal { 1 }
sub describe ($) { shift->{'description'} }

sub new ($$$$)
{
    my $class = ref($_[0]) || $_[0];

    my $pattern = $_[1];

    my $desc = $pattern;
    $desc=~s/\\/\\\\/g;
    $desc=~s/}/\\}/g;
    $desc=~s/{/\\{/g;

    bless
    {
        "pattern"     => $pattern,
        "lookahead"   => $_[2],
        "line"        => $_[3],
        "description" => "'$desc'",
    }, $class;
}


sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [' . $self->describe
                      . ']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{' . ($rule->hasleftmost($self) ? ''
                : $self->describe ) . '})->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) . '

        ' . ($self->{"lookahead"}<0?'if':'unless')
        . ' ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and '
        . ($check->{itempos}? 'do {'.Parse::RecDescent::_Runtime::Production::incitempos().' 1} and ' : '')
        . '  $text =~ m/\A' . quotemeta($self->{"pattern"}) . '/)
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '$text = $lastsep . $text if defined $lastsep;').'
            '. ($check->{itempos} ? Parse::RecDescent::_Runtime::Production::unincitempos() : '') . '
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn\'t match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{'.$self->{hashname}.'}=$current_match;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
';

    return $code;
}

1;

package Parse::RecDescent::_Runtime::InterpLit;

sub sethashname { $_[0]->{hashname} = '__STRING' . ++$_[1]->{strcount} . '__'; }

sub issubrule { undef }
sub isterminal { 1 }
sub describe ($) { shift->{'description'} }

sub new ($$$$)
{
    my $class = ref($_[0]) || $_[0];

    my $pattern = $_[1];
    $pattern =~ s#/#\\/#g;

    my $desc = $pattern;
    $desc=~s/\\/\\\\/g;
    $desc=~s/}/\\}/g;
    $desc=~s/{/\\{/g;

    bless
    {
        "pattern"   => $pattern,
        "lookahead" => $_[2],
        "line"      => $_[3],
        "description" => "'$desc'",
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [' . $self->describe
                      . ']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{name} . '},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{' . ($rule->hasleftmost($self) ? ''
                : $self->describe ) . '})->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) . '

        ' . ($self->{"lookahead"}<0?'if':'unless')
        . ' ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and '
        . ($check->{itempos}? 'do {'.Parse::RecDescent::_Runtime::Production::incitempos().' 1} and ' : '')
        . '  do { $_tok = "' . $self->{"pattern"} . '"; 1 } and
             substr($text,0,length($_tok)) eq $_tok and
             do { substr($text,0,length($_tok)) = ""; 1; }
        )
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '$text = $lastsep . $text if defined $lastsep;').'
            '. ($check->{itempos} ? Parse::RecDescent::_Runtime::Production::unincitempos() : '') . '
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn\'t match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $_tok . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{'.$self->{hashname}.'}=$_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
';

    return $code;
}

1;

package Parse::RecDescent::_Runtime::Subrule;

sub issubrule ($) { return $_[0]->{"subrule"} }
sub isterminal { 0 }
sub sethashname {}

sub describe ($)
{
    my $desc = $_[0]->{"implicit"} || $_[0]->{"subrule"};
    $desc = "<matchrule:$desc>" if $_[0]->{"matchrule"};
    return $desc;
}

sub callsyntax($$)
{
    if ($_[0]->{"matchrule"})
    {
        return "&{'$_[1]'.qq{$_[0]->{subrule}}}";
    }
    else
    {
        return $_[1].$_[0]->{"subrule"};
    }
}

sub new ($$$$;$$$)
{
    my $class = ref($_[0]) || $_[0];
    bless
    {
        "subrule"   => $_[1],
        "lookahead" => $_[2],
        "line"      => $_[3],
        "implicit"  => $_[4] || undef,
        "matchrule" => $_[5],
        "argcode"   => $_[6] || undef,
    }, $class;
}


sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

'
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [' . $self->{"subrule"} . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{' . $rule->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(' . ($rule->hasleftmost($self) ? 'q{}'
                # WAS : 'qq{'.$self->describe.'}' ) . ')->at($text);
                : 'q{'.$self->describe.'}' ) . ')->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' )
        . ($self->{"lookahead"}<0?'if':'unless')
        . ' (defined ($_tok = '
        . $self->callsyntax($namespace.'::')
        . '($thisparser,$text,$repeating,'
        . ($self->{"lookahead"}?'1':'$_noactions')
        . ($self->{argcode} ? ",sub { return $self->{argcode} }"
                   : ',sub { \\@arg }')
        . ($check->{"itempos"}?',$itempos[$#itempos]':',undef')
        . ')))
        {
            '.($self->{"lookahead"} ? '$text = $_savetext;' : '').'
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' subrule: ['
            . $self->{subrule} . ']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{' . $rule->{"name"} .'},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' subrule: ['
                    . $self->{subrule} . ']<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{' . $self->{subrule} . '}} = $_tok;
        push @item, $_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'
        }
'
}

package Parse::RecDescent::_Runtime::Repetition;

sub issubrule ($) { return $_[0]->{"subrule"} }
sub isterminal { 0 }
sub sethashname {  }

sub describe ($)
{
    my $desc = $_[0]->{"expected"} || $_[0]->{"subrule"};
    $desc = "<matchrule:$desc>" if $_[0]->{"matchrule"};
    return $desc;
}

sub callsyntax($$)
{
    if ($_[0]->{matchrule})
        { return "sub { goto &{''.qq{$_[1]$_[0]->{subrule}}} }"; }
    else
        { return "\\&$_[1]$_[0]->{subrule}"; }
}

sub new ($$$$$$$$$$)
{
    my ($self, $subrule, $repspec, $min, $max, $lookahead, $line, $parser, $matchrule, $argcode) = @_;
    my $class = ref($self) || $self;
    ($max, $min) = ( $min, $max) if ($max<$min);

    my $desc;
    if ($subrule=~/\A_alternation_\d+_of_production_\d+_of_rule/)
        { $desc = $parser->{"rules"}{$subrule}->expected }

    if ($lookahead)
    {
        if ($min>0)
        {
           return new Parse::RecDescent::_Runtime::Subrule($subrule,$lookahead,$line,$desc,$matchrule,$argcode);
        }
        else
        {
            Parse::RecDescent::_Runtime::_error("Not symbol (\"!\") before
                        \"$subrule\" doesn't make
                        sense.",$line);
            Parse::RecDescent::_Runtime::_hint("Lookahead for negated optional
                       repetitions (such as
                       \"!$subrule($repspec)\" can never
                       succeed, since optional items always
                       match (zero times at worst).
                       Did you mean a single \"!$subrule\",
                       instead?");
        }
    }
    bless
    {
        "subrule"   => $subrule,
        "repspec"   => $repspec,
        "min"       => $min,
        "max"       => $max,
        "lookahead" => $lookahead,
        "line"      => $line,
        "expected"  => $desc,
        "argcode"   => $argcode || undef,
        "matchrule" => $matchrule,
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

    my ($subrule, $repspec, $min, $max, $lookahead) =
        @{$self}{ qw{subrule repspec min max lookahead} };

'
        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [' . $self->describe . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{' . $rule->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(' . ($rule->hasleftmost($self) ? 'q{}'
                # WAS : 'qq{'.$self->describe.'}' ) . ')->at($text);
                : 'q{'.$self->describe.'}' ) . ')->at($text);
        ' . ($self->{"lookahead"} ? '$_savetext = $text;' : '' ) .'
        unless (defined ($_tok = $thisparser->_parserepeat($text, '
        . $self->callsyntax($namespace.'::')
        . ', ' . $min . ', ' . $max . ', '
        . ($self->{"lookahead"}?'1':'$_noactions')
        . ',$expectation,'
        . ($self->{argcode} ? "sub { return $self->{argcode} }"
                        : 'sub { \\@arg }')
        . ($check->{"itempos"}?',$itempos[$#itempos]':',undef')
        . ')))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' repeated subrule: ['
            . $self->describe . ']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{' . $rule->{"name"} .'},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' repeated subrule: ['
                    . $self->{subrule} . ']<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{' . "$self->{subrule}($self->{repspec})" . '}} = $_tok;
        push @item, $_tok;
        ' . ($self->{"lookahead"} ? '$text = $_savetext;' : '' ) .'

'
}

package Parse::RecDescent::_Runtime::Result;

sub issubrule { 0 }
sub isterminal { 0 }
sub describe { '' }

sub new
{
    my ($class, $pos) = @_;

    bless {}, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule) = @_;

    '
        $return = $item[-1];
    ';
}

package Parse::RecDescent::_Runtime::Operator;

my @opertype = ( " non-optional", "n optional" );

sub issubrule { 0 }
sub isterminal { 0 }

sub describe { $_[0]->{"expected"} }
sub sethashname { $_[0]->{hashname} = '__DIRECTIVE' . ++$_[1]->{dircount} .  '__'; }


sub new
{
    my ($class, $type, $minrep, $maxrep, $leftarg, $op, $rightarg) = @_;

    bless
    {
        "type"      => "${type}op",
        "leftarg"   => $leftarg,
        "op"        => $op,
        "min"       => $minrep,
        "max"       => $maxrep,
        "rightarg"  => $rightarg,
        "expected"  => "<${type}op: ".$leftarg->describe." ".$op->describe." ".$rightarg->describe.">",
    }, $class;
}

sub code($$$$)
{
    my ($self, $namespace, $rule, $check) = @_;

    my @codeargs = @_[1..$#_];

    my ($leftarg, $op, $rightarg) =
        @{$self}{ qw{leftarg op rightarg} };

    my $code = '
        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [' . $self->describe . ']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{' . $rule->{"name"} . '},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(' . ($rule->hasleftmost($self) ? 'q{}'
                # WAS : 'qq{'.$self->describe.'}' ) . ')->at($text);
                : 'q{'.$self->describe.'}' ) . ')->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;
';

    $code .= '
          my  $_itempos = $itempos[-1];
          my  $itemposfirst;
' if $check->{itempos};

    if ($self->{type} eq "leftop" )
    {
        $code .= '
          # MATCH LEFTARG
          ' . $leftarg->code(@codeargs) . '

';

        $code .= '
          if (defined($_itempos) and !defined($itemposfirst))
          {
              $itemposfirst = Parse::RecDescent::_Runtime::Production::_duplicate_itempos($_itempos);
          }
' if $check->{itempos};

        $code .= '
          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < ' . $self->{max} . ')
          {
            $backtrack = 0;
            ' . $op->code(@codeargs) . '
            ' . ($op->isterminal() ? 'pop @item;' : '$backtrack=1;' ) . '
            ' . (ref($op) eq 'Parse::RecDescent::_Runtime::Token'
                ? 'if (defined $1) {push @item, $item{'.($self->{name}||$self->{hashname}).'}=$1; $backtrack=1;}'
                : "" ) . '
            ' . $rightarg->code(@codeargs) . '
            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          ';
    }
    else
    {
        $code .= '
          my $savetext = $text;
          my $backtrack;
          # MATCH (LEFTARG OP)(s)
          while ($repcount < ' . $self->{max} . ')
          {
            $backtrack = 0;
            ' . $leftarg->code(@codeargs) . '
';
        $code .= '
            if (defined($_itempos) and !defined($itemposfirst))
            {
                $itemposfirst = Parse::RecDescent::_Runtime::Production::_duplicate_itempos($_itempos);
            }
' if $check->{itempos};

        $code .= '
            $repcount++;
            $backtrack = 1;
            ' . $op->code(@codeargs) . '
            $savetext = $text;
            ' . ($op->isterminal() ? 'pop @item;' : "" ) . '
            ' . (ref($op) eq 'Parse::RecDescent::_Runtime::Token' ? 'do { push @item, $item{'.($self->{name}||$self->{hashname}).'}=$1; } if defined $1;' : "" ) . '
          }
          $text = $savetext;
          pop @item if $backtrack;

          # MATCH RIGHTARG
          ' . $rightarg->code(@codeargs) . '
          $repcount++;
          ';
    }

    $code .= 'unless (@item) { undef $_tok; last }' unless $self->{min}==0;

    $code .= '
          $_tok = [ @item ];
';


    $code .= '
          if (defined $itemposfirst)
          {
              Parse::RecDescent::_Runtime::Production::_update_itempos(
                  $_itempos, $itemposfirst, undef, [qw(from)]);
          }
' if $check->{itempos};

    $code .= '
          last;
        } # end of OPLOOP
';

    $code .= '
        unless ($repcount>='.$self->{min}.')
        {
            Parse::RecDescent::_Runtime::_trace(q{<<'.Parse::RecDescent::_Runtime::_matchtracemessage($self,1).' operator: ['
                          . $self->describe
                          . ']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{' . $rule->{"name"} .'},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>'.Parse::RecDescent::_Runtime::_matchtracemessage($self).' operator: ['
                      . $self->describe
                      . ']<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{' . $rule->{"name"} .'},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{'.($self->{name}||$self->{hashname}).'}=$_tok||[];
';

    return $code;
}


package Parse::RecDescent::_Runtime::Expectation;

sub new ($)
{
    bless {
        "failed"      => 0,
        "expected"    => "",
        "unexpected"      => "",
        "lastexpected"    => "",
        "lastunexpected"  => "",
        "defexpected"     => $_[1],
          };
}

sub is ($$)
{
    $_[0]->{lastexpected} = $_[1]; return $_[0];
}

sub at ($$)
{
    $_[0]->{lastunexpected} = $_[1]; return $_[0];
}

sub failed ($)
{
    return unless $_[0]->{lastexpected};
    $_[0]->{expected}   = $_[0]->{lastexpected}   unless $_[0]->{failed};
    $_[0]->{unexpected} = $_[0]->{lastunexpected} unless $_[0]->{failed};
    $_[0]->{failed} = 1;
}

sub message ($)
{
    my ($self) = @_;
    $self->{expected} = $self->{defexpected} unless $self->{expected};
    $self->{expected} =~ s/_/ /g;
    if (!$self->{unexpected} || $self->{unexpected} =~ /\A\s*\Z/s)
    {
        return "Was expecting $self->{expected}";
    }
    else
    {
        $self->{unexpected} =~ /\s*(.*)/;
        return "Was expecting $self->{expected} but found \"$1\" instead";
    }
}

1;

package Parse::RecDescent::_Runtime;

use Carp;
use vars qw ( $AUTOLOAD $VERSION $_FILENAME);

my $ERRORS = 0;

our $VERSION = '1.967009';
$VERSION = eval $VERSION;
$_FILENAME=__FILE__;

# BUILDING A PARSER

my $nextnamespace = "namespace000001";

sub _nextnamespace()
{
    return "Parse::RecDescent::_Runtime::" . $nextnamespace++;
}

# ARGS ARE: $class, $grammar, $compiling, $namespace
sub new ($$$$)
{
    my $class = ref($_[0]) || $_[0];
    local $Parse::RecDescent::_Runtime::compiling = $_[2];
    my $name_space_name = defined $_[3]
        ? "Parse::RecDescent::_Runtime::".$_[3]
        : _nextnamespace();
    my $self =
    {
        "rules"     => {},
        "namespace" => $name_space_name,
        "startcode" => '',
        "localvars" => '',
        "_AUTOACTION" => undef,
        "_AUTOTREE"   => undef,

        # Precompiled parsers used to set _precompiled, but that
        # wasn't present in some versions of Parse::RecDescent::_Runtime used to
        # build precompiled parsers.  Instead, set a new
        # _not_precompiled flag, which is remove from future
        # Precompiled parsers at build time.
        "_not_precompiled" => 1,
    };


    if ($::RD_AUTOACTION) {
        my $sourcecode = $::RD_AUTOACTION;
        $sourcecode = "{ $sourcecode }"
            unless $sourcecode =~ /\A\s*\{.*\}\s*\Z/;
        $self->{_check}{itempos} =
            $sourcecode =~ /\@itempos\b|\$itempos\s*\[/;
        $self->{_AUTOACTION}
            = new Parse::RecDescent::_Runtime::Action($sourcecode,0,-1)
    }

    bless $self, $class;
    return $self->Replace($_[1])
}

sub Compile($$$$) {
    die "Compilation of Parse::RecDescent::_Runtime grammars not yet implemented\n";
}

sub DESTROY {
    my ($self) = @_;
    my $namespace = $self->{namespace};
    $namespace =~ s/Parse::RecDescent::_Runtime:://;
    if ($self->{_not_precompiled}) {
        # BEGIN WORKAROUND
        # Perl has a bug that creates a circular reference between
        # @ISA and that variable's stash:
        #   https://rt.perl.org/rt3/Ticket/Display.html?id=92708
        # Emptying the array before deleting the stash seems to
        # prevent the leak.  Once the ticket above has been resolved,
        # these two lines can be removed.
        no strict 'refs';
        @{$self->{namespace} . '::ISA'} = ();
        # END WORKAROUND

        # Some grammars may contain circular references between rules,
        # such as:
        #   a: 'ID' | b
        #   b: '(' a ')'
        # Unless these references are broken, the subs stay around on
        # stash deletion below.  Iterate through the stash entries and
        # for each defined code reference, set it to reference sub {}
        # instead.
        {
            local $^W; # avoid 'sub redefined' warnings.
            my $blank_sub = sub {};
            while (my ($name, $glob) = each %{"Parse::RecDescent::_Runtime::$namespace\::"}) {
                *$glob = $blank_sub if defined &$glob;
            }
        }

        # Delete the namespace's stash
        delete $Parse::RecDescent::_Runtime::{$namespace.'::'};
    }
}

# BUILDING A GRAMMAR....

# ARGS ARE: $self, $grammar, $isimplicit, $isleftop
sub Replace ($$)
{
    # set $replace = 1 for _generate
    splice(@_, 2, 0, 1);

    return _generate(@_);
}

# ARGS ARE: $self, $grammar, $isimplicit, $isleftop
sub Extend ($$)
{
    # set $replace = 0 for _generate
    splice(@_, 2, 0, 0);

    return _generate(@_);
}

sub _no_rule ($$;$)
{
    _error("Ruleless $_[0] at start of grammar.",$_[1]);
    my $desc = $_[2] ? "\"$_[2]\"" : "";
    _hint("You need to define a rule for the $_[0] $desc
           to be part of.");
}

my $NEGLOOKAHEAD    = '\G(\s*\.\.\.\!)';
my $POSLOOKAHEAD    = '\G(\s*\.\.\.)';
my $RULE        = '\G\s*(\w+)[ \t]*:';
my $PROD        = '\G\s*([|])';
my $TOKEN       = q{\G\s*/((\\\\/|\\\\\\\\|[^/])*)/([cgimsox]*)};
my $MTOKEN      = q{\G\s*(m\s*[^\w\s])};
my $LITERAL     = q{\G\s*'((\\\\['\\\\]|[^'])*)'};
my $INTERPLIT       = q{\G\s*"((\\\\["\\\\]|[^"])*)"};
my $SUBRULE     = '\G\s*(\w+)';
my $MATCHRULE       = '\G(\s*<matchrule:)';
my $SIMPLEPAT       = '((\\s+/[^/\\\\]*(?:\\\\.[^/\\\\]*)*/)?)';
my $OPTIONAL        = '\G\((\?)'.$SIMPLEPAT.'\)';
my $ANY         = '\G\((s\?)'.$SIMPLEPAT.'\)';
my $MANY        = '\G\((s|\.\.)'.$SIMPLEPAT.'\)';
my $EXACTLY     = '\G\(([1-9]\d*)'.$SIMPLEPAT.'\)';
my $BETWEEN     = '\G\((\d+)\.\.([1-9]\d*)'.$SIMPLEPAT.'\)';
my $ATLEAST     = '\G\((\d+)\.\.'.$SIMPLEPAT.'\)';
my $ATMOST      = '\G\(\.\.([1-9]\d*)'.$SIMPLEPAT.'\)';
my $BADREP      = '\G\((-?\d+)?\.\.(-?\d+)?'.$SIMPLEPAT.'\)';
my $ACTION      = '\G\s*\{';
my $IMPLICITSUBRULE = '\G\s*\(';
my $COMMENT     = '\G\s*(#.*)';
my $COMMITMK        = '\G\s*<commit>';
my $UNCOMMITMK      = '\G\s*<uncommit>';
my $QUOTELIKEMK     = '\G\s*<perl_quotelike>';
my $CODEBLOCKMK     = '\G\s*<perl_codeblock(?:\s+([][()<>{}]+))?>';
my $VARIABLEMK      = '\G\s*<perl_variable>';
my $NOCHECKMK       = '\G\s*<nocheck>';
my $AUTOACTIONPATMK = '\G\s*<autoaction:';
my $AUTOTREEMK      = '\G\s*<autotree(?::\s*([\w:]+)\s*)?>';
my $AUTOSTUBMK      = '\G\s*<autostub>';
my $AUTORULEMK      = '\G\s*<autorule:(.*?)>';
my $REJECTMK        = '\G\s*<reject>';
my $CONDREJECTMK    = '\G\s*<reject:';
my $SCOREMK     = '\G\s*<score:';
my $AUTOSCOREMK     = '\G\s*<autoscore:';
my $SKIPMK      = '\G\s*<skip:';
my $OPMK        = '\G\s*<(left|right)op(?:=(\'.*?\'))?:';
my $ENDDIRECTIVEMK  = '\G\s*>';
my $RESYNCMK        = '\G\s*<resync>';
my $RESYNCPATMK     = '\G\s*<resync:';
my $RULEVARPATMK    = '\G\s*<rulevar:';
my $DEFERPATMK      = '\G\s*<defer:';
my $TOKENPATMK      = '\G\s*<token:';
my $AUTOERRORMK     = '\G\s*<error(\??)>';
my $MSGERRORMK      = '\G\s*<error(\??):';
my $NOCHECK     = '\G\s*<nocheck>';
my $WARNMK      = '\G\s*<warn((?::\s*(\d+)\s*)?)>';
my $HINTMK      = '\G\s*<hint>';
my $TRACEBUILDMK    = '\G\s*<trace_build((?::\s*(\d+)\s*)?)>';
my $TRACEPARSEMK    = '\G\s*<trace_parse((?::\s*(\d+)\s*)?)>';
my $UNCOMMITPROD    = $PROD.'\s*<uncommit';
my $ERRORPROD       = $PROD.'\s*<error';
my $LONECOLON       = '\G\s*:';
my $OTHER       = '\G\s*([^\s]+)';

my @lines = 0;

sub _generate
{
    my ($self, $grammar, $replace, $isimplicit, $isleftop) = (@_, 0);

    my $aftererror = 0;
    my $lookahead = 0;
    my $lookaheadspec = "";
    my $must_pop_lines;
    if (! $lines[-1]) {
        push @lines, _linecount($grammar) ;
        $must_pop_lines = 1;
    }
    $self->{_check}{itempos} = ($grammar =~ /\@itempos\b|\$itempos\s*\[/)
        unless $self->{_check}{itempos};
    for (qw(thisoffset thiscolumn prevline prevoffset prevcolumn))
    {
        $self->{_check}{$_} =
            ($grammar =~ /\$$_/) || $self->{_check}{itempos}
                unless $self->{_check}{$_};
    }
    my $line;

    my $rule = undef;
    my $prod = undef;
    my $item = undef;
    my $lastgreedy = '';
    pos $grammar = 0;
    study $grammar;

    local $::RD_HINT  = $::RD_HINT;
    local $::RD_WARN  = $::RD_WARN;
    local $::RD_TRACE = $::RD_TRACE;
    local $::RD_CHECK = $::RD_CHECK;

    while (pos $grammar < length $grammar)
    {
        $line = $lines[-1] - _linecount($grammar) + 1;
        my $commitonly;
        my $code = "";
        my @components = ();
        if ($grammar =~ m/$COMMENT/gco)
        {
            _parse("a comment",0,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            next;
        }
        elsif ($grammar =~ m/$NEGLOOKAHEAD/gco)
        {
            _parse("a negative lookahead",$aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $lookahead = $lookahead ? -$lookahead : -1;
            $lookaheadspec .= $1;
            next;   # SKIP LOOKAHEAD RESET AT END OF while LOOP
        }
        elsif ($grammar =~ m/$POSLOOKAHEAD/gco)
        {
            _parse("a positive lookahead",$aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $lookahead = $lookahead ? $lookahead : 1;
            $lookaheadspec .= $1;
            next;   # SKIP LOOKAHEAD RESET AT END OF while LOOP
        }
        elsif ($grammar =~ m/(?=$ACTION)/gco
            and do { ($code) = extract_codeblock($grammar); $code })
        {
            _parse("an action", $aftererror, $line, $code);
            $item = new Parse::RecDescent::_Runtime::Action($code,$lookahead,$line);
            $prod and $prod->additem($item)
                  or  $self->_addstartcode($code);
        }
        elsif ($grammar =~ m/(?=$IMPLICITSUBRULE)/gco
            and do { ($code) = extract_codeblock($grammar,'{([',undef,'(',1);
                $code })
        {
            $code =~ s/\A\s*\(|\)\Z//g;
            _parse("an implicit subrule", $aftererror, $line,
                "( $code )");
            my $implicit = $rule->nextimplicit;
            return undef
                if !$self->_generate("$implicit : $code",$replace,1);
            my $pos = pos $grammar;
            substr($grammar,$pos,0,$implicit);
            pos $grammar = $pos;;
        }
        elsif ($grammar =~ m/$ENDDIRECTIVEMK/gco)
        {

        # EXTRACT TRAILING REPETITION SPECIFIER (IF ANY)

            my ($minrep,$maxrep) = (1,$MAXREP);
            if ($grammar =~ m/\G[(]/gc)
            {
                pos($grammar)--;

                if ($grammar =~ m/$OPTIONAL/gco)
                    { ($minrep, $maxrep) = (0,1) }
                elsif ($grammar =~ m/$ANY/gco)
                    { $minrep = 0 }
                elsif ($grammar =~ m/$EXACTLY/gco)
                    { ($minrep, $maxrep) = ($1,$1) }
                elsif ($grammar =~ m/$BETWEEN/gco)
                    { ($minrep, $maxrep) = ($1,$2) }
                elsif ($grammar =~ m/$ATLEAST/gco)
                    { $minrep = $1 }
                elsif ($grammar =~ m/$ATMOST/gco)
                    { $maxrep = $1 }
                elsif ($grammar =~ m/$MANY/gco)
                    { }
                elsif ($grammar =~ m/$BADREP/gco)
                {
                    _parse("an invalid repetition specifier", 0,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                    _error("Incorrect specification of a repeated directive",
                           $line);
                    _hint("Repeated directives cannot have
                           a maximum repetition of zero, nor can they have
                           negative components in their ranges.");
                }
            }

            $prod && $prod->enddirective($line,$minrep,$maxrep);
        }
        elsif ($grammar =~ m/\G\s*<[^m]/gc)
        {
            pos($grammar)-=2;

            if ($grammar =~ m/$OPMK/gco)
            {
                # $DB::single=1;
                _parse("a $1-associative operator directive", $aftererror, $line, "<$1op:...>");
                $prod->adddirective($1, $line,$2||'');
            }
            elsif ($grammar =~ m/$UNCOMMITMK/gco)
            {
                _parse("an uncommit marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive('$commit=0;1',
                                  $lookahead,$line,"<uncommit>");
                $prod and $prod->additem($item)
                      or  _no_rule("<uncommit>",$line);
            }
            elsif ($grammar =~ m/$QUOTELIKEMK/gco)
            {
                _parse("an perl quotelike marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                    'my ($match,@res);
                     ($match,$text,undef,@res) =
                          Text::Balanced::extract_quotelike($text,$skip);
                      $match ? \@res : undef;
                    ', $lookahead,$line,"<perl_quotelike>");
                $prod and $prod->additem($item)
                      or  _no_rule("<perl_quotelike>",$line);
            }
            elsif ($grammar =~ m/$CODEBLOCKMK/gco)
            {
                my $outer = $1||"{}";
                _parse("an perl codeblock marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                    'Text::Balanced::extract_codeblock($text,undef,$skip,\''.$outer.'\');
                    ', $lookahead,$line,"<perl_codeblock>");
                $prod and $prod->additem($item)
                      or  _no_rule("<perl_codeblock>",$line);
            }
            elsif ($grammar =~ m/$VARIABLEMK/gco)
            {
                _parse("an perl variable marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                    'Text::Balanced::extract_variable($text,$skip);
                    ', $lookahead,$line,"<perl_variable>");
                $prod and $prod->additem($item)
                      or  _no_rule("<perl_variable>",$line);
            }
            elsif ($grammar =~ m/$NOCHECKMK/gco)
            {
                _parse("a disable checking marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                if ($rule)
                {
                    _error("<nocheck> directive not at start of grammar", $line);
                    _hint("The <nocheck> directive can only
                           be specified at the start of a
                           grammar (before the first rule
                           is defined.");
                }
                else
                {
                    local $::RD_CHECK = 1;
                }
            }
            elsif ($grammar =~ m/$AUTOSTUBMK/gco)
            {
                _parse("an autostub marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $::RD_AUTOSTUB = "";
            }
            elsif ($grammar =~ m/$AUTORULEMK/gco)
            {
                _parse("an autorule marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $::RD_AUTOSTUB = $1;
            }
            elsif ($grammar =~ m/$AUTOTREEMK/gco)
            {
                my $base = defined($1) ? $1 : "";
                my $current_match = substr($grammar, $-[0], $+[0] - $-[0]);
                $base .= "::" if $base && $base !~ /::$/;
                _parse("an autotree marker", $aftererror,$line, $current_match);
                if ($rule)
                {
                    _error("<autotree> directive not at start of grammar", $line);
                    _hint("The <autotree> directive can only
                           be specified at the start of a
                           grammar (before the first rule
                           is defined.");
                }
                else
                {
                    undef $self->{_AUTOACTION};
                    $self->{_AUTOTREE}{NODE}
                        = new Parse::RecDescent::_Runtime::Action(q({bless \%item, ').$base.q('.$item[0]}),0,-1);
                    $self->{_AUTOTREE}{TERMINAL}
                        = new Parse::RecDescent::_Runtime::Action(q({bless {__VALUE__=>$item[1]}, ').$base.q('.$item[0]}),0,-1);
                }
            }

            elsif ($grammar =~ m/$REJECTMK/gco)
            {
                _parse("an reject marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::UncondReject($lookahead,$line,"<reject>");
                $prod and $prod->additem($item)
                      or  _no_rule("<reject>",$line);
            }
            elsif ($grammar =~ m/(?=$CONDREJECTMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                      $code })
            {
                _parse("a (conditional) reject marker", $aftererror,$line, $code );
                $code =~ /\A\s*<reject:(.*)>\Z/s;
                my $cond = $1;
                $item = new Parse::RecDescent::_Runtime::Directive(
                          "($1) ? undef : 1", $lookahead,$line,"<reject:$cond>");
                $prod and $prod->additem($item)
                      or  _no_rule("<reject:$cond>",$line);
            }
            elsif ($grammar =~ m/(?=$SCOREMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                      $code })
            {
                _parse("a score marker", $aftererror,$line, $code );
                $code =~ /\A\s*<score:(.*)>\Z/s;
                $prod and $prod->addscore($1, $lookahead, $line)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/(?=$AUTOSCOREMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("an autoscore specifier", $aftererror,$line,$code);
                $code =~ /\A\s*<autoscore:(.*)>\Z/s;

                $rule and $rule->addautoscore($1,$self)
                      or  _no_rule($code,$line);

                $item = new Parse::RecDescent::_Runtime::UncondReject($lookahead,$line,$code);
                $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/$RESYNCMK/gco)
            {
                _parse("a resync to newline marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive(
                          'if ($text =~ s/(\A[^\n]*\n)//) { $return = 0; $1; } else { undef }',
                          $lookahead,$line,"<resync>");
                $prod and $prod->additem($item)
                      or  _no_rule("<resync>",$line);
            }
            elsif ($grammar =~ m/(?=$RESYNCPATMK)/gco
                and do { ($code) = extract_bracketed($grammar,'<');
                      $code })
            {
                _parse("a resync with pattern marker", $aftererror,$line, $code );
                $code =~ /\A\s*<resync:(.*)>\Z/s;
                $item = new Parse::RecDescent::_Runtime::Directive(
                          'if ($text =~ s/(\A'.$1.')//) { $return = 0; $1; } else { undef }',
                          $lookahead,$line,$code);
                $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/(?=$SKIPMK)/gco
                and do { ($code) = extract_codeblock($grammar,'<');
                      $code })
            {
                _parse("a skip marker", $aftererror,$line, $code );
                $code =~ /\A\s*<skip:(.*)>\Z/s;
                if ($rule) {
                    $item = new Parse::RecDescent::_Runtime::Directive(
                        'my $oldskip = $skip; $skip='.$1.'; $oldskip',
                        $lookahead,$line,$code);
                    $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
                } else {
                    #global <skip> directive
                    $self->{skip} = $1;
                }
            }
            elsif ($grammar =~ m/(?=$RULEVARPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("a rule variable specifier", $aftererror,$line,$code);
                $code =~ /\A\s*<rulevar:(.*)>\Z/s;

                $rule and $rule->addvar($1,$self)
                      or  _no_rule($code,$line);

                $item = new Parse::RecDescent::_Runtime::UncondReject($lookahead,$line,$code);
                $prod and $prod->additem($item)
                      or  _no_rule($code,$line);
            }
            elsif ($grammar =~ m/(?=$AUTOACTIONPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("an autoaction specifier", $aftererror,$line,$code);
                $code =~ s/\A\s*<autoaction:(.*)>\Z/$1/s;
                if ($code =~ /\A\s*[^{]|[^}]\s*\Z/) {
                    $code = "{ $code }"
                }
        $self->{_check}{itempos} =
            $code =~ /\@itempos\b|\$itempos\s*\[/;
        $self->{_AUTOACTION}
            = new Parse::RecDescent::_Runtime::Action($code,0,-$line)
            }
            elsif ($grammar =~ m/(?=$DEFERPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("a deferred action specifier", $aftererror,$line,$code);
                $code =~ s/\A\s*<defer:(.*)>\Z/$1/s;
                if ($code =~ /\A\s*[^{]|[^}]\s*\Z/)
                {
                    $code = "{ $code }"
                }

                $item = new Parse::RecDescent::_Runtime::Directive(
                          "push \@{\$thisparser->{deferred}}, sub $code;",
                          $lookahead,$line,"<defer:$code>");
                $prod and $prod->additem($item)
                      or  _no_rule("<defer:$code>",$line);

                $self->{deferrable} = 1;
            }
            elsif ($grammar =~ m/(?=$TOKENPATMK)/gco
                and do { ($code) = extract_codeblock($grammar,'{',undef,'<');
                     $code;
                       } )
            {
                _parse("a token constructor", $aftererror,$line,$code);
                $code =~ s/\A\s*<token:(.*)>\Z/$1/s;

                my $types = eval 'no strict; local $SIG{__WARN__} = sub {0}; my @arr=('.$code.'); @arr' || ();
                if (!$types)
                {
                    _error("Incorrect token specification: \"$@\"", $line);
                    _hint("The <token:...> directive requires a list
                           of one or more strings representing possible
                           types of the specified token. For example:
                           <token:NOUN,VERB>");
                }
                else
                {
                    $item = new Parse::RecDescent::_Runtime::Directive(
                              'no strict;
                               $return = { text => $item[-1] };
                               @{$return->{type}}{'.$code.'} = (1..'.$types.');',
                              $lookahead,$line,"<token:$code>");
                    $prod and $prod->additem($item)
                          or  _no_rule("<token:$code>",$line);
                }
            }
            elsif ($grammar =~ m/$COMMITMK/gco)
            {
                _parse("an commit marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Directive('$commit = 1',
                                  $lookahead,$line,"<commit>");
                $prod and $prod->additem($item)
                      or  _no_rule("<commit>",$line);
            }
            elsif ($grammar =~ m/$NOCHECKMK/gco) {
                _parse("an hint request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_CHECK = 0;
        }
            elsif ($grammar =~ m/$HINTMK/gco) {
                _parse("an hint request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_HINT = $self->{__HINT__} = 1;
        }
            elsif ($grammar =~ m/$WARNMK/gco) {
                _parse("an warning request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_WARN = $self->{__WARN__} = $1 ? $2+0 : 1;
        }
            elsif ($grammar =~ m/$TRACEBUILDMK/gco) {
                _parse("an grammar build trace request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $::RD_TRACE = $1 ? $2+0 : 1;
        }
            elsif ($grammar =~ m/$TRACEPARSEMK/gco) {
                _parse("an parse trace request", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
        $self->{__TRACE__} = $1 ? $2+0 : 1;
        }
            elsif ($grammar =~ m/$AUTOERRORMK/gco)
            {
                $commitonly = $1;
                _parse("an error marker", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
                $item = new Parse::RecDescent::_Runtime::Error('',$lookahead,$1,$line);
                $prod and $prod->additem($item)
                      or  _no_rule("<error>",$line);
                $aftererror = !$commitonly;
            }
            elsif ($grammar =~ m/(?=$MSGERRORMK)/gco
                and do { $commitonly = $1;
                     ($code) = extract_bracketed($grammar,'<');
                    $code })
            {
                _parse("an error marker", $aftererror,$line,$code);
                $code =~ /\A\s*<error\??:(.*)>\Z/s;
                $item = new Parse::RecDescent::_Runtime::Error($1,$lookahead,$commitonly,$line);
                $prod and $prod->additem($item)
                      or  _no_rule("$code",$line);
                $aftererror = !$commitonly;
            }
            elsif (do { $commitonly = $1;
                     ($code) = extract_bracketed($grammar,'<');
                    $code })
            {
                if ($code =~ /^<[A-Z_]+>$/)
                {
                    _error("Token items are not yet
                    supported: \"$code\"",
                           $line);
                    _hint("Items like $code that consist of angle
                    brackets enclosing a sequence of
                    uppercase characters will eventually
                    be used to specify pre-lexed tokens
                    in a grammar. That functionality is not
                    yet implemented. Or did you misspell
                    \"$code\"?");
                }
                else
                {
                    _error("Untranslatable item encountered: \"$code\"",
                           $line);
                    _hint("Did you misspell \"$code\"
                           or forget to comment it out?");
                }
            }
        }
        elsif ($grammar =~ m/$RULE/gco)
        {
            _parseunneg("a rule declaration", 0,
                    $lookahead,$line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;
            my $rulename = $1;
            if ($rulename =~ /Replace|Extend|Precompile|Save/ )
            {
                _warn(2,"Rule \"$rulename\" hidden by method
                       Parse::RecDescent::_Runtime::$rulename",$line)
                and
                _hint("The rule named \"$rulename\" cannot be directly
                       called through the Parse::RecDescent::_Runtime object
                       for this grammar (although it may still
                       be used as a subrule of other rules).
                       It can't be directly called because
                       Parse::RecDescent::_Runtime::$rulename is already defined (it
                       is the standard method of all
                       parsers).");
            }
            $rule = new Parse::RecDescent::_Runtime::Rule($rulename,$self,$line,$replace);
            $prod->check_pending($line) if $prod;
            $prod = $rule->addprod( new Parse::RecDescent::_Runtime::Production );
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$UNCOMMITPROD/gco)
        {
            pos($grammar)-=9;
            _parseunneg("a new (uncommitted) production",
                    0, $lookahead, $line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;

            $prod->check_pending($line) if $prod;
            $prod = new Parse::RecDescent::_Runtime::Production($line,1);
            $rule and $rule->addprod($prod)
                  or  _no_rule("<uncommit>",$line);
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$ERRORPROD/gco)
        {
            pos($grammar)-=6;
            _parseunneg("a new (error) production", $aftererror,
                    $lookahead,$line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;
            $prod->check_pending($line) if $prod;
            $prod = new Parse::RecDescent::_Runtime::Production($line,0,1);
            $rule and $rule->addprod($prod)
                  or  _no_rule("<error>",$line);
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$PROD/gco)
        {
            _parseunneg("a new production", 0,
                    $lookahead,$line, substr($grammar, $-[0], $+[0] - $-[0]) ) or next;
            $rule
              and (!$prod || $prod->check_pending($line))
              and $prod = $rule->addprod(new Parse::RecDescent::_Runtime::Production($line))
            or  _no_rule("production",$line);
            $aftererror = 0;
        }
        elsif ($grammar =~ m/$LITERAL/gco)
        {
            my $literal = $1;
            ($code = $literal) =~ s/\\\\/\\/g;
            _parse("a literal terminal", $aftererror,$line,$literal);
            $item = new Parse::RecDescent::_Runtime::Literal($code,$lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("literal terminal",$line,"'$literal'");
        }
        elsif ($grammar =~ m/$INTERPLIT/gco)
        {
            _parse("an interpolated literal terminal", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $item = new Parse::RecDescent::_Runtime::InterpLit($1,$lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("interpolated literal terminal",$line,"'$1'");
        }
        elsif ($grammar =~ m/$TOKEN/gco)
        {
            _parse("a /../ pattern terminal", $aftererror,$line, substr($grammar, $-[0], $+[0] - $-[0]) );
            $item = new Parse::RecDescent::_Runtime::Token($1,'/',$3?$3:'',$lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("pattern terminal",$line,"/$1/");
        }
        elsif ($grammar =~ m/(?=$MTOKEN)/gco
            and do { ($code, undef, @components)
                    = extract_quotelike($grammar);
                 $code }
              )

        {
            _parse("an m/../ pattern terminal", $aftererror,$line,$code);
            $item = new Parse::RecDescent::_Runtime::Token(@components[3,2,8],
                                 $lookahead,$line);
            $prod and $prod->additem($item)
                  or  _no_rule("pattern terminal",$line,$code);
        }
        elsif ($grammar =~ m/(?=$MATCHRULE)/gco
                and do { ($code) = extract_bracketed($grammar,'<');
                     $code
                       }
               or $grammar =~ m/$SUBRULE/gco
                and $code = $1)
        {
            my $name = $code;
            my $matchrule = 0;
            if (substr($name,0,1) eq '<')
            {
                $name =~ s/$MATCHRULE\s*//;
                $name =~ s/\s*>\Z//;
                $matchrule = 1;
            }

        # EXTRACT TRAILING ARG LIST (IF ANY)

            my ($argcode) = extract_codeblock($grammar, "[]",'') || '';

        # EXTRACT TRAILING REPETITION SPECIFIER (IF ANY)

            if ($grammar =~ m/\G[(]/gc)
            {
                pos($grammar)--;

                if ($grammar =~ m/$OPTIONAL/gco)
                {
                    _parse("an zero-or-one subrule match", $aftererror,$line,"$code$argcode($1)");
                    $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,0,1,
                                       $lookahead,$line,
                                       $self,
                                       $matchrule,
                                       $argcode);
                    $prod and $prod->additem($item)
                          or  _no_rule("repetition",$line,"$code$argcode($1)");

                    !$matchrule and $rule and $rule->addcall($name);
                }
                elsif ($grammar =~ m/$ANY/gco)
                {
                    _parse("a zero-or-more subrule match", $aftererror,$line,"$code$argcode($1)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name(s?)': $name $2 $name>(s?) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,0,$MAXREP,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1)");

                        !$matchrule and $rule and $rule->addcall($name);

                        _check_insatiable($name,$1,$grammar,$line) if $::RD_CHECK;
                    }
                }
                elsif ($grammar =~ m/$MANY/gco)
                {
                    _parse("a one-or-more subrule match", $aftererror,$line,"$code$argcode($1)");
                    if ($2)
                    {
                        # $DB::single=1;
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name(s)': $name $2 $name> ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,1,$MAXREP,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);

                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1)");

                        !$matchrule and $rule and $rule->addcall($name);

                        _check_insatiable($name,$1,$grammar,$line) if $::RD_CHECK;
                    }
                }
                elsif ($grammar =~ m/$EXACTLY/gco)
                {
                    _parse("an exactly-$1-times subrule match", $aftererror,$line,"$code$argcode($1)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name($1)': $name $2 $name>($1) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,$1,$1,$1,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1)");

                        !$matchrule and $rule and $rule->addcall($name);
                    }
                }
                elsif ($grammar =~ m/$BETWEEN/gco)
                {
                    _parse("a $1-to-$2 subrule match", $aftererror,$line,"$code$argcode($1..$2)");
                    if ($3)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name($1..$2)': $name $3 $name>($1..$2) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,"$1..$2",$1,$2,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1..$2)");

                        !$matchrule and $rule and $rule->addcall($name);
                    }
                }
                elsif ($grammar =~ m/$ATLEAST/gco)
                {
                    _parse("a $1-or-more subrule match", $aftererror,$line,"$code$argcode($1..)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name($1..)': $name $2 $name>($1..) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,"$1..",$1,$MAXREP,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode($1..)");

                        !$matchrule and $rule and $rule->addcall($name);
                        _check_insatiable($name,"$1..",$grammar,$line) if $::RD_CHECK;
                    }
                }
                elsif ($grammar =~ m/$ATMOST/gco)
                {
                    _parse("a one-to-$1 subrule match", $aftererror,$line,"$code$argcode(..$1)");
                    if ($2)
                    {
                        my $pos = pos $grammar;
                        substr($grammar,$pos,0,
                               "<leftop='$name(..$1)': $name $2 $name>(..$1) ");

                        pos $grammar = $pos;
                    }
                    else
                    {
                        $item = new Parse::RecDescent::_Runtime::Repetition($name,"..$1",1,$1,
                                           $lookahead,$line,
                                           $self,
                                           $matchrule,
                                           $argcode);
                        $prod and $prod->additem($item)
                              or  _no_rule("repetition",$line,"$code$argcode(..$1)");

                        !$matchrule and $rule and $rule->addcall($name);
                    }
                }
                elsif ($grammar =~ m/$BADREP/gco)
                {
                    my $current_match = substr($grammar, $-[0], $+[0] - $-[0]);
                    _parse("an subrule match with invalid repetition specifier", 0,$line, $current_match);
                    _error("Incorrect specification of a repeated subrule",
                           $line);
                    _hint("Repeated subrules like \"$code$argcode$current_match\" cannot have
                           a maximum repetition of zero, nor can they have
                           negative components in their ranges.");
                }
            }
            else
            {
                _parse("a subrule match", $aftererror,$line,$code);
                my $desc;
                if ($name=~/\A_alternation_\d+_of_production_\d+_of_rule/)
                    { $desc = $self->{"rules"}{$name}->expected }
                $item = new Parse::RecDescent::_Runtime::Subrule($name,
                                       $lookahead,
                                       $line,
                                       $desc,
                                       $matchrule,
                                       $argcode);

                $prod and $prod->additem($item)
                      or  _no_rule("(sub)rule",$line,$name);

                !$matchrule and $rule and $rule->addcall($name);
            }
        }
        elsif ($grammar =~ m/$LONECOLON/gco   )
        {
            _error("Unexpected colon encountered", $line);
            _hint("Did you mean \"|\" (to start a new production)?
                   Or perhaps you forgot that the colon
                   in a rule definition must be
                   on the same line as the rule name?");
        }
        elsif ($grammar =~ m/$ACTION/gco   ) # BAD ACTION, ALREADY FAILED
        {
            _error("Malformed action encountered",
                   $line);
            _hint("Did you forget the closing curly bracket
                   or is there a syntax error in the action?");
        }
        elsif ($grammar =~ m/$OTHER/gco   )
        {
            _error("Untranslatable item encountered: \"$1\"",
                   $line);
            _hint("Did you misspell \"$1\"
                   or forget to comment it out?");
        }

        if ($lookaheadspec =~ tr /././ > 3)
        {
            $lookaheadspec =~ s/\A\s+//;
            $lookahead = $lookahead<0
                    ? 'a negative lookahead ("...!")'
                    : 'a positive lookahead ("...")' ;
            _warn(1,"Found two or more lookahead specifiers in a
                   row.",$line)
            and
            _hint("Multiple positive and/or negative lookaheads
                   are simply multiplied together to produce a
                   single positive or negative lookahead
                   specification. In this case the sequence
                   \"$lookaheadspec\" was reduced to $lookahead.
                   Was this your intention?");
        }
        $lookahead = 0;
        $lookaheadspec = "";

        $grammar =~ m/\G\s+/gc;
    }

    if ($must_pop_lines) {
        pop @lines;
    }

    unless ($ERRORS or $isimplicit or !$::RD_CHECK)
    {
        $self->_check_grammar();
    }

    unless ($ERRORS or $isimplicit or $Parse::RecDescent::_Runtime::compiling)
    {
        my $code = $self->_code();
        if (defined $::RD_TRACE)
        {
            my $mode = ($nextnamespace eq "namespace000002") ? '>' : '>>';
            print STDERR "printing code (", length($code),") to RD_TRACE\n";
            local *TRACE_FILE;
            open TRACE_FILE, $mode, "RD_TRACE"
            and print TRACE_FILE "my \$ERRORS;\n$code"
            and close TRACE_FILE;
        }

        unless ( eval "$code 1" )
        {
            _error("Internal error in generated parser code!");
            $@ =~ s/at grammar/in grammar at/;
            _hint($@);
        }
    }

    if ($ERRORS and !_verbosity("HINT"))
    {
        local $::RD_HINT = defined $::RD_HINT ? $::RD_HINT : 1;
        _hint('Set $::RD_HINT (or -RD_HINT if you\'re using "perl -s")
               for hints on fixing these problems.  Use $::RD_HINT = 0
               to disable this message.');
    }
    if ($ERRORS) { $ERRORS=0; return }
    return $self;
}


sub _addstartcode($$)
{
    my ($self, $code) = @_;
    $code =~ s/\A\s*\{(.*)\}\Z/$1/s;

    $self->{"startcode"} .= "$code;\n";
}

# CHECK FOR GRAMMAR PROBLEMS....

sub _check_insatiable($$$$)
{
    my ($subrule,$repspec,$grammar,$line) = @_;
    pos($grammar)=pos($_[2]);
    return if $grammar =~ m/$OPTIONAL/gco || $grammar =~ m/$ANY/gco;
    my $min = 1;
    if ( $grammar =~ m/$MANY/gco
      || $grammar =~ m/$EXACTLY/gco
      || $grammar =~ m/$ATMOST/gco
      || $grammar =~ m/$BETWEEN/gco && do { $min=$2; 1 }
      || $grammar =~ m/$ATLEAST/gco && do { $min=$2; 1 }
      || $grammar =~ m/$SUBRULE(?!\s*:)/gco
       )
    {
        return unless $1 eq $subrule && $min > 0;
        my $current_match = substr($grammar, $-[0], $+[0] - $-[0]);
        _warn(3,"Subrule sequence \"$subrule($repspec) $current_match\" will
               (almost certainly) fail.",$line)
        and
        _hint("Unless subrule \"$subrule\" performs some cunning
               lookahead, the repetition \"$subrule($repspec)\" will
               insatiably consume as many matches of \"$subrule\" as it
               can, leaving none to match the \"$current_match\" that follows.");
    }
}

sub _check_grammar ($)
{
    my $self = shift;
    my $rules = $self->{"rules"};
    my $rule;
    foreach $rule ( values %$rules )
    {
        next if ! $rule->{"changed"};

    # CHECK FOR UNDEFINED RULES

        my $call;
        foreach $call ( @{$rule->{"calls"}} )
        {
            if (!defined ${$rules}{$call}
              &&!defined &{"Parse::RecDescent::_Runtime::$call"})
            {
                if (!defined $::RD_AUTOSTUB)
                {
                    _warn(3,"Undefined (sub)rule \"$call\"
                          used in a production.")
                    and
                    _hint("Will you be providing this rule
                           later, or did you perhaps
                           misspell \"$call\"? Otherwise
                           it will be treated as an
                           immediate <reject>.");
                    eval "sub $self->{namespace}::$call {undef}";
                }
                else    # EXPERIMENTAL
                {
                    my $rule = qq{'$call'};
                    if ($::RD_AUTOSTUB and $::RD_AUTOSTUB ne "1") {
                        $rule = $::RD_AUTOSTUB;
                    }
                    _warn(1,"Autogenerating rule: $call")
                    and
                    _hint("A call was made to a subrule
                           named \"$call\", but no such
                           rule was specified. However,
                           since \$::RD_AUTOSTUB
                           was defined, a rule stub
                           ($call : $rule) was
                           automatically created.");

                    $self->_generate("$call: $rule",0,1);
                }
            }
        }

    # CHECK FOR LEFT RECURSION

        if ($rule->isleftrec($rules))
        {
            _error("Rule \"$rule->{name}\" is left-recursive.");
            _hint("Redesign the grammar so it's not left-recursive.
                   That will probably mean you need to re-implement
                   repetitions using the '(s)' notation.
                   For example: \"$rule->{name}(s)\".");
            next;
        }

    # CHECK FOR PRODUCTIONS FOLLOWING EMPTY PRODUCTIONS
      {
          my $hasempty;
          my $prod;
          foreach $prod ( @{$rule->{"prods"}} ) {
              if ($hasempty) {
                  _error("Production " . $prod->describe . " for \"$rule->{name}\"
                         will never be reached (preceding empty production will
                         always match first).");
                  _hint("Reorder the grammar so that the empty production
                         is last in the list or productions.");
                  last;
              }
              $hasempty ||= $prod->isempty();
          }
      }
    }
}

# GENERATE ACTUAL PARSER CODE

sub _code($)
{
    my $self = shift;
    my $initial_skip = defined($self->{skip}) ? $self->{skip} : $skip;

    my $code = qq!
package $self->{namespace};
use strict;
use vars qw(\$skip \$AUTOLOAD $self->{localvars} );
\@$self->{namespace}\::ISA = ();
\$skip = '$initial_skip';
$self->{startcode}

{
local \$SIG{__WARN__} = sub {0};
# PRETEND TO BE IN Parse::RecDescent::_Runtime NAMESPACE
*$self->{namespace}::AUTOLOAD   = sub
{
    no strict 'refs';
!
# This generated code uses ${"AUTOLOAD"} rather than $AUTOLOAD in
# order to avoid the circular reference documented here:
#    https://rt.perl.org/rt3/Public/Bug/Display.html?id=110248
# As a result of the investigation of
#    https://rt.cpan.org/Ticket/Display.html?id=53710
. qq!
    \${"AUTOLOAD"} =~ s/^$self->{namespace}/Parse::RecDescent::_Runtime/;
    goto &{\${"AUTOLOAD"}};
}
}

!;
    $code .= "push \@$self->{namespace}\::ISA, 'Parse::RecDescent::_Runtime';";
    $self->{"startcode"} = '';

    my $rule;
    foreach $rule ( values %{$self->{"rules"}} )
    {
        if ($rule->{"changed"})
        {
            $code .= $rule->code($self->{"namespace"},$self);
            $rule->{"changed"} = 0;
        }
    }

    return $code;
}


# EXECUTING A PARSE....

sub AUTOLOAD    # ($parser, $text; $linenum, @args)
{
    croak "Could not find method: $AUTOLOAD\n" unless ref $_[0];
    my $class = ref($_[0]) || $_[0];
    my $text = ref($_[1]) eq 'SCALAR' ? ${$_[1]} : "$_[1]";
    $_[0]->{lastlinenum} = _linecount($text);
    $_[0]->{lastlinenum} += ($_[2]||0) if @_ > 2;
    $_[0]->{offsetlinenum} = $_[0]->{lastlinenum};
    $_[0]->{fulltext} = $text;
    $_[0]->{fulltextlen} = length $text;
    $_[0]->{linecounter_cache} = {};
    $_[0]->{deferred} = [];
    $_[0]->{errors} = [];
    my @args = @_[3..$#_];
    my $args = sub { [ @args ] };

    $AUTOLOAD =~ s/$class/$_[0]->{namespace}/;
    no strict "refs";

    local $::RD_WARN  = $::RD_WARN  || $_[0]->{__WARN__};
    local $::RD_HINT  = $::RD_HINT  || $_[0]->{__HINT__};
    local $::RD_TRACE = $::RD_TRACE || $_[0]->{__TRACE__};

    croak "Unknown starting rule ($AUTOLOAD) called\n"
        unless defined &$AUTOLOAD;
    my $retval = &{$AUTOLOAD}(
        $_[0], # $parser
        $text, # $text
        undef, # $repeating
        undef, # $_noactions
        $args, # \@args
        undef, # $_itempos
    );


    if (defined $retval)
    {
        foreach ( @{$_[0]->{deferred}} ) { &$_; }
    }
    else
    {
        foreach ( @{$_[0]->{errors}} ) { _error(@$_); }
    }

    if (ref $_[1] eq 'SCALAR') { ${$_[1]} = $text }

    $ERRORS = 0;
    return $retval;
}

sub _parserepeat($$$$$$$$$)    # RETURNS A REF TO AN ARRAY OF MATCHES
{
    my ($parser, $text, $prod, $min, $max, $_noactions, $expectation, $argcode, $_itempos) = @_;
    my @tokens = ();

    my $itemposfirst;
    my $reps;
    for ($reps=0; $reps<$max;)
    {
        $expectation->at($text);
        my $_savetext = $text;
        my $prevtextlen = length $text;
        my $_tok;
        if (! defined ($_tok = &$prod($parser,$text,1,$_noactions,$argcode,$_itempos)))
        {
            $text = $_savetext;
            last;
        }

        if (defined($_itempos) and !defined($itemposfirst))
        {
            $itemposfirst = Parse::RecDescent::_Runtime::Production::_duplicate_itempos($_itempos);
        }

        push @tokens, $_tok if defined $_tok;
        last if ++$reps >= $min and $prevtextlen == length $text;
    }

    do { $expectation->failed(); return undef} if $reps<$min;

    if (defined $itemposfirst)
    {
        Parse::RecDescent::_Runtime::Production::_update_itempos($_itempos, $itemposfirst, undef, [qw(from)]);
    }

    $_[1] = $text;
    return [@tokens];
}

sub set_autoflush {
    my $orig_selected = select $_[0];
    $| = 1;
    select $orig_selected;
    return;
}

# ERROR REPORTING....

sub _write_ERROR {
    my ($errorprefix, $errortext) = @_;
    return if $errortext !~ /\S/;
    $errorprefix =~ s/\s+\Z//;
    local $^A = q{};

    formline(<<'END_FORMAT', $errorprefix, $errortext);
@>>>>>>>>>>>>>>>>>>>>: ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
END_FORMAT
    formline(<<'END_FORMAT', $errortext);
~~                     ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
END_FORMAT
    print {*STDERR} $^A;
}

# TRACING

my $TRACE_FORMAT = <<'END_FORMAT';
@>|@|||||||||@^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<|
  | ~~       |^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<|
END_FORMAT

my $TRACECONTEXT_FORMAT = <<'END_FORMAT';
@>|@|||||||||@                                      |^<<<<<<<<<<<<<<<<<<<<<<<<<<<
  | ~~       |                                      |^<<<<<<<<<<<<<<<<<<<<<<<<<<<
END_FORMAT

sub _write_TRACE {
    my ($tracelevel, $tracerulename, $tracemsg) = @_;
    return if $tracemsg !~ /\S/;
    $tracemsg =~ s/\s*\Z//;
    local $^A = q{};
    my $bar = '|';
    formline($TRACE_FORMAT, $tracelevel, $tracerulename, $bar, $tracemsg, $tracemsg);
    print {*STDERR} $^A;
}

sub _write_TRACECONTEXT {
    my ($tracelevel, $tracerulename, $tracecontext) = @_;
    return if $tracecontext !~ /\S/;
    $tracecontext =~ s/\s*\Z//;
    local $^A = q{};
    my $bar = '|';
    formline($TRACECONTEXT_FORMAT, $tracelevel, $tracerulename, $bar, $tracecontext, $tracecontext);
    print {*STDERR} $^A;
}

sub _verbosity($)
{
       defined $::RD_TRACE
    or defined $::RD_HINT    and  $::RD_HINT   and $_[0] =~ /ERRORS|WARN|HINT/
    or defined $::RD_WARN    and  $::RD_WARN   and $_[0] =~ /ERRORS|WARN/
    or defined $::RD_ERRORS  and  $::RD_ERRORS and $_[0] =~ /ERRORS/
}

sub _error($;$)
{
    $ERRORS++;
    return 0 if ! _verbosity("ERRORS");
    my $errortext   = $_[0];
    my $errorprefix = "ERROR" .  ($_[1] ? " (line $_[1])" : "");
    $errortext =~ s/\s+/ /g;
    print {*STDERR} "\n" if _verbosity("WARN");
    _write_ERROR($errorprefix, $errortext);
    return 1;
}

sub _warn($$;$)
{
    return 0 unless _verbosity("WARN") && ($::RD_HINT || $_[0] >= ($::RD_WARN||1));
    my $errortext   = $_[1];
    my $errorprefix = "Warning" .  ($_[2] ? " (line $_[2])" : "");
    print {*STDERR} "\n" if _verbosity("HINT");
    $errortext =~ s/\s+/ /g;
    _write_ERROR($errorprefix, $errortext);
    return 1;
}

sub _hint($)
{
    return 0 unless $::RD_HINT;
    my $errortext = $_[0];
    my $errorprefix = "Hint" .  ($_[1] ? " (line $_[1])" : "");
    $errortext =~ s/\s+/ /g;
    _write_ERROR($errorprefix, $errortext);
    return 1;
}

sub _tracemax($)
{
    if (defined $::RD_TRACE
        && $::RD_TRACE =~ /\d+/
        && $::RD_TRACE>1
        && $::RD_TRACE+10<length($_[0]))
    {
        my $count = length($_[0]) - $::RD_TRACE;
        return substr($_[0],0,$::RD_TRACE/2)
            . "...<$count>..."
            . substr($_[0],-$::RD_TRACE/2);
    }
    else
    {
        return substr($_[0],0,500);
    }
}

sub _tracefirst($)
{
    if (defined $::RD_TRACE
        && $::RD_TRACE =~ /\d+/
        && $::RD_TRACE>1
        && $::RD_TRACE+10<length($_[0]))
    {
        my $count = length($_[0]) - $::RD_TRACE;
        return substr($_[0],0,$::RD_TRACE) . "...<+$count>";
    }
    else
    {
        return substr($_[0],0,500);
    }
}

my $lastcontext = '';
my $lastrulename = '';
my $lastlevel = '';

sub _trace($;$$$)
{
    my $tracemsg      = $_[0];
    my $tracecontext  = $_[1]||$lastcontext;
    my $tracerulename = $_[2]||$lastrulename;
    my $tracelevel    = $_[3]||$lastlevel;
    if ($tracerulename) { $lastrulename = $tracerulename }
    if ($tracelevel)    { $lastlevel = $tracelevel }

    $tracecontext =~ s/\n/\\n/g;
    $tracecontext =~ s/\s+/ /g;
    $tracerulename = qq{$tracerulename};
    _write_TRACE($tracelevel, $tracerulename, $tracemsg);
    if ($tracecontext ne $lastcontext)
    {
        if ($tracecontext)
        {
            $lastcontext = _tracefirst($tracecontext);
            $tracecontext = qq{"$tracecontext"};
        }
        else
        {
            $tracecontext = qq{<NO TEXT LEFT>};
        }
        _write_TRACECONTEXT($tracelevel, $tracerulename, $tracecontext);
    }
}

sub _matchtracemessage
{
    my ($self, $reject) = @_;

    my $prefix = '';
    my $postfix = '';
    my $matched = not $reject;
    my @t = ("Matched", "Didn't match");
    if (exists $self->{lookahead} and $self->{lookahead})
    {
        $postfix = $reject ? "(reject)" : "(keep)";
        $prefix = "...";
        if ($self->{lookahead} < 0)
        {
            $prefix .= '!';
            $matched = not $matched;
        }
    }
    $prefix . ($matched ? $t[0] : $t[1]) . $postfix;
}

sub _parseunneg($$$$$)
{
    _parse($_[0],$_[1],$_[3],$_[4]);
    if ($_[2]<0)
    {
        _error("Can't negate \"$_[4]\".",$_[3]);
        _hint("You can't negate $_[0]. Remove the \"...!\" before
               \"$_[4]\".");
        return 0;
    }
    return 1;
}

sub _parse($$$$)
{
    my $what = $_[3];
       $what =~ s/^\s+//;
    if ($_[1])
    {
        _warn(3,"Found $_[0] ($what) after an unconditional <error>",$_[2])
        and
        _hint("An unconditional <error> always causes the
               production containing it to immediately fail.
               \u$_[0] that follows an <error>
               will never be reached.  Did you mean to use
               <error?> instead?");
    }

    return if ! _verbosity("TRACE");
    my $errortext = "Treating \"$what\" as $_[0]";
    my $errorprefix = "Parse::RecDescent::_Runtime";
    $errortext =~ s/\s+/ /g;
    _write_ERROR($errorprefix, $errortext);
}

sub _linecount($) {
    scalar substr($_[0], pos $_[0]||0) =~ tr/\n//
}


package main;

use vars qw ( $RD_ERRORS $RD_WARN $RD_HINT $RD_TRACE $RD_CHECK );
$::RD_CHECK = 1;
$::RD_ERRORS = 1;
$::RD_WARN = 3;

1;

}
package Tree_Query::Grammar;
{ my $ERRORS;


package Parse::RecDescent::_Runtime::Tree_Query::Grammar;
use strict;
use vars qw($skip $AUTOLOAD  );
@Parse::RecDescent::_Runtime::Tree_Query::Grammar::ISA = ();
$skip = '\s*';

      $skip = '\s*(?:[#][^\n]*\s*)*';

      {
	package Tree_Query::ParserError;
	use UNIVERSAL::DOES;
	use overload '""' => \&as_text;
	sub new {
	  my $class=shift;
	  my $array_ref = shift;
	  bless $array_ref, $class;
	}
	sub as_text {
	  my $self = shift;
	  return 'Parse error at line '.$self->line.': '.$self->message;
	}
	sub message { return $_[0]->[0] }
	sub line    { return $_[0]->[1] }
      }

      sub report_error {
        my ($thisparser,$rule,$text)=@_;
        die
        Tree_Query::ParserError->new($thisparser->{errors}[0]||['Syntax error in '.$rule.' near '.substr($$text,0,20).'...',1]);
      }
      sub new_node {
        my ($hash,$children)=@_;
        my $new = Treex::PML::Factory->createNode($hash,1);
        if ($children) {
          if (ref($children) eq 'ARRAY') {
             for (reverse @$children) {
                if (UNIVERSAL::DOES::does($_,'Treex::PML::Node')) {
                  $_->paste_on($new)
                } else {
                   warn "new_node: child of $hash->{'#name'} is not a node:\n".
                        Data::Dumper::Dumper([$_]);
                }
             }
          } else {
            warn "new_node: 2nd argument of constructor to $hash->{'#name'} is not an ARRAYref:\n".
            Data::Dumper::Dumper([$children]);
          }
        }
        return $new;
      }
      sub new_struct {
        return Treex::PML::Factory->createStructure($_[0],1);
      }
      sub new_relation {
        my ($type,$opts)=@_;
	$opts||={};
	$opts->{label}||=$type;
	return Treex::PML::Factory->createSeq([
          Treex::PML::Seq::Element->new( 
            $type => Treex::PML::Factory->createContainer(undef,$opts)
          )
        ]);
      }
      sub _error ($;$){
	# Parse::RecDescent::_Runtime::_old_error(@_);
        warn("Syntax error at line $_[1]: $_[0]\n");
	return 1;
      }
      sub parse_or_die {
        my $self=shift;
        my $rule=shift;
        #local *Parse::RecDescent::_Runtime::_old_error = \&Parse::RecDescent::_Runtime::_error;
        #local *Parse::RecDescent::_Runtime::_error = \&_error;
        #local *Parse::RecDescent::_Runtime::ERROR;
        $self->$rule(@_);
      } 
      BEGIN {
      no strict qw(refs);
      foreach my $r (qw(query selectors filters node test expression flat_expression
              conditions column_expression)) {
	my $rule=$r;
	*{__PACKAGE__."::parse_".$rule} = sub {
            my $self=shift;
            my $call = "pmltq_$rule";
	    my $old_error = \*Parse::RecDescent::_Runtime::_error;
	    my @captured_errors;
            *Parse::RecDescent::_Runtime::_error = sub ($;$) { push @captured_errors, [@_] };
	    my $ret = $self->$call(@_);
            *Parse::RecDescent::_Runtime::_error = $old_error;
            if (@captured_errors) {
	      my %seen;
              croak(join("\n",
	        grep !$seen{$_}++,
	        map {
		 my $e="Syntax error at line $_->[1]: $_->[0]";
		 $e=~s/\r//g;
		 $e;
		} @captured_errors)	);
	    } elsif (!defined($ret)) {
              die "Unknown syntax error while parsing $rule\n";
            }
            return $ret;
          };
      }
      }
    ;


{
local $SIG{__WARN__} = sub {0};
# PRETEND TO BE IN Parse::RecDescent::_Runtime NAMESPACE
*Parse::RecDescent::_Runtime::Tree_Query::Grammar::AUTOLOAD   = sub
{
    no strict 'refs';

    ${"AUTOLOAD"} =~ s/^Parse::RecDescent::_Runtime::Tree_Query::Grammar/Parse::RecDescent::_Runtime/;
    goto &{${"AUTOLOAD"}};
}
}

push @Parse::RecDescent::_Runtime::Tree_Query::Grammar::ISA, 'Parse::RecDescent::_Runtime';
# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_column_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_column_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{column_expression});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [column_expression end_of_string]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_column_expression});
        %item = (__RULE__ => q{pmltq_column_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_column_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_string})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_string($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_string]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_column_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_string]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_string}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [column_expression end_of_string]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_column_expression});
        %item = (__RULE__ => q{pmltq_column_expression});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_column_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_column_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_column_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_member_or_subq
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_member_or_subq"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_member_or_subq]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_member_or_subq},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'::'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['::']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_member_or_subq},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_member_or_subq});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_member_or_subq});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['::']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_member_or_subq},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\:\:/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['::']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_member_or_subq},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_member_or_subq},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_member_or_subq},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_member_or_subq},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_member_or_subq},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_and_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_and_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_and_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: col_not_clause 'and' col_not_clause>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: col_not_clause 'and' col_not_clause>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_and_clause});
        %item = (__RULE__ => q{col_and_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: col_not_clause 'and' col_not_clause>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_not_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_not_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_not_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_not_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_and_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_not_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_not_clause}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['and']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'and'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aand/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_not_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_not_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_not_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_not_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_and_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_not_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_not_clause}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: col_not_clause 'and' col_not_clause>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_and_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: col_not_clause 'and' col_not_clause>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = (@{$item[1]} > 1 ? new_node({ '#name' => 'and'}, $item[1]) : $item[1][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: col_not_clause 'and' col_not_clause>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_and_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_and_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_and_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_conditions
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_conditions"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{lone_conditions});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [lone_conditions end_of_conditions]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_conditions});
        %item = (__RULE__ => q{pmltq_conditions});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [lone_conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::lone_conditions, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [lone_conditions]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [lone_conditions]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{lone_conditions(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_conditions})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_conditions($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_conditions]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_conditions]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_conditions}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1][0] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [lone_conditions end_of_conditions]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_conditions});
        %item = (__RULE__ => q{pmltq_conditions});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_conditions},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_conditions},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_conditions},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::subquery
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"subquery"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [subquery]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{subquery},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{occurrences_alt});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [occurrences_alt <commit> member_or_subq]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{subquery});
        %item = (__RULE__ => q{subquery});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [occurrences_alt]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{subquery},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::occurrences_alt($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [occurrences_alt]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{subquery},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [occurrences_alt]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{occurrences_alt}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [member_or_subq]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{subquery},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{member_or_subq})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::member_or_subq($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [member_or_subq]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{subquery},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [member_or_subq]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{member_or_subq}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do {
    	        $return = $item[3];
		$return->{occurrences}=$item[1];
		1
              };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [occurrences_alt <commit> member_or_subq]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{subquery});
        %item = (__RULE__ => q{subquery});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{subquery},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{subquery},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{subquery},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{subquery},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_query
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"end_of_query"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [end_of_query]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{end_of_query},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/^\\Z/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^\\Z/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{end_of_query});
        %item = (__RULE__ => q{end_of_query});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^\\Z/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^\Z)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^\\Z/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{end_of_query},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{end_of_query},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{end_of_query},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::lone_conditions
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"lone_conditions"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [lone_conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{lone_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: condition /and|,/ condition>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: condition /and|,/ condition> end_of_conditions]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{lone_conditions});
        %item = (__RULE__ => q{lone_conditions});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: condition /and|,/ condition>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{lone_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [condition]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{lone_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{condition})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::condition($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [condition]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{lone_conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [condition]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{condition}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/and|,/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/and|,/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:and|,)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

            pop @item;
            if (defined $1) {push @item, $item{__DIRECTIVE1__}=$1; $backtrack=1;}
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [condition]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{lone_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{condition})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::condition($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [condition]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{lone_conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [condition]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{condition}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: condition /and|,/ condition>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{lone_conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: condition /and|,/ condition>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{lone_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_conditions})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_conditions($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_conditions]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{lone_conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_conditions]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_conditions}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: condition /and|,/ condition> end_of_conditions]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{lone_conditions});
        %item = (__RULE__ => q{lone_conditions});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{lone_conditions},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{lone_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{lone_conditions},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{lone_conditions},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::group_by
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"group_by"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [group_by]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{group_by},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'for', or 'give'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['for' <commit> columns 'give']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{group_by});
        %item = (__RULE__ => q{group_by});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['for']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Afor/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [columns]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{group_by},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{columns})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::columns($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [columns]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{group_by},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [columns]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{columns}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['give']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'give'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Agive/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$item[3]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['for' <commit> columns 'give']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['give' <commit>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{group_by});
        %item = (__RULE__ => q{group_by});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['give']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Agive/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['give' <commit>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        
        my $_savetext;
        @item = (q{group_by});
        %item = (__RULE__ => q{group_by});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{group_by},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{group_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{group_by},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{group_by},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::indexed_name
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"indexed_name"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [indexed_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{indexed_name},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/\\[\\s*[0-9]+\\s*\\]/, or XMLNAME});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\[\\s*[0-9]+\\s*\\]/ <skip: ''> XMLNAME]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{indexed_name});
        %item = (__RULE__ => q{indexed_name});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\[\\s*[0-9]+\\s*\\]/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\[\s*[0-9]+\s*\])/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip: ''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip= ''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [XMLNAME]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{indexed_name},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{XMLNAME})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::XMLNAME($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [XMLNAME]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{indexed_name},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [XMLNAME]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{XMLNAME}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1].$item[3] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\[\\s*[0-9]+\\s*\\]/ <skip: ''> XMLNAME]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [XMLNAME <skip: ''> /\\[\\s*[0-9]+\\s*\\]/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{indexed_name});
        %item = (__RULE__ => q{indexed_name});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [XMLNAME]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{indexed_name},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::XMLNAME($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [XMLNAME]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{indexed_name},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [XMLNAME]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{XMLNAME}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip: ''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip= ''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\[\\s*[0-9]+\\s*\\]/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\[\\s*[0-9]+\\s*\\]/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\[\s*[0-9]+\s*\])/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1].$item[3] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [XMLNAME <skip: ''> /\\[\\s*[0-9]+\\s*\\]/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{indexed_name},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{indexed_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{indexed_name},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{indexed_name},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_result_function
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_result_function"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_result_function]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_result_function},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{FUNC, or 'first_defined'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [FUNC]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_result_function});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_result_function});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [FUNC]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_result_function},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::FUNC($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [FUNC]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_result_function},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [FUNC]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{FUNC}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [FUNC]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['first_defined']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_result_function});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_result_function});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['first_defined']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Afirst_defined/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['first_defined']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_result_function},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_result_function},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_sort_term
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"column_sort_term"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [column_sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{column_sort_term},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{column_term});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [column_term 'asc', or 'desc']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_sort_term});
        %item = (__RULE__ => q{column_sort_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_sort_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_sort_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_term}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: ['asc', or 'desc']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_sort_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{'asc', or 'desc'})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_column_sort_term, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: ['asc', or 'desc']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_sort_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_column_sort_term]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_column_sort_term(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$item[1],@{$item[2]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [column_term 'asc', or 'desc']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{column_sort_term},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{column_sort_term},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{column_sort_term},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"column_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: column_exp OP column_exp>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: column_exp OP column_exp>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_expression});
        %item = (__RULE__ => q{column_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: column_exp OP column_exp>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{column_exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_exp}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [OP]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{OP})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::OP($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [OP]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [OP]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{OP}} = $_tok;
        push @item, $_tok;
        
        }

            $backtrack=1;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{column_exp})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_exp($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_exp]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_exp]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_exp}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: column_exp OP column_exp>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: column_exp OP column_exp>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { @{$item[1]}>1 ? ['EXP',@{$item[1]}] : $item[1][0] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: column_exp OP column_exp>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{column_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{column_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{column_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{column_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_selectors
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_selectors"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_selectors]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_selectors},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{selectors});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [selectors]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_selectors});
        %item = (__RULE__ => q{pmltq_selectors});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selectors]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_selectors},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selectors($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selectors]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_selectors},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selectors]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selectors}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [selectors]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_selectors});
        %item = (__RULE__ => q{pmltq_selectors});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_selectors},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_selectors},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_selectors},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_conditions
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"end_of_conditions"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [end_of_conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{end_of_conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/,?(?:\\#[^\\n]*\\n|\\s+)*\\Z/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/,?(?:\\#[^\\n]*\\n|\\s+)*\\Z/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{end_of_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{end_of_conditions});
        %item = (__RULE__ => q{end_of_conditions});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/,?(?:\\#[^\\n]*\\n|\\s+)*\\Z/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{end_of_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:,?(?:\#[^\n]*\n|\s+)*\Z)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/,?(?:\\#[^\\n]*\\n|\\s+)*\\Z/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{end_of_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{end_of_conditions},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{end_of_conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{end_of_conditions},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{end_of_conditions},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::optional_nested_selector
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"optional_nested_selector"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [optional_nested_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{optional_nested_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'?'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['?' nested_selector]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{optional_nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{optional_nested_selector});
        %item = (__RULE__ => q{optional_nested_selector});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['?']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{optional_nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\?/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [nested_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{optional_nested_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{nested_selector})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::nested_selector($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [nested_selector]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{optional_nested_selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [nested_selector]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{optional_nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{nested_selector}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{optional_nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[2]; $return->{optional}=1; 1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['?' nested_selector]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{optional_nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{optional_nested_selector},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{optional_nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{optional_nested_selector},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{optional_nested_selector},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::sort_by
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"sort_by"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [sort_by]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{sort_by},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?:>>(?:\\#[^\\n]*\\n|\\s+)*)?/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:>>(?:\\#[^\\n]*\\n|\\s+)*)?/ 'sort' <commit> 'by' <leftop: sort_term ',' sort_term>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{sort_by});
        %item = (__RULE__ => q{sort_by});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:>>(?:\\#[^\\n]*\\n|\\s+)*)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:>>(?:\#[^\n]*\n|\s+)*)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['sort']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'sort'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Asort/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: ['by']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{'by'})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_sort_by, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: ['by']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_sort_by]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_sort_by(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: sort_term ',' sort_term>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{<leftop: sort_term ',' sort_term>})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{sort_term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::sort_term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [sort_term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [sort_term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{sort_term}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{sort_term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::sort_term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [sort_term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [sort_term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{sort_term}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: sort_term ',' sort_term>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: sort_term ',' sort_term>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE2__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[5] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:>>(?:\\#[^\\n]*\\n|\\s+)*)?/ 'sort' <commit> 'by' <leftop: sort_term ',' sort_term>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{sort_by});
        %item = (__RULE__ => q{sort_by});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{sort_by},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{sort_by},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{sort_by},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::member
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"member"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [member]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{member},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{simple_attribute});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [simple_attribute name_assignment '[' /(?!\\s*\\d+\\s*\\])/ <commit> conditions /,?(?:\\#[^\\n]*\\n|\\s+)*/ ']']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{member});
        %item = (__RULE__ => q{member});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [simple_attribute]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{member},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::simple_attribute($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [simple_attribute]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{member},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [simple_attribute]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{simple_attribute}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [name_assignment]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{member},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{name_assignment})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::name_assignment, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [name_assignment]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{member},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [name_assignment]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{name_assignment(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['[']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'['})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\[/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?!\\s*\\d+\\s*\\])/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/(?!\\s*\\d+\\s*\\])/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?!\s*\d+\s*\]))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{member},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{conditions})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::conditions, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [conditions]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{member},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [conditions]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{conditions(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/,?(?:\\#[^\\n]*\\n|\\s+)*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/,?(?:\\#[^\\n]*\\n|\\s+)*/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:,?(?:\#[^\n]*\n|\s+)*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [']']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{']'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\]/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	    my $attr = $item[1];
	    my $path = join('/',@$attr[1..$#$attr]);
	    new_node({ '#name' => 'node', 
	    	       'node-type' => $path, 
		       'name' => $item[2][0],
		       'relation' => new_relation('member'),
 		     }, $item[6][0]) 
          };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [simple_attribute name_assignment '[' /(?!\\s*\\d+\\s*\\])/ <commit> conditions /,?(?:\\#[^\\n]*\\n|\\s+)*/ ']']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{member});
        %item = (__RULE__ => q{member});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{member},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{member},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{member},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{member},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::ref
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"ref"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [ref]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{ref},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{RELATION});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [RELATION selector_name]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{ref},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{ref});
        %item = (__RULE__ => q{ref});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{ref},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [RELATION]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{ref},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ref},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selector_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{ref},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{selector_name})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector_name($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selector_name]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{ref},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selector_name]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ref},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selector_name}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ref},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = new_node({ '#name' => 'ref', target=>$item[2], relation=>$item[1][0] });
         };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [RELATION selector_name]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ref},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{ref},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{ref},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{ref},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{ref},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_5_of_rule_RELATION
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_2_of_production_5_of_rule_RELATION"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_2_of_production_5_of_rule_RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_2_of_production_5_of_rule_RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/[0-9]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[0-9]+/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_2_of_production_5_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_2_of_production_5_of_rule_RELATION});
        %item = (__RULE__ => q{_alternation_2_of_production_5_of_rule_RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[0-9]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_2_of_production_5_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[0-9]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[0-9]+/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_2_of_production_5_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_2_of_production_5_of_rule_RELATION},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_2_of_production_5_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_2_of_production_5_of_rule_RELATION},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_2_of_production_5_of_rule_RELATION},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::nested_selector
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"nested_selector"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [nested_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{nested_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'+'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['+' RELATION selector]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{nested_selector});
        %item = (__RULE__ => q{nested_selector});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: ['+']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{nested_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_nested_selector, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: ['+']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{nested_selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_nested_selector]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_nested_selector(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{nested_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{RELATION})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [RELATION]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{nested_selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{nested_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{selector})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selector]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{nested_selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selector]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selector}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do {
      $return = $item[3]; 
      $return->{overlapping}=$item[1][0] ? 1 : 0;
      $return->{relation}=$item[2][0];
      1 
    };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['+' RELATION selector]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{nested_selector},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{nested_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{nested_selector},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{nested_selector},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::selectors
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"selectors"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [selectors]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{selectors},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: toplevel_selector ';' toplevel_selector>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: toplevel_selector ';' toplevel_selector>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{selectors});
        %item = (__RULE__ => q{selectors});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: toplevel_selector ';' toplevel_selector>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selectors},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [toplevel_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selectors},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{toplevel_selector})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::toplevel_selector($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [toplevel_selector]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selectors},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [toplevel_selector]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{toplevel_selector}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [';']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{';'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\;/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [toplevel_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selectors},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{toplevel_selector})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::toplevel_selector($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [toplevel_selector]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selectors},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [toplevel_selector]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{toplevel_selector}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: toplevel_selector ';' toplevel_selector>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selectors},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: toplevel_selector ';' toplevel_selector>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: toplevel_selector ';' toplevel_selector>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{selectors});
        %item = (__RULE__ => q{selectors});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{selectors},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{selectors},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{selectors},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{selectors},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{expression});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [expression end_of_string]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_expression});
        %item = (__RULE__ => q{pmltq_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_string})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_string($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_string]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_string]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_string}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [expression end_of_string]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_expression});
        %item = (__RULE__ => q{pmltq_expression});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"flat_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [flat_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    my  ($start,$t);


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<rulevar: ($start,$t)>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{flat_expression});
        %item = (__RULE__ => q{flat_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <rulevar: ($start,$t)>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<rulevar: ($start,$t)>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: term OP term>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{flat_expression});
        %item = (__RULE__ => q{flat_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ($start,$t)=($thisoffset,$text) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: term OP term>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{<leftop: term OP term>})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{term}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [OP]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{OP})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::OP($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [OP]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [OP]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{OP}} = $_tok;
        push @item, $_tok;
        
        }

            $backtrack=1;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{term}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: term OP term>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: term OP term>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=substr($t,0,$thisoffset-$start); $return=~s/^\s*//; 1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: term OP term>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{flat_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{flat_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{flat_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_simple_test
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_simple_test"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_simple_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{col_predicate, or '('});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [col_predicate]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_simple_test});
        %item = (__RULE__ => q{col_simple_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_predicate]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_predicate($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_predicate]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_simple_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_predicate]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_predicate}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [col_predicate]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['(' col_comma_clause ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_simple_test});
        %item = (__RULE__ => q{col_simple_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_comma_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_comma_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_comma_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_comma_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_simple_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_comma_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_comma_clause}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['(' col_comma_clause ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_simple_test},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_simple_test},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_simple_test},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::arguments
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"arguments"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [arguments]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{arguments},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: argument ',' argument>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: argument ',' argument>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{arguments});
        %item = (__RULE__ => q{arguments});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: argument ',' argument>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{arguments},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [argument]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{arguments},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{argument})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::argument($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [argument]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{arguments},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [argument]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{argument}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [argument]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{arguments},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{argument})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::argument($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [argument]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{arguments},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [argument]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{argument}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: argument ',' argument>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{arguments},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: argument ',' argument>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: argument ',' argument>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{arguments},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{arguments},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{arguments},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{arguments},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::conditions
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"conditions"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: condition /and|,/ condition>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: condition /and|,/ condition>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{conditions});
        %item = (__RULE__ => q{conditions});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: condition /and|,/ condition>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [condition]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{condition})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::condition($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [condition]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [condition]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{condition}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/and|,/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/and|,/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:and|,)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

            pop @item;
            if (defined $1) {push @item, $item{__DIRECTIVE1__}=$1; $backtrack=1;}
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [condition]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{conditions},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{condition})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::condition($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [condition]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [condition]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{condition}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: condition /and|,/ condition>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{conditions},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: condition /and|,/ condition>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: condition /and|,/ condition>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{conditions});
        %item = (__RULE__ => q{conditions});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{conditions},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{conditions},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{conditions},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{conditions},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_node
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_node"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_node]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_node},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{member_selector, or nested_selector, or optional_nested_selector});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [member_selector, or nested_selector, or optional_nested_selector end_of_query]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_node});
        %item = (__RULE__ => q{pmltq_node});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [_alternation_1_of_production_1_of_rule_pmltq_node]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_node},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_pmltq_node($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [_alternation_1_of_production_1_of_rule_pmltq_node]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_node},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [_alternation_1_of_production_1_of_rule_pmltq_node]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_pmltq_node}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_query]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_node},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_query})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_query($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_query]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_node},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_query]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_query}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [member_selector, or nested_selector, or optional_nested_selector end_of_query]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_node});
        %item = (__RULE__ => q{pmltq_node});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_node},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_node},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_node},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_node},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_simple_attribute
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_simple_attribute"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_simple_attribute]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_simple_attribute},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'[]', or 'content()', or indexed_name, or XMLNAME});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['[]']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_simple_attribute});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_simple_attribute});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['[]']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\[\]/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['[]']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['content()']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_simple_attribute});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_simple_attribute});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['content()']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Acontent\(\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['content()']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [indexed_name]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_simple_attribute});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_simple_attribute});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [indexed_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_simple_attribute},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::indexed_name($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [indexed_name]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_simple_attribute},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [indexed_name]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{indexed_name}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [indexed_name]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [XMLNAME]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_simple_attribute});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_simple_attribute});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [XMLNAME]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_simple_attribute},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::XMLNAME($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [XMLNAME]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_simple_attribute},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [XMLNAME]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{XMLNAME}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [XMLNAME]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_simple_attribute},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_simple_attribute},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_test
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_test"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_test},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{test});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [test end_of_query]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_test});
        %item = (__RULE__ => q{pmltq_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{test}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_query]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_query})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_query($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_query]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_query]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_query}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [test end_of_query]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_test});
        %item = (__RULE__ => q{pmltq_test});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_test},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_test},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_test},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::function
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"function"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [function]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{function},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{FUNC});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [FUNC <skip: ''> '(' <skip: '\s*(?:[#][^\n]*\s*)*'> <commit> arguments ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{function});
        %item = (__RULE__ => q{function});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [FUNC]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{function},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::FUNC($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [FUNC]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{function},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [FUNC]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{FUNC}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip: ''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip= ''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'('})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip: '\s*(?:[#][^\n]*\s*)*'>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip= '\s*(?:[#][^\n]*\s*)*'; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE3__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [arguments]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{function},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{arguments})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::arguments, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [arguments]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{function},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [arguments]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{arguments(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['FUNC', $item[1],$item[6][0] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [FUNC <skip: ''> '(' <skip: '\s*(?:[#][^\n]*\s*)*'> <commit> arguments ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{function});
        %item = (__RULE__ => q{function});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{function},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{function},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{function},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_sort_term
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_sort_term"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_sort_term},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'asc', or 'desc'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['asc']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_sort_term});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_sort_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['asc']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aasc/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['asc']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['desc']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_sort_term});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_sort_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['desc']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Adesc/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['desc']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_sort_term},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_sort_term},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_predicate
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_predicate"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_predicate]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_predicate},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{flat_column_expression});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [flat_column_expression /!?\\s*in/ <commit> flat_set_col_expression]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_predicate});
        %item = (__RULE__ => q{col_predicate});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_predicate},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_column_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_column_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_predicate},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_column_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_column_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/!?\\s*in/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/!?\\s*in/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:!?\s*in)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_set_col_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_predicate},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{flat_set_col_expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_set_col_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_set_col_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_predicate},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_set_col_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_set_col_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { my $op = $item[2]; $op=~s/\s+//g;
	         new_node({ '#name' => 'test', a=>$item[1], operator=>$op, b=>$item[4] })
	       };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [flat_column_expression /!?\\s*in/ <commit> flat_set_col_expression]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [flat_column_expression CMP <commit> flat_column_expression]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_predicate});
        %item = (__RULE__ => q{col_predicate});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_predicate},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_column_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_column_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_predicate},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_column_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_column_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [CMP]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_predicate},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{CMP})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::CMP($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [CMP]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_predicate},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [CMP]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{CMP}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_predicate},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{flat_column_expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_column_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_column_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_predicate},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_column_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_column_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
                  my $op = $item[2];
		  $return = new_node({ '#name' => 'test', a=>$item[1], operator=>$op, b=>$item[4] });
	       };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [flat_column_expression CMP <commit> flat_column_expression]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_predicate},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_predicate},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_predicate},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_predicate},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_test
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_test"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_test},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{col_or_clause});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [col_or_clause]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_test});
        %item = (__RULE__ => q{col_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_or_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_or_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_or_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_or_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_or_clause}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [col_or_clause]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_test},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_test},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_test},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::STRING
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"STRING"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [STRING]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{STRING},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/'([^'\\\\]+|\\\\.)*'|"([^"\\\\]+|\\\\.)*"/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/'([^'\\\\]+|\\\\.)*'|"([^"\\\\]+|\\\\.)*"/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{STRING},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{STRING});
        %item = (__RULE__ => q{STRING});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/'([^'\\\\]+|\\\\.)*'|"([^"\\\\]+|\\\\.)*"/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{STRING},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:'([^'\\]+|\\.)*'|"([^"\\]+|\\.)*")/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/'([^'\\\\]+|\\\\.)*'|"([^"\\\\]+|\\\\.)*"/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{STRING},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{STRING},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{STRING},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{STRING},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{STRING},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::argument
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"argument"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [argument]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{argument},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{expression, or VAR_OR_SELF});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [expression]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{argument});
        %item = (__RULE__ => q{argument});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{argument},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{argument},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [expression]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [VAR_OR_SELF]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{argument});
        %item = (__RULE__ => q{argument});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [VAR_OR_SELF]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{argument},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::VAR_OR_SELF($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [VAR_OR_SELF]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{argument},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [VAR_OR_SELF]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{VAR_OR_SELF}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { '$'.$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [VAR_OR_SELF]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        
        my $_savetext;
        @item = (q{argument});
        %item = (__RULE__ => q{argument});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{argument},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{argument},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{argument},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{argument},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_sort_by
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_sort_by"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_sort_by]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_sort_by},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'by'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['by']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_sort_by});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_sort_by});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['by']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aby/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['by']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_sort_by},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_sort_by},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_sort_by},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_sort_by},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::analytic_function
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"analytic_function"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [analytic_function]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{analytic_function},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{ANALYTIC_FUNC});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [ANALYTIC_FUNC <skip: ''> '(' <commit> <skip: '\s*(?:[#][^\n]*\s*)*'> result_arguments over_clause ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{analytic_function});
        %item = (__RULE__ => q{analytic_function});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [ANALYTIC_FUNC]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{analytic_function},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::ANALYTIC_FUNC($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [ANALYTIC_FUNC]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{analytic_function},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [ANALYTIC_FUNC]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{ANALYTIC_FUNC}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip: ''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip= ''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'('})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip: '\s*(?:[#][^\n]*\s*)*'>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip= '\s*(?:[#][^\n]*\s*)*'; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE3__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [result_arguments]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{analytic_function},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{result_arguments})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::result_arguments, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [result_arguments]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{analytic_function},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [result_arguments]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{result_arguments(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [over_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{analytic_function},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{over_clause})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::over_clause, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [over_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{analytic_function},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [over_clause]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{over_clause(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['ANALYTIC_FUNC', $item[1], $item[6][0], @{$item[7][0]||[]} ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [ANALYTIC_FUNC <skip: ''> '(' <commit> <skip: '\s*(?:[#][^\n]*\s*)*'> result_arguments over_clause ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{analytic_function},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{analytic_function},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{analytic_function},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{analytic_function},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_8_of_rule_RELATION
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_8_of_rule_RELATION"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_8_of_rule_RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_8_of_rule_RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/[0-9]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[0-9]+/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_8_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_8_of_rule_RELATION});
        %item = (__RULE__ => q{_alternation_1_of_production_8_of_rule_RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[0-9]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_8_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[0-9]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[0-9]+/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_8_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_8_of_rule_RELATION},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_8_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_8_of_rule_RELATION},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_8_of_rule_RELATION},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::VAR_OR_SELF
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"VAR_OR_SELF"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [VAR_OR_SELF]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{VAR_OR_SELF},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'$$', or selector_name});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['$$']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{VAR_OR_SELF});
        %item = (__RULE__ => q{VAR_OR_SELF});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['$$']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\$\$/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return='$' };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['$$']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [selector_name]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{VAR_OR_SELF});
        %item = (__RULE__ => q{VAR_OR_SELF});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selector_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{VAR_OR_SELF},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector_name($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selector_name]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{VAR_OR_SELF},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selector_name]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selector_name}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [selector_name]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{VAR_OR_SELF},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{VAR_OR_SELF},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{VAR_OR_SELF},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{VAR_OR_SELF},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::RELATION
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"RELATION"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?:sibling|descendant|ancestor|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/, or /(?:child|parent|sibling|descendant|ancestor|same-tree-as|same-document-as|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/, or /[_a-zA-Z][-.\\/_a-zA-Z]*(?=\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$))/, or /[_a-zA-Z][-.\\/_a-zA-Z]*(?=(?:\\\{[0-9]*,[0-9]*\\\})(?:\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$)))/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:sibling|descendant|ancestor|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /-?[0-9]+/ ',' /-?[0-9]+/ '\}' /\\s*(?!->)(::)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:sibling|descendant|ancestor|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:sibling|descendant|ancestor|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\{'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:'\s*(?:[#][^\n]*\s*)*'>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip='\s*(?:[#][^\n]*\s*)*'; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE3__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/-?[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/-?[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/-?[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/-?[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/-?[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_1_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/-?[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_2_of_production_1_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_2_of_production_1_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!->)(::)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!->)(::)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!->)(::)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation($item[1],{min_length=>$item[6][0],max_length=>$item[8][0]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:sibling|descendant|ancestor|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /-?[0-9]+/ ',' /-?[0-9]+/ '\}' /\\s*(?!->)(::)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:child|parent|sibling|descendant|ancestor|same-tree-as|same-document-as|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/ /\\s*(?!->)(::)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:child|parent|sibling|descendant|ancestor|same-tree-as|same-document-as|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:child|parent|sibling|descendant|ancestor|same-tree-as|same-document-as|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!->)(::)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!->)(::)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!->)(::)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation($item[1]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:child|parent|sibling|descendant|ancestor|same-tree-as|same-document-as|depth-first-precedes|depth-first-follows|order-precedes|order-follows)\\b/ /\\s*(?!->)(::)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:$\{Tree_Query::user_defined\})\\b/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /[0-9]+/ ',' /[0-9]+/ '\}' /\\s*(?!->)(::)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $Tree_Query::user_defined ? 1 : undef };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:$\{Tree_Query::user_defined\})\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/(?:$\{Tree_Query::user_defined\})\\b/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:${Tree_Query::user_defined})\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\{'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:'\s*(?:[#][^\n]*\s*)*'>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip='\s*(?:[#][^\n]*\s*)*'; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE3__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_3_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_3_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_3_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_3_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_2_of_production_3_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_2_of_production_3_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!->)(::)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!->)(::)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!->)(::)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation('user-defined',{label => $item[2],category=>'implementation',
              min_length=>$item[7][0], max_length=>$item[9][0]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:$\{Tree_Query::user_defined\})\\b/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /[0-9]+/ ',' /[0-9]+/ '\}' /\\s*(?!->)(::)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:$\{Tree_Query::user_defined\})\\b/ /\\s*(?!->)(::)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $Tree_Query::user_defined ? 1 : undef };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:$\{Tree_Query::user_defined\})\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/(?:$\{Tree_Query::user_defined\})\\b/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:${Tree_Query::user_defined})\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!->)(::)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!->)(::)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!->)(::)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation('user-defined',{label => $item[2],category=>'implementation'}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:$\{Tree_Query::user_defined\})\\b/ /\\s*(?!->)(::)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:$\{Tree_Query::pmlrf_relations\})\\b/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /[0-9]+/ ',' /[0-9]+/ '\}' /\\s*(?!::)(->)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $Tree_Query::pmlrf_relations ? 1 : undef };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:$\{Tree_Query::pmlrf_relations\})\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/(?:$\{Tree_Query::pmlrf_relations\})\\b/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:${Tree_Query::pmlrf_relations})\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\{'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:'\s*(?:[#][^\n]*\s*)*'>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip='\s*(?:[#][^\n]*\s*)*'; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE3__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_5_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_5_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_5_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_5_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_2_of_production_5_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_2_of_production_5_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!::)(->)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!::)(->)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!::)(->)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation('user-defined',{label => $item[2],
	      category=>'pmlrf',
              min_length=>$item[7][0], max_length=>$item[9][0]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:$\{Tree_Query::pmlrf_relations\})\\b/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /[0-9]+/ ',' /[0-9]+/ '\}' /\\s*(?!::)(->)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?:$\{Tree_Query::pmlrf_relations\})\\b/ /\\s*(?!::)(->)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $Tree_Query::pmlrf_relations ? 1 : undef };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?:$\{Tree_Query::pmlrf_relations\})\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/(?:$\{Tree_Query::pmlrf_relations\})\\b/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?:${Tree_Query::pmlrf_relations})\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!::)(->)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!::)(->)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!::)(->)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation('user-defined',{label => $item[2],category=>'pmlrf'}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?:$\{Tree_Query::pmlrf_relations\})\\b/ /\\s*(?!::)(->)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[_a-zA-Z][-.\\/_a-zA-Z]*(?=\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$))/ /\\s*(?!::)(->)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[6];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[_a-zA-Z][-.\\/_a-zA-Z]*(?=\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$))/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[_a-zA-Z][-.\/_a-zA-Z]*(?=\s*->|\s*\$[[:alpha:]_][[:alnum:]_]*\b(?!\s*:=)|\s+${Treex::PML::Schema::CDATA::Name}\s*(?:\[|\$)))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!::)(->)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!::)(->)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!::)(->)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation('user-defined',{label => $item[1],category=>'pmlrf'}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[_a-zA-Z][-.\\/_a-zA-Z]*(?=\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$))/ /\\s*(?!::)(->)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        local $skip = defined($skip) ? $skip : $Parse::RecDescent::_Runtime::skip;
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/[_a-zA-Z][-.\\/_a-zA-Z]*(?=(?:\\\{[0-9]*,[0-9]*\\\})(?:\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$)))/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /[0-9]+/ ',' /[0-9]+/ '\}' /\\s*(?!::)(->)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[7];
        $text = $_[1];
        my $_savetext;
        @item = (q{RELATION});
        %item = (__RULE__ => q{RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/[_a-zA-Z][-.\\/_a-zA-Z]*(?=(?:\\\{[0-9]*,[0-9]*\\\})(?:\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$)))/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:[_a-zA-Z][-.\/_a-zA-Z]*(?=(?:\{[0-9]*,[0-9]*\})(?:\s*->|\s*\$[[:alpha:]_][[:alnum:]_]*\b(?!\s*:=)|\s+${Treex::PML::Schema::CDATA::Name}\s*(?:\[|\$))))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:''>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip=''; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\{'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE2__}=$_tok;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<skip:'\s*(?:[#][^\n]*\s*)*'>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { my $oldskip = $skip; $skip='\s*(?:[#][^\n]*\s*)*'; $oldskip };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE3__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_8_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_8_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_8_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[0-9]+/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[0-9]+/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_8_of_rule_RELATION, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[0-9]+/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{RELATION},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_2_of_production_8_of_rule_RELATION]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_2_of_production_8_of_rule_RELATION(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*(?!::)(->)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/\\s*(?!::)(->)?/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*(?!::)(->)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_relation('user-defined',{label => $item[1], category=>'pmlrf', min_length=>$item[6][0], max_length=>$item[8][0]}) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/[_a-zA-Z][-.\\/_a-zA-Z]*(?=(?:\\\{[0-9]*,[0-9]*\\\})(?:\\s*->|\\s*\\$[[:alpha:]_][[:alnum:]_]*\\b(?!\\s*:=)|\\s+$\{Treex::PML::Schema::CDATA::Name\}\\s*(?:\\[|\\$)))/ <skip:''> '\{' <commit> <skip:'\s*(?:[#][^\n]*\s*)*'> /[0-9]+/ ',' /[0-9]+/ '\}' /\\s*(?!::)(->)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{RELATION},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{RELATION},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{RELATION},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_toplevel_selector
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_toplevel_selector"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_toplevel_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'+'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['+']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_toplevel_selector});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_toplevel_selector});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['+']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\+/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['+']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_toplevel_selector},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::NUMBER
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"NUMBER"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [NUMBER]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{NUMBER},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/-?[0-9]+(\\.[0-9]+)?/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/-?[0-9]+(\\.[0-9]+)?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{NUMBER},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{NUMBER});
        %item = (__RULE__ => q{NUMBER});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/-?[0-9]+(\\.[0-9]+)?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{NUMBER},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:-?[0-9]+(\.[0-9]+)?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/-?[0-9]+(\\.[0-9]+)?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{NUMBER},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{NUMBER},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{NUMBER},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{NUMBER},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{NUMBER},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::content_look_ahead
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"content_look_ahead"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [content_look_ahead]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{content_look_ahead},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(\\s*\\d+\\s*\\])/, or /\\s*\\]\\s*\\//});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(\\s*\\d+\\s*\\])/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{content_look_ahead});
        %item = (__RULE__ => q{content_look_ahead});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(\\s*\\d+\\s*\\])/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(\s*\d+\s*\]))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(\\s*\\d+\\s*\\])/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\s*\\]\\s*\\//]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{content_look_ahead});
        %item = (__RULE__ => q{content_look_ahead});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\s*\\]\\s*\\//]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\s*\]\s*\/)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\s*\\]\\s*\\//]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{content_look_ahead},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{content_look_ahead},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{content_look_ahead},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{content_look_ahead},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::ANALYTIC_FUNC
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"ANALYTIC_FUNC"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [ANALYTIC_FUNC]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{ANALYTIC_FUNC},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/min|max|sum|avg|count|ratio|concat|row_number|rank|dense_rank/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/min|max|sum|avg|count|ratio|concat|row_number|rank|dense_rank/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{ANALYTIC_FUNC},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{ANALYTIC_FUNC});
        %item = (__RULE__ => q{ANALYTIC_FUNC});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/min|max|sum|avg|count|ratio|concat|row_number|rank|dense_rank/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ANALYTIC_FUNC},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:min|max|sum|avg|count|ratio|concat|row_number|rank|dense_rank)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/min|max|sum|avg|count|ratio|concat|row_number|rank|dense_rank/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{ANALYTIC_FUNC},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{ANALYTIC_FUNC},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{ANALYTIC_FUNC},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{ANALYTIC_FUNC},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{ANALYTIC_FUNC},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_not_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_not_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_not_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_not_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'!', or col_simple_test});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['!' <commit> col_simple_test]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_not_clause});
        %item = (__RULE__ => q{col_not_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['!']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\!/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_simple_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_not_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_simple_test})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_simple_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_simple_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_not_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_simple_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_simple_test}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do {  $return = $item[3];
                 if ($return->{'#name'} eq 'and') {
                   $return->{'#name'} = 'not';
                 } else {
                   $return = new_node({ '#name' => 'not'}, [$return])
                 }
              };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['!' <commit> col_simple_test]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [col_simple_test]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_not_clause});
        %item = (__RULE__ => q{col_not_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_simple_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_not_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_simple_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_simple_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_not_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_simple_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_simple_test}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [col_simple_test]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_not_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_not_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_not_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::over_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"over_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [over_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{over_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'over', or sort_by_clause});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['over' over_columns sort_by_clause]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{over_clause});
        %item = (__RULE__ => q{over_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['over']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aover/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [over_columns]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{over_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{over_columns})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::over_columns($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [over_columns]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{over_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [over_columns]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{over_columns}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [sort_by_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{over_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{sort_by_clause})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::sort_by_clause, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [sort_by_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{over_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [sort_by_clause]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{sort_by_clause(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [$item[2],$item[3][0]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['over' over_columns sort_by_clause]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [sort_by_clause]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{over_clause});
        %item = (__RULE__ => q{over_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [sort_by_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{over_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::sort_by_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [sort_by_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{over_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [sort_by_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{sort_by_clause}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [['ALL'],$item[3][0]] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [sort_by_clause]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{over_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{over_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{over_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{over_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_reference
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"column_reference"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [column_reference]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{column_reference},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/^\\$[1-9][0-9]*/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^\\$[1-9][0-9]*/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_reference},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_reference});
        %item = (__RULE__ => q{column_reference});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^\\$[1-9][0-9]*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_reference},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^\$[1-9][0-9]*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^\\$[1-9][0-9]*/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_reference},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{column_reference},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{column_reference},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{column_reference},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{column_reference},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_term
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"column_term"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [column_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'(', or column_reference, or result_function, or attribute_path, or selector_name, or literal});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['(' <commit> column_expression ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{column_expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[3] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['(' <commit> column_expression ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [column_reference <commit>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_reference]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_reference($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_reference]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_reference]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_reference}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [column_reference <commit>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [result_function]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [result_function]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::result_function($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [result_function]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [result_function]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{result_function}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [result_function]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [attribute_path]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [attribute_path]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::attribute_path($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [attribute_path]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [attribute_path]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{attribute_path}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [attribute_path]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [selector_name]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selector_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector_name($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selector_name]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selector_name]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selector_name}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return='$'.$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [selector_name]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [literal]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [literal]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_term},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::literal($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [literal]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_term},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [literal]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{literal}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [literal]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[6];
        
        my $_savetext;
        @item = (q{column_term});
        %item = (__RULE__ => q{column_term});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{column_term},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{column_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{column_term},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{column_term},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::filter
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"filter"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [filter]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{filter},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/>>\\s*(?:filter|where)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/>>\\s*(?:filter|where)/ <commit> col_comma_clause]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{filter});
        %item = (__RULE__ => q{filter});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/>>\\s*(?:filter|where)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:>>\s*(?:filter|where))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_comma_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{filter},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_comma_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_comma_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_comma_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{filter},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_comma_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_comma_clause}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do {
            $return=Treex::PML::Factory->createStructure({'where'=>new_node({'#name'=>'and'},[$item[3]])})
          };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/>>\\s*(?:filter|where)/ <commit> col_comma_clause]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{filter});
        %item = (__RULE__ => q{filter});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{filter},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{filter},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{filter},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{filter},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::or_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"or_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [or_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: and_clause 'or' and_clause>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: and_clause 'or' and_clause>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{or_clause});
        %item = (__RULE__ => q{or_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: and_clause 'or' and_clause>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [and_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{and_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::and_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [and_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{or_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [and_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{and_clause}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['or']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'or'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aor/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [and_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{and_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::and_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [and_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{or_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [and_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{and_clause}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: and_clause 'or' and_clause>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{or_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: and_clause 'or' and_clause>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return =  (@{$item[1]} > 1 ? new_node({ '#name' => 'or'}, $item[1]) : $item[1][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: and_clause 'or' and_clause>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{or_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{or_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{or_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_filters
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_filters"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_filters]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{output_filters});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [output_filters end_of_query]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_filters});
        %item = (__RULE__ => q{pmltq_filters});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [output_filters]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::output_filters($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [output_filters]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [output_filters]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{output_filters}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_query]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_query})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_query($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_query]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_query]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_query}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=@{$item[1]}; 1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [output_filters end_of_query]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_filters});
        %item = (__RULE__ => q{pmltq_filters});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_filters},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_filters},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_filters},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector_name
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"selector_name"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [selector_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{selector_name},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/\\$[[:alpha:]_][[:alnum:]_]*\\b/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\$[[:alpha:]_][[:alnum:]_]*\\b/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{selector_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{selector_name});
        %item = (__RULE__ => q{selector_name});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\$[[:alpha:]_][[:alnum:]_]*\\b/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\$[[:alpha:]_][[:alnum:]_]*\b)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=$item[1]; $return=~s/^\$//; 1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\$[[:alpha:]_][[:alnum:]_]*\\b/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector_name},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{selector_name},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{selector_name},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{selector_name},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{selector_name},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::every
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"every"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [every]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{every},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'*'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['*' <commit> attribute_path, or simple_attribute]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{every});
        %item = (__RULE__ => q{every});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['*']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\*/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [_alternation_1_of_production_1_of_rule_every]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{every},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{attribute_path, or simple_attribute})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_every($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [_alternation_1_of_production_1_of_rule_every]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{every},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [_alternation_1_of_production_1_of_rule_every]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_every}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { [ 'EVERY',$item[3] ] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['*' <commit> attribute_path, or simple_attribute]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{every});
        %item = (__RULE__ => q{every});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{every},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{every},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{every},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{every},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::comma_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"comma_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [comma_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: test ',' test>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: test ',' test>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{comma_clause});
        %item = (__RULE__ => q{comma_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: test ',' test>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{test})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{comma_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{test}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{test})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{comma_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{test}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: test ',' test>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{comma_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: test ',' test>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return =  (@{$item[1]} > 1 ? new_node({ '#name' => 'and'}, $item[1]) : $item[1][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: test ',' test>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{comma_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{comma_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{comma_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::comma_or_end_of_query
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"comma_or_end_of_query"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [comma_or_end_of_query]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{comma_or_end_of_query},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{','});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [',' /^\\Z/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{comma_or_end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{comma_or_end_of_query});
        %item = (__RULE__ => q{comma_or_end_of_query});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [',']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{comma_or_end_of_query},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_comma_or_end_of_query, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [',']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{comma_or_end_of_query},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_comma_or_end_of_query]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_or_end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_comma_or_end_of_query(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^\\Z/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_or_end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/^\\Z/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^\Z)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [',' /^\\Z/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{comma_or_end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{comma_or_end_of_query},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{comma_or_end_of_query},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{comma_or_end_of_query},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{comma_or_end_of_query},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_exp
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"column_exp"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [column_exp]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{column_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{column_term, or analytic_function, or if_function});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [column_term]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_exp});
        %item = (__RULE__ => q{column_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_term}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [column_term]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [analytic_function]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_exp});
        %item = (__RULE__ => q{column_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [analytic_function]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::analytic_function($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [analytic_function]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [analytic_function]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{analytic_function}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [analytic_function]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [if_function]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{column_exp});
        %item = (__RULE__ => q{column_exp});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [if_function]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{column_exp},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::if_function($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [if_function]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{column_exp},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [if_function]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{if_function}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [if_function]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{column_exp},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{column_exp},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{column_exp},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{column_exp},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_2_of_rule_output_filters_or_query_end
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_2_of_rule_output_filters_or_query_end"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_2_of_rule_output_filters_or_query_end]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{';'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [';']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end});
        %item = (__RULE__ => q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [';']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\;/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [';']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_2_of_rule_output_filters_or_query_end},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_1_of_rule_RELATION
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_2_of_production_1_of_rule_RELATION"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_2_of_production_1_of_rule_RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_2_of_production_1_of_rule_RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/-?[0-9]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/-?[0-9]+/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_2_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_2_of_production_1_of_rule_RELATION});
        %item = (__RULE__ => q{_alternation_2_of_production_1_of_rule_RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/-?[0-9]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_2_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:-?[0-9]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/-?[0-9]+/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_2_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_2_of_production_1_of_rule_RELATION},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_2_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_2_of_production_1_of_rule_RELATION},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_2_of_production_1_of_rule_RELATION},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::sort_by_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"sort_by_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [sort_by_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{sort_by_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/sort\\s+by/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/sort\\s+by/ <leftop: column_sort_term ',' column_sort_term>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{sort_by_clause});
        %item = (__RULE__ => q{sort_by_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/sort\\s+by/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:sort\s+by)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: column_sort_term ',' column_sort_term>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{<leftop: column_sort_term ',' column_sort_term>})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{column_sort_term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_sort_term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_sort_term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_sort_term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_sort_term}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [column_sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{sort_by_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{column_sort_term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::column_sort_term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [column_sort_term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [column_sort_term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{column_sort_term}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: column_sort_term ',' column_sort_term>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{sort_by_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: column_sort_term ',' column_sort_term>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/sort\\s+by/ <leftop: column_sort_term ',' column_sort_term>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{sort_by_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{sort_by_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{sort_by_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{sort_by_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_RELATION
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_RELATION"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_RELATION]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_RELATION},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/-?[0-9]+/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/-?[0-9]+/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_RELATION});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_RELATION});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/-?[0-9]+/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:-?[0-9]+)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/-?[0-9]+/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_RELATION},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_RELATION},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_RELATION},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_RELATION},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_or_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_or_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_or_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: col_and_clause 'or' col_and_clause>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: col_and_clause 'or' col_and_clause>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_or_clause});
        %item = (__RULE__ => q{col_or_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: col_and_clause 'or' col_and_clause>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_and_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_and_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_and_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_and_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_or_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_and_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_and_clause}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['or']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'or'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aor/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_and_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_or_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_and_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_and_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_and_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_or_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_and_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_and_clause}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: col_and_clause 'or' col_and_clause>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_or_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: col_and_clause 'or' col_and_clause>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return =  (@{$item[1]} > 1 ? new_node({ '#name' => 'or'}, $item[1]) : $item[1][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: col_and_clause 'or' col_and_clause>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_or_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_or_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_or_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_or_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::test
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"test"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [test]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{test},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{or_clause});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [or_clause]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{test});
        %item = (__RULE__ => q{test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [or_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::or_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [or_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [or_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{or_clause}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [or_clause]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{test},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{test},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{test},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::XMLNAME
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"XMLNAME"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [XMLNAME]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{XMLNAME},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/$\{Treex::PML::Schema::CDATA::Name\}/o});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/$\{Treex::PML::Schema::CDATA::Name\}/o]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{XMLNAME},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{XMLNAME});
        %item = (__RULE__ => q{XMLNAME});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/$\{Treex::PML::Schema::CDATA::Name\}/o]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{XMLNAME},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:${Treex::PML::Schema::CDATA::Name})/o)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/$\{Treex::PML::Schema::CDATA::Name\}/o]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{XMLNAME},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{XMLNAME},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{XMLNAME},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{XMLNAME},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{XMLNAME},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::pmltq_flat_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"pmltq_flat_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [pmltq_flat_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{pmltq_flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{flat_expression});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [flat_expression end_of_string]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{pmltq_flat_expression});
        %item = (__RULE__ => q{pmltq_flat_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_flat_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_expression}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [end_of_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{pmltq_flat_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{end_of_string})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_string($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [end_of_string]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{pmltq_flat_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [end_of_string]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{end_of_string}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [flat_expression end_of_string]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{pmltq_flat_expression});
        %item = (__RULE__ => q{pmltq_flat_expression});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{pmltq_flat_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{pmltq_flat_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{pmltq_flat_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{pmltq_flat_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::occurrences
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"occurrences"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [occurrences]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{occurrences},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{NUMBER});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [NUMBER '..' <commit> NUMBER]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{occurrences});
        %item = (__RULE__ => q{occurrences});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [NUMBER]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{occurrences},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::NUMBER($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [NUMBER]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{occurrences},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [NUMBER]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{NUMBER}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['..']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'..'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\.\./)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [NUMBER]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{occurrences},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{NUMBER})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::NUMBER($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [NUMBER]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{occurrences},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [NUMBER]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{NUMBER}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = new_struct({ min=>$item[1], max=>$item[4] }) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [NUMBER '..' <commit> NUMBER]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [NUMBER /[-+]/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{occurrences});
        %item = (__RULE__ => q{occurrences});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [NUMBER]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{occurrences},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::NUMBER($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [NUMBER]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{occurrences},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [NUMBER]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{NUMBER}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [/[-+]/]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{occurrences},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{/[-+]/})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_2_of_rule_occurrences, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [/[-+]/]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{occurrences},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_2_of_rule_occurrences]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_2_of_rule_occurrences(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
	         $return = !$item[2][0] 
		       ? new_struct({ min=>$item[1], max=>$item[1] })
	               : new_struct({ ($item[2][0] eq '-' ? 'max' : 'min') =>
		                      $item[1] })
		 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [NUMBER /[-+]/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        
        my $_savetext;
        @item = (q{occurrences});
        %item = (__RULE__ => q{occurrences});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{occurrences},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{occurrences},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{occurrences},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{occurrences},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_set_expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"flat_set_expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [flat_set_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{flat_set_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    my  ($start,$t);


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<rulevar: ($start,$t)>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{flat_set_expression});
        %item = (__RULE__ => q{flat_set_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <rulevar: ($start,$t)>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<rulevar: ($start,$t)>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['\{' <leftop: flat_expression ',' flat_expression> '\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{flat_set_expression});
        %item = (__RULE__ => q{flat_set_expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ($start,$t)=($thisoffset,$text) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\{'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: flat_expression ',' flat_expression>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_set_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{<leftop: flat_expression ',' flat_expression>})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_set_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{flat_expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_set_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_expression}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [flat_expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{flat_set_expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{flat_expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::flat_expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [flat_expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_set_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [flat_expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{flat_expression}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: flat_expression ',' flat_expression>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{flat_set_expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: flat_expression ',' flat_expression>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return=substr($t,0,$thisoffset-$start); $return=~s/^\s*//; 1 };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION2__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['\{' <leftop: flat_expression ',' flat_expression> '\}']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{flat_set_expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{flat_set_expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{flat_set_expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{flat_set_expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::and_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"and_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [and_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: not_clause 'and' not_clause>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: not_clause 'and' not_clause>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{and_clause});
        %item = (__RULE__ => q{and_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: not_clause 'and' not_clause>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [not_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{not_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::not_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [not_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{and_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [not_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{not_clause}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['and']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'and'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aand/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [not_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{and_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{not_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::not_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [not_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{and_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [not_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{not_clause}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: not_clause 'and' not_clause>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{and_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: not_clause 'and' not_clause>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = (@{$item[1]} > 1 ? new_node({ '#name' => 'and'}, $item[1]) : $item[1][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: not_clause 'and' not_clause>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{and_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{and_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{and_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{and_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::NODE_TYPE
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"NODE_TYPE"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [NODE_TYPE]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{NODE_TYPE},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/\\*|$\{Treex::PML::Schema::CDATA::NCName\}(?:[:][*])?/o});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\*|$\{Treex::PML::Schema::CDATA::NCName\}(?:[:][*])?/o]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{NODE_TYPE},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{NODE_TYPE});
        %item = (__RULE__ => q{NODE_TYPE});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\*|$\{Treex::PML::Schema::CDATA::NCName\}(?:[:][*])?/o]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{NODE_TYPE},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\*|${Treex::PML::Schema::CDATA::NCName}(?:[:][*])?)/o)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\*|$\{Treex::PML::Schema::CDATA::NCName\}(?:[:][*])?/o]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{NODE_TYPE},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{NODE_TYPE},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{NODE_TYPE},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{NODE_TYPE},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{NODE_TYPE},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_column_sort_term
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_column_sort_term"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_column_sort_term]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_column_sort_term},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'asc', or 'desc'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['asc']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_column_sort_term});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_column_sort_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['asc']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Aasc/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['asc']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['desc']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_column_sort_term});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_column_sort_term});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['desc']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Adesc/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['desc']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_column_sort_term},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_column_sort_term},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_comma_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"col_comma_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [col_comma_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{col_comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{<leftop: col_test ',' col_test>});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: col_test ',' col_test>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{col_comma_clause});
        %item = (__RULE__ => q{col_comma_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: col_test ',' col_test>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_test})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_comma_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_test}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [col_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{col_comma_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{col_test})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::col_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [col_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_comma_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [col_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{col_test}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: col_test ',' col_test>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{col_comma_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: col_test ',' col_test>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return =  (@{$item[1]} > 1 ? new_node({ '#name' => 'and'}, $item[1]) : $item[1][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: col_test ',' col_test>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{col_comma_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{col_comma_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{col_comma_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{col_comma_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::literal
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"literal"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [literal]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{literal},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{NUMBER, or STRING});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [NUMBER]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{literal});
        %item = (__RULE__ => q{literal});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [NUMBER]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{literal},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::NUMBER($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [NUMBER]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{literal},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [NUMBER]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{NUMBER}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [NUMBER]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [STRING]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{literal});
        %item = (__RULE__ => q{literal});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [STRING]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{literal},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::STRING($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [STRING]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{literal},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [STRING]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{STRING}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [STRING]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{literal},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{literal},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{literal},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{literal},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::step
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"step"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [step]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{step},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'content()', or '.', or indexed_name, or XMLNAME, or /\\[\\s*[0-9]+\\s*\\]/, or '[]'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['content()']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['content()']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\Acontent\(\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['content()']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['.']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['.']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\./)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['.']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [indexed_name]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [indexed_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{step},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::indexed_name($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [indexed_name]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{step},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [indexed_name]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{indexed_name}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [indexed_name]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [XMLNAME]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [XMLNAME]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{step},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::XMLNAME($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [XMLNAME]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{step},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [XMLNAME]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{XMLNAME}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [XMLNAME]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/\\[\\s*[0-9]+\\s*\\]/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[4];
        $text = $_[1];
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/\\[\\s*[0-9]+\\s*\\]/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:\[\s*[0-9]+\s*\])/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/\\[\\s*[0-9]+\\s*\\]/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['[]']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[5];
        $text = $_[1];
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['[]']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\[\]/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['[]']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[6];
        
        my $_savetext;
        @item = (q{step});
        %item = (__RULE__ => q{step});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{step},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{step},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{step},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{step},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::toplevel_selector
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"toplevel_selector"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [toplevel_selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{toplevel_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?!>>)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?!>>)/ '+' selector]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{toplevel_selector});
        %item = (__RULE__ => q{toplevel_selector});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?!>>)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?!>>))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: ['+']},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{toplevel_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{'+'})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_toplevel_selector, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: ['+']>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{toplevel_selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_toplevel_selector]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_toplevel_selector(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{toplevel_selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{selector})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selector]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{toplevel_selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selector]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selector}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { 
      $return = $item[3]; 
      $return->{overlapping}=$item[2][0] ? 1 : 0;
    };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?!>>)/ '+' selector]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{toplevel_selector});
        %item = (__RULE__ => q{toplevel_selector});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{toplevel_selector},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{toplevel_selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{toplevel_selector},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{toplevel_selector},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::expression
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"expression"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'\{', or <leftop: term OP term>, or VAR_OR_SELF});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['\{' <commit> <leftop: expression ',' expression> '\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{expression});
        %item = (__RULE__ => q{expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\{']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\{/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: expression ',' expression>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{<leftop: expression ',' expression>})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{expression}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [',']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{','})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\,/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

            pop @item;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [expression]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{expression})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::expression($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [expression]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [expression]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{expression}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: expression ',' expression>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: expression ',' expression>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE2__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['\}']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'\}'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\}/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING3__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { ['SET',@{$item[3]}] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['\{' <commit> <leftop: expression ',' expression> '\}']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<leftop: term OP term>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{expression});
        %item = (__RULE__ => q{expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying operator: [<leftop: term OP term>]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);

        $_tok = undef;
        OPLOOP: while (1)
        {
          $repcount = 0;
          my @item;
          my %item;

          # MATCH LEFTARG
          
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{term}} = $_tok;
        push @item, $_tok;
        
        }



          $repcount++;

          my $savetext = $text;
          my $backtrack;

          # MATCH (OP RIGHTARG)(s)
          while ($repcount < 100000000)
          {
            $backtrack = 0;
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [OP]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{OP})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::OP($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [OP]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [OP]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{OP}} = $_tok;
        push @item, $_tok;
        
        }

            $backtrack=1;
            
            
        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [term]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{term})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::term($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [term]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [term]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{term}} = $_tok;
        push @item, $_tok;
        
        }

            $savetext = $text;
            $repcount++;
          }
          $text = $savetext;
          pop @item if $backtrack;

          unless (@item) { undef $_tok; last }
          $_tok = [ @item ];

          last;
        } # end of OPLOOP

        unless ($repcount>=1)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match operator: [<leftop: term OP term>]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched operator: [<leftop: term OP term>]<< (return value: [}
                      . qq{@{$_tok||[]}} . q{]},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;

        push @item, $item{__DIRECTIVE1__}=$_tok||[];

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { @{$item[1]}>1 ? ['EXP',@{$item[1]}] : $item[1][0] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<leftop: term OP term>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [VAR_OR_SELF]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{expression});
        %item = (__RULE__ => q{expression});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [VAR_OR_SELF]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{expression},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::VAR_OR_SELF($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [VAR_OR_SELF]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{expression},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [VAR_OR_SELF]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{VAR_OR_SELF}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { '$'.$item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [VAR_OR_SELF]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        
        my $_savetext;
        @item = (q{expression});
        %item = (__RULE__ => q{expression});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{expression},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{expression},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{expression},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{expression},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::not_clause
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"not_clause"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [not_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{not_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{'!', or simple_test});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['!' <commit> simple_test]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{not_clause});
        %item = (__RULE__ => q{not_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['!']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\!/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [simple_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{not_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{simple_test})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::simple_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [simple_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{not_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [simple_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{simple_test}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do {  $return = $item[3];
                 if ($return->{'#name'} eq 'and') {
                   $return->{'#name'} = 'not';
                 } else {
                   $return = new_node({ '#name' => 'not'}, [$return])
                 }
              };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['!' <commit> simple_test]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [simple_test]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{not_clause});
        %item = (__RULE__ => q{not_clause});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [simple_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{not_clause},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::simple_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [simple_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{not_clause},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [simple_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{simple_test}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [simple_test]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{not_clause},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{not_clause},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{not_clause},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{not_clause},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::CMP
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"CMP"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [CMP]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{CMP},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/!?=|!?~\\*?|[<>]=?/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/!?=|!?~\\*?|[<>]=?/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{CMP},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{CMP});
        %item = (__RULE__ => q{CMP});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/!?=|!?~\\*?|[<>]=?/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{CMP},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:!?=|!?~\*?|[<>]=?)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/!?=|!?~\\*?|[<>]=?/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{CMP},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{CMP},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{CMP},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{CMP},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{CMP},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_output_filters
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_1_of_production_1_of_rule_output_filters"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_1_of_production_1_of_rule_output_filters]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_1_of_production_1_of_rule_output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{filter, or return});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [filter]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_output_filters});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_output_filters});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [filter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::filter($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [filter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_output_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [filter]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{filter}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [filter]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [return]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_output_filters});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_output_filters});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [return]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{_alternation_1_of_production_1_of_rule_output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::return($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [return]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{_alternation_1_of_production_1_of_rule_output_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [return]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{return}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [return]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error...>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        
        my $_savetext;
        @item = (q{_alternation_1_of_production_1_of_rule_output_filters});
        %item = (__RULE__ => q{_alternation_1_of_production_1_of_rule_output_filters});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if (1) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error...>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_1_of_production_1_of_rule_output_filters},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_1_of_production_1_of_rule_output_filters},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_2_of_production_1_of_rule_return
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"_alternation_2_of_production_1_of_rule_return"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [_alternation_2_of_production_1_of_rule_return]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{_alternation_2_of_production_1_of_rule_return},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{';'});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [';']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{_alternation_2_of_production_1_of_rule_return},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{_alternation_2_of_production_1_of_rule_return});
        %item = (__RULE__ => q{_alternation_2_of_production_1_of_rule_return});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [';']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_2_of_production_1_of_rule_return},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\;/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [';']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{_alternation_2_of_production_1_of_rule_return},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{_alternation_2_of_production_1_of_rule_return},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{_alternation_2_of_production_1_of_rule_return},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{_alternation_2_of_production_1_of_rule_return},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{_alternation_2_of_production_1_of_rule_return},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::simple_test
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"simple_test"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [simple_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{predicate, or subquery_test, or ref, or '('});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [predicate]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{simple_test});
        %item = (__RULE__ => q{simple_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [predicate]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::predicate($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [predicate]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{simple_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [predicate]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{predicate}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [predicate]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [subquery_test]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        $text = $_[1];
        my $_savetext;
        @item = (q{simple_test});
        %item = (__RULE__ => q{simple_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [subquery_test]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::subquery_test($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [subquery_test]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{simple_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [subquery_test]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{subquery_test}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [subquery_test]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [ref]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[2];
        $text = $_[1];
        my $_savetext;
        @item = (q{simple_test});
        %item = (__RULE__ => q{simple_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [ref]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::ref($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [ref]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{simple_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [ref]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{ref}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [ref]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: ['(' comma_clause ')']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[3];
        $text = $_[1];
        my $_savetext;
        @item = (q{simple_test});
        %item = (__RULE__ => q{simple_test});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['(']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\(/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [comma_clause]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{simple_test},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{comma_clause})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::comma_clause($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [comma_clause]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{simple_test},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [comma_clause]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{comma_clause}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [')']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{')'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = $item[2] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: ['(' comma_clause ')']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{simple_test},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{simple_test},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{simple_test},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{simple_test},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::output_filters
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"output_filters"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [output_filters]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{filter});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [filter return <commit> filter, or return]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{output_filters});
        %item = (__RULE__ => q{output_filters});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [filter]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::filter, 0, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [filter]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{output_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [filter]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{filter(s?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [return]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{return})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::return($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [return]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{output_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [return]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{return}} = $_tok;
        push @item, $_tok;
        
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [filter, or return]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{output_filters},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{filter, or return})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::_alternation_1_of_production_1_of_rule_output_filters, 0, 100000000, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [filter, or return]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{output_filters},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [_alternation_1_of_production_1_of_rule_output_filters]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{_alternation_1_of_production_1_of_rule_output_filters(s?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { Treex::PML::Factory->createList([@{$item[1]}, $item[2], @{$item[4]}],1) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [filter return <commit> filter, or return]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{output_filters});
        %item = (__RULE__ => q{output_filters});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{output_filters},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{output_filters},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{output_filters},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{output_filters},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::name_assignment
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"name_assignment"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [name_assignment]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{name_assignment},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{selector_name});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [selector_name ':=' <commit> '[']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{name_assignment});
        %item = (__RULE__ => q{name_assignment});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [selector_name]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{name_assignment},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{})->at($text);
        unless (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector_name($thisparser,$text,$repeating,$_noactions,sub { \@arg },undef)))
        {
            
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match subrule: [selector_name]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{name_assignment},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched subrule: [selector_name]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{selector_name}} = $_tok;
        push @item, $_tok;
        
        }

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [':=']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{':='})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\:\=/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['[']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'['})->at($text);
        $_savetext = $text;

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\[/)
        {
            $text = $_savetext;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        $text = $_savetext;

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { $return = $item[1] };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [selector_name ':=' <commit> '[']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{name_assignment});
        %item = (__RULE__ => q{name_assignment});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{name_assignment},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{name_assignment},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{name_assignment},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{name_assignment},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::end_of_string
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"end_of_string"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [end_of_string]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{end_of_string},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/^\\Z/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/^\\Z/]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{end_of_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{end_of_string});
        %item = (__RULE__ => q{end_of_string});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/^\\Z/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{end_of_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:^\Z)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/^\\Z/]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{end_of_string},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{end_of_string},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{end_of_string},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{end_of_string},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
                      Parse::RecDescent::_Runtime::_tracemax(substr($_[1],0,-length($text))) . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      , q{end_of_string},
                      $tracelevel)
    }
    $_[1] = $text;
    return $return;
}

# ARGS ARE: ($parser, $text; $repeating, $_noactions, \@args, $_itempos)
sub Parse::RecDescent::_Runtime::Tree_Query::Grammar::selector
{
	my $thisparser = $_[0];
	use vars q{$tracelevel};
	local $tracelevel = ($tracelevel||0)+1;
	$ERRORS = 0;
    my $thisrule = $thisparser->{"rules"}{"selector"};

    Parse::RecDescent::_Runtime::_trace(q{Trying rule: [selector]},
                  Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                  q{selector},
                  $tracelevel)
                    if defined $::RD_TRACE;

    
    my $err_at = @{$thisparser->{errors}};

    my $score;
    my $score_return;
    my $_tok;
    my $return = undef;
    my $_matched=0;
    my $commit=0;
    my @item = ();
    my %item = ();
    my $repeating =  $_[2];
    my $_noactions = $_[3];
    my @arg =    defined $_[4] ? @{ &{$_[4]} } : ();
    my $_itempos = $_[5];
    my %arg =    ($#arg & 01) ? @arg : (@arg, undef);
    my $text;
    my $lastsep;
    my $current_match;
    my $expectation = new Parse::RecDescent::_Runtime::Expectation(q{/(?!>>)/});
    $expectation->at($_[1]);
    
    my $thisoffset;
    tie $thisoffset, q{Parse::RecDescent::_Runtime::OffsetCounter}, \$text, $thisparser;
    
    my $thisline;
    tie $thisline, q{Parse::RecDescent::_Runtime::LineCounter}, \$text, $thisparser;

    

    while (!$_matched && !$commit)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [/(?!>>)/ type name_assignment '[' content_look_ahead <commit> conditions /,?(?:\\#[^\\n]*\\n|\\s+)*/ ']']},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[0];
        $text = $_[1];
        my $_savetext;
        @item = (q{selector});
        %item = (__RULE__ => q{selector});
        my $repcount = 0;


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/(?!>>)/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:(?!>>))/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [type]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{type})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::type, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [type]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [type]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{type(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [name_assignment]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{name_assignment})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::name_assignment, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [name_assignment]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [name_assignment]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{name_assignment(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: ['[']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{'['})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\[/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING1__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying subrule: [content_look_ahead]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        if (1) { no strict qw{refs};
        $expectation->is(q{content_look_ahead})->at($text);
        $_savetext = $text;if (defined ($_tok = Parse::RecDescent::_Runtime::Tree_Query::Grammar::content_look_ahead($thisparser,$text,$repeating,1,sub { \@arg },undef)))
        {
            $text = $_savetext;
            Parse::RecDescent::_Runtime::_trace(q{<<...!Matched(reject) subrule: [content_look_ahead]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            $expectation->failed();
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>...!Didn't match(keep) subrule: [content_look_ahead]<< (return value: [}
                    . $_tok . q{]},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{content_look_ahead}} = $_tok;
        push @item, $_tok;
        $text = $_savetext;
        }

        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<commit>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { $commit = 1 };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying repeated subrule: [conditions]},
                  Parse::RecDescent::_Runtime::_tracefirst($text),
                  q{selector},
                  $tracelevel)
                    if defined $::RD_TRACE;
        $expectation->is(q{conditions})->at($text);
        
        unless (defined ($_tok = $thisparser->_parserepeat($text, \&Parse::RecDescent::_Runtime::Tree_Query::Grammar::conditions, 0, 1, $_noactions,$expectation,sub { \@arg },undef)))
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match repeated subrule: [conditions]>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text),
                          q{selector},
                          $tracelevel)
                            if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched repeated subrule: [conditions]<< (}
                    . @$_tok . q{ times)},

                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $item{q{conditions(?)}} = $_tok;
        push @item, $_tok;
        


        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [/,?(?:\\#[^\\n]*\\n|\\s+)*/]}, Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{/,?(?:\\#[^\\n]*\\n|\\s+)*/})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A(?:,?(?:\#[^\n]*\n|\s+)*)/)
        {
            $text = $lastsep . $text if defined $lastsep;
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;

            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                    if defined $::RD_TRACE;
        push @item, $item{__PATTERN2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying terminal: [']']},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $lastsep;
        $expectation->is(q{']'})->at($text);
        

        unless ($text =~ s/\A($skip)/$lastsep=$1 and ""/e and   $text =~ m/\A\]/)
        {
            $text = $lastsep . $text if defined $lastsep;
            
            $expectation->failed();
            Parse::RecDescent::_Runtime::_trace(qq{<<Didn't match terminal>>},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
            last;
        }
        $current_match = substr($text, $-[0], $+[0] - $-[0]);
        substr($text,0,length($current_match),q{});
        Parse::RecDescent::_Runtime::_trace(q{>>Matched terminal<< (return value: [}
                        . $current_match . q{])},
                          Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        push @item, $item{__STRING2__}=$current_match;
        

        Parse::RecDescent::_Runtime::_trace(q{Trying action},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        

        $_tok = ($_noactions) ? 0 : do { new_node({ '#name' => 'node', 'node-type' => $item[2][0], name => $item[3][0]}, $item[7][0]) };
        unless (defined $_tok)
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match action>> (return value: [undef])})
                    if defined $::RD_TRACE;
            last;
        }
        Parse::RecDescent::_Runtime::_trace(q{>>Matched action<< (return value: [}
                      . $_tok . q{])},
                      Parse::RecDescent::_Runtime::_tracefirst($text))
                        if defined $::RD_TRACE;
        push @item, $_tok;
        $item{__ACTION1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [/(?!>>)/ type name_assignment '[' content_look_ahead <commit> conditions /,?(?:\\#[^\\n]*\\n|\\s+)*/ ']']<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    while (!$_matched)
    {
        
        Parse::RecDescent::_Runtime::_trace(q{Trying production: [<error?:...> <reject>]},
                      Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        my $thisprod = $thisrule->{"prods"}[1];
        
        my $_savetext;
        @item = (q{selector});
        %item = (__RULE__ => q{selector});
        my $repcount = 0;


        

        Parse::RecDescent::_Runtime::_trace(q{Trying directive: [<error?:...>]},
                    Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE; 
        $_tok = do { if ($commit) { do {
        my $rule = $item[0];
           $rule =~ s/_/ /g;
        #WAS: Parse::RecDescent::_Runtime::_error("Invalid $rule: " . $expectation->message() ,$thisline);
        push @{$thisparser->{errors}}, ["Invalid $rule: " . $expectation->message() ,$thisline];
        } unless  $_noactions; undef } else {0} };
        if (defined($_tok))
        {
            Parse::RecDescent::_Runtime::_trace(q{>>Matched directive<< (return value: [}
                        . $_tok . q{])},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        else
        {
            Parse::RecDescent::_Runtime::_trace(q{<<Didn't match directive>>},
                        Parse::RecDescent::_Runtime::_tracefirst($text))
                            if defined $::RD_TRACE;
        }
        
        last unless defined $_tok;
        push @item, $item{__DIRECTIVE1__}=$_tok;
        

        Parse::RecDescent::_Runtime::_trace(q{>>Rejecting production<< (found <reject>)},
                     Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        undef $return;
        

        $_tok = undef;
        
        last unless defined $_tok;

        Parse::RecDescent::_Runtime::_trace(q{>>Matched production: [<error?:...> <reject>]<<},
                      Parse::RecDescent::_Runtime::_tracefirst($text),
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;



        $_matched = 1;
        last;
    }


    unless ( $_matched || defined($score) )
    {
        

        $_[1] = $text;  # NOT SURE THIS IS NEEDED
        Parse::RecDescent::_Runtime::_trace(q{<<Didn't match rule>>},
                     Parse::RecDescent::_Runtime::_tracefirst($_[1]),
                     q{selector},
                     $tracelevel)
                    if defined $::RD_TRACE;
        return undef;
    }
    if (!defined($return) && defined($score))
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Accepted scored production<<}, "",
                      q{selector},
                      $tracelevel)
                        if defined $::RD_TRACE;
        $return = $score_return;
    }
    splice @{$thisparser->{errors}}, $err_at;
    $return = $item[$#item] unless defined $return;
    if (defined $::RD_TRACE)
    {
        Parse::RecDescent::_Runtime::_trace(q{>>Matched rule<< (return value: [} .
                      $return . q{])}, "",
                      q{selector},
                      $tracelevel);
        Parse::RecDescent::_Runtime::_trace(q{(consumed: [} .
               