/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <sysexits.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <mail.h>
#include "scheduler.h"
#include NDIR_H

#if	defined(USE_INET) && (defined(USE_HOSTS) || defined(USE_RESOLV) || defined(USE_YP))
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <sys/file.h>
#endif	/* USE_INET */

#ifdef sgi
flock() { return 0; }
#endif

/*
 * This program must be installed suid to the uid the scheduler runs as
 * (usually root).  Unfortunately.
 */

char	*progname;
char	*postoffice;

int	debug, verbose, user, onlyuser;

char	*optarg;
int	optind;

int
main(argc, argv)
	int argc;
	char *argv[];
{
	char *path;
	int fd, c, errflg, status, eval;
	struct passwd *pw;
#ifdef	AF_INET
	short port = 0;
	u_long naddr;
	char *host = NULL;
	struct hostent *hp, he;
	struct sockaddr_in sad;
	struct servent *serv = NULL;
#else	/* !AF_INET */
	char *rendezvous;
	FILE *fp;
	struct stat stbuf;
	int r, pid, dsflag;
#endif	/* AF_INET */
	extern char *whathost(), *dottedquad();
	extern int errno, pokedaemon();
	extern void prversion(), checkrouter(), checkscheduler();
	extern void docat();

	progname = argv[0];
	verbose = debug = errflg = status = user = onlyuser = 0;
	while ((c = getopt(argc, argv, "dip:stu:vV")) != EOF) {
		switch (c) {
		case 'd':
			++debug;
			break;
		case 'i':
			user = getuid();
			onlyuser = 1;
			if (verbose == 0)
				++verbose;
			break;
#ifdef	AF_INET
		case 'p':
			if ((port = (short)atoi(optarg)) <= 0) {
				(void) fprintf(stderr, "%s: illegal port: %s\n",
						progname, optarg);
				++errflg;
			} else
				port = htons(port);
			break;
#endif	/* AF_INET */
		case 's':
			++status;
			break;
		case 't':
			verbose = 0;
			break;
		case 'u':
			if (optarg == NULL) {
				++errflg;
				break;
			}
			if ((pw = getpwnam(optarg)) == NULL) {
				fprintf(stderr, "%s: unknown user '%s'\n",
					progname, optarg);
				++errflg;
				break;
			}
			user = pw->pw_uid;
			onlyuser = 1;
			if (verbose == 0)
				++verbose;
			break;
		case 'v':
			++verbose;
			break;
		case 'V':
			prversion("mailq");
			exit(0);
			break;
		default:
			++errflg;
			break;
		}
	}
	if (optind < argc) {
#ifdef	AF_INET
			if (optind != argc - 1) {
				fprintf(stderr, "%s: too many hosts\n",
						progname);
				++errflg;
			} else
				host = argv[optind];
#else	/* !AF_INET */
			fprintf(stderr, "%s: not compiled with AF_INET\n",
					progname);
			++errflg;
#endif	/* AF_INET */
	}
	if (errflg) {
#ifdef	AF_INET
		fprintf(stderr, "Usage: %s [-isvt] [-p#] [host]\n", progname);
#else	/* !AF_INET */
		fprintf(stderr, "Usage: %s [-isvt]\n", progname);
#endif	/* AF_INET */
		exit(EX_USAGE);
	}
	if ((postoffice = getzenv("POSTOFFICE")) == NULL)
		postoffice = POSTOFFICE;
	path = emalloc((unsigned)(strlen(postoffice)+1+strlen(PID_SCHEDULER)+1));
	(void) sprintf(path, "%s/%s", postoffice, PID_SCHEDULER);
	errno = 0;
#ifdef	AF_INET
	if (port == 0 && (serv = getservbyname("mailq", "tcp")) == NULL) {
		fprintf(stderr, "%s: cannot find 'mailq' tcp service\n",
				progname);
		exit(EX_TEMPFAIL);
	} else if (port == 0)
		port = serv->s_port;
	if (host == NULL) {
		if (((host = getzenv("MAILSERVER")) == NULL || *host == '\n')
		    && (host = whathost(path)) == NULL) {
			if (status > 0)
				host = "localhost";
			else {
				if (whathost(postoffice)) {
					fprintf(stderr, "%s: %s is not active",
							progname, postoffice);
					fprintf(stderr,
						" (\"%s\" does not exist)\n",
						path);
				} else
					fprintf(stderr,
					    "%s: cannot find postoffice host\n",
						progname);
				(void) free(path);
				exit(EX_OSFILE);
			}
		}
	}
	if ((hp = gethostbyname(host)) == NULL) {
		if ((naddr = inet_addr(host)) != (u_long)-1) {
			naddr = ntohl(naddr);
			if ((hp = gethostbyaddr(&naddr, 4, AF_INET)) == NULL) {
				hp = &he;
				he.h_name = host;
				he.h_length = 4;
			}
		} else if (hp == NULL) {
			fprintf(stderr, "%s: cannot find address of %s\n",
					progname, host);
			exit(EX_UNAVAILABLE);
		}
	}
	if (strcmp(host, "localhost") != 0)
		(void) printf("[%s]\n", hp->h_name);
	if (status) {
		checkrouter();
		checkscheduler();
		if (status > 1)
			exit(0);
	}
	/* try grabbing a port */
	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf(stderr, "%s: ", progname);
		perror("socket");
		exit(EX_UNAVAILABLE);
	}
	sad.sin_family = AF_INET;
	if (hp == &he)
		sad.sin_addr.s_addr = htonl(naddr);
	else {
		char *addr;
		extern void hp_init();
		extern char **hp_getaddr(), **hp_nextaddr();

		eval = EFAULT;
		hp_init(hp);
		while ((addr = *hp_getaddr()) != NULL) {
			sad.sin_addr.s_addr = 0;
			sad.sin_port = htons((short)0);
			if (bind(fd, (caddr_t)&sad, sizeof sad) < 0) {
				(void) fprintf(stderr, "%s: ", progname);
				perror("bind");
				exit(EX_UNAVAILABLE);
			}
			sad.sin_port = port;
			bcopy(addr, (char *)&sad.sin_addr, hp->h_length);
			if (connect(fd, (char *)&sad, sizeof sad) < 0) {
				eval = errno;
				(void) fprintf(stderr,
					       "%s: connect failed to %s",
						progname,
						dottedquad(&sad.sin_addr));
				if ((addr = *hp_nextaddr()) != NULL) {
					bcopy(addr, (char *)&sad.sin_addr,
						    hp->h_length);
					(void) fprintf(stderr,
						       ", trying %s...\n",
						       dottedquad(&sad.sin_addr));
				} else
					(void) putc('\n', stderr);
				(void) close(fd);
				if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0){
					fprintf(stderr, "%s: ", progname);
					perror("socket");
					exit(EX_UNAVAILABLE);
				}
				continue;
			}
			break;
		}
		if (*hp_getaddr() == NULL) {
			fprintf(stderr, "%s: ", progname);
			eval = errno;
			perror("connect");
			fprintf(stderr,
				"%s: unable to contact scheduler on %s\n",
				progname, hp->h_name);
			exit(EX_UNAVAILABLE);
		}
	}
	docat((char *)NULL, fd);
#else	/* !AF_INET */
	if (status) {
		checkrouter();
		checkscheduler();
		if (status > 1)
			exit(0);
	}
	r = isalive(PID_SCHEDULER, &pid, &fp);
	if (r == EX_OSFILE)
		exit(r);
	else if (r == EX_UNAVAILABLE) {
		fprintf(stderr, "%s: no active scheduler process\n", progname);
		exit(r);
	} else if (fp != NULL)
		(void) fclose(fp);
	if (rendezvous == NULL && (rendezvous = getzenv("RENDEZVOUS")) == NULL) {
		extern char *qoutputfile;
		
		rendezvous = qoutputfile;
	}
#ifdef	S_IFIFO
	if (stat(rendezvous, &stbuf) < 0) {
		extern char *strerror();

		(void) unlink(rendezvous);
		if (mknod(rendezvous, S_IFIFO|0666, 0) < 0) {
			fprintf(stderr, "%s: mknod: %s\n",
					progname, strerror(errno));
			exit(EX_UNAVAILABLE);
		}
		stbuf.st_mode |= S_IFIFO;	/* cheat on the next test... */
	}
	if (stbuf.st_mode & S_IFIFO) {
		if ((fd = open(rendezvous, O_RDONLY|O_NDELAY, 0)) < 0) {
			fprintf(stderr, "%s: %s: %s\n", progname,
					rendezvous, strerror(errno));
			exit(EX_OSFILE);
		}
		dsflag = fcntl(fd, F_GETFL, 0);
		dsflag &= ~O_NDELAY;
		(void) fcntl(fd, F_SETFL, dsflag);
		pokedaemon(pid);
		/* XX: reset uid in case we are suid - we need to play games */
		sleep(1);	/* this makes it work reliably. WHY??! */
		docat((char *)NULL, fd);
	} else
#endif	/* S_IFIFO */
	{
		pokedaemon(pid);
		/* XX: reset uid in case we are suid */
		/* sleep until mtime < ctime */
		do {
			sleep(1);
			if (stat(rendezvous, &stbuf) < 0)
				continue;
			if (stbuf.st_mtime < stbuf.st_ctime)
				break;
		} while (1);
		docat(rendezvous, -1);
	}
#endif	/* AF_INET */
	exit(EX_OK);
	/* NOTREACHED */
	return 0;
}

void
docat(file, fd)
	char *file;
	int fd;
{
	char buf[BUFSIZ];
	int n;
	FILE *fp;
	extern char *strerror();
	extern void report();

	if (fd < 0 && (fp = fopen(file, "r")) == NULL) {
		(void) fprintf(stderr, "%s: %s: %s\n",
				progname, file, strerror(errno));
		exit(EX_OSFILE);
		/* NOTREACHED */
	} else if (fd >= 0)
		fp = fdopen(fd, "r");
	if (debug)
		while ((n = fread(buf, sizeof buf[0], sizeof buf, fp)) > 0)
			(void) fwrite(buf, sizeof buf[0], n, stdout);
	else
		report(fp);
	(void) fclose(fp);
}

/*
 * Determine if the Router is alive, how many entries are in the queue,
 * and whether the router dumped core last time it died.
 */
 
void
checkrouter()
{
	int pid, n, r;
	FILE *fp;
	struct stat pidbuf, corebuf;
	struct NDIR_TYPE *dp;
	DIR *dirp;
	char *path;
	extern int isalive();

	if (postoffice == NULL)
		return;
	path = emalloc((unsigned)(strlen(postoffice)+1+strlen(ROUTERDIR)+1));
	(void) sprintf(path, "%s/%s", postoffice, ROUTERDIR);
	if ((dirp = opendir(path)) == NULL) {
		fprintf(stderr, "%s: opendir(%s):", progname, path);
		perror(" ");
		(void) free(path);
		return;
	}
	(void) free(path);
	for (dp = readdir(dirp), n = 0; dp != NULL; dp = readdir(dirp)) {
		if (isascii(dp->d_name[0]) && isdigit(dp->d_name[0]))
			++n;
	}
	(void) closedir(dirp);
	(void) printf("%d entr%s in router queue: ", n, n != 1 ? "ies" : "y");

	r = isalive(PID_ROUTER, &pid, &fp);
	switch (r) {
	case EX_UNAVAILABLE:
		/* if the .router.pid file is younger than any core file,
		   then the router dumped core... so let'em know about it. */
		path = emalloc((unsigned)(strlen(postoffice)+strlen(ROUTERDIR)+sizeof "//core"+1));
		(void) sprintf(path, "%s/%s/core",postoffice,ROUTERDIR);
		if (fstat(fileno(fp), &pidbuf) < 0) {
			(void) fprintf(stderr, "\n%s: ", progname);
			perror("fstat: ");
		} else if (stat(path, &corebuf) == 0
			   && pidbuf.st_mtime < corebuf.st_mtime)
			(void) printf("core dumped\n");
		else
			(void) printf("no daemon\n");
		(void) fclose(fp);
		free(path);
		break;
	case EX_OK:
		if (n)
			printf("processing\n");
		else
			printf("idle\n");
		(void) fclose(fp);
		break;
	default:
		printf("never started\n");
		break;
	}

	path = emalloc((unsigned)(strlen(postoffice)+1+strlen(DEFERREDDIR)+1));
	(void) sprintf(path, "%s/%s", postoffice, DEFERREDDIR);
	if ((dirp = opendir(path)) == NULL) {
		(void) fprintf(stderr, "%s: opendir(%s):", progname, path);
		perror(" ");
		(void) free(path);
		return;
	}
	(void) free(path);
	for (dp = readdir(dirp), n = 0; dp != NULL; dp = readdir(dirp)) {
		if (isascii(dp->d_name[0]) && isdigit(dp->d_name[0]))
			++n;
	}
	(void) closedir(dirp);
	if (n)
		printf("%d message%s deferred\n", n, n != 1 ? "s" : "");
}

void
checkscheduler()
{
	int pid, n, tn, r, sawcore;
	FILE *fp;
	struct stat corebuf;
	struct NDIR_TYPE *dp;
	DIR *dirp;
	char *path;
	extern int isalive();

	if (postoffice == NULL)
		return;
	path = emalloc((unsigned)(strlen(postoffice)+1+strlen(SCHEDULERDIR)+1));
	(void) sprintf(path, "%s/%s", postoffice, SCHEDULERDIR);
	if ((dirp = opendir(path)) == NULL) {
		(void) fprintf(stderr, "%s: opendir(%s):", progname, path);
		perror(" ");
		free(path);
		return;
	}
	free(path);
	for (dp = readdir(dirp), n = tn = 0; dp != NULL; dp = readdir(dirp)) {
		/* beware of directories that fit the pattern... */
		if (isascii(dp->d_name[0]) && isdigit(dp->d_name[0]))
			++n;
		else
			++tn;
	}
	(void) closedir(dirp);
	tn -= 2; /* . and .. */
	(void) printf("%d entr%s in scheduler queue: ", n, n != 1 ? "ies" : "y");

	path = emalloc((unsigned)(strlen(postoffice)+strlen(SCHEDULERDIR)+sizeof "//core"+1));
	(void) sprintf(path, "%s/%s/core", postoffice, SCHEDULERDIR);
	sawcore = (stat(path, &corebuf) == 0);
	free(path);
	tn -= sawcore;

	r = isalive(PID_SCHEDULER, &pid, &fp);
	switch (r) {
	case EX_UNAVAILABLE:
		(void) printf("no daemon");
		(void) fclose(fp);
		break;
	case EX_OK:
		if (n) {
			if (tn > 0)
				printf("busy");
			else
				printf("processing control files");
		} else if (tn > 0)
			printf("scheduling transports");
		else
			printf("idle");
		(void) fclose(fp);
		break;
	default:
		printf("never started");
		if (tn > 0)
			printf(" \"%s/%s\" polluted", postoffice, SCHEDULERDIR);
		break;
	}
	if (sawcore)
		printf(" (core exists)");
	printf("\n");

	path = emalloc((unsigned)(strlen(postoffice)+1+strlen(TRANSPORTDIR)+1));
	(void) sprintf(path, "%s/%s", postoffice, TRANSPORTDIR);
	if ((dirp = opendir(path)) == NULL) {
		fprintf(stderr, "%s: opendir(%s):", progname, path);
		perror(" ");
		(void) free(path);
		return;
	}
	(void) free(path);
	for (dp = readdir(dirp), n = 0; dp != NULL; dp = readdir(dirp)) {
		if (isascii(dp->d_name[0]) && isdigit(dp->d_name[0]))
			++n;
	}
	(void) closedir(dirp);
	if (n)
		(void) printf("%d message%s in transport queue\n",
			   n, n != 1 ? "s" : "");
}

int
isalive(pidfile, pidp, fpp)
	char *pidfile;
	int *pidp;
	FILE **fpp;
{
	char *path;
	extern int errno;

	if (postoffice == NULL)
		return 0;
	path = emalloc((unsigned)(strlen(postoffice)+1+strlen(pidfile)+1));
	(void) sprintf(path, "%s/%s", postoffice, pidfile);
	
	if ((*fpp = fopen(path, "r")) == NULL) {
		/* (void) fprintf(stderr, "%s: cannot open %s (%s)\n",
			progname, path, strerror(errno)); */
		(void) free(path);
		return EX_OSFILE;
	}
	(void) free(path);
	if (fscanf(*fpp, "%d", pidp) != 1) {
		(void) fprintf(stderr, "%s: cannot read process id\n", progname);
		(void) fclose(*fpp);
		*fpp = NULL;
		return EX_OSFILE;
	}
	if (kill(*pidp, 0) < 0 && errno == ESRCH)
		return EX_UNAVAILABLE;
	return EX_OK;
}

#define	MAGIC_PREAMBLE		"version "
#define	LEN_MAGIC_PREAMBLE	(sizeof MAGIC_PREAMBLE - 1)
#define	VERSION_ID		"zmailer 1.0"
#define	GETLINE(fp)		\
	if (fgets(buf, sizeof buf, fp) == NULL) {\
		fprintf(stderr, "%s: no input from scheduler\n", progname);\
		buf[0] = '\0';\
		return 0;\
	}\
	if (buf[strlen(buf)-1] = '\n')\
		buf[strlen(buf)-1] = '\0';
				

char	*names[SIZE_L] = { "Vertices:", "Hosts:", "Channels:" };

#define	L_VERTEX	SIZE_L
struct sptree *spt_ids[SIZE_L+1];

#define	EQNSTR(a,b)	(!strncmp(a,b,strlen(b)))

int
parse(fp)
	FILE *fp;
{
	register char *cp;
	register struct vertex *v;
	register struct web *w;
	register struct ctlfile *cfp;
	register int	i;
	u_int	list, key;
	struct spblk *spl;
	char	buf[8192*8], *ocp;

	GETLINE(fp);
	if (!(EQNSTR(buf, MAGIC_PREAMBLE)
		    && EQNSTR(buf+LEN_MAGIC_PREAMBLE, VERSION_ID))) {
		fprintf(stderr, "%s: version mismatch, input is \"%s\".\n",
				progname, buf);
		return 0;
	}
	GETLINE(fp);
	if (!EQNSTR(buf, names[L_CTLFILE]))
		return 0;
	list = L_CTLFILE;
	spt_ids[L_CTLFILE] = sp_init();
	spt_ids[L_VERTEX] = sp_init();
	spt_ids[L_CHANNEL] = sp_init();
	spt_ids[L_HOST] = sp_init();
	while (fgets(buf, sizeof buf, fp) != NULL) {
		switch (list) {
		case L_CTLFILE:
			/* decid:\tfile\tnaddr; off1[,off2,...][\t#message] */
			if (!isdigit(buf[0]) && EQNSTR(buf, names[L_CHANNEL])) {
				list = L_CHANNEL;
				break;
			}
			if (!isdigit(buf[0])
				    || (cp = strchr(buf, ':')) == NULL) {
				fprintf(stderr,
					"%s: %s: orphaned pending recovery\n",
					progname, buf);
				break;
			}
			*cp++ = '\0';
			key = atoi(buf);
			while (isascii(*cp) && isspace(*cp))
				++cp;
			ocp = cp;
			while (isascii(*cp) && !isspace(*cp))
				++cp;
			*cp++ = '\0';
			spl = sp_lookup(symbol((u_char *)ocp), spt_ids[L_CTLFILE]);
			if (spl == NULL
			    || (cfp = (struct ctlfile *)spl->data) == NULL) {
				cfp = (struct ctlfile *)emalloc(sizeof (struct ctlfile));
				cfp->fd = -1;
				cfp->haderror = 0;
				cfp->head = NULL;
				cfp->nlines = 0;
				cfp->contents = NULL;
				cfp->logident = NULL;
				cfp->id = 0;
				cfp->mid = emalloc(strlen(ocp)+1);
				(void) strcpy(cfp->mid, ocp);
				cfp->mark = 0;
				sp_install(symbol((u_char *)ocp), (u_char *)cfp, 0,
						 spt_ids[L_CTLFILE]);
			}
			while (isascii(*cp) && isspace(*cp))
				++cp;
			ocp = cp;
			while (isascii(*cp) && isdigit(*cp))
				++cp;
			*cp++ = '\0';
			if ((i = atoi(ocp)) < 1) {
				fprintf(stderr,
					"%s: bad number of addresses: '%s'\n",
					progname, ocp);
				break;
			}
			v = (struct vertex *)emalloc(sizeof (struct vertex) +
					    (i - 1) * sizeof (long));
			v->ngroup = i;
			v->cfp = cfp;
			while (isascii(*cp) && isspace(*cp))
				++cp;
			for (i = 0; isascii(*cp) && isdigit(*cp); ++cp) {
				ocp = cp;
				while (isascii(*cp) && isdigit(*cp))
					++cp;
				*cp = '\0';
				v->index[i++] = atoi(ocp);
			}
			while (*cp != '\0' && *cp != '\n' && *cp != '#')
				++cp;
			if (*cp == '#') {
				ocp = ++cp;
				while (*cp != '\0' && *cp != '\n')
					++cp;
				*cp = '\0';
				v->message = emalloc(strlen(ocp)+1);
				(void) strcpy(v->message, ocp);
			} else
				v->message = NULL;
			v->next[L_CTLFILE] = cfp->head;
			if (cfp->head == NULL)
				cfp->head = v;
			else
				cfp->head->prev[L_CTLFILE] = v;
			v->prev[L_CTLFILE] = NULL;
			cfp->head = v;
			v->orig[L_CTLFILE] = v->orig[L_CHANNEL] = v->orig[L_HOST] = NULL;
			v->next[L_CHANNEL] = v->next[L_HOST] = NULL;
			v->prev[L_CHANNEL] = v->prev[L_HOST] = NULL;
			v->nextmark = NULL;
			sp_install(key, (u_char *)v, 0, spt_ids[L_VERTEX]);
			break;
		case L_CHANNEL:
			/* (channel|host):\tdecid[>decid...] */
			if (EQNSTR(buf, names[L_HOST])) {
				list = L_HOST;
				break;
			}
			/* FALL THROUGH */
		case L_HOST:
			cp = buf-1;
			do
				cp = strchr(cp+1, ':');
			while (cp != 0 && (*(cp+1) != '\t' || *(cp+2) != '>'));

			if (cp == NULL) {
				fprintf(stderr,
					"%s: %s: orphaned pending recovery\n",
					progname, buf);
				break;
			}
			*cp++ = '\0';

			spl = sp_lookup(symbol((u_char *)buf), spt_ids[list]);
			if (spl == NULL
			    || (w = (struct web *)spl->data) == NULL) {
				w = (struct web *)emalloc(sizeof (struct web));
				w->name = emalloc(strlen(buf)+1);
				(void) strcpy(w->name, buf);
				w->kids = 0;
				w->link = w->lastlink = NULL;
				w->fifohead = w->fifotail = NULL;
				sp_install(symbol((u_char *)buf),
						 (u_char *)w, 0, spt_ids[list]);
			}
			while (isascii(*cp) && isspace(*cp))
				++cp;
			++cp;	/* skip the first '>' */
			while (*cp != '\0' && isascii(*cp) && isdigit(*cp)) {
				ocp = cp;
				while (isascii(*cp) && isdigit(*cp))
					++cp;
				*cp++ = '\0';
				spl = sp_lookup((u_int)atoi(ocp),
						spt_ids[L_VERTEX]);
				if (spl == NULL
				    || (v = (struct vertex *)spl->data)==NULL) {
					fprintf(stderr, "%s: unknown key %s\n",
							progname, ocp);
				} else {
					if (w->link)
						w->link->prev[list] = v;
					else
						w->lastlink = v;
					v->next[list] = w->link;
					w->link = v;
					if (v->orig[list] == NULL)
						v->orig[list] = w;
				}
			}
			break;
		}
	}
	return 1;
}

static int r_i;

int
repscan(spl)
	struct spblk *spl;
{
	register struct vertex *v, *vv;
	struct web *w;
	int fd, flag;
	char *path;
	struct stat stbuf;
	extern void printaddrs();

	w = (struct web *)spl->data;
	/* assert w != NULL */
	for (vv = w->link; vv != NULL; vv = vv->next[L_CHANNEL]) {
		if (vv->ngroup == 0)
			continue;
		if (!onlyuser)
			(void) printf("%s/%s:\n",
				      w->name, vv->orig[L_HOST]->name);
		else
			flag = 0;
		for (v = vv; v != NULL; v = v->next[L_HOST]) {
			if (v->ngroup == 0)
				continue;
			if (onlyuser) {
				path = emalloc((unsigned)(strlen(postoffice)+1
					+strlen(TRANSPORTDIR)+1
					+strlen(v->cfp->mid)+1));
				(void) sprintf(path, "%s/%s/%s",
					postoffice, TRANSPORTDIR, v->cfp->mid);
				if ((fd = open(path, O_RDONLY, 0)) < 0) {
					(void) free(path);
					continue;
				}
				free(path);
				if (fstat(fd, &stbuf) < 0
				    || stbuf.st_uid != user) {
					(void) close(fd);
					continue;
				}
				(void) close(fd);
				if (flag == 0)
					printf("%s/%s:\n", w->name,
						vv->orig[L_HOST]->name);
			}
			flag = 1;
			printf("\t%s", v->cfp->mid);
			if (v->ngroup > 1)
				printf("/%d", v->ngroup);
			putchar(':');
			if (v->message)
				printf("\t%s\n", v->message);
			else
				putchar('\n');
			if (verbose)
				printaddrs(v);
			for (r_i = 0; r_i < SIZE_L; ++r_i) {
				if (v->next[r_i] != NULL)
					v->next[r_i]->prev[r_i] = v->prev[r_i];
				if (v->prev[r_i] != NULL)
					v->prev[r_i]->next[r_i] = v->next[r_i];
			}
			/* if we are verbose, space becomes important */
			if (v->next[L_CTLFILE] == NULL
			    && v->prev[L_CTLFILE] == NULL) {
				/* we can free the control file */
				if (v->cfp->contents != NULL)
					(void) free(v->cfp->contents);
				free((char *)v->cfp);
			}
			/* we can't free v! so mark it instead */
			v->ngroup = 0;
		}
	}
	return 0;
}

void
report(fp)
	FILE *fp;
{
	if (!parse(fp))
		return;

	r_i = 0;
	sp_scan(repscan, (struct spblk *)NULL, spt_ids[L_CHANNEL]);
	if (!r_i) {
		if (onlyuser)
			(void) printf("No user messages found\n");
		else
			(void) printf("Transport queue is empty\n");
	}
}

void
printaddrs(v)
	struct vertex *v;
{
	register char *cp;
	int	i, fd;
	struct stat stbuf;
	char	*path, *ocp;
	extern int errno;
	extern char *strerror();

	if (v->cfp->contents == NULL) {
		path = emalloc((unsigned)(strlen(postoffice)+1+strlen(TRANSPORTDIR)+1+strlen(v->cfp->mid)+1));
		(void) sprintf(path, "%s/%s/%s",
				     postoffice, TRANSPORTDIR, v->cfp->mid);
		if ((fd = open(path, O_RDONLY, 0)) < 0) {
#if	0
			(void) printf("\t\t%s: %s\n", path, strerror(errno));
#endif
			free(path);
			return;
		}
		if (fstat(fd, &stbuf) < 0) {
			(void) printf("\t\tfstat(%s): %s\n",
				      path, strerror(errno));
			(void) close(fd);
			free(path);
			return;
		}
		free(path);
		v->cfp->contents = emalloc((u_int)stbuf.st_size);
		if (v->cfp->contents == NULL) {
			(void) printf("\t\tmalloc(%d): out of memory!\n",
					  stbuf.st_size);
			(void) close(fd);
			return;
		}
		errno = 0;
		if (read(fd, v->cfp->contents, stbuf.st_size) < stbuf.st_size) {
			(void) printf("\t\tread(%d): %s\n",
				errno == 0 ? "failed" : strerror(errno));
			(void) close(fd);
			return;
		}
		(void) close(fd);
		for (cp = v->cfp->contents, i = 0;
		     cp < v->cfp->contents + stbuf.st_size - 1; ++cp) {
			if (*cp == '\n') {
				*cp = '\0';
				if (*++cp == 's')
					break;
				switch (*cp) {
				case 'l':
					v->cfp->logident = cp + 2;
					break;
				case 'e':
					/* overload cfp->mark to be from addr*/
					v->cfp->mark = cp+2 - v->cfp->contents;
					break;
				}
			}
		}
		if (verbose > 1) {
			path = emalloc((unsigned)(strlen(postoffice)+1+strlen(QUEUEDIR)+1+strlen(v->cfp->mid)+1));
			(void) sprintf(path, "%s/%s/%s",
					     postoffice, QUEUEDIR, v->cfp->mid);
			if (stat(path, &stbuf) == 0)
				/* overload offset[] to be size of message */
				v->cfp->offset[0] = stbuf.st_size;
			else
				v->cfp->offset[0] = 0;
			free(path);
		}
	}
	if (v->cfp->logident)
		printf("\t  id\t%s", v->cfp->logident);
	if (verbose > 1 && v->cfp->offset[0] > 0)
		printf(", %d bytes", v->cfp->offset[0]);
	putchar('\n');
	if (v->cfp->mark > 0)
		printf("\t  from\t%s\n", v->cfp->contents + v->cfp->mark);
	for (i = 0; i < v->ngroup; ++i) {
		cp = v->cfp->contents + v->index[i] + 2;
		while (isascii(*cp) && !isspace(*cp))
			++cp;
		while (isascii(*cp) && isspace(*cp))
			++cp;
		while (isascii(*cp) && !isspace(*cp))
			++cp;
		while (isascii(*cp) && isspace(*cp))
			++cp;
		ocp = cp;
		while (*cp != '\0' && *cp != '\n')
			++cp;
		if (*cp-- == '\n') {
			while (isascii(*cp) && !isspace(*cp))
				--cp;
			while (isascii(*cp) && isspace(*cp))
				--cp;
			if (cp >= ocp)
				*++cp = '\0';
			else {
				cp = ocp;
				while (*cp != '\0' && *cp != '\n')
					++cp;
				*cp = '\0';
			}
		}
		putchar('\t');
		if (i == 0)
			printf("  to");
		putchar('\t');
		(void) puts(ocp);
	}
}

#ifdef	AF_INET

#ifdef	USE_NFS
#ifdef	ultrix
#include <sys/param.h>
#include <sys/mount.h>
#else	/* Sun-derived NFS */
#include <mntent.h>
#endif

/*
 * Given a name like /usr/src/etc/foo.c returns the mount point
 * for the file system it lives in, or NULL in case of any error.
 */
static char *
getmntpt(file, dir)
	char	*file, **dir;
{
#ifdef	ultrix
	int	mountind, nummount;
	struct fs_data	mounts[1];
#else	/* Sun-derived NFS */
	FILE	*mntp;
	struct mntent	*mnt;
#endif	
	struct stat	filestat, dirstat;

	if (stat(file, &filestat) < 0) {
		(void) fprintf(stderr, "%s: ", progname);
		perror(file);
		return(NULL);
	}
#ifdef	ultrix
	mountind = 0;
	while ((nummount = getmountent(&mountind, mounts, 1)) > 0) {
		if ((stat(mounts[0].fd_path, &dirstat) >= 0) &&
		   (filestat.st_dev == dirstat.st_dev)) {
			*dir = mounts[0].fd_path;
			return mounts[0].fd_devname;
		}
	}
	if (nummount == -1)
		perror("Can't get mount information");
#else	/* Sun-derived NFS */
	if ((mntp = setmntent(MOUNTED, "r")) == NULL) {
		fprintf(stderr, "%s: ", progname);
		perror(MOUNTED);
		return(NULL);
	}
	while ((mnt = getmntent(mntp)) != 0) {
		if (strcmp(mnt->mnt_type, MNTTYPE_IGNORE) == 0 ||
		    strcmp(mnt->mnt_type, MNTTYPE_SWAP) == 0)
			continue;
		if ((stat(mnt->mnt_dir, &dirstat) >= 0) &&
		   (filestat.st_dev == dirstat.st_dev)) {
			endmntent(mntp);
			*dir = mnt->mnt_dir;
			return mnt->mnt_fsname;
		}
	}
	endmntent(mntp);
#endif
	return NULL;
}
#endif	/* USE_NFS */

char *
whathost(file)
	char	*file;
{
#ifdef	USE_NFS
	struct stat	statb;
	char		*mnt, *cp, *dir;
#ifndef	MAXHOSTNAMELEN
#define	MAXHOSTNAMELEN 64
#endif	/* MAXHOSTNAMELEN */
	static char	hostname[MAXHOSTNAMELEN+1];
	extern char	*getmntpt();

	if (stat(file, &statb) < 0)
		return NULL;
	if ((statb.st_mode & S_IFMT & (S_IFREG|S_IFDIR))
	    && (mnt = getmntpt(file, &dir)) != NULL) {
		if ((cp = strchr(mnt, ':')) == 0)
			return "localhost";
		/* file is remote - use locking daemon! */
		*cp++ = '\0';
		strcpy(hostname, mnt);
		return hostname;
	}
	return NULL;
#else	/* !USE_NFS */
	return "localhost";
#endif	/* USE_NFS */
}

#else	/* !AF_INET */

pokedaemon(pid)
	int pid;
{
	if (kill(pid, SIGUSR2) < 0) {
		if (errno == EPERM) {
			/* if we can use other types of signals, try here */
			fprintf(stderr, "%s: not owner of scheduler process\n",
					progname);
			exit(EX_NOPERM);
		} else if (errno == ESRCH) {
			fprintf(stderr, "%s: scheduler just died\n", progname);
			exit(EX_UNAVAILABLE);
		} else {
			fprintf(stderr, "%s: cannot signal scheduler (%s)\n",
				progname, strerror(errno));
			exit(EX_UNAVAILABLE);
		}
	}
}
#endif	/* AF_INET */
