/* File: toolbox_tools.c
 * Created: 3/26/92 by John Butare(jb7p@andrew.cmu.edu)
 * Description:
 *	
 * Modifications:
 */

#include <InterViews/bitmap.h>
#include <InterViews/stencil.h>
#include <InterViews/tiff.h>
#include <InterViews/raster.h>
#include <InterViews/image.h>
#include <InterViews/background.h>
#include <InterViews/box.h>
#include <InterViews/deck.h>
#include <InterViews/glue.h>
#include <InterViews/label.h>
#include <InterViews/margin.h>
#include <InterViews/style.h>
#include <InterViews/action.h>
#include <InterViews/border.h>
#include <InterViews/center.h>
     
#include <InterViews/window.h>
#include <InterViews/display.h>

#include <IV-look/kit.h>
     
#include <OS/string.h>
#include <OS/list.h>

#include <stdio.h>
#include <strings.h>

#include <UI/generic_button.h>
#include <UI/toggle_button.h>
#include <UI/radio_button.h>
#include <UI/rbbox.h>
#include <UI/date_editor.h>
#include <UI/kit.h>
#include <UI/text_editor.h>
#include <UI/string_list.h>
#include <UI/list_box.h>

#include "gems.h"
#include "map.h"
#include "toolbox.h"
#include "toolbox_tools.h"
#include "simulation_tools.h"
#include "particle_tracker.h"
#include "query_particle_tracker.h"
#include "color_scale.h"

declare(ActionCallback, ToolboxTool);
implement(ActionCallback, ToolboxTool);
     
/*
 * ToolboxListBox
 */

declare(ActionCallback, ToolboxListBox);
implement(ActionCallback, ToolboxListBox);

declareList(ToolboxEditorList, GemsTextEditor*);
implementList(ToolboxEditorList, GemsTextEditor*);

ToolboxListBox::ToolboxListBox(
       ToolboxTool* toolbox_tool, Coord label_width, 
       Coord list_height, Coord list_width
   ) : Patch(nil)
{
  Gems* gems = Gems::instance();
  Style* style = gems->style();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();

  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);

  _toolbox_tool = toolbox_tool;
  _editor_list = new ToolboxEditorList(10);
  _button_press_action = 
       new ActionCallback(ToolboxListBox)(this, &ToolboxListBox::button_press);

  _labels =
       new RBBox(
       	   new TBBox(
       	       new VGlue(0.0, 0.0, 0.0),
       	       new HGlue(label_width, 0.0, fil)
       	   )
       );

  _ok_action = new ActionCallback(ToolboxListBox)(this, &ToolboxListBox::editor_ok);

  _editors =
       new TBBox(
	 new VGlue(0.0, 0.0, 0.0),
       	 new VGlue(3.0, 0.0, 0.0)
       );

  _parameters = 
       new LRBox(
       	   _labels,
           new HGlue(4.0, 0.0, 0.0),
       	   _editors
       );

  body(_deck = new Deck());

  _deck->append(
       _structure = new VCenter(
       	   new Overlay(
       	       new VGlue(list_height, fil, 0.0),
       	       new HGlue(list_width, fil, 0.0)
       	   ),
       1.0)
   );

  _deck->append(_simulation_toolbox = new SimulationToolbox(this));
}     

void ToolboxListBox::append_list(char* label, StringList* list)
{
  Gems* gems = Gems::instance();
  Style* style = gems->style();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();

  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  GemsTextEditor* ed;

  _labels->append(new LRBox(stretch_hsep, kit->label(label, style), stretch_hsep), style, _button_press_action);
  char* default_text = (list && list->count() ? list->item(0) : "");
  _editor_list->append(ed = gemsKit->text_editor(style, default_text, _ok_action));
  _editors->remove(_editors->count()-1);
  _editors->append(new TMargin(ed, 3.0, fil, 0.0));
  _editors->append(new VGlue(3.0, 0.0, 0.0));
  _deck->append(new VCenter(new ListBox(list, ed, style), 1.0));

  _deck->flip_to(2);
  _labels->choose(1);
}

void ToolboxListBox::replace_list(GlyphIndex i, StringList* list)
{
  Gems* gems = Gems::instance();
  Style* style = gems->style();

  _deck->replace(i+1, new VCenter(new ListBox(list, editor(i), style), 1.0));
  editor(i)->string((list && list->count()) ? list->item(0) : "");
  
  if (choice() == i)
    use(i);
}

ToolboxListBox::~ToolboxListBox()
{ 
  delete _simulation_toolbox;
}

int ToolboxListBox::choice()
{
  return(labels()->chosen());
}

char* ToolboxListBox::value()
{
  if (choice() > 0)
    return(editor(choice())->string());
  else
    Gems::instance()->error("ToolboxListBox::value() no choice for value 0");
}

void ToolboxListBox::editor_ok()
{
  printf("ToolboxListBox::editor_ok()\n");
  printf("\t _labels->chosen()=%d\n", _labels->chosen());
  printf("\t _deck->card()=%d\n", _deck->card());

  if (_deck->card() == 1) /* a simulation is showing */
    _simulation_toolbox->current()->update_value(_labels->chosen());
  else
    _toolbox_tool->ok_cb();
}

void ToolboxListBox::button_press()
{
  printf("ToolboxListBox::button_press\n");
  printf("\t_labels->chosen()=%d\n", choice());

  use(_labels->chosen());
}

GlyphIndex ToolboxListBox::current()
{
  return(_deck->card()-1);
}

GemsTextEditor* ToolboxListBox::editor(GlyphIndex i)
{
  return(_editor_list->item(i-1));
}

void ToolboxListBox::use(GlyphIndex i)
{
  if (choice() != i)
    _labels->choose(i);

  _deck->flip_to(i+1);
  redraw();
  reallocate();
  redraw();
}

/*
 * ToolboxTool
 */

ToolboxTool::ToolboxTool(Coord label_width, Coord list_height, Coord list_width)
{
  _select_action = new ActionCallback(ToolboxTool)(this, &ToolboxTool::select);
  _list_box = new ToolboxListBox(this, label_width, list_height, list_width);
}

void ToolboxTool::append_list(char* label, StringList* list)
{
  _list_box->append_list(label, list);
}

void ToolboxTool::replace_list(GlyphIndex i, StringList* list)
{
  _list_box->replace_list(i, list);
}

void ToolboxTool::select()
{
  _toolbox->use(this);
}

void ToolboxTool::icon(char* bitmap_file_name, char* tiff_file_name)
{
  Gems* gems = Gems::instance();
  Style* style = gems->style();
  char* file_name;

  if (style->value_is_on("pretty") && tiff_file_name) {
    file_name = gems->datapath(tiff_file_name);
    Raster* icon = TIFFRaster::load(file_name, false);
    if (icon) _icon = new Image(icon);
    
  } else if (bitmap_file_name) {
    file_name = gems->datapath(bitmap_file_name);
    Bitmap* icon =  Bitmap::open(file_name);
    if (icon) _icon = new Stencil(icon, style->foreground());
  } else
    _icon = nil;

  if (_icon == nil) {
    char s[200];
    sprintf(s, "Unable to open ToolboxTool icon from file %s.\n", file_name);
    gems->error(s);
  }
  else
    {
      Viewer* view = new Viewer(_icon);
      view->zoom_to_fit(50,50);
      _icon = view;
    }
}

/* 
 * MapViewTool
 */

declare(ActionCallback, MapViewTool);
implement(ActionCallback, MapViewTool);

MapViewTool::MapViewTool(
       Coord label_width, Coord list_height, Coord list_width, 
       char* item1, char* item2,
       char* item3, char* item4,
       char* item5, char* item6,
       char* item7, char* item8,
       char* item9, char* item10
   ) : ToolboxTool(label_width, list_height, list_width)
{
  Gems* gems = Gems::instance();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();
  Style* style = gems->style();

  Glyph* stretch_vsep = new VGlue(2.0, fil, 0.0);
  Glyph* fixed_vsep = new VGlue(2.0, 0.0, 0.0);
  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);

  /*
   * Icon
   */
  icon("pictures/lens.bm", "pictures/Rivers");

  /*
   * Parameters
   */

  Action* cb = new ActionCallback(MapViewTool)(this, &MapViewTool::view_cb);
  ToggleButton *v1, *v2, *v3, *v4, *v5, *v6, *v7, *v8, *v9, *v10;

  v1 = (item1 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item1, style)) : nil);
  v2 = (item2 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item2, style)) : nil);
  v3 = (item3 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item3, style)): nil);
  v4 = (item4 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item4, style)):nil);
  v5 = (item5 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item5, style)):nil);
  v6 = (item6 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item6, style)):nil);
  v7 = (item7 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item7, style)) : nil);
  v8 = (item8 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item8, style)) : nil);
  v9 = (item9 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item9, style)) : nil);
  v10 = (item10 ? gemsKit->toggle_button(style, cb, cb, nil, kit->label(item10, style)) : nil);

  
  _view = new RBBox(nil, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10);
  _view->multiple_selections(true);

  Glyph* box1 = new TBBox(new VGlue(0.0, 0.0, 0.0));
  Glyph* box2 = new TBBox(new VGlue(0.0, 0.0, 0.0));
  
  if (v1) box1->append(new TMargin(v1, 2.0, fil, 0.0));
  if (v2) box2->append(new TMargin(v2, 2.0, fil, 0.0));
  if (v3) box1->append(new TMargin(v3, 2.0, fil, 0.0));
  if (v4) box2->append(new TMargin(v4, 2.0, fil, 0.0));
  if (v5) box1->append(new TMargin(v5, 2.0, fil, 0.0));
  if (v6) box2->append(new TMargin(v6, 2.0, fil, 0.0));
  if (v7) box1->append(new TMargin(v7, 2.0, fil, 0.0));
  if (v8) box2->append(new TMargin(v8, 2.0, fil, 0.0));
  if (v9) box1->append(new TMargin(v9, 2.0, fil, 0.0));
  if (v10) box2->append(new TMargin(v10, 2.0, fil, 0.0));

  box1->append(stretch_vsep);
  box2->append(stretch_vsep);

  Glyph* parameters = 
       new LRBox(
       	   stretch_hsep,
	   box1,
       	   stretch_hsep,
	   box2,
       	   stretch_hsep
       );
  /*
   * ListBox
   */

  _list_box = new ToolboxListBox(this, label_width, list_height, list_width);

  /*
   * Control Buttons
   */

  Glyph* help = gemsKit->push_button("Help", style,
       new ActionCallback(MapViewTool)(this, &MapViewTool::help_cb)
   );

  Glyph* update = gemsKit->push_button("Update", style,
       new ActionCallback(MapViewTool)(this, &MapViewTool::ok_cb)
   );

  Glyph* control_buttons =
       new LRBox(
       	   help,
       	   stretch_hsep,
       	   update
       );
  
  /* 
   * Layout
   */

  Glyph* name = 
       new LRBox(
       	   stretch_hsep, 
       	   kit->label("Map View", style),
	   stretch_hsep
       );
       
  _tool = new Background(
       new TBBox(
	   new VGlue(0.0, 0.0, 0.0),
       	   name,
	   fixed_vsep,
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   parameters,
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   _list_box->parameters(),
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
      	   _list_box,
	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       ),
       style->flat()
  ); 

  _tool->append(control_buttons);
}

/*
 * MapViewTool Callbacks
 */

void MapViewTool::view_cb()
{
  _view_item = _view->chosen();
  _view_on = _view->chosen_button()->chosen();

  if (_view_on == false)
    _view->chosen_button()->choose(false);

  printf("MapViewTool::view_cb\n");
  printf("\t%d %s\n", _view_item, (_view_on ? "on" : "off"));
}

void MapViewTool::ok_cb()
{
  printf("MapViewTool::ok_cb\n");
  printf("\t%d=%s\n", choice(), value());
}

void MapViewTool::help_cb()
{
  printf("MapViewTool help\n");
}

/*
 * WhereIsTool
 */

declare(ActionCallback, WhereIsTool);
implement(ActionCallback, WhereIsTool);

WhereIsTool::WhereIsTool(Coord label_width, Coord list_height, Coord list_width)
   : ToolboxTool(label_width, list_height, list_width)
{
  Gems* gems = Gems::instance();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();
  Style* style = gems->style();

  Glyph* stretch_vsep = new VGlue(2.0, fil, 0.0);
  Glyph* fixed_vsep = new VGlue(2.0, 0.0, 0.0);
  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);
  
  /*
   * Icon
   */
  icon("pictures/whereis.bm", "pictures/Magnify");

  /*
   * Control Buttons
   */

  Glyph* help = gemsKit->push_button("Help", style,
       new ActionCallback(WhereIsTool)(this, &WhereIsTool::help_cb)
   );
  
  Glyph* info = gemsKit->push_button("Info", style,
       new ActionCallback(WhereIsTool)(this, &WhereIsTool::info_cb)
  );

  Glyph* find = gemsKit->push_button("Find", style,
       new ActionCallback(WhereIsTool)(this, &WhereIsTool::ok_cb)
   );
  	   
  Glyph* control_buttons =
       new LRBox(
       	   help,
       	   stretch_hsep,
       	   info,          
       	   stretch_hsep,
           find
       );

  /* 
   * Layout
   */

  Glyph* name =
       new LRBox(
       	   stretch_hsep,
       	   kit->label("Where Is Tool", style),
       	   stretch_hsep
       );

  _tool = new Background(
       new TBBox(
	   new VGlue(0.0, 0.0, 0.0),
       	   name,
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   _list_box->parameters(),
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
      	   _list_box,
	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
	   control_buttons
       ),
       style->flat()
   );
}

/*
 * WhereIsTool Callbacks
 */

void WhereIsTool::help_cb()
{
  printf("WhereIs::help\n");
}

void WhereIsTool::info_cb()
{
  printf("WhereIsTool::info\n");
}

void WhereIsTool::ok_cb()
{
  printf("WhereIsTool::ok_cb\n");
  printf("\t%d=%s\n", choice(), value());
}

/*
 * ChemistryTool
 */

declare(ActionCallback, ChemistryTool);
implement(ActionCallback, ChemistryTool);

ChemistryTool::ChemistryTool(
       Map* map, Coord label_width, Coord list_height, Coord list_width
   ) : ToolboxTool(label_width, list_height, list_width)
{
  Gems* gems = Gems::instance();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();
  Style* style = gems->style();

  Glyph* stretch_vsep = new VGlue(2.0, fil, 0.0);
  Glyph* fixed_vsep = new VGlue(2.0, 0.0, 0.0);
  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);

  _map = map;
  
  /*
   * Icon
   */
  icon("pictures/chem.bm", "pictures/Chemical");

  /*
   * Control Buttons
   */

  Glyph* help = gemsKit->push_button("Help", style,
       new ActionCallback(ChemistryTool)(this, &ChemistryTool::help_cb)
   );
  
  Glyph* create = gemsKit->push_button("Create", style,
       new ActionCallback(ChemistryTool)(this, &ChemistryTool::ok_cb)
   );
  	   
  Glyph* control_buttons =
       new LRBox(
       	   help,
       	   stretch_hsep,
       	   _list_box->simulation_toolbox()->controls(),
       	   stretch_hsep,
           create
       );

  /* 
   * Layout
   */


  Glyph* name =
       new LRBox(
       	   stretch_hsep,
       	   kit->label("Chemistry Lab", style),
       	   stretch_hsep
       );
       
  _tool = new Background(
       new TBBox(
	   new VGlue(0.0, 0.0, 0.0),
       	   name,
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   _list_box->parameters(),
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   _list_box,
	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
	   control_buttons
       ),
       style->flat()
   );
}

/*
 * ChemistryTool Callbacks
 */

void ChemistryTool::ok_cb()
{
  printf("ChemistryTool::ok_cb()\n");
  printf("\t value()=%s\n", value());
  
  if (strcmp(value(), "Particle Track") == 0)
    particle_track();
  else if (strcmp(value(), "Contor Map") == 0)
     contor_map();
}

void ChemistryTool::contor_map()
{
}

void ChemistryTool::particle_track()
{
  Gems* gems = Gems::instance();
  Style* style = gems->style();
  SimulationController* simulation_controller = gems->controller();
  Display* d = gems->display();

//  DrawingBox* b = new DrawingBox(500.0, 500.0);
  DrawingBox* b = _map->drawing_box();

  ParticleTracker* particle_tracker = 
    (gems->_air_db ? 
     new QueryParticleTracker(simulation_controller, b) :
     new ParticleTracker(simulation_controller, b)
     );

  _list_box->simulation_toolbox()->append(
       	   new ParticleTrackerTool(
       	       _list_box->simulation_toolbox(), 
       	       particle_tracker
       	   )
   );

  particle_tracker->visible(true);
  particle_tracker->current(true);

/*  TopLevelWindow* windowa = new TopLevelWindow(new Background(b, style->background()));
  windowa->name("LA Particle Tracker");
  windowa->place(d->width() / 2.0, d->height() / 2.0);
  windowa->align(0.5, 0.5);
  windowa->map();
*/

}

void ChemistryTool::help_cb()
{
}

/*
 * PopulationTool
 */

declare(ActionCallback, PopulationTool);
implement(ActionCallback, PopulationTool);

PopulationTool::PopulationTool(
       Coord label_width, Coord list_height, Coord list_width
   ) : ToolboxTool(label_width, list_height, list_width)
{
  Gems* gems = Gems::instance();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();
  Style* style = gems->style();

  Glyph* stretch_vsep = new VGlue(2.0, fil, 0.0);
  Glyph* fixed_vsep = new VGlue(2.0, 0.0, 0.0);
  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);
  
  /*
   * Icon
   */
  icon("pictures/population.bm", "pictures/People");

  /*
   * Parameters
   */

  Glyph* parameters = 
       new LRBox(
       	  stretch_hsep,
       	  gemsKit->toggle_button(style, nil, nil, nil, kit->label("Link to Chemistry Lab", style), &_link_to_chem_lab),
       	  stretch_hsep
       );       	  

  /*
   * Control Buttons
   */

  Glyph* help = gemsKit->push_button("Help", style,
       new ActionCallback(PopulationTool)(this, &PopulationTool::help_cb)
   );
  
  Glyph* create = gemsKit->push_button("Create", style,
       new ActionCallback(PopulationTool)(this, &PopulationTool::ok_cb)
   );
  	   
  Glyph* control_buttons =
       new LRBox(
       	   help,
       	   stretch_hsep,
       	   _list_box->simulation_toolbox()->controls(),
       	   stretch_hsep,
           create
       );

  /* 
   * Layout
   */

  Glyph* name =
       new LRBox(
       	   stretch_hsep,
       	   kit->label("Population Tool", style),
       	   stretch_hsep
       );
       
  _tool = new Background(
       new TBBox(
	   new VGlue(0.0, 0.0, 0.0),
       	   name,
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
	   parameters,
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   _list_box->parameters(),
       	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
       	   _list_box,
	   new VMargin(kit->outset_frame(fixed_vsep, style), 2.0),
	   control_buttons
       ),
       style->flat()
   );
}

/*
 * PointSourceTool
 */

declare(ActionCallback, PointSourceTool);
implement(ActionCallback, PointSourceTool);

PointSourceTool::PointSourceTool(
       Coord label_width, Coord list_height, Coord list_width
   ) : ToolboxTool(label_width, list_height, list_width)
{
  Gems* gems = Gems::instance();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();
  Style* style = gems->style();

  Glyph* stretch_vsep = new VGlue(2.0, fil, 0.0);
  Glyph* fixed_vsep = new VGlue(2.0, 0.0, 0.0);
  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);
  
  /*
   * Icon
   */
  icon("pictures/pointsource.bm", "pictures/Point");

  /* 
   * Layout
   */

  _tool = new Background( 
       gems->color_scale(),
       style->flat()
   );
}
