/****************************************************************************
 *
 * Class:  Tabular implementation
 * Author: Mark Roseman
 * 
 * This class manages a table of glyphs arranged in rows and columns, where
 * the size of each row or column is the maximum of the sizes of all elements
 * in the row or column.
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 08/14/92 MR        initial version
 *
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk-ui/tabular.h>
#include <IV-look/kit.h>
#include <InterViews/layout.h>
#include <InterViews/patch.h>
#include <InterViews/canvas.h>
#include <InterViews/polyglyph.h>
#include <stdio.h>

/**************************************************************************
 * 
 * Macros to get at the different parts of the tabular.  It is arranged
 * like this:
 *
 *   --------------------------------------------------------
 *   | ROWS                                                 |
 *   |                                                      |
 *   |  --------------------------------------------------  |
 *   |  | VFIX(0)                                        |  |
 *   |  |  --------------------------------------------  |  |
 *   |  |  | ROW(0)                                   |  |  |
 *   |  |  |                                          |  |  |
 *   |  |  |  ------------------  ------------------  |  |  |
 *   |  |  |  | HFIX(0,0)      |  | HFIX(1,0)      |  |  |  |
 *   |  |  |  |  ------------- |  |  ------------- |  |  |  |
 *   |  |  |  |  | CELL(0,0) | |  |  | CELL(1,0) | |  |  |  |
 *   |  |  |  |  ------------- |  |  ------------- |  |  |  |
 *   |  |  |  ------------------  ------------------  |  |  |
 *   |  |  |                                          |  |  |
 *   |  |  --------------------------------------------  |  |
 *   |  --------------------------------------------------  |
 *   |                                                      |
 *   |  --------------------------------------------------  |
 *   |  | VFIX(1)                                        |  |
 *   |  |  --------------------------------------------  |  |
 *   |  |  | ROW(1)                                   |  |  |
 *   |  |  |                                          |  |  |
 *   |  |  |  ------------------  ------------------  |  |  |
 *   |  |  |  | HFIX(0,1)      |  | HFIX(1,1)      |  |  |  |
 *   |  |  |  |  ------------- |  |  ------------- |  |  |  |
 *   |  |  |  |  | CELL(0,1) | |  |  | CELL(1,1) | |  |  |  |
 *   |  |  |  |  ------------- |  |  ------------- |  |  |  |
 *   |  |  |  ------------------  ------------------  |  |  |
 *   |  |  |                                          |  |  |
 *   |  |  --------------------------------------------  |  |
 *   |  --------------------------------------------------  |
 *   |                                                      |
 *   --------------------------------------------------------
 *
 **************************************************************************/

#define ROWS        ((PolyGlyph*)((Patch*)body())->body())
#define VFIX(i)     ((MonoGlyph*)ROWS->component(i*2))
#define ROW(i)      ((PolyGlyph*)VFIX(i)->body())
#define HFIX(i,j)   ((MonoGlyph*)ROW(i)->component(j*2))
#define CELL(i,j)   ((Glyph*)HFIX(i,j)->body())


#define HSPACE   15.0
#define VSPACE   10.0

/**************************************************************************
 * 
 * create a tabular.
 *
 **************************************************************************/

Tabular::Tabular(int x, int y) : MonoGlyph(nil) {
  kit_ = WidgetKit::instance();
  layout_ = LayoutKit::instance();
  body( 
    new Patch(
      layout_->vbox( 
        layout_->vfixed(
	  layout_->hbox(
	    layout_->hfixed(
              kit_->label(" ")
              ,10.0
            ),
	    layout_->hglue(HSPACE)
          )
          ,10.0
        )
	, layout_->vglue(VSPACE)
      )
    ) 
  );
  rows_ = cols_ = 1;
  while ( x-- > 1 ) 
    insert_col( cols_ );
  while ( y-- > 1 )
    insert_row( rows_ );
}


/**************************************************************************
 * 
 * insert a row
 *
 **************************************************************************/

void Tabular::insert_row( int y ) {
  PolyGlyph* newrow = layout_->hbox();
  for( int i=0; i < cols_; i++ ) {
    newrow->append( layout_->hfixed( kit_->label(" "), 10.0 ) );
    newrow->append( layout_->hglue(HSPACE));
  }
  ROWS->insert( y*2, layout_->vfixed( newrow, 10.0) );
  ROWS->insert(y*2+1, layout_->vglue(VSPACE));
  rows_++;
  recalc();
}

/**************************************************************************
 * 
 * insert a column
 *
 **************************************************************************/

void Tabular::insert_col( int x ) {
  for( int i=0; i < rows_; i++ ) {
    ROW(i)->insert( x*2, layout_->hfixed( kit_->label(" "), 10.0));
    ROW(i)->insert( x*2+1, layout_->hglue(HSPACE));
  }
  cols_++;
  recalc();
}

/**************************************************************************
 * 
 * delete a row
 *
 **************************************************************************/

void Tabular::delete_row( int y ) {
  ROWS->remove( y*2 );
  ROWS->remove( y*2 );
  recalc();
}

/**************************************************************************
 * 
 * delete a column
 *
 **************************************************************************/

void Tabular::delete_col( int x ) {
  for( int i=0; i < rows_; i++ ) {
    ROW(i)->remove(x*2);
    ROW(i)->remove(x*2);
  }
  recalc();
}

/**************************************************************************
 * 
 * replace the glyph at the given (column, row) with a new glyph
 *
 **************************************************************************/

void Tabular::replace( int x, int y, Glyph* g ) {
  HFIX(y,x)->body(g);
  width(x);
  height(y);
  ((Patch*)body())->reallocate();
  if( ((Patch*)body())->canvas() != nil )
    ((Patch*)body())->canvas()->damage_all();
}

/**************************************************************************
 * 
 * return the glyph at the given (column, row)
 *
 **************************************************************************/

Glyph* Tabular::item( int x, int y ) {
  return CELL(x,y);
}

/**************************************************************************
 * 
 * return the number of rows and columns in the tabular (note: counting
 * starts at 0, so last row is ROW( rows()-1 )
 *
 **************************************************************************/

int Tabular::rows() { return rows_; }
int Tabular::cols() { return cols_; }

/**************************************************************************
 * 
 * calculate the maximum width of all the elements in the given column
 *
 **************************************************************************/

void Tabular::width(int x) {
  Requisition req;
  Coord max = 0.0;
  for( int i=0; i < rows_; i++) {
    CELL(i,x)->request(req);
    max = ( req.x_requirement().natural() > max ? req.x_requirement().natural() : max );
  }
  for ( int j = 0; j < rows_; j++ ) {
    Glyph* g = CELL(j,x);
    Resource::ref(g);
    ROW(j)->replace( x*2, layout_->hfixed( g, max ));
    ROW(j)->modified( x*2 );
    Resource::unref(g);
  }
}


/**************************************************************************
 * 
 * calculate the maximum height of all the elements in the given row
 *
 **************************************************************************/

void Tabular::height( int y ) {
  Requisition req;
  Coord max = 0.0;
  for ( int i=0; i < cols_; i++) {
    CELL(y,i)->request(req);
    max = ( req.y_requirement().natural() > max ? req.y_requirement().natural() : max );
  }
  Glyph* g = ROW(y);
  Resource::ref(g);
  ROWS->replace( y*2, layout_->vfixed( g, max  ));
  ROWS->modified( y*2 );
  Resource::unref(g);
}
  
/**************************************************************************
 * 
 * figure out the new size for the tabular
 *
 **************************************************************************/

void Tabular::recalc() {
  for(int i = 0; i < cols_; i++) 
    width(i);
  for(int j = 0; j < rows_; j++) 
    height(j);
  ((Patch*)body())->reallocate();
  if(((Patch*)body())->canvas() != nil) {
    ((Patch*)body())->canvas()->damage_all();
    ((Patch*)body())->redraw();
  }
  ((Patch*)body())->change(0);
}
