/****************************************************************************
 *
 * Classes: SlaveRegClient implementation
 * Author:  Mark Roseman
 *
 * Do an "slave" registrar client, suitable for use by a user in a strictly
 * facilitated conference.
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 10/13/92 MR        initial version
 *
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk-reg/coordinator.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <IV-look/kit.h>
#include <InterViews/layout.h>
#include <gk-ui/strbrowser.h>
#include <IV-look/dialogs.h>
#include <InterViews/window.h>
#include "slaverc.h"
#include <gk/groupsession.h>
#include <gk-ui/labscrlst.h>
#include <InterViews/display.h>
#include <InterViews/patch.h>
#include <OS/table.h>
#include <InterViews/action.h>

declareActionCallback(SlaveRegClient)
implementActionCallback(SlaveRegClient)

declareTable(IntTable,int,int);
implementTable(IntTable,int,int);

/****************************************************************************
 *
 * Constructor.  Pop up a dialog asking the user to select the facilitated
 * conference to join.
 *
 ****************************************************************************/


SlaveRegClient::SlaveRegClient(const char* host, int port, Coordinator* c) : 
                  RegistrarClient(host,port,c), connected_(false) 
{
  cnum_ = new IntTable(20);
  kit_ = WidgetKit::instance();
  LayoutKit& layout = *LayoutKit::instance();
  confs_ = new StringBrowser( kit_, new ActionCallback(SlaveRegClient)
			     (this,&SlaveRegClient::selected_conf), nil );
  patch_ = new Patch( new LabelledScrollList( 
          kit_, "Select a Facilitated Conference:", confs_, 100, 100 ));
  tw_ = new TransientWindow(
    kit_->inset_frame( 
      layout.margin( 
        patch_
        , 10
      )
    )
  );
  tw_->place( Session::instance()->default_display()->width() / 2.0,
	      Session::instance()->default_display()->height() / 2.0);
  tw_->align( 0.5, 0.5 );
  tw_->map();
}

/****************************************************************************
 * 
 * found a new conference - if we're still waiting to join update the
 *    dialog box
 *
 ****************************************************************************/

void SlaveRegClient::foundNewConference(AttributeList* al) {
  char conf[80];
  al->find_attribute("confnum", conf);
  conference_tbl_->insert( atoi(conf), al );
  users_tbl_->insert( atoi(conf), new AttrListTable(30));
  if ( !connected_ )
    update_browser();
}


/****************************************************************************
 * 
 * update the dialog box with the list of facilitated conferences the user
 * could join - walk the list of conferences and stick them in
 *
 ****************************************************************************/

void SlaveRegClient::update_browser() 
{
  char type[80], name[80], conf[80];
  confs_->Clear();
  for ( TableIterator(AttrLstTbl) i(*conference_tbl_); i.more(); i.next()) {
    i.cur_value()->find_attribute( "type", type);
    if (strcmp(type, "FacilitatedConference")==0) {
      i.cur_value()->find_attribute( "name", name );
      confs_->Append(name);
      i.cur_value()->find_attribute( "confnum", conf );
      cnum_->remove( confs_->Count()-1 );
      cnum_->insert( confs_->Count()-1, atoi(conf));
    }
  }
  patch_->redraw();
}


/****************************************************************************
 * 
 * user selected a facilitated conference from the browser - join it, and
 *   then delete the dialog box
 *
 ****************************************************************************/

void SlaveRegClient::selected_conf() {
  int confnum;
  if (cnum_->find( confnum, (int)confs_->selected() )) {
    callJoinConference( confnum );
    tw_->unmap();
  } else
    fprintf(stderr, "could not find it\n");
}


/****************************************************************************
 *
 * found a deleted conference - update the dialog box if displayed
 *
 ****************************************************************************/

void SlaveRegClient::foundDeletedConference(int conf) {
  conference_tbl_->remove(conf);
  users_tbl_->remove(conf);
  coord_->deleteConference(conf);
  if( !connected_ ) 
    update_browser();
}


/****************************************************************************
 *
 * found new user for a conference - if the user is us:
 *    
 *   (a) create the conference
 *   (b) pass on "local info" (e.g. id) to the conference
 *   (c) join up to all other users in the conference
 *
 ****************************************************************************/

void SlaveRegClient::foundNewUser(AttributeList* al) {
  char conf[80], usernum[80], hostname[80], port[10], hostnm[80], portnm[80];
  AttrListTable* usrs;
  AttributeList* conf_al;
  al->find_attribute("confnum", conf);
  al->find_attribute("usernum", usernum);
  al->find_attribute("host", hostname);
  al->find_attribute("port", port);
  if(users_tbl_->find( usrs, atoi(conf))) {
    usrs->insert( atoi(usernum), al);

  if (( strcmp( hostname, GroupSession::host_name())==0) &&
      ( atoi(port)==lPort())) {
    if (conference_tbl_->find( conf_al, atoi(conf))) {
      coord_->createConference( conf_al );
      coord_->setLocalInfo( al );
	for(TableIterator(AttrLstTbl) i(*usrs); i.more(); i.next()) {
	  i.cur_value()->find_attribute("host", hostnm );
	  i.cur_value()->find_attribute("port", portnm );
	  if ( (strcmp(hostnm, GroupSession::host_name())!=0) || 
	      (atoi(portnm) != lPort())) 
	    coord_->joinTo(i.cur_value());
	}
      }
    }
  }
}


/****************************************************************************
 *
 * found a deleted user - pass it on to the conference via the coordinator
 *
 ****************************************************************************/

void SlaveRegClient::foundDeletedUser(int conf, int user) {
  fprintf(stderr, "deleting user\n");
  AttrListTable* users;
  users_tbl_->find( users, conf );
  users->remove( user );
  coord_->deleteUser(conf,user);
}



/****************************************************************************
 * 
 * one of the local conferences has been destroyed, tell the registrar
 *     the user has left the conference
 *
 ****************************************************************************/

void SlaveRegClient::userLeft(int conf_id, int /* user_id */) {
  char port[80], host[80], usernum[80];
  AttrListTable* usrs;
  users_tbl_->find( usrs, conf_id );
  for ( TableIterator(AttrLstTbl) i(*usrs); i.more(); i.next()) {
    i.cur_value()->find_attribute("host", host);
    i.cur_value()->find_attribute("port", port);
    if ((atoi(port)==lPort()) && (strcmp(host,GroupSession::host_name())==0)) {
      i.cur_value()->find_attribute("usernum", usernum);
      callLeaveConference( conf_id, atoi(usernum));
    }
  }
}





