/***************************************************************************
 *
 * Class:  Connection and ConnectionList
 * Author: Mark Roseman
 * 
 * This class provides a convenient way of bundling a RpcWriter and an
 * RpcReader together.  Nothing complicated but useful.  The list similarly
 * provides a few straightforward conveniences, the most important of having
 * a set of callback routines which are used by every RpcReader in the list.
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 02/17/92 MR        initial version
 * 02/20/92 MR        declared ConnectionList class here
 * 05/17/92 MR        added "id" field / findConnection for use by conference
 * 05/28/92 MR        redid to work with CallbackRpcReader and RpcWriter
 * 08/14/92 MR        clean up
 * 09/17/92 MR        changed add routines to also return created connection
 * 
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#ifndef __connection_h
#define __connection_h

#include <OS/list.h>

class Connection {
public:

  /*
  ** constructor / destructor
  */
  Connection(const char* host, int port, int id=-1, 
	     class ReaderCallbackTable* = nil);
  Connection(int fd, int id=-1, class ReaderCallbackTable* = nil);
  Connection(class rpcstream*, int id=-1, class ReaderCallbackTable* = nil);
  virtual ~Connection();

  /*
  ** id number of the remote user -- needs to be assigned by several
  ** different classes, so left public (essentially its write-once-read-many)
  */
  int id_;

  /*
  ** access methods 
  */
  class CallbackRpcReader* reader();
  class Writer* writer();
  class ReaderCallbackTable* callbacks();

private:
  class ReaderCallbackTable* callbacks_;
  class CallbackRpcReader* reader_;
  class Writer* writer_;
};


/*
** connectionlist
*/

declarePtrList(ConnList, Connection);

class ConnectionList {
public:

  /*
  ** constructor / destructor 
  */
  ConnectionList();
  ~ConnectionList();

  /*
  ** add a new connection to the list
  */
  virtual Connection* add(char* host, int port, int id=-1);
  virtual Connection* add(int fd, int id=-1);
  virtual Connection* add(class rpcstream*, int id=-1);

  /*
  ** given the id of a connection, find its index in the list
  */
  long find(int id);

  /* 
  ** send a message out on a single connection, or all of them 
  */
  void sendTo(int id, class MsgSender*);
  void toAll(class MsgSender*);

  /*
  ** access methods 
  */
  ConnList* list();
  class ReaderCallbackTable* callbacks();

protected:
  class ReaderCallbackTable* callbacks_;
  ConnList* list_;
};

#endif
