#line 1
package Test::More;

use 5.008001;
use strict;
use warnings;

our $VERSION = '1.301001_096';
$VERSION = eval $VERSION;    ## no critic (BuiltinFunctions::ProhibitStringyEval)

use Test::Stream 1.301001 '-internal';
use Test::Stream::Util qw/protect try spoof/;
use Test::Stream::Toolset qw/is_tester init_tester context before_import/;
use Test::Stream::Subtest qw/subtest/;

use Test::Stream::Carp qw/croak carp/;
use Scalar::Util qw/blessed/;

use Test::More::Tools;
use Test::More::DeepCheck::Strict;

use Test::Builder;

use Test::Stream::Exporter qw/
    default_export default_exports export_to export_to_level
/;

our $TODO;
default_export '$TODO' => \$TODO;
default_exports qw{
    plan done_testing

    ok
    is isnt
    like unlike
    cmp_ok
    is_deeply
    eq_array eq_hash eq_set
    can_ok isa_ok new_ok
    pass fail
    require_ok use_ok
    subtest

    explain

    diag note

    skip todo_skip
    BAIL_OUT
};
Test::Stream::Exporter->cleanup;

{
    no warnings 'once';
    $Test::Builder::Level ||= 1;
}

sub import {
    my $class = shift;
    my $caller = caller;
    my @args = @_;

    my $stash = $class->before_import($caller, \@args) if $class->can('before_import');
    export_to($class, $caller, @args);
    $class->after_import($caller, $stash, @args) if $class->can('after_import');
    $class->import_extra(@args);
}

sub import_extra { 1 };

sub builder { Test::Builder->new }

sub ok ($;$) {
    my ($test, $name) = @_;
    my $ctx  = context();
    if($test) {
        $ctx->ok(1, $name);
        return 1;
    }
    else {
        $ctx->ok(0, $name);
        return 0;
    }
}

sub plan {
    return unless @_;
    my ($directive, $arg) = @_;
    my $ctx = context();

    if ($directive eq 'tests') {
        $ctx->plan($arg);
    }
    else {
        $ctx->plan(0, $directive, $arg);
    }
}

sub done_testing {
    my ($num) = @_;
    my $ctx = context();
    $ctx->done_testing($num);
}

sub is($$;$) {
    my ($got, $want, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = tmt->is_eq($got, $want);
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub isnt ($$;$) {
    my ($got, $forbid, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = tmt->isnt_eq($got, $forbid);
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

{
    no warnings 'once';
    *isn't = \&isnt;
    # ' to unconfuse syntax higlighters
}

sub like ($$;$) {
    my ($got, $check, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = tmt->regex_check($got, $check, '=~');
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub unlike ($$;$) {
    my ($got, $forbid, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = tmt->regex_check($got, $forbid, '!~');
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub cmp_ok($$$;$) {
    my ($got, $type, $expect, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = tmt->cmp_check($got, $type, $expect);
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub can_ok($@) {
    my ($thing, @methods) = @_;
    my $ctx = context();

    my $class = ref $thing || $thing || '';
    my ($ok, @diag);

    if (!@methods) {
        ($ok, @diag) = (0, "    can_ok() called with no methods");
    }
    elsif (!$class) {
        ($ok, @diag) = (0, "    can_ok() called with empty class or reference");
    }
    else {
        ($ok, @diag) = tmt->can_check($thing, $class, @methods);
    }

    my $name = (@methods == 1 && defined $methods[0])
        ? "$class\->can('$methods[0]')"
        : "$class\->can(...)";

    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub isa_ok ($$;$) {
    my ($thing, $class, $thing_name) = @_;
    my $ctx = context();
    $thing_name = "'$thing_name'" if $thing_name;
    my ($ok, @diag) = tmt->isa_check($thing, $class, \$thing_name);
    my $name = "$thing_name isa '$class'";
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub new_ok {
    croak "new_ok() must be given at least a class" unless @_;
    my ($class, $args, $object_name) = @_;
    my $ctx = context();
    my ($obj, $name, $ok, @diag) = tmt->new_check($class, $args, $object_name);
    $ctx->ok($ok, $name, \@diag);
    return $obj;
}

sub pass (;$) {
    my $ctx = context();
    return $ctx->ok(1, @_);
}

sub fail (;$) {
    my $ctx = context();
    return $ctx->ok(0, @_);
}

sub explain {
    my $ctx = context();
    tmt->explain(@_);
}

sub diag {
    my $ctx = context();
    $ctx->diag($_) for @_;
}

sub note {
    my $ctx = context();
    $ctx->note($_) for @_;
}

sub skip {
    my( $why, $how_many ) = @_;
    my $ctx = context();

    _skip($why, $how_many, 'skip', 1);

    no warnings 'exiting';
    last SKIP;
}

sub _skip {
    my( $why, $how_many, $func, $bool ) = @_;
    my $ctx = context();

    my $plan = $ctx->stream->plan;

    # If there is no plan we do not need to worry about counts
    my $need_count = $plan ? !($plan->directive && $plan->directive eq 'NO PLAN') : 0;

    if ($need_count && !defined $how_many) {
        $ctx->alert("$func() needs to know \$how_many tests are in the block");
    }

    $ctx->alert("$func() was passed a non-numeric number of tests.  Did you get the arguments backwards?")
        if defined $how_many and $how_many =~ /\D/;

    $how_many = 1 unless defined $how_many;
    $ctx->set_skip($why);
    for( 1 .. $how_many ) {
        $ctx->ok($bool, '');
    }
}

sub todo_skip {
    my($why, $how_many) = @_;

    my $ctx = context();
    $ctx->set_in_todo(1);
    $ctx->set_todo($why);
    _skip($why, $how_many, 'todo_skip', 0);

    no warnings 'exiting';
    last TODO;
}

sub BAIL_OUT {
    my ($reason) = @_;
    my $ctx = context();
    $ctx->bail($reason);
}

sub is_deeply {
    my ($got, $want, $name) = @_;

    my $ctx = context();

    unless( @_ == 2 or @_ == 3 ) {
        my $msg = <<'WARNING';
is_deeply() takes two or three args, you gave %d.
This usually means you passed an array or hash instead
of a reference to it
WARNING
        chop $msg;    # clip off newline so carp() will put in line/file

        $ctx->alert(sprintf $msg, scalar @_);

        $ctx->ok(0, undef, ['incorrect number of args']);
        return 0;
    }

    my ($ok, @diag) = Test::More::DeepCheck::Strict->check($got, $want);
    $ctx->ok($ok, $name, \@diag);
    return $ok;
}

sub eq_array {
    my ($got, $want, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = Test::More::DeepCheck::Strict->check_array($got, $want);
    return $ok;
}

sub eq_hash {
    my ($got, $want, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = Test::More::DeepCheck::Strict->check_hash($got, $want);
    return $ok;
}

sub eq_set {
    my ($got, $want, $name) = @_;
    my $ctx = context();
    my ($ok, @diag) = Test::More::DeepCheck::Strict->check_set($got, $want);
    return $ok;
}

sub require_ok($;$) {
    my($module) = shift;
    my $ctx = context();

    # Try to determine if we've been given a module name or file.
    # Module names must be barewords, files not.
    $module = qq['$module'] unless _is_module_name($module);

    my ($ret, $err);
    {
        local $SIG{__DIE__};
        ($ret, $err) = spoof [caller] => "require $module";
    }

    my @diag;
    unless ($ret) {
        chomp $err;
        push @diag => <<"        DIAG";
    Tried to require '$module'.
    Error:  $err
        DIAG
    }

    $ctx->ok( $ret, "require $module;", \@diag );
    return $ret ? 1 : 0;
}

sub _is_module_name {
    my $module = shift;

    # Module names start with a letter.
    # End with an alphanumeric.
    # The rest is an alphanumeric or ::
    $module =~ s/\b::\b//g;

    return $module =~ /^[a-zA-Z]\w*$/ ? 1 : 0;
}

sub use_ok($;@) {
    my ($module, @imports) = @_;
    @imports = () unless @imports;
    my $ctx = context();

    my($pack, $filename, $line) = caller;
    $filename =~ y/\n\r/_/; # so it doesn't run off the "#line $line $f" line

    my ($ret, $err, $newdie, @diag);
    {
        local $SIG{__DIE__};

        if( @imports == 1 and $imports[0] =~ /^\d+(?:\.\d+)?$/ ) {
            # probably a version check.  Perl needs to see the bare number
            # for it to work with non-Exporter based modules.
            ($ret, $err) = spoof [$pack, $filename, $line] => "use $module $imports[0]";
        }
        else {
            ($ret, $err) = spoof [$pack, $filename, $line] => "use $module \@args", @imports;
        }

        $newdie = $SIG{__DIE__};
    }

    $SIG{__DIE__} = $newdie if defined $newdie;

    unless ($ret) {
        chomp $err;
        push @diag => <<"        DIAG";
    Tried to use '$module'.
    Error:  $err
        DIAG
    }

    $ctx->ok($ret, "use $module;", \@diag);

    return $ret ? 1 : 0;
}

1;

__END__

